//
// Syd: rock-solid application kernel
// src/confine.rs: Sandboxing utilities
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
// SPDX-License-Identifier: GPL-3.0

use std::{
    arch::asm,
    fmt::Display,
    os::{
        fd::{AsFd, AsRawFd, RawFd},
        unix::process::ExitStatusExt,
    },
    path::Path,
    process::{exit, Command},
    str::FromStr,
    sync::atomic::Ordering,
};

use btoi::btoi;
use libc::{c_int, c_ulong, prctl, EACCES, EOPNOTSUPP};
use libseccomp::{scmp_cmp, ScmpAction, ScmpArch, ScmpFilterContext, ScmpSyscall};
use nix::{
    dir::Dir,
    errno::Errno,
    fcntl::OFlag,
    mount::MsFlags,
    sched::{unshare, CloneFlags},
    sys::{
        personality::Persona,
        stat::Mode,
        wait::{waitpid, Id, WaitPidFlag},
    },
    unistd::{fork, ForkResult, Gid, Pid, Uid},
};
use procfs_core::process::{MMPermissions, MMapPath, MemoryMap};
use serde::Serialize;

use crate::{
    caps,
    compat::waitid,
    config::{HAVE_RWF_NOAPPEND, MMAP_MIN_ADDR, UNSAFE_PERSONAS},
    err::{err2no, SydResult},
    fs::{readlinkat, safe_clone, FileType, AT_BADFD},
    info,
    landlock::{
        path_beneath_rules, Access, AccessFs, AccessNet, NetPort, RestrictSelfFlags,
        RestrictionStatus, Ruleset, RulesetAttr, RulesetCreatedAttr, RulesetError, RulesetStatus,
        Scope, ABI,
    },
    path::{mask_path, XPathBuf},
    sandbox::{IoctlMap, Sandbox},
};

/// Confine current process using MDWE prctl(2).
///
/// Use `no_inherit` to prevent inheriting the restriction to children.
pub fn confine_mdwe(no_inherit: bool) -> Result<(), Errno> {
    const PR_SET_MDWE: c_int = 65;
    const PR_MDWE_REFUSE_EXEC_GAIN: c_ulong = 1;
    const PR_MDWE_NO_INHERIT: c_ulong = 2;

    let mut flags = PR_MDWE_REFUSE_EXEC_GAIN;
    if no_inherit {
        flags |= PR_MDWE_NO_INHERIT;
    }

    // SAFETY: In libc we trust.
    Errno::result(unsafe { prctl(PR_SET_MDWE, flags, 0, 0, 0) }).map(drop)
}

/// Apply W^X memory restrictions using _seccomp_(2).
pub fn confine_scmp_wx() -> SydResult<()> {
    let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;
    // We don't want ECANCELED, we want actual errnos.
    let _ = ctx.set_api_sysrawrc(true);
    // We kill for bad system call and bad arch.
    let _ = ctx.set_act_badarch(ScmpAction::KillProcess);
    // Use a binary tree sorted by syscall number.
    let _ = ctx.set_ctl_optimize(2);

    seccomp_add_architectures(&mut ctx)?;

    // Seccomp W^X restrictions:
    //
    // - Prevent mmap(addr<${mmap_min_addr}, MAP_FIXED).
    // - Prohibit attempts to create memory mappings
    //   that are writable and executable at the same time, or to
    //   change existing memory mappings to become executable, or
    //   mapping shared memory segments as executable.
    // - Deny unsafe personality(2) personas.

    const MAP_FIXED: u64 = libc::MAP_FIXED as u64;
    const MAP_FIXED_NOREPLACE: u64 = libc::MAP_FIXED_NOREPLACE as u64;
    const W: u64 = libc::PROT_WRITE as u64;
    const X: u64 = libc::PROT_EXEC as u64;
    const WX: u64 = W | X;
    const SHM_X: u64 = libc::SHM_EXEC as u64;
    const MAP_A: u64 = libc::MAP_ANONYMOUS as u64;
    const MAP_S: u64 = libc::MAP_SHARED as u64;

    let mmap_min_addr = *MMAP_MIN_ADDR;
    for sysname in ["mmap", "mmap2"] {
        let syscall = if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            syscall
        } else {
            continue;
        };

        // Prevent fixed mappings under mmap_min_addr.
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[
                scmp_cmp!($arg0 < mmap_min_addr),
                scmp_cmp!($arg3 & MAP_FIXED == MAP_FIXED),
            ],
        )?;
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[
                scmp_cmp!($arg0 < mmap_min_addr),
                scmp_cmp!($arg3 & MAP_FIXED_NOREPLACE == MAP_FIXED_NOREPLACE),
            ],
        )?;

        // Prevent writable and executable memory.
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[scmp_cmp!($arg2 & WX == WX)],
        )?;

        // Prevent executable anonymous memory.
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[scmp_cmp!($arg2 & X == X), scmp_cmp!($arg3 & MAP_A == MAP_A)],
        )?;

        // Prevent executable shared memory.
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[scmp_cmp!($arg2 & X == X), scmp_cmp!($arg3 & MAP_S == MAP_S)],
        )?;
    }

    for sysname in ["mprotect", "pkey_mprotect"] {
        let syscall = if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            syscall
        } else {
            continue;
        };

        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[scmp_cmp!($arg2 & X == X)],
        )?;
    }

    if let Ok(syscall) = ScmpSyscall::from_name("shmat") {
        ctx.add_rule_conditional(
            ScmpAction::KillProcess,
            syscall,
            &[scmp_cmp!($arg2 & SHM_X == SHM_X)],
        )?;
    }

    if let Ok(syscall) = ScmpSyscall::from_name("personality") {
        #[allow(clippy::cast_sign_loss)]
        for persona in UNSAFE_PERSONAS {
            let persona = persona.bits() as u64;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                syscall,
                &[scmp_cmp!($arg0 & persona == persona)],
            )?;
        }
    }

    ctx.load()?;

    Ok(())
}

/// Add per-architecture seccomp(2) filters to deny given ioctl(2) requests.
///
/// Set `ssb` to true to disable Speculative Store Bypass mitigations.
pub fn confine_scmp_ioctl(denylist: &IoctlMap, ssb: bool) -> SydResult<()> {
    let syscall = ScmpSyscall::from_name("ioctl").or(Err(Errno::ENOSYS))?;
    for arch in SCMP_ARCH {
        let denylist = if let Some(denylist) = denylist.get(arch) {
            denylist
        } else {
            continue;
        };

        // Prepare per-architecture seccomp(2) filter.
        let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;

        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;

        // Enable Speculative Store Bypass mitigations.
        ctx.set_ctl_ssb(ssb)?;

        // Do not synchronize filter to all threads.
        ctx.set_ctl_tsync(false)?;

        // Allow bad/unsupported architectures,
        // this is a per-architecture filter.
        ctx.set_act_badarch(ScmpAction::Allow)?;

        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);

        // We don't want ECANCELED, we want actual errnos.
        let _ = ctx.set_api_sysrawrc(true);

        // Remove native architecture from filter,
        // and add the specific architecture.
        ctx.remove_arch(ScmpArch::native())?;
        ctx.add_arch(*arch)?;

        #[allow(clippy::useless_conversion)]
        for request in denylist {
            let request = (*request).into();
            ctx.add_rule_conditional(
                ScmpAction::Errno(EACCES),
                syscall,
                &[scmp_cmp!($arg1 == request)],
            )?;
            if let Some(request) = extend_ioctl(request) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(EACCES),
                    syscall,
                    &[scmp_cmp!($arg1 == request)],
                )?;
            }
        }

        ctx.load()?;
    }

    Ok(())
}

/// pwritev2(2) flag for per-IO negation of O_APPEND
pub const RWF_NOAPPEND: u64 = 0x00000020;

/// Deny pwritev2(2) system call when flags include
/// RWF_NOAPPEND with the EOPNOTSUPP errno.
///
/// Optimized so that:
///   - if SCMP_ARCH contains X32: install per-arch filters (X32 uses $arg4)
///   - else: install a single filter using $arg5 (libseccomp/natural ABI)
///
/// Set `ssb` to true to disable Speculative Store Bypass mitigations.
pub fn confine_scmp_pwritev2(ssb: bool) -> SydResult<()> {
    if !*HAVE_RWF_NOAPPEND {
        // RWF_NOAPPEND not supported, nothing to do.
        return Ok(());
    }

    let syscall = if let Ok(syscall) = ScmpSyscall::from_name("pwritev2") {
        syscall
    } else {
        // pwritev2(2) not supported, nothing to do.
        return Ok(());
    };

    if !SCMP_ARCH.contains(&ScmpArch::X32) {
        // Fast path: all supported archs have flags at $arg5.
        let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;

        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;

        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(ssb)?;

        // Synchronize filter to all threads.
        ctx.set_ctl_tsync(true)?;

        // We deny with ENOSYS for bad/unsupported system call, and kill process for bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;

        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);

        // We don't want ECANCELED, we want actual errnos.
        let _ = ctx.set_api_sysrawrc(true);

        // Add supported architectures.
        seccomp_add_architectures(&mut ctx)?;

        // Deny pwritev2(2) using RWF_NOAPPEND with EOPNOTSUPP.
        let rule = scmp_cmp!($arg5 & RWF_NOAPPEND == RWF_NOAPPEND);
        ctx.add_rule_conditional(ScmpAction::Errno(EOPNOTSUPP), syscall, &[rule])?;

        // Load the arch-agnostic filter and return.
        return Ok(ctx.load()?);
    }

    // Slow path with x32 flags at $arg4 and others at $arg5.
    // Install per-arch filters with the correct index.
    for arch in SCMP_ARCH {
        // Prepare per-architecture seccomp(2) filter.
        let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;

        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;

        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(ssb)?;

        // Do not synchronize filter to all threads.
        ctx.set_ctl_tsync(false)?;

        // Allow bad/unsupported architectures,
        // this is a per-architecture filter.
        ctx.set_act_badarch(ScmpAction::Allow)?;

        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);

        // We don't want ECANCELED, we want actual errnos.
        let _ = ctx.set_api_sysrawrc(true);

        // Remove native architecture from filter,
        // and add the specific architecture.
        ctx.remove_arch(ScmpArch::native())?;
        ctx.add_arch(*arch)?;

        // x32: flags is $arg4; everybody else here: $arg5.
        let rule = if *arch == ScmpArch::X32 {
            scmp_cmp!($arg4 & RWF_NOAPPEND == RWF_NOAPPEND)
        } else {
            scmp_cmp!($arg5 & RWF_NOAPPEND == RWF_NOAPPEND)
        };
        ctx.add_rule_conditional(ScmpAction::Errno(EOPNOTSUPP), syscall, &[rule])?;

        // Load the arch-specific filter.
        ctx.load()?;
    }

    Ok(())
}

/// Allow writes to sandbox `SYD_LOG_FD` only.
///
/// If logging is disabled:
///
/// a. If `max` is `None`, deny write(2) completely.
/// b. If `max` is `Some(limit)`, allow writes up to `max` bytes.
///
/// It is OK for the `SYD_LOG_FD` to be negative,
/// in which case no rule will be inserted
/// for the fd.
///
/// # Exceptions
///
/// 1. Allow write(2) globally if profiling is enabled.
/// 2. Allow write(2) globally if `chk_mem` is true,
///    and `Sandbox::memory_access` is less than 2.
///    This is required for proc_pid_mem(5) access.
pub fn confine_scmp_write(
    ctx: &mut ScmpFilterContext,
    max: Option<u64>,
    chk_mem: bool,
) -> SydResult<()> {
    let syscall = match ScmpSyscall::from_name("write") {
        Ok(syscall) => syscall,
        Err(_) => {
            info!("ctx": "confine", "op": "allow_syscall",
                "msg": "invalid or unsupported syscall write");
            return Ok(());
        }
    };

    if cfg!(feature = "prof") && (chk_mem && Sandbox::memory_access() < 2) {
        ctx.add_rule(ScmpAction::Allow, syscall)?;
        return Ok(());
    }

    if let Ok(log_fd) = u64::try_from(crate::log::LOG_FD.load(Ordering::Relaxed)) {
        ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg0 == log_fd)])?;
        if let Some(max) = max {
            ctx.add_rule_conditional(
                ScmpAction::Allow,
                syscall,
                &[scmp_cmp!($arg0 != log_fd), scmp_cmp!($arg2 <= max)],
            )?;
        }
    } else if let Some(max) = max {
        ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg2 <= max)])?;
    } // else deny write(2) completely.

    Ok(())
}

/// Convenience `Command` run wrapper which returns:
///
/// - Same exit code as the process on clean exit.
/// - 128 plus signal number on unclean termination.
/// - `Errno` number if executing the process fails.
pub fn run_cmd(cmd: &mut Command) -> u8 {
    #![allow(clippy::arithmetic_side_effects)]
    #![allow(clippy::cast_possible_truncation)]
    #![allow(clippy::cast_sign_loss)]
    match cmd.status() {
        Ok(status) => {
            if let Some(code) = status.code() {
                code as u8
            } else if let Some(sig) = status.signal() {
                128 + (sig as u8)
            } else {
                127
            }
        }
        Err(error) => err2no(&error) as i32 as u8,
    }
}

/// Simple wrapper over ScmpSyscall and ScmpArch to provide Display.
#[derive(Copy, Clone, Debug, Eq, PartialEq, Ord, PartialOrd, Hash)]
pub struct Sydcall(pub ScmpSyscall, pub u32);

impl Display for Sydcall {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let arch = match scmp_arch(self.1) {
            Ok(arch) => arch,
            Err(_) => return write!(f, "?"),
        };

        match self.0.get_name_by_arch(arch).ok() {
            Some(name) => write!(f, "{name}"),
            None => write!(f, "?"),
        }
    }
}

impl Serialize for Sydcall {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let arch = match scmp_arch(self.1) {
            Ok(arch) => arch,
            Err(_) => return serializer.serialize_none(),
        };

        match self.0.get_name_by_arch(arch).ok() {
            Some(name) => serializer.serialize_str(&name),
            None => serializer.serialize_none(),
        }
    }
}

pub(crate) struct SydArch(pub(crate) ScmpArch);

impl Display for SydArch {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        let arch = format!("{:?}", self.0).to_ascii_lowercase();
        let arch = if arch == { "x8664" } { "x86_64" } else { &arch };
        write!(f, "{arch}")
    }
}

impl Serialize for SydArch {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let arch = format!("{:?}", self.0).to_ascii_lowercase();
        let arch = if arch == { "x8664" } { "x86_64" } else { &arch };
        serializer.serialize_str(arch)
    }
}

/// A wrapper type that wraps MemoryMap and provides `Serialize`.
#[derive(Debug, PartialEq, Eq, Clone)]
pub struct SydMemoryMap(pub MemoryMap);

impl SydMemoryMap {
    /// Checks if the memory map points to a stack.
    pub fn is_stack(&self) -> bool {
        matches!(self.0.pathname, MMapPath::Stack | MMapPath::TStack(_))
    }
}

impl Display for SydMemoryMap {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        let mmap = &self.0;

        // Build permissions string.
        let perms = format!(
            "{}{}{}{}",
            if mmap.perms.contains(MMPermissions::READ) {
                "r"
            } else {
                "-"
            },
            if mmap.perms.contains(MMPermissions::WRITE) {
                "w"
            } else {
                "-"
            },
            if mmap.perms.contains(MMPermissions::EXECUTE) {
                "x"
            } else {
                "-"
            },
            if mmap.perms.contains(MMPermissions::SHARED) {
                "s"
            } else if mmap.perms.contains(MMPermissions::PRIVATE) {
                "p"
            } else {
                "-"
            }
        );

        // Map pathname.
        let pathname = match &mmap.pathname {
            MMapPath::Path(path) => mask_path(path),
            MMapPath::Heap => "[heap]".to_string(),
            MMapPath::Stack => "[stack]".to_string(),
            MMapPath::TStack(tid) => format!("[stack:{tid}]"),
            MMapPath::Vdso => "[vdso]".to_string(),
            MMapPath::Vvar => "[vvar]".to_string(),
            MMapPath::Vsyscall => "[vsyscall]".to_string(),
            MMapPath::Rollup => "[rollup]".to_string(),
            MMapPath::Anonymous => "[anon]".to_string(),
            MMapPath::Vsys(key) => format!("[vsys:{key}]"),
            MMapPath::Other(pseudo_path) => mask_path(Path::new(pseudo_path)),
        };

        // Format output line.
        write!(
            f,
            "{:x}-{:x} {perms:<4} {:08x} {:02x}:{:02x} {:<10} {pathname}",
            mmap.address.0, mmap.address.1, mmap.offset, mmap.dev.0, mmap.dev.1, mmap.inode,
        )
    }
}

impl Serialize for SydMemoryMap {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(&self.to_string())
    }
}

/// A type that wraps personality(2) return value and implements Display.
pub(crate) struct SydPersona(pub(crate) Persona);

impl Display for SydPersona {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        // Execution domain constants, taken from sys/personality.h
        const PER_LINUX: c_int = 0;
        const PER_LINUX_32BIT: c_int = PER_LINUX | ADDR_LIMIT_32BIT;
        const PER_LINUX_FDPIC: c_int = PER_LINUX | FDPIC_FUNCPTRS;
        const PER_SVR4: c_int = 1 | STICKY_TIMEOUTS | MMAP_PAGE_ZERO;
        const PER_SVR3: c_int = 2 | STICKY_TIMEOUTS | SHORT_INODE;
        const PER_SCOSVR3: c_int = 3 | STICKY_TIMEOUTS | WHOLE_SECONDS | SHORT_INODE;
        const PER_OSR5: c_int = 3 | STICKY_TIMEOUTS | WHOLE_SECONDS;
        const PER_WYSEV386: c_int = 4 | STICKY_TIMEOUTS | SHORT_INODE;
        const PER_ISCR4: c_int = 5 | STICKY_TIMEOUTS;
        const PER_BSD: c_int = 6;
        const PER_SUNOS: c_int = PER_BSD | STICKY_TIMEOUTS;
        const PER_XENIX: c_int = 7 | STICKY_TIMEOUTS | SHORT_INODE;
        const PER_LINUX32: c_int = 8;
        const PER_LINUX32_3GB: c_int = PER_LINUX32 | ADDR_LIMIT_3GB;
        const PER_IRIX32: c_int = 9 | STICKY_TIMEOUTS;
        const PER_IRIXN32: c_int = 0xa | STICKY_TIMEOUTS;
        const PER_IRIX64: c_int = 0x0b | STICKY_TIMEOUTS;
        const PER_RISCOS: c_int = 0xc;
        const PER_SOLARIS: c_int = 0xd | STICKY_TIMEOUTS;
        const PER_UW7: c_int = 0xe | STICKY_TIMEOUTS | MMAP_PAGE_ZERO;
        const PER_OSF4: c_int = 0xf;
        const PER_HPUX: c_int = 0x10;
        const PER_MASK: c_int = 0xff;

        // Flag constants, taken from sys/personality.h
        const UNAME26: c_int = 0x0020000;
        const ADDR_NO_RANDOMIZE: c_int = 0x0040000;
        const FDPIC_FUNCPTRS: c_int = 0x0080000;
        const MMAP_PAGE_ZERO: c_int = 0x0100000;
        const ADDR_COMPAT_LAYOUT: c_int = 0x0200000;
        const READ_IMPLIES_EXEC: c_int = 0x0400000;
        const ADDR_LIMIT_32BIT: c_int = 0x0800000;
        const SHORT_INODE: c_int = 0x1000000;
        const WHOLE_SECONDS: c_int = 0x2000000;
        const STICKY_TIMEOUTS: c_int = 0x4000000;
        const ADDR_LIMIT_3GB: c_int = 0x8000000;

        let domain = match self.0.bits() & PER_MASK {
            PER_LINUX => "linux",
            PER_LINUX_32BIT => "linux_32bit",
            PER_LINUX_FDPIC => "linux_fdpic",
            PER_SVR4 => "svr4",
            PER_SVR3 => "svr3",
            PER_SCOSVR3 => "scosvr3",
            PER_OSR5 => "osr5",
            PER_WYSEV386 => "wysev386",
            PER_ISCR4 => "iscr4",
            PER_BSD => "bsd",
            PER_SUNOS => "sunos",
            PER_XENIX => "xenix",
            PER_LINUX32 => "linux32",
            PER_LINUX32_3GB => "linux32_3gb",
            PER_IRIX32 => "irix32",
            PER_IRIXN32 => "irixn32",
            PER_IRIX64 => "irix64",
            PER_RISCOS => "riscos",
            PER_SOLARIS => "solaris",
            PER_UW7 => "uw7",
            PER_OSF4 => "osf4",
            PER_HPUX => "hpux",
            _ => "unknown",
        };

        let flags = [
            (UNAME26, "uname26"),
            (ADDR_NO_RANDOMIZE, "addr-no-randomize"),
            (FDPIC_FUNCPTRS, "fdpic-funcptrs"),
            (MMAP_PAGE_ZERO, "mmap-page-zero"),
            (ADDR_COMPAT_LAYOUT, "addr-compat-layout"),
            (READ_IMPLIES_EXEC, "read-implies-exec"),
            (ADDR_LIMIT_32BIT, "addr-limit-32bit"),
            (SHORT_INODE, "short-inode"),
            (WHOLE_SECONDS, "whole-seconds"),
            (STICKY_TIMEOUTS, "sticky-timeouts"),
            (ADDR_LIMIT_3GB, "addr-limit-3gb"),
        ]
        .iter()
        .filter_map(|&(flag, name)| {
            if self.0.bits() & flag == flag {
                Some(name)
            } else {
                None
            }
        })
        .collect::<Vec<_>>()
        .join(",");

        if flags.is_empty() {
            write!(f, "{domain}")
        } else {
            write!(f, "{domain},{flags}")
        }
    }
}

/// MS_NOSYMFOLLOW is Linux>=5.10 and not defined by libc yet.
pub const MS_NOSYMFOLLOW: MsFlags = MsFlags::from_bits_retain(256);

/// SydMsFlags wraps MsFlags and provides from_name.
/// This is already available in newer bitflags versions.
pub(crate) struct SydMsFlags(pub(crate) MsFlags);

impl SydMsFlags {
    pub(crate) fn from_name(name: &str) -> Option<Self> {
        match name {
            "ro" => Some(SydMsFlags(MsFlags::MS_RDONLY)),
            "nosuid" => Some(SydMsFlags(MsFlags::MS_NOSUID)),
            "nodev" => Some(SydMsFlags(MsFlags::MS_NODEV)),
            "noexec" => Some(SydMsFlags(MsFlags::MS_NOEXEC)),
            "nosymfollow" => Some(SydMsFlags(MS_NOSYMFOLLOW)),
            "sync" => Some(SydMsFlags(MsFlags::MS_SYNCHRONOUS)),
            "remount" => Some(SydMsFlags(MsFlags::MS_REMOUNT)),
            "mandlock" => Some(SydMsFlags(MsFlags::MS_MANDLOCK)),
            "dirsync" => Some(SydMsFlags(MsFlags::MS_DIRSYNC)),
            "noatime" => Some(SydMsFlags(MsFlags::MS_NOATIME)),
            "nodiratime" => Some(SydMsFlags(MsFlags::MS_NODIRATIME)),
            "bind" => Some(SydMsFlags(MsFlags::MS_BIND)),
            "move" => Some(SydMsFlags(MsFlags::MS_MOVE)),
            "rec" => Some(SydMsFlags(MsFlags::MS_REC)),
            "silent" => Some(SydMsFlags(MsFlags::MS_SILENT)),
            "posixacl" => Some(SydMsFlags(MsFlags::MS_POSIXACL)),
            "unbindable" => Some(SydMsFlags(MsFlags::MS_UNBINDABLE)),
            "private" => Some(SydMsFlags(MsFlags::MS_PRIVATE)),
            "slave" => Some(SydMsFlags(MsFlags::MS_SLAVE)),
            "shared" => Some(SydMsFlags(MsFlags::MS_SHARED)),
            "relatime" => Some(SydMsFlags(MsFlags::MS_RELATIME)),
            "i_version" => Some(SydMsFlags(MsFlags::MS_I_VERSION)),
            "strictatime" => Some(SydMsFlags(MsFlags::MS_STRICTATIME)),
            "lazytime" => Some(SydMsFlags(MsFlags::MS_LAZYTIME)),
            // Deprecated: Should only be used in-kernel.
            //"kernmount" => Some(SydMsFlags(MsFlags::MS_KERNMOUNT)),
            // "active" => Some(SydMsFlags(MsFlags::MS_ACTIVE)),
            // "nouser" => Some(SydMsFlags(MsFlags::MS_NOUSER)),
            _ => None,
        }
    }

    /// Convert MsFlags to a vector of flag names
    #[allow(clippy::cognitive_complexity)]
    pub(crate) fn to_names(&self) -> Vec<&str> {
        let mut names = Vec::new();

        if self.0.contains(MsFlags::MS_RDONLY) {
            names.push("ro");
        }
        if self.0.contains(MsFlags::MS_NOSUID) {
            names.push("nosuid");
        }
        if self.0.contains(MsFlags::MS_NODEV) {
            names.push("nodev");
        }
        if self.0.contains(MsFlags::MS_NOEXEC) {
            names.push("noexec");
        }
        if self.0.contains(MS_NOSYMFOLLOW) {
            names.push("nosymfollow");
        }
        if self.0.contains(MsFlags::MS_SYNCHRONOUS) {
            names.push("sync");
        }
        if self.0.contains(MsFlags::MS_REMOUNT) {
            names.push("remount");
        }
        if self.0.contains(MsFlags::MS_MANDLOCK) {
            names.push("mandlock");
        }
        if self.0.contains(MsFlags::MS_DIRSYNC) {
            names.push("dirsync");
        }
        if self.0.contains(MsFlags::MS_NOATIME) {
            names.push("noatime");
        }
        if self.0.contains(MsFlags::MS_NODIRATIME) {
            names.push("nodiratime");
        }
        if self.0.contains(MsFlags::MS_BIND) {
            names.push("bind");
        }
        if self.0.contains(MsFlags::MS_MOVE) {
            names.push("move");
        }
        if self.0.contains(MsFlags::MS_REC) {
            names.push("rec");
        }
        if self.0.contains(MsFlags::MS_SILENT) {
            names.push("silent");
        }
        if self.0.contains(MsFlags::MS_POSIXACL) {
            names.push("posixacl");
        }
        if self.0.contains(MsFlags::MS_UNBINDABLE) {
            names.push("unbindable");
        }
        if self.0.contains(MsFlags::MS_PRIVATE) {
            names.push("private");
        }
        if self.0.contains(MsFlags::MS_SLAVE) {
            names.push("slave");
        }
        if self.0.contains(MsFlags::MS_SHARED) {
            names.push("shared");
        }
        if self.0.contains(MsFlags::MS_RELATIME) {
            names.push("relatime");
        }
        if self.0.contains(MsFlags::MS_I_VERSION) {
            names.push("i_version");
        }
        if self.0.contains(MsFlags::MS_STRICTATIME) {
            names.push("strictatime");
        }
        if self.0.contains(MsFlags::MS_LAZYTIME) {
            names.push("lazytime");
        }

        names
    }
}

#[inline]
pub(crate) fn op2name(op: u8) -> &'static str {
    match op {
        0x2 => "bind",
        0x3 => "connect",
        0x5 => "accept",
        0xb => "sendto",
        0x10 => "sendmsg",
        0x12 => "accept4",
        0x14 => "sendmmsg",
        _ => unreachable!(),
    }
}

#[inline]
pub(crate) fn op2errno(op: u8) -> Errno {
    match op {
        0x2 /*bind*/ => Errno::EADDRNOTAVAIL,
        0x3 /*connect*/ =>  Errno::ECONNREFUSED,
        0x5 | 0x12 /*accept{,4}*/ => Errno::ECONNABORTED,
        _ /*send{to,{m,}msg}*/ => Errno::ENOTCONN,
    }
}

/// Checks if the given namespaces are enabled.
pub fn ns_enabled(ns_flags: CloneFlags) -> Result<bool, Errno> {
    const SAFE_CLONE_FLAGS: libc::c_int = libc::CLONE_FS
        | libc::CLONE_FILES
        | libc::CLONE_IO
        | libc::CLONE_VM
        | libc::CLONE_VFORK
        | libc::CLONE_SIGHAND;

    // All set, spawn the thread to check unprivileged userns.
    let mut stack = [0u8; crate::config::MINI_STACK_SIZE];
    #[allow(clippy::blocks_in_conditions)]
    let pid_fd = safe_clone(
        Box::new(|| -> isize {
            if unshare(ns_flags).is_ok() {
                0
            } else {
                127
            }
        }),
        &mut stack[..],
        SAFE_CLONE_FLAGS,
        Some(libc::SIGCHLD),
    )?;

    loop {
        break match waitid(Id::PIDFd(pid_fd.as_fd()), WaitPidFlag::WEXITED) {
            Ok(crate::compat::WaitStatus::Exited(_, 0)) => Ok(true),
            Ok(_) => Ok(false),
            Err(Errno::EINTR) => continue,
            Err(errno) => Err(errno),
        };
    }
}

/// Checks if the given LandLock ABI is supported.
/// Returns:
/// - 0: Fully enforced
/// - 1: Partially enforced
/// - 2: Not enforced
/// - 127: Unsupported
pub fn lock_enabled(abi: ABI) -> u8 {
    let path_ro = vec![XPathBuf::from("/")];
    let path_rw = vec![XPathBuf::from("/")];
    // Landlock network is ABI>=4.
    let port_if = if abi as i32 >= ABI::V4 as i32 {
        Some((2525, 22))
    } else {
        None
    };

    // A helper function to wrap the operations and reduce duplication
    #[allow(clippy::disallowed_methods)]
    fn landlock_operation(
        abi: ABI,
        path_ro: &[XPathBuf],
        path_rw: &[XPathBuf],
        port_if: Option<(u16, u16)>,
    ) -> Result<RestrictionStatus, RulesetError> {
        // from_all includes IoctlDev of ABI >= 5 as necessary.
        let mut ruleset = Ruleset::default().handle_access(AccessFs::from_all(abi))?;
        let ruleset_ref = &mut ruleset;

        let mut network_rules: Vec<Result<NetPort, RulesetError>> = vec![];
        if let Some((port_bind, port_conn)) = port_if {
            ruleset_ref.handle_access(AccessNet::BindTcp)?;
            network_rules.push(Ok(NetPort::new(port_bind, AccessNet::BindTcp)));

            ruleset_ref.handle_access(AccessNet::ConnectTcp)?;
            network_rules.push(Ok(NetPort::new(port_conn, AccessNet::ConnectTcp)));
        }

        // Landlock network is ABI>=6.
        if abi as i32 >= ABI::V6 as i32 {
            ruleset_ref.scope(Scope::AbstractUnixSocket)?;
            ruleset_ref.scope(Scope::Signal)?;
        }

        ruleset
            .create()?
            .add_rules(path_beneath_rules(path_ro, AccessFs::from_read(abi)))?
            .add_rules(path_beneath_rules(path_rw, AccessFs::from_all(abi)))?
            .add_rules(network_rules)?
            .restrict_self(RestrictSelfFlags::empty())
    }

    match landlock_operation(abi, &path_ro, &path_rw, port_if) {
        Ok(status) => match status.ruleset {
            RulesetStatus::FullyEnforced => 0,
            RulesetStatus::PartiallyEnforced => 1,
            RulesetStatus::NotEnforced => 2,
        },
        Err(_) => 127,
    }
}

/// Returns true if we are running under syd.
#[allow(clippy::disallowed_methods)]
pub fn syd_enabled() -> bool {
    // This will not work if the sandbox is locked.
    // Path::new("/dev/syd").exists() || Path::new("/dev/syd").exists()
    // SAFETY: In libc, we trust.
    match unsafe { fork() } {
        Ok(ForkResult::Parent { child, .. }) => {
            match waitpid(child, None) {
                Ok(nix::sys::wait::WaitStatus::Exited(_, code)) => {
                    // Check the child's exit status.
                    // Exit status of 0 means syd is enabled.
                    code == 0
                }
                _ => {
                    // If there's an error waiting on the
                    // child, assume syd is not enabled.
                    false
                }
            }
        }
        Ok(ForkResult::Child) => {
            let mut ctx = match ScmpFilterContext::new(ScmpAction::Allow) {
                Ok(ctx) => ctx,
                Err(_) => exit(1),
            };

            let syscall = ScmpSyscall::from_name("open").unwrap();
            if ctx.add_rule(ScmpAction::Notify, syscall).is_err() {
                exit(1);
            }

            if ctx.load().is_err() && Errno::last() == Errno::EBUSY {
                // seccomp filter exists
                // syd is in business.
                exit(0);
            } else {
                // seccomp filter does not exist
                exit(1);
            }
        }
        Err(_) => {
            // If there's an error forking,
            // assume syd is not enabled.
            false
        }
    }
}

/// Returns the name of the libsecc☮mp native architecture.
pub(crate) fn seccomp_arch_native_name() -> Option<&'static str> {
    match ScmpArch::native() {
        ScmpArch::X86 => Some("x86"),
        ScmpArch::X8664 => Some("x86_64"),
        ScmpArch::X32 => Some("x32"),
        ScmpArch::Arm => Some("arm"),
        ScmpArch::Aarch64 => Some("aarch64"),
        ScmpArch::Loongarch64 => Some("loongarch64"),
        ScmpArch::M68k => Some("m68k"),
        ScmpArch::Mips => Some("mips"),
        ScmpArch::Mips64 => Some("mips64"),
        ScmpArch::Mips64N32 => Some("mips64n32"),
        ScmpArch::Mipsel => Some("mipsel"),
        ScmpArch::Mipsel64 => Some("mipsel64"),
        ScmpArch::Mipsel64N32 => Some("mipsel64n32"),
        ScmpArch::Ppc => Some("ppc"),
        ScmpArch::Ppc64 => Some("ppc64"),
        ScmpArch::Ppc64Le => Some("ppc64le"),
        ScmpArch::S390 => Some("s390"),
        ScmpArch::S390X => Some("s390x"),
        ScmpArch::Parisc => Some("parisc"),
        ScmpArch::Parisc64 => Some("parisc64"),
        ScmpArch::Riscv64 => Some("riscv64"),
        ScmpArch::Sheb => Some("sheb"),
        ScmpArch::Sh => Some("sh"),
        _ => None,
    }
}

const SECCOMP_ARCH_LIST: &[ScmpArch] = &[
    ScmpArch::X86,
    ScmpArch::X8664,
    ScmpArch::X32,
    ScmpArch::Arm,
    ScmpArch::Aarch64,
    ScmpArch::Loongarch64,
    ScmpArch::M68k,
    ScmpArch::Mips,
    ScmpArch::Mips64,
    ScmpArch::Mips64N32,
    ScmpArch::Mipsel,
    ScmpArch::Mipsel64,
    ScmpArch::Mipsel64N32,
    ScmpArch::Ppc,
    ScmpArch::Ppc64,
    ScmpArch::Ppc64Le,
    ScmpArch::S390,
    ScmpArch::S390X,
    ScmpArch::Parisc,
    ScmpArch::Parisc64,
    ScmpArch::Riscv64,
    ScmpArch::Sheb,
    ScmpArch::Sh,
];

/// Print list of libseccomp's supported architectures
/// Used by `syd --arch list`
pub fn print_seccomp_architectures() {
    let native = ScmpArch::native();
    for arch in SECCOMP_ARCH_LIST {
        let mut repr = format!("{arch:?}").to_ascii_lowercase();
        if repr == "x8664" {
            // Fix potential confusion.
            repr = "x86_64".to_string();
        }
        if *arch == native {
            println!("- {repr} [*]")
        } else {
            println!("- {repr}");
        }
    }
}

// x32 bit for arch-specific syscalls.
pub(crate) const X32_SYSCALL_BIT: i32 = 0x4000_0000;

// List of libseccomp supported architectures for the current system.
#[cfg(all(target_arch = "x86_64", target_pointer_width = "64",))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::X8664, ScmpArch::X86, ScmpArch::X32];
#[cfg(all(target_arch = "x86_64", target_pointer_width = "32",))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::X32, ScmpArch::X86];
#[cfg(target_arch = "x86")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::X86];
#[cfg(target_arch = "arm")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Arm];
#[cfg(target_arch = "aarch64")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Aarch64, ScmpArch::Arm];
#[cfg(target_arch = "m68k")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::M68k];
#[cfg(all(target_arch = "mips", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mips];
#[cfg(all(target_arch = "mips", target_endian = "little"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mipsel];
#[cfg(all(target_arch = "mips32r6", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mips];
#[cfg(all(target_arch = "mips32r6", target_endian = "little"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mipsel];
#[cfg(all(target_arch = "mips64", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mips64, ScmpArch::Mips64N32, ScmpArch::Mips];
#[cfg(all(target_arch = "mips64", target_endian = "little"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] =
    &[ScmpArch::Mipsel64, ScmpArch::Mipsel64N32, ScmpArch::Mipsel];
#[cfg(all(target_arch = "mips64r6", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Mips64, ScmpArch::Mips64N32, ScmpArch::Mips];
#[cfg(all(target_arch = "mips64r6", target_endian = "little"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] =
    &[ScmpArch::Mipsel64, ScmpArch::Mipsel64N32, ScmpArch::Mipsel];
#[cfg(all(target_arch = "powerpc", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Ppc];
#[cfg(all(target_arch = "powerpc64", target_endian = "big"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Ppc64, ScmpArch::Ppc];
#[cfg(all(target_arch = "powerpc64", target_endian = "little"))]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Ppc64Le];
//#[cfg(target_arch = "parisc")]
//pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Parisc];
//#[cfg(target_arch = "parisc64")]
//pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Parisc64, ScmpArch::Parisc];
#[cfg(target_arch = "riscv64")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Riscv64];
#[cfg(target_arch = "s390x")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::S390X, ScmpArch::S390];
#[cfg(target_arch = "loongarch64")]
pub(crate) const SCMP_ARCH: &[ScmpArch] = &[ScmpArch::Loongarch64];

/// List of architectures with the ipc(2) multiplexer system call.
pub(crate) const IPC_ARCH: &[ScmpArch] = &[
    ScmpArch::X86,
    ScmpArch::M68k,
    ScmpArch::Mips,
    ScmpArch::Mipsel,
    ScmpArch::Ppc,
    ScmpArch::Ppc64,
    ScmpArch::Ppc64Le,
    ScmpArch::S390X,
    ScmpArch::S390,
    ScmpArch::Sheb,
    ScmpArch::Sh,
];

/// Return true if native architecture has the multiplexed ipc(2) system call.
///
/// Panics if it cannot determine the native architecture.
pub fn seccomp_native_has_ipc() -> bool {
    IPC_ARCH.contains(&ScmpArch::native())
}

/// Return true if native architecture has the multiplexed socketcall(2) system call.
///
/// Panics if it cannot determine the native architecture.
pub fn seccomp_native_has_socketcall() -> bool {
    matches!(
        ScmpArch::native(),
        ScmpArch::X86
            | ScmpArch::M68k
            | ScmpArch::Mips
            | ScmpArch::Mipsel
            | ScmpArch::Ppc
            | ScmpArch::Ppc64
            | ScmpArch::Ppc64Le
            | ScmpArch::S390
            | ScmpArch::S390X
    )
}

/// Add all supported architectures to the given filter.
pub fn seccomp_add_architectures(ctx: &mut ScmpFilterContext) -> SydResult<()> {
    // Add architectures based on the current architecture
    for arch in SCMP_ARCH {
        seccomp_add_arch(ctx, *arch)?;
    }
    Ok(())
}

fn seccomp_add_arch(ctx: &mut ScmpFilterContext, arch: ScmpArch) -> SydResult<()> {
    Ok(ctx.add_arch(arch).map(drop)?)
}

/// Check if arch is 64-bit or 32-bit.
#[inline]
pub const fn scmp_arch_bits(arch: ScmpArch) -> usize {
    match arch {
        ScmpArch::X8664
        | ScmpArch::X32
        | ScmpArch::Aarch64
        | ScmpArch::Loongarch64
        | ScmpArch::Mips64
        | ScmpArch::Mips64N32
        | ScmpArch::Mipsel64
        | ScmpArch::Mipsel64N32
        | ScmpArch::Ppc64
        | ScmpArch::Ppc64Le
        | ScmpArch::Parisc64
        | ScmpArch::Riscv64
        | ScmpArch::S390X => 64,
        ScmpArch::X86
        | ScmpArch::Arm
        | ScmpArch::M68k
        | ScmpArch::Mips
        | ScmpArch::Mipsel
        | ScmpArch::Ppc
        | ScmpArch::Parisc
        | ScmpArch::S390
        | ScmpArch::Sheb
        | ScmpArch::Sh => 32,
        _ => 64, // sane default for non-exhaustive enum.
    }
}

/// Helper function to determine if the architecture is big-endian.
#[inline]
pub fn scmp_big_endian(arch: ScmpArch) -> bool {
    matches!(
        arch,
        ScmpArch::Mips
            | ScmpArch::Mips64
            | ScmpArch::Ppc
            | ScmpArch::Ppc64
            | ScmpArch::S390
            | ScmpArch::S390X
            | ScmpArch::Parisc
            | ScmpArch::Parisc64
    )
}

/// Represents seccomp notify data.
/// We redefine this because libseccomp struct is non-exhaustive.
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub(crate) struct ScmpNotifData {
    pub(crate) syscall: ScmpSyscall,
    pub(crate) arch: ScmpArch,
    pub(crate) instr_pointer: u64,
    pub(crate) args: [u64; 6],
}

/// Represents a seccomp notify request.
/// We redefine this because libseccomp struct is non-exhaustive.
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub struct ScmpNotifReq {
    pub(crate) id: u64,
    pub(crate) pid: u32,
    pub(crate) flags: u32,
    pub(crate) data: ScmpNotifData,
}

impl ScmpNotifData {
    fn from_sys(data: libc::seccomp_data) -> Result<Self, Errno> {
        Ok(Self {
            syscall: ScmpSyscall::from(data.nr),
            arch: scmp_arch(data.arch)?,
            instr_pointer: data.instruction_pointer,
            args: data.args,
        })
    }
}

impl ScmpNotifReq {
    pub(crate) fn from_sys(req: libc::seccomp_notif) -> Result<Self, Errno> {
        Ok(Self {
            id: req.id,
            pid: req.pid,
            flags: req.flags,
            data: ScmpNotifData::from_sys(req.data)?,
        })
    }

    #[inline(always)]
    pub(crate) fn pid(&self) -> Pid {
        #[allow(clippy::cast_possible_wrap)]
        Pid::from_raw(self.pid as libc::pid_t)
    }
}

/// Helper function to convert raw arch value to ScmpArch.
///
/// We need this because ScmpArch::from_sys is not imported.
pub const fn scmp_arch(arch: u32) -> Result<ScmpArch, Errno> {
    match arch {
        libseccomp_sys::SCMP_ARCH_NATIVE => Ok(ScmpArch::Native),
        libseccomp_sys::SCMP_ARCH_X86 => Ok(ScmpArch::X86),
        libseccomp_sys::SCMP_ARCH_X86_64 => Ok(ScmpArch::X8664),
        libseccomp_sys::SCMP_ARCH_X32 => Ok(ScmpArch::X32),
        libseccomp_sys::SCMP_ARCH_ARM => Ok(ScmpArch::Arm),
        libseccomp_sys::SCMP_ARCH_AARCH64 => Ok(ScmpArch::Aarch64),
        libseccomp_sys::SCMP_ARCH_LOONGARCH64 => Ok(ScmpArch::Loongarch64),
        libseccomp_sys::SCMP_ARCH_M68K => Ok(ScmpArch::M68k),
        libseccomp_sys::SCMP_ARCH_MIPS => Ok(ScmpArch::Mips),
        libseccomp_sys::SCMP_ARCH_MIPS64 => Ok(ScmpArch::Mips64),
        libseccomp_sys::SCMP_ARCH_MIPS64N32 => Ok(ScmpArch::Mips64N32),
        libseccomp_sys::SCMP_ARCH_MIPSEL => Ok(ScmpArch::Mipsel),
        libseccomp_sys::SCMP_ARCH_MIPSEL64 => Ok(ScmpArch::Mipsel64),
        libseccomp_sys::SCMP_ARCH_MIPSEL64N32 => Ok(ScmpArch::Mipsel64N32),
        libseccomp_sys::SCMP_ARCH_PPC => Ok(ScmpArch::Ppc),
        libseccomp_sys::SCMP_ARCH_PPC64 => Ok(ScmpArch::Ppc64),
        libseccomp_sys::SCMP_ARCH_PPC64LE => Ok(ScmpArch::Ppc64Le),
        libseccomp_sys::SCMP_ARCH_S390 => Ok(ScmpArch::S390),
        libseccomp_sys::SCMP_ARCH_S390X => Ok(ScmpArch::S390X),
        libseccomp_sys::SCMP_ARCH_PARISC => Ok(ScmpArch::Parisc),
        libseccomp_sys::SCMP_ARCH_PARISC64 => Ok(ScmpArch::Parisc64),
        libseccomp_sys::SCMP_ARCH_RISCV64 => Ok(ScmpArch::Riscv64),
        libseccomp_sys::SCMP_ARCH_SHEB => Ok(ScmpArch::Sheb),
        libseccomp_sys::SCMP_ARCH_SH => Ok(ScmpArch::Sh),
        _ => Err(Errno::ENOSYS),
    }
}

/// Helper function to convert ScmpArch to raw arch values.
///
/// We need this because ScmpArch::from_sys is not imported.
/// This function panics on invalid/unsupported architecture.
pub const fn scmp_arch_raw(arch: ScmpArch) -> u32 {
    match arch {
        ScmpArch::Native => libseccomp_sys::SCMP_ARCH_NATIVE,
        ScmpArch::X86 => libseccomp_sys::SCMP_ARCH_X86,
        ScmpArch::X8664 => libseccomp_sys::SCMP_ARCH_X86_64,
        ScmpArch::X32 => libseccomp_sys::SCMP_ARCH_X32,
        ScmpArch::Arm => libseccomp_sys::SCMP_ARCH_ARM,
        ScmpArch::Aarch64 => libseccomp_sys::SCMP_ARCH_AARCH64,
        ScmpArch::Loongarch64 => libseccomp_sys::SCMP_ARCH_LOONGARCH64,
        ScmpArch::M68k => libseccomp_sys::SCMP_ARCH_M68K,
        ScmpArch::Mips => libseccomp_sys::SCMP_ARCH_MIPS,
        ScmpArch::Mips64 => libseccomp_sys::SCMP_ARCH_MIPS64,
        ScmpArch::Mips64N32 => libseccomp_sys::SCMP_ARCH_MIPS64N32,
        ScmpArch::Mipsel => libseccomp_sys::SCMP_ARCH_MIPSEL,
        ScmpArch::Mipsel64 => libseccomp_sys::SCMP_ARCH_MIPSEL64,
        ScmpArch::Mipsel64N32 => libseccomp_sys::SCMP_ARCH_MIPSEL64N32,
        ScmpArch::Ppc => libseccomp_sys::SCMP_ARCH_PPC,
        ScmpArch::Ppc64 => libseccomp_sys::SCMP_ARCH_PPC64,
        ScmpArch::Ppc64Le => libseccomp_sys::SCMP_ARCH_PPC64LE,
        ScmpArch::S390 => libseccomp_sys::SCMP_ARCH_S390,
        ScmpArch::S390X => libseccomp_sys::SCMP_ARCH_S390X,
        ScmpArch::Parisc => libseccomp_sys::SCMP_ARCH_PARISC,
        ScmpArch::Parisc64 => libseccomp_sys::SCMP_ARCH_PARISC64,
        ScmpArch::Riscv64 => libseccomp_sys::SCMP_ARCH_RISCV64,
        ScmpArch::Sheb => libseccomp_sys::SCMP_ARCH_SHEB,
        ScmpArch::Sh => libseccomp_sys::SCMP_ARCH_SH,
        _ => unreachable!(),
    }
}

/// Confine creation of the given file type using mknod(2) and mknodat(2).
#[allow(clippy::cognitive_complexity)]
pub(crate) fn scmp_add_mknod(
    ctx: &mut ScmpFilterContext,
    action: ScmpAction,
    f_type: FileType,
) -> SydResult<()> {
    const S_IFMT: u64 = libc::S_IFMT as u64;
    let f_type = u64::from(f_type.mode().ok_or(Errno::EINVAL)?);

    let sysname = "mknod";
    if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
        ctx.add_rule_conditional(action, syscall, &[scmp_cmp!($arg1 & S_IFMT == f_type)])?;
    } else {
        info!("ctx": "confine", "op": "deny_syscall",
            "msg": format!("invalid or unsupported syscall {sysname}"));
    }

    let sysname = "mknodat";
    if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
        ctx.add_rule_conditional(action, syscall, &[scmp_cmp!($arg2 & S_IFMT == f_type)])?;
    } else {
        info!("ctx": "confine", "op": "deny_syscall",
            "msg": format!("invalid or unsupported syscall {sysname}"));
    }

    Ok(())
}

/// Add UID/GID change rules for SafeSetId.
#[allow(clippy::cognitive_complexity)]
pub(crate) fn scmp_add_setid_rules(
    tag: &str,
    ctx: &mut ScmpFilterContext,
    safe_setuid: bool,
    safe_setgid: bool,
    transit_uids: &[(Uid, Uid)],
    transit_gids: &[(Gid, Gid)],
) -> SydResult<()> {
    const NULL_ID: u64 = u64::MAX;
    let op_a = format!("allow_{tag}_syscall");
    let op_f = format!("filter_{tag}_syscall");

    // SAFETY: Signal system calls are necessary to handle reserved signals.
    for sysname in ["sigreturn", "rt_sigreturn"] {
        match ScmpSyscall::from_name(sysname) {
            Ok(syscall) => {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            }
            Err(_) => {
                info!("ctx": "confine", "op": &op_a,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }
    }

    // SAFETY: Only allow defined UID transitions.
    if safe_setuid {
        let source_uid = Uid::current();

        for sysname in &["setuid", "setuid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_uid, t_uid) in transit_uids {
                    if source_uid == *s_uid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[scmp_cmp!($arg0 == u64::from(t_uid.as_raw()))],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        for sysname in &["setreuid", "setreuid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_uid, t_uid) in transit_uids {
                    if source_uid == *s_uid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                            ],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        for sysname in &["setresuid", "setresuid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_uid, t_uid) in transit_uids {
                    if source_uid == *s_uid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg2 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg2 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == u64::from(t_uid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_uid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }
    }

    // SAFETY: Only allow defined GID transitions.
    if safe_setgid {
        let source_gid = Gid::current();

        for sysname in &["setgid", "setgid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_gid, t_gid) in transit_gids {
                    if source_gid == *s_gid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[scmp_cmp!($arg0 == u64::from(t_gid.as_raw()))],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        for sysname in &["setregid", "setregid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_gid, t_gid) in transit_gids {
                    if source_gid == *s_gid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                            ],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        for sysname in &["setresgid", "setresgid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                for (s_gid, t_gid) in transit_gids {
                    if source_gid == *s_gid {
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg2 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg2 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == u64::from(t_gid.as_raw())),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == NULL_ID),
                                scmp_cmp!($arg1 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                        ctx.add_rule_conditional(
                            ScmpAction::Allow,
                            syscall,
                            &[
                                scmp_cmp!($arg0 == u64::from(t_gid.as_raw())),
                                scmp_cmp!($arg1 == NULL_ID),
                                scmp_cmp!($arg2 == NULL_ID),
                            ],
                        )?;
                    }
                }
            } else {
                info!("ctx": "confine", "op": &op_f,
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }
    }

    Ok(())
}

/// CLONE_NEWTIME constant to create time namespaces.
pub const CLONE_NEWTIME: CloneFlags = CloneFlags::from_bits_retain(128);

pub(crate) const NAMESPACE_FLAGS: &[libc::c_int] = &[
    libc::CLONE_NEWNS,
    libc::CLONE_NEWIPC,
    libc::CLONE_NEWNET,
    libc::CLONE_NEWPID,
    libc::CLONE_NEWUTS,
    libc::CLONE_NEWUSER,
    libc::CLONE_NEWCGROUP,
    CLONE_NEWTIME.bits(),
];

pub(crate) const NAMESPACE_FLAGS_ALL: libc::c_int = libc::CLONE_NEWNS
    | libc::CLONE_NEWIPC
    | libc::CLONE_NEWNET
    | libc::CLONE_NEWPID
    | libc::CLONE_NEWUTS
    | libc::CLONE_NEWUSER
    | libc::CLONE_NEWCGROUP
    | CLONE_NEWTIME.bits();

pub(crate) const NAMESPACE_NAMES: &[&str] = &[
    "mount", "ipc", "net", "pid", "uts", "user", "cgroup", "time",
];

/// Convert CLONE namespace flags to a Vector of Strings.
pub fn nsflags_name(flags: libc::c_int) -> Vec<String> {
    let mut names = Vec::with_capacity(NAMESPACE_FLAGS.len());
    for &flag in NAMESPACE_FLAGS {
        if flags & flag != 0 {
            names.push(nsflag_name(flag));
        }
    }

    names
}

/// Convert a CLONE namespace flag to its String representation.
pub fn nsflag_name(flag: libc::c_int) -> String {
    match flag {
        libc::CLONE_NEWNS => "mount",
        libc::CLONE_NEWIPC => "ipc",
        libc::CLONE_NEWNET => "net",
        libc::CLONE_NEWPID => "pid",
        libc::CLONE_NEWUTS => "uts",
        libc::CLONE_NEWUSER => "user",
        libc::CLONE_NEWCGROUP => "cgroup",
        n if n == CLONE_NEWTIME.bits() => "time",
        _ => "?",
    }
    .to_string()
}

/// Check for file descriptor leaks above the standard input, output, and error.
///
/// This function examines the `/proc/self/fd` directory to identify
/// open file descriptors. It prints any open file descriptors other
/// than the standard input (0), output (1), and error (2), indicating
/// potential resource leaks.
///
/// # Parameters
/// - `fd_max`: An optional parameter that sets a maximum file
///   descriptor number to check. If not specified, only the standard
///   file descriptors are considered normal.
///
/// # Returns
/// Returns `true` if leaks are found, otherwise `false`.
pub fn check_fd_leaks(fd_max: Option<RawFd>) -> u32 {
    let proc_fd_path = Path::new("/proc/self/fd");
    let mut dir = match Dir::open(proc_fd_path, OFlag::O_RDONLY, Mode::empty()) {
        Ok(d) => d,
        Err(e) => {
            eprintln!("Failed to open /proc/self/fd: {e}");
            return u32::MAX;
        }
    };

    let mut leaks_found: u32 = 0;
    let dir_fd = dir.as_raw_fd();
    let fd_limit = fd_max.unwrap_or(2); // Default limit only std fds

    for entry in dir.iter() {
        let entry = match entry {
            Ok(e) => e,
            Err(_) => continue,
        };

        let fd_str = entry.file_name().to_string_lossy(); // Use lossy conversion
        let fd = match fd_str.parse::<RawFd>() {
            Ok(fd) => fd,
            Err(_) => continue,
        };

        // Ignore standard file descriptors and the directory stream FD itself
        if fd <= fd_limit || fd == dir_fd {
            continue;
        }

        // Create a PathBuf from the string representation of the file descriptor
        let link_path = proc_fd_path.join(fd_str.into_owned()); // Convert Cow<str> into a String and then into a PathBuf

        #[allow(clippy::disallowed_methods)]
        match std::fs::read_link(&link_path) {
            Ok(target_path) => {
                eprintln!("!!! Leaked file descriptor {fd} -> {target_path:?} !!!");
                leaks_found = leaks_found.saturating_add(1);
            }
            Err(error) => {
                eprintln!("Failed to read link for FD {fd}: {error}");
            }
        }
    }

    leaks_found
}

/// Print list of file descriptors to standard error.
pub fn list_fds(pid: Option<Pid>) {
    let mut path = match pid {
        Some(pid) => XPathBuf::from(format!("/proc/{}/fd", pid.as_raw())),
        None => XPathBuf::from("/proc/self/fd"),
    };

    let mut dir = match Dir::open(&path, OFlag::O_RDONLY, Mode::empty()) {
        Ok(dir) => dir,
        Err(errno) => {
            eprintln!("list_fds: Failed to open {path}: {errno}");
            return;
        }
    };

    // Header
    eprintln!(
        "list_fds: {}",
        pid.map(|p| p.as_raw().to_string())
            .unwrap_or_else(|| "self".to_string())
    );
    eprintln!("fd\ttarget");

    let dfd = dir.as_raw_fd();
    for entry in dir.iter() {
        let entry = match entry {
            Ok(entry) => entry,
            Err(_) => continue,
        };

        let fd = match btoi::<RawFd>(entry.file_name().to_bytes()) {
            Ok(fd) => fd,
            Err(_) => continue,
        };

        // Skip our dir FD.
        if fd == dfd {
            continue;
        }

        path.push_fd(fd);
        match readlinkat(AT_BADFD, &path) {
            Ok(target) => eprintln!("{fd}\t{target}"),
            Err(errno) => eprintln!("{fd}\t!!! {errno}"),
        }
        path.pop();
    }
}

/// Extends the ioctl value if necessary.
///
/// In musl, ioctl is defined as:
/// `int ioctl(int fd, int req, ...);`
///
/// In glibc, ioctl is defined as:
/// `int ioctl(int fd, unsigned long request, ...);`
///
/// This difference can cause issues when handling ioctl values that are
/// larger than what a signed 32-bit integer can represent.
/// Specifically, values with the high bit set (0x80000000) or the next
/// highest bit set (0x40000000) can be interpreted differently
/// depending on the implementation.
///
/// In a 32-bit signed integer, the high bit (0x80000000) is used as the
/// sign bit, indicating whether the number is positive or negative. If
/// this bit is set, the number is interpreted as negative. The next
/// highest bit (0x40000000) is the largest value that a signed 32-bit
/// integer can represent without becoming negative.
///
/// Therefore, ioctl values that have either of these bits set can cause
/// compatibility issues between musl and glibc. To ensure
/// compatibility, we need to extend such ioctl values to 64 bits by
/// prefixing them with `0xffffffff`, converting them to their unsigned
/// representation.
///
/// # Arguments
///
/// * `value` - The original ioctl value.
///
/// # Returns
///
/// * `Some(extended_value)` - If the value requires extension.
/// * `None` - If the value does not require extension.
#[inline]
pub fn extend_ioctl(value: u64) -> Option<u64> {
    // Check if the high bit (0x80000000) or the next highest bit
    // (0x40000000) is set.  These bits can cause the value to be
    // interpreted as a negative number in a signed 32-bit context.
    if (value & 0x80000000 == 0x80000000) || (value & 0x40000000 == 0x40000000) {
        // If the value requires extension, return the extended value by
        // prefixing with `0xffffffff`.
        Some(0xffffffff00000000 | value)
    } else {
        // If the value does not require extension, return None.
        None
    }
}

/// Drop a Capability from the Effective, Ambient, Inheritable and Permitted capsets.
pub fn safe_drop_cap(cap: caps::Capability) -> Result<(), caps::errors::CapsError> {
    caps::drop(None, caps::CapSet::Effective, cap)?;
    caps::drop(None, caps::CapSet::Ambient, cap)?;
    caps::drop(None, caps::CapSet::Inheritable, cap)?;
    caps::drop(None, caps::CapSet::Permitted, cap)
}

/// Return true if the given signal has default action Core.
#[inline]
#[allow(unreachable_patterns)]
pub(crate) fn is_coredump(sig: i32) -> bool {
    matches!(
        sig,
        libc::SIGABRT
            | libc::SIGBUS
            | libc::SIGFPE
            | libc::SIGILL
            | libc::SIGIOT
            | libc::SIGKILL
            | libc::SIGQUIT
            | libc::SIGSEGV
            | libc::SIGSYS
            | libc::SIGTRAP
            | libc::SIGXCPU
            | libc::SIGXFSZ
    )
}

/// Seccomp sandbox profile export modes.
#[derive(Debug, Clone, Copy, Eq, PartialEq)]
pub enum ExportMode {
    /// Berkeley Packet Filter (binary, machine readable)
    BerkeleyPacketFilter,
    /// Pseudo Filter Code (text, human readable)
    PseudoFiltercode,
}

impl FromStr for ExportMode {
    type Err = Errno;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.to_ascii_lowercase().as_str() {
            "bpf" => Ok(Self::BerkeleyPacketFilter),
            "pfc" => Ok(Self::PseudoFiltercode),
            _ => Err(Errno::EINVAL),
        }
    }
}

impl ExportMode {
    /// Return the export mode specified by the environment.
    #[allow(clippy::disallowed_methods)]
    pub fn from_env() -> Option<ExportMode> {
        Self::from_str(&std::env::var(crate::config::ENV_DUMP_SCMP).ok()?).ok()
    }
}

#[cfg(target_arch = "x86")]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "mov eax, 0x2", // 0x2 is the syscall number for fork on x86
        "int 0x80",     // Interrupt to make the syscall
        out("eax") _,
    );
}

#[cfg(target_arch = "x86_64")]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    // Inline assembly for x86-64
    asm!(
        "mov rax, 57", // 57 is the syscall number for fork on x86-64
        "syscall",
        out("rax") _,
    );
}

#[cfg(target_arch = "aarch64")]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "mov x0, 17",  // SIGCHLD
        "mov x1, 0",   // child_stack (null, not recommended)
        "mov x8, 220", // syscall number for clone
        "svc 0",
        options(nostack),
    );
}

#[cfg(target_arch = "arm")]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "mov r7, #2", // 2 is the syscall number for fork on ARM
        "swi #0",     // Software interrupt to make the syscall
        out("r0") _,
        options(nostack),
    );
}

/*
 * error[E0658]: inline assembly is not stable yet on this architecture
#[cfg(any(target_arch = "powerpc", target_arch = "powerpc64"))]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "li 0, 2",     // Load immediate 2 into register r0 (syscall number for fork)
        "sc",          // System call
        out("r3") _,   // Output from r3 (return value of fork)
    );
}
*/

#[cfg(target_arch = "riscv64")]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "li a7, 220",  // syscall number for clone on riscv64
        "li a0, 17",   // SIGCHLD
        "li a1, 0",    // child_stack (null, not recommended)
        "ecall",       // make the syscall
        out("a0") _,   // store return value in a0
        options(nostack),
    );
}

/*
 * error[E0658]: inline assembly is not stable yet on this architecture
#[cfg(any(target_arch = "s390x"))]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    asm!(
        "lgr %r1, 2", // Load syscall number for fork (2) directly into %r1.
        "svc 0",      // Supervisor Call to invoke the syscall.
    );
}
*/

#[cfg(any(
    target_arch = "powerpc",
    target_arch = "powerpc64",
    target_arch = "s390x"
))]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    let _ = libc::syscall(libc::SYS_fork);
}

#[cfg(not(any(
    target_arch = "aarch64",
    target_arch = "arm",
    target_arch = "powerpc",
    target_arch = "powerpc64",
    target_arch = "riscv64",
    target_arch = "riscv64",
    target_arch = "s390x",
    target_arch = "x86",
    target_arch = "x86_64",
)))]
#[inline(always)]
/// Fork fast.
///
/// # Safety
///
/// Unsafe to be fast!
pub unsafe fn fork_fast() {
    let _ = fork();
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_extend_ioctl() {
        const IOCTLS: &[(u64, bool)] = &[
            (0x5451, false),
            (0x5450, false),
            (0x541B, false),
            (0x5421, false),
            (0x5452, false),
            (0x4B66, false),
            (0x5401, false),
            (0x5402, false),
            (0x5403, false),
            (0x5404, false),
            (0x5405, false),
            (0x5406, false),
            (0x5407, false),
            (0x5408, false),
            (0x5456, false),
            (0x5457, false),
            (0x5413, false),
            (0x5414, false),
            (0x5409, false),
            (0x5425, false),
            (0x5427, false),
            (0x5428, false),
            (0x540A, false),
            (0x5411, false),
            (0x540B, false),
            (0x80045430, true),
            (0x80045432, true),
            (0x5432, false),
            (0x5433, false),
            (0x5434, false),
            (0x5435, false),
            (0x40045436, true),
            (0x5437, false),
            (0x80045438, true),
            (0x80045439, true),
            (0x80045440, true),
            (0x5441, false),
            (0x540E, false),
            (0x540F, false),
            (0x5410, false),
            (0x5429, false),
            (0x540C, false),
            (0x80045440, true),
            (0x540D, false),
            (0x5424, false),
            (0x5423, false),
            (0x5420, false),
            (0x80045438, true),
            (0x40045431, true),
            (0x80045439, true),
            (0x5441, false),
            (0x80086601, true),
            (0x5419, false),
            (0x541A, false),
            (0x8910, false),
            (0x8912, false),
            (0x8913, false),
            (0x8915, false),
            (0x8917, false),
            (0x8919, false),
            (0x891b, false),
            (0x891d, false),
            (0x891f, false),
            (0x892, false),
            (0x8925, false),
            (0x8927, false),
            (0x8929, false),
            (0x8933, false),
            (0x8935, false),
            (0x8938, false),
            (0x8940, false),
            (0x8942, false),
            (0x8947, false),
            (0x8948, false),
            (0x894C, false),
            (0x2400, false),
            (0x2401, false),
            (0x2402, false),
            (0x2403, false),
            (0x2405, false),
            (0x40082404, true),
            (0x40082406, true),
            (0x80082407, true),
            (0x40042408, true),
            (0x40042409, true),
            (0xc008240a, true),
            (0x4008240b, true),
        ];

        for (request, extend) in IOCTLS.iter() {
            if *extend {
                assert!(
                    extend_ioctl(*request).is_some(),
                    "OOPS: {request}->{extend}"
                );
            } else {
                assert!(
                    extend_ioctl(*request).is_none(),
                    "OOPS: {request}->{extend}"
                );
            }
        }
    }
}
