import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import React, { useCallback, useEffect, useMemo, useReducer } from 'react';
import { createPortal } from 'react-dom';
import { combine } from '@atlaskit/pragmatic-drag-and-drop/combine';
import { monitorForElements } from '@atlaskit/pragmatic-drag-and-drop/element/adapter';
import { getHiddenTextElementId } from '../drag-drop-context/hooks/use-hidden-text-element';
import { useDragDropContext } from '../drag-drop-context/internal-context';
import { useMonitorForLifecycle } from '../drag-drop-context/lifecycle-context';
import { rbdInvariant } from '../drag-drop-context/rbd-invariant';
import { isDraggableData } from '../draggable/data';
import { getDraggableProvidedStyle } from '../draggable/get-draggable-provided-style';
import { idleState, reducer } from '../draggable/state';
import { useDraggableStateSnapshot } from '../draggable/use-draggable-state-snapshot';
import { useDraggableDimensions } from '../hooks/use-captured-dimensions';
import { attributes } from '../utils/attributes';
import { findDragHandle } from '../utils/find-drag-handle';
import { findDropIndicator } from '../utils/find-drop-indicator';
import { findPlaceholder } from '../utils/find-placeholder';
function getBody() {
  return document.body;
}

/**
 * Calls the `renderClone` function.
 *
 * Only rendered during drags.
 */
function DraggableCloneInner(_ref) {
  var children = _ref.children,
    droppableId = _ref.droppableId,
    type = _ref.type,
    draggableId = _ref.draggableId,
    index = _ref.index,
    draggingOver = _ref.draggingOver,
    style = _ref.style,
    _ref$getContainerForC = _ref.getContainerForClone,
    getContainerForClone = _ref$getContainerForC === void 0 ? getBody : _ref$getContainerForC,
    mode = _ref.mode;
  var _useDragDropContext = useDragDropContext(),
    contextId = _useDragDropContext.contextId;

  /**
   * The handle should maintain focus during a drag,
   * if it had focus before the drag started.
   */
  var focusDragHandle = useCallback(function (element) {
    if (!element) {
      return;
    }
    var dragHandle = findDragHandle({
      contextId: contextId,
      draggableId: draggableId
    });
    dragHandle === null || dragHandle === void 0 || dragHandle.focus();
  }, [contextId, draggableId]);
  var provided = useMemo(function () {
    return {
      innerRef: focusDragHandle,
      draggableProps: _defineProperty(_defineProperty(_defineProperty({}, attributes.draggable.contextId, contextId), attributes.draggable.id, draggableId), "style", style),
      dragHandleProps: _defineProperty(_defineProperty(_defineProperty(_defineProperty(_defineProperty({
        role: 'button',
        'aria-describedby': getHiddenTextElementId(contextId)
      }, attributes.dragHandle.contextId, contextId), attributes.dragHandle.draggableId, draggableId), "tabIndex", 0), "draggable", false), "onDragStart", function onDragStart() {})
    };
  }, [contextId, draggableId, focusDragHandle, style]);
  var snapshot = useDraggableStateSnapshot({
    draggingOver: draggingOver,
    isClone: true,
    isDragging: true,
    mode: mode
  });
  var rubric = useMemo(function () {
    return {
      draggableId: draggableId,
      type: type,
      source: {
        droppableId: droppableId,
        index: index
      }
    };
  }, [draggableId, droppableId, index, type]);
  return /*#__PURE__*/createPortal(children(provided, snapshot, rubric), getContainerForClone());
}

/**
 * Wrapper that is always rendered if there is a `renderClone` function.
 *
 * It sets up a monitor, and needs to observe the entire lifecycle.
 */
export function DraggableClone(_ref2) {
  var children = _ref2.children,
    droppableId = _ref2.droppableId,
    type = _ref2.type,
    getContainerForClone = _ref2.getContainerForClone;
  var _useDragDropContext2 = useDragDropContext(),
    contextId = _useDragDropContext2.contextId,
    getDragState = _useDragDropContext2.getDragState;
  var draggableDimensions = useDraggableDimensions();
  var _useReducer = useReducer(reducer, idleState),
    _useReducer2 = _slicedToArray(_useReducer, 2),
    state = _useReducer2[0],
    dispatch = _useReducer2[1];
  var monitorForLifecycle = useMonitorForLifecycle();
  useEffect(function () {
    return combine(monitorForLifecycle({
      onPendingDragStart: function onPendingDragStart(_ref3) {
        var start = _ref3.start,
          droppable = _ref3.droppable;
        if (droppableId !== start.source.droppableId) {
          return;
        }
        if (start.mode === 'FLUID') {
          return dispatch({
            type: 'START_POINTER_DRAG',
            payload: {
              start: start
            }
          });
        }
        if (start.mode === 'SNAP') {
          var dragState = getDragState();
          rbdInvariant(dragState.isDragging && dragState.draggableDimensions);
          return dispatch({
            type: 'START_KEYBOARD_DRAG',
            payload: {
              start: start,
              draggableDimensions: dragState.draggableDimensions,
              droppable: droppable
            }
          });
        }
      },
      onPendingDragUpdate: function onPendingDragUpdate(_ref4) {
        var update = _ref4.update,
          droppable = _ref4.droppable;
        if (state.type !== 'dragging') {
          return;
        }
        if (state.draggableId !== update.draggableId) {
          return;
        }
        dispatch({
          type: 'UPDATE_DRAG',
          payload: {
            update: update
          }
        });
        if (update.mode === 'SNAP') {
          /**
           * Updating the position in a microtask to resolve timing issues.
           *
           * When doing cross-axis dragging, the drop indicator in the new
           * droppable will mount and update in a `onPendingDragUpdate` too.
           *
           * The microtask ensures that the indicator will have updated by
           * the time this runs, so the preview will have the correct
           * location of the indicator.
           */
          queueMicrotask(function () {
            /**
             * Because this update occurs in a microtask, we need to check
             * that the drag is still happening.
             *
             * If it has ended we should not try to update the preview.
             */
            var dragState = getDragState();
            if (!dragState.isDragging) {
              return;
            }

            /**
             * The placeholder might not exist if its associated
             * draggable unmounts in a virtual list.
             */
            var placeholder = findPlaceholder(contextId);
            var placeholderRect = placeholder ? placeholder.getBoundingClientRect() : null;

            /**
             * The drop indicator might not exist if the current target
             * is null
             */
            var dropIndicator = findDropIndicator();
            var dropIndicatorRect = dropIndicator ? dropIndicator.getBoundingClientRect() : null;
            dispatch({
              type: 'UPDATE_KEYBOARD_PREVIEW',
              payload: {
                update: update,
                draggableDimensions: draggableDimensions,
                droppable: droppable,
                placeholderRect: placeholderRect,
                dropIndicatorRect: dropIndicatorRect
              }
            });
          });
        }
      },
      onBeforeDragEnd: function onBeforeDragEnd(_ref5) {
        var draggableId = _ref5.draggableId;
        if (state.type !== 'dragging') {
          return;
        }
        if (draggableId !== state.draggableId) {
          return;
        }
        dispatch({
          type: 'DROP'
        });
      }
    }), monitorForElements({
      canMonitor: function canMonitor(_ref6) {
        var source = _ref6.source;
        if (!isDraggableData(source.data)) {
          // not dragging something from the migration layer
          // we should not monitor it
          return false;
        }
        return source.data.contextId === contextId && source.data.droppableId === droppableId;
      },
      onDrag: function onDrag(_ref7) {
        var location = _ref7.location;
        dispatch({
          type: 'UPDATE_POINTER_PREVIEW',
          payload: {
            pointerLocation: location
          }
        });
      }
    }));
  }, [droppableId, contextId, monitorForLifecycle, state, draggableDimensions, getDragState]);
  if (state.type !== 'dragging') {
    return null;
  }
  var style = getDraggableProvidedStyle({
    draggableDimensions: draggableDimensions,
    draggableState: state
  });
  return /*#__PURE__*/React.createElement(DraggableCloneInner, {
    droppableId: droppableId,
    type: type,
    draggableId: state.draggableId,
    index: state.start.index,
    draggingOver: state.draggingOver,
    mode: state.mode
    // eslint-disable-next-line @atlaskit/ui-styling-standard/enforce-style-prop -- Ignored via go/DSP-18766
    ,
    style: style,
    getContainerForClone: getContainerForClone
  }, children);
}