#![allow(dead_code)]

use super::buffer::{hb_buffer_t, hb_glyph_info_t};
use super::hb_tag_t;
use super::ot_shape_plan::hb_ot_shape_plan_t;
use super::{aat_layout_kerx_table, aat_layout_morx_table, aat_layout_trak_table};
use super::{aat_map, hb_font_t};
use crate::hb::aat_layout_common::hb_aat_apply_context_t;

pub type hb_aat_layout_feature_type_t = u8;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_INVALID: u8 = 0xFF;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_ALL_TYPOGRAPHIC: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CURISVE_CONNECTION: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_LETTER_CASE: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_SUBSTITUTION: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_LINGUISTIC_REARRANGEMENT: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_SPACING: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_SMART_SWASH_TYPE: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_DIACRITICS_TYPE: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_FRACTIONS: u8 = 11;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_OVERLAPPING_CHARACTERS_TYPE: u8 = 13;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_TYPOGRAPHIC_EXTRAS: u8 = 14;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_MATHEMATICAL_EXTRAS: u8 = 15;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_ORNAMENT_SETS_TYPE: u8 = 16;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_ALTERNATIVES: u8 = 17;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_DESIGN_COMPLEXITY_TYPE: u8 = 18;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_STYLE_OPTIONS: u8 = 19;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE: u8 = 20;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_CASE: u8 = 21;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING: u8 = 22;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_TRANSLITERATION: u8 = 23;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_ANNOTATION_TYPE: u8 = 24;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_KANA_SPACING_TYPE: u8 = 25;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_IDEOGRAPHIC_SPACING_TYPE: u8 = 26;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_UNICODE_DECOMPOSITION_TYPE: u8 = 27;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_RUBY_KANA: u8 = 28;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CJK_SYMBOL_ALTERNATIVES_TYPE: u8 = 29;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_IDEOGRAPHIC_ALTERNATIVES_TYPE: u8 = 30;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CJK_VERTICAL_ROMAN_PLACEMENT_TYPE: u8 = 31;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_ITALIC_CJK_ROMAN: u8 = 32;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CASE_SENSITIVE_LAYOUT: u8 = 33;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_ALTERNATE_KANA: u8 = 34;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES: u8 = 35;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CONTEXTUAL_ALTERNATIVES: u8 = 36;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_LOWER_CASE: u8 = 37;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_UPPER_CASE: u8 = 38;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_LANGUAGE_TAG_TYPE: u8 = 39;
pub const HB_AAT_LAYOUT_FEATURE_TYPE_CJK_ROMAN_SPACING_TYPE: u8 = 103;

pub type hb_aat_layout_feature_selector_t = u8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INVALID: u8 = 0xFF;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_ALL_TYPOGRAPHIC */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALL_TYPE_FEATURES_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALL_TYPE_FEATURES_OFF: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_REQUIRED_LIGATURES_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_REQUIRED_LIGATURES_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_COMMON_LIGATURES_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_COMMON_LIGATURES_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_RARE_LIGATURES_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_RARE_LIGATURES_OFF: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LOGOS_ON: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LOGOS_OFF: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_REBUS_PICTURES_ON: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_REBUS_PICTURES_OFF: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DIPHTHONG_LIGATURES_ON: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DIPHTHONG_LIGATURES_OFF: u8 = 11;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SQUARED_LIGATURES_ON: u8 = 12;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SQUARED_LIGATURES_OFF: u8 = 13;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ABBREV_SQUARED_LIGATURES_ON: u8 = 14;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ABBREV_SQUARED_LIGATURES_OFF: u8 = 15;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SYMBOL_LIGATURES_ON: u8 = 16;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SYMBOL_LIGATURES_OFF: u8 = 17;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_LIGATURES_ON: u8 = 18;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_LIGATURES_OFF: u8 = 19;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HISTORICAL_LIGATURES_ON: u8 = 20;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HISTORICAL_LIGATURES_OFF: u8 = 21;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_UNCONNECTED: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PARTIALLY_CONNECTED: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CURSIVE: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_LETTER_CASE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_AND_LOWER_CASE: u8 = 0; /* deprecated */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALL_CAPS: u8 = 1; /* deprecated */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALL_LOWER_CASE: u8 = 2; /* deprecated */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SMALL_CAPS: u8 = 3; /* deprecated */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INITIAL_CAPS: u8 = 4; /* deprecated */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INITIAL_CAPS_AND_SMALL_CAPS: u8 = 5; /* deprecated */

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_SUBSTITUTION */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_OFF: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_LINGUISTIC_REARRANGEMENT */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINGUISTIC_REARRANGEMENT_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINGUISTIC_REARRANGEMENT_OFF: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_SPACING */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_MONOSPACED_NUMBERS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_NUMBERS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_THIRD_WIDTH_NUMBERS: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_QUARTER_WIDTH_NUMBERS: u8 = 3;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_SMART_SWASH_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_WORD_INITIAL_SWASHES_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_WORD_INITIAL_SWASHES_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_WORD_FINAL_SWASHES_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_WORD_FINAL_SWASHES_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINE_INITIAL_SWASHES_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINE_INITIAL_SWASHES_OFF: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINE_FINAL_SWASHES_ON: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LINE_FINAL_SWASHES_OFF: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NON_FINAL_SWASHES_ON: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NON_FINAL_SWASHES_OFF: u8 = 9;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_DIACRITICS_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SHOW_DIACRITICS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HIDE_DIACRITICS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DECOMPOSE_DIACRITICS: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NORMAL_POSITION: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SUPERIORS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INFERIORS: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ORDINALS: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SCIENTIFIC_INFERIORS: u8 = 4;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_FRACTIONS */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_FRACTIONS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_VERTICAL_FRACTIONS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DIAGONAL_FRACTIONS: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_OVERLAPPING_CHARACTERS_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PREVENT_OVERLAP_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PREVENT_OVERLAP_OFF: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_TYPOGRAPHIC_EXTRAS */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHENS_TO_EM_DASH_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHENS_TO_EM_DASH_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHEN_TO_EN_DASH_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHEN_TO_EN_DASH_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASHED_ZERO_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASHED_ZERO_OFF: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FORM_INTERROBANG_ON: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FORM_INTERROBANG_OFF: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SMART_QUOTES_ON: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SMART_QUOTES_OFF: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PERIODS_TO_ELLIPSIS_ON: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PERIODS_TO_ELLIPSIS_OFF: u8 = 11;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_MATHEMATICAL_EXTRAS */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHEN_TO_MINUS_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HYPHEN_TO_MINUS_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ASTERISK_TO_MULTIPLY_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ASTERISK_TO_MULTIPLY_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASH_TO_DIVIDE_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASH_TO_DIVIDE_OFF: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INEQUALITY_LIGATURES_ON: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INEQUALITY_LIGATURES_OFF: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_EXPONENTS_ON: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_EXPONENTS_OFF: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_MATHEMATICAL_GREEK_ON: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_MATHEMATICAL_GREEK_OFF: u8 = 11;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_ORNAMENT_SETS_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_ORNAMENTS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DINGBATS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PI_CHARACTERS: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FLEURONS: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DECORATIVE_BORDERS: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INTERNATIONAL_SYMBOLS: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_MATH_SYMBOLS: u8 = 6;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_ALTERNATIVES */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_ALTERNATES: u8 = 0;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_DESIGN_COMPLEXITY_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DESIGN_LEVEL1: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DESIGN_LEVEL2: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DESIGN_LEVEL3: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DESIGN_LEVEL4: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DESIGN_LEVEL5: u8 = 4;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_STYLE_OPTIONS */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_STYLE_OPTIONS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DISPLAY_TEXT: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ENGRAVED_TEXT: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ILLUMINATED_CAPS: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TITLING_CAPS: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TALL_CAPS: u8 = 5;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_CHARACTERS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SIMPLIFIED_CHARACTERS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1978_CHARACTERS: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1983_CHARACTERS: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1990_CHARACTERS: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_ALT_ONE: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_ALT_TWO: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_ALT_THREE: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_ALT_FOUR: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_ALT_FIVE: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_EXPERT_CHARACTERS: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS2004_CHARACTERS: u8 = 11;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HOJO_CHARACTERS: u8 = 12;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NLCCHARACTERS: u8 = 13;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_NAMES_CHARACTERS: u8 = 14;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_CASE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_NUMBERS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_NUMBERS: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_TEXT: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_MONOSPACED_TEXT: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HALF_WIDTH_TEXT: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_THIRD_WIDTH_TEXT: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_QUARTER_WIDTH_TEXT: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_PROPORTIONAL_TEXT: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_HALF_WIDTH_TEXT: u8 = 6;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_TRANSLITERATION */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_TRANSLITERATION: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HANJA_TO_HANGUL: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HIRAGANA_TO_KATAKANA: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_KATAKANA_TO_HIRAGANA: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_KANA_TO_ROMANIZATION: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ROMANIZATION_TO_HIRAGANA: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ROMANIZATION_TO_KATAKANA: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HANJA_TO_HANGUL_ALT_ONE: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HANJA_TO_HANGUL_ALT_TWO: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HANJA_TO_HANGUL_ALT_THREE: u8 = 9;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_ANNOTATION_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_ANNOTATION: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_BOX_ANNOTATION: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ROUNDED_BOX_ANNOTATION: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CIRCLE_ANNOTATION: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INVERTED_CIRCLE_ANNOTATION: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PARENTHESIS_ANNOTATION: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PERIOD_ANNOTATION: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ROMAN_NUMERAL_ANNOTATION: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DIAMOND_ANNOTATION: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INVERTED_BOX_ANNOTATION: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_INVERTED_ROUNDED_BOX_ANNOTATIO: u8 = 10;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_KANA_SPACING_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FULL_WIDTH_KANA: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_KANA: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_IDEOGRAPHIC_SPACING_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FULL_WIDTH_IDEOGRAPHS: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_IDEOGRAPHS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HALF_WIDTH_IDEOGRAPHS: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_UNICODE_DECOMPOSITION_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CANONICAL_COMPOSITION_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CANONICAL_COMPOSITION_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_COMPATIBILITY_COMPOSITION_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_COMPATIBILITY_COMPOSITION_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRANSCODING_COMPOSITION_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_TRANSCODING_COMPOSITION_OFF: u8 = 5;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_RUBY_KANA */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_RUBY_KANA: u8 = 0; /* deprecated - use HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_OFF instead */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA: u8 = 1; /* deprecated - use HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_ON instead */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_OFF: u8 = 3;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CJK_SYMBOL_ALTERNATIVES_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_CJK_SYMBOL_ALTERNATIVES: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_SYMBOL_ALT_ONE: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_SYMBOL_ALT_TWO: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_SYMBOL_ALT_THREE: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_SYMBOL_ALT_FOUR: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_SYMBOL_ALT_FIVE: u8 = 5;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_IDEOGRAPHIC_ALTERNATIVES_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_IDEOGRAPHIC_ALTERNATIVES: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_IDEOGRAPHIC_ALT_ONE: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_IDEOGRAPHIC_ALT_TWO: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_IDEOGRAPHIC_ALT_THREE: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_IDEOGRAPHIC_ALT_FOUR: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_IDEOGRAPHIC_ALT_FIVE: u8 = 5;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CJK_VERTICAL_ROMAN_PLACEMENT_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_VERTICAL_ROMAN_CENTERED: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_VERTICAL_ROMAN_HBASELINE: u8 = 1;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_ITALIC_CJK_ROMAN */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_CJK_ITALIC_ROMAN: u8 = 0; /* deprecated - use HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_OFF instead */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN: u8 = 1; /* deprecated - use HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_ON instead */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_OFF: u8 = 3;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CASE_SENSITIVE_LAYOUT */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_LAYOUT_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_LAYOUT_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_SPACING_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_SPACING_OFF: u8 = 3;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_ALTERNATE_KANA */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_HORIZ_KANA_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_HORIZ_KANA_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_VERT_KANA_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_VERT_KANA_OFF: u8 = 3;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_STYLISTIC_ALTERNATES: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ONE_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ONE_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWO_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWO_OFF: u8 = 5;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THREE_ON: u8 = 6;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THREE_OFF: u8 = 7;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOUR_ON: u8 = 8;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOUR_OFF: u8 = 9;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIVE_ON: u8 = 10;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIVE_OFF: u8 = 11;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIX_ON: u8 = 12;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIX_OFF: u8 = 13;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVEN_ON: u8 = 14;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVEN_OFF: u8 = 15;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHT_ON: u8 = 16;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHT_OFF: u8 = 17;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINE_ON: u8 = 18;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINE_OFF: u8 = 19;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TEN_ON: u8 = 20;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TEN_OFF: u8 = 21;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ELEVEN_ON: u8 = 22;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ELEVEN_OFF: u8 = 23;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWELVE_ON: u8 = 24;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWELVE_OFF: u8 = 25;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THIRTEEN_ON: u8 = 26;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THIRTEEN_OFF: u8 = 27;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOURTEEN_ON: u8 = 28;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOURTEEN_OFF: u8 = 29;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIFTEEN_ON: u8 = 30;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIFTEEN_OFF: u8 = 31;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIXTEEN_ON: u8 = 32;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIXTEEN_OFF: u8 = 33;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVENTEEN_ON: u8 = 34;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVENTEEN_OFF: u8 = 35;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHTEEN_ON: u8 = 36;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHTEEN_OFF: u8 = 37;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINETEEN_ON: u8 = 38;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINETEEN_OFF: u8 = 39;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWENTY_ON: u8 = 40;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWENTY_OFF: u8 = 41;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CONTEXTUAL_ALTERNATIVES */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_ALTERNATES_ON: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_ALTERNATES_OFF: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SWASH_ALTERNATES_ON: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_SWASH_ALTERNATES_OFF: u8 = 3;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_SWASH_ALTERNATES_ON: u8 = 4;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_SWASH_ALTERNATES_OFF: u8 = 5;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_LOWER_CASE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_LOWER_CASE: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_SMALL_CAPS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_PETITE_CAPS: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_UPPER_CASE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_UPPER_CASE: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_SMALL_CAPS: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_PETITE_CAPS: u8 = 2;

/* Selectors for #HB_AAT_LAYOUT_FEATURE_TYPE_CJK_ROMAN_SPACING_TYPE */
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_HALF_WIDTH_CJK_ROMAN: u8 = 0;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_CJK_ROMAN: u8 = 1;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_CJK_ROMAN: u8 = 2;
pub const HB_AAT_LAYOUT_FEATURE_SELECTOR_FULL_WIDTH_CJK_ROMAN: u8 = 3;

pub struct hb_aat_feature_mapping_t {
    pub ot_feature_tag: hb_tag_t,
    pub aat_feature_type: hb_aat_layout_feature_type_t,
    pub selector_to_enable: u8,
    pub selector_to_disable: u8,
}

impl hb_aat_feature_mapping_t {
    const fn new(
        ot_feature_tag: &[u8; 4],
        aat_feature_type: hb_aat_layout_feature_type_t,
        selector_to_enable: u8,
        selector_to_disable: u8,
    ) -> Self {
        hb_aat_feature_mapping_t {
            ot_feature_tag: hb_tag_t::from_bytes(ot_feature_tag),
            aat_feature_type,
            selector_to_enable,
            selector_to_disable,
        }
    }
}

/// Mapping from OpenType feature tags to AAT feature names and selectors.
///
/// Table data courtesy of Apple.
/// Converted from mnemonics to integers when moving to this file.
#[rustfmt::skip]
pub const feature_mappings: &[hb_aat_feature_mapping_t] = &[
    hb_aat_feature_mapping_t::new(b"afrc", HB_AAT_LAYOUT_FEATURE_TYPE_FRACTIONS, HB_AAT_LAYOUT_FEATURE_SELECTOR_VERTICAL_FRACTIONS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_FRACTIONS),
    hb_aat_feature_mapping_t::new(b"c2pc", HB_AAT_LAYOUT_FEATURE_TYPE_UPPER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_PETITE_CAPS, HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_UPPER_CASE),
    hb_aat_feature_mapping_t::new(b"c2sc", HB_AAT_LAYOUT_FEATURE_TYPE_UPPER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_SMALL_CAPS, HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_UPPER_CASE),
    hb_aat_feature_mapping_t::new(b"calt", HB_AAT_LAYOUT_FEATURE_TYPE_CONTEXTUAL_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_ALTERNATES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_ALTERNATES_OFF),
    hb_aat_feature_mapping_t::new(b"case", HB_AAT_LAYOUT_FEATURE_TYPE_CASE_SENSITIVE_LAYOUT, HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_LAYOUT_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_LAYOUT_OFF),
    hb_aat_feature_mapping_t::new(b"clig", HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_LIGATURES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_LIGATURES_OFF),
    hb_aat_feature_mapping_t::new(b"cpsp", HB_AAT_LAYOUT_FEATURE_TYPE_CASE_SENSITIVE_LAYOUT, HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_SPACING_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CASE_SENSITIVE_SPACING_OFF),
    hb_aat_feature_mapping_t::new(b"cswh", HB_AAT_LAYOUT_FEATURE_TYPE_CONTEXTUAL_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_SWASH_ALTERNATES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CONTEXTUAL_SWASH_ALTERNATES_OFF),
    hb_aat_feature_mapping_t::new(b"dlig", HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES, HB_AAT_LAYOUT_FEATURE_SELECTOR_RARE_LIGATURES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_RARE_LIGATURES_OFF),
    hb_aat_feature_mapping_t::new(b"expt", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_EXPERT_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"frac", HB_AAT_LAYOUT_FEATURE_TYPE_FRACTIONS, HB_AAT_LAYOUT_FEATURE_SELECTOR_DIAGONAL_FRACTIONS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_FRACTIONS),
    hb_aat_feature_mapping_t::new(b"fwid", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_MONOSPACED_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"halt", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_HALF_WIDTH_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"hist", 40, 0, 1),
    hb_aat_feature_mapping_t::new(b"hkna", HB_AAT_LAYOUT_FEATURE_TYPE_ALTERNATE_KANA, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_HORIZ_KANA_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_HORIZ_KANA_OFF),
    hb_aat_feature_mapping_t::new(b"hlig", HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES, HB_AAT_LAYOUT_FEATURE_SELECTOR_HISTORICAL_LIGATURES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_HISTORICAL_LIGATURES_OFF),
    hb_aat_feature_mapping_t::new(b"hngl", HB_AAT_LAYOUT_FEATURE_TYPE_TRANSLITERATION, HB_AAT_LAYOUT_FEATURE_SELECTOR_HANJA_TO_HANGUL, HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_TRANSLITERATION),
    hb_aat_feature_mapping_t::new(b"hojo", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_HOJO_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"hwid", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_HALF_WIDTH_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"ital", HB_AAT_LAYOUT_FEATURE_TYPE_ITALIC_CJK_ROMAN, HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_CJK_ITALIC_ROMAN_OFF),
    hb_aat_feature_mapping_t::new(b"jp04", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS2004_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"jp78", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1978_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"jp83", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1983_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"jp90", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_JIS1990_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"liga", HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES, HB_AAT_LAYOUT_FEATURE_SELECTOR_COMMON_LIGATURES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_COMMON_LIGATURES_OFF),
    hb_aat_feature_mapping_t::new(b"lnum", HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_UPPER_CASE_NUMBERS, 2),
    hb_aat_feature_mapping_t::new(b"mgrk", HB_AAT_LAYOUT_FEATURE_TYPE_MATHEMATICAL_EXTRAS, HB_AAT_LAYOUT_FEATURE_SELECTOR_MATHEMATICAL_GREEK_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_MATHEMATICAL_GREEK_OFF),
    hb_aat_feature_mapping_t::new(b"nlck", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_NLCCHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"onum", HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_NUMBERS, 2),
    hb_aat_feature_mapping_t::new(b"ordn", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION, HB_AAT_LAYOUT_FEATURE_SELECTOR_ORDINALS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NORMAL_POSITION),
    hb_aat_feature_mapping_t::new(b"palt", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_PROPORTIONAL_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"pcap", HB_AAT_LAYOUT_FEATURE_TYPE_LOWER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_PETITE_CAPS, HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_LOWER_CASE),
    hb_aat_feature_mapping_t::new(b"pkna", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"pnum", HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_NUMBERS, 4),
    hb_aat_feature_mapping_t::new(b"pwid", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_PROPORTIONAL_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"qwid", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_QUARTER_WIDTH_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"rlig", HB_AAT_LAYOUT_FEATURE_TYPE_LIGATURES, HB_AAT_LAYOUT_FEATURE_SELECTOR_REQUIRED_LIGATURES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_REQUIRED_LIGATURES_OFF),
    hb_aat_feature_mapping_t::new(b"ruby", HB_AAT_LAYOUT_FEATURE_TYPE_RUBY_KANA, HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_RUBY_KANA_OFF),
    hb_aat_feature_mapping_t::new(b"sinf", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION, HB_AAT_LAYOUT_FEATURE_SELECTOR_SCIENTIFIC_INFERIORS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NORMAL_POSITION),
    hb_aat_feature_mapping_t::new(b"smcp", HB_AAT_LAYOUT_FEATURE_TYPE_LOWER_CASE, HB_AAT_LAYOUT_FEATURE_SELECTOR_LOWER_CASE_SMALL_CAPS, HB_AAT_LAYOUT_FEATURE_SELECTOR_DEFAULT_LOWER_CASE),
    hb_aat_feature_mapping_t::new(b"smpl", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_SIMPLIFIED_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"ss01", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ONE_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ONE_OFF),
    hb_aat_feature_mapping_t::new(b"ss02", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWO_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWO_OFF),
    hb_aat_feature_mapping_t::new(b"ss03", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THREE_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THREE_OFF),
    hb_aat_feature_mapping_t::new(b"ss04", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOUR_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOUR_OFF),
    hb_aat_feature_mapping_t::new(b"ss05", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIVE_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIVE_OFF),
    hb_aat_feature_mapping_t::new(b"ss06", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIX_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIX_OFF),
    hb_aat_feature_mapping_t::new(b"ss07", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss08", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHT_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHT_OFF),
    hb_aat_feature_mapping_t::new(b"ss09", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINE_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINE_OFF),
    hb_aat_feature_mapping_t::new(b"ss10", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss11", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ELEVEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_ELEVEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss12", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWELVE_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWELVE_OFF),
    hb_aat_feature_mapping_t::new(b"ss13", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THIRTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_THIRTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss14", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOURTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FOURTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss15", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIFTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_FIFTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss16", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIXTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SIXTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss17", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVENTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_SEVENTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss18", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHTEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_EIGHTEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss19", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINETEEN_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_NINETEEN_OFF),
    hb_aat_feature_mapping_t::new(b"ss20", HB_AAT_LAYOUT_FEATURE_TYPE_STYLISTIC_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWENTY_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_STYLISTIC_ALT_TWENTY_OFF),
    hb_aat_feature_mapping_t::new(b"subs", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION, HB_AAT_LAYOUT_FEATURE_SELECTOR_INFERIORS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NORMAL_POSITION),
    hb_aat_feature_mapping_t::new(b"sups", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_POSITION, HB_AAT_LAYOUT_FEATURE_SELECTOR_SUPERIORS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NORMAL_POSITION),
    hb_aat_feature_mapping_t::new(b"swsh", HB_AAT_LAYOUT_FEATURE_TYPE_CONTEXTUAL_ALTERNATIVES, HB_AAT_LAYOUT_FEATURE_SELECTOR_SWASH_ALTERNATES_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_SWASH_ALTERNATES_OFF),
    hb_aat_feature_mapping_t::new(b"titl", HB_AAT_LAYOUT_FEATURE_TYPE_STYLE_OPTIONS, HB_AAT_LAYOUT_FEATURE_SELECTOR_TITLING_CAPS, HB_AAT_LAYOUT_FEATURE_SELECTOR_NO_STYLE_OPTIONS),
    hb_aat_feature_mapping_t::new(b"tnam", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_NAMES_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"tnum", HB_AAT_LAYOUT_FEATURE_TYPE_NUMBER_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_MONOSPACED_NUMBERS, 4),
    hb_aat_feature_mapping_t::new(b"trad", HB_AAT_LAYOUT_FEATURE_TYPE_CHARACTER_SHAPE, HB_AAT_LAYOUT_FEATURE_SELECTOR_TRADITIONAL_CHARACTERS, 16),
    hb_aat_feature_mapping_t::new(b"twid", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_THIRD_WIDTH_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"unic", HB_AAT_LAYOUT_FEATURE_TYPE_LETTER_CASE, 14, 15),
    hb_aat_feature_mapping_t::new(b"valt", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_PROPORTIONAL_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"vert", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_SUBSTITUTION, HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_OFF),
    hb_aat_feature_mapping_t::new(b"vhal", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_HALF_WIDTH_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"vkna", HB_AAT_LAYOUT_FEATURE_TYPE_ALTERNATE_KANA, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_VERT_KANA_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALTERNATE_VERT_KANA_OFF),
    hb_aat_feature_mapping_t::new(b"vpal", HB_AAT_LAYOUT_FEATURE_TYPE_TEXT_SPACING, HB_AAT_LAYOUT_FEATURE_SELECTOR_ALT_PROPORTIONAL_TEXT, 7),
    hb_aat_feature_mapping_t::new(b"vrt2", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_SUBSTITUTION, HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_SUBSTITUTE_VERTICAL_FORMS_OFF),
    hb_aat_feature_mapping_t::new(b"vrtr", HB_AAT_LAYOUT_FEATURE_TYPE_VERTICAL_SUBSTITUTION, 2, 3),
    hb_aat_feature_mapping_t::new(b"zero", HB_AAT_LAYOUT_FEATURE_TYPE_TYPOGRAPHIC_EXTRAS, HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASHED_ZERO_ON, HB_AAT_LAYOUT_FEATURE_SELECTOR_SLASHED_ZERO_OFF),
];

mod AAT {
    pub const DELETED_GLYPH: u32 = 0xFFFF;
}

pub fn hb_aat_layout_substitute(
    plan: &hb_ot_shape_plan_t,
    face: &hb_font_t,
    buffer: &mut hb_buffer_t,
) {
    let mut builder = aat_map::hb_aat_map_builder_t::default();

    for feature in &plan.user_features {
        builder.add_feature(face, feature);
    }

    let mut aat_map = aat_map::hb_aat_map_t::default();
    if plan.apply_morx {
        builder.compile(face, &mut aat_map);
    }

    let mut c = hb_aat_apply_context_t::new(face, buffer);
    aat_layout_morx_table::apply(&mut c, &mut aat_map);
}

pub fn hb_aat_layout_zero_width_deleted_glyphs(buffer: &mut hb_buffer_t) {
    for i in 0..buffer.len {
        if buffer.info[i].glyph_id == AAT::DELETED_GLYPH {
            buffer.pos[i].x_advance = 0;
            buffer.pos[i].y_advance = 0;
            buffer.pos[i].x_offset = 0;
            buffer.pos[i].y_offset = 0;
        }
    }
}

fn is_deleted_glyph(info: &hb_glyph_info_t) -> bool {
    info.glyph_id == AAT::DELETED_GLYPH
}

pub fn hb_aat_layout_remove_deleted_glyphs(buffer: &mut hb_buffer_t) {
    buffer.delete_glyphs_inplace(is_deleted_glyph)
}

pub fn hb_aat_layout_position(
    plan: &hb_ot_shape_plan_t,
    face: &hb_font_t,
    buffer: &mut hb_buffer_t,
) {
    aat_layout_kerx_table::apply(plan, face, buffer);
}

pub fn hb_aat_layout_track(plan: &hb_ot_shape_plan_t, face: &hb_font_t, buffer: &mut hb_buffer_t) {
    aat_layout_trak_table::apply(plan, face, buffer);
}
