use crate::buffer::Buffer;

const MACHINE_TRANS_KEYS: &[u8] = &[
    1, 32, 3, 30, 5, 29, 5, 8, 5, 29, 3, 25, 5, 25, 5, 25,
    3, 29, 3, 29, 3, 29, 3, 29, 1, 16, 3, 29, 3, 29, 3, 29,
    3, 29, 3, 29, 3, 30, 3, 29, 3, 29, 3, 29, 3, 29, 3, 29,
    5, 29, 5, 8, 5, 29, 3, 25, 5, 25, 5, 25, 3, 29, 3, 29,
    3, 29, 3, 29, 1, 16, 3, 30, 3, 29, 3, 29, 3, 29, 3, 29,
    3, 29, 3, 30, 3, 29, 3, 29, 3, 29, 3, 29, 3, 29, 3, 30,
    3, 29, 1, 32, 1, 32, 8, 8, 0
];

const MACHINE_KEY_SPANS: &[u8] = &[
    32, 28, 25, 4, 25, 23, 21, 21,
    27, 27, 27, 27, 16, 27, 27, 27,
    27, 27, 28, 27, 27, 27, 27, 27,
    25, 4, 25, 23, 21, 21, 27, 27,
    27, 27, 16, 28, 27, 27, 27, 27,
    27, 28, 27, 27, 27, 27, 27, 28,
    27, 32, 32, 1
];

const MACHINE_INDEX_OFFSETS: &[u16] = &[
    0, 33, 62, 88, 93, 119, 143, 165,
    187, 215, 243, 271, 299, 316, 344, 372,
    400, 428, 456, 485, 513, 541, 569, 597,
    625, 651, 656, 682, 706, 728, 750, 778,
    806, 834, 862, 879, 908, 936, 964, 992,
    1020, 1048, 1077, 1105, 1133, 1161, 1189, 1217,
    1246, 1274, 1307, 1340
];

const MACHINE_INDICIES: &[u8] = &[
    1, 1, 2, 3, 4, 4, 0, 5,
    0, 6, 1, 0, 0, 0, 0, 7,
    0, 8, 9, 0, 10, 11, 12, 13,
    14, 15, 16, 17, 18, 19, 20, 1,
    0, 22, 23, 24, 24, 21, 25, 21,
    26, 21, 21, 21, 21, 21, 21, 21,
    27, 21, 21, 28, 29, 30, 31, 32,
    33, 34, 35, 36, 37, 21, 24, 24,
    21, 25, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 38, 21, 21, 21, 21,
    21, 21, 32, 21, 21, 21, 36, 21,
    24, 24, 21, 25, 21, 24, 24, 21,
    25, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 32, 21, 21, 21, 36, 21, 39,
    21, 24, 24, 21, 25, 21, 32, 21,
    21, 21, 21, 21, 21, 21, 40, 21,
    21, 21, 21, 21, 21, 32, 21, 24,
    24, 21, 25, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 40, 21, 21, 21,
    21, 21, 21, 32, 21, 24, 24, 21,
    25, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 32, 21, 22, 21, 24, 24, 21,
    25, 21, 26, 21, 21, 21, 21, 21,
    21, 21, 41, 21, 21, 41, 21, 21,
    21, 32, 42, 21, 21, 36, 21, 22,
    21, 24, 24, 21, 25, 21, 26, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 32, 21, 21,
    21, 36, 21, 22, 21, 24, 24, 21,
    25, 21, 26, 21, 21, 21, 21, 21,
    21, 21, 41, 21, 21, 21, 21, 21,
    21, 32, 42, 21, 21, 36, 21, 22,
    21, 24, 24, 21, 25, 21, 26, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 32, 42, 21,
    21, 36, 21, 1, 1, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 1, 21, 22, 21, 24, 24,
    21, 25, 21, 26, 21, 21, 21, 21,
    21, 21, 21, 27, 21, 21, 28, 29,
    30, 31, 32, 33, 34, 35, 36, 21,
    22, 21, 24, 24, 21, 25, 21, 26,
    21, 21, 21, 21, 21, 21, 21, 43,
    21, 21, 21, 21, 21, 21, 32, 33,
    34, 35, 36, 21, 22, 21, 24, 24,
    21, 25, 21, 26, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 32, 33, 34, 35, 36, 21,
    22, 21, 24, 24, 21, 25, 21, 26,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 32, 33,
    34, 21, 36, 21, 22, 21, 24, 24,
    21, 25, 21, 26, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 32, 21, 34, 21, 36, 21,
    22, 21, 24, 24, 21, 25, 21, 26,
    21, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 32, 33,
    34, 35, 36, 43, 21, 22, 21, 24,
    24, 21, 25, 21, 26, 21, 21, 21,
    21, 21, 21, 21, 21, 21, 21, 28,
    21, 30, 21, 32, 33, 34, 35, 36,
    21, 22, 21, 24, 24, 21, 25, 21,
    26, 21, 21, 21, 21, 21, 21, 21,
    43, 21, 21, 28, 21, 21, 21, 32,
    33, 34, 35, 36, 21, 22, 21, 24,
    24, 21, 25, 21, 26, 21, 21, 21,
    21, 21, 21, 21, 44, 21, 21, 28,
    29, 30, 21, 32, 33, 34, 35, 36,
    21, 22, 21, 24, 24, 21, 25, 21,
    26, 21, 21, 21, 21, 21, 21, 21,
    21, 21, 21, 28, 29, 30, 21, 32,
    33, 34, 35, 36, 21, 22, 23, 24,
    24, 21, 25, 21, 26, 21, 21, 21,
    21, 21, 21, 21, 27, 21, 21, 28,
    29, 30, 31, 32, 33, 34, 35, 36,
    21, 46, 46, 45, 5, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 47, 45,
    45, 45, 45, 45, 45, 14, 45, 45,
    45, 18, 45, 46, 46, 45, 5, 45,
    46, 46, 45, 5, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 14, 45, 45, 45,
    18, 45, 48, 45, 46, 46, 45, 5,
    45, 14, 45, 45, 45, 45, 45, 45,
    45, 49, 45, 45, 45, 45, 45, 45,
    14, 45, 46, 46, 45, 5, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 49,
    45, 45, 45, 45, 45, 45, 14, 45,
    46, 46, 45, 5, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 14, 45, 2, 45,
    46, 46, 45, 5, 45, 6, 45, 45,
    45, 45, 45, 45, 45, 50, 45, 45,
    50, 45, 45, 45, 14, 51, 45, 45,
    18, 45, 2, 45, 46, 46, 45, 5,
    45, 6, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    14, 45, 45, 45, 18, 45, 2, 45,
    46, 46, 45, 5, 45, 6, 45, 45,
    45, 45, 45, 45, 45, 50, 45, 45,
    45, 45, 45, 45, 14, 51, 45, 45,
    18, 45, 2, 45, 46, 46, 45, 5,
    45, 6, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    14, 51, 45, 45, 18, 45, 52, 52,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 52, 45, 2,
    3, 46, 46, 45, 5, 45, 6, 45,
    45, 45, 45, 45, 45, 45, 8, 45,
    45, 10, 11, 12, 13, 14, 15, 16,
    17, 18, 19, 45, 2, 45, 46, 46,
    45, 5, 45, 6, 45, 45, 45, 45,
    45, 45, 45, 8, 45, 45, 10, 11,
    12, 13, 14, 15, 16, 17, 18, 45,
    2, 45, 46, 46, 45, 5, 45, 6,
    45, 45, 45, 45, 45, 45, 45, 53,
    45, 45, 45, 45, 45, 45, 14, 15,
    16, 17, 18, 45, 2, 45, 46, 46,
    45, 5, 45, 6, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 14, 15, 16, 17, 18, 45,
    2, 45, 46, 46, 45, 5, 45, 6,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 14, 15,
    16, 45, 18, 45, 2, 45, 46, 46,
    45, 5, 45, 6, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 14, 45, 16, 45, 18, 45,
    2, 45, 46, 46, 45, 5, 45, 6,
    45, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 14, 15,
    16, 17, 18, 53, 45, 2, 45, 46,
    46, 45, 5, 45, 6, 45, 45, 45,
    45, 45, 45, 45, 45, 45, 45, 10,
    45, 12, 45, 14, 15, 16, 17, 18,
    45, 2, 45, 46, 46, 45, 5, 45,
    6, 45, 45, 45, 45, 45, 45, 45,
    53, 45, 45, 10, 45, 45, 45, 14,
    15, 16, 17, 18, 45, 2, 45, 46,
    46, 45, 5, 45, 6, 45, 45, 45,
    45, 45, 45, 45, 54, 45, 45, 10,
    11, 12, 45, 14, 15, 16, 17, 18,
    45, 2, 45, 46, 46, 45, 5, 45,
    6, 45, 45, 45, 45, 45, 45, 45,
    45, 45, 45, 10, 11, 12, 45, 14,
    15, 16, 17, 18, 45, 2, 3, 46,
    46, 45, 5, 45, 6, 45, 45, 45,
    45, 45, 45, 45, 8, 45, 45, 10,
    11, 12, 13, 14, 15, 16, 17, 18,
    45, 22, 23, 24, 24, 21, 25, 21,
    26, 21, 21, 21, 21, 21, 21, 21,
    55, 21, 21, 28, 29, 30, 31, 32,
    33, 34, 35, 36, 37, 21, 22, 56,
    24, 24, 21, 25, 21, 26, 21, 21,
    21, 21, 21, 21, 21, 27, 21, 21,
    28, 29, 30, 31, 32, 33, 34, 35,
    36, 21, 1, 1, 2, 3, 46, 46,
    45, 5, 45, 6, 1, 45, 45, 45,
    45, 1, 45, 8, 45, 45, 10, 11,
    12, 13, 14, 15, 16, 17, 18, 19,
    45, 1, 45, 1, 1, 57, 57, 57,
    57, 57, 57, 57, 57, 1, 57, 57,
    57, 57, 1, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 1, 57, 58, 57, 0
];

const MACHINE_TRANS_TARGS: &[u8] = &[
    0, 1, 24, 34, 0, 25, 31, 47,
    36, 50, 37, 42, 43, 44, 27, 39,
    40, 41, 30, 46, 51, 0, 2, 12,
    0, 3, 9, 13, 14, 19, 20, 21,
    5, 16, 17, 18, 8, 23, 4, 6,
    7, 10, 11, 15, 22, 0, 0, 26,
    28, 29, 32, 33, 35, 38, 45, 48,
    49, 0, 0
];

const MACHINE_TRANS_ACTIONS: &[u8] = &[
    3, 0, 0, 0, 4, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 5, 0, 0,
    6, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 7, 8, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 9, 10
];

const MACHINE_TO_STATE_ACTIONS: &[u8] = &[
    1, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0
];

const MACHINE_FROM_STATE_ACTIONS: &[u8] = &[
    2, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0
];

const MACHINE_EOF_TRANS: &[u8] = &[
    0, 22, 22, 22, 22, 22, 22, 22,
    22, 22, 22, 22, 22, 22, 22, 22,
    22, 22, 22, 22, 22, 22, 22, 22,
    46, 46, 46, 46, 46, 46, 46, 46,
    46, 46, 46, 46, 46, 46, 46, 46,
    46, 46, 46, 46, 46, 46, 46, 22,
    22, 46, 58, 58
];

#[derive(Clone, Copy)]
pub enum SyllableType {
    ConsonantSyllable = 0,
    PunctuationCluster,
    BrokenCluster,
    NonMyanmarCluster,
}

pub fn find_syllables_myanmar(buffer: &mut Buffer) {
    let mut cs = 0usize;
    let mut ts = 0;
    let mut te;
    let mut p = 0;
    let pe = buffer.len;
    let eof = buffer.len;
    let mut syllable_serial = 1u8;
    let mut reset = true;
    let mut slen;
    let mut trans = 0;
    if p == pe {
        if MACHINE_EOF_TRANS[cs] > 0 {
            trans = (MACHINE_EOF_TRANS[cs] - 1) as usize;
        }
    }

    loop {
        if reset {
            if MACHINE_FROM_STATE_ACTIONS[cs] == 2 {
                ts = p;
            }

            slen = MACHINE_KEY_SPANS[cs] as usize;
            let cs_idx = ((cs as i32) << 1) as usize;
            let i = if slen > 0 &&
                MACHINE_TRANS_KEYS[cs_idx] <= buffer.info[p].indic_category() as u8 &&
                buffer.info[p].indic_category() as u8 <= MACHINE_TRANS_KEYS[cs_idx + 1]
            {
                (buffer.info[p].indic_category() as u8 - MACHINE_TRANS_KEYS[cs_idx]) as usize
            } else {
                slen
            };
            trans = MACHINE_INDICIES[MACHINE_INDEX_OFFSETS[cs] as usize + i] as usize;
        }
        reset = true;

        cs = MACHINE_TRANS_TARGS[trans] as usize;

        if MACHINE_TRANS_ACTIONS[trans] != 0 {
            match MACHINE_TRANS_ACTIONS[trans] {
                6 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::ConsonantSyllable, buffer);
                }
                4 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::NonMyanmarCluster, buffer);
                }
                10 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::PunctuationCluster, buffer);
                }
                8 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::BrokenCluster, buffer);
                }
                3 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::NonMyanmarCluster, buffer);
                }
                5 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::ConsonantSyllable, buffer);
                }
                7 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::BrokenCluster, buffer);
                }
                9 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::NonMyanmarCluster, buffer);
                }
                _ => {}
            }
        }

        if MACHINE_TO_STATE_ACTIONS[cs] == 1 {
            ts = 0;
        }

        p += 1;
        if p != pe {
            continue;
        }

        if p == eof {
            if MACHINE_EOF_TRANS[cs] > 0 {
                trans = (MACHINE_EOF_TRANS[cs] - 1) as usize;
                reset = false;
                continue;
            }
        }

        break;
    }
}

#[inline]
fn found_syllable(
    start: usize,
    end: usize,
    syllable_serial: &mut u8,
    kind: SyllableType,
    buffer: &mut Buffer,
) {
    for i in start..end {
        buffer.info[i].set_syllable((*syllable_serial << 4) | kind as u8);
    }

    *syllable_serial += 1;

    if *syllable_serial == 16 {
        *syllable_serial = 1;
    }
}
