/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*                      PROTOTYPES FOR FILES IN UTIL                       */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/


#ifndef __UTIL_H
#define __UTIL_H



/***************************************************************************/
/*                                                                         */
/*                             allocrus.c                                  */
/*                                                                         */
/***************************************************************************/

/***************************************************************************/
/*                                                                         */
/*                   MEMORY ALLOCATION MACROS                              */
/*                                                                         */
/*                           TSP CODE                                      */
/*                                                                         */
/*                                                                         */
/*  Written by:  Applegate, Bixby, Chvatal, and Cook                       */
/*  Date: February 24, 1995 (cofeb24)                                      */
/*                                                                         */
/*                                                                         */
/*  EXPORTED MACROS:                                                       */
/*    CC_SAFE_MALLOC (nnum,type)                                           */
/*         int nnum (the number of objects to be malloced)                 */
/*         data type (the sort of objects to be malloced)                  */
/*         RETURNS a pointer to the allocated space. If out of memory,     */
/*                 it prints an error message and returns NULL.            */
/*                                                                         */
/*    CC_FREE (object,type)                                                */
/*         type *object (pointer to previously allocated space)            */
/*         data type (the sort of object)                                  */
/*         ACTION: frees the memory and sets the object to NULL.           */
/*                                                                         */
/*    CC_IFFREE (object,type)                                              */
/*         type *object (pointer to previously allocated space)            */
/*         data type (the sort of object)                                  */
/*         ACTION: if *object is not NULL, frees the memory and sets       */
/*                 the object to NULL.                                     */
/*                                                                         */
/*    CC_PTR_ALLOC_ROUTINE (type, functionname, chunklist, freelist)       */
/*         data type (the sort of objects)                                 */
/*         string functionname (the generated function)                    */
/*         CCbigchunkptr *chunklist (used to accumulate bigchunks)         */
/*         type *freelist (used for the linked list of objects)            */
/*         ACTION: Generates a function ("functionname") that returns      */
/*                 (type *) objects, keeping the free ones on freelist     */
/*                 and getting its space from calls to bigchunkalloc.      */
/*                                                                         */
/*    CC_PTR_FREE_ROUTINE (type, functionname, freelist)                   */
/*         Parameters as above.                                            */
/*         ACTION: Generates a function that adds an object to the         */
/*                 freelist.                                               */
/*                                                                         */
/*    CC_PTR_FREE_LIST_ROUTINE (type, functionname, freefunction)          */
/*         Parameters defined as above, with freefunction the function     */
/*         generated by PTR_FREE_ROUTINE.                                  */
/*         ACTION: Generates a function to free a linked list of           */
/*                 objects using calls to freefunction.                    */
/*                                                                         */
/*    CC_PTR_FREE_WORLD_ROUTINE( type, functionname, chunklist, freelist)  */
/*         Parameters defined as above.                                    */
/*         ACTION: Generates a function that returns all of the            */
/*                 memory used in the PTR_ALLOC_ROUTINE allocations        */
/*                 back to the global supply of CCbigchunkptrs.            */
/*                                                                         */
/*    CC_PTR_LEAKS_ROUTINE (type, name, chunklist, freelist, field,        */
/*            fieldtype)                                                   */
/*          As above, with "field" the name of a "fieldtype" field in the  */
/*          object type that can be set to 0 or to 1.                      */
/*          ACTION: Generates a function that checks to see that we have   */
/*                  not leaked any of the objects.                         */
/*                                                                         */
/*    CC_PTR_STATUS_ROUTINE (type, name, chunklist, freelist)              */
/*          ACTION: Like LEAKS, but does not check for duplicates (and so  */
/*                  does not corrupt the objects).                         */
/*                                                                         */
/*  NOTES:                                                                 */
/*     These routines use the functions in allocrus.c.  The PTR macros     */
/*  The PTR macros generate the functions for allocating objects for       */
/*  linked lists. They get their raw memory from the bigchunk supply, so   */
/*  so foo_free_world (geneated by PTR_FREE_WORLD_ROUTINE) should be       */
/*  called for each type of linked object "foo" when closing down the      */
/*  local memory.                                                          */
/*     To use these functions, put the macros near the top of the file     */
/*  before any calls to the functions (since the macros also write the     */
/*  function prototypes). If you use PTR_FREE_LIST_ROUTINE for foo, you    */
/*  must also use PTR_FREE_ROUTINE, and PTR_FREE_LIST_ROUTINE must be      */
/*  listed after CC_PTR_FREE_ROUTINE (to get the prototype).               */
/*                                                                         */
/***************************************************************************/


#define CC_SAFE_MALLOC(nnum,type)                                          \
    (type *) CCutil_allocrus (((unsigned int) (nnum)) * sizeof (type))

#define CC_FREE(object,type) {                                             \
    CCutil_freerus ((void *) (object));                                    \
    object = (type *) NULL;                                                \
}

#define CC_IFFREE(object,type) {                                           \
    if ((object)) CC_FREE ((object),type);                                 \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_ALLOC_ROUTINE(type, functionname)                    \
static type * functionname (void);                                         \
static type * functionname (void)
#else
#define CC_HEADER_PTR_ALLOC_ROUTINE(type, functionname)                    \
static type * functionname ();                                             \
static type * functionname ()
#endif

#define CC_PTR_ALLOC_ROUTINE(type, functionname, chunklist, freelist)      \
static  type * freelist = ( type * ) NULL;                                 \
static  CCbigchunkptr * chunklist = ( CCbigchunkptr * ) NULL;              \
 CC_HEADER_PTR_ALLOC_ROUTINE (type, functionname)                          \
{                                                                          \
    type *p;                                                               \
                                                                           \
    if (! freelist ) {                                                     \
        int count = CC_BIGCHUNK / sizeof ( type );                         \
        CCbigchunkptr *bp;                                                 \
                                                                           \
        bp = CCutil_bigchunkalloc ();                                      \
        if (!bp) {                                                         \
            fprintf (stderr, "ptr alloc failed\n");                        \
            return ( type * ) NULL;                                        \
        }                                                                  \
        freelist = ( type * ) bp->this;                                    \
        bp->next = chunklist ;                                             \
        chunklist = bp;                                                    \
                                                                           \
        for (p = freelist + count - 2; p >= freelist ; p--)                \
            p->next = p + 1;                                               \
        freelist [count - 1].next = ( type * ) NULL;                       \
    }                                                                      \
    p = freelist ;                                                         \
    freelist = p->next;                                                    \
                                                                           \
    return p;                                                              \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                     \
static void functionname ( type *p );                                      \
static void functionname ( type *p )
#else
#define CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                     \
static void functionname ();                                               \
static void functionname ( p )                                             \
type *p;
#endif

#define CC_PTR_FREE_ROUTINE(type, functionname, freelist)                  \
 CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                            \
{                                                                          \
    p->next = freelist ;                                                   \
    freelist = p;                                                          \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_LIST_ROUTINE(type, functionname)                \
static void functionname ( type *p );                                      \
static void functionname ( type *p )
#else
#define CC_HEADER_PTR_FREE_LIST_ROUTINE(type, functionname)                \
static void functionname ();                                               \
static void functionname ( p )                                             \
type *p;
#endif

#define CC_PTR_FREE_LIST_ROUTINE(type, functionname, freefunction)         \
 CC_HEADER_PTR_FREE_LIST_ROUTINE (type, functionname)                      \
{                                                                          \
    type *next;                                                            \
                                                                           \
    while (p) {                                                            \
        next = p->next;                                                    \
        freefunction (p);                                                  \
        p = next;                                                          \
    }                                                                      \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                     \
static void functionname (void);                                           \
static void functionname (void)
#else
#define CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                     \
static void functionname ();                                               \
static void functionname ()
#endif

#define CC_PTR_FREE_WORLD_ROUTINE(type, functionname, chunklist, freelist) \
 CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                            \
{                                                                          \
    CCbigchunkptr *bp, *bpnext;                                            \
                                                                           \
    for (bp = chunklist ; bp; bp = bpnext) {                               \
        bpnext = bp->next;                                                 \
        CCutil_bigchunkfree (bp);                                          \
    }                                                                      \
    chunklist = (CCbigchunkptr *) NULL;                                    \
    freelist = (type *) NULL;                                              \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_LEAKS_ROUTINE(functionname)                          \
static int functionname (int *total, int *onlist);                         \
static int functionname (int *total, int *onlist)
#else
#define CC_HEADER_PTR_LEAKS_ROUTINE(functionname)                          \
static int functionname ();                                                \
static int functionname (total, onlist)                                    \
int *total, *onlist;
#endif

#define CC_PTR_LEAKS_ROUTINE(type,name,chunklist,freelist,field,fieldtype) \
 CC_HEADER_PTR_LEAKS_ROUTINE(name)                                         \
{                                                                          \
    int count = CC_BIGCHUNK / sizeof ( type );                             \
    int duplicates = 0;                                                    \
    type * p;                                                              \
    CCbigchunkptr *bp;                                                     \
                                                                           \
    *total = 0;                                                            \
    *onlist = 0;                                                           \
                                                                           \
    for (bp = chunklist ; bp; bp = bp->next)                               \
        (*total) += count;                                                 \
                                                                           \
    for (p = freelist ; p; p = p->next) {                                  \
        (*onlist)++;                                                       \
        p-> field = ( fieldtype ) 0;                                       \
    }                                                                      \
    for (p = freelist ; p; p = p->next) {                                  \
        if (p-> field == ( fieldtype ) 1)                                  \
            duplicates++;                                                  \
        else                                                               \
            p-> field = ( fieldtype ) 1;                                   \
    }                                                                      \
    if (duplicates) {                                                      \
        fprintf (stderr, "WARNING: %d duplicates on ptr free list \n",     \
                 duplicates);                                              \
    }                                                                      \
    return *total - *onlist;                                               \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_STATUS_ROUTINE(functionname)                         \
static int functionname (int *total, int *onlist);                         \
static int functionname (int *total, int *onlist)
#else
#define CC_HEADER_PTR_STATUS_ROUTINE(functionname)                         \
static int functionname ();                                                \
static int functionname (total, onlist)                                    \
int *total, *onlist;
#endif

#define CC_PTR_STATUS_ROUTINE(type, name, chunklist, freelist)             \
 CC_HEADER_PTR_STATUS_ROUTINE(name)                                        \
{                                                                          \
    int count = CC_BIGCHUNK / sizeof ( type );                             \
    type * p;                                                              \
    CCbigchunkptr *bp;                                                     \
                                                                           \
    *total = 0;                                                            \
    *onlist = 0;                                                           \
                                                                           \
    for (bp = chunklist ; bp; bp = bp->next)                               \
        (*total) += count;                                                 \
                                                                           \
    for (p = freelist ; p; p = p->next)                                    \
        (*onlist)++;                                                       \
    return *total - *onlist;                                               \
}


#define CC_BIGCHUNK ((int) ((1<<16)-16))

typedef struct CCbigchunkptr {
    void                 *this;
    struct CCbigchunkptr *next;
} CCbigchunkptr;


#ifdef CC_PROTOTYPE_ANSI

void
   *CCutil_allocrus (unsigned int size),
   *CCutil_reallocrus (void *ptr, unsigned int size),
    CCutil_freerus (void *p),
    CCutil_bigchunkquery (int *total, int *reserve),
    CCutil_bigchunkfree (CCbigchunkptr *bp);

int
    CCutil_reallocrus_scale (void **pptr, int *pnnum, int count, double scale,
                      unsigned int size),
    CCutil_reallocrus_count (void **pptr, int count, unsigned int size),
    CCutil_bigchunk_free_world (void);

CCbigchunkptr
   *CCutil_bigchunkalloc (void);

#else

void
   *CCutil_allocrus (),
   *CCutil_reallocrus (),
    CCutil_freerus (),
    CCutil_bigchunkquery (),
    CCutil_bigchunkfree ();

int
    CCutil_reallocrus_scale (),
    CCutil_reallocrus_count (),
    CCutil_bigchunk_free_world ();

CCbigchunkptr
    *CCutil_bigchunkalloc ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             bgetopt.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_bix_getopt (int, char **, char *);

#else

int
    CCutil_bix_getopt ();

#endif

#define CC_BIX_GETOPT_UNKNOWN -3038

extern int CCutil_bix_optind;
extern char *CCutil_bix_optarg;



/***************************************************************************/
/*                                                                         */
/*                             dheaps_i.c                                  */
/*                                                                         */
/***************************************************************************/

typedef struct CCdheap {
    double  *key;
    int     *entry;
    int     *loc;
    int     total_space;
    int     size;
} CCdheap;

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_dheap_free (CCdheap *h),
    CCutil_dheap_insert (CCdheap *h, int i),
    CCutil_dheap_delete (CCdheap *h, int i),
    CCutil_dheap_changekey (CCdheap *h, int i, double newkey);
int
    CCutil_dheap_init (CCdheap *h, int k),
    CCutil_dheap_resize (CCdheap *h, int newsize),
    CCutil_dheap_findmin (CCdheap *h),
    CCutil_dheap_deletemin (CCdheap *h);

#else

void
    CCutil_dheap_free (),
    CCutil_dheap_insert (),
    CCutil_dheap_delete (),
    CCutil_dheap_changekey ();
int
    CCutil_dheap_init (),
    CCutil_dheap_resize (),
    CCutil_dheap_findmin (),
    CCutil_dheap_deletemin ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             edg2cyc.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_edge_to_cycle (int ncount, int *elist, int *cyc);

#else

int
    CCutil_edge_to_cycle ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             edgelen.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCdatagroup {
    double  *x;
    double  *y;
    double  *z;
    int    **adj;
    int      norm;
} CCdatagroup;


#ifdef CC_PROTOTYPE_ANSI

extern int
  (*CCutil_dat_edgelen) (int i, int j, CCdatagroup *dat);
int
    CCutil_init_dat_edgelen (CCdatagroup *dat),
    CCutil_max_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_ibm_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid_ceiling_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid3d_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_geographic_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_att_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_dsjrand_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_crystal_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_matrix_edgelen (int i, int j, CCdatagroup *dat);
void
    CCutil_dsjrand_init (int maxdist, int seed),
    CCutil_freedatagroup (int ncount, CCdatagroup *dat);

#else

extern int
  (*CCutil_dat_edgelen) ();
int
    CCutil_init_dat_edgelen (),
    CCutil_max_edgelen (),
    CCutil_euclid_edgelen (),
    CCutil_ibm_edgelen (),
    CCutil_euclid_ceiling_edgelen (),
    CCutil_euclid3d_edgelen (),
    CCutil_geographic_edgelen (),
    CCutil_att_edgelen (),
    CCutil_dsjrand_edgelen (),
    CCutil_crystal_edgelen (),
    CCutil_matrix_edgelen ();
void
    CCutil_dsjrand_init (),
    CCutil_freedatagroup ();

#endif


#define CC_KD_NORM_TYPE    128            /* Kdtrees work      */
#define CC_X_NORM_TYPE     256            /* Old nearest works */
#define CC_JUNK_NORM_TYPE  512            /* Nothing works     */

#define CC_D2_NORM_SIZE      1024         /* x,y coordinates   */
#define CC_D3_NORM_SIZE      2048         /* x,y,z coordinates */
#define CC_MATRIX_NORM_SIZE  4096         /* adj matrix        */

#define CC_NORM_BITS      (CC_KD_NORM_TYPE | CC_X_NORM_TYPE | CC_JUNK_NORM_TYPE)
#define CC_NORM_SIZE_BITS (CC_D2_NORM_SIZE | CC_D3_NORM_SIZE | CC_MATRIX_NORM_SIZE)

#define CC_MAXNORM        (0 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN_CEIL (1 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN      (2 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN_3D   (3 |    CC_X_NORM_TYPE |     CC_D3_NORM_SIZE)
#define CC_IBM            (4 | CC_JUNK_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_ATT            (5 |    CC_X_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_GEOGRAPHIC     (6 |    CC_X_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_MATRIXNORM     (7 | CC_JUNK_NORM_TYPE | CC_MATRIX_NORM_SIZE)
#define CC_DSJRANDNORM    (8 | CC_JUNK_NORM_TYPE)
#define CC_CRYSTAL        (9 |    CC_X_NORM_TYPE |     CC_D3_NORM_SIZE)

#define CC_GEOGRAPHIC_SCALE (6378.388 * 3.14 / 180.0) /*    see edgelen.c   */
#define CC_ATT_SCALE (.31622)                         /*    sqrt(1/10)      */

/* For X-NORMS, scales are such that |x[i] - x[j]| * scale <= edgelen(i,j). */
/* Ggeographic is slightly off, since the fractional part of x[i] is really */
/* really minutes, not fractional degrees.                                  */



/***************************************************************************/
/*                                                                         */
/*                             fastread.c                                  */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_readint (FILE *);

#else

int
    CCutil_readint ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             genhash.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCgenhash {
    int                     nelem;
    int                     maxelem;
    int                     size;
#ifdef CC_PROTOTYPE_ANSI
    int                   (*hcmp) (void *key1, void *key2, void *u_data);
    unsigned int          (*hfunc) (void *key, void *u_data);
#else
    int                   (*hcmp) ();
    unsigned int          (*hfunc) ();
#endif
    void                   *u_data;
    double                  maxdensity;
    double                  lowdensity;
    struct CCgenhash_elem **table;
} CCgenhash;

typedef struct CCgenhash_iter {
    int                    i;
    struct CCgenhash_elem *next;
} CCgenhash_iter;

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_genhash_init (CCgenhash *h, int size,
            int (*hcmp) (void *key1, void *key2, void *u_data),
            unsigned int (*hfunc) (void *key, void *u_data),
            void *u_data, double maxdensity, double lowdensity),
    CCutil_genhash_insert (CCgenhash *h, void *key, void *data),
    CCutil_genhash_insert_h (CCgenhash *h, unsigned int hashval, void *key,
            void *data),
    CCutil_genhash_replace (CCgenhash *h, void *key, void *data),
    CCutil_genhash_replace_h (CCgenhash *h, unsigned int hashval, void *key,
                       void *data),
    CCutil_genhash_delete (CCgenhash *h, void *key),
    CCutil_genhash_delete_h (CCgenhash *h, unsigned int hashval, void *key);

unsigned int
    CCutil_genhash_hash (CCgenhash *h, void *key);

void
   *CCutil_genhash_lookup (CCgenhash *h, void *key),
   *CCutil_genhash_lookup_h (CCgenhash *h, unsigned int hashval, void *key),
   *CCutil_genhash_next (CCgenhash *h, CCgenhash_iter *iter, void **key,
            int *keysize);

void
    CCutil_genhash_u_data (CCgenhash *h, void *u_data),
    CCutil_genhash_free (CCgenhash *h, void (*freefunc)(void *key, void *data,
            void *u_data)),
    CCutil_genhash_start (CCgenhash *h, CCgenhash_iter *iter);

#else

int
    CCutil_genhash_init (),
    CCutil_genhash_insert (),
    CCutil_genhash_insert_h (),
    CCutil_genhash_replace (),
    CCutil_genhash_replace_h (),
    CCutil_genhash_delete (),
    CCutil_genhash_delete_h ();

unsigned int
    CCutil_genhash_hash ();

void
   *CCutil_genhash_lookup (),
   *CCutil_genhash_lookup_h (),
   *CCutil_genhash_next ();

void
    CCutil_genhash_u_data (),
    CCutil_genhash_free (),
    CCutil_genhash_start ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             getdata.c                                   */
/*                                                                         */
/***************************************************************************/

#define CC_MASTER_NO_DAT  100
#define CC_MASTER_DAT     101

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_getdata (char *datname, int binary_in, int innorm, int *ncount,
            CCdatagroup *dat),
    CCutil_writemaster (char *mastername, int ncount, CCdatagroup *dat,
            int *perm),
    CCutil_getmaster (char *mastername, int *ncount, CCdatagroup *dat,
            int **perm),
    CCutil_getnodeweights (char *weightname, int ncount, int weight_limit,
            double **wcoord),
    CCutil_gettsplib (char *datname, int *ncount, CCdatagroup *dat),
    CCutil_datagroup_perm (int ncount, CCdatagroup *dat, int *perm),
    CCutil_getedgelist (int ncount, char *fname, int *ecount, int **elist,
            int **elen),
    CCutil_getedgelist_n (int *ncount, char *fname, int *ecount, int **elist,
            int **elen),
    CCutil_getcycle_edgelist (int ncount, char *cyclename, int *outcycle),
    CCutil_getcycle (int ncount, char *cyclename, int *outcycle),
    CCutil_getedges_double (int *ncount, char *fname, int *ecount, int **elist,
            double **elen, int binary_in),
    CCutil_writeedges (int ncount, char *outedgename, int ecount, int *elist,
            CCdatagroup *dat),
    CCutil_writecycle_edgelist (int ncount, char *outedgename, int *cycle,
            CCdatagroup *dat),
    CCutil_writecycle (int ncount, char *outcyclename, int *cycle),
    CCutil_writeedges_double (int ncount, char *outedgename, int ecount,
            int *elist, double *elen, int binary_out);

#else

int
    CCutil_getdata (),
    CCutil_writemaster (),
    CCutil_getmaster (),
    CCutil_getnodeweights (),
    CCutil_gettsplib (),
    CCutil_datagroup_perm (),
    CCutil_getedgelist (),
    CCutil_getedgelist_n (),
    CCutil_getcycle_edgelist (),
    CCutil_getcycle (),
    CCutil_getedges_double (),
    CCutil_writeedges (),
    CCutil_writecycle_edgelist (),
    CCutil_writecycle (),
    CCutil_writeedges_double ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             priority.c                                  */
/*                                                                         */
/***************************************************************************/

typedef struct CCpriority {
    CCdheap   heap;
    union pri_data {
        void *data;
        int  next;
    }        *pri_info;
    int       space;
    int       freelist;
} CCpriority;

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_priority_free (CCpriority *pri),
    CCutil_priority_delete (CCpriority *pri, int handle),
    CCutil_priority_changekey (CCpriority *pri, int handle, double newkey),
   *CCutil_priority_findmin (CCpriority *pri, double *keyval),
   *CCutil_priority_deletemin (CCpriority *pri, double *keyval);

int
    CCutil_priority_init (CCpriority *pri, int k),
    CCutil_priority_insert (CCpriority *pri, void *data, double keyval);

#else

void
    CCutil_priority_free (),
    CCutil_priority_delete (),
    CCutil_priority_changekey (),
   *CCutil_priority_findmin (),
   *CCutil_priority_deletemin ();

int
    CCutil_priority_init (),
    CCutil_priority_insert ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             safe_io.c                                   */
/*                                                                         */
/***************************************************************************/

#define CC_SBUFFER_SIZE (4000)
#define CC_SFNAME_SIZE (32)

typedef struct CC_SFILE {
    int           status;
    int           desc;
    int           chars_in_buffer;
    int           current_buffer_char;     /* only used for reading */
    int           bits_in_last_char;       /* writing: number of empty bits in
                                            * buffer[chars_in_buffer];
                                            * reading: number of full bits in
                                            * buffer[?] */
    int           pos;
    char          fname[CC_SFNAME_SIZE];
    unsigned char buffer[CC_SBUFFER_SIZE];
} CC_SFILE;

#ifdef CC_PROTOTYPE_ANSI

CC_SFILE
   *CCutil_sopen (char *f, char *s),
   *CCutil_sdopen (int d, char *s);

int
    CCutil_swrite (CC_SFILE *f, unsigned char *buf, int size),
    CCutil_swrite_bits (CC_SFILE *f, unsigned int x, int xbits),
    CCutil_swrite_char (CC_SFILE *f, unsigned char x),
    CCutil_swrite_string (CC_SFILE *f, unsigned char *x),
    CCutil_swrite_short (CC_SFILE *f, unsigned short x),
    CCutil_swrite_int (CC_SFILE *f, unsigned int x),
    CCutil_swrite_double (CC_SFILE *f, double x),
    CCutil_sread (CC_SFILE *f, unsigned char *buf, int size),
    CCutil_sread_bits (CC_SFILE *f, unsigned int *x, int xbits),
    CCutil_sread_char (CC_SFILE *f, unsigned char *x),
    CCutil_sread_string (CC_SFILE *f, unsigned char *x, int maxlen),
    CCutil_sread_short (CC_SFILE *f, unsigned short *x),
    CCutil_sread_short_r (CC_SFILE *f, unsigned short *x),
    CCutil_sread_int (CC_SFILE *f, unsigned int *x),
    CCutil_sread_int_r (CC_SFILE *f, unsigned int *x),
    CCutil_sread_double (CC_SFILE *f, double *x),
    CCutil_sread_double_r (CC_SFILE *f, double *x),
    CCutil_sflush (CC_SFILE *f),
    CCutil_stell (CC_SFILE *f),
    CCutil_sseek (CC_SFILE *f, int offset),
    CCutil_srewind (CC_SFILE *f),
    CCutil_sclose (CC_SFILE *f),
    CCutil_sbits (unsigned int x),
    CCutil_sdelete_file (char *fname),
    CCutil_sdelete_file_backup (char *fname);

#else

CC_SFILE
   *CCutil_sopen (),
   *CCutil_sdopen ();

int
    CCutil_swrite (),
    CCutil_swrite_bits (),
    CCutil_swrite_char (),
    CCutil_swrite_string (),
    CCutil_swrite_short (),
    CCutil_swrite_int (),
    CCutil_swrite_double (),
    CCutil_sread (),
    CCutil_sread_bits (),
    CCutil_sread_char (),
    CCutil_sread_string (),
    CCutil_sread_short (),
    CCutil_sread_short_r (),
    CCutil_sread_int (),
    CCutil_sread_int_r (),
    CCutil_sread_double (),
    CCutil_sread_double_r (),
    CCutil_sflush (),
    CCutil_stell (),
    CCutil_sseek (),
    CCutil_srewind (),
    CCutil_sclose (),
    CCutil_sbits (),
    CCutil_sdelete_file (),
    CCutil_sdelete_file_backup ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             sortrus.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_int_array_quicksort (int *len, int n),
    CCutil_int_perm_quicksort (int *perm, int *len, int n),
    CCutil_double_perm_quicksort (int *perm, double *len, int n),
    CCutil_rselect (int *arr, int l, int r, int m, double *coord);

char
   *CCutil_linked_radixsort (char *data, char *datanext, char *dataval,
            int valsize);

#else

void
    CCutil_int_array_quicksort (),
    CCutil_int_perm_quicksort (),
    CCutil_double_perm_quicksort (),
    CCutil_rselect ();

char
   *CCutil_linked_radixsort ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             urandom.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_sprand (int);
int
    CCutil_lprand (void);

#else

void
    CCutil_sprand ();
int
    CCutil_lprand ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             util.c                                      */
/*                                                                         */
/***************************************************************************/


#ifdef CC_PROTOTYPE_ANSI

char
   *CCutil_strrchr (char *s, int c);

unsigned int
    CCutil_nextprime (unsigned int x);

int
    CCutil_our_gcd (int a, int b);

#else

char
   *CCutil_strrchr ();

unsigned int
    CCutil_nextprime ();

int
    CCutil_our_gcd ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             zeit.c                                      */
/*                                                                         */
/***************************************************************************/


#ifdef CC_PROTOTYPE_ANSI

double
    CCutil_zeit (void),
    CCutil_real_zeit (void);

#else

double
    CCutil_zeit (),
    CCutil_real_zeit ();

#endif


#endif /* __UTIL_H */
