/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Y1 zeros and extremum on [0, 52] interval
Generated by SageMath:
```python
R120 = RealField(120)

zeros = []

mp.prec = 150

step = mpf("0.1")
epsilon = mpf("1e-35")
x = mpf("1.25")

previous_zero = R120(0)
y1_zeros = []

while x < mpf("52.0"):
    f1 = bessely(1, x)
    f2 = bessely(1, x + step)
    if f1 * f2 < 0:
        zero = findroot(lambda t: bessely(1, t), (x, x + step), solver='secant', tol=mp.mpf("1e-41"))
        previous_zero = zero
        y1_zeros.append(zero)
    if previous_zero is not None and abs(x - mpf(f'{round(x)}')) < epsilon:
        zeros.append(previous_zero)
    x += step

y1_extrema = []

x = mpf("1.25")
while x < mpf("52.0"):
    d1 = mp.diff(lambda t: bessely(1, t), x)
    d2 = mp.diff(lambda t: bessely(1, t), x + step)
    if d1 * d2 < 0:
        extremum = findroot(lambda t: mp.diff(lambda u: bessely(1, u), t), (x, x + step), solver='secant', tol=mp.mpf("1e-41"))
        y1_extrema.append(extremum)
    x += step

y1_zeros.extend(y1_extrema)

y1_zeros = sorted(y1_zeros)

print(f"pub(crate) static Y1_ZEROS: [(u64, u64); {len(y1_zeros)}] = [")
print(f"(0x0, 0x0),")
for z in y1_zeros:
    k = split_double_double(z)
    hi = double_to_hex(k[1])
    lo = double_to_hex(k[0])
    print(f"({lo}, {hi}),")

print("];")
```
**/
pub(crate) static Y1_ZEROS: [(u64, u64); 33] = [
    (0x0, 0x0), // not really used, just a stab to avoid indices messing
    (0xbc8bd1e50d219bfd, 0x400193bed4dff243),
    (0x3c53bac0714e4129, 0x400d76d4affba175),
    (0x3cbdfe7bac228e8c, 0x4015b7fe4e87b02e),
    (0x3ca7960b6b1c46ac, 0x401bc41890588553),
    (0x3cb479cc068d9046, 0x40213127ae6169b4),
    (0x3cc8f4ba5d68e440, 0x40243f2ee51e8c7e),
    (0x3c80fc786ce06080, 0x40277f9138d43206),
    (0xbcaf6ef7a3571593, 0x402a924ee4a3e52c),
    (0xbcc5e091a50f8e05, 0x402dcb7d88de848b),
    (0x3cc07320221cd5e5, 0x403070a7a43daae6),
    (0xbcda1ee4c5487ede, 0x40320b1c695f1e3b),
    (0xbcd2903124fef7e3, 0x4033971a15717510),
    (0x3cd391b14410528f, 0x40353025492188cd),
    (0xbcc15ec09721b746, 0x4036bcefd7de87a3),
    (0x3cb52f75f025b205, 0x403854fa303820ca),
    (0x3cb6f57f7696f493, 0x4039e262715f12a9),
    (0xbcbcf130fbea3b24, 0x403b79acee8cfb7d),
    (0xbc912142b10a5c65, 0x403d079247e8f51b),
    (0xbc9e7a77047d6166, 0x403e9e480605283c),
    (0x3cb1452eb07cd937, 0x40401649819af8fa),
    (0xbce96beabef7ecf4, 0x4040e16907f8fb56),
    (0x3cec6086fb5dd335, 0x4041a8b8a142d536),
    (0x3cd2481e87adfe57, 0x404273a7b35a7aff),
    (0x3cd7df5b6f701c7a, 0x40433b1ac0375e31),
    (0x3cda8ffacaac8461, 0x404405e18393afb5),
    (0xbce5b5acaff0a867, 0x4044cd72d2adfb0c),
    (0x3cbfe463face2c1c, 0x4045981787d668db),
    (0xbcefcba6ea61df1b, 0x40465fc2f7ca5b81),
    (0xbce26390f25f01cb, 0x40472a4a85cc317e),
    (0xbcbba46ca6ef9b6f, 0x4047f20cbfc32967),
    (0xbcdcc667e557a177, 0x4048bc7b10ed3960),
    (0x3cea473d4f209faf, 0x4049845158040451),
];

/**
Value at zero or extremum of Y1 belongs to [Y1_ZEROS]
Generated by MPFR:
```text
let mut arr = vec![];
for zeros in Y1_ZEROS.iter() {
    if zeros.1 == 0 {
        arr.push(0);
    } else {
        let mpfr = Float::with_val(107, f64::from_bits(zeros.1)).y1();
        arr.push(mpfr.to_f64().to_bits());
    }
}
println!(
    "arr: [{}]",
    arr.iter()
        .map(|x| format!("0x{:016x}", x))
        .collect::<Vec<_>>()
        .join(", ")
);
```
**/
pub(crate) static Y1_ZEROS_VALUES: [u64; 33] = [
    0x0000000000000000,
    0x3c7cf9f8d5e1a475,
    0x3fdaabb4011ed330,
    0x3ca46a40b234169c,
    0xbfd36732d4b96094,
    0xbc963bc010b45f46,
    0x3fd00ef3745e0e3c,
    0x3c5f96d32c02f147,
    0xbfcc075da85beb4f,
    0x3ca213201464272b,
    0x3fc931a5a0ae5aa0,
    0xbcb39d4c41d5839f,
    0xbfc713fc51664c74,
    0xbcab20138dd047ec,
    0x3fc56b97f8091ac5,
    0x3c8b67dbfd3bd179,
    0xbfc413644356a52b,
    0x3c919e290514c619,
    0x3fc2f4e70d6c7e01,
    0xbc719366994a86bc,
    0xbfc20198200b699d,
    0x3cbbec2345ac18e2,
    0x3fc12f9870d68e18,
    0x3ca334eb74fdcfd1,
    0xbfc077eede4a0d89,
    0xbcaaca291aee3c35,
    0x3fbfab0b166d23d8,
    0x3c8ef92aa9c8e53c,
    0xbfbe891b327da16d,
    0x3cb13e3c695ea088,
    0x3fbd84391bb2748d,
    0xbcaa1c861fdd1438,
    0xbfbc97d79918527d,
];

/**
Series expansion at point for Y1

Generated by SageMath and Sollya:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = 2 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.05
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(y1_zeros)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 250;
bessel_y1 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_y1(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RR(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

degree = 18

print(f"pub(crate) static Y1F_COEFFS: [[u64;{degree + 1}]; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print("[")
    for c in coeffs:
        print(double_to_hex(c) + ",")
    print("],")
print("];")
```
**/
pub(crate) static Y1F_COEFFS: [[u64; 19]; 32] = [
    [
        0x3bdca2ee18606a4b,
        0x3fe0aa48442f014b,
        0xbfbe56f82217b8f4,
        0xbfa0d2af4e932400,
        0xbf73a6dec3726cd5,
        0x3f7e671c7d12ea48,
        0xbf65429dc5c0e9d4,
        0x3f517ab4af4655e4,
        0xbf40b2d8647a250d,
        0x3f2eea7b1b675766,
        0xbf1c3fb728e7d2ff,
        0x3f09d1da72e12f44,
        0xbef7964bf8511e22,
        0x3ee57c2a83e1f972,
        0xbed33f4211a00375,
        0x3ec02bcdac2103fd,
        0xbea6fefcf033ab9d,
        0x3e874128ed97d3bb,
        0xbe57d5b1eac16658,
    ],
    [
        0x3fdaabb4011ed330,
        0x3c54da7c52fcf446,
        0xbfc8b45babe797b6,
        0x3f8e147099a6f00d,
        0x3f88c5af1eeb2143,
        0xbf4133fa47d8ea48,
        0xbf3bf8af93e7a2f0,
        0x3f021d64bd4e2cd8,
        0x3eb44d2c32fdaf23,
        0x3eb14c3b9e7960c1,
        0xbe9b8ee25c629be6,
        0x3e7a85b5b497dc6c,
        0xbe5bfa422fb8d949,
        0x3e3f0ad81d293f5a,
        0xbe20e6844c6faba7,
        0x3e0214b2d826d072,
        0xbde4ff658967d425,
        0x3dcb549cdc774a83,
        0xbda6b5df4d9c1682,
    ],
    [
        0x3b7ff35240713789,
        0xbfd5c7c556f0c19a,
        0x3fa00b9f8571ca1f,
        0x3faa15d92dfe3e27,
        0xbf710a329e2c23f5,
        0xbf61be6db9923ac9,
        0x3f2337c7e138eb84,
        0x3f085b940eb5f37f,
        0xbec80619146a1e65,
        0xbea255e6cf4b3254,
        0x3e5b62ccdc392c5a,
        0x3e380b1a5a61e6b5,
        0xbdfa7ec7fd0d2925,
        0x3d840d04ff01d1b2,
        0xbd938dc1b2e33eca,
        0x3d74839c586126ca,
        0xbd4b045bc7ad769b,
        0x3d261d10a8575c45,
        0xbd052a6cc14bcc54,
    ],
    [
        0xbfd36732d4b96094,
        0x3b3886a5ed6fd628,
        0x3fc3001c8002caf8,
        0xbf7bf5a03bab4999,
        0xbf8751ea028c1953,
        0x3f423874cd8d0402,
        0x3f364f6610d6493b,
        0xbef02978de38394f,
        0xbed72f0766d0d9c7,
        0x3e8f2081874e556c,
        0x3e6defd5dce91973,
        0xbe2205c70046a2c7,
        0xbdfb6432eb3ab7ea,
        0x3db028a1c0572973,
        0x3d807791dcab03a0,
        0xbd29778204deee13,
        0xbd08342db2e7148e,
        0x3cc898efb37f9dad,
        0xbc84e2adc305e2ab,
    ],
    [
        0xbac1435819592d4c,
        0x3fd15f993fceab5c,
        0xbf902b3933cf21b1,
        0xbfa6395dfe49fcd4,
        0x3f63ced2a2e69180,
        0x3f607a678d6000bb,
        0xbf1b50d7e1d3201e,
        0xbf06f7bab104f34b,
        0x3ec176e72bf94a3a,
        0x3ea2becb2b6bacd1,
        0xbe5a384eebfb23c2,
        0xbe341e7a921f7f66,
        0x3de9e3284b918a26,
        0x3dbec40b21f2c78f,
        0xbd726865da6190a9,
        0xbd416f4fe7eed351,
        0x3cf3160bd2bd6c64,
        0x3cbf6d61c945b95c,
        0xbc706809636e0aec,
    ],
    [
        0x3fd00ef3745e0e3c,
        0x3aff192f298c81c3,
        0xbfbfcdacdda138f2,
        0x3f706cc34cd829fa,
        0x3f84641bb10c16cb,
        0xbf37fac943e2a16d,
        0xbf34769ed32e14a2,
        0x3ee80608ecda1508,
        0x3ed5cc8242d77e23,
        0xbe888c8f2538feb8,
        0xbe6ce5908c1e5174,
        0x3e1ed16257e17417,
        0x3dfa30d623eda066,
        0xbdaa5076123e3ecf,
        0xbd814cd297d2be7e,
        0x3d306166947e23e9,
        0x3d01635f73179569,
        0xbcaeafcf4c2f127b,
        0xbc7b0828175d92fa,
    ],
    [
        0x3aba1488e1b7782d,
        0xbfcdc14ea14e89f9,
        0x3f84429fef5b5fbd,
        0x3fa367d7d608e4ba,
        0xbf59d6eb2bc49e35,
        0xbf5dc4f991b3db86,
        0x3f1315ec04d6e6bb,
        0x3f0571814a1aa2f5,
        0xbeba2977fa42f00f,
        0xbea1e864230850b8,
        0x3e54a7b82d3fa1e5,
        0x3e33906609f9fe4c,
        0xbde549e8b0e16969,
        0xbdbe32cf2ce99d6f,
        0x3d6eff542dd345c3,
        0x3d415e2a9c2f4933,
        0xbcf0d48dde3c3ffe,
        0xbcbeac3c36b4bce2,
        0x3c6af1612c5ddab0,
    ],
    [
        0xbfcc075da85beb4f,
        0xbafcfa84f4024782,
        0x3fbbdeb6ff9f55e1,
        0xbf661eefb74da882,
        0xbf8229ea914b846e,
        0x3f30cbcc6778fd37,
        0x3f32aa59f5091f7b,
        0xbee1c15d5251ae54,
        0xbed4583f15abd654,
        0x3e831d151a12624a,
        0x3e6b74e57c21e022,
        0xbe19044f1339b061,
        0xbdf93b1ec70c7bbc,
        0x3da61a4e437e8105,
        0x3d80d4305f038451,
        0xbd2c3aad6f3b35c7,
        0xbd010dec3a02c58c,
        0x3cab15901b6d0925,
        0x3c7ab2531f00c501,
    ],
    [
        0xbab392a85abdc950,
        0x3fca7022be084d99,
        0xbf7c650b6b83109a,
        0xbfa163191c30aa62,
        0x3f526b045287ddca,
        0x3f5b17602840abf5,
        0xbf0c0a9cee3c8429,
        0xbf03e398cbc472de,
        0x3eb3f35db1ff19f5,
        0x3ea0e9b612dbc0ea,
        0xbe5056babcd79a11,
        0xbe32c1a8c8d768b1,
        0x3de161b6a84838d0,
        0x3dbd4ca9d2d67d78,
        0xbd69fdd67a999eab,
        0xbd4101919ce84a07,
        0x3cecd91fa7851496,
        0x3cbe3f8588ebbfdf,
        0xbc67a4499c96e38d,
    ],
    [
        0x3fc931a5a0ae5aa0,
        0x3afa23fd08be9891,
        0xbfb919c8a3f203fa,
        0x3f602a38da6262a9,
        0x3f807ced48910819,
        0xbf2900f33a00690a,
        0xbf31278d46fd153c,
        0x3edb2595529cf19f,
        0x3ed2f7c2d608e0bb,
        0xbe7e212d23787793,
        0xbe69f3fcf3631e9c,
        0x3e144fbf033f1974,
        0x3df82268e7ab0cdb,
        0xbda26cc2714815d4,
        0xbd80418b35c32375,
        0x3d28122e50410f0a,
        0x3d009aba27e11464,
        0xbca78943175d4e84,
        0xbc7a379f959c0224,
    ],
    [
        0x3aaf25ce7e30cbc6,
        0xbfc80781c32422e7,
        0x3f754eda697a0098,
        0x3f9fbe6df840847f,
        0xbf4be318d61276e1,
        0xbf58efee4094379c,
        0x3f059145b4f0e4dd,
        0x3f0282d26a74c382,
        0xbeaf56c29d9ad6c8,
        0xbe9fdd03174f6b47,
        0x3e4a44a7907d0ec6,
        0x3e31df6533090779,
        0xbddc96e9cb6ee22b,
        0xbdbc3439a99213c4,
        0x3d65d387fc8083e0,
        0x3d40830db4ec8a6e,
        0xbce8ad426f9ce3f5,
        0xbcbd93c0cf35d116,
        0x3c649b19a5449ffa,
    ],
    [
        0xbfc713fc51664c74,
        0xbaf73aab14face16,
        0x3fb7049760cde490,
        0xbf58ef5f1cbe4874,
        0xbf7e5f53caf3bead,
        0x3f237b0b62ddadd1,
        0x3f2fd3bac08286da,
        0xbed5789803de3adb,
        0xbed1c0faa8999393,
        0x3e7845b49b063dc7,
        0x3e6886872800e226,
        0xbe10b03677687883,
        0xbdf7049d17bd230b,
        0x3d9edd9ca057f252,
        0x3d7f445f42a168e6,
        0xbd24866878075342,
        0xbd0015a37275b46d,
        0x3ca463bd3d4059be,
        0x3c79974848138496,
    ],
    [
        0xbaa9a62f9227c851,
        0x3fc62d94d97e859c,
        0xbf70bf614807033c,
        0xbf9d5f857a2a6107,
        0x3f46081b0b7fe572,
        0x3f57307b03e248f8,
        0xbf0132c0aa83d0db,
        0xbf0154ed4598d2e4,
        0x3ea94f64f476e3f5,
        0x3e9e1272585385c0,
        0xbe4588c758dd66db,
        0xbe31021cdd7a4f3a,
        0x3dd7cfa7a39f5d48,
        0x3dbb0e00d41ec645,
        0xbd6276c9a451cdb1,
        0xbd3fe8cf17671ae1,
        0x3ce52f1a6f7ae06f,
        0x3cbcc2eb893d62ce,
        0xbc61f4c0af8bd0fb,
    ],
    [
        0x3fc56b97f8091ac5,
        0x3af48a947d2475cd,
        0xbfb560fcc8c08469,
        0x3f53fafa39618883,
        0x3f7c49141623372f,
        0xbf1f69980694fd17,
        0xbf2dc5f848aa9d33,
        0x3ed178fc979b779d,
        0x3ed0b494a4bafca8,
        0xbe73fc3884c243a5,
        0xbe673afb9fb48ff7,
        0x3e0bd903464b077a,
        0x3df5f3bafabcdabe,
        0xbd9a1c27612b5f03,
        0xbd7e04553366c10e,
        0x3d219970f1564c7c,
        0x3cff128fbd867c78,
        0xbca1b4d2be53f3ad,
        0xbc78e13fb654b036,
    ],
    [
        0x3aa5951bb8e2b477,
        0xbfc4b2a38f1ab9b4,
        0x3f6b3878aadeb34d,
        0x3f9b750d89a9b35f,
        0xbf41f6911725a956,
        0xbf55beee6fd51c8a,
        0x3efc3625d7a65087,
        0x3f005375a588a71f,
        0xbea4ee5e4e7cafc0,
        0xbe9c7b3d81b5dc31,
        0x3e41fce14f464e1e,
        0x3e30346643a98dcb,
        0xbdd41c86191a49ce,
        0xbdb9eed9da04017a,
        0x3d5f8cee5e5b42b4,
        0x3d3ec41075d33352,
        0xbce24e44459e28b0,
        0xbcbbe16f7d769c15,
        0x3c5f670ad9138f1f,
    ],
    [
        0xbfc413644356a52b,
        0xbaf22d9ab9060f8f,
        0x3fb40bb88c6f2b85,
        0xbf5078d13cfc400e,
        0xbf7a9191262ab9d5,
        0x3f1a005297618f35,
        0x3f2c0cbad847a60e,
        0xbecd1a72e7c35fa0,
        0xbecf9a2654099c0b,
        0x3e70c6b06e20d1c0,
        0x3e66136d6425acf0,
        0xbe0797767778226d,
        0xbdf4f77b30ed58c3,
        0x3d96572059bf2445,
        0x3d7cd12649b82d6f,
        0xbd1e6ce514a88f2d,
        0xbcfdfabaf5c37514,
        0x3c9eea8202989176,
        0x3c782260f7596e02,
    ],
    [
        0xbaa27e57c2b07d4b,
        0x3fc37aaceac987b9,
        0xbf66afe4fe0bc0f7,
        0xbf99de7a33bc3a97,
        0x3f3e024f567ac487,
        0x3f548843c426abe0,
        0xbef7a8e14711c0f4,
        0xbefeeceb341ad81c,
        0x3ea1a743e05b383f,
        0x3e9b143d39c8eb5f,
        0xbe3e8e00011fabc3,
        0xbe2ef28e31ff924c,
        0x3dd137a1bd136742,
        0x3db8e0878264a773,
        0xbd5b3dc655a5a5f4,
        0xbd3da652e8239897,
        0x3cdfe34eace42448,
        0x3cbafd0cc7251807,
        0xbc5b9b0102453020,
    ],
    [
        0x3fc2f4e70d6c7e01,
        0x3af022defda0ec45,
        0xbfb2ef24d6f7526a,
        0x3f4bc33c9dc6ec82,
        0x3f7920414ee2acbe,
        0xbf15f9173916a219,
        0xbf2a94fdbdcec471,
        0x3ec8b309990f94db,
        0x3ece087ff4517bd5,
        0xbe6ca22ab12c685c,
        0xbe650d1f28632753,
        0x3e044415529c950b,
        0x3df411b8a7d9d1bc,
        0xbd9354e8c7a8bfd7,
        0xbd7bb16e8ee8c711,
        0x3d1a881fddcb8d86,
        0x3cfcecee70233b69,
        0xbc9b2b6cccd3802a,
        0xbc77637662fa6ba8,
    ],
    [
        0x3aa00f5dbb23e90b,
        0xbfc2740819f1caaa,
        0x3f6349369dc780bb,
        0x3f98868d7401bf2e,
        0xbf398cd1bebe1445,
        0xbf537eef9aadeee2,
        0x3ef43394c95b2d29,
        0x3efd6dfcdb026013,
        0xbe9e448fbc8a1c95,
        0xbe99d764ee07a6b7,
        0x3e3a53958c8a71d8,
        0x3e2da0e1c86368ce,
        0xbdcdd7f914e496e5,
        0xbdb7e67ff45daf48,
        0x3d57c2e32861f41c,
        0x3d3c96e18ab6db69,
        0xbcdc0099b11f0478,
        0xbcba1dfeafeb6e19,
        0x3c586b4c940f74bc,
    ],
    [
        0xbfc20198200b699d,
        0xbaecc875d54af9d0,
        0x3fb1fd242a74e630,
        0xbf47cf261dfbf19a,
        0xbf77e4820ec1dde4,
        0x3f12e1bd281dfcba,
        0x3f2950bb06c6fdf9,
        0xbec54a38ab6af51a,
        0xbecca94f38024fb4,
        0x3e68c7e75971843f,
        0x3e6423fc7e24ed40,
        0xbe019fe1d8a6e0d8,
        0xbdf34198c7517f5a,
        0x3d90e78c95f157aa,
        0x3d7aa74c4042e051,
        0xbd1756942b9afcaf,
        0xbcfbedc3e7dae4e4,
        0x3c980b9567289463,
        0x3c76a9e024cc6a52,
    ],
    [
        0xba9c33661811b8ff,
        0x3fc192f2627a74e3,
        0xbf60a846a83fecf2,
        0xbf975eceaabf7f86,
        0x3f3617c581be35b1,
        0x3f529934b7a84483,
        0xbef18123e8751889,
        0xbefc1f05a2d85150,
        0x3e9a4e0bc09262e9,
        0x3e98be81ad44b8a4,
        0xbe36f73795dfb5c7,
        0xbe2c70ab155167d1,
        0x3dca26218cc79400,
        0x3db7011269271056,
        0xbd54ec138a5f86cd,
        0xbd3b98bf6fa2fe47,
        0x3cd8c95d73f0c84c,
        0x3cb948b2dd021429,
        0xbc55bd7c63fa9765,
    ],
    [
        0x3fc12f9870d68e18,
        0x3ae9cd1ac1fa64f9,
        0xbfb12c11811945f9,
        0x3f44b638f21f0f76,
        0x3f76d2a897d58353,
        0xbf10732e5458ba20,
        0xbf2835929300df3e,
        0x3ec297283816a814,
        0x3ecb73adedf11a1f,
        0xbe65b455b903b389,
        0xbe6353f0797a3bf1,
        0x3dfefc9ac10b87d9,
        0x3df2853545ffa79d,
        0xbd8dd8945079a88f,
        0xbd79b28860cd63f8,
        0x3d14b29ba2797832,
        0x3cfaff02f362ca7e,
        0xbc956d8436ee55ed,
        0xbc75f881bb0137f5,
    ],
    [
        0x3a9900b85a085cfa,
        0xbfc0cf3ee98f769b,
        0x3f5d26e7af251f79,
        0x3f965d05948a946a,
        0xbf335959b8482e40,
        0xbf51cff175d05c2a,
        0x3eeeb59416879104,
        0x3efaf7544eeac751,
        0xbe9720522bb1fa69,
        0xbe97c41261703475,
        0x3e343fa0ea5ba663,
        0x3e2b5e23abb21a5f,
        0xbdc722397b59adb6,
        0xbdb62f213532a0b8,
        0x3d5294a89e377c98,
        0x3d3aac95aead6ada,
        0xbcd61abb584f3fc2,
        0xbcb87f3345758fd5,
        0x3c537a3b70fc94b7,
    ],
    [
        0xbfc077eede4a0d89,
        0xbae73fb2e67b1968,
        0x3fb0751548b2924d,
        0xbf423b5d46a73864,
        0xbf75e2467c8fb832,
        0x3f0cfe5c189d6e4d,
        0x3f273bbd8c7aef2c,
        0xbec06974d3d04263,
        0xbeca6081d36e6a0b,
        0x3e6334a83cf5d21d,
        0x3e6299571cb4bb1e,
        0xbdfb7f5bc046450f,
        0xbdf1da63b49ed896,
        0x3d8a92885fb339ae,
        0x3d78d1d6e93bb23d,
        0xbd127ea4434f9fb5,
        0xbcfa20dab6b920e9,
        0x3c93389d892643a0,
        0x3c7550b88147fd02,
    ],
    [
        0xba96582ab366c758,
        0x3fc0230ba90f2871,
        0xbf59ca16f0c9734e,
        0xbf9579c1bdbcfc99,
        0x3f3120ecfac5c017,
        0x3f511dd26bbe2946,
        0xbeeb37e7c9a57147,
        0xbef9f01e7c19098c,
        0x3e94887fe7a88a4d,
        0x3e96e3723883fe87,
        0xbe3204b644d485a1,
        0xbe2a659b13b69c6b,
        0x3dc4a40c8498625a,
        0x3db56f0212f628e4,
        0xbd50a0fef4ac5a44,
        0xbd39d1d92cf50973,
        0x3cd3d93917ae0666,
        0x3cb7c222c421cbf8,
        0xbc518e27cddeecfd,
    ],
    [
        0x3fbfab0b166d23d8,
        0x3ae50cd9856106aa,
        0xbfafa65c1ce7ebd6,
        0x3f4035bf503ffc1f,
        0x3f750d1b04713c41,
        0xbf09cd14a92842a1,
        0xbf265d504af5d8fe,
        0x3ebd3feeb33d9cae,
        0x3ec96a257062f750,
        0xbe61254f302b04d0,
        0xbe61f11585e02bfc,
        0x3df89a7674827723,
        0x3df13f0ba458182e,
        0xbd87d67ae3559fb7,
        0xbd78038124810666,
        0x3d10a3cffab7b16e,
        0x3cf952a4679b4020,
        0xbc9158ffabf6b26f,
        0xbc74b2e8c7ca451f,
    ],
    [
        0x3a9422b204fbf27f,
        0xbfbf13fb0c0e6fcd,
        0x3f5706ed3d935d00,
        0x3f94af74cbd77bef,
        0xbf2e9a9e66e5a792,
        0xbf507ec9ed824fcb,
        0x3ee856d4518ab29c,
        0x3ef9040de830648a,
        0xbe9262f69c56c4a2,
        0xbe9618c94a54555e,
        0x3e3029d2c8bd8b0e,
        0x3e2983bca06d479e,
        0xbdc28e29fd7e309a,
        0xbdb4beea8ebaabe8,
        0x3d4df87d00b82fa1,
        0x3d39076f370434b4,
        0xbcd1ef67bd03c16c,
        0xbcb7115a994eb5d5,
        0x3c4fd28ad1effa7c,
    ],
    [
        0xbfbe891b327da16d,
        0xbae325aba995f36e,
        0x3fae8533ce07bdb8,
        0xbf3d1253218e31b0,
        0xbf744e6826476498,
        0x3f07271a9b5e3cb4,
        0x3f2595b697c8ec04,
        0xbeba46b03ecb3892,
        0xbec88c173e076203,
        0x3e5ed9b1754f626a,
        0x3e615891ef312cd4,
        0xbdf62ca3527c988f,
        0xbdf0b14767922479,
        0x3d85879b58ff9d05,
        0x3d7745bb7346aea9,
        0xbd0e20fb122a7c2a,
        0xbcf89354d05f7b8c,
        0x3c8f7da941b1f5c8,
        0x3c741ef462ba56a3,
    ],
    [
        0xba923c4506ec812e,
        0x3fbe018dac1c17e3,
        0xbf54b994dd05c1fb,
        0xbf93f9e0db07e7ef,
        0x3f2b8e55b75b13ab,
        0x3f4fdf68a78bb3d2,
        0xbee5ee9d17106a07,
        0xbef82ee6dfdfedd8,
        0x3e90962d7f6d601e,
        0x3e9560edce7d4b08,
        0xbe2d34381cff4d39,
        0xbe28b5a0e715b17d,
        0x3dc0cab951aa6b0b,
        0x3db41d218be1cc79,
        0xbd4b2d2a51de7089,
        0xbd384c06737cf8d9,
        0x3cd04c96e113bd0a,
        0x3cb66c4b3dc74284,
        0xbc4cfe4908cedcb8,
    ],
    [
        0x3fbd84391bb2748d,
        0x3ae17f232c3596e7,
        0xbfad80edb3c4ea05,
        0x3f3a431f5421f7ef,
        0x3f73a282fe7b63a8,
        0xbf04ed653e607bdc,
        0xbf24e15832bda3c4,
        0x3eb7c5832dd13719,
        0x3ec7c2b80da8df7e,
        0xbe5bf160ccd30135,
        0xbe60cda2141b0afb,
        0x3df41d31d25837e8,
        0x3df02f6ca822716e,
        0xbd83901f0d351db4,
        0xbd7696cdd91c27b7,
        0x3d0b6e1487e46414,
        0x3cf7e1bb61dba0fc,
        0xbc8cbb194749390a,
        0xbc7394699d841c82,
    ],
    [
        0x3a909e71c2163ed3,
        0xbfbd09b21e36c0bd,
        0x3f52c84acfb586b4,
        0x3f9355b904fbf7ee,
        0xbf28fb570465af0d,
        0xbf4edc3292ba6cfd,
        0x3ee3e552ee8c2575,
        0x3ef76d44f6a83510,
        0xbe8e1ee2dc4a3910,
        0xbe94b944bbd4ab57,
        0x3e2a8c3a4ce04c1b,
        0x3e27f8c9ef133245,
        0xbdbe92957079603b,
        0xbdb388138abed52f,
        0x3d48c7ca6d80cb32,
        0x3d379e45475751e8,
        0xbccdc773f7abde68,
        0xbcb5d2331c76adb9,
        0x3c4a8aa5ca753696,
    ],
    [
        0xbfbc97d79918527d,
        0xbae4d780a21f2057,
        0x3fac95081ab2b511,
        0xbf37e0b14f7d7c3f,
        0xbf730688f6836a76,
        0x3f030941f6e78e36,
        0x3f243d5898657a6f,
        0xbeb5a39a94f2ad1c,
        0xbec70b18406146b4,
        0x3e597607f952cd69,
        0x3e604e788f4f18d9,
        0xbdf2598918fbb9ee,
        0xbdef701541acccd2,
        0x3d81df5bcb47b899,
        0x3d75f524c73bc009,
        0xbd0919aede2dcbf2,
        0xbcf73c65c9ce91ed,
        0x3c8a553df0f1b569,
        0x3c72fe9f3f0dcae2,
    ],
];
