/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2025 the original author or authors.
 */
package org.assertj.core.api.yearmonth;

import static java.util.Arrays.asList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatIllegalArgumentException;
import static org.assertj.core.error.ShouldBeIn.shouldBeIn;
import static org.assertj.core.util.AssertionsUtil.assertThatAssertionErrorIsThrownBy;

import java.time.YearMonth;

import org.assertj.core.api.ThrowableAssert.ThrowingCallable;
import org.junit.jupiter.api.Test;

/**
 * Only test String based assertion (tests with {@link YearMonth} are already done)
 */
class YearMonthAssert_isIn_Test extends YearMonthAssertBaseTest {

  @Test
  void should_pass_if_actual_is_in_dates_as_string_array_parameter() {
    assertThat(REFERENCE).isIn(REFERENCE.toString(), AFTER.toString());
  }

  @Test
  void should_fail_if_actual_is_not_in_dates_as_string_array_parameter() {
    // WHEN
    ThrowingCallable code = () -> assertThat(REFERENCE).isIn(AFTER.toString(), BEFORE.toString());
    // THEN
    assertThatAssertionErrorIsThrownBy(code).withMessage(shouldBeIn(REFERENCE, asList(AFTER, BEFORE)).create());
  }

  @Test
  void should_fail_if_dates_as_string_array_parameter_is_null() {
    // GIVEN
    String[] otherYearMonthsAsString = null;
    // WHEN
    ThrowingCallable code = () -> assertThat(YearMonth.now()).isIn(otherYearMonthsAsString);
    // THEN
    assertThatIllegalArgumentException().isThrownBy(code)
                                        .withMessage("The given YearMonth array should not be null");
  }

  @Test
  void should_fail_if_dates_as_string_array_parameter_is_empty() {
    // GIVEN
    String[] otherYearMonthsAsString = new String[0];
    // WHEN
    ThrowingCallable code = () -> assertThat(YearMonth.now()).isIn(otherYearMonthsAsString);
    // THEN
    assertThatIllegalArgumentException().isThrownBy(code)
                                        .withMessage("The given YearMonth array should not be empty");
  }

}
