//go:build fips

package fips

import (
	"testing"

	"github.com/stretchr/testify/require"
	"golang.org/x/crypto/ssh"
)

func TestSupportedAlgorithms_FipsEnabled(t *testing.T) {
	algorithms := SupportedAlgorithms()

	// When FIPS is enabled, should return FIPS-compliant algorithms
	t.Run("FipsEnabled", func(t *testing.T) {
		// Test that we get FIPS algorithms
		require.Equal(t, fipsCiphers, algorithms.Ciphers, "Ciphers should match FIPS-approved list")
		require.Equal(t, fipsMACs, algorithms.MACs, "MACs should match FIPS-approved list")
		require.Equal(t, fipsKexAlgos, algorithms.KeyExchanges, "KeyExchanges should match FIPS-approved list")
		require.Equal(t, fipsHostKeyAlgos, algorithms.HostKeys, "HostKeys should match FIPS-approved list")
		require.Equal(t, fipsPubKeyAuthAlgos, algorithms.PublicKeyAuths, "PublicKeyAuths should match FIPS-approved list")

		// Verify these are clones, not the original slices
		require.False(t, &algorithms.Ciphers == &fipsCiphers, "Ciphers should be cloned")
		require.False(t, &algorithms.MACs == &fipsMACs, "MACs should be cloned")
		require.False(t, &algorithms.KeyExchanges == &fipsKexAlgos, "KeyExchanges should be cloned")
		require.False(t, &algorithms.HostKeys == &fipsHostKeyAlgos, "HostKeys should be cloned")
		require.False(t, &algorithms.PublicKeyAuths == &fipsPubKeyAuthAlgos, "PublicKeyAuths should be cloned")

		// Verify modifying returned algorithms doesn't affect originals
		originalCiphersLen := len(fipsCiphers)
		algorithms.Ciphers = append(algorithms.Ciphers, "test-cipher")
		require.Equal(t, originalCiphersLen, len(fipsCiphers), "Original fipsCiphers should be unchanged")
	})

	t.Run("FipsAlgorithmsContent", func(t *testing.T) {
		// Test that FIPS algorithms contain expected values
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES128GCM, "Should contain AES-128-GCM")
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES256GCM, "Should contain AES-256-GCM")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP256, "Should contain ECDH P-256")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP384, "Should contain ECDH P-384")
		require.Contains(t, algorithms.MACs, ssh.HMACSHA256, "Should contain HMAC-SHA256")
		require.Contains(t, algorithms.MACs, ssh.HMACSHA512, "Should contain HMAC-SHA512")

		// Verify no non-FIPS algorithms are included
		require.NotContains(t, algorithms.Ciphers, "chacha20-poly1305@openssh.com", "Should not contain ChaCha20")
		require.NotContains(t, algorithms.KeyExchanges, "curve25519-sha256", "Should not contain Curve25519")
	})
}
