/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "comment.h"

Comment::Comment(PumpController *pumpController,
                 QVariantMap commentMap,
                 QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;


    QSizePolicy sizePolicy;
    sizePolicy.setHeightForWidth(false);
    sizePolicy.setWidthForHeight(false);
    //sizePolicy.setHorizontalStretch(1);
    //sizePolicy.setVerticalStretch(6);
    sizePolicy.setHorizontalPolicy(QSizePolicy::MinimumExpanding);
    sizePolicy.setVerticalPolicy(QSizePolicy::MinimumExpanding);
    this->setSizePolicy(sizePolicy);


    this->commentID = commentMap.value("id").toString();
    this->objectType = commentMap.value("objectType").toString();
    QVariantMap commentAuthorMap = commentMap.value("author").toMap();

    QString commentAuthorID = commentAuthorMap.value("id").toString();
    commentAuthorID.remove(0, 5); // remove "acct:" at the beginning
    if (commentAuthorID == pController->currentUserId())
    {
        commentIsOwn = true; // Comment is ours!

        // Different frame style depending on whether the comment is ours or not
        this->setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
    }
    else
    {
        commentIsOwn = false;

        this->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);
    }



    // Avatar pixmap
    avatarLabel = new QLabel();
    avatarLabel->setToolTip(commentAuthorID);
    this->commentAuthorAvatarUrl = commentAuthorMap.value("image").toMap().value("url").toString();
    QString commentAuthorAvatarFile = MiscHelpers::getCachedAvatarFilename(commentAuthorAvatarUrl);
    if (QFile::exists(commentAuthorAvatarFile))
    {
        avatarLabel->setPixmap(QPixmap(commentAuthorAvatarFile)
                               .scaledToWidth(32, Qt::SmoothTransformation));
    }
    else
    {
        // Set placeholder
        avatarLabel->setPixmap(QIcon::fromTheme("user-identity").pixmap(32, 32));
        // and download avatar for next time
        connect(pController, SIGNAL(avatarStored(QString,QString)),
                this, SLOT(redrawAvatar(QString,QString)));
        this->pController->getAvatar(commentAuthorAvatarUrl);
    }


    QFont commentsFont;
    commentsFont.setPointSize(commentsFont.pointSize() - 1); // 1 point less than default


    // Name, with ID as tooltip
    commentsFont.setBold(true);

    fullNameLabel = new QLabel(commentAuthorMap.value("displayName").toString());
    fullNameLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    fullNameLabel->setFont(commentsFont);
    fullNameLabel->setToolTip(commentAuthorID);


    // Timestamp
    commentsFont.setBold(false);
    commentsFont.setItalic(true);

    QString timestamp = commentMap.value("published").toString();
    QString commentExactTime = Timestamp::localTimeDate(timestamp);
    QString commentFuzzyTime = Timestamp::fuzzyTime(timestamp);
    timestampLabel = new QLabel(commentFuzzyTime);
    timestampLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    timestampLabel->setFont(commentsFont);
    timestampLabel->setToolTip(commentExactTime);



    // Like and Delete "buttons"
    commentsFont.setBold(true);
    commentsFont.setItalic(false);

    likeLabel = new QLabel("*like*");
    likeLabel->setContextMenuPolicy(Qt::NoContextMenu);
    likeLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    likeLabel->setFont(commentsFont);
    likeLabel->setToolTip(tr("Like or unlike this comment"));
    connect(likeLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(likeComment(QString)));

    commentIsLiked = commentMap.value("liked").toBool();
    this->fixLikeLabelText();


    quoteLabel = new QLabel("<a href=\"quote://\">" + tr("Quote", "This is a verb, infinitive") + "</a>");
    quoteLabel->setContextMenuPolicy(Qt::NoContextMenu);
    quoteLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    quoteLabel->setFont(commentsFont);
    quoteLabel->setToolTip(tr("Reply quoting this comment"));
    connect(quoteLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(quoteComment()));


    deleteLabel = new QLabel("<a href=\"delete://\">" + tr("Delete") + "</a>");
    deleteLabel->setContextMenuPolicy(Qt::NoContextMenu);
    deleteLabel->setAlignment(Qt::AlignTop | Qt::AlignRight);
    deleteLabel->setFont(commentsFont);
    deleteLabel->setToolTip(tr("Delete this comment"));
    connect(deleteLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(deleteComment()));


    // The likes count
    likesCountLabel = new QLabel();
    likesCountLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    commentsFont.setBold(false);
    likesCountLabel->setFont(commentsFont);

    this->setLikesCount(commentMap.value("likes").toMap().value("totalItems").toInt(),
                        commentMap.value("likes").toMap().value("items").toList());

    // The comment itself
    this->commentOriginalText = commentMap.value("content").toString();


    contentLabel = new QLabel();
    contentLabel->setAlignment(Qt::AlignLeft);
    contentLabel->setFont(commentsFont);
    contentLabel->setTextInteractionFlags(Qt::TextBrowserInteraction);
    contentLabel->setWordWrap(true);
    contentLabel->setOpenExternalLinks(true);
    contentLabel->setTextFormat(Qt::RichText);
    contentLabel->setSizePolicy(sizePolicy);
    connect(contentLabel, SIGNAL(linkHovered(QString)),
            this, SLOT(showUrlInfo(QString)));



    QStringList commentImageList = MiscHelpers::htmlWithReplacedImages(commentOriginalText,
                                                                       64); // Arbitrary width
    commentImageList.removeFirst(); // First is the HTML with images replaced

    // If the image list is not empty, get them
    if (!commentImageList.isEmpty())
    {
        qDebug() << "Comment has" << commentImageList.size() << "images included...";

        foreach (QString imageUrl, commentImageList)
        {
            this->enqueueImageForDownload(imageUrl);
        }
    }


    urlInfoLabel = new QLabel();
    urlInfoLabel->setFont(commentsFont);
    urlInfoLabel->setAutoFillBackground(true);
    urlInfoLabel->setForegroundRole(QPalette::ToolTipText);
    urlInfoLabel->setBackgroundRole(QPalette::ToolTipBase);
    urlInfoLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    urlInfoLabel->hide(); // Hidden initially

    contentLabelLayout = new QHBoxLayout();
    contentLabelLayout->addWidget(urlInfoLabel, 0, Qt::AlignRight | Qt::AlignBottom);
    contentLabel->setLayout(contentLabelLayout);



    leftLayout = new QVBoxLayout();
    leftLayout->setAlignment(Qt::AlignLeft);
    leftLayout->addWidget(avatarLabel,     0, Qt::AlignLeft | Qt::AlignTop);
    leftLayout->addWidget(likesCountLabel, 1, Qt::AlignHCenter | Qt::AlignTop);

    rightTopLayout = new QHBoxLayout();
    rightTopLayout->addWidget(fullNameLabel,  0, Qt::AlignLeft);
    rightTopLayout->addWidget(timestampLabel, 0, Qt::AlignLeft);
    rightTopLayout->addWidget(likeLabel,      0, Qt::AlignLeft);
    rightTopLayout->addWidget(quoteLabel,     0, Qt::AlignLeft);
    rightTopLayout->addStretch(1);
    if (commentIsOwn)
    {
        rightTopLayout->addWidget(deleteLabel, 0, Qt::AlignRight);
    }

    rightLayout = new QVBoxLayout();
    rightLayout->addLayout(rightTopLayout, 0);
    rightLayout->addSpacing(4); // 4px vertical space separation
    rightLayout->addWidget(contentLabel,   10);


    mainLayout = new QHBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);
    mainLayout->addLayout(leftLayout);
    mainLayout->addLayout(rightLayout);
    this->setLayout(mainLayout);


    setCommentContents();


    qDebug() << "Comment created" << this->commentID;
}


Comment::~Comment()
{
    qDebug() << "Comment destroyed" << this->commentID;
}



void Comment::fixLikeLabelText()
{
    if (commentIsLiked)
    {
        this->likeLabel->setText("<a href=\"unlike://\">" + tr("Unlike") +"</a>");
    }
    else
    {
        this->likeLabel->setText("<a href=\"like://\">" + tr("Like") +"</a>");
    }
}


void Comment::setLikesCount(int count, QVariantList namesVariantList)
{
    if (count > 0)
    {
        QString likesString;
        foreach (QVariant likesMap, namesVariantList)
        {
            likesString.append(likesMap.toMap().value("displayName").toString() + ", ");
        }
        likesString.remove(-2, 2); // remove last comma+space: ", "
        if (count > 1)
        {
            likesString = tr("%1 like this comment",
                             "Plural: %1=list of people like John, Jane, Smith").arg(likesString);
        }
        else
        {
            likesString = tr("%1 likes this comment",
                             "Singular: %1=name of just 1 person").arg(likesString);
        }


        likesCountLabel->setText(QString::fromUtf8("\342\231\245")  // heart symbol
                                 + QString(" %1").arg(count));
        // set tooltip as HTML, so it gets wordwrapped
        likesCountLabel->setToolTip(likesString + "<b></b>");
    }
    else
    {
        likesCountLabel->clear();
        likesCountLabel->setToolTip("");
    }
}


/*
 * Set the contents of the comment, parsing images, etc.
 *
 */
void Comment::setCommentContents()
{
    int imageWidth = this->contentLabel->width() - 20; // Kinda TMP

    QStringList commentImageList = MiscHelpers::htmlWithReplacedImages(commentOriginalText,
                                                                       imageWidth);

    QString commentContents = commentImageList.takeAt(0); // Comment's HTML with images replaced
    this->contentLabel->setText(commentContents);

    setCommentHeight();
}



void Comment::setCommentHeight()
{
//    int height = this->contentLabel->height();

/*
    if (height > 200)
    {
        height = 200;
    }
*/

/*
    this->contentLabel->setMinimumHeight(height);
    this->contentLabel->setMaximumHeight(height);
*/

}


void Comment::enqueueImageForDownload(QString url)
{
    if (QFile::exists(MiscHelpers::getCachedImageFilename(url))
     || pendingImagesList.contains(url))
    {
        qDebug() << "Comment::enqueueImageForDownload(), "
                    "Using cached post image, or requested image is pending download...";
    }
    else
    {
        qDebug() << "Comment::enqueueImageForDownload(), "
                    "post image not cached, downloading" << url;

        connect(pController, SIGNAL(imageStored(QString)),
                this, SLOT(redrawImages(QString)));

        this->pendingImagesList.append(url);
        pController->getImage(url);
    }
}



/*
 * Ensure urlInfoLabel is hidden when the mouse leaves the comment
 *
 */
void Comment::leaveEvent(QEvent *event)
{
    this->urlInfoLabel->clear();
    this->urlInfoLabel->hide();

    event->accept();
}




/****************************************************************************/
/******************************** SLOTS *************************************/
/****************************************************************************/



void Comment::likeComment(QString clickedLink)
{

    if (clickedLink == "like://")
    {
        commentIsLiked = true;
    }
    else // unlike://
    {
        commentIsLiked = false;
    }

    this->pController->likePost(this->commentID,
                                this->objectType,
                                this->commentIsLiked);
    this->fixLikeLabelText();
}


/*
 * Take the contents of a comment and put them, as a quote block
 * in the comment composer
 *
 * FIXME: It'd be nice to only quote the selected text,
 * unless nothing is selected, then use full comment, but clicking
 * the "quote" label removes the selection
 */
void Comment::quoteComment()
{
    QString quotedComment = MiscHelpers::quotedText(this->fullNameLabel->text(),
                                                    this->contentLabel->text());

    emit commentQuoteRequested(quotedComment);
}



void Comment::deleteComment()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete comment?"),
                                             tr("Are you sure you want to delete this comment?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting comment" << this->commentID;
        this->pController->deletePost(this->commentID, this->objectType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation canceled, not deleting the comment";
    }
}


/*
 * Show the URL of a link hovered in a comment
 *
 */
void Comment::showUrlInfo(QString url)
{
    if (url.isEmpty())
    {
        this->urlInfoLabel->clear();
        this->urlInfoLabel->hide();
    }
    else
    {
        qDebug() << "Link hovered in comment:" << url;
        this->urlInfoLabel->setText(url);
        this->urlInfoLabel->show();
    }
}



/*
 * Redraw comment author's avatar after receiving it
 *
 */void Comment::redrawAvatar(QString avatarUrl, QString avatarFilename)
{
    if (avatarUrl == this->commentAuthorAvatarUrl)
    {
        avatarLabel->setPixmap(QPixmap(avatarFilename)
                               .scaledToWidth(32, Qt::SmoothTransformation));

        disconnect(pController, SIGNAL(avatarStored(QString,QString)),
                   this, SLOT(redrawAvatar(QString,QString)));
    }
}


/*
 * Redraw comment contents after receiving downloaded images
 *
 */
void Comment::redrawImages(QString imageUrl)
{
    if (pendingImagesList.contains(imageUrl))
    {
        this->pendingImagesList.removeAll(imageUrl);
        if (pendingImagesList.isEmpty()) // If there are no more, disconnect
        {
            disconnect(pController, SIGNAL(imageStored(QString)),
                       this, SLOT(redrawImages(QString)));
        }

        setCommentContents();
    }
}
