/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { HttpStatusCode, TalerError, TalerMerchantApi } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useState } from "preact/hooks";
import { ProductForm } from "../../../../components/product/ProductForm.js";
import { useSessionContext } from "../../../../context/session.js";
import { useInstanceCategories } from "../../../../hooks/category.js";
import { Loading } from "../../../../components/exception/loading.js";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";

export interface Props {
  onCreate: () => void;
  onBack?: () => void;
}

export function CreatePage({ onCreate, onBack }: Props): VNode {
  const { state: session, lib } = useSessionContext();
  const [form, setForm] = useState<TalerMerchantApi.ProductAddDetail>();

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const create = safeFunctionHandler(
    lib.instance.addProduct.bind(lib.instance),
    !session.token || !form ? undefined : [session.token, form],
  );
  create.onSuccess = onCreate;
  create.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
    }
  };

  const { i18n } = useTranslationContext();

  // FIXME: if the category list is big the will bring a lot of info
  // we could find a lazy way to add up on searches
  const categoriesResult = useInstanceCategories();
  if (!categoriesResult) return <Loading />;
  if (categoriesResult instanceof TalerError) {
    return <ErrorLoadingMerchant error={categoriesResult} />;
  }
  const categories =
    categoriesResult.type === "fail" ? [] : categoriesResult.body.categories;

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section is-main-section">
        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <ProductForm onSubscribe={setForm} categories={categories} />

            <div class="buttons is-right mt-5">
              {onBack && (
                <button class="button" onClick={onBack}>
                  <i18n.Translate>Cancel</i18n.Translate>
                </button>
              )}
              <ButtonBetterBulma
                onClick={create}
                data-tooltip={
                  !create.args
                    ? i18n.str`Please complete the marked fields`
                    : "confirm operation"
                }
                type="submit"
              >
                <i18n.Translate>Confirm</i18n.Translate>
              </ButtonBetterBulma>
            </div>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
