/*
This file is part of Telegram Desktop,
the official desktop application for the Telegram messaging service.

For license and copyright information please follow this link:
https://github.com/telegramdesktop/tdesktop/blob/master/LEGAL
*/
#include "boxes/about_sponsored_box.h"

#include "core/file_utilities.h"
#include "lang/lang_keys.h"
#include "ui/layers/generic_box.h"
#include "ui/text/text_utilities.h"
#include "ui/widgets/buttons.h"
#include "ui/widgets/labels.h"
#include "styles/style_boxes.h"
#include "styles/style_layers.h"

namespace Ui {
namespace {

constexpr auto kUrl = "https://ads.telegram.org"_cs;

} // namespace

void AboutSponsoredBox(not_null<Ui::GenericBox*> box) {
	box->setTitle(tr::lng_sponsored_title());
	box->setWidth(st::boxWideWidth);
	box->addButton(tr::lng_box_ok(), [=] { box->closeBox(); });

	const auto addUrl = [&] {
		const auto &st = st::sponsoredUrlButton;
		const auto row = box->addRow(object_ptr<RpWidget>(box));
		row->resize(0, st.height + st.padding.top() + st.padding.bottom());
		const auto button = Ui::CreateChild<RoundButton>(
			row,
			rpl::single<QString>(kUrl.utf8()),
			st);
		button->setBrushOverride(Qt::NoBrush);
		button->setPenOverride(QPen(st::historyLinkInFg));
		button->setTextTransform(Ui::RoundButton::TextTransform::NoTransform);
		rpl::combine(
			row->sizeValue(),
			button->sizeValue()
		) | rpl::on_next([=](
				const QSize &rowSize,
				const QSize &buttonSize) {
			button->moveToLeft(
				(rowSize.width() - buttonSize.width()) / 2,
				(rowSize.height() - buttonSize.height()) / 2);
		}, row->lifetime());
		button->addClickHandler([=] {
			File::OpenUrl(kUrl.utf8());
		});
	};

	const auto &stLabel = st::aboutLabel;
	auto text1 = tr::lng_sponsored_info_description1_linked(
		lt_link,
		rpl::combine(
			tr::lng_sponsored_info_description1_link(),
			tr::lng_sponsored_info_description1_url()
		) | rpl::map([](const QString &text, const QString &url) {
			return Ui::Text::Link(text, url);
		}),
		Ui::Text::RichLangValue);
	box->addRow(object_ptr<FlatLabel>(box, std::move(text1), stLabel));

	box->addSkip(st::sponsoredUrlButtonSkip);
	addUrl();
	box->addSkip(st::sponsoredUrlButtonSkip);

	const auto info2 = box->addRow(object_ptr<FlatLabel>(box, stLabel));
	info2->setText(tr::lng_sponsored_info_description2(tr::now));
}

} // namespace Ui
