import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '..';
import * as Terminal from './terminal';
/**
 * An implementation of a terminal interface.
 */
export declare class TerminalConnection implements Terminal.ITerminalConnection {
    /**
     * Construct a new terminal session.
     */
    constructor(options: Terminal.ITerminalConnection.IOptions);
    /**
     * A signal emitted when the session is disposed.
     */
    get disposed(): ISignal<this, void>;
    /**
     * A signal emitted when a message is received from the server.
     */
    get messageReceived(): ISignal<this, Terminal.IMessage>;
    /**
     * Get the name of the terminal session.
     */
    get name(): string;
    /**
     * Get the model for the terminal session.
     */
    get model(): Terminal.IModel;
    /**
     * The server settings for the session.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the session is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the session.
     */
    dispose(): void;
    /**
     * Send a message to the terminal session.
     *
     * #### Notes
     * If the connection is down, the message will be queued for sending when
     * the connection comes back up.
     */
    send(message: Terminal.IMessage): void;
    /**
     * Send a message on the websocket, or possibly queue for later sending.
     *
     * @param queue - whether to queue the message if it cannot be sent
     */
    _sendMessage(message: Terminal.IMessage, queue?: boolean): void;
    /**
     * Send pending messages to the kernel.
     */
    private _sendPending;
    /**
     * Reconnect to a terminal.
     *
     * #### Notes
     * This may try multiple times to reconnect to a terminal, and will sever
     * any existing connection.
     */
    reconnect(): Promise<void>;
    /**
     * Attempt a connection if we have not exhausted connection attempts.
     */
    _reconnect(): void;
    /**
     * Forcefully clear the socket state.
     *
     * #### Notes
     * This will clear all socket state without calling any handlers and will
     * not update the connection status. If you call this method, you are
     * responsible for updating the connection status as needed and recreating
     * the socket if you plan to reconnect.
     */
    private _clearSocket;
    /**
     * Shut down the terminal session.
     */
    shutdown(): Promise<void>;
    /**
     * Clone the current terminal connection.
     */
    clone(): Terminal.ITerminalConnection;
    /**
     * Create the terminal websocket connection and add socket status handlers.
     *
     * #### Notes
     * You are responsible for updating the connection status as appropriate.
     */
    private _createSocket;
    private _onWSMessage;
    private _onWSClose;
    /**
     * Handle connection status changes.
     */
    private _updateConnectionStatus;
    /**
     * Utility function to throw an error if this instance is disposed.
     */
    private _errorIfDisposed;
    /**
     * A signal emitted when the terminal connection status changes.
     */
    get connectionStatusChanged(): ISignal<this, Terminal.ConnectionStatus>;
    /**
     * The current connection status of the terminal connection.
     */
    get connectionStatus(): Terminal.ConnectionStatus;
    private _connectionStatus;
    private _connectionStatusChanged;
    private _isDisposed;
    private _disposed;
    private _messageReceived;
    private _name;
    private _reconnectTimeout;
    private _ws;
    private _noOp;
    private _reconnectLimit;
    private _reconnectAttempt;
    private _pendingMessages;
}
