// Copyright contributors to the openqasm-parser project
// SPDX-License-Identifier: Apache-2.0

//! Syntax Tree library used throughout the rust-analyzer.
//!
//! Properties:
//!   - easy and fast incremental re-parsing. (Removed for OpenQASM 3, GJL)
//!   - graceful handling of errors
//!   - full-fidelity representation (*any* text can be precisely represented as
//!     a syntax tree)
//!
//! For more information, see the [RFC]. Current implementation is inspired by
//! the [Swift] one.
//!
//! The most interesting modules here are `syntax_node` (which defines concrete
//! syntax tree) and `ast` (which defines abstract syntax tree on top of the
//! CST). The actual parser live in a separate `parser` crate, though the
//! lexer lives in this crate.
//!
//! See `api_walkthrough` test in this file for a quick API tour!
//!
//! [RFC]: <https://github.com/rust-lang/rfcs/pull/2256>
//! [Swift]: <https://github.com/apple/swift/blob/13d593df6f359d0cb2fc81cfaac273297c539455/lib/Syntax/README.md>

#![warn(
    rust_2018_idioms,
    unused_lifetimes,
    semicolon_in_expressions_from_macros
)]

mod parsing;
mod ptr;
mod sourcegen;
mod syntax_error;
pub mod syntax_node;
#[cfg(test)]
mod tests;
mod token_text;
mod validation;

pub mod ast;
pub mod ted;

use std::marker::PhantomData;

use triomphe::Arc;

pub use crate::{
    ast::{AstNode, AstToken, HasTextName},
    parsing::parse_text,
    ptr::{AstPtr, SyntaxNodePtr},
    syntax_error::SyntaxError,
    syntax_node::{
        OpenQASM3Language, PreorderWithTokens, SyntaxElement, SyntaxNode, SyntaxToken,
        SyntaxTreeBuilder,
    },
    token_text::TokenText,
};
pub use oq3_parser::{SyntaxKind, T};
pub use rowan::{
    api::Preorder, Direction, GreenNode, NodeOrToken, SyntaxText, TextRange, TextSize,
    TokenAtOffset, WalkEvent,
};
pub use smol_str::SmolStr;

pub trait AsString {
    fn string(&self) -> String;
}
impl AsString for SyntaxNode {
    fn string(&self) -> String {
        self.text().to_string()
    }
}

/// `Parse` is the result of the parsing: a syntax tree and a collection of
/// errors.
///
/// Note that we always produce a syntax tree, even for completely invalid
/// files.
#[derive(Debug, PartialEq, Eq)]
pub struct Parse<T> {
    green: GreenNode,
    errors: Arc<Vec<SyntaxError>>,
    _ty: PhantomData<fn() -> T>,
}

impl<T> Clone for Parse<T> {
    fn clone(&self) -> Parse<T> {
        Parse {
            green: self.green.clone(),
            errors: self.errors.clone(),
            _ty: PhantomData,
        }
    }
}

impl<T> Parse<T> {
    fn new(green: GreenNode, errors: Vec<SyntaxError>) -> Parse<T> {
        Parse {
            green,
            errors: Arc::new(errors),
            _ty: PhantomData,
        }
    }

    pub fn syntax_node(&self) -> SyntaxNode {
        SyntaxNode::new_root(self.green.clone())
    }
    pub fn errors(&self) -> &[SyntaxError] {
        &self.errors
    }
}

impl<T: AstNode> Parse<T> {
    pub fn tree(&self) -> T {
        T::cast(self.syntax_node()).unwrap()
    }

    pub fn ok(self) -> Result<T, Arc<Vec<SyntaxError>>> {
        if self.errors.is_empty() {
            Ok(self.tree())
        } else {
            Err(self.errors)
        }
    }
}

impl Parse<SyntaxNode> {
    pub fn cast<N: AstNode>(self) -> Option<Parse<N>> {
        if N::cast(self.syntax_node()).is_some() {
            Some(Parse {
                green: self.green,
                errors: self.errors,
                _ty: PhantomData,
            })
        } else {
            None
        }
    }
}

impl Parse<SourceFile> {
    pub fn debug_dump(&self) -> String {
        let mut buf = format!("{:#?}", self.tree().syntax());
        for err in self.errors.iter() {
            format_to!(buf, "error {:?}: {}\n", err.range(), err);
        }
        buf
    }
}

// We have preserved `Parse<T>` above which is needed at least by make.rs.
// But it's probably best to consolidate `Parse` and `ParseOrErrors`. The
// former is still used in make.rs. We make very little use of this, only for
// a few tests. Some of these tests trigger lexer errors, some do not. We
// inherited the treatement from r-a, which always proceeds from lexing to parsing,
// and makes no distinction between lexer and parser errors.
// But for OQ3, it is convenient to demand:
//   The parser only runs if there were no lexer errors.
//   The semantic analysis only runs if there were no parser errors.
// Corollaries to these requirements are
//   An improperly lexed stream, say one with tokens deemed "illegal", being ingested
//   by the parser, implies a bug outside the parser.
//   A syntactically incorrect ast being ingested by the semantic analyzer implies a bug
//   outside the semantic analyzer.
/// Same as Parse<T> except that the `GreenNode` is wrapped in `Option`.
/// The `Option` is `None` if lexer errors were recorded, in which case no
/// parsing was done. In the same case, all errors will be lexer errors.
/// If there are no lexer errors, the parsing was done, and there is a `GreenNode`.
/// In this case any errors are parser errors.
#[derive(Debug, PartialEq, Eq)]
pub struct ParseOrErrors<T> {
    green: Option<GreenNode>,
    errors: Arc<Vec<SyntaxError>>,
    _ty: PhantomData<fn() -> T>,
}

impl<T> Clone for ParseOrErrors<T> {
    fn clone(&self) -> ParseOrErrors<T> {
        ParseOrErrors {
            green: self.green.clone(),
            errors: self.errors.clone(),
            _ty: PhantomData,
        }
    }
}

impl<T> ParseOrErrors<T> {
    pub fn syntax_node(&self) -> SyntaxNode {
        SyntaxNode::new_root(self.green.clone().unwrap())
    }
    pub fn errors(&self) -> &[SyntaxError] {
        &self.errors
    }
    pub fn have_parse(&self) -> bool {
        self.green.is_some()
    }
}

impl<T: AstNode> ParseOrErrors<T> {
    pub fn tree(&self) -> T {
        T::cast(self.syntax_node()).unwrap()
    }
}

/// `SourceFile` represents a parse tree for a single OQ3 file.
pub use crate::ast::SourceFile;

impl SourceFile {
    pub fn parse(text: &str) -> Parse<SourceFile> {
        let (green, mut errors) = parsing::parse_text(text);
        let root = SyntaxNode::new_root(green.clone());
        errors.extend(validation::validate(&root));
        assert_eq!(root.kind(), SyntaxKind::SOURCE_FILE);
        Parse {
            green,
            errors: Arc::new(errors),
            _ty: PhantomData,
        }
    }

    /// Lex `text` and parse the result if there are no lexing errors.
    /// The green tree is wrapped in `Option` to account for the case
    /// that there *are* lexing errors and no parsing is done and no
    /// tree is built.
    pub fn parse_check_lex(text: &str) -> ParseOrErrors<SourceFile> {
        let (green_maybe, mut errors) = parsing::parse_text_check_lex(text);
        if let Some(ref green) = green_maybe {
            let root = SyntaxNode::new_root(green.clone());
            errors.extend(validation::validate(&root));
            assert_eq!(root.kind(), SyntaxKind::SOURCE_FILE);
        }
        ParseOrErrors {
            green: green_maybe,
            errors: Arc::new(errors),
            _ty: PhantomData,
        }
    }
}

/// Matches a `SyntaxNode` against an `ast` type.
///
/// # Example:
///
/// ```ignore
/// match_ast! {
///     match node {
///         ast::CallExpr(it) => { ... },
///         ast::MethodCallExpr(it) => { ... },
///         ast::MacroCall(it) => { ... },
///         _ => None,
///     }
/// }
/// ```
#[macro_export]
macro_rules! match_ast {
    (match $node:ident { $($tt:tt)* }) => { $crate::match_ast!(match ($node) { $($tt)* }) };

    (match ($node:expr) {
        $( $( $path:ident )::+ ($it:pat) => $res:expr, )*
        _ => $catch_all:expr $(,)?
    }) => {{
        $( if let Some($it) = $($path::)+cast($node.clone()) { $res } else )*
        { $catch_all }
    }};
}

// /// This test does not assert anything and instead just shows off the crate's
// /// API.
#[test]
fn api_walkthrough() {
    use ast::HasName;

    let source_code = "
        def foo() {
            1 + 1;
        }
    ";
    // `SourceFile` is the main entry point.
    //
    // The `parse` method returns a `Parse` -- a pair of syntax tree and a list
    // of errors. That is, syntax tree is constructed even in presence of errors.
    let parse = SourceFile::parse(source_code);
    assert!(parse.errors().is_empty());

    // The `tree` method returns an owned syntax node of type `SourceFile`.
    // Owned nodes are cheap: inside, they are `Rc` handles to the underling data.
    let file: SourceFile = parse.tree();

    // `SourceFile` is the root of the syntax tree. We can iterate file's stmts.
    // Let's fetch the `foo` function.
    let mut func = None;
    for stmt in file.statements() {
        match stmt {
            ast::Stmt::Def(f) => func = Some(f),
            _ => unreachable!(),
        }
    }
    let func: ast::Def = func.unwrap();

    // Each AST node has a bunch of getters for children. All getters return
    // `Option`s though, to account for incomplete code. Some getters are common
    // for several kinds of node. In this case, a trait like `ast::NameOwner`
    // usually exists. By convention, all ast types should be used with `ast::`
    // qualifier.
    let name: Option<ast::Name> = func.name();
    let name = name.unwrap();
    assert_eq!(name.text(), "foo");

    // Let's get the `1 + 1` expression!
    let body: ast::BlockExpr = func.body().unwrap();
    assert_eq!(
        body.syntax().first_child_or_token().map(|it| it.kind()),
        Some(T!['{'])
    );
    let mut stmt_list = body.statements();
    let expr0 = stmt_list.next().unwrap();
    let expr = match expr0 {
        ast::Stmt::ExprStmt(expr_stmt) => expr_stmt.expr().unwrap(),
        _ => unreachable!(),
    };
    // Enums are used to group related ast nodes together, and can be used for
    // matching. However, because there are no public fields, it's possible to
    // match only the top level enum: that is the price we pay for increased API
    // flexibility
    let bin_expr: &ast::BinExpr = match &expr {
        ast::Expr::BinExpr(e) => e,
        _ => unreachable!(),
    };
    // Besides the "typed" AST API, there's an untyped CST one as well.
    // To switch from AST to CST, call `.syntax()` method:
    let expr_syntax: &SyntaxNode = expr.syntax();

    // Note how `expr` and `bin_expr` are in fact the same node underneath:
    assert!(expr_syntax == bin_expr.syntax());

    // To go from CST to AST, `AstNode::cast` function is used:
    let _expr: ast::Expr = match ast::Expr::cast(expr_syntax.clone()) {
        Some(e) => e,
        None => unreachable!(),
    };

    // The two properties each syntax node has is a `SyntaxKind`:
    assert_eq!(expr_syntax.kind(), SyntaxKind::BIN_EXPR);

    // And text range:
    assert_eq!(
        expr_syntax.text_range(),
        TextRange::new(33.into(), 38.into())
    );

    // You can get node's text as a `SyntaxText` object, which will traverse the
    // tree collecting token's text:
    let text: SyntaxText = expr_syntax.text();
    assert_eq!(text.to_string(), "1 + 1");

    // There's a bunch of traversal methods on `SyntaxNode`:
    // Following line broken because for OQ3 we simplified BlockExpr
    // assert_eq!(expr_syntax.parent().as_ref(), Some(stmt_list.syntax()));
    assert_eq!(
        body.syntax().first_child_or_token().map(|it| it.kind()),
        Some(T!['{'])
    );
    assert_eq!(
        expr_syntax.next_sibling_or_token().map(|it| it.kind()),
        Some(SyntaxKind::SEMICOLON) // OQ3 requires semicolon on every, including last, statement in block
                                    //        Some(SyntaxKind::WHITESPACE)
    );

    // As well as some iterator helpers:
    let f = expr_syntax.ancestors().find_map(ast::Def::cast);
    assert_eq!(f, Some(func));
    //    assert!(expr_syntax.siblings_with_tokens(Direction::Next).any(|it| it.kind() == T!['}']));
    // Not sure if following is meaningful. where is semicolon?
    assert_eq!(
        expr_syntax.descendants_with_tokens().count(),
        8, // 5 tokens `1`, ` `, `+`, ` `, `!`
           // 2 child literal expressions: `1`, `1`
           // 1 the node itself: `1 + 1`
    );

    // There's also a `preorder` method with a more fine-grained iteration control:
    let mut buf = String::new();
    let mut indent = 0;
    for event in expr_syntax.preorder_with_tokens() {
        match event {
            WalkEvent::Enter(node) => {
                let text = match &node {
                    NodeOrToken::Node(it) => it.text().to_string(),
                    NodeOrToken::Token(it) => it.text().to_string(),
                };
                format_to!(
                    buf,
                    "{:indent$}{:?} {:?}\n",
                    " ",
                    text,
                    node.kind(),
                    indent = indent
                );
                indent += 2;
            }
            WalkEvent::Leave(_) => indent -= 2,
        }
    }
    assert_eq!(indent, 0);
    assert_eq!(
        buf.trim(),
        r#"
"1 + 1" BIN_EXPR
  "1" LITERAL
    "1" INT_NUMBER
  " " WHITESPACE
  "+" PLUS
  " " WHITESPACE
  "1" LITERAL
    "1" INT_NUMBER
"#
        .trim()
    );

    // To recursively process the tree, there are three approaches:
    // 1. explicitly call getter methods on AST nodes.
    // 2. use descendants and `AstNode::cast`.
    // 3. use descendants and `match_ast!`.
    //
    // Here's how the first one looks like:
    let exprs_cast: Vec<String> = file
        .syntax()
        .descendants()
        .filter_map(ast::Expr::cast)
        .map(|expr| expr.syntax().text().to_string())
        .collect();

    // An alternative is to use a macro.
    let mut exprs_visit = Vec::new();
    for node in file.syntax().descendants() {
        match_ast! {
            match node {
                ast::Expr(it) => {
                    let res = it.syntax().text().to_string();
                    exprs_visit.push(res);
                },
                _ => (),
            }
        }
    }
    assert_eq!(exprs_cast, exprs_visit);
}

// #229
// This macro is copied from ra_ap_stdx. It is the only feature
// we needed from ra_ap_stdx. But that crate put us in a version bind.
/// Appends formatted string to a `String`.
#[macro_export]
macro_rules! format_to {
    ($buf:expr) => ();
    ($buf:expr, $lit:literal $($arg:tt)*) => {
        {
            use ::std::fmt::Write as _;
            // We can't do ::std::fmt::Write::write_fmt($buf, format_args!($lit $($arg)*))
            // unfortunately, as that loses out on autoref behavior.
            _ = $buf.write_fmt(format_args!($lit $($arg)*))
        }
    };
}
