/*
Copyright (c) 2022 Advanced Micro Devices, Inc. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include <hip_test_common.hh>
#include <hip/hip_runtime_api.h>

/**
 * @addtogroup hipExtGetLinkTypeAndHopCount hipExtGetLinkTypeAndHopCount
 * @{
 * @ingroup DeviceTest
 * `hipExtGetLinkTypeAndHopCount(int device1, int device2, uint32_t* linktype, uint32_t* hopcount)`
 * - Returns the link type and hop count between two devices.
 */

#if __linux__
#if HT_AMD
/**
 * Test Description
 * ------------------------
 *  - Check commutativity of devices for every device combination.
 * Test source
 * ------------------------
 *  - unit/device/hipExtGetLinkTypeAndHopCount.cc
 * Test requirements
 * ------------------------
 *  - Platform specific (AMD)
 *  - HIP_VERSION >= 5.2
 */
TEST_CASE("Unit_hipExtGetLinkTypeAndHopCount_Positive_Basic") {
  const auto device1 = GENERATE(range(0, HipTest::getDeviceCount()));
  const auto device2 = GENERATE(range(0, HipTest::getDeviceCount()));

  if (device1 == device2) {
    return;
  }

  int can_access_peer = 0;
  HIP_CHECK(hipDeviceCanAccessPeer(&can_access_peer, device1, device2));
  if (!can_access_peer) {
    std::string msg =
        "Skipped as peer access is not supported between devices : " + std::to_string(device1) +
        " " + std::to_string(device2);
    HipTest::HIP_SKIP_TEST(msg.c_str());
    return;
  }

  uint32_t link_type1 = -1, hop_count1 = -1;
  uint32_t link_type2 = -1, hop_count2 = -1;

  HIP_CHECK(hipExtGetLinkTypeAndHopCount(device1, device2, &link_type1, &hop_count1));
  HIP_CHECK(hipExtGetLinkTypeAndHopCount(device2, device1, &link_type2, &hop_count2));

  REQUIRE(hop_count1 > 0);

  REQUIRE(hop_count1 == hop_count2);
  REQUIRE(link_type1 == link_type2);
}

/**
 * Test Description
 * ------------------------
 *  - Validates handling of invalid arguments:
 *    -# When creating the link between the same device
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When device ordinance for the first device is out of bounds
 *      - Expected output: return `hipErrorInvalidDevice`
 *    -# When device ordinance for the second device is out of bounds
 *      - Expected output: return `hipErrorInvalidDevice`
 *    -# When device ordinance for both devices is out of bounds
 *      - Expected output: return `hipErrorInvalidDevice`
 *    -# When device ordinance for the first device is < 0
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When device ordinance for the second device is < 0
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When device ordinance for both devices is < 0
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When pointer to the link type is `nullptr`
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When pointer to the hop count is `nullptr`
 *      - Expected output: return `hipErrorInvalidValue`
 *    -# When both pointers are `nullptr`
 *      - Expected output: return `hipErrorInvalidValue`
 * Test source
 * ------------------------
 *  - unit/device/hipExtGetLinkTypeAndHopCount.cc
 * Test requirements
 * ------------------------
 *  - Platform specific (AMD)
 *  - HIP_VERSION >= 5.2
 */
TEST_CASE("Unit_hipExtGetLinkTypeAndHopCount_Negative_Parameters") {
  uint32_t link_type, hop_count;
  SECTION("same device") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(0, 0, &link_type, &hop_count),
                    hipErrorInvalidValue);
  }

  SECTION("device ordinance 1 too large") {
    HIP_CHECK_ERROR(
        hipExtGetLinkTypeAndHopCount(HipTest::getDeviceCount(), 0, &link_type, &hop_count),
        hipErrorInvalidDevice);
  }

  SECTION("device ordinance 2 too large") {
    HIP_CHECK_ERROR(
        hipExtGetLinkTypeAndHopCount(0, HipTest::getDeviceCount(), &link_type, &hop_count),
        hipErrorInvalidDevice);
  }

  SECTION("device ordinances too large") {
    HIP_CHECK_ERROR(
        hipExtGetLinkTypeAndHopCount(HipTest::getDeviceCount(), HipTest::getDeviceCount() + 1,
                                     &link_type, &hop_count),
        hipErrorInvalidDevice);
  }

  SECTION("device 1 < 0") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(-1, 0, &link_type, &hop_count),
                    hipErrorInvalidValue);
  }

  SECTION("device 2 < 0") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(0, -1, &link_type, &hop_count),
                    hipErrorInvalidValue);
  }

  SECTION("both devices < 0") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(-1, -2, &link_type, &hop_count),
                    hipErrorInvalidValue);
  }

  SECTION("linktype == nullptr") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(0, 1, nullptr, &hop_count), hipErrorInvalidValue);
  }

  SECTION("hopcount == nullptr") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(0, 1, &link_type, nullptr), hipErrorInvalidValue);
  }

  SECTION("linktype and hopcount == nullptr") {
    HIP_CHECK_ERROR(hipExtGetLinkTypeAndHopCount(0, 1, nullptr, nullptr), hipErrorInvalidValue);
  }
}
#endif
#endif

/**
 * End doxygen group DeviceTest.
 * @}
 */
