# frozen_string_literal: true

require "abstract_unit"

module AbstractController
  module Testing
    class MyCollector
      include AbstractController::Collector
      attr_accessor :responses

      def initialize
        @responses = []
      end

      def custom(mime, *args, **kwargs, &block)
        @responses << [mime, args, kwargs, block]
      end
    end

    class TestCollector < ActiveSupport::TestCase
      test "responds to default mime types" do
        collector = MyCollector.new
        assert_respond_to collector, :html
        assert_respond_to collector, :text
      end

      test "does not respond to unknown mime types" do
        collector = MyCollector.new
        assert_not_respond_to collector, :unknown
      end

      test "register mime types on method missing" do
        AbstractController::Collector.remove_method :js
        begin
          collector = MyCollector.new
          assert_not_respond_to collector, :js
          collector.js
          assert_respond_to collector, :js
        ensure
          unless AbstractController::Collector.method_defined? :js
            AbstractController::Collector.generate_method_for_mime :js
          end
        end
      end

      test "does not register unknown mime types" do
        collector = MyCollector.new
        assert_raise NoMethodError do
          collector.unknown
        end
      end

      test "generated methods call custom with arguments received" do
        collector = MyCollector.new
        collector.html
        collector.text(:foo, bar: :baz)
        collector.js(:bar) { :baz }
        assert_equal [Mime[:html], [], {}, nil], collector.responses[0]
        assert_equal [Mime[:text], [:foo], { bar: :baz }, nil], collector.responses[1]
        assert_equal [Mime[:js], [:bar], {}], collector.responses[2][0, 3]
        assert_equal :baz, collector.responses[2][3].call
      end
    end
  end
end
