/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.internal.auth;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.openhab.core.auth.Authentication;
import org.openhab.core.auth.AuthenticationException;
import org.openhab.core.auth.Credentials;
import org.openhab.core.auth.ManagedUser;
import org.openhab.core.auth.User;
import org.openhab.core.auth.UserApiToken;
import org.openhab.core.auth.UserApiTokenCredentials;
import org.openhab.core.auth.UserProvider;
import org.openhab.core.auth.UserRegistry;
import org.openhab.core.auth.UserSession;
import org.openhab.core.auth.UsernamePasswordCredentials;
import org.openhab.core.common.registry.AbstractRegistry;
import org.openhab.core.internal.auth.ManagedUserProvider;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
@Component(service={UserRegistry.class}, immediate=true)
public class UserRegistryImpl
extends AbstractRegistry<User, String, UserProvider>
implements UserRegistry {
    private final Logger logger = LoggerFactory.getLogger(UserRegistryImpl.class);
    private static final int PASSWORD_ITERATIONS = 65536;
    private static final int APITOKEN_ITERATIONS = 1024;
    private static final String APITOKEN_PREFIX = "oh";
    private static final int KEY_LENGTH = 512;
    private static final String ALGORITHM = "PBKDF2WithHmacSHA512";
    private static final SecureRandom RAND = new SecureRandom();

    @Activate
    public UserRegistryImpl(BundleContext context, Map<String, Object> properties) {
        super(UserProvider.class);
        super.activate(context);
    }

    @Override
    @Deactivate
    protected void deactivate() {
        super.deactivate();
    }

    @Reference(cardinality=ReferenceCardinality.OPTIONAL, policy=ReferencePolicy.DYNAMIC)
    protected void setManagedProvider(ManagedUserProvider managedProvider) {
        super.setManagedProvider(managedProvider);
        super.addProvider(managedProvider);
    }

    protected void unsetManagedProvider(ManagedUserProvider managedProvider) {
        super.unsetManagedProvider(managedProvider);
        super.removeProvider(managedProvider);
    }

    @Override
    public User register(String username, String password, Set<String> roles) {
        String passwordSalt = this.generateSalt(64).get();
        String passwordHash = this.hash(password, passwordSalt, 65536).get();
        ManagedUser user = new ManagedUser(username, passwordSalt, passwordHash);
        user.setRoles(new HashSet<String>(roles));
        super.add(user);
        return user;
    }

    private Optional<String> generateSalt(int length) {
        if (length < 1) {
            this.logger.error("error in generateSalt: length must be > 0");
            return Optional.empty();
        }
        byte[] salt = new byte[length];
        RAND.nextBytes(salt);
        return Optional.of(Base64.getEncoder().encodeToString(salt));
    }

    private Optional<String> hash(String password, String salt, int iterations) {
        char[] chars = password.toCharArray();
        byte[] bytes = salt.getBytes();
        PBEKeySpec spec = new PBEKeySpec(chars, bytes, iterations, 512);
        Arrays.fill(chars, '\u0000');
        try {
            SecretKeyFactory fac = SecretKeyFactory.getInstance(ALGORITHM);
            byte[] securePassword = fac.generateSecret(spec).getEncoded();
            Optional<String> optional = Optional.of(Base64.getEncoder().encodeToString(securePassword));
            return optional;
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            this.logger.error("Exception encountered while hashing", (Throwable)e);
            Optional<String> optional = Optional.empty();
            return optional;
        }
        finally {
            spec.clearPassword();
        }
    }

    @Override
    public Authentication authenticate(Credentials credentials) throws AuthenticationException {
        if (credentials instanceof UsernamePasswordCredentials) {
            UsernamePasswordCredentials usernamePasswordCreds = (UsernamePasswordCredentials)credentials;
            User user = (User)this.get(usernamePasswordCreds.getUsername());
            if (user == null) {
                throw new AuthenticationException("User not found: " + usernamePasswordCreds.getUsername());
            }
            ManagedUser managedUser = (ManagedUser)user;
            String hashedPassword = this.hash(usernamePasswordCreds.getPassword(), managedUser.getPasswordSalt(), 65536).get();
            if (!hashedPassword.equals(managedUser.getPasswordHash())) {
                throw new AuthenticationException("Wrong password for user " + usernamePasswordCreds.getUsername());
            }
            return new Authentication(managedUser.getName(), (String[])managedUser.getRoles().stream().toArray(String[]::new));
        }
        if (credentials instanceof UserApiTokenCredentials) {
            UserApiTokenCredentials apiTokenCreds = (UserApiTokenCredentials)credentials;
            String[] apiTokenParts = apiTokenCreds.getApiToken().split("\\.");
            if (apiTokenParts.length != 3 || !APITOKEN_PREFIX.equals(apiTokenParts[0])) {
                throw new AuthenticationException("Invalid API token format");
            }
            for (User user : this.getAll()) {
                ManagedUser managedUser = (ManagedUser)user;
                for (UserApiToken userApiToken : managedUser.getApiTokens()) {
                    if (!userApiToken.getName().equals(apiTokenParts[1])) continue;
                    String[] existingTokenHashAndSalt = userApiToken.getApiToken().split(":");
                    String incomingTokenHash = this.hash(apiTokenCreds.getApiToken(), existingTokenHashAndSalt[1], 1024).get();
                    if (!incomingTokenHash.equals(existingTokenHashAndSalt[0])) continue;
                    return new Authentication(managedUser.getName(), (String[])managedUser.getRoles().stream().toArray(String[]::new), userApiToken.getScope());
                }
            }
            throw new AuthenticationException("Unknown API token");
        }
        throw new IllegalArgumentException("Invalid credential type");
    }

    @Override
    public void changePassword(User user, String newPassword) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        ManagedUser managedUser = (ManagedUser)user;
        String passwordSalt = this.generateSalt(64).get();
        String passwordHash = this.hash(newPassword, passwordSalt, 65536).get();
        managedUser.setPasswordSalt(passwordSalt);
        managedUser.setPasswordHash(passwordHash);
        this.update(user);
    }

    @Override
    public void addUserSession(User user, UserSession session) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        ManagedUser managedUser = (ManagedUser)user;
        managedUser.getSessions().add(session);
        this.update(user);
    }

    @Override
    public void removeUserSession(User user, UserSession session) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        ManagedUser managedUser = (ManagedUser)user;
        managedUser.getSessions().remove(session);
        this.update(user);
    }

    @Override
    public void clearSessions(User user) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        ManagedUser managedUser = (ManagedUser)user;
        managedUser.getSessions().clear();
        this.update(user);
    }

    @Override
    public String addUserApiToken(User user, String name, String scope) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        if (!name.matches("[a-zA-Z0-9]*")) {
            throw new IllegalArgumentException("API token name format invalid, alphanumeric characters only");
        }
        ManagedUser managedUser = (ManagedUser)user;
        String tokenSalt = this.generateSalt(64).get();
        byte[] rnd = new byte[64];
        RAND.nextBytes(rnd);
        String token = "oh." + name + "." + Base64.getEncoder().encodeToString(rnd).replaceAll("(\\+|/|=)", "");
        String tokenHash = this.hash(token, tokenSalt, 1024).get();
        UserApiToken userApiToken = new UserApiToken(name, tokenHash + ":" + tokenSalt, scope);
        managedUser.getApiTokens().add(userApiToken);
        this.update(user);
        return token;
    }

    @Override
    public void removeUserApiToken(User user, UserApiToken userApiToken) {
        if (!(user instanceof ManagedUser)) {
            throw new IllegalArgumentException("User is not managed: " + user.getName());
        }
        ManagedUser managedUser = (ManagedUser)user;
        managedUser.getApiTokens().remove(userApiToken);
        this.update(user);
    }

    @Override
    public boolean supports(Class<? extends Credentials> type) {
        return UsernamePasswordCredentials.class.isAssignableFrom(type);
    }
}

