/*
 * Windows 932 codepage (Japanese Shift-JIS) functions
 *
 * Copyright (c) 2008-2013, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <types.h>

#include "libuna_codepage_windows_932.h"
#include "libuna_inline.h"
#include "libuna_libcerror.h"
#include "libuna_types.h"

/* Extended ASCII to Unicode character lookup table for Windows 932 codepage
 * Unknown are filled with the Unicode replacement character 0xfffd
 */
const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8140[ 109 ] = {
	0x3000, 0x3001, 0x3002, 0xff0c, 0xff0e, 0x30fb, 0xff1a, 0xff1b,
	0xff1f, 0xff01, 0x309b, 0x309c, 0x00b4, 0xff40, 0x00a8, 0xff3e,
	0xffe3, 0xff3f, 0x30fd, 0x30fe, 0x309d, 0x309e, 0x3003, 0x4edd,
	0x3005, 0x3006, 0x3007, 0x30fc, 0x2015, 0x2010, 0xff0f, 0xff3c,
	0xff5e, 0x2225, 0xff5c, 0x2026, 0x2025, 0x2018, 0x2019, 0x201c,
	0x201d, 0xff08, 0xff09, 0x3014, 0x3015, 0xff3b, 0xff3d, 0xff5b,
	0xff5d, 0x3008, 0x3009, 0x300a, 0x300b, 0x300c, 0x300d, 0x300e,
	0x300f, 0x3010, 0x3011, 0xff0b, 0xff0d, 0x00b1, 0x00d7, 0xfffd,
	0x00f7, 0xff1d, 0x2260, 0xff1c, 0xff1e, 0x2266, 0x2267, 0x221e,
	0x2234, 0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xffe5,
	0xff04, 0xffe0, 0xffe1, 0xff05, 0xff03, 0xff06, 0xff0a, 0xff20,
	0x00a7, 0x2606, 0x2605, 0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6,
	0x25a1, 0x25a0, 0x25b3, 0x25b2, 0x25bd, 0x25bc, 0x203b, 0x3012,
	0x2192, 0x2190, 0x2191, 0x2193, 0x3013
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81b8[ 8 ] = {
	0x2208, 0x220b, 0x2286, 0x2287, 0x2282, 0x2283, 0x222a, 0x2229
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81c8[ 7 ] = {
	0x2227, 0x2228, 0xffe2, 0x21d2, 0x21d4, 0x2200, 0x2203
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81da[ 15 ] = {
	                0x2220, 0x22a5, 0x2312, 0x2202, 0x2207, 0x2261,
	0x2252, 0x226a, 0x226b, 0x221a, 0x223d, 0x221d, 0x2235, 0x222b,
	0x222c
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81f0[ 8 ] = {
	0x212b, 0x2030, 0x266f, 0x266d, 0x266a, 0x2020, 0x2021, 0x00b6
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8440[ 33 ] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401, 0x0416,
	0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e,
	0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426,
	0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e,
	0x042f
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8470[ 15 ] = {
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, 0x0436,
	0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x849f[ 32 ] = {
	                                                        0x2500,
	0x2502, 0x250c, 0x2510, 0x2518, 0x2514, 0x251c, 0x252c, 0x2524,
	0x2534, 0x253c, 0x2501, 0x2503, 0x250f, 0x2513, 0x251b, 0x2517,
	0x2523, 0x2533, 0x252b, 0x253b, 0x254b, 0x2520, 0x252f, 0x2528,
	0x2537, 0x253f, 0x251d, 0x2530, 0x2525, 0x2538, 0x2542
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x875f[ 23 ] = {
	                                                        0x3349,
	0x3314, 0x3322, 0x334d, 0x3318, 0x3327, 0x3303, 0x3336, 0x3351,
	0x3357, 0x330d, 0x3326, 0x3323, 0x332b, 0x334a, 0x333b, 0x339c,
	0x339d, 0x339e, 0x338e, 0x338f, 0x33c4, 0x33a1
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8780[ 29 ] = {
	0x301d, 0x301f, 0x2116, 0x33cd, 0x2121, 0x32a4, 0x32a5, 0x32a6,
	0x32a7, 0x32a8, 0x3231, 0x3232, 0x3239, 0x337e, 0x337d, 0x337c,
	0x2252, 0x2261, 0x222b, 0x222e, 0x2211, 0x221a, 0x22a5, 0x2220,
	0x221f, 0x22bf, 0x2235, 0x2229, 0x222a
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x889f[ 94 ] = {
	                                                        0x4e9c,
	0x5516, 0x5a03, 0x963f, 0x54c0, 0x611b, 0x6328, 0x59f6, 0x9022,
	0x8475, 0x831c, 0x7a50, 0x60aa, 0x63e1, 0x6e25, 0x65ed, 0x8466,
	0x82a6, 0x9bf5, 0x6893, 0x5727, 0x65a1, 0x6271, 0x5b9b, 0x59d0,
	0x867b, 0x98f4, 0x7d62, 0x7dbe, 0x9b8e, 0x6216, 0x7c9f, 0x88b7,
	0x5b89, 0x5eb5, 0x6309, 0x6697, 0x6848, 0x95c7, 0x978d, 0x674f,
	0x4ee5, 0x4f0a, 0x4f4d, 0x4f9d, 0x5049, 0x56f2, 0x5937, 0x59d4,
	0x5a01, 0x5c09, 0x60df, 0x610f, 0x6170, 0x6613, 0x6905, 0x70ba,
	0x754f, 0x7570, 0x79fb, 0x7dad, 0x7def, 0x80c3, 0x840e, 0x8863,
	0x8b02, 0x9055, 0x907a, 0x533b, 0x4e95, 0x4ea5, 0x57df, 0x80b2,
	0x90c1, 0x78ef, 0x4e00, 0x58f1, 0x6ea2, 0x9038, 0x7a32, 0x8328,
	0x828b, 0x9c2f, 0x5141, 0x5370, 0x54bd, 0x54e1, 0x56e0, 0x59fb,
	0x5f15, 0x98f2, 0x6deb, 0x80e4, 0x852d
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8940[ 189 ] = {
	0x9662, 0x9670, 0x96a0, 0x97fb, 0x540b, 0x53f3, 0x5b87, 0x70cf,
	0x7fbd, 0x8fc2, 0x96e8, 0x536f, 0x9d5c, 0x7aba, 0x4e11, 0x7893,
	0x81fc, 0x6e26, 0x5618, 0x5504, 0x6b1d, 0x851a, 0x9c3b, 0x59e5,
	0x53a9, 0x6d66, 0x74dc, 0x958f, 0x5642, 0x4e91, 0x904b, 0x96f2,
	0x834f, 0x990c, 0x53e1, 0x55b6, 0x5b30, 0x5f71, 0x6620, 0x66f3,
	0x6804, 0x6c38, 0x6cf3, 0x6d29, 0x745b, 0x76c8, 0x7a4e, 0x9834,
	0x82f1, 0x885b, 0x8a60, 0x92ed, 0x6db2, 0x75ab, 0x76ca, 0x99c5,
	0x60a6, 0x8b01, 0x8d8a, 0x95b2, 0x698e, 0x53ad, 0x5186, 0xfffd,
	0x5712, 0x5830, 0x5944, 0x5bb4, 0x5ef6, 0x6028, 0x63a9, 0x63f4,
	0x6cbf, 0x6f14, 0x708e, 0x7114, 0x7159, 0x71d5, 0x733f, 0x7e01,
	0x8276, 0x82d1, 0x8597, 0x9060, 0x925b, 0x9d1b, 0x5869, 0x65bc,
	0x6c5a, 0x7525, 0x51f9, 0x592e, 0x5965, 0x5f80, 0x5fdc, 0x62bc,
	0x65fa, 0x6a2a, 0x6b27, 0x6bb4, 0x738b, 0x7fc1, 0x8956, 0x9d2c,
	0x9d0e, 0x9ec4, 0x5ca1, 0x6c96, 0x837b, 0x5104, 0x5c4b, 0x61b6,
	0x81c6, 0x6876, 0x7261, 0x4e59, 0x4ffa, 0x5378, 0x6069, 0x6e29,
	0x7a4f, 0x97f3, 0x4e0b, 0x5316, 0x4eee, 0x4f55, 0x4f3d, 0x4fa1,
	0x4f73, 0x52a0, 0x53ef, 0x5609, 0x590f, 0x5ac1, 0x5bb6, 0x5be1,
	0x79d1, 0x6687, 0x679c, 0x67b6, 0x6b4c, 0x6cb3, 0x706b, 0x73c2,
	0x798d, 0x79be, 0x7a3c, 0x7b87, 0x82b1, 0x82db, 0x8304, 0x8377,
	0x83ef, 0x83d3, 0x8766, 0x8ab2, 0x5629, 0x8ca8, 0x8fe6, 0x904e,
	0x971e, 0x868a, 0x4fc4, 0x5ce8, 0x6211, 0x7259, 0x753b, 0x81e5,
	0x82bd, 0x86fe, 0x8cc0, 0x96c5, 0x9913, 0x99d5, 0x4ecb, 0x4f1a,
	0x89e3, 0x56de, 0x584a, 0x58ca, 0x5efb, 0x5feb, 0x602a, 0x6094,
	0x6062, 0x61d0, 0x6212, 0x62d0, 0x6539
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8a40[ 189 ] = {
	0x9b41, 0x6666, 0x68b0, 0x6d77, 0x7070, 0x754c, 0x7686, 0x7d75,
	0x82a5, 0x87f9, 0x958b, 0x968e, 0x8c9d, 0x51f1, 0x52be, 0x5916,
	0x54b3, 0x5bb3, 0x5d16, 0x6168, 0x6982, 0x6daf, 0x788d, 0x84cb,
	0x8857, 0x8a72, 0x93a7, 0x9ab8, 0x6d6c, 0x99a8, 0x86d9, 0x57a3,
	0x67ff, 0x86ce, 0x920e, 0x5283, 0x5687, 0x5404, 0x5ed3, 0x62e1,
	0x64b9, 0x683c, 0x6838, 0x6bbb, 0x7372, 0x78ba, 0x7a6b, 0x899a,
	0x89d2, 0x8d6b, 0x8f03, 0x90ed, 0x95a3, 0x9694, 0x9769, 0x5b66,
	0x5cb3, 0x697d, 0x984d, 0x984e, 0x639b, 0x7b20, 0x6a2b, 0xfffd,
	0x6a7f, 0x68b6, 0x9c0d, 0x6f5f, 0x5272, 0x559d, 0x6070, 0x62ec,
	0x6d3b, 0x6e07, 0x6ed1, 0x845b, 0x8910, 0x8f44, 0x4e14, 0x9c39,
	0x53f6, 0x691b, 0x6a3a, 0x9784, 0x682a, 0x515c, 0x7ac3, 0x84b2,
	0x91dc, 0x938c, 0x565b, 0x9d28, 0x6822, 0x8305, 0x8431, 0x7ca5,
	0x5208, 0x82c5, 0x74e6, 0x4e7e, 0x4f83, 0x51a0, 0x5bd2, 0x520a,
	0x52d8, 0x52e7, 0x5dfb, 0x559a, 0x582a, 0x59e6, 0x5b8c, 0x5b98,
	0x5bdb, 0x5e72, 0x5e79, 0x60a3, 0x611f, 0x6163, 0x61be, 0x63db,
	0x6562, 0x67d1, 0x6853, 0x68fa, 0x6b3e, 0x6b53, 0x6c57, 0x6f22,
	0x6f97, 0x6f45, 0x74b0, 0x7518, 0x76e3, 0x770b, 0x7aff, 0x7ba1,
	0x7c21, 0x7de9, 0x7f36, 0x7ff0, 0x809d, 0x8266, 0x839e, 0x89b3,
	0x8acc, 0x8cab, 0x9084, 0x9451, 0x9593, 0x9591, 0x95a2, 0x9665,
	0x97d3, 0x9928, 0x8218, 0x4e38, 0x542b, 0x5cb8, 0x5dcc, 0x73a9,
	0x764c, 0x773c, 0x5ca9, 0x7feb, 0x8d0b, 0x96c1, 0x9811, 0x9854,
	0x9858, 0x4f01, 0x4f0e, 0x5371, 0x559c, 0x5668, 0x57fa, 0x5947,
	0x5b09, 0x5bc4, 0x5c90, 0x5e0c, 0x5e7e, 0x5fcc, 0x63ee, 0x673a,
	0x65d7, 0x65e2, 0x671f, 0x68cb, 0x68c4
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8b40[ 189 ] = {
	0x6a5f, 0x5e30, 0x6bc5, 0x6c17, 0x6c7d, 0x757f, 0x7948, 0x5b63,
	0x7a00, 0x7d00, 0x5fbd, 0x898f, 0x8a18, 0x8cb4, 0x8d77, 0x8ecc,
	0x8f1d, 0x98e2, 0x9a0e, 0x9b3c, 0x4e80, 0x507d, 0x5100, 0x5993,
	0x5b9c, 0x622f, 0x6280, 0x64ec, 0x6b3a, 0x72a0, 0x7591, 0x7947,
	0x7fa9, 0x87fb, 0x8abc, 0x8b70, 0x63ac, 0x83ca, 0x97a0, 0x5409,
	0x5403, 0x55ab, 0x6854, 0x6a58, 0x8a70, 0x7827, 0x6775, 0x9ecd,
	0x5374, 0x5ba2, 0x811a, 0x8650, 0x9006, 0x4e18, 0x4e45, 0x4ec7,
	0x4f11, 0x53ca, 0x5438, 0x5bae, 0x5f13, 0x6025, 0x6551, 0xfffd,
	0x673d, 0x6c42, 0x6c72, 0x6ce3, 0x7078, 0x7403, 0x7a76, 0x7aae,
	0x7b08, 0x7d1a, 0x7cfe, 0x7d66, 0x65e7, 0x725b, 0x53bb, 0x5c45,
	0x5de8, 0x62d2, 0x62e0, 0x6319, 0x6e20, 0x865a, 0x8a31, 0x8ddd,
	0x92f8, 0x6f01, 0x79a6, 0x9b5a, 0x4ea8, 0x4eab, 0x4eac, 0x4f9b,
	0x4fa0, 0x50d1, 0x5147, 0x7af6, 0x5171, 0x51f6, 0x5354, 0x5321,
	0x537f, 0x53eb, 0x55ac, 0x5883, 0x5ce1, 0x5f37, 0x5f4a, 0x602f,
	0x6050, 0x606d, 0x631f, 0x6559, 0x6a4b, 0x6cc1, 0x72c2, 0x72ed,
	0x77ef, 0x80f8, 0x8105, 0x8208, 0x854e, 0x90f7, 0x93e1, 0x97ff,
	0x9957, 0x9a5a, 0x4ef0, 0x51dd, 0x5c2d, 0x6681, 0x696d, 0x5c40,
	0x66f2, 0x6975, 0x7389, 0x6850, 0x7c81, 0x50c5, 0x52e4, 0x5747,
	0x5dfe, 0x9326, 0x65a4, 0x6b23, 0x6b3d, 0x7434, 0x7981, 0x79bd,
	0x7b4b, 0x7dca, 0x82b9, 0x83cc, 0x887f, 0x895f, 0x8b39, 0x8fd1,
	0x91d1, 0x541f, 0x9280, 0x4e5d, 0x5036, 0x53e5, 0x533a, 0x72d7,
	0x7396, 0x77e9, 0x82e6, 0x8eaf, 0x99c6, 0x99c8, 0x99d2, 0x5177,
	0x611a, 0x865e, 0x55b0, 0x7a7a, 0x5076, 0x5bd3, 0x9047, 0x9685,
	0x4e32, 0x6adb, 0x91e7, 0x5c51, 0x5c48
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8c40[ 189 ] = {
	0x6398, 0x7a9f, 0x6c93, 0x9774, 0x8f61, 0x7aaa, 0x718a, 0x9688,
	0x7c82, 0x6817, 0x7e70, 0x6851, 0x936c, 0x52f2, 0x541b, 0x85ab,
	0x8a13, 0x7fa4, 0x8ecd, 0x90e1, 0x5366, 0x8888, 0x7941, 0x4fc2,
	0x50be, 0x5211, 0x5144, 0x5553, 0x572d, 0x73ea, 0x578b, 0x5951,
	0x5f62, 0x5f84, 0x6075, 0x6176, 0x6167, 0x61a9, 0x63b2, 0x643a,
	0x656c, 0x666f, 0x6842, 0x6e13, 0x7566, 0x7a3d, 0x7cfb, 0x7d4c,
	0x7d99, 0x7e4b, 0x7f6b, 0x830e, 0x834a, 0x86cd, 0x8a08, 0x8a63,
	0x8b66, 0x8efd, 0x981a, 0x9d8f, 0x82b8, 0x8fce, 0x9be8, 0xfffd,
	0x5287, 0x621f, 0x6483, 0x6fc0, 0x9699, 0x6841, 0x5091, 0x6b20,
	0x6c7a, 0x6f54, 0x7a74, 0x7d50, 0x8840, 0x8a23, 0x6708, 0x4ef6,
	0x5039, 0x5026, 0x5065, 0x517c, 0x5238, 0x5263, 0x55a7, 0x570f,
	0x5805, 0x5acc, 0x5efa, 0x61b2, 0x61f8, 0x62f3, 0x6372, 0x691c,
	0x6a29, 0x727d, 0x72ac, 0x732e, 0x7814, 0x786f, 0x7d79, 0x770c,
	0x80a9, 0x898b, 0x8b19, 0x8ce2, 0x8ed2, 0x9063, 0x9375, 0x967a,
	0x9855, 0x9a13, 0x9e78, 0x5143, 0x539f, 0x53b3, 0x5e7b, 0x5f26,
	0x6e1b, 0x6e90, 0x7384, 0x73fe, 0x7d43, 0x8237, 0x8a00, 0x8afa,
	0x9650, 0x4e4e, 0x500b, 0x53e4, 0x547c, 0x56fa, 0x59d1, 0x5b64,
	0x5df1, 0x5eab, 0x5f27, 0x6238, 0x6545, 0x67af, 0x6e56, 0x72d0,
	0x7cca, 0x88b4, 0x80a1, 0x80e1, 0x83f0, 0x864e, 0x8a87, 0x8de8,
	0x9237, 0x96c7, 0x9867, 0x9f13, 0x4e94, 0x4e92, 0x4f0d, 0x5348,
	0x5449, 0x543e, 0x5a2f, 0x5f8c, 0x5fa1, 0x609f, 0x68a7, 0x6a8e,
	0x745a, 0x7881, 0x8a9e, 0x8aa4, 0x8b77, 0x9190, 0x4e5e, 0x9bc9,
	0x4ea4, 0x4f7c, 0x4faf, 0x5019, 0x5016, 0x5149, 0x516c, 0x529f,
	0x52b9, 0x52fe, 0x539a, 0x53e3, 0x5411
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8d40[ 189 ] = {
	0x540e, 0x5589, 0x5751, 0x57a2, 0x597d, 0x5b54, 0x5b5d, 0x5b8f,
	0x5de5, 0x5de7, 0x5df7, 0x5e78, 0x5e83, 0x5e9a, 0x5eb7, 0x5f18,
	0x6052, 0x614c, 0x6297, 0x62d8, 0x63a7, 0x653b, 0x6602, 0x6643,
	0x66f4, 0x676d, 0x6821, 0x6897, 0x69cb, 0x6c5f, 0x6d2a, 0x6d69,
	0x6e2f, 0x6e9d, 0x7532, 0x7687, 0x786c, 0x7a3f, 0x7ce0, 0x7d05,
	0x7d18, 0x7d5e, 0x7db1, 0x8015, 0x8003, 0x80af, 0x80b1, 0x8154,
	0x818f, 0x822a, 0x8352, 0x884c, 0x8861, 0x8b1b, 0x8ca2, 0x8cfc,
	0x90ca, 0x9175, 0x9271, 0x783f, 0x92fc, 0x95a4, 0x964d, 0xfffd,
	0x9805, 0x9999, 0x9ad8, 0x9d3b, 0x525b, 0x52ab, 0x53f7, 0x5408,
	0x58d5, 0x62f7, 0x6fe0, 0x8c6a, 0x8f5f, 0x9eb9, 0x514b, 0x523b,
	0x544a, 0x56fd, 0x7a40, 0x9177, 0x9d60, 0x9ed2, 0x7344, 0x6f09,
	0x8170, 0x7511, 0x5ffd, 0x60da, 0x9aa8, 0x72db, 0x8fbc, 0x6b64,
	0x9803, 0x4eca, 0x56f0, 0x5764, 0x58be, 0x5a5a, 0x6068, 0x61c7,
	0x660f, 0x6606, 0x6839, 0x68b1, 0x6df7, 0x75d5, 0x7d3a, 0x826e,
	0x9b42, 0x4e9b, 0x4f50, 0x53c9, 0x5506, 0x5d6f, 0x5de6, 0x5dee,
	0x67fb, 0x6c99, 0x7473, 0x7802, 0x8a50, 0x9396, 0x88df, 0x5750,
	0x5ea7, 0x632b, 0x50b5, 0x50ac, 0x518d, 0x6700, 0x54c9, 0x585e,
	0x59bb, 0x5bb0, 0x5f69, 0x624d, 0x63a1, 0x683d, 0x6b73, 0x6e08,
	0x707d, 0x91c7, 0x7280, 0x7815, 0x7826, 0x796d, 0x658e, 0x7d30,
	0x83dc, 0x88c1, 0x8f09, 0x969b, 0x5264, 0x5728, 0x6750, 0x7f6a,
	0x8ca1, 0x51b4, 0x5742, 0x962a, 0x583a, 0x698a, 0x80b4, 0x54b2,
	0x5d0e, 0x57fc, 0x7895, 0x9dfa, 0x4f5c, 0x524a, 0x548b, 0x643e,
	0x6628, 0x6714, 0x67f5, 0x7a84, 0x7b56, 0x7d22, 0x932f, 0x685c,
	0x9bad, 0x7b39, 0x5319, 0x518a, 0x5237
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8e40[ 189 ] = {
	0x5bdf, 0x62f6, 0x64ae, 0x64e6, 0x672d, 0x6bba, 0x85a9, 0x96d1,
	0x7690, 0x9bd6, 0x634c, 0x9306, 0x9bab, 0x76bf, 0x6652, 0x4e09,
	0x5098, 0x53c2, 0x5c71, 0x60e8, 0x6492, 0x6563, 0x685f, 0x71e6,
	0x73ca, 0x7523, 0x7b97, 0x7e82, 0x8695, 0x8b83, 0x8cdb, 0x9178,
	0x9910, 0x65ac, 0x66ab, 0x6b8b, 0x4ed5, 0x4ed4, 0x4f3a, 0x4f7f,
	0x523a, 0x53f8, 0x53f2, 0x55e3, 0x56db, 0x58eb, 0x59cb, 0x59c9,
	0x59ff, 0x5b50, 0x5c4d, 0x5e02, 0x5e2b, 0x5fd7, 0x601d, 0x6307,
	0x652f, 0x5b5c, 0x65af, 0x65bd, 0x65e8, 0x679d, 0x6b62, 0xfffd,
	0x6b7b, 0x6c0f, 0x7345, 0x7949, 0x79c1, 0x7cf8, 0x7d19, 0x7d2b,
	0x80a2, 0x8102, 0x81f3, 0x8996, 0x8a5e, 0x8a69, 0x8a66, 0x8a8c,
	0x8aee, 0x8cc7, 0x8cdc, 0x96cc, 0x98fc, 0x6b6f, 0x4e8b, 0x4f3c,
	0x4f8d, 0x5150, 0x5b57, 0x5bfa, 0x6148, 0x6301, 0x6642, 0x6b21,
	0x6ecb, 0x6cbb, 0x723e, 0x74bd, 0x75d4, 0x78c1, 0x793a, 0x800c,
	0x8033, 0x81ea, 0x8494, 0x8f9e, 0x6c50, 0x9e7f, 0x5f0f, 0x8b58,
	0x9d2b, 0x7afa, 0x8ef8, 0x5b8d, 0x96eb, 0x4e03, 0x53f1, 0x57f7,
	0x5931, 0x5ac9, 0x5ba4, 0x6089, 0x6e7f, 0x6f06, 0x75be, 0x8cea,
	0x5b9f, 0x8500, 0x7be0, 0x5072, 0x67f4, 0x829d, 0x5c61, 0x854a,
	0x7e1e, 0x820e, 0x5199, 0x5c04, 0x6368, 0x8d66, 0x659c, 0x716e,
	0x793e, 0x7d17, 0x8005, 0x8b1d, 0x8eca, 0x906e, 0x86c7, 0x90aa,
	0x501f, 0x52fa, 0x5c3a, 0x6753, 0x707c, 0x7235, 0x914c, 0x91c8,
	0x932b, 0x82e5, 0x5bc2, 0x5f31, 0x60f9, 0x4e3b, 0x53d6, 0x5b88,
	0x624b, 0x6731, 0x6b8a, 0x72e9, 0x73e0, 0x7a2e, 0x816b, 0x8da3,
	0x9152, 0x9996, 0x5112, 0x53d7, 0x546a, 0x5bff, 0x6388, 0x6a39,
	0x7dac, 0x9700, 0x56da, 0x53ce, 0x5468
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8f40[ 189 ] = {
	0x5b97, 0x5c31, 0x5dde, 0x4fee, 0x6101, 0x62fe, 0x6d32, 0x79c0,
	0x79cb, 0x7d42, 0x7e4d, 0x7fd2, 0x81ed, 0x821f, 0x8490, 0x8846,
	0x8972, 0x8b90, 0x8e74, 0x8f2f, 0x9031, 0x914b, 0x916c, 0x96c6,
	0x919c, 0x4ec0, 0x4f4f, 0x5145, 0x5341, 0x5f93, 0x620e, 0x67d4,
	0x6c41, 0x6e0b, 0x7363, 0x7e26, 0x91cd, 0x9283, 0x53d4, 0x5919,
	0x5bbf, 0x6dd1, 0x795d, 0x7e2e, 0x7c9b, 0x587e, 0x719f, 0x51fa,
	0x8853, 0x8ff0, 0x4fca, 0x5cfb, 0x6625, 0x77ac, 0x7ae3, 0x821c,
	0x99ff, 0x51c6, 0x5faa, 0x65ec, 0x696f, 0x6b89, 0x6df3, 0xfffd,
	0x6e96, 0x6f64, 0x76fe, 0x7d14, 0x5de1, 0x9075, 0x9187, 0x9806,
	0x51e6, 0x521d, 0x6240, 0x6691, 0x66d9, 0x6e1a, 0x5eb6, 0x7dd2,
	0x7f72, 0x66f8, 0x85af, 0x85f7, 0x8af8, 0x52a9, 0x53d9, 0x5973,
	0x5e8f, 0x5f90, 0x6055, 0x92e4, 0x9664, 0x50b7, 0x511f, 0x52dd,
	0x5320, 0x5347, 0x53ec, 0x54e8, 0x5546, 0x5531, 0x5617, 0x5968,
	0x59be, 0x5a3c, 0x5bb5, 0x5c06, 0x5c0f, 0x5c11, 0x5c1a, 0x5e84,
	0x5e8a, 0x5ee0, 0x5f70, 0x627f, 0x6284, 0x62db, 0x638c, 0x6377,
	0x6607, 0x660c, 0x662d, 0x6676, 0x677e, 0x68a2, 0x6a1f, 0x6a35,
	0x6cbc, 0x6d88, 0x6e09, 0x6e58, 0x713c, 0x7126, 0x7167, 0x75c7,
	0x7701, 0x785d, 0x7901, 0x7965, 0x79f0, 0x7ae0, 0x7b11, 0x7ca7,
	0x7d39, 0x8096, 0x83d6, 0x848b, 0x8549, 0x885d, 0x88f3, 0x8a1f,
	0x8a3c, 0x8a54, 0x8a73, 0x8c61, 0x8cde, 0x91a4, 0x9266, 0x937e,
	0x9418, 0x969c, 0x9798, 0x4e0a, 0x4e08, 0x4e1e, 0x4e57, 0x5197,
	0x5270, 0x57ce, 0x5834, 0x58cc, 0x5b22, 0x5e38, 0x60c5, 0x64fe,
	0x6761, 0x6756, 0x6d44, 0x72b6, 0x7573, 0x7a63, 0x84b8, 0x8b72,
	0x91b8, 0x9320, 0x5631, 0x57f4, 0x98fe
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9040[ 189 ] = {
	0x62ed, 0x690d, 0x6b96, 0x71ed, 0x7e54, 0x8077, 0x8272, 0x89e6,
	0x98df, 0x8755, 0x8fb1, 0x5c3b, 0x4f38, 0x4fe1, 0x4fb5, 0x5507,
	0x5a20, 0x5bdd, 0x5be9, 0x5fc3, 0x614e, 0x632f, 0x65b0, 0x664b,
	0x68ee, 0x699b, 0x6d78, 0x6df1, 0x7533, 0x75b9, 0x771f, 0x795e,
	0x79e6, 0x7d33, 0x81e3, 0x82af, 0x85aa, 0x89aa, 0x8a3a, 0x8eab,
	0x8f9b, 0x9032, 0x91dd, 0x9707, 0x4eba, 0x4ec1, 0x5203, 0x5875,
	0x58ec, 0x5c0b, 0x751a, 0x5c3d, 0x814e, 0x8a0a, 0x8fc5, 0x9663,
	0x976d, 0x7b25, 0x8acf, 0x9808, 0x9162, 0x56f3, 0x53a8, 0xfffd,
	0x9017, 0x5439, 0x5782, 0x5e25, 0x63a8, 0x6c34, 0x708a, 0x7761,
	0x7c8b, 0x7fe0, 0x8870, 0x9042, 0x9154, 0x9310, 0x9318, 0x968f,
	0x745e, 0x9ac4, 0x5d07, 0x5d69, 0x6570, 0x67a2, 0x8da8, 0x96db,
	0x636e, 0x6749, 0x6919, 0x83c5, 0x9817, 0x96c0, 0x88fe, 0x6f84,
	0x647a, 0x5bf8, 0x4e16, 0x702c, 0x755d, 0x662f, 0x51c4, 0x5236,
	0x52e2, 0x59d3, 0x5f81, 0x6027, 0x6210, 0x653f, 0x6574, 0x661f,
	0x6674, 0x68f2, 0x6816, 0x6b63, 0x6e05, 0x7272, 0x751f, 0x76db,
	0x7cbe, 0x8056, 0x58f0, 0x88fd, 0x897f, 0x8aa0, 0x8a93, 0x8acb,
	0x901d, 0x9192, 0x9752, 0x9759, 0x6589, 0x7a0e, 0x8106, 0x96bb,
	0x5e2d, 0x60dc, 0x621a, 0x65a5, 0x6614, 0x6790, 0x77f3, 0x7a4d,
	0x7c4d, 0x7e3e, 0x810a, 0x8cac, 0x8d64, 0x8de1, 0x8e5f, 0x78a9,
	0x5207, 0x62d9, 0x63a5, 0x6442, 0x6298, 0x8a2d, 0x7a83, 0x7bc0,
	0x8aac, 0x96ea, 0x7d76, 0x820c, 0x8749, 0x4ed9, 0x5148, 0x5343,
	0x5360, 0x5ba3, 0x5c02, 0x5c16, 0x5ddd, 0x6226, 0x6247, 0x64b0,
	0x6813, 0x6834, 0x6cc9, 0x6d45, 0x6d17, 0x67d3, 0x6f5c, 0x714e,
	0x717d, 0x65cb, 0x7a7f, 0x7bad, 0x7dda
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9140[ 189 ] = {
	0x7e4a, 0x7fa8, 0x817a, 0x821b, 0x8239, 0x85a6, 0x8a6e, 0x8cce,
	0x8df5, 0x9078, 0x9077, 0x92ad, 0x9291, 0x9583, 0x9bae, 0x524d,
	0x5584, 0x6f38, 0x7136, 0x5168, 0x7985, 0x7e55, 0x81b3, 0x7cce,
	0x564c, 0x5851, 0x5ca8, 0x63aa, 0x66fe, 0x66fd, 0x695a, 0x72d9,
	0x758f, 0x758e, 0x790e, 0x7956, 0x79df, 0x7c97, 0x7d20, 0x7d44,
	0x8607, 0x8a34, 0x963b, 0x9061, 0x9f20, 0x50e7, 0x5275, 0x53cc,
	0x53e2, 0x5009, 0x55aa, 0x58ee, 0x594f, 0x723d, 0x5b8b, 0x5c64,
	0x531d, 0x60e3, 0x60f3, 0x635c, 0x6383, 0x633f, 0x63bb, 0xfffd,
	0x64cd, 0x65e9, 0x66f9, 0x5de3, 0x69cd, 0x69fd, 0x6f15, 0x71e5,
	0x4e89, 0x75e9, 0x76f8, 0x7a93, 0x7cdf, 0x7dcf, 0x7d9c, 0x8061,
	0x8349, 0x8358, 0x846c, 0x84bc, 0x85fb, 0x88c5, 0x8d70, 0x9001,
	0x906d, 0x9397, 0x971c, 0x9a12, 0x50cf, 0x5897, 0x618e, 0x81d3,
	0x8535, 0x8d08, 0x9020, 0x4fc3, 0x5074, 0x5247, 0x5373, 0x606f,
	0x6349, 0x675f, 0x6e2c, 0x8db3, 0x901f, 0x4fd7, 0x5c5e, 0x8cca,
	0x65cf, 0x7d9a, 0x5352, 0x8896, 0x5176, 0x63c3, 0x5b58, 0x5b6b,
	0x5c0a, 0x640d, 0x6751, 0x905c, 0x4ed6, 0x591a, 0x592a, 0x6c70,
	0x8a51, 0x553e, 0x5815, 0x59a5, 0x60f0, 0x6253, 0x67c1, 0x8235,
	0x6955, 0x9640, 0x99c4, 0x9a28, 0x4f53, 0x5806, 0x5bfe, 0x8010,
	0x5cb1, 0x5e2f, 0x5f85, 0x6020, 0x614b, 0x6234, 0x66ff, 0x6cf0,
	0x6ede, 0x80ce, 0x817f, 0x82d4, 0x888b, 0x8cb8, 0x9000, 0x902e,
	0x968a, 0x9edb, 0x9bdb, 0x4ee3, 0x53f0, 0x5927, 0x7b2c, 0x918d,
	0x984c, 0x9df9, 0x6edd, 0x7027, 0x5353, 0x5544, 0x5b85, 0x6258,
	0x629e, 0x62d3, 0x6ca2, 0x6fef, 0x7422, 0x8a17, 0x9438, 0x6fc1,
	0x8afe, 0x8338, 0x51e7, 0x86f8, 0x53ea
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9240[ 189 ] = {
	0x53e9, 0x4f46, 0x9054, 0x8fb0, 0x596a, 0x8131, 0x5dfd, 0x7aea,
	0x8fbf, 0x68da, 0x8c37, 0x72f8, 0x9c48, 0x6a3d, 0x8ab0, 0x4e39,
	0x5358, 0x5606, 0x5766, 0x62c5, 0x63a2, 0x65e6, 0x6b4e, 0x6de1,
	0x6e5b, 0x70ad, 0x77ed, 0x7aef, 0x7baa, 0x7dbb, 0x803d, 0x80c6,
	0x86cb, 0x8a95, 0x935b, 0x56e3, 0x58c7, 0x5f3e, 0x65ad, 0x6696,
	0x6a80, 0x6bb5, 0x7537, 0x8ac7, 0x5024, 0x77e5, 0x5730, 0x5f1b,
	0x6065, 0x667a, 0x6c60, 0x75f4, 0x7a1a, 0x7f6e, 0x81f4, 0x8718,
	0x9045, 0x99b3, 0x7bc9, 0x755c, 0x7af9, 0x7b51, 0x84c4, 0xfffd,
	0x9010, 0x79e9, 0x7a92, 0x8336, 0x5ae1, 0x7740, 0x4e2d, 0x4ef2,
	0x5b99, 0x5fe0, 0x62bd, 0x663c, 0x67f1, 0x6ce8, 0x866b, 0x8877,
	0x8a3b, 0x914e, 0x92f3, 0x99d0, 0x6a17, 0x7026, 0x732a, 0x82e7,
	0x8457, 0x8caf, 0x4e01, 0x5146, 0x51cb, 0x558b, 0x5bf5, 0x5e16,
	0x5e33, 0x5e81, 0x5f14, 0x5f35, 0x5f6b, 0x5fb4, 0x61f2, 0x6311,
	0x66a2, 0x671d, 0x6f6e, 0x7252, 0x753a, 0x773a, 0x8074, 0x8139,
	0x8178, 0x8776, 0x8abf, 0x8adc, 0x8d85, 0x8df3, 0x929a, 0x9577,
	0x9802, 0x9ce5, 0x52c5, 0x6357, 0x76f4, 0x6715, 0x6c88, 0x73cd,
	0x8cc3, 0x93ae, 0x9673, 0x6d25, 0x589c, 0x690e, 0x69cc, 0x8ffd,
	0x939a, 0x75db, 0x901a, 0x585a, 0x6802, 0x63b4, 0x69fb, 0x4f43,
	0x6f2c, 0x67d8, 0x8fbb, 0x8526, 0x7db4, 0x9354, 0x693f, 0x6f70,
	0x576a, 0x58f7, 0x5b2c, 0x7d2c, 0x722a, 0x540a, 0x91e3, 0x9db4,
	0x4ead, 0x4f4e, 0x505c, 0x5075, 0x5243, 0x8c9e, 0x5448, 0x5824,
	0x5b9a, 0x5e1d, 0x5e95, 0x5ead, 0x5ef7, 0x5f1f, 0x608c, 0x62b5,
	0x633a, 0x63d0, 0x68af, 0x6c40, 0x7887, 0x798e, 0x7a0b, 0x7de0,
	0x8247, 0x8a02, 0x8ae6, 0x8e44, 0x9013
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9340[ 189 ] = {
	0x90b8, 0x912d, 0x91d8, 0x9f0e, 0x6ce5, 0x6458, 0x64e2, 0x6575,
	0x6ef4, 0x7684, 0x7b1b, 0x9069, 0x93d1, 0x6eba, 0x54f2, 0x5fb9,
	0x64a4, 0x8f4d, 0x8fed, 0x9244, 0x5178, 0x586b, 0x5929, 0x5c55,
	0x5e97, 0x6dfb, 0x7e8f, 0x751c, 0x8cbc, 0x8ee2, 0x985b, 0x70b9,
	0x4f1d, 0x6bbf, 0x6fb1, 0x7530, 0x96fb, 0x514e, 0x5410, 0x5835,
	0x5857, 0x59ac, 0x5c60, 0x5f92, 0x6597, 0x675c, 0x6e21, 0x767b,
	0x83df, 0x8ced, 0x9014, 0x90fd, 0x934d, 0x7825, 0x783a, 0x52aa,
	0x5ea6, 0x571f, 0x5974, 0x6012, 0x5012, 0x515a, 0x51ac, 0xfffd,
	0x51cd, 0x5200, 0x5510, 0x5854, 0x5858, 0x5957, 0x5b95, 0x5cf6,
	0x5d8b, 0x60bc, 0x6295, 0x642d, 0x6771, 0x6843, 0x68bc, 0x68df,
	0x76d7, 0x6dd8, 0x6e6f, 0x6d9b, 0x706f, 0x71c8, 0x5f53, 0x75d8,
	0x7977, 0x7b49, 0x7b54, 0x7b52, 0x7cd6, 0x7d71, 0x5230, 0x8463,
	0x8569, 0x85e4, 0x8a0e, 0x8b04, 0x8c46, 0x8e0f, 0x9003, 0x900f,
	0x9419, 0x9676, 0x982d, 0x9a30, 0x95d8, 0x50cd, 0x52d5, 0x540c,
	0x5802, 0x5c0e, 0x61a7, 0x649e, 0x6d1e, 0x77b3, 0x7ae5, 0x80f4,
	0x8404, 0x9053, 0x9285, 0x5ce0, 0x9d07, 0x533f, 0x5f97, 0x5fb3,
	0x6d9c, 0x7279, 0x7763, 0x79bf, 0x7be4, 0x6bd2, 0x72ec, 0x8aad,
	0x6803, 0x6a61, 0x51f8, 0x7a81, 0x6934, 0x5c4a, 0x9cf6, 0x82eb,
	0x5bc5, 0x9149, 0x701e, 0x5678, 0x5c6f, 0x60c7, 0x6566, 0x6c8c,
	0x8c5a, 0x9041, 0x9813, 0x5451, 0x66c7, 0x920d, 0x5948, 0x90a3,
	0x5185, 0x4e4d, 0x51ea, 0x8599, 0x8b0e, 0x7058, 0x637a, 0x934b,
	0x6962, 0x99b4, 0x7e04, 0x7577, 0x5357, 0x6960, 0x8edf, 0x96e3,
	0x6c5d, 0x4e8c, 0x5c3c, 0x5f10, 0x8fe9, 0x5302, 0x8cd1, 0x8089,
	0x8679, 0x5eff, 0x65e5, 0x4e73, 0x5165
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9440[ 189 ] = {
	0x5982, 0x5c3f, 0x97ee, 0x4efb, 0x598a, 0x5fcd, 0x8a8d, 0x6fe1,
	0x79b0, 0x7962, 0x5be7, 0x8471, 0x732b, 0x71b1, 0x5e74, 0x5ff5,
	0x637b, 0x649a, 0x71c3, 0x7c98, 0x4e43, 0x5efc, 0x4e4b, 0x57dc,
	0x56a2, 0x60a9, 0x6fc3, 0x7d0d, 0x80fd, 0x8133, 0x81bf, 0x8fb2,
	0x8997, 0x86a4, 0x5df4, 0x628a, 0x64ad, 0x8987, 0x6777, 0x6ce2,
	0x6d3e, 0x7436, 0x7834, 0x5a46, 0x7f75, 0x82ad, 0x99ac, 0x4ff3,
	0x5ec3, 0x62dd, 0x6392, 0x6557, 0x676f, 0x76c3, 0x724c, 0x80cc,
	0x80ba, 0x8f29, 0x914d, 0x500d, 0x57f9, 0x5a92, 0x6885, 0xfffd,
	0x6973, 0x7164, 0x72fd, 0x8cb7, 0x58f2, 0x8ce0, 0x966a, 0x9019,
	0x877f, 0x79e4, 0x77e7, 0x8429, 0x4f2f, 0x5265, 0x535a, 0x62cd,
	0x67cf, 0x6cca, 0x767d, 0x7b94, 0x7c95, 0x8236, 0x8584, 0x8feb,
	0x66dd, 0x6f20, 0x7206, 0x7e1b, 0x83ab, 0x99c1, 0x9ea6, 0x51fd,
	0x7bb1, 0x7872, 0x7bb8, 0x8087, 0x7b48, 0x6ae8, 0x5e61, 0x808c,
	0x7551, 0x7560, 0x516b, 0x9262, 0x6e8c, 0x767a, 0x9197, 0x9aea,
	0x4f10, 0x7f70, 0x629c, 0x7b4f, 0x95a5, 0x9ce9, 0x567a, 0x5859,
	0x86e4, 0x96bc, 0x4f34, 0x5224, 0x534a, 0x53cd, 0x53db, 0x5e06,
	0x642c, 0x6591, 0x677f, 0x6c3e, 0x6c4e, 0x7248, 0x72af, 0x73ed,
	0x7554, 0x7e41, 0x822c, 0x85e9, 0x8ca9, 0x7bc4, 0x91c6, 0x7169,
	0x9812, 0x98ef, 0x633d, 0x6669, 0x756a, 0x76e4, 0x78d0, 0x8543,
	0x86ee, 0x532a, 0x5351, 0x5426, 0x5983, 0x5e87, 0x5f7c, 0x60b2,
	0x6249, 0x6279, 0x62ab, 0x6590, 0x6bd4, 0x6ccc, 0x75b2, 0x76ae,
	0x7891, 0x79d8, 0x7dcb, 0x7f77, 0x80a5, 0x88ab, 0x8ab9, 0x8cbb,
	0x907f, 0x975e, 0x98db, 0x6a0b, 0x7c38, 0x5099, 0x5c3e, 0x5fae,
	0x6787, 0x6bd8, 0x7435, 0x7709, 0x7f8e
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9540[ 189 ] = {
	0x9f3b, 0x67ca, 0x7a17, 0x5339, 0x758b, 0x9aed, 0x5f66, 0x819d,
	0x83f1, 0x8098, 0x5f3c, 0x5fc5, 0x7562, 0x7b46, 0x903c, 0x6867,
	0x59eb, 0x5a9b, 0x7d10, 0x767e, 0x8b2c, 0x4ff5, 0x5f6a, 0x6a19,
	0x6c37, 0x6f02, 0x74e2, 0x7968, 0x8868, 0x8a55, 0x8c79, 0x5edf,
	0x63cf, 0x75c5, 0x79d2, 0x82d7, 0x9328, 0x92f2, 0x849c, 0x86ed,
	0x9c2d, 0x54c1, 0x5f6c, 0x658c, 0x6d5c, 0x7015, 0x8ca7, 0x8cd3,
	0x983b, 0x654f, 0x74f6, 0x4e0d, 0x4ed8, 0x57e0, 0x592b, 0x5a66,
	0x5bcc, 0x51a8, 0x5e03, 0x5e9c, 0x6016, 0x6276, 0x6577, 0xfffd,
	0x65a7, 0x666e, 0x6d6e, 0x7236, 0x7b26, 0x8150, 0x819a, 0x8299,
	0x8b5c, 0x8ca0, 0x8ce6, 0x8d74, 0x961c, 0x9644, 0x4fae, 0x64ab,
	0x6b66, 0x821e, 0x8461, 0x856a, 0x90e8, 0x5c01, 0x6953, 0x98a8,
	0x847a, 0x8557, 0x4f0f, 0x526f, 0x5fa9, 0x5e45, 0x670d, 0x798f,
	0x8179, 0x8907, 0x8986, 0x6df5, 0x5f17, 0x6255, 0x6cb8, 0x4ecf,
	0x7269, 0x9b92, 0x5206, 0x543b, 0x5674, 0x58b3, 0x61a4, 0x626e,
	0x711a, 0x596e, 0x7c89, 0x7cde, 0x7d1b, 0x96f0, 0x6587, 0x805e,
	0x4e19, 0x4f75, 0x5175, 0x5840, 0x5e63, 0x5e73, 0x5f0a, 0x67c4,
	0x4e26, 0x853d, 0x9589, 0x965b, 0x7c73, 0x9801, 0x50fb, 0x58c1,
	0x7656, 0x78a7, 0x5225, 0x77a5, 0x8511, 0x7b86, 0x504f, 0x5909,
	0x7247, 0x7bc7, 0x7de8, 0x8fba, 0x8fd4, 0x904d, 0x4fbf, 0x52c9,
	0x5a29, 0x5f01, 0x97ad, 0x4fdd, 0x8217, 0x92ea, 0x5703, 0x6355,
	0x6b69, 0x752b, 0x88dc, 0x8f14, 0x7a42, 0x52df, 0x5893, 0x6155,
	0x620a, 0x66ae, 0x6bcd, 0x7c3f, 0x83e9, 0x5023, 0x4ff8, 0x5305,
	0x5446, 0x5831, 0x5949, 0x5b9d, 0x5cf0, 0x5cef, 0x5d29, 0x5e96,
	0x62b1, 0x6367, 0x653e, 0x65b9, 0x670b
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9640[ 189 ] = {
	0x6cd5, 0x6ce1, 0x70f9, 0x7832, 0x7e2b, 0x80de, 0x82b3, 0x840c,
	0x84ec, 0x8702, 0x8912, 0x8a2a, 0x8c4a, 0x90a6, 0x92d2, 0x98fd,
	0x9cf3, 0x9d6c, 0x4e4f, 0x4ea1, 0x508d, 0x5256, 0x574a, 0x59a8,
	0x5e3d, 0x5fd8, 0x5fd9, 0x623f, 0x66b4, 0x671b, 0x67d0, 0x68d2,
	0x5192, 0x7d21, 0x80aa, 0x81a8, 0x8b00, 0x8c8c, 0x8cbf, 0x927e,
	0x9632, 0x5420, 0x982c, 0x5317, 0x50d5, 0x535c, 0x58a8, 0x64b2,
	0x6734, 0x7267, 0x7766, 0x7a46, 0x91e6, 0x52c3, 0x6ca1, 0x6b86,
	0x5800, 0x5e4c, 0x5954, 0x672c, 0x7ffb, 0x51e1, 0x76c6, 0xfffd,
	0x6469, 0x78e8, 0x9b54, 0x9ebb, 0x57cb, 0x59b9, 0x6627, 0x679a,
	0x6bce, 0x54e9, 0x69d9, 0x5e55, 0x819c, 0x6795, 0x9baa, 0x67fe,
	0x9c52, 0x685d, 0x4ea6, 0x4fe3, 0x53c8, 0x62b9, 0x672b, 0x6cab,
	0x8fc4, 0x4fad, 0x7e6d, 0x9ebf, 0x4e07, 0x6162, 0x6e80, 0x6f2b,
	0x8513, 0x5473, 0x672a, 0x9b45, 0x5df3, 0x7b95, 0x5cac, 0x5bc6,
	0x871c, 0x6e4a, 0x84d1, 0x7a14, 0x8108, 0x5999, 0x7c8d, 0x6c11,
	0x7720, 0x52d9, 0x5922, 0x7121, 0x725f, 0x77db, 0x9727, 0x9d61,
	0x690b, 0x5a7f, 0x5a18, 0x51a5, 0x540d, 0x547d, 0x660e, 0x76df,
	0x8ff7, 0x9298, 0x9cf4, 0x59ea, 0x725d, 0x6ec5, 0x514d, 0x68c9,
	0x7dbf, 0x7dec, 0x9762, 0x9eba, 0x6478, 0x6a21, 0x8302, 0x5984,
	0x5b5f, 0x6bdb, 0x731b, 0x76f2, 0x7db2, 0x8017, 0x8499, 0x5132,
	0x6728, 0x9ed9, 0x76ee, 0x6762, 0x52ff, 0x9905, 0x5c24, 0x623b,
	0x7c7e, 0x8cb0, 0x554f, 0x60b6, 0x7d0b, 0x9580, 0x5301, 0x4e5f,
	0x51b6, 0x591c, 0x723a, 0x8036, 0x91ce, 0x5f25, 0x77e2, 0x5384,
	0x5f79, 0x7d04, 0x85ac, 0x8a33, 0x8e8d, 0x9756, 0x67f3, 0x85ae,
	0x9453, 0x6109, 0x6108, 0x6cb9, 0x7652
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9740[ 189 ] = {
	0x8aed, 0x8f38, 0x552f, 0x4f51, 0x512a, 0x52c7, 0x53cb, 0x5ba5,
	0x5e7d, 0x60a0, 0x6182, 0x63d6, 0x6709, 0x67da, 0x6e67, 0x6d8c,
	0x7336, 0x7337, 0x7531, 0x7950, 0x88d5, 0x8a98, 0x904a, 0x9091,
	0x90f5, 0x96c4, 0x878d, 0x5915, 0x4e88, 0x4f59, 0x4e0e, 0x8a89,
	0x8f3f, 0x9810, 0x50ad, 0x5e7c, 0x5996, 0x5bb9, 0x5eb8, 0x63da,
	0x63fa, 0x64c1, 0x66dc, 0x694a, 0x69d8, 0x6d0b, 0x6eb6, 0x7194,
	0x7528, 0x7aaf, 0x7f8a, 0x8000, 0x8449, 0x84c9, 0x8981, 0x8b21,
	0x8e0a, 0x9065, 0x967d, 0x990a, 0x617e, 0x6291, 0x6b32, 0xfffd,
	0x6c83, 0x6d74, 0x7fcc, 0x7ffc, 0x6dc0, 0x7f85, 0x87ba, 0x88f8,
	0x6765, 0x83b1, 0x983c, 0x96f7, 0x6d1b, 0x7d61, 0x843d, 0x916a,
	0x4e71, 0x5375, 0x5d50, 0x6b04, 0x6feb, 0x85cd, 0x862d, 0x89a7,
	0x5229, 0x540f, 0x5c65, 0x674e, 0x68a8, 0x7406, 0x7483, 0x75e2,
	0x88cf, 0x88e1, 0x91cc, 0x96e2, 0x9678, 0x5f8b, 0x7387, 0x7acb,
	0x844e, 0x63a0, 0x7565, 0x5289, 0x6d41, 0x6e9c, 0x7409, 0x7559,
	0x786b, 0x7c92, 0x9686, 0x7adc, 0x9f8d, 0x4fb6, 0x616e, 0x65c5,
	0x865c, 0x4e86, 0x4eae, 0x50da, 0x4e21, 0x51cc, 0x5bee, 0x6599,
	0x6881, 0x6dbc, 0x731f, 0x7642, 0x77ad, 0x7a1c, 0x7ce7, 0x826f,
	0x8ad2, 0x907c, 0x91cf, 0x9675, 0x9818, 0x529b, 0x7dd1, 0x502b,
	0x5398, 0x6797, 0x6dcb, 0x71d0, 0x7433, 0x81e8, 0x8f2a, 0x96a3,
	0x9c57, 0x9e9f, 0x7460, 0x5841, 0x6d99, 0x7d2f, 0x985e, 0x4ee4,
	0x4f36, 0x4f8b, 0x51b7, 0x52b1, 0x5dba, 0x601c, 0x73b2, 0x793c,
	0x82d3, 0x9234, 0x96b7, 0x96f6, 0x970a, 0x9e97, 0x9f62, 0x66a6,
	0x6b74, 0x5217, 0x52a3, 0x70c8, 0x88c2, 0x5ec9, 0x604b, 0x6190,
	0x6f23, 0x7149, 0x7c3e, 0x7df4, 0x806f
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9840[ 51 ] = {
	0x84ee, 0x9023, 0x932c, 0x5442, 0x9b6f, 0x6ad3, 0x7089, 0x8cc2,
	0x8def, 0x9732, 0x52b4, 0x5a41, 0x5eca, 0x5f04, 0x6717, 0x697c,
	0x6994, 0x6d6a, 0x6f0f, 0x7262, 0x72fc, 0x7bed, 0x8001, 0x807e,
	0x874b, 0x90ce, 0x516d, 0x9e93, 0x7984, 0x808b, 0x9332, 0x8ad6,
	0x502d, 0x548c, 0x8a71, 0x6b6a, 0x8cc4, 0x8107, 0x60d1, 0x67a0,
	0x9df2, 0x4e99, 0x4e98, 0x9c10, 0x8a6b, 0x85c1, 0x8568, 0x6900,
	0x6e7e, 0x7897, 0x8155
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x989f[ 94 ] = {
	                                                        0x5f0c,
	0x4e10, 0x4e15, 0x4e2a, 0x4e31, 0x4e36, 0x4e3c, 0x4e3f, 0x4e42,
	0x4e56, 0x4e58, 0x4e82, 0x4e85, 0x8c6b, 0x4e8a, 0x8212, 0x5f0d,
	0x4e8e, 0x4e9e, 0x4e9f, 0x4ea0, 0x4ea2, 0x4eb0, 0x4eb3, 0x4eb6,
	0x4ece, 0x4ecd, 0x4ec4, 0x4ec6, 0x4ec2, 0x4ed7, 0x4ede, 0x4eed,
	0x4edf, 0x4ef7, 0x4f09, 0x4f5a, 0x4f30, 0x4f5b, 0x4f5d, 0x4f57,
	0x4f47, 0x4f76, 0x4f88, 0x4f8f, 0x4f98, 0x4f7b, 0x4f69, 0x4f70,
	0x4f91, 0x4f6f, 0x4f86, 0x4f96, 0x5118, 0x4fd4, 0x4fdf, 0x4fce,
	0x4fd8, 0x4fdb, 0x4fd1, 0x4fda, 0x4fd0, 0x4fe4, 0x4fe5, 0x501a,
	0x5028, 0x5014, 0x502a, 0x5025, 0x5005, 0x4f1c, 0x4ff6, 0x5021,
	0x5029, 0x502c, 0x4ffe, 0x4fef, 0x5011, 0x5006, 0x5043, 0x5047,
	0x6703, 0x5055, 0x5050, 0x5048, 0x505a, 0x5056, 0x506c, 0x5078,
	0x5080, 0x509a, 0x5085, 0x50b4, 0x50b2
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9940[ 189 ] = {
	0x50c9, 0x50ca, 0x50b3, 0x50c2, 0x50d6, 0x50de, 0x50e5, 0x50ed,
	0x50e3, 0x50ee, 0x50f9, 0x50f5, 0x5109, 0x5101, 0x5102, 0x5116,
	0x5115, 0x5114, 0x511a, 0x5121, 0x513a, 0x5137, 0x513c, 0x513b,
	0x513f, 0x5140, 0x5152, 0x514c, 0x5154, 0x5162, 0x7af8, 0x5169,
	0x516a, 0x516e, 0x5180, 0x5182, 0x56d8, 0x518c, 0x5189, 0x518f,
	0x5191, 0x5193, 0x5195, 0x5196, 0x51a4, 0x51a6, 0x51a2, 0x51a9,
	0x51aa, 0x51ab, 0x51b3, 0x51b1, 0x51b2, 0x51b0, 0x51b5, 0x51bd,
	0x51c5, 0x51c9, 0x51db, 0x51e0, 0x8655, 0x51e9, 0x51ed, 0xfffd,
	0x51f0, 0x51f5, 0x51fe, 0x5204, 0x520b, 0x5214, 0x520e, 0x5227,
	0x522a, 0x522e, 0x5233, 0x5239, 0x524f, 0x5244, 0x524b, 0x524c,
	0x525e, 0x5254, 0x526a, 0x5274, 0x5269, 0x5273, 0x527f, 0x527d,
	0x528d, 0x5294, 0x5292, 0x5271, 0x5288, 0x5291, 0x8fa8, 0x8fa7,
	0x52ac, 0x52ad, 0x52bc, 0x52b5, 0x52c1, 0x52cd, 0x52d7, 0x52de,
	0x52e3, 0x52e6, 0x98ed, 0x52e0, 0x52f3, 0x52f5, 0x52f8, 0x52f9,
	0x5306, 0x5308, 0x7538, 0x530d, 0x5310, 0x530f, 0x5315, 0x531a,
	0x5323, 0x532f, 0x5331, 0x5333, 0x5338, 0x5340, 0x5346, 0x5345,
	0x4e17, 0x5349, 0x534d, 0x51d6, 0x535e, 0x5369, 0x536e, 0x5918,
	0x537b, 0x5377, 0x5382, 0x5396, 0x53a0, 0x53a6, 0x53a5, 0x53ae,
	0x53b0, 0x53b6, 0x53c3, 0x7c12, 0x96d9, 0x53df, 0x66fc, 0x71ee,
	0x53ee, 0x53e8, 0x53ed, 0x53fa, 0x5401, 0x543d, 0x5440, 0x542c,
	0x542d, 0x543c, 0x542e, 0x5436, 0x5429, 0x541d, 0x544e, 0x548f,
	0x5475, 0x548e, 0x545f, 0x5471, 0x5477, 0x5470, 0x5492, 0x547b,
	0x5480, 0x5476, 0x5484, 0x5490, 0x5486, 0x54c7, 0x54a2, 0x54b8,
	0x54a5, 0x54ac, 0x54c4, 0x54c8, 0x54a8
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9a40[ 189 ] = {
	0x54ab, 0x54c2, 0x54a4, 0x54be, 0x54bc, 0x54d8, 0x54e5, 0x54e6,
	0x550f, 0x5514, 0x54fd, 0x54ee, 0x54ed, 0x54fa, 0x54e2, 0x5539,
	0x5540, 0x5563, 0x554c, 0x552e, 0x555c, 0x5545, 0x5556, 0x5557,
	0x5538, 0x5533, 0x555d, 0x5599, 0x5580, 0x54af, 0x558a, 0x559f,
	0x557b, 0x557e, 0x5598, 0x559e, 0x55ae, 0x557c, 0x5583, 0x55a9,
	0x5587, 0x55a8, 0x55da, 0x55c5, 0x55df, 0x55c4, 0x55dc, 0x55e4,
	0x55d4, 0x5614, 0x55f7, 0x5616, 0x55fe, 0x55fd, 0x561b, 0x55f9,
	0x564e, 0x5650, 0x71df, 0x5634, 0x5636, 0x5632, 0x5638, 0xfffd,
	0x566b, 0x5664, 0x562f, 0x566c, 0x566a, 0x5686, 0x5680, 0x568a,
	0x56a0, 0x5694, 0x568f, 0x56a5, 0x56ae, 0x56b6, 0x56b4, 0x56c2,
	0x56bc, 0x56c1, 0x56c3, 0x56c0, 0x56c8, 0x56ce, 0x56d1, 0x56d3,
	0x56d7, 0x56ee, 0x56f9, 0x5700, 0x56ff, 0x5704, 0x5709, 0x5708,
	0x570b, 0x570d, 0x5713, 0x5718, 0x5716, 0x55c7, 0x571c, 0x5726,
	0x5737, 0x5738, 0x574e, 0x573b, 0x5740, 0x574f, 0x5769, 0x57c0,
	0x5788, 0x5761, 0x577f, 0x5789, 0x5793, 0x57a0, 0x57b3, 0x57a4,
	0x57aa, 0x57b0, 0x57c3, 0x57c6, 0x57d4, 0x57d2, 0x57d3, 0x580a,
	0x57d6, 0x57e3, 0x580b, 0x5819, 0x581d, 0x5872, 0x5821, 0x5862,
	0x584b, 0x5870, 0x6bc0, 0x5852, 0x583d, 0x5879, 0x5885, 0x58b9,
	0x589f, 0x58ab, 0x58ba, 0x58de, 0x58bb, 0x58b8, 0x58ae, 0x58c5,
	0x58d3, 0x58d1, 0x58d7, 0x58d9, 0x58d8, 0x58e5, 0x58dc, 0x58e4,
	0x58df, 0x58ef, 0x58fa, 0x58f9, 0x58fb, 0x58fc, 0x58fd, 0x5902,
	0x590a, 0x5910, 0x591b, 0x68a6, 0x5925, 0x592c, 0x592d, 0x5932,
	0x5938, 0x593e, 0x7ad2, 0x5955, 0x5950, 0x594e, 0x595a, 0x5958,
	0x5962, 0x5960, 0x5967, 0x596c, 0x5969
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9b40[ 189 ] = {
	0x5978, 0x5981, 0x599d, 0x4f5e, 0x4fab, 0x59a3, 0x59b2, 0x59c6,
	0x59e8, 0x59dc, 0x598d, 0x59d9, 0x59da, 0x5a25, 0x5a1f, 0x5a11,
	0x5a1c, 0x5a09, 0x5a1a, 0x5a40, 0x5a6c, 0x5a49, 0x5a35, 0x5a36,
	0x5a62, 0x5a6a, 0x5a9a, 0x5abc, 0x5abe, 0x5acb, 0x5ac2, 0x5abd,
	0x5ae3, 0x5ad7, 0x5ae6, 0x5ae9, 0x5ad6, 0x5afa, 0x5afb, 0x5b0c,
	0x5b0b, 0x5b16, 0x5b32, 0x5ad0, 0x5b2a, 0x5b36, 0x5b3e, 0x5b43,
	0x5b45, 0x5b40, 0x5b51, 0x5b55, 0x5b5a, 0x5b5b, 0x5b65, 0x5b69,
	0x5b70, 0x5b73, 0x5b75, 0x5b78, 0x6588, 0x5b7a, 0x5b80, 0xfffd,
	0x5b83, 0x5ba6, 0x5bb8, 0x5bc3, 0x5bc7, 0x5bc9, 0x5bd4, 0x5bd0,
	0x5be4, 0x5be6, 0x5be2, 0x5bde, 0x5be5, 0x5beb, 0x5bf0, 0x5bf6,
	0x5bf3, 0x5c05, 0x5c07, 0x5c08, 0x5c0d, 0x5c13, 0x5c20, 0x5c22,
	0x5c28, 0x5c38, 0x5c39, 0x5c41, 0x5c46, 0x5c4e, 0x5c53, 0x5c50,
	0x5c4f, 0x5b71, 0x5c6c, 0x5c6e, 0x4e62, 0x5c76, 0x5c79, 0x5c8c,
	0x5c91, 0x5c94, 0x599b, 0x5cab, 0x5cbb, 0x5cb6, 0x5cbc, 0x5cb7,
	0x5cc5, 0x5cbe, 0x5cc7, 0x5cd9, 0x5ce9, 0x5cfd, 0x5cfa, 0x5ced,
	0x5d8c, 0x5cea, 0x5d0b, 0x5d15, 0x5d17, 0x5d5c, 0x5d1f, 0x5d1b,
	0x5d11, 0x5d14, 0x5d22, 0x5d1a, 0x5d19, 0x5d18, 0x5d4c, 0x5d52,
	0x5d4e, 0x5d4b, 0x5d6c, 0x5d73, 0x5d76, 0x5d87, 0x5d84, 0x5d82,
	0x5da2, 0x5d9d, 0x5dac, 0x5dae, 0x5dbd, 0x5d90, 0x5db7, 0x5dbc,
	0x5dc9, 0x5dcd, 0x5dd3, 0x5dd2, 0x5dd6, 0x5ddb, 0x5deb, 0x5df2,
	0x5df5, 0x5e0b, 0x5e1a, 0x5e19, 0x5e11, 0x5e1b, 0x5e36, 0x5e37,
	0x5e44, 0x5e43, 0x5e40, 0x5e4e, 0x5e57, 0x5e54, 0x5e5f, 0x5e62,
	0x5e64, 0x5e47, 0x5e75, 0x5e76, 0x5e7a, 0x9ebc, 0x5e7f, 0x5ea0,
	0x5ec1, 0x5ec2, 0x5ec8, 0x5ed0, 0x5ecf
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9c40[ 189 ] = {
	0x5ed6, 0x5ee3, 0x5edd, 0x5eda, 0x5edb, 0x5ee2, 0x5ee1, 0x5ee8,
	0x5ee9, 0x5eec, 0x5ef1, 0x5ef3, 0x5ef0, 0x5ef4, 0x5ef8, 0x5efe,
	0x5f03, 0x5f09, 0x5f5d, 0x5f5c, 0x5f0b, 0x5f11, 0x5f16, 0x5f29,
	0x5f2d, 0x5f38, 0x5f41, 0x5f48, 0x5f4c, 0x5f4e, 0x5f2f, 0x5f51,
	0x5f56, 0x5f57, 0x5f59, 0x5f61, 0x5f6d, 0x5f73, 0x5f77, 0x5f83,
	0x5f82, 0x5f7f, 0x5f8a, 0x5f88, 0x5f91, 0x5f87, 0x5f9e, 0x5f99,
	0x5f98, 0x5fa0, 0x5fa8, 0x5fad, 0x5fbc, 0x5fd6, 0x5ffb, 0x5fe4,
	0x5ff8, 0x5ff1, 0x5fdd, 0x60b3, 0x5fff, 0x6021, 0x6060, 0xfffd,
	0x6019, 0x6010, 0x6029, 0x600e, 0x6031, 0x601b, 0x6015, 0x602b,
	0x6026, 0x600f, 0x603a, 0x605a, 0x6041, 0x606a, 0x6077, 0x605f,
	0x604a, 0x6046, 0x604d, 0x6063, 0x6043, 0x6064, 0x6042, 0x606c,
	0x606b, 0x6059, 0x6081, 0x608d, 0x60e7, 0x6083, 0x609a, 0x6084,
	0x609b, 0x6096, 0x6097, 0x6092, 0x60a7, 0x608b, 0x60e1, 0x60b8,
	0x60e0, 0x60d3, 0x60b4, 0x5ff0, 0x60bd, 0x60c6, 0x60b5, 0x60d8,
	0x614d, 0x6115, 0x6106, 0x60f6, 0x60f7, 0x6100, 0x60f4, 0x60fa,
	0x6103, 0x6121, 0x60fb, 0x60f1, 0x610d, 0x610e, 0x6147, 0x613e,
	0x6128, 0x6127, 0x614a, 0x613f, 0x613c, 0x612c, 0x6134, 0x613d,
	0x6142, 0x6144, 0x6173, 0x6177, 0x6158, 0x6159, 0x615a, 0x616b,
	0x6174, 0x616f, 0x6165, 0x6171, 0x615f, 0x615d, 0x6153, 0x6175,
	0x6199, 0x6196, 0x6187, 0x61ac, 0x6194, 0x619a, 0x618a, 0x6191,
	0x61ab, 0x61ae, 0x61cc, 0x61ca, 0x61c9, 0x61f7, 0x61c8, 0x61c3,
	0x61c6, 0x61ba, 0x61cb, 0x7f79, 0x61cd, 0x61e6, 0x61e3, 0x61f6,
	0x61fa, 0x61f4, 0x61ff, 0x61fd, 0x61fc, 0x61fe, 0x6200, 0x6208,
	0x6209, 0x620d, 0x620c, 0x6214, 0x621b
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9d40[ 189 ] = {
	0x621e, 0x6221, 0x622a, 0x622e, 0x6230, 0x6232, 0x6233, 0x6241,
	0x624e, 0x625e, 0x6263, 0x625b, 0x6260, 0x6268, 0x627c, 0x6282,
	0x6289, 0x627e, 0x6292, 0x6293, 0x6296, 0x62d4, 0x6283, 0x6294,
	0x62d7, 0x62d1, 0x62bb, 0x62cf, 0x62ff, 0x62c6, 0x64d4, 0x62c8,
	0x62dc, 0x62cc, 0x62ca, 0x62c2, 0x62c7, 0x629b, 0x62c9, 0x630c,
	0x62ee, 0x62f1, 0x6327, 0x6302, 0x6308, 0x62ef, 0x62f5, 0x6350,
	0x633e, 0x634d, 0x641c, 0x634f, 0x6396, 0x638e, 0x6380, 0x63ab,
	0x6376, 0x63a3, 0x638f, 0x6389, 0x639f, 0x63b5, 0x636b, 0xfffd,
	0x6369, 0x63be, 0x63e9, 0x63c0, 0x63c6, 0x63e3, 0x63c9, 0x63d2,
	0x63f6, 0x63c4, 0x6416, 0x6434, 0x6406, 0x6413, 0x6426, 0x6436,
	0x651d, 0x6417, 0x6428, 0x640f, 0x6467, 0x646f, 0x6476, 0x644e,
	0x652a, 0x6495, 0x6493, 0x64a5, 0x64a9, 0x6488, 0x64bc, 0x64da,
	0x64d2, 0x64c5, 0x64c7, 0x64bb, 0x64d8, 0x64c2, 0x64f1, 0x64e7,
	0x8209, 0x64e0, 0x64e1, 0x62ac, 0x64e3, 0x64ef, 0x652c, 0x64f6,
	0x64f4, 0x64f2, 0x64fa, 0x6500, 0x64fd, 0x6518, 0x651c, 0x6505,
	0x6524, 0x6523, 0x652b, 0x6534, 0x6535, 0x6537, 0x6536, 0x6538,
	0x754b, 0x6548, 0x6556, 0x6555, 0x654d, 0x6558, 0x655e, 0x655d,
	0x6572, 0x6578, 0x6582, 0x6583, 0x8b8a, 0x659b, 0x659f, 0x65ab,
	0x65b7, 0x65c3, 0x65c6, 0x65c1, 0x65c4, 0x65cc, 0x65d2, 0x65db,
	0x65d9, 0x65e0, 0x65e1, 0x65f1, 0x6772, 0x660a, 0x6603, 0x65fb,
	0x6773, 0x6635, 0x6636, 0x6634, 0x661c, 0x664f, 0x6644, 0x6649,
	0x6641, 0x665e, 0x665d, 0x6664, 0x6667, 0x6668, 0x665f, 0x6662,
	0x6670, 0x6683, 0x6688, 0x668e, 0x6689, 0x6684, 0x6698, 0x669d,
	0x66c1, 0x66b9, 0x66c9, 0x66be, 0x66bc
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9e40[ 189 ] = {
	0x66c4, 0x66b8, 0x66d6, 0x66da, 0x66e0, 0x663f, 0x66e6, 0x66e9,
	0x66f0, 0x66f5, 0x66f7, 0x670f, 0x6716, 0x671e, 0x6726, 0x6727,
	0x9738, 0x672e, 0x673f, 0x6736, 0x6741, 0x6738, 0x6737, 0x6746,
	0x675e, 0x6760, 0x6759, 0x6763, 0x6764, 0x6789, 0x6770, 0x67a9,
	0x677c, 0x676a, 0x678c, 0x678b, 0x67a6, 0x67a1, 0x6785, 0x67b7,
	0x67ef, 0x67b4, 0x67ec, 0x67b3, 0x67e9, 0x67b8, 0x67e4, 0x67de,
	0x67dd, 0x67e2, 0x67ee, 0x67b9, 0x67ce, 0x67c6, 0x67e7, 0x6a9c,
	0x681e, 0x6846, 0x6829, 0x6840, 0x684d, 0x6832, 0x684e, 0xfffd,
	0x68b3, 0x682b, 0x6859, 0x6863, 0x6877, 0x687f, 0x689f, 0x688f,
	0x68ad, 0x6894, 0x689d, 0x689b, 0x6883, 0x6aae, 0x68b9, 0x6874,
	0x68b5, 0x68a0, 0x68ba, 0x690f, 0x688d, 0x687e, 0x6901, 0x68ca,
	0x6908, 0x68d8, 0x6922, 0x6926, 0x68e1, 0x690c, 0x68cd, 0x68d4,
	0x68e7, 0x68d5, 0x6936, 0x6912, 0x6904, 0x68d7, 0x68e3, 0x6925,
	0x68f9, 0x68e0, 0x68ef, 0x6928, 0x692a, 0x691a, 0x6923, 0x6921,
	0x68c6, 0x6979, 0x6977, 0x695c, 0x6978, 0x696b, 0x6954, 0x697e,
	0x696e, 0x6939, 0x6974, 0x693d, 0x6959, 0x6930, 0x6961, 0x695e,
	0x695d, 0x6981, 0x696a, 0x69b2, 0x69ae, 0x69d0, 0x69bf, 0x69c1,
	0x69d3, 0x69be, 0x69ce, 0x5be8, 0x69ca, 0x69dd, 0x69bb, 0x69c3,
	0x69a7, 0x6a2e, 0x6991, 0x69a0, 0x699c, 0x6995, 0x69b4, 0x69de,
	0x69e8, 0x6a02, 0x6a1b, 0x69ff, 0x6b0a, 0x69f9, 0x69f2, 0x69e7,
	0x6a05, 0x69b1, 0x6a1e, 0x69ed, 0x6a14, 0x69eb, 0x6a0a, 0x6a12,
	0x6ac1, 0x6a23, 0x6a13, 0x6a44, 0x6a0c, 0x6a72, 0x6a36, 0x6a78,
	0x6a47, 0x6a62, 0x6a59, 0x6a66, 0x6a48, 0x6a38, 0x6a22, 0x6a90,
	0x6a8d, 0x6aa0, 0x6a84, 0x6aa2, 0x6aa3
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9f40[ 189 ] = {
	0x6a97, 0x8617, 0x6abb, 0x6ac3, 0x6ac2, 0x6ab8, 0x6ab3, 0x6aac,
	0x6ade, 0x6ad1, 0x6adf, 0x6aaa, 0x6ada, 0x6aea, 0x6afb, 0x6b05,
	0x8616, 0x6afa, 0x6b12, 0x6b16, 0x9b31, 0x6b1f, 0x6b38, 0x6b37,
	0x76dc, 0x6b39, 0x98ee, 0x6b47, 0x6b43, 0x6b49, 0x6b50, 0x6b59,
	0x6b54, 0x6b5b, 0x6b5f, 0x6b61, 0x6b78, 0x6b79, 0x6b7f, 0x6b80,
	0x6b84, 0x6b83, 0x6b8d, 0x6b98, 0x6b95, 0x6b9e, 0x6ba4, 0x6baa,
	0x6bab, 0x6baf, 0x6bb2, 0x6bb1, 0x6bb3, 0x6bb7, 0x6bbc, 0x6bc6,
	0x6bcb, 0x6bd3, 0x6bdf, 0x6bec, 0x6beb, 0x6bf3, 0x6bef, 0xfffd,
	0x9ebe, 0x6c08, 0x6c13, 0x6c14, 0x6c1b, 0x6c24, 0x6c23, 0x6c5e,
	0x6c55, 0x6c62, 0x6c6a, 0x6c82, 0x6c8d, 0x6c9a, 0x6c81, 0x6c9b,
	0x6c7e, 0x6c68, 0x6c73, 0x6c92, 0x6c90, 0x6cc4, 0x6cf1, 0x6cd3,
	0x6cbd, 0x6cd7, 0x6cc5, 0x6cdd, 0x6cae, 0x6cb1, 0x6cbe, 0x6cba,
	0x6cdb, 0x6cef, 0x6cd9, 0x6cea, 0x6d1f, 0x884d, 0x6d36, 0x6d2b,
	0x6d3d, 0x6d38, 0x6d19, 0x6d35, 0x6d33, 0x6d12, 0x6d0c, 0x6d63,
	0x6d93, 0x6d64, 0x6d5a, 0x6d79, 0x6d59, 0x6d8e, 0x6d95, 0x6fe4,
	0x6d85, 0x6df9, 0x6e15, 0x6e0a, 0x6db5, 0x6dc7, 0x6de6, 0x6db8,
	0x6dc6, 0x6dec, 0x6dde, 0x6dcc, 0x6de8, 0x6dd2, 0x6dc5, 0x6dfa,
	0x6dd9, 0x6de4, 0x6dd5, 0x6dea, 0x6dee, 0x6e2d, 0x6e6e, 0x6e2e,
	0x6e19, 0x6e72, 0x6e5f, 0x6e3e, 0x6e23, 0x6e6b, 0x6e2b, 0x6e76,
	0x6e4d, 0x6e1f, 0x6e43, 0x6e3a, 0x6e4e, 0x6e24, 0x6eff, 0x6e1d,
	0x6e38, 0x6e82, 0x6eaa, 0x6e98, 0x6ec9, 0x6eb7, 0x6ed3, 0x6ebd,
	0x6eaf, 0x6ec4, 0x6eb2, 0x6ed4, 0x6ed5, 0x6e8f, 0x6ea5, 0x6ec2,
	0x6e9f, 0x6f41, 0x6f11, 0x704c, 0x6eec, 0x6ef8, 0x6efe, 0x6f3f,
	0x6ef2, 0x6f31, 0x6eef, 0x6f32, 0x6ecc
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe040[ 189 ] = {
	0x6f3e, 0x6f13, 0x6ef7, 0x6f86, 0x6f7a, 0x6f78, 0x6f81, 0x6f80,
	0x6f6f, 0x6f5b, 0x6ff3, 0x6f6d, 0x6f82, 0x6f7c, 0x6f58, 0x6f8e,
	0x6f91, 0x6fc2, 0x6f66, 0x6fb3, 0x6fa3, 0x6fa1, 0x6fa4, 0x6fb9,
	0x6fc6, 0x6faa, 0x6fdf, 0x6fd5, 0x6fec, 0x6fd4, 0x6fd8, 0x6ff1,
	0x6fee, 0x6fdb, 0x7009, 0x700b, 0x6ffa, 0x7011, 0x7001, 0x700f,
	0x6ffe, 0x701b, 0x701a, 0x6f74, 0x701d, 0x7018, 0x701f, 0x7030,
	0x703e, 0x7032, 0x7051, 0x7063, 0x7099, 0x7092, 0x70af, 0x70f1,
	0x70ac, 0x70b8, 0x70b3, 0x70ae, 0x70df, 0x70cb, 0x70dd, 0xfffd,
	0x70d9, 0x7109, 0x70fd, 0x711c, 0x7119, 0x7165, 0x7155, 0x7188,
	0x7166, 0x7162, 0x714c, 0x7156, 0x716c, 0x718f, 0x71fb, 0x7184,
	0x7195, 0x71a8, 0x71ac, 0x71d7, 0x71b9, 0x71be, 0x71d2, 0x71c9,
	0x71d4, 0x71ce, 0x71e0, 0x71ec, 0x71e7, 0x71f5, 0x71fc, 0x71f9,
	0x71ff, 0x720d, 0x7210, 0x721b, 0x7228, 0x722d, 0x722c, 0x7230,
	0x7232, 0x723b, 0x723c, 0x723f, 0x7240, 0x7246, 0x724b, 0x7258,
	0x7274, 0x727e, 0x7282, 0x7281, 0x7287, 0x7292, 0x7296, 0x72a2,
	0x72a7, 0x72b9, 0x72b2, 0x72c3, 0x72c6, 0x72c4, 0x72ce, 0x72d2,
	0x72e2, 0x72e0, 0x72e1, 0x72f9, 0x72f7, 0x500f, 0x7317, 0x730a,
	0x731c, 0x7316, 0x731d, 0x7334, 0x732f, 0x7329, 0x7325, 0x733e,
	0x734e, 0x734f, 0x9ed8, 0x7357, 0x736a, 0x7368, 0x7370, 0x7378,
	0x7375, 0x737b, 0x737a, 0x73c8, 0x73b3, 0x73ce, 0x73bb, 0x73c0,
	0x73e5, 0x73ee, 0x73de, 0x74a2, 0x7405, 0x746f, 0x7425, 0x73f8,
	0x7432, 0x743a, 0x7455, 0x743f, 0x745f, 0x7459, 0x7441, 0x745c,
	0x7469, 0x7470, 0x7463, 0x746a, 0x7476, 0x747e, 0x748b, 0x749e,
	0x74a7, 0x74ca, 0x74cf, 0x74d4, 0x73f1
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe140[ 189 ] = {
	0x74e0, 0x74e3, 0x74e7, 0x74e9, 0x74ee, 0x74f2, 0x74f0, 0x74f1,
	0x74f8, 0x74f7, 0x7504, 0x7503, 0x7505, 0x750c, 0x750e, 0x750d,
	0x7515, 0x7513, 0x751e, 0x7526, 0x752c, 0x753c, 0x7544, 0x754d,
	0x754a, 0x7549, 0x755b, 0x7546, 0x755a, 0x7569, 0x7564, 0x7567,
	0x756b, 0x756d, 0x7578, 0x7576, 0x7586, 0x7587, 0x7574, 0x758a,
	0x7589, 0x7582, 0x7594, 0x759a, 0x759d, 0x75a5, 0x75a3, 0x75c2,
	0x75b3, 0x75c3, 0x75b5, 0x75bd, 0x75b8, 0x75bc, 0x75b1, 0x75cd,
	0x75ca, 0x75d2, 0x75d9, 0x75e3, 0x75de, 0x75fe, 0x75ff, 0xfffd,
	0x75fc, 0x7601, 0x75f0, 0x75fa, 0x75f2, 0x75f3, 0x760b, 0x760d,
	0x7609, 0x761f, 0x7627, 0x7620, 0x7621, 0x7622, 0x7624, 0x7634,
	0x7630, 0x763b, 0x7647, 0x7648, 0x7646, 0x765c, 0x7658, 0x7661,
	0x7662, 0x7668, 0x7669, 0x766a, 0x7667, 0x766c, 0x7670, 0x7672,
	0x7676, 0x7678, 0x767c, 0x7680, 0x7683, 0x7688, 0x768b, 0x768e,
	0x7696, 0x7693, 0x7699, 0x769a, 0x76b0, 0x76b4, 0x76b8, 0x76b9,
	0x76ba, 0x76c2, 0x76cd, 0x76d6, 0x76d2, 0x76de, 0x76e1, 0x76e5,
	0x76e7, 0x76ea, 0x862f, 0x76fb, 0x7708, 0x7707, 0x7704, 0x7729,
	0x7724, 0x771e, 0x7725, 0x7726, 0x771b, 0x7737, 0x7738, 0x7747,
	0x775a, 0x7768, 0x776b, 0x775b, 0x7765, 0x777f, 0x777e, 0x7779,
	0x778e, 0x778b, 0x7791, 0x77a0, 0x779e, 0x77b0, 0x77b6, 0x77b9,
	0x77bf, 0x77bc, 0x77bd, 0x77bb, 0x77c7, 0x77cd, 0x77d7, 0x77da,
	0x77dc, 0x77e3, 0x77ee, 0x77fc, 0x780c, 0x7812, 0x7926, 0x7820,
	0x792a, 0x7845, 0x788e, 0x7874, 0x7886, 0x787c, 0x789a, 0x788c,
	0x78a3, 0x78b5, 0x78aa, 0x78af, 0x78d1, 0x78c6, 0x78cb, 0x78d4,
	0x78be, 0x78bc, 0x78c5, 0x78ca, 0x78ec
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe240[ 189 ] = {
	0x78e7, 0x78da, 0x78fd, 0x78f4, 0x7907, 0x7912, 0x7911, 0x7919,
	0x792c, 0x792b, 0x7940, 0x7960, 0x7957, 0x795f, 0x795a, 0x7955,
	0x7953, 0x797a, 0x797f, 0x798a, 0x799d, 0x79a7, 0x9f4b, 0x79aa,
	0x79ae, 0x79b3, 0x79b9, 0x79ba, 0x79c9, 0x79d5, 0x79e7, 0x79ec,
	0x79e1, 0x79e3, 0x7a08, 0x7a0d, 0x7a18, 0x7a19, 0x7a20, 0x7a1f,
	0x7980, 0x7a31, 0x7a3b, 0x7a3e, 0x7a37, 0x7a43, 0x7a57, 0x7a49,
	0x7a61, 0x7a62, 0x7a69, 0x9f9d, 0x7a70, 0x7a79, 0x7a7d, 0x7a88,
	0x7a97, 0x7a95, 0x7a98, 0x7a96, 0x7aa9, 0x7ac8, 0x7ab0, 0xfffd,
	0x7ab6, 0x7ac5, 0x7ac4, 0x7abf, 0x9083, 0x7ac7, 0x7aca, 0x7acd,
	0x7acf, 0x7ad5, 0x7ad3, 0x7ad9, 0x7ada, 0x7add, 0x7ae1, 0x7ae2,
	0x7ae6, 0x7aed, 0x7af0, 0x7b02, 0x7b0f, 0x7b0a, 0x7b06, 0x7b33,
	0x7b18, 0x7b19, 0x7b1e, 0x7b35, 0x7b28, 0x7b36, 0x7b50, 0x7b7a,
	0x7b04, 0x7b4d, 0x7b0b, 0x7b4c, 0x7b45, 0x7b75, 0x7b65, 0x7b74,
	0x7b67, 0x7b70, 0x7b71, 0x7b6c, 0x7b6e, 0x7b9d, 0x7b98, 0x7b9f,
	0x7b8d, 0x7b9c, 0x7b9a, 0x7b8b, 0x7b92, 0x7b8f, 0x7b5d, 0x7b99,
	0x7bcb, 0x7bc1, 0x7bcc, 0x7bcf, 0x7bb4, 0x7bc6, 0x7bdd, 0x7be9,
	0x7c11, 0x7c14, 0x7be6, 0x7be5, 0x7c60, 0x7c00, 0x7c07, 0x7c13,
	0x7bf3, 0x7bf7, 0x7c17, 0x7c0d, 0x7bf6, 0x7c23, 0x7c27, 0x7c2a,
	0x7c1f, 0x7c37, 0x7c2b, 0x7c3d, 0x7c4c, 0x7c43, 0x7c54, 0x7c4f,
	0x7c40, 0x7c50, 0x7c58, 0x7c5f, 0x7c64, 0x7c56, 0x7c65, 0x7c6c,
	0x7c75, 0x7c83, 0x7c90, 0x7ca4, 0x7cad, 0x7ca2, 0x7cab, 0x7ca1,
	0x7ca8, 0x7cb3, 0x7cb2, 0x7cb1, 0x7cae, 0x7cb9, 0x7cbd, 0x7cc0,
	0x7cc5, 0x7cc2, 0x7cd8, 0x7cd2, 0x7cdc, 0x7ce2, 0x9b3b, 0x7cef,
	0x7cf2, 0x7cf4, 0x7cf6, 0x7cfa, 0x7d06
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe340[ 189 ] = {
	0x7d02, 0x7d1c, 0x7d15, 0x7d0a, 0x7d45, 0x7d4b, 0x7d2e, 0x7d32,
	0x7d3f, 0x7d35, 0x7d46, 0x7d73, 0x7d56, 0x7d4e, 0x7d72, 0x7d68,
	0x7d6e, 0x7d4f, 0x7d63, 0x7d93, 0x7d89, 0x7d5b, 0x7d8f, 0x7d7d,
	0x7d9b, 0x7dba, 0x7dae, 0x7da3, 0x7db5, 0x7dc7, 0x7dbd, 0x7dab,
	0x7e3d, 0x7da2, 0x7daf, 0x7ddc, 0x7db8, 0x7d9f, 0x7db0, 0x7dd8,
	0x7ddd, 0x7de4, 0x7dde, 0x7dfb, 0x7df2, 0x7de1, 0x7e05, 0x7e0a,
	0x7e23, 0x7e21, 0x7e12, 0x7e31, 0x7e1f, 0x7e09, 0x7e0b, 0x7e22,
	0x7e46, 0x7e66, 0x7e3b, 0x7e35, 0x7e39, 0x7e43, 0x7e37, 0xfffd,
	0x7e32, 0x7e3a, 0x7e67, 0x7e5d, 0x7e56, 0x7e5e, 0x7e59, 0x7e5a,
	0x7e79, 0x7e6a, 0x7e69, 0x7e7c, 0x7e7b, 0x7e83, 0x7dd5, 0x7e7d,
	0x8fae, 0x7e7f, 0x7e88, 0x7e89, 0x7e8c, 0x7e92, 0x7e90, 0x7e93,
	0x7e94, 0x7e96, 0x7e8e, 0x7e9b, 0x7e9c, 0x7f38, 0x7f3a, 0x7f45,
	0x7f4c, 0x7f4d, 0x7f4e, 0x7f50, 0x7f51, 0x7f55, 0x7f54, 0x7f58,
	0x7f5f, 0x7f60, 0x7f68, 0x7f69, 0x7f67, 0x7f78, 0x7f82, 0x7f86,
	0x7f83, 0x7f88, 0x7f87, 0x7f8c, 0x7f94, 0x7f9e, 0x7f9d, 0x7f9a,
	0x7fa3, 0x7faf, 0x7fb2, 0x7fb9, 0x7fae, 0x7fb6, 0x7fb8, 0x8b71,
	0x7fc5, 0x7fc6, 0x7fca, 0x7fd5, 0x7fd4, 0x7fe1, 0x7fe6, 0x7fe9,
	0x7ff3, 0x7ff9, 0x98dc, 0x8006, 0x8004, 0x800b, 0x8012, 0x8018,
	0x8019, 0x801c, 0x8021, 0x8028, 0x803f, 0x803b, 0x804a, 0x8046,
	0x8052, 0x8058, 0x805a, 0x805f, 0x8062, 0x8068, 0x8073, 0x8072,
	0x8070, 0x8076, 0x8079, 0x807d, 0x807f, 0x8084, 0x8086, 0x8085,
	0x809b, 0x8093, 0x809a, 0x80ad, 0x5190, 0x80ac, 0x80db, 0x80e5,
	0x80d9, 0x80dd, 0x80c4, 0x80da, 0x80d6, 0x8109, 0x80ef, 0x80f1,
	0x811b, 0x8129, 0x8123, 0x812f, 0x814b
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe440[ 189 ] = {
	0x968b, 0x8146, 0x813e, 0x8153, 0x8151, 0x80fc, 0x8171, 0x816e,
	0x8165, 0x8166, 0x8174, 0x8183, 0x8188, 0x818a, 0x8180, 0x8182,
	0x81a0, 0x8195, 0x81a4, 0x81a3, 0x815f, 0x8193, 0x81a9, 0x81b0,
	0x81b5, 0x81be, 0x81b8, 0x81bd, 0x81c0, 0x81c2, 0x81ba, 0x81c9,
	0x81cd, 0x81d1, 0x81d9, 0x81d8, 0x81c8, 0x81da, 0x81df, 0x81e0,
	0x81e7, 0x81fa, 0x81fb, 0x81fe, 0x8201, 0x8202, 0x8205, 0x8207,
	0x820a, 0x820d, 0x8210, 0x8216, 0x8229, 0x822b, 0x8238, 0x8233,
	0x8240, 0x8259, 0x8258, 0x825d, 0x825a, 0x825f, 0x8264, 0xfffd,
	0x8262, 0x8268, 0x826a, 0x826b, 0x822e, 0x8271, 0x8277, 0x8278,
	0x827e, 0x828d, 0x8292, 0x82ab, 0x829f, 0x82bb, 0x82ac, 0x82e1,
	0x82e3, 0x82df, 0x82d2, 0x82f4, 0x82f3, 0x82fa, 0x8393, 0x8303,
	0x82fb, 0x82f9, 0x82de, 0x8306, 0x82dc, 0x8309, 0x82d9, 0x8335,
	0x8334, 0x8316, 0x8332, 0x8331, 0x8340, 0x8339, 0x8350, 0x8345,
	0x832f, 0x832b, 0x8317, 0x8318, 0x8385, 0x839a, 0x83aa, 0x839f,
	0x83a2, 0x8396, 0x8323, 0x838e, 0x8387, 0x838a, 0x837c, 0x83b5,
	0x8373, 0x8375, 0x83a0, 0x8389, 0x83a8, 0x83f4, 0x8413, 0x83eb,
	0x83ce, 0x83fd, 0x8403, 0x83d8, 0x840b, 0x83c1, 0x83f7, 0x8407,
	0x83e0, 0x83f2, 0x840d, 0x8422, 0x8420, 0x83bd, 0x8438, 0x8506,
	0x83fb, 0x846d, 0x842a, 0x843c, 0x855a, 0x8484, 0x8477, 0x846b,
	0x84ad, 0x846e, 0x8482, 0x8469, 0x8446, 0x842c, 0x846f, 0x8479,
	0x8435, 0x84ca, 0x8462, 0x84b9, 0x84bf, 0x849f, 0x84d9, 0x84cd,
	0x84bb, 0x84da, 0x84d0, 0x84c1, 0x84c6, 0x84d6, 0x84a1, 0x8521,
	0x84ff, 0x84f4, 0x8517, 0x8518, 0x852c, 0x851f, 0x8515, 0x8514,
	0x84fc, 0x8540, 0x8563, 0x8558, 0x8548
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe540[ 189 ] = {
	0x8541, 0x8602, 0x854b, 0x8555, 0x8580, 0x85a4, 0x8588, 0x8591,
	0x858a, 0x85a8, 0x856d, 0x8594, 0x859b, 0x85ea, 0x8587, 0x859c,
	0x8577, 0x857e, 0x8590, 0x85c9, 0x85ba, 0x85cf, 0x85b9, 0x85d0,
	0x85d5, 0x85dd, 0x85e5, 0x85dc, 0x85f9, 0x860a, 0x8613, 0x860b,
	0x85fe, 0x85fa, 0x8606, 0x8622, 0x861a, 0x8630, 0x863f, 0x864d,
	0x4e55, 0x8654, 0x865f, 0x8667, 0x8671, 0x8693, 0x86a3, 0x86a9,
	0x86aa, 0x868b, 0x868c, 0x86b6, 0x86af, 0x86c4, 0x86c6, 0x86b0,
	0x86c9, 0x8823, 0x86ab, 0x86d4, 0x86de, 0x86e9, 0x86ec, 0xfffd,
	0x86df, 0x86db, 0x86ef, 0x8712, 0x8706, 0x8708, 0x8700, 0x8703,
	0x86fb, 0x8711, 0x8709, 0x870d, 0x86f9, 0x870a, 0x8734, 0x873f,
	0x8737, 0x873b, 0x8725, 0x8729, 0x871a, 0x8760, 0x875f, 0x8778,
	0x874c, 0x874e, 0x8774, 0x8757, 0x8768, 0x876e, 0x8759, 0x8753,
	0x8763, 0x876a, 0x8805, 0x87a2, 0x879f, 0x8782, 0x87af, 0x87cb,
	0x87bd, 0x87c0, 0x87d0, 0x96d6, 0x87ab, 0x87c4, 0x87b3, 0x87c7,
	0x87c6, 0x87bb, 0x87ef, 0x87f2, 0x87e0, 0x880f, 0x880d, 0x87fe,
	0x87f6, 0x87f7, 0x880e, 0x87d2, 0x8811, 0x8816, 0x8815, 0x8822,
	0x8821, 0x8831, 0x8836, 0x8839, 0x8827, 0x883b, 0x8844, 0x8842,
	0x8852, 0x8859, 0x885e, 0x8862, 0x886b, 0x8881, 0x887e, 0x889e,
	0x8875, 0x887d, 0x88b5, 0x8872, 0x8882, 0x8897, 0x8892, 0x88ae,
	0x8899, 0x88a2, 0x888d, 0x88a4, 0x88b0, 0x88bf, 0x88b1, 0x88c3,
	0x88c4, 0x88d4, 0x88d8, 0x88d9, 0x88dd, 0x88f9, 0x8902, 0x88fc,
	0x88f4, 0x88e8, 0x88f2, 0x8904, 0x890c, 0x890a, 0x8913, 0x8943,
	0x891e, 0x8925, 0x892a, 0x892b, 0x8941, 0x8944, 0x893b, 0x8936,
	0x8938, 0x894c, 0x891d, 0x8960, 0x895e
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe640[ 189 ] = {
	0x8966, 0x8964, 0x896d, 0x896a, 0x896f, 0x8974, 0x8977, 0x897e,
	0x8983, 0x8988, 0x898a, 0x8993, 0x8998, 0x89a1, 0x89a9, 0x89a6,
	0x89ac, 0x89af, 0x89b2, 0x89ba, 0x89bd, 0x89bf, 0x89c0, 0x89da,
	0x89dc, 0x89dd, 0x89e7, 0x89f4, 0x89f8, 0x8a03, 0x8a16, 0x8a10,
	0x8a0c, 0x8a1b, 0x8a1d, 0x8a25, 0x8a36, 0x8a41, 0x8a5b, 0x8a52,
	0x8a46, 0x8a48, 0x8a7c, 0x8a6d, 0x8a6c, 0x8a62, 0x8a85, 0x8a82,
	0x8a84, 0x8aa8, 0x8aa1, 0x8a91, 0x8aa5, 0x8aa6, 0x8a9a, 0x8aa3,
	0x8ac4, 0x8acd, 0x8ac2, 0x8ada, 0x8aeb, 0x8af3, 0x8ae7, 0xfffd,
	0x8ae4, 0x8af1, 0x8b14, 0x8ae0, 0x8ae2, 0x8af7, 0x8ade, 0x8adb,
	0x8b0c, 0x8b07, 0x8b1a, 0x8ae1, 0x8b16, 0x8b10, 0x8b17, 0x8b20,
	0x8b33, 0x97ab, 0x8b26, 0x8b2b, 0x8b3e, 0x8b28, 0x8b41, 0x8b4c,
	0x8b4f, 0x8b4e, 0x8b49, 0x8b56, 0x8b5b, 0x8b5a, 0x8b6b, 0x8b5f,
	0x8b6c, 0x8b6f, 0x8b74, 0x8b7d, 0x8b80, 0x8b8c, 0x8b8e, 0x8b92,
	0x8b93, 0x8b96, 0x8b99, 0x8b9a, 0x8c3a, 0x8c41, 0x8c3f, 0x8c48,
	0x8c4c, 0x8c4e, 0x8c50, 0x8c55, 0x8c62, 0x8c6c, 0x8c78, 0x8c7a,
	0x8c82, 0x8c89, 0x8c85, 0x8c8a, 0x8c8d, 0x8c8e, 0x8c94, 0x8c7c,
	0x8c98, 0x621d, 0x8cad, 0x8caa, 0x8cbd, 0x8cb2, 0x8cb3, 0x8cae,
	0x8cb6, 0x8cc8, 0x8cc1, 0x8ce4, 0x8ce3, 0x8cda, 0x8cfd, 0x8cfa,
	0x8cfb, 0x8d04, 0x8d05, 0x8d0a, 0x8d07, 0x8d0f, 0x8d0d, 0x8d10,
	0x9f4e, 0x8d13, 0x8ccd, 0x8d14, 0x8d16, 0x8d67, 0x8d6d, 0x8d71,
	0x8d73, 0x8d81, 0x8d99, 0x8dc2, 0x8dbe, 0x8dba, 0x8dcf, 0x8dda,
	0x8dd6, 0x8dcc, 0x8ddb, 0x8dcb, 0x8dea, 0x8deb, 0x8ddf, 0x8de3,
	0x8dfc, 0x8e08, 0x8e09, 0x8dff, 0x8e1d, 0x8e1e, 0x8e10, 0x8e1f,
	0x8e42, 0x8e35, 0x8e30, 0x8e34, 0x8e4a
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe740[ 189 ] = {
	0x8e47, 0x8e49, 0x8e4c, 0x8e50, 0x8e48, 0x8e59, 0x8e64, 0x8e60,
	0x8e2a, 0x8e63, 0x8e55, 0x8e76, 0x8e72, 0x8e7c, 0x8e81, 0x8e87,
	0x8e85, 0x8e84, 0x8e8b, 0x8e8a, 0x8e93, 0x8e91, 0x8e94, 0x8e99,
	0x8eaa, 0x8ea1, 0x8eac, 0x8eb0, 0x8ec6, 0x8eb1, 0x8ebe, 0x8ec5,
	0x8ec8, 0x8ecb, 0x8edb, 0x8ee3, 0x8efc, 0x8efb, 0x8eeb, 0x8efe,
	0x8f0a, 0x8f05, 0x8f15, 0x8f12, 0x8f19, 0x8f13, 0x8f1c, 0x8f1f,
	0x8f1b, 0x8f0c, 0x8f26, 0x8f33, 0x8f3b, 0x8f39, 0x8f45, 0x8f42,
	0x8f3e, 0x8f4c, 0x8f49, 0x8f46, 0x8f4e, 0x8f57, 0x8f5c, 0xfffd,
	0x8f62, 0x8f63, 0x8f64, 0x8f9c, 0x8f9f, 0x8fa3, 0x8fad, 0x8faf,
	0x8fb7, 0x8fda, 0x8fe5, 0x8fe2, 0x8fea, 0x8fef, 0x9087, 0x8ff4,
	0x9005, 0x8ff9, 0x8ffa, 0x9011, 0x9015, 0x9021, 0x900d, 0x901e,
	0x9016, 0x900b, 0x9027, 0x9036, 0x9035, 0x9039, 0x8ff8, 0x904f,
	0x9050, 0x9051, 0x9052, 0x900e, 0x9049, 0x903e, 0x9056, 0x9058,
	0x905e, 0x9068, 0x906f, 0x9076, 0x96a8, 0x9072, 0x9082, 0x907d,
	0x9081, 0x9080, 0x908a, 0x9089, 0x908f, 0x90a8, 0x90af, 0x90b1,
	0x90b5, 0x90e2, 0x90e4, 0x6248, 0x90db, 0x9102, 0x9112, 0x9119,
	0x9132, 0x9130, 0x914a, 0x9156, 0x9158, 0x9163, 0x9165, 0x9169,
	0x9173, 0x9172, 0x918b, 0x9189, 0x9182, 0x91a2, 0x91ab, 0x91af,
	0x91aa, 0x91b5, 0x91b4, 0x91ba, 0x91c0, 0x91c1, 0x91c9, 0x91cb,
	0x91d0, 0x91d6, 0x91df, 0x91e1, 0x91db, 0x91fc, 0x91f5, 0x91f6,
	0x921e, 0x91ff, 0x9214, 0x922c, 0x9215, 0x9211, 0x925e, 0x9257,
	0x9245, 0x9249, 0x9264, 0x9248, 0x9295, 0x923f, 0x924b, 0x9250,
	0x929c, 0x9296, 0x9293, 0x929b, 0x925a, 0x92cf, 0x92b9, 0x92b7,
	0x92e9, 0x930f, 0x92fa, 0x9344, 0x932e
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe840[ 189 ] = {
	0x9319, 0x9322, 0x931a, 0x9323, 0x933a, 0x9335, 0x933b, 0x935c,
	0x9360, 0x937c, 0x936e, 0x9356, 0x93b0, 0x93ac, 0x93ad, 0x9394,
	0x93b9, 0x93d6, 0x93d7, 0x93e8, 0x93e5, 0x93d8, 0x93c3, 0x93dd,
	0x93d0, 0x93c8, 0x93e4, 0x941a, 0x9414, 0x9413, 0x9403, 0x9407,
	0x9410, 0x9436, 0x942b, 0x9435, 0x9421, 0x943a, 0x9441, 0x9452,
	0x9444, 0x945b, 0x9460, 0x9462, 0x945e, 0x946a, 0x9229, 0x9470,
	0x9475, 0x9477, 0x947d, 0x945a, 0x947c, 0x947e, 0x9481, 0x947f,
	0x9582, 0x9587, 0x958a, 0x9594, 0x9596, 0x9598, 0x9599, 0xfffd,
	0x95a0, 0x95a8, 0x95a7, 0x95ad, 0x95bc, 0x95bb, 0x95b9, 0x95be,
	0x95ca, 0x6ff6, 0x95c3, 0x95cd, 0x95cc, 0x95d5, 0x95d4, 0x95d6,
	0x95dc, 0x95e1, 0x95e5, 0x95e2, 0x9621, 0x9628, 0x962e, 0x962f,
	0x9642, 0x964c, 0x964f, 0x964b, 0x9677, 0x965c, 0x965e, 0x965d,
	0x965f, 0x9666, 0x9672, 0x966c, 0x968d, 0x9698, 0x9695, 0x9697,
	0x96aa, 0x96a7, 0x96b1, 0x96b2, 0x96b0, 0x96b4, 0x96b6, 0x96b8,
	0x96b9, 0x96ce, 0x96cb, 0x96c9, 0x96cd, 0x894d, 0x96dc, 0x970d,
	0x96d5, 0x96f9, 0x9704, 0x9706, 0x9708, 0x9713, 0x970e, 0x9711,
	0x970f, 0x9716, 0x9719, 0x9724, 0x972a, 0x9730, 0x9739, 0x973d,
	0x973e, 0x9744, 0x9746, 0x9748, 0x9742, 0x9749, 0x975c, 0x9760,
	0x9764, 0x9766, 0x9768, 0x52d2, 0x976b, 0x9771, 0x9779, 0x9785,
	0x977c, 0x9781, 0x977a, 0x9786, 0x978b, 0x978f, 0x9790, 0x979c,
	0x97a8, 0x97a6, 0x97a3, 0x97b3, 0x97b4, 0x97c3, 0x97c6, 0x97c8,
	0x97cb, 0x97dc, 0x97ed, 0x9f4f, 0x97f2, 0x7adf, 0x97f6, 0x97f5,
	0x980f, 0x980c, 0x9838, 0x9824, 0x9821, 0x9837, 0x983d, 0x9846,
	0x984f, 0x984b, 0x986b, 0x986f, 0x9870
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe940[ 189 ] = {
	0x9871, 0x9874, 0x9873, 0x98aa, 0x98af, 0x98b1, 0x98b6, 0x98c4,
	0x98c3, 0x98c6, 0x98e9, 0x98eb, 0x9903, 0x9909, 0x9912, 0x9914,
	0x9918, 0x9921, 0x991d, 0x991e, 0x9924, 0x9920, 0x992c, 0x992e,
	0x993d, 0x993e, 0x9942, 0x9949, 0x9945, 0x9950, 0x994b, 0x9951,
	0x9952, 0x994c, 0x9955, 0x9997, 0x9998, 0x99a5, 0x99ad, 0x99ae,
	0x99bc, 0x99df, 0x99db, 0x99dd, 0x99d8, 0x99d1, 0x99ed, 0x99ee,
	0x99f1, 0x99f2, 0x99fb, 0x99f8, 0x9a01, 0x9a0f, 0x9a05, 0x99e2,
	0x9a19, 0x9a2b, 0x9a37, 0x9a45, 0x9a42, 0x9a40, 0x9a43, 0xfffd,
	0x9a3e, 0x9a55, 0x9a4d, 0x9a5b, 0x9a57, 0x9a5f, 0x9a62, 0x9a65,
	0x9a64, 0x9a69, 0x9a6b, 0x9a6a, 0x9aad, 0x9ab0, 0x9abc, 0x9ac0,
	0x9acf, 0x9ad1, 0x9ad3, 0x9ad4, 0x9ade, 0x9adf, 0x9ae2, 0x9ae3,
	0x9ae6, 0x9aef, 0x9aeb, 0x9aee, 0x9af4, 0x9af1, 0x9af7, 0x9afb,
	0x9b06, 0x9b18, 0x9b1a, 0x9b1f, 0x9b22, 0x9b23, 0x9b25, 0x9b27,
	0x9b28, 0x9b29, 0x9b2a, 0x9b2e, 0x9b2f, 0x9b32, 0x9b44, 0x9b43,
	0x9b4f, 0x9b4d, 0x9b4e, 0x9b51, 0x9b58, 0x9b74, 0x9b93, 0x9b83,
	0x9b91, 0x9b96, 0x9b97, 0x9b9f, 0x9ba0, 0x9ba8, 0x9bb4, 0x9bc0,
	0x9bca, 0x9bb9, 0x9bc6, 0x9bcf, 0x9bd1, 0x9bd2, 0x9be3, 0x9be2,
	0x9be4, 0x9bd4, 0x9be1, 0x9c3a, 0x9bf2, 0x9bf1, 0x9bf0, 0x9c15,
	0x9c14, 0x9c09, 0x9c13, 0x9c0c, 0x9c06, 0x9c08, 0x9c12, 0x9c0a,
	0x9c04, 0x9c2e, 0x9c1b, 0x9c25, 0x9c24, 0x9c21, 0x9c30, 0x9c47,
	0x9c32, 0x9c46, 0x9c3e, 0x9c5a, 0x9c60, 0x9c67, 0x9c76, 0x9c78,
	0x9ce7, 0x9cec, 0x9cf0, 0x9d09, 0x9d08, 0x9ceb, 0x9d03, 0x9d06,
	0x9d2a, 0x9d26, 0x9daf, 0x9d23, 0x9d1f, 0x9d44, 0x9d15, 0x9d12,
	0x9d41, 0x9d3f, 0x9d3e, 0x9d46, 0x9d48
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xea40[ 101 ] = {
	0x9d5d, 0x9d5e, 0x9d64, 0x9d51, 0x9d50, 0x9d59, 0x9d72, 0x9d89,
	0x9d87, 0x9dab, 0x9d6f, 0x9d7a, 0x9d9a, 0x9da4, 0x9da9, 0x9db2,
	0x9dc4, 0x9dc1, 0x9dbb, 0x9db8, 0x9dba, 0x9dc6, 0x9dcf, 0x9dc2,
	0x9dd9, 0x9dd3, 0x9df8, 0x9de6, 0x9ded, 0x9def, 0x9dfd, 0x9e1a,
	0x9e1b, 0x9e1e, 0x9e75, 0x9e79, 0x9e7d, 0x9e81, 0x9e88, 0x9e8b,
	0x9e8c, 0x9e92, 0x9e95, 0x9e91, 0x9e9d, 0x9ea5, 0x9ea9, 0x9eb8,
	0x9eaa, 0x9ead, 0x9761, 0x9ecc, 0x9ece, 0x9ecf, 0x9ed0, 0x9ed4,
	0x9edc, 0x9ede, 0x9edd, 0x9ee0, 0x9ee5, 0x9ee8, 0x9eef, 0xfffd,
	0x9ef4, 0x9ef6, 0x9ef7, 0x9ef9, 0x9efb, 0x9efc, 0x9efd, 0x9f07,
	0x9f08, 0x76b7, 0x9f15, 0x9f21, 0x9f2c, 0x9f3e, 0x9f4a, 0x9f52,
	0x9f54, 0x9f63, 0x9f5f, 0x9f60, 0x9f61, 0x9f66, 0x9f67, 0x9f6c,
	0x9f6a, 0x9f77, 0x9f72, 0x9f76, 0x9f95, 0x9f9c, 0x9fa0, 0x582f,
	0x69c7, 0x9059, 0x7464, 0x51dc, 0x7199
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xed40[ 189 ] = {
	0x7e8a, 0x891c, 0x9348, 0x9288, 0x84dc, 0x4fc9, 0x70bb, 0x6631,
	0x68c8, 0x92f9, 0x66fb, 0x5f45, 0x4e28, 0x4ee1, 0x4efc, 0x4f00,
	0x4f03, 0x4f39, 0x4f56, 0x4f92, 0x4f8a, 0x4f9a, 0x4f94, 0x4fcd,
	0x5040, 0x5022, 0x4fff, 0x501e, 0x5046, 0x5070, 0x5042, 0x5094,
	0x50f4, 0x50d8, 0x514a, 0x5164, 0x519d, 0x51be, 0x51ec, 0x5215,
	0x529c, 0x52a6, 0x52c0, 0x52db, 0x5300, 0x5307, 0x5324, 0x5372,
	0x5393, 0x53b2, 0x53dd, 0xfa0e, 0x549c, 0x548a, 0x54a9, 0x54ff,
	0x5586, 0x5759, 0x5765, 0x57ac, 0x57c8, 0x57c7, 0xfa0f, 0xfffd,
	0xfa10, 0x589e, 0x58b2, 0x590b, 0x5953, 0x595b, 0x595d, 0x5963,
	0x59a4, 0x59ba, 0x5b56, 0x5bc0, 0x752f, 0x5bd8, 0x5bec, 0x5c1e,
	0x5ca6, 0x5cba, 0x5cf5, 0x5d27, 0x5d53, 0xfa11, 0x5d42, 0x5d6d,
	0x5db8, 0x5db9, 0x5dd0, 0x5f21, 0x5f34, 0x5f67, 0x5fb7, 0x5fde,
	0x605d, 0x6085, 0x608a, 0x60de, 0x60d5, 0x6120, 0x60f2, 0x6111,
	0x6137, 0x6130, 0x6198, 0x6213, 0x62a6, 0x63f5, 0x6460, 0x649d,
	0x64ce, 0x654e, 0x6600, 0x6615, 0x663b, 0x6609, 0x662e, 0x661e,
	0x6624, 0x6665, 0x6657, 0x6659, 0xfa12, 0x6673, 0x6699, 0x66a0,
	0x66b2, 0x66bf, 0x66fa, 0x670e, 0xf929, 0x6766, 0x67bb, 0x6852,
	0x67c0, 0x6801, 0x6844, 0x68cf, 0xfa13, 0x6968, 0xfa14, 0x6998,
	0x69e2, 0x6a30, 0x6a6b, 0x6a46, 0x6a73, 0x6a7e, 0x6ae2, 0x6ae4,
	0x6bd6, 0x6c3f, 0x6c5c, 0x6c86, 0x6c6f, 0x6cda, 0x6d04, 0x6d87,
	0x6d6f, 0x6d96, 0x6dac, 0x6dcf, 0x6df8, 0x6df2, 0x6dfc, 0x6e39,
	0x6e5c, 0x6e27, 0x6e3c, 0x6ebf, 0x6f88, 0x6fb5, 0x6ff5, 0x7005,
	0x7007, 0x7028, 0x7085, 0x70ab, 0x710f, 0x7104, 0x715c, 0x7146,
	0x7147, 0xfa15, 0x71c1, 0x71fe, 0x72b1
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xee40[ 189 ] = {
	0x72be, 0x7324, 0xfa16, 0x7377, 0x73bd, 0x73c9, 0x73d6, 0x73e3,
	0x73d2, 0x7407, 0x73f5, 0x7426, 0x742a, 0x7429, 0x742e, 0x7462,
	0x7489, 0x749f, 0x7501, 0x756f, 0x7682, 0x769c, 0x769e, 0x769b,
	0x76a6, 0xfa17, 0x7746, 0x52af, 0x7821, 0x784e, 0x7864, 0x787a,
	0x7930, 0xfa18, 0xfa19, 0xfa1a, 0x7994, 0xfa1b, 0x799b, 0x7ad1,
	0x7ae7, 0xfa1c, 0x7aeb, 0x7b9e, 0xfa1d, 0x7d48, 0x7d5c, 0x7db7,
	0x7da0, 0x7dd6, 0x7e52, 0x7f47, 0x7fa1, 0xfa1e, 0x8301, 0x8362,
	0x837f, 0x83c7, 0x83f6, 0x8448, 0x84b4, 0x8553, 0x8559, 0xfffd,
	0x856b, 0xfa1f, 0x85b0, 0xfa20, 0xfa21, 0x8807, 0x88f5, 0x8a12,
	0x8a37, 0x8a79, 0x8aa7, 0x8abe, 0x8adf, 0xfa22, 0x8af6, 0x8b53,
	0x8b7f, 0x8cf0, 0x8cf4, 0x8d12, 0x8d76, 0xfa23, 0x8ecf, 0xfa24,
	0xfa25, 0x9067, 0x90de, 0xfa26, 0x9115, 0x9127, 0x91da, 0x91d7,
	0x91de, 0x91ed, 0x91ee, 0x91e4, 0x91e5, 0x9206, 0x9210, 0x920a,
	0x923a, 0x9240, 0x923c, 0x924e, 0x9259, 0x9251, 0x9239, 0x9267,
	0x92a7, 0x9277, 0x9278, 0x92e7, 0x92d7, 0x92d9, 0x92d0, 0xfa27,
	0x92d5, 0x92e0, 0x92d3, 0x9325, 0x9321, 0x92fb, 0xfa28, 0x931e,
	0x92ff, 0x931d, 0x9302, 0x9370, 0x9357, 0x93a4, 0x93c6, 0x93de,
	0x93f8, 0x9431, 0x9445, 0x9448, 0x9592, 0xf9dc, 0xfa29, 0x969d,
	0x96af, 0x9733, 0x973b, 0x9743, 0x974d, 0x974f, 0x9751, 0x9755,
	0x9857, 0x9865, 0xfa2a, 0xfa2b, 0x9927, 0xfa2c, 0x999e, 0x9a4e,
	0x9ad9, 0x9adc, 0x9b75, 0x9b72, 0x9b8f, 0x9bb1, 0x9bbb, 0x9c00,
	0x9d70, 0x9d6b, 0xfa2d, 0x9e19, 0x9ed1, 0xfffd, 0xfffd, 0x2170,
	0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177, 0x2178,
	0x2179, 0xffe2, 0xffe4, 0xff07, 0xff02
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfa40[ 189 ] = {
	0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177,
	0x2178, 0x2179, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165,
	0x2166, 0x2167, 0x2168, 0x2169, 0xffe2, 0xffe4, 0xff07, 0xff02,
	0x3231, 0x2116, 0x2121, 0x2235, 0x7e8a, 0x891c, 0x9348, 0x9288,
	0x84dc, 0x4fc9, 0x70bb, 0x6631, 0x68c8, 0x92f9, 0x66fb, 0x5f45,
	0x4e28, 0x4ee1, 0x4efc, 0x4f00, 0x4f03, 0x4f39, 0x4f56, 0x4f92,
	0x4f8a, 0x4f9a, 0x4f94, 0x4fcd, 0x5040, 0x5022, 0x4fff, 0x501e,
	0x5046, 0x5070, 0x5042, 0x5094, 0x50f4, 0x50d8, 0x514a, 0xfffd,
	0x5164, 0x519d, 0x51be, 0x51ec, 0x5215, 0x529c, 0x52a6, 0x52c0,
	0x52db, 0x5300, 0x5307, 0x5324, 0x5372, 0x5393, 0x53b2, 0x53dd,
	0xfa0e, 0x549c, 0x548a, 0x54a9, 0x54ff, 0x5586, 0x5759, 0x5765,
	0x57ac, 0x57c8, 0x57c7, 0xfa0f, 0xfa10, 0x589e, 0x58b2, 0x590b,
	0x5953, 0x595b, 0x595d, 0x5963, 0x59a4, 0x59ba, 0x5b56, 0x5bc0,
	0x752f, 0x5bd8, 0x5bec, 0x5c1e, 0x5ca6, 0x5cba, 0x5cf5, 0x5d27,
	0x5d53, 0xfa11, 0x5d42, 0x5d6d, 0x5db8, 0x5db9, 0x5dd0, 0x5f21,
	0x5f34, 0x5f67, 0x5fb7, 0x5fde, 0x605d, 0x6085, 0x608a, 0x60de,
	0x60d5, 0x6120, 0x60f2, 0x6111, 0x6137, 0x6130, 0x6198, 0x6213,
	0x62a6, 0x63f5, 0x6460, 0x649d, 0x64ce, 0x654e, 0x6600, 0x6615,
	0x663b, 0x6609, 0x662e, 0x661e, 0x6624, 0x6665, 0x6657, 0x6659,
	0xfa12, 0x6673, 0x6699, 0x66a0, 0x66b2, 0x66bf, 0x66fa, 0x670e,
	0xf929, 0x6766, 0x67bb, 0x6852, 0x67c0, 0x6801, 0x6844, 0x68cf,
	0xfa13, 0x6968, 0xfa14, 0x6998, 0x69e2, 0x6a30, 0x6a6b, 0x6a46,
	0x6a73, 0x6a7e, 0x6ae2, 0x6ae4, 0x6bd6, 0x6c3f, 0x6c5c, 0x6c86,
	0x6c6f, 0x6cda, 0x6d04, 0x6d87, 0x6d6f
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfb40[ 189 ] = {
	0x6d96, 0x6dac, 0x6dcf, 0x6df8, 0x6df2, 0x6dfc, 0x6e39, 0x6e5c,
	0x6e27, 0x6e3c, 0x6ebf, 0x6f88, 0x6fb5, 0x6ff5, 0x7005, 0x7007,
	0x7028, 0x7085, 0x70ab, 0x710f, 0x7104, 0x715c, 0x7146, 0x7147,
	0xfa15, 0x71c1, 0x71fe, 0x72b1, 0x72be, 0x7324, 0xfa16, 0x7377,
	0x73bd, 0x73c9, 0x73d6, 0x73e3, 0x73d2, 0x7407, 0x73f5, 0x7426,
	0x742a, 0x7429, 0x742e, 0x7462, 0x7489, 0x749f, 0x7501, 0x756f,
	0x7682, 0x769c, 0x769e, 0x769b, 0x76a6, 0xfa17, 0x7746, 0x52af,
	0x7821, 0x784e, 0x7864, 0x787a, 0x7930, 0xfa18, 0xfa19, 0xfffd,
	0xfa1a, 0x7994, 0xfa1b, 0x799b, 0x7ad1, 0x7ae7, 0xfa1c, 0x7aeb,
	0x7b9e, 0xfa1d, 0x7d48, 0x7d5c, 0x7db7, 0x7da0, 0x7dd6, 0x7e52,
	0x7f47, 0x7fa1, 0xfa1e, 0x8301, 0x8362, 0x837f, 0x83c7, 0x83f6,
	0x8448, 0x84b4, 0x8553, 0x8559, 0x856b, 0xfa1f, 0x85b0, 0xfa20,
	0xfa21, 0x8807, 0x88f5, 0x8a12, 0x8a37, 0x8a79, 0x8aa7, 0x8abe,
	0x8adf, 0xfa22, 0x8af6, 0x8b53, 0x8b7f, 0x8cf0, 0x8cf4, 0x8d12,
	0x8d76, 0xfa23, 0x8ecf, 0xfa24, 0xfa25, 0x9067, 0x90de, 0xfa26,
	0x9115, 0x9127, 0x91da, 0x91d7, 0x91de, 0x91ed, 0x91ee, 0x91e4,
	0x91e5, 0x9206, 0x9210, 0x920a, 0x923a, 0x9240, 0x923c, 0x924e,
	0x9259, 0x9251, 0x9239, 0x9267, 0x92a7, 0x9277, 0x9278, 0x92e7,
	0x92d7, 0x92d9, 0x92d0, 0xfa27, 0x92d5, 0x92e0, 0x92d3, 0x9325,
	0x9321, 0x92fb, 0xfa28, 0x931e, 0x92ff, 0x931d, 0x9302, 0x9370,
	0x9357, 0x93a4, 0x93c6, 0x93de, 0x93f8, 0x9431, 0x9445, 0x9448,
	0x9592, 0xf9dc, 0xfa29, 0x969d, 0x96af, 0x9733, 0x973b, 0x9743,
	0x974d, 0x974f, 0x9751, 0x9755, 0x9857, 0x9865, 0xfa2a, 0xfa2b,
	0x9927, 0xfa2c, 0x999e, 0x9a4e, 0x9ad9
};

const uint16_t libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfc40[ 12 ] = {
	0x9adc, 0x9b75, 0x9b72, 0x9b8f, 0x9bb1, 0x9bbb, 0x9c00, 0x9d70,
	0x9d6b, 0xfa2d, 0x9e19, 0x9ed1
};

/* Unicode to ASCII character lookup table for Windows 932 codepage
 * Unknown are filled with the ASCII replacement character 0x1a
 */

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0080[ 128 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9881,
	0x4e81, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8b81, 0x7d81, 0x001a, 0x001a, 0x4c81, 0x001a, 0xf781, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7e81,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8081,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0380[ 256 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9f83, 0xa083, 0xa183, 0xa283, 0xa383, 0xa483, 0xa583,
	0xa683, 0xa783, 0xa883, 0xa983, 0xaa83, 0xab83, 0xac83, 0xad83,
	0xae83, 0xaf83, 0x001a, 0xb083, 0xb183, 0xb283, 0xb383, 0xb483,
	0xb583, 0xb683, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xbf83, 0xc083, 0xc183, 0xc283, 0xc383, 0xc483, 0xc583,
	0xc683, 0xc783, 0xc883, 0xc983, 0xca83, 0xcb83, 0xcc83, 0xcd83,
	0xce83, 0xcf83, 0x001a, 0xd083, 0xd183, 0xd283, 0xd383, 0xd483,
	0xd583, 0xd683, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4684, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4084, 0x4184, 0x4284, 0x4384, 0x4484, 0x4584, 0x4784, 0x4884,
	0x4984, 0x4a84, 0x4b84, 0x4c84, 0x4d84, 0x4e84, 0x4f84, 0x5084,
	0x5184, 0x5284, 0x5384, 0x5484, 0x5584, 0x5684, 0x5784, 0x5884,
	0x5984, 0x5a84, 0x5b84, 0x5c84, 0x5d84, 0x5e84, 0x5f84, 0x6084,
	0x7084, 0x7184, 0x7284, 0x7384, 0x7484, 0x7584, 0x7784, 0x7884,
	0x7984, 0x7a84, 0x7b84, 0x7c84, 0x7d84, 0x7e84, 0x8084, 0x8184,
	0x8284, 0x8384, 0x8484, 0x8584, 0x8684, 0x8784, 0x8884, 0x8984,
	0x8a84, 0x8b84, 0x8c84, 0x8d84, 0x8e84, 0x8f84, 0x9084, 0x9184,
	0x001a, 0x7684, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2000[ 832 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5d81, 0x001a, 0x001a, 0x001a, 0x001a, 0x5c81, 0x001a, 0x001a,
	0x6581, 0x6681, 0x001a, 0x001a, 0x6781, 0x6881, 0x001a, 0x001a,
	0xf581, 0xf681, 0x001a, 0x001a, 0x001a, 0x6481, 0x6381, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf181, 0x001a, 0x8c81, 0x8d81, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa681, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x8e81, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8287, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8487, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf081, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5487, 0x5587, 0x5687, 0x5787, 0x5887, 0x5987, 0x5a87, 0x5b87,
	0x5c87, 0x5d87, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xefee, 0xf0ee, 0xf1ee, 0xf2ee, 0xf3ee, 0xf4ee, 0xf5ee, 0xf6ee,
	0xf7ee, 0xf8ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa981, 0xaa81, 0xa881, 0xab81, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcb81, 0x001a, 0xcc81, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcd81, 0x001a, 0xdd81, 0xce81, 0x001a, 0x001a, 0x001a, 0xde81,
	0xb881, 0x001a, 0x001a, 0xb981, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9487, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe381, 0x001a, 0x001a, 0xe581, 0x8781, 0x9887,
	0xda81, 0x001a, 0x001a, 0x001a, 0x001a, 0x6181, 0x001a, 0xc881,
	0xc981, 0xbf81, 0xbe81, 0xe781, 0xe881, 0x001a, 0x9387, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8881, 0xe681, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe481, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe081, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8281, 0xdf81, 0x001a, 0x001a, 0x001a, 0x001a, 0x8581, 0x8681,
	0x001a, 0x001a, 0xe181, 0xe281, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xbc81, 0xbd81, 0x001a, 0x001a, 0xba81, 0xbb81,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdb81, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9987,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xdc81, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2440[ 576 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4087, 0x4187, 0x4287, 0x4387, 0x4487, 0x4587, 0x4687, 0x4787,
	0x4887, 0x4987, 0x4a87, 0x4b87, 0x4c87, 0x4d87, 0x4e87, 0x4f87,
	0x5087, 0x5187, 0x5287, 0x5387, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9f84, 0xaa84, 0xa084, 0xab84, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa184, 0x001a, 0x001a, 0xac84,
	0xa284, 0x001a, 0x001a, 0xad84, 0xa484, 0x001a, 0x001a, 0xaf84,
	0xa384, 0x001a, 0x001a, 0xae84, 0xa584, 0xba84, 0x001a, 0x001a,
	0xb584, 0x001a, 0x001a, 0xb084, 0xa784, 0xbc84, 0x001a, 0x001a,
	0xb784, 0x001a, 0x001a, 0xb284, 0xa684, 0x001a, 0x001a, 0xb684,
	0xbb84, 0x001a, 0x001a, 0xb184, 0xa884, 0x001a, 0x001a, 0xb884,
	0xbd84, 0x001a, 0x001a, 0xb384, 0xa984, 0x001a, 0x001a, 0xb984,
	0x001a, 0x001a, 0xbe84, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb484, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa181, 0xa081, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa381, 0xa281, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa581, 0xa481, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9f81, 0x9e81,
	0x001a, 0x001a, 0x001a, 0x9b81, 0x001a, 0x001a, 0x9d81, 0x9c81,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xfc81,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9a81, 0x9981, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8a81, 0x001a, 0x8981, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf481, 0x001a, 0x001a, 0xf381, 0x001a, 0xf281,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3000[ 256 ] = {
	0x4081, 0x4181, 0x4281, 0x5681, 0x001a, 0x5881, 0x5981, 0x5a81,
	0x7181, 0x7281, 0x7381, 0x7481, 0x7581, 0x7681, 0x7781, 0x7881,
	0x7981, 0x7a81, 0xa781, 0xac81, 0x6b81, 0x6c81, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8087, 0x001a, 0x8187,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9f82, 0xa082, 0xa182, 0xa282, 0xa382, 0xa482, 0xa582,
	0xa682, 0xa782, 0xa882, 0xa982, 0xaa82, 0xab82, 0xac82, 0xad82,
	0xae82, 0xaf82, 0xb082, 0xb182, 0xb282, 0xb382, 0xb482, 0xb582,
	0xb682, 0xb782, 0xb882, 0xb982, 0xba82, 0xbb82, 0xbc82, 0xbd82,
	0xbe82, 0xbf82, 0xc082, 0xc182, 0xc282, 0xc382, 0xc482, 0xc582,
	0xc682, 0xc782, 0xc882, 0xc982, 0xca82, 0xcb82, 0xcc82, 0xcd82,
	0xce82, 0xcf82, 0xd082, 0xd182, 0xd282, 0xd382, 0xd482, 0xd582,
	0xd682, 0xd782, 0xd882, 0xd982, 0xda82, 0xdb82, 0xdc82, 0xdd82,
	0xde82, 0xdf82, 0xe082, 0xe182, 0xe282, 0xe382, 0xe482, 0xe582,
	0xe682, 0xe782, 0xe882, 0xe982, 0xea82, 0xeb82, 0xec82, 0xed82,
	0xee82, 0xef82, 0xf082, 0xf182, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4a81, 0x4b81, 0x5481, 0x5581, 0x001a,
	0x001a, 0x4083, 0x4183, 0x4283, 0x4383, 0x4483, 0x4583, 0x4683,
	0x4783, 0x4883, 0x4983, 0x4a83, 0x4b83, 0x4c83, 0x4d83, 0x4e83,
	0x4f83, 0x5083, 0x5183, 0x5283, 0x5383, 0x5483, 0x5583, 0x5683,
	0x5783, 0x5883, 0x5983, 0x5a83, 0x5b83, 0x5c83, 0x5d83, 0x5e83,
	0x5f83, 0x6083, 0x6183, 0x6283, 0x6383, 0x6483, 0x6583, 0x6683,
	0x6783, 0x6883, 0x6983, 0x6a83, 0x6b83, 0x6c83, 0x6d83, 0x6e83,
	0x6f83, 0x7083, 0x7183, 0x7283, 0x7383, 0x7483, 0x7583, 0x7683,
	0x7783, 0x7883, 0x7983, 0x7a83, 0x7b83, 0x7c83, 0x7d83, 0x7e83,
	0x8083, 0x8183, 0x8283, 0x8383, 0x8483, 0x8583, 0x8683, 0x8783,
	0x8883, 0x8983, 0x8a83, 0x8b83, 0x8c83, 0x8d83, 0x8e83, 0x8f83,
	0x9083, 0x9183, 0x9283, 0x9383, 0x9483, 0x9583, 0x9683, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4581, 0x5b81, 0x5281, 0x5381, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3200[ 512 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8a87, 0x8b87, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8c87, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8587, 0x8687, 0x8787, 0x8887,
	0x8987, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6587, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6987, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6087, 0x001a, 0x001a, 0x001a,
	0x6387, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6187, 0x6b87, 0x001a, 0x001a, 0x6a87, 0x6487,
	0x001a, 0x001a, 0x001a, 0x6c87, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6687, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6e87, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5f87, 0x6d87, 0x001a, 0x001a, 0x6287, 0x001a, 0x001a,
	0x001a, 0x6787, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6887,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7e87, 0x8f87, 0x8e87, 0x8d87, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7287, 0x7387,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6f87, 0x7087, 0x7187, 0x001a,
	0x001a, 0x7587, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7487, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8387, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0x4e00[ 22288 ] = {
	0xea88, 0x9a92, 0x001a, 0xb58e, 0x001a, 0x001a, 0x001a, 0x9c96,
	0xe48f, 0x4f8e, 0xe38f, 0xba89, 0x001a, 0x7395, 0x5e97, 0x001a,
	0xa098, 0x4e89, 0x001a, 0x001a, 0x8e8a, 0xa198, 0xa290, 0xc099,
	0x758b, 0xb895, 0x001a, 0x001a, 0x001a, 0x001a, 0xe58f, 0x001a,
	0x001a, 0xbc97, 0x001a, 0x001a, 0x001a, 0x001a, 0xc095, 0x001a,
	0x4ced, 0x001a, 0xa298, 0x001a, 0x001a, 0x8692, 0x001a, 0x001a,
	0x001a, 0xa398, 0xf88b, 0x001a, 0x001a, 0x001a, 0xa498, 0x001a,
	0xdb8a, 0x4f92, 0x001a, 0xe58e, 0xa598, 0x001a, 0x001a, 0xa698,
	0x001a, 0x001a, 0xa798, 0x5494, 0x001a, 0x768b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5694, 0x001a, 0xe193, 0xc18c, 0x5296,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x68e5, 0xa898, 0xe68f,
	0xa998, 0xb389, 0x001a, 0x001a, 0x001a, 0xe38b, 0xee8c, 0xe796,
	0x001a, 0x001a, 0xa49b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9097, 0x001a, 0xfb93, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa38a, 0x001a,
	0x548b, 0x001a, 0xaa98, 0x001a, 0x001a, 0xab98, 0xb997, 0x001a,
	0x5c97, 0x8891, 0xad98, 0x968e, 0xf193, 0x001a, 0xb098, 0x001a,
	0x001a, 0x5d89, 0xdd8c, 0x001a, 0xdc8c, 0xe488, 0x001a, 0x001a,
	0x6a98, 0x6998, 0x001a, 0xb18d, 0x9f88, 0x001a, 0xb198, 0xb298,
	0xb398, 0x5396, 0xb498, 0x001a, 0xf08c, 0xe588, 0x9296, 0x001a,
	0x9c8b, 0x001a, 0x001a, 0x9d8b, 0x9e8b, 0xe092, 0xba97, 0x001a,
	0xb598, 0x001a, 0x001a, 0xb698, 0x001a, 0x001a, 0xb798, 0x001a,
	0x001a, 0x001a, 0x6c90, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x598f, 0x6d90, 0xbc98, 0x001a, 0xba98, 0x001a, 0xbb98, 0x778b,
	0x001a, 0x001a, 0xa18d, 0xee89, 0x001a, 0xb998, 0xb898, 0xa795,
	0x001a, 0x001a, 0x001a, 0x001a, 0x658e, 0x648e, 0xbc91, 0xbd98,
	0x7495, 0xe590, 0x001a, 0x001a, 0x001a, 0x5781, 0xbe98, 0xc098,
	0x001a, 0x4ded, 0x001a, 0xe391, 0xdf97, 0xc888, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbf98, 0xbc89, 0x001a,
	0xc28b, 0x001a, 0x8792, 0x001a, 0x001a, 0x001a, 0x8f8c, 0xc198,
	0x001a, 0x001a, 0x001a, 0x4394, 0x4eed, 0x001a, 0x001a, 0x001a,
	0x4fed, 0xe98a, 0x001a, 0x50ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc298, 0xc988, 0x001a, 0x001a, 0xde8c, 0xea8a, 0x9a95,
	0xb094, 0x788b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xef89, 0x001a, 0xe598, 0x6093, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8c94,
	0xc498, 0x001a, 0x001a, 0x001a, 0xba94, 0x001a, 0xe097, 0x001a,
	0x4c90, 0x51ed, 0x668e, 0x001a, 0x978e, 0xbe89, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xcf92, 0x001a, 0x001a, 0x4192, 0xc898,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xca88, 0xe192, 0x5a8f,
	0xb28d, 0x4397, 0x001a, 0xcc91, 0x001a, 0xbd89, 0x52ed, 0xc798,
	0x001a, 0x5d97, 0xc398, 0xc598, 0xec8d, 0xc698, 0x439b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xce98, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd198,
	0xcf98, 0x001a, 0x001a, 0xc089, 0x001a, 0xb995, 0xc998, 0x001a,
	0x001a, 0x001a, 0x001a, 0xcd98, 0xf18c, 0x001a, 0x001a, 0x678e,
	0x001a, 0x001a, 0x001a, 0xa48a, 0x001a, 0x001a, 0xd298, 0x001a,
	0xca98, 0x001a, 0x54ed, 0xe197, 0x001a, 0x988e, 0x001a, 0xcb98,
	0x001a, 0xd098, 0x53ed, 0x001a, 0x56ed, 0x001a, 0xd398, 0x001a,
	0xcc98, 0x001a, 0x55ed, 0x9f8b, 0x001a, 0xcb88, 0x001a, 0x001a,
	0xa08b, 0xbf89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x449b, 0x001a, 0x9996, 0x8e95, 0xf28c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4e90, 0xb597, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd695,
	0x001a, 0x001a, 0x578c, 0xa391, 0xe289, 0x001a, 0x001a, 0x001a,
	0x001a, 0x45ed, 0x728f, 0x001a, 0x001a, 0x57ed, 0xd798, 0x001a,
	0xdc98, 0xda98, 0x001a, 0x001a, 0xd598, 0x001a, 0x001a, 0xad91,
	0xd898, 0x001a, 0xdb98, 0xd998, 0x001a, 0xdb95, 0x001a, 0xd698,
	0x001a, 0x4d90, 0x001a, 0x9396, 0xdd98, 0xde98, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x438f, 0xeb98,
	0x001a, 0x001a, 0x001a, 0x6f94, 0x001a, 0x5595, 0xe698, 0x001a,
	0xee95, 0x001a, 0xb489, 0x001a, 0x001a, 0x001a, 0xea98, 0x5aed,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe498, 0xed98, 0x001a,
	0x001a, 0x7191, 0x001a, 0xc28c, 0x001a, 0x7b94, 0x001a, 0xc5e0,
	0x001a, 0xec98, 0x7c93, 0x001a, 0xe198, 0x001a, 0xf48c, 0x001a,
	0x001a, 0xf38c, 0xdf98, 0x001a, 0x001a, 0x001a, 0x5bed, 0xd88e,
	0x001a, 0xe798, 0x59ed, 0xed95, 0x6c92, 0xe398, 0x918c, 0x001a,
	0xe098, 0xe898, 0xe298, 0xcf97, 0xe998, 0x6098, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe48b, 0x001a,
	0x001a, 0x908c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x58ed, 0x001a, 0x5eed, 0xee98, 0x001a, 0x001a, 0x5ced, 0xef98,
	0xf398, 0xcc88, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xce95,
	0xf298, 0x001a, 0x001a, 0x001a, 0x001a, 0xf198, 0xf598, 0x001a,
	0x001a, 0x001a, 0xf498, 0x001a, 0xe292, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x928c, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf698, 0x001a, 0x001a, 0x001a,
	0x5ded, 0x001a, 0xc38e, 0x001a, 0xa491, 0xe392, 0xf48b, 0x001a,
	0xf798, 0x001a, 0x001a, 0x001a, 0x001a, 0x558b, 0x001a, 0x001a,
	0xf898, 0x001a, 0x001a, 0x001a, 0x001a, 0xfa98, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5496, 0x001a, 0x001a,
	0x001a, 0x868c, 0x001a, 0x001a, 0x5fed, 0x001a, 0x001a, 0x001a,
	0x508e, 0xf594, 0xf998, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc38d, 0x6297, 0x001a, 0x001a,
	0x001a, 0x001a, 0xfc98, 0x4299, 0xfb98, 0xc28d, 0x001a, 0x9d8f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x588c, 0x001a,
	0x001a, 0x001a, 0x4399, 0x001a, 0x001a, 0xcd8b, 0x001a, 0x001a,
	0x001a, 0x4099, 0x4199, 0x001a, 0x001a, 0xad93, 0x001a, 0x9c91,
	0x001a, 0xa18b, 0x001a, 0x001a, 0x001a, 0x6c96, 0x4499, 0x001a,
	0x61ed, 0x001a, 0xbb97, 0x001a, 0x001a, 0x001a, 0x4599, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4899, 0x001a, 0x4699, 0x001a, 0x6d91,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4799, 0x4999, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x60ed, 0x4b99, 0x001a, 0x001a,
	0x001a, 0x4a99, 0x001a, 0xc695, 0x001a, 0x001a, 0x001a, 0x001a,
	0x568b, 0x4d99, 0x4e99, 0x001a, 0xad89, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4c99, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf28e, 0x001a, 0x5199, 0x5099, 0x4f99, 0x001a,
	0xd498, 0x001a, 0x5299, 0x001a, 0x001a, 0x001a, 0x001a, 0x9e8f,
	0x001a, 0x5399, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4497, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd796, 0x001a, 0x001a, 0x001a, 0x001a, 0x5599,
	0x001a, 0x001a, 0x5499, 0x5799, 0x5699, 0x001a, 0x001a, 0x5899,
	0x5999, 0xf288, 0x001a, 0xb38c, 0x5a8c, 0x5b8f, 0x9b92, 0xa28b,
	0xe690, 0xf58c, 0x62ed, 0x8e8d, 0x5b99, 0xc696, 0x6593, 0x001a,
	0x998e, 0x001a, 0x5a99, 0x001a, 0x5c99, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7d93, 0x001a, 0x958a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5d99, 0x001a, 0x63ed, 0xfc93, 0x001a, 0x001a,
	0x5391, 0x5f99, 0x6099, 0xaa94, 0xf68c, 0x5a98, 0x6199, 0x001a,
	0x001a, 0xa48b, 0x001a, 0x001a, 0x001a, 0xba95, 0xb491, 0xef8b,
	0x5493, 0x001a, 0x001a, 0x001a, 0x938c, 0x001a, 0x001a, 0x001a,
	0x6299, 0x001a, 0x6399, 0x001a, 0x001a, 0xe093, 0x7e89, 0x001a,
	0x001a, 0x6699, 0xfb8d, 0x001a, 0x6599, 0xc48d, 0x001a, 0x6799,
	0xece3, 0x6899, 0x6096, 0x6999, 0x001a, 0x6a99, 0x6b99, 0xe78f,
	0x001a, 0xca8e, 0x001a, 0x001a, 0x001a, 0x64ed, 0x001a, 0x001a,
	0xa58a, 0x001a, 0x6e99, 0x001a, 0x6c99, 0xbb96, 0x6d99, 0x001a,
	0x7995, 0x6f99, 0x7099, 0x7199, 0x7e93, 0x001a, 0x001a, 0x001a,
	0x7599, 0x7399, 0x7499, 0x7299, 0xe18d, 0x7699, 0xe896, 0xe297,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7799, 0x65ed, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa690, 0x7899, 0x798f, 0x001a,
	0x001a, 0x7999, 0x001a, 0x9c92, 0xbd97, 0x8093, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc399, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7a99, 0xa3ea, 0xc38b, 0x001a, 0x001a,
	0x7b99, 0x7d96, 0x001a, 0x001a, 0x001a, 0x001a, 0x888f, 0xfa91,
	0x001a, 0x7d99, 0xe293, 0x001a, 0x66ed, 0x7e99, 0x001a, 0x001a,
	0x8099, 0x4d8a, 0x001a, 0x001a, 0x001a, 0x8199, 0xa58b, 0x001a,
	0xca93, 0x9a89, 0x6f8f, 0x001a, 0x001a, 0x9f94, 0x8299, 0x001a,
	0x8193, 0x001a, 0x001a, 0x6e90, 0x8399, 0x001a, 0xaa95, 0xd890,
	0xa08a, 0x001a, 0xa78a, 0x8499, 0x001a, 0x001a, 0x8699, 0x001a,
	0x001a, 0x598c, 0x001a, 0x001a, 0x8599, 0x67ed, 0x001a, 0xf197,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x898f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbb94, 0xca95, 0x001a, 0x8799,
	0x001a, 0x9897, 0x8899, 0x001a, 0x001a, 0x001a, 0x8999, 0x001a,
	0x9e93, 0x001a, 0x001a, 0x8a99, 0x001a, 0x001a, 0xa790, 0xfc8d,
	0x948c, 0x8b99, 0x688e, 0x8f8d, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe492, 0x8d99, 0x001a, 0x001a, 0xa591,
	0x001a, 0x001a, 0xed8d, 0x8e99, 0x8f99, 0x4f91, 0x001a, 0x8c99,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9199, 0x001a, 0x5596, 0x001a,
	0x001a, 0x001a, 0x001a, 0x848d, 0x001a, 0x001a, 0x9099, 0x001a,
	0x001a, 0x001a, 0x001a, 0x958c, 0xdc8d, 0x8d94, 0x001a, 0x001a,
	0x001a, 0x9499, 0x9299, 0x001a, 0x001a, 0x001a, 0x001a, 0x9b95,
	0xe88f, 0x9b99, 0x848a, 0x9599, 0x9399, 0x6e91, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9799, 0x001a, 0x9699,
	0x001a, 0x001a, 0x001a, 0x638a, 0x001a, 0x001a, 0x001a, 0x808c,
	0x9c99, 0xab97, 0x001a, 0x001a, 0x001a, 0x9899, 0x001a, 0x001a,
	0x001a, 0x9d99, 0x9a99, 0x001a, 0x9999, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xcd97, 0x68ed, 0x001a, 0x001a, 0xf78c,
	0xc189, 0x001a, 0x001a, 0xf297, 0x001a, 0x001a, 0x69ed, 0x001a,
	0x001a, 0x958f, 0x7793, 0x858d, 0xa099, 0xa199, 0x001a, 0x5bee,
	0x001a, 0xe397, 0x001a, 0x001a, 0x4a98, 0xa399, 0x001a, 0x001a,
	0x001a, 0xf88c, 0x001a, 0x001a, 0xa299, 0x001a, 0x4e8a, 0x001a,
	0x6aed, 0xa499, 0x001a, 0x7596, 0x001a, 0xba92, 0x001a, 0x4597,
	0x001a, 0xd795, 0x001a, 0x001a, 0x001a, 0xa599, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd3e8, 0x001a, 0x001a, 0xae93, 0x001a, 0xa699,
	0xa88a, 0xb196, 0x001a, 0x6bed, 0x001a, 0x9f8f, 0xa799, 0xe595,
	0xab99, 0x001a, 0xa890, 0xa899, 0xce8b, 0x001a, 0xa999, 0xa98a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4d8c, 0xac99, 0x001a, 0xad99, 0x001a, 0x001a,
	0xae99, 0xaf99, 0xd98e, 0x001a, 0x001a, 0x001a, 0xf98c, 0xdc96,
	0x6ced, 0xe696, 0xf593, 0x001a, 0x001a, 0xef95, 0xb099, 0x6ded,
	0xb199, 0x001a, 0x001a, 0x001a, 0x001a, 0xb399, 0x001a, 0xb599,
	0xb499, 0x001a, 0x001a, 0x001a, 0x001a, 0xb699, 0xbb89, 0x6b96,
	0x001a, 0xfa8d, 0xb799, 0x001a, 0x001a, 0x7891, 0x001a, 0x001a,
	0xa08f, 0xa78b, 0x001a, 0xb899, 0x6eed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd994, 0x001a, 0x001a, 0x001a, 0x001a, 0xb999,
	0x001a, 0xba99, 0x001a, 0xbb99, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbc99, 0x4395, 0xe68b, 0xe388, 0x001a, 0x001a, 0x001a, 0xbd93,
	0xbd99, 0x5c8f, 0x001a, 0xe790, 0x001a, 0xbf99, 0xbe99, 0xa18f,
	0xdf8c, 0xc199, 0xbc94, 0x001a, 0x001a, 0xc299, 0x001a, 0x001a,
	0x001a, 0xda94, 0xb291, 0xec91, 0xa68b, 0x001a, 0x001a, 0xec93,
	0x5092, 0x001a, 0x8e94, 0x001a, 0x6d96, 0x001a, 0xc499, 0x001a,
	0xe890, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x548c, 0x001a,
	0x001a, 0xc599, 0x001a, 0x001a, 0x001a, 0x001a, 0xc699, 0x4b89,
	0xf388, 0xeb8a, 0x6fed, 0xa691, 0x708b, 0x9197, 0x001a, 0xc999,
	0xb589, 0x001a, 0x001a, 0xc899, 0x001a, 0x001a, 0x001a, 0xa88b,
	0x001a, 0x001a, 0xca99, 0x001a, 0xef96, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x70ed, 0x001a, 0x001a, 0xcb99, 0x001a,
	0xd097, 0x001a, 0xfa8c, 0x001a, 0x001a, 0x001a, 0x001a, 0xb48c,
	0xcc99, 0x001a, 0x001a, 0x001a, 0x001a, 0xce99, 0xcd99, 0x001a,
	0x7e90, 0x5889, 0x001a, 0x001a, 0x001a, 0x7d89, 0xcf99, 0x001a,
	0xd099, 0x001a, 0x71ed, 0xb58c, 0x001a, 0x001a, 0xd199, 0x001a,
	0x001a, 0x001a, 0x001a, 0x8e8b, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x518e, 0xd299, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9496, 0xb38d, 0x798b, 0x4697, 0x6f91, 0xbd94, 0xfb8e, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x668f, 0x001a, 0xe68e, 0xf38e,
	0x001a, 0x968f, 0x001a, 0xbe94, 0x001a, 0x72ed, 0x001a, 0xd599,
	0x001a, 0x6289, 0x7091, 0xfb8c, 0xc38c, 0xe58b, 0x001a, 0x001a,
	0xd999, 0x4092, 0xfc91, 0xa98b, 0xa28f, 0xda99, 0xd899, 0xc289,
	0xe491, 0xb68e, 0x6a8e, 0x4589, 0x001a, 0x001a, 0x908a, 0x868d,
	0x698e, 0x001a, 0xdb99, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdc99, 0x001a, 0x688b, 0x658a, 0x001a, 0x001a, 0x001a,
	0x878d, 0x678b, 0xdd92, 0x4489, 0xaf93, 0xbc96, 0x408d, 0x9997,
	0x6693, 0xfc8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4e8c, 0x001a, 0xe599, 0x001a, 0xe18b,
	0x6996, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdb94, 0x001a,
	0x001a, 0xe499, 0x001a, 0xdc8a, 0xdf99, 0xe099, 0xe299, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe399, 0x001a,
	0x7a8b, 0x8190, 0x001a, 0xab95, 0xe199, 0xdd99, 0xe18c, 0x001a,
	0xde99, 0x001a, 0x4398, 0x001a, 0x001a, 0x001a, 0xf095, 0x001a,
	0xe692, 0xe08c, 0x908d, 0x001a, 0x001a, 0x001a, 0xe699, 0x001a,
	0x001a, 0xdb93, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xea99,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfc8e, 0x001a, 0xf48e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xed99, 0xeb99, 0x001a, 0xa196, 0x001a, 0xe899, 0xf199, 0xec99,
	0x001a, 0x001a, 0x001a, 0xef99, 0xc48c, 0xbd96, 0x001a, 0x001a,
	0xf099, 0x001a, 0x001a, 0x001a, 0xf299, 0x001a, 0xf499, 0x001a,
	0x001a, 0x001a, 0x75ed, 0xee8d, 0x6198, 0x001a, 0xe999, 0xe799,
	0xf399, 0x001a, 0xee99, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x74ed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf699, 0x001a, 0x429a, 0xf899, 0x001a, 0x001a,
	0xfc99, 0x76ed, 0x001a, 0x409a, 0xf999, 0x001a, 0x001a, 0x5d9a,
	0x001a, 0x001a, 0xe78d, 0x508a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf799, 0x001a, 0x001a, 0x001a, 0x449a, 0xf488, 0x439a, 0x001a,
	0xa388, 0x6995, 0x419a, 0x001a, 0xfa99, 0x001a, 0x001a, 0xf599,
	0xfb99, 0xc68d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x459a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf588, 0x4e9a, 0x001a, 0x001a, 0x469a, 0x479a, 0x001a,
	0xa38f, 0x8996, 0x001a, 0x001a, 0x001a, 0x4c9a, 0x4b9a, 0x001a,
	0x001a, 0x001a, 0x4e93, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4d9a, 0x001a, 0x001a, 0x4a9a, 0x001a, 0x77ed,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5389, 0x001a, 0xb48d, 0x4f90,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x489a,
	0x8293, 0x001a, 0x001a, 0x001a, 0x499a, 0x001a, 0xa088, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x539a, 0x4297,
	0x001a, 0xa58f, 0x001a, 0x599a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x589a, 0x4f9a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc191, 0x001a,
	0x509a, 0x001a, 0x001a, 0x001a, 0xed91, 0x559a, 0xa48f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x529a, 0x001a, 0x001a, 0xe296,
	0x001a, 0x001a, 0x001a, 0x5b8c, 0x001a, 0x001a, 0x569a, 0x579a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x549a, 0x5a9a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x519a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x609a, 0x659a, 0x001a, 0x619a, 0x001a,
	0x5c9a, 0x001a, 0x001a, 0x669a, 0x5091, 0x001a, 0x78ed, 0x689a,
	0x001a, 0x418d, 0x5e9a, 0x9d92, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x629a, 0x5b9a, 0xab8a, 0x001a, 0xec8a, 0x858a, 0x639a, 0x5f9a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x968c,
	0x699a, 0x679a, 0x7291, 0x698b, 0xaa8b, 0x001a, 0x649a, 0x001a,
	0xf28b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6389, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6d9a, 0x6b9a, 0x001a, 0xa59a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x709a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6a9a, 0x001a, 0x6e9a, 0x001a, 0x001a, 0x6c9a,
	0x001a, 0x001a, 0x001a, 0x6b8e, 0x6f9a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x729a,
	0x001a, 0x779a, 0x001a, 0x001a, 0x001a, 0x759a, 0x749a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5192, 0x001a,
	0x001a, 0xc389, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x719a, 0x001a, 0x739a, 0xa68f,
	0x5289, 0x001a, 0x001a, 0x769a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdc89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x829a,
	0x001a, 0xfa8f, 0x7d9a, 0x001a, 0x7b9a, 0x001a, 0x7c9a, 0x001a,
	0x7e9a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5c89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5891, 0x001a, 0x789a, 0x001a,
	0x799a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9a8a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x819a, 0x001a, 0x001a, 0x001a,
	0xed8a, 0x001a, 0x849a, 0x809a, 0x839a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xac95, 0x001a, 0x001a, 0x001a,
	0xd393, 0x001a, 0xb694, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x869a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x859a, 0x648a,
	0x001a, 0x001a, 0x879a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8a9a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x899a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x889a, 0x001a, 0x5894, 0x001a, 0x001a, 0x8b9a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8c9a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8e9a, 0x001a, 0x8d9a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x909a, 0x001a, 0x001a, 0x001a,
	0x939a, 0x919a, 0x8f9a, 0x929a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x949a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x959a, 0x001a,
	0x001a, 0x969a, 0x001a, 0x979a, 0x001a, 0x001a, 0x001a, 0x989a,
	0x6499, 0x001a, 0xfa8e, 0x6c8e, 0x001a, 0x001a, 0xf189, 0x001a,
	0xf688, 0x001a, 0x001a, 0x6392, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x999a, 0x001a,
	0xa28d, 0x001a, 0xcd88, 0x7d90, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9a9a, 0xc58c, 0x001a, 0x001a, 0x918d, 0x001a, 0x9c9a,
	0x9b9a, 0x001a, 0x001a, 0xde95, 0x9d9a, 0x001a, 0x001a, 0x001a,
	0x9f9a, 0x9e9a, 0x001a, 0xa09a, 0x001a, 0xa19a, 0x001a, 0x978c,
	0x001a, 0x001a, 0x8089, 0xa29a, 0x001a, 0x001a, 0xa49a, 0x001a,
	0xa39a, 0x001a, 0x001a, 0x001a, 0xa69a, 0x001a, 0x001a, 0x7993,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa79a, 0xb388,
	0xdd8d, 0x001a, 0x001a, 0x001a, 0x001a, 0x5c8c, 0x001a, 0x001a,
	0x6e92, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa89a,
	0xa99a, 0x001a, 0x001a, 0xab9a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xac9a, 0x001a, 0xe28d, 0x001a, 0x001a, 0x001a, 0x001a, 0xcf8b,
	0x001a, 0x001a, 0x5696, 0x001a, 0x001a, 0x001a, 0xaa9a, 0xad9a,
	0xbf8d, 0x428d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x79ed, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb19a, 0x001a, 0x001a, 0xa38d, 0x7aed, 0x5292, 0x001a,
	0x001a, 0xae9a, 0xd892, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb29a,
	0x001a, 0x001a, 0x8290, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb09a, 0xb39a, 0x001a, 0x5e8c, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb49a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb59a, 0x001a, 0x438d, 0x5f8a, 0xb79a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb89a, 0x001a, 0x7bed, 0x001a, 0x001a, 0x001a,
	0xb99a, 0x001a, 0x001a, 0xb69a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xaf9a, 0x001a, 0x001a, 0xba9a, 0x001a, 0x001a, 0xbb9a, 0x7ded,
	0x7ced, 0x001a, 0x001a, 0x8496, 0x001a, 0x001a, 0xe98f, 0x001a,
	0x001a, 0x001a, 0xbd9a, 0xbe9a, 0xbc9a, 0x001a, 0xc09a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5794, 0x001a, 0x001a, 0xe688,
	0x7595, 0x001a, 0x001a, 0xc19a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfb8f, 0x001a, 0x001a, 0xb78e,
	0x001a, 0x7c94, 0xee8a, 0x001a, 0xe98d, 0x001a, 0x001a, 0x001a,
	0x7896, 0x001a, 0xb093, 0x001a, 0x001a, 0x988c, 0xcd91, 0x001a,
	0x001a, 0x001a, 0xbf9a, 0xc29a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc291, 0x001a, 0x001a,
	0x001a, 0xc39a, 0x001a, 0x001a, 0x001a, 0xc49a, 0x001a, 0x001a,
	0x001a, 0xc69a, 0x001a, 0x001a, 0xe792, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xac8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9fea,
	0x8189, 0xf195, 0x001a, 0x001a, 0xea8f, 0x6793, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe48d, 0x001a, 0x001a, 0xcc9a, 0x001a, 0x001a,
	0xbb95, 0xdb97, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf289, 0xc89a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5991, 0xcb9a, 0x001a, 0x8393, 0x001a, 0x001a, 0x6893,
	0x8493, 0xb794, 0xcb92, 0x001a, 0x001a, 0x001a, 0xc78d, 0x001a,
	0x001a, 0x001a, 0xc79a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9689, 0x001a, 0x5593, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc99a, 0x001a, 0xc59a, 0x001a, 0x001a, 0x6f90, 0x001a, 0x001a,
	0x001a, 0xcd9a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6d8f, 0x001a,
	0x001a, 0x001a, 0x001a, 0xab8b, 0x001a, 0xce9a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe695, 0x001a, 0x001a, 0x001a, 0x9d91,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc492, 0x001a, 0x81ed, 0xd09a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6e96, 0x001a, 0x001a, 0xd19a, 0x001a, 0x001a, 0xd69a, 0x001a,
	0x001a, 0x001a, 0x82ed, 0xad95, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd59a, 0xcf9a, 0xd29a, 0xd49a, 0x001a, 0x001a, 0xa48d, 0x001a,
	0x001a, 0xc795, 0x001a, 0x001a, 0x001a, 0xd79a, 0x001a, 0x6492,
	0x001a, 0x001a, 0xf389, 0x001a, 0xeb8f, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd99a, 0x001a, 0xd89a, 0x001a, 0x888d, 0x001a, 0xda9a,
	0xdc9a, 0xdb9a, 0x001a, 0x001a, 0xde9a, 0x001a, 0xd39a, 0xe09a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xdf9a, 0xdd9a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6d8e, 0x7090, 0x001a, 0x7391, 0xe19a,
	0xba90, 0xeb88, 0x8494, 0x001a, 0x001a, 0x001a, 0x001a, 0xd992,
	0x001a, 0xe39a, 0xe29a, 0xe49a, 0xe59a, 0xe69a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe79a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcf95, 0xe89a, 0x83ed, 0x001a, 0x001a, 0x001a, 0xc489,
	0xe99a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5b97, 0x4f8a, 0x001a,
	0xc799, 0x678f, 0xbd91, 0xea9a, 0xe996, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb296, 0x001a, 0x001a, 0xec9a, 0x001a, 0xe591,
	0x001a, 0x5693, 0xbe91, 0x7695, 0xed9a, 0xee9a, 0x9b89, 0x001a,
	0x001a, 0xb88e, 0xef9a, 0x001a, 0x001a, 0x001a, 0x001a, 0xce88,
	0xf09a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf19a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8289, 0x001a, 0x001a, 0xef8a,
	0xde93, 0xf295, 0x001a, 0x001a, 0x001a, 0x001a, 0xf59a, 0x7491,
	0xf49a, 0x5f8c, 0x001a, 0x84ed, 0x7a96, 0xf39a, 0x001a, 0x8593,
	0xf79a, 0x001a, 0xf69a, 0x85ed, 0x001a, 0x86ed, 0x001a, 0x001a,
	0xf99a, 0x001a, 0xf89a, 0x87ed, 0x001a, 0x9c89, 0x001a, 0xfa9a,
	0xa78f, 0xfc9a, 0x4492, 0x001a, 0xfb9a, 0x001a, 0xb195, 0x001a,
	0x001a, 0x001a, 0x001a, 0x978f, 0x7a93, 0x001a, 0x001a, 0x001a,
	0x409b, 0x001a, 0x001a, 0x001a, 0x001a, 0x448d, 0x001a, 0x001a,
	0x001a, 0x419b, 0x4094, 0xdc94, 0xcf96, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4494, 0x001a, 0x001a, 0x4a9b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x578b, 0x001a, 0x001a, 0x6497, 0x001a,
	0x001a, 0xad96, 0x001a, 0xaa9b, 0x001a, 0x429b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x459b, 0x88ed, 0xc391, 0x001a, 0x001a,
	0x5796, 0x001a, 0x001a, 0x001a, 0x6993, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x469b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8596, 0x89ed, 0xc88d, 0x001a, 0x001a, 0xa88f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x479b, 0x001a,
	0x001a, 0x6f8e, 0x001a, 0x6e8e, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb788, 0xc68c, 0x001a, 0xa990, 0xcf88, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4b9b, 0x4c9b, 0x001a, 0x499b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5789, 0xad8a, 0x001a,
	0x489b, 0x001a, 0xc396, 0x5095, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa688, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf788, 0x001a, 0x001a, 0x001a, 0x708e,
	0x001a, 0xd088, 0x001a, 0xa188, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x519b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4f9b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xba96, 0x001a, 0x529b, 0x001a, 0x509b, 0x001a, 0x001a, 0x4e9b,
	0x5090, 0x001a, 0x001a, 0x001a, 0x001a, 0x4d9b, 0x001a, 0x001a,
	0x001a, 0xd895, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe28c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x569b, 0x579b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa98f, 0x001a, 0x001a, 0x001a,
	0x539b, 0x4b98, 0x001a, 0x001a, 0x001a, 0x001a, 0x6b94, 0x001a,
	0x001a, 0x559b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa58d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x589b, 0x001a, 0x001a, 0x001a, 0x7795, 0x001a,
	0x001a, 0x001a, 0x599b, 0x001a, 0x549b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb996,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7d94, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5a9b, 0x5195, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5b9b, 0x5f9b, 0x5c9b, 0x001a,
	0x001a, 0xc589, 0x5e9b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb98e, 0x001a, 0x5d9b, 0x998c, 0x001a, 0x001a, 0x001a,
	0x6b9b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x649b, 0x619b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8492, 0x001a, 0x609b, 0x001a, 0x001a, 0x629b, 0x001a,
	0x001a, 0x639b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x659b, 0x669b, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf08a, 0x001a, 0x689b, 0x679b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x699b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xec8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6c9b, 0x001a, 0xda92, 0x001a, 0x001a, 0x001a,
	0x6489, 0x001a, 0x6a9b, 0x001a, 0x001a, 0x001a, 0x6d9b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6e9b, 0x001a,
	0x719b, 0x001a, 0x001a, 0x6f9b, 0x001a, 0x709b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x718e, 0x729b, 0x001a, 0x001a, 0x458d, 0x739b, 0x8aed, 0x9a8e,
	0xb691, 0x001a, 0x749b, 0x759b, 0x798e, 0x468d, 0x001a, 0xd096,
	0x001a, 0x001a, 0x001a, 0x478b, 0xc78c, 0x769b, 0x778a, 0x001a,
	0x001a, 0x779b, 0x001a, 0xb791, 0x001a, 0x001a, 0x001a, 0x001a,
	0x789b, 0xa19b, 0x001a, 0x799b, 0x001a, 0x7a9b, 0x001a, 0x001a,
	0x7b9b, 0x001a, 0x7d9b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7e9b, 0x001a, 0x001a, 0x809b, 0x001a, 0xee91, 0x001a, 0x4689,
	0xe78e, 0xc088, 0x001a, 0x7691, 0xae8a, 0xb38e, 0x001a, 0x478d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8693, 0x001a, 0x408f,
	0xaf8a, 0x8892, 0xe892, 0xb688, 0x588b, 0xf395, 0x001a, 0xc08e,
	0x001a, 0x001a, 0x718b, 0xe990, 0xba8e, 0x4797, 0x819b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7b8b, 0x001a,
	0xc98d, 0x001a, 0x001a, 0x518a, 0x8389, 0xaa8f, 0xc689, 0x001a,
	0x829b, 0x6597, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x688f,
	0x8bed, 0x001a, 0xe28e, 0x839b, 0xf18a, 0xd093, 0xa796, 0x849b,
	0x001a, 0x859b, 0x001a, 0x001a, 0x7895, 0x001a, 0x001a, 0x001a,
	0x879b, 0x001a, 0xa68a, 0xf58b, 0x869b, 0x001a, 0x001a, 0x001a,
	0x8ded, 0x001a, 0x001a, 0xb08a, 0x001a, 0x5190, 0x8b9b, 0x408e,
	0x001a, 0xc789, 0x8a9b, 0x001a, 0x889b, 0x8c9b, 0x899b, 0x4a94,
	0xcb9e, 0x5290, 0x001a, 0x8d9b, 0x8eed, 0x001a, 0xbe97, 0x001a,
	0x8e9b, 0x001a, 0x001a, 0x909b, 0x001a, 0x9e92, 0x8f9b, 0x001a,
	0xa190, 0x001a, 0x9b8e, 0x001a, 0x001a, 0x001a, 0xce91, 0xf58e,
	0x001a, 0x9595, 0xea90, 0x001a, 0xcb8e, 0x919b, 0xab8f, 0x929b,
	0x939b, 0xd188, 0xb891, 0x7190, 0x001a, 0x949b, 0xb193, 0xac8f,
	0x001a, 0xad8f, 0x001a, 0x959b, 0x001a, 0x001a, 0xeb90, 0x001a,
	0x001a, 0x001a, 0xae8f, 0x001a, 0x001a, 0x001a, 0x8fed, 0x001a,
	0x969b, 0x001a, 0x979b, 0x001a, 0xde96, 0x001a, 0x001a, 0x001a,
	0x989b, 0x001a, 0x001a, 0x001a, 0x001a, 0xc48b, 0x001a, 0x001a,
	0x001a, 0x418f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x999b, 0x9a9b, 0xda8e, 0x4b90, 0xf293, 0x7390, 0xf694, 0x4194,
	0xc78b, 0x9b9b, 0x001a, 0x001a, 0x001a, 0x8f8b, 0x9c9b, 0x001a,
	0xfc8b, 0x001a, 0xcd93, 0xae89, 0x001a, 0x728e, 0x9d9b, 0xa09b,
	0x9f9b, 0xfb8b, 0x001a, 0x9e9b, 0x001a, 0x5793, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xae91, 0x001a,
	0x6a93, 0xc68e, 0x001a, 0x001a, 0x7791, 0x9a97, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa29b, 0x001a, 0xa39b, 0xd493,
	0x001a, 0x528e, 0x001a, 0x001a, 0x001a, 0x001a, 0xa59b, 0x001a,
	0x001a, 0xa69b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa79b, 0x001a, 0x001a, 0x001a,
	0xf28a, 0xa89b, 0x001a, 0x001a, 0xa99b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xaa89, 0x001a, 0x001a, 0x001a, 0x001a, 0x90ed, 0x001a,
	0x5a91, 0xe28a, 0x001a, 0xab9b, 0xa696, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd091, 0x001a, 0x788a, 0x001a, 0x001a, 0xad9b, 0xaf9b,
	0xdd8a, 0x001a, 0x91ed, 0xac9b, 0xae9b, 0x001a, 0xb19b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb09b, 0x001a, 0xb29b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb39b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbb93, 0xac8b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe389, 0xb49b, 0xb99b, 0x001a, 0x001a, 0xb79b, 0x001a, 0xf595,
	0xf495, 0x001a, 0x001a, 0x001a, 0x001a, 0x92ed, 0x8793, 0x001a,
	0x001a, 0x001a, 0xb69b, 0x738f, 0x001a, 0xb59b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9290,
	0x001a, 0x001a, 0x001a, 0xba9b, 0x001a, 0x001a, 0xe88d, 0x001a,
	0x001a, 0xc09b, 0x001a, 0x001a, 0xc19b, 0xbb9b, 0x528a, 0xbc9b,
	0xc59b, 0xc49b, 0xc39b, 0xbf9b, 0x001a, 0x001a, 0x001a, 0xbe9b,
	0x001a, 0x001a, 0xc29b, 0x001a, 0x001a, 0x001a, 0x001a, 0x93ed,
	0x001a, 0xf695, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x96ed, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc99b, 0xc69b, 0x001a, 0xc89b, 0x001a,
	0x9297, 0x001a, 0xc79b, 0x94ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbd9b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9390, 0x001a, 0x001a, 0xca9b, 0x97ed, 0x001a, 0xb58d,
	0x001a, 0x001a, 0x001a, 0xcb9b, 0x001a, 0x001a, 0xcc9b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcf9b, 0x001a, 0xce9b, 0x001a, 0x001a, 0xcd9b,
	0x001a, 0x001a, 0x001a, 0x8893, 0xb89b, 0x001a, 0x001a, 0x001a,
	0xd59b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd19b, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd09b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd29b, 0x001a, 0xd39b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd69b,
	0x98ed, 0x99ed, 0xe497, 0x001a, 0xd79b, 0xd49b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd89b, 0x001a, 0x001a, 0xde8a, 0xd99b, 0x001a, 0x001a,
	0x9aed, 0x001a, 0xdb9b, 0xda9b, 0x001a, 0x001a, 0xdc9b, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdd9b, 0x001a, 0xec90, 0x428f, 0x001a,
	0x001a, 0x848f, 0x001a, 0x8391, 0x001a, 0x488d, 0xb68d, 0x498d,
	0x908b, 0x001a, 0x001a, 0xde9b, 0x001a, 0x001a, 0xb78d, 0x001a,
	0x001a, 0xc88c, 0xdf9b, 0xa496, 0x6294, 0xe09b, 0x001a, 0x4a8d,
	0x001a, 0x001a, 0x001a, 0xaa8a, 0x001a, 0x4692, 0xd08b, 0x001a,
	0x001a, 0x001a, 0x738e, 0x7a95, 0x001a, 0x001a, 0xbf94, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe19b, 0xf38a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe49b, 0x001a, 0x001a, 0x001a, 0x001a, 0x9f92, 0x001a,
	0x001a, 0xe39b, 0xe29b, 0xe59b, 0x001a, 0xe992, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8390, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x748e, 0x001a, 0xc890, 0x001a, 0xd191,
	0x418b, 0x001a, 0x001a, 0xa092, 0x001a, 0x001a, 0xe69b, 0xe79b,
	0xed8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x5896, 0x001a, 0x001a,
	0xea9b, 0x001a, 0x001a, 0xe99b, 0xe89b, 0x9d95, 0x001a, 0xf19b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7996, 0x001a, 0xeb9b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xed9b, 0x8b96, 0x001a, 0xec9b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xee9b,
	0x001a, 0xa694, 0xef9b, 0xbc95, 0xf09b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb18a, 0xbd95, 0x4e94, 0xf29b, 0xf39b, 0x001a,
	0x4b8d, 0xb28a, 0xf49b, 0xb68c, 0x6397, 0x4897, 0xf48a, 0xf69b,
	0x001a, 0xa192, 0x001a, 0x4c8d, 0xaf8f, 0x001a, 0x001a, 0xdd94,
	0x001a, 0x001a, 0xb08f, 0x001a, 0x001a, 0x001a, 0x001a, 0x988f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xea92, 0xf795, 0x5893,
	0x001a, 0x001a, 0x4d8d, 0x001a, 0x7b95, 0x001a, 0x001a, 0x001a,
	0xf79b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7893, 0xc08d,
	0x001a, 0x001a, 0x001a, 0xc98c, 0x001a, 0xeb92, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc188, 0x8e8f, 0x4e8d,
	0x6697, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf89b, 0xf99b, 0x7094, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfa9b, 0xf597, 0x4c98, 0x001a, 0x001a, 0x001a, 0x001a, 0xfc9b,
	0xfb9b, 0x001a, 0x001a, 0x668a, 0x001a, 0x001a, 0x409c, 0x001a,
	0x001a, 0x001a, 0x439c, 0x449c, 0x001a, 0x429c, 0x001a, 0x5f95,
	0xb18f, 0x469c, 0x459c, 0x419c, 0x001a, 0x001a, 0x001a, 0x001a,
	0x479c, 0x489c, 0x001a, 0x001a, 0x499c, 0x001a, 0x001a, 0x001a,
	0x4c9c, 0x4a9c, 0x001a, 0x4b9c, 0x4d9c, 0x001a, 0x8489, 0xec92,
	0x4e9c, 0x001a, 0x9a8c, 0xf489, 0x5594, 0x001a, 0x4f9c, 0xf993,
	0x001a, 0xd995, 0x001a, 0x509c, 0x4d98, 0x001a, 0x001a, 0x001a,
	0x001a, 0x519c, 0xbe95, 0x549c, 0x9f98, 0xaf98, 0x001a, 0xae8e,
	0xf393, 0x559c, 0x001a, 0x7c8b, 0xa292, 0xf888, 0x569c, 0xa495,
	0x4f8d, 0x001a, 0x001a, 0x6f92, 0x001a, 0x001a, 0x001a, 0xed92,
	0x001a, 0x9bed, 0x001a, 0x001a, 0x001a, 0xed96, 0xb78c, 0xca8c,
	0x001a, 0x579c, 0x001a, 0x001a, 0x001a, 0x589c, 0x001a, 0x5e9c,
	0x001a, 0xe38e, 0x001a, 0x001a, 0x9ced, 0xa392, 0x001a, 0xad8b,
	0x599c, 0x001a, 0x001a, 0x001a, 0x4a95, 0x001a, 0x6592, 0x001a,
	0x001a, 0x5a9c, 0x001a, 0x001a, 0x001a, 0x4bed, 0x001a, 0x001a,
	0x5b9c, 0x001a, 0xae8b, 0x001a, 0x5c9c, 0x001a, 0x5d9c, 0x001a,
	0x001a, 0x5f9c, 0x001a, 0x9693, 0x001a, 0x001a, 0x609c, 0x619c,
	0x001a, 0x629c, 0x001a, 0x001a, 0x539c, 0x529c, 0x001a, 0x001a,
	0x001a, 0x639c, 0x608c, 0x001a, 0x001a, 0x001a, 0x4695, 0x9ded,
	0x001a, 0xca8d, 0x5695, 0xa492, 0x6a95, 0x649c, 0x001a, 0x001a,
	0xb28f, 0x6589, 0x001a, 0x659c, 0x001a, 0x001a, 0x001a, 0x669c,
	0x001a, 0xf096, 0x001a, 0x001a, 0xde94, 0x001a, 0x001a, 0x699c,
	0x9d89, 0xaa90, 0x689c, 0x679c, 0x618c, 0xd291, 0x001a, 0x6d9c,
	0x6b9c, 0x001a, 0x6a9c, 0xa597, 0xe38c, 0x001a, 0x001a, 0x001a,
	0x998f, 0x6c9c, 0x6b93, 0x5d8f, 0x001a, 0x001a, 0x001a, 0xbe93,
	0x709c, 0x6f9c, 0x001a, 0x001a, 0x001a, 0x001a, 0x6e9c, 0x001a,
	0x719c, 0xe48c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x729c, 0x9c95, 0x7a8f, 0x001a, 0x001a, 0x739c, 0xf794, 0x001a,
	0x001a, 0x001a, 0x001a, 0xbf93, 0xa592, 0x001a, 0x001a, 0x9eed,
	0x001a, 0x4f93, 0x001a, 0x001a, 0x749c, 0x4a8b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5390, 0x001a, 0x4b95, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf58a, 0x4594, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x759c, 0x758e,
	0x5996, 0x5a96, 0x001a, 0x001a, 0x9e89, 0x7a9c, 0x9fed, 0x001a,
	0x8992, 0x001a, 0x001a, 0x001a, 0x779c, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf589, 0x001a, 0x001a, 0x001a, 0x001a,
	0xab9c, 0x799c, 0x001a, 0x001a, 0x001a, 0x4f94, 0x001a, 0x001a,
	0x789c, 0x001a, 0x001a, 0x769c, 0x001a, 0x9a8d, 0x001a, 0x7c9c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x839c, 0x899c,
	0x819c, 0x001a, 0x7b93, 0x001a, 0x001a, 0x869c, 0x7c95, 0x001a,
	0x001a, 0x809c, 0x001a, 0x859c, 0xe597, 0x768e, 0x001a, 0x001a,
	0xd391, 0x7d9c, 0x001a, 0x001a, 0x001a, 0x7d8b, 0x889c, 0xab90,
	0x8589, 0x829c, 0xf689, 0x879c, 0x001a, 0x001a, 0x001a, 0xaf8b,
	0x001a, 0x849c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8a9c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8c9c, 0x969c, 0x949c, 0x001a, 0x001a, 0x919c, 0x001a,
	0x001a, 0x001a, 0x909c, 0xf697, 0x001a, 0x929c, 0x001a, 0x001a,
	0xb08b, 0x001a, 0x508d, 0x001a, 0x001a, 0x9a8f, 0x001a, 0x001a,
	0x001a, 0x999c, 0x8b9c, 0x001a, 0x001a, 0xa0ed, 0x001a, 0x8f9c,
	0x7e9c, 0x001a, 0xf889, 0x939c, 0x959c, 0x7092, 0x001a, 0x001a,
	0xa68d, 0xb689, 0x8d9c, 0x989c, 0x979c, 0xb18b, 0x001a, 0xa791,
	0x868a, 0x001a, 0x001a, 0x001a, 0x001a, 0x628c, 0x001a, 0x8e9c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9a9c, 0x001a, 0x9d9c, 0x9f9c, 0xa1ed, 0x001a, 0x001a,
	0x001a, 0xbb8e, 0xa2ed, 0xa59c, 0xee92, 0x9b9c, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa39c, 0x001a, 0xf789, 0x001a, 0xa19c, 0xa29c,
	0x001a, 0x001a, 0x9e9c, 0xa09c, 0x001a, 0x001a, 0x001a, 0xe58c,
	0x4997, 0x001a, 0x001a, 0xb38a, 0x001a, 0x001a, 0x7889, 0xa49c,
	0x001a, 0x5994, 0xab88, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xdf94, 0x7b9c, 0xaa9c, 0xae9c, 0xe396, 0x001a,
	0xa79c, 0x001a, 0x001a, 0x001a, 0x8993, 0xac9c, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xee8f, 0xad9c, 0xd593,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6698, 0x001a, 0xa99c, 0x001a, 0xa4ed, 0x001a, 0x001a,
	0xaf9c, 0x001a, 0x9b8d, 0x001a, 0xc990, 0x001a, 0xa3ed, 0xd288,
	0xa89c, 0xa69c, 0x001a, 0x7991, 0x001a, 0x001a, 0x001a, 0x9c9c,
	0x538e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc491, 0xbb9c, 0xa6ed, 0x7a91, 0xb69c, 0x001a, 0xb39c, 0xb49c,
	0x001a, 0xe48e, 0xb79c, 0xba9c, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb59c, 0x448f, 0x001a, 0xb89c, 0x001a, 0x001a, 0xb29c, 0x001a,
	0xfa96, 0xf996, 0x001a, 0x001a, 0x001a, 0xbc9c, 0xbd9c, 0xd388,
	0x001a, 0xa7ed, 0x001a, 0x001a, 0x001a, 0xb19c, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf08b, 0xa488, 0x001a, 0x001a, 0x001a, 0xb48a,
	0xa5ed, 0xb99c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc19c,
	0xc09c, 0x001a, 0x001a, 0x001a, 0xc59c, 0x001a, 0x001a, 0x001a,
	0xa9ed, 0x001a, 0x001a, 0x001a, 0xc69c, 0x001a, 0x001a, 0xa8ed,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc49c, 0xc79c, 0xbf9c, 0xc39c,
	0x001a, 0x001a, 0xc89c, 0x001a, 0xc99c, 0x001a, 0x001a, 0xbe9c,
	0x9c8e, 0x001a, 0xc29c, 0xd491, 0x518d, 0xb09c, 0x5490, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd69c, 0x001a, 0xe795, 0x001a, 0x001a,
	0xcc9c, 0xcd9c, 0xce9c, 0x001a, 0x001a, 0xd59c, 0x001a, 0xd49c,
	0x001a, 0x001a, 0x9d96, 0xb58a, 0x001a, 0xd29c, 0x001a, 0x648c,
	0x538a, 0x001a, 0x001a, 0xcf9c, 0x001a, 0x001a, 0xb697, 0xd19c,
	0xd488, 0xd39c, 0x001a, 0xca9c, 0xd09c, 0xd79c, 0x638c, 0xcb9c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7c97, 0x001a,
	0x001a, 0x001a, 0x4a97, 0x001a, 0x001a, 0x001a, 0x001a, 0xda9c,
	0x001a, 0x001a, 0xde9c, 0x001a, 0x001a, 0x001a, 0x9e91, 0x001a,
	0xf797, 0xdf9c, 0x001a, 0x001a, 0xdc9c, 0x001a, 0xd99c, 0x001a,
	0xaaed, 0xd89c, 0xdd9c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xae95, 0x001a, 0x001a, 0xb293,
	0x001a, 0x658c, 0x001a, 0xe09c, 0xdb9c, 0x001a, 0xe19c, 0x001a,
	0x001a, 0x001a, 0x9b8c, 0x001a, 0x001a, 0x001a, 0xaf89, 0x001a,
	0x001a, 0x001a, 0xe99c, 0x001a, 0x001a, 0x001a, 0xb68a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe79c, 0x001a, 0x001a, 0xe89c, 0xa78d,
	0xe69c, 0xe49c, 0xe39c, 0xea9c, 0xe29c, 0xec9c, 0x001a, 0x001a,
	0xf989, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xee9c, 0x001a, 0x001a, 0xed9c, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa692, 0x001a, 0xf19c, 0x001a, 0xef9c, 0xe59c,
	0x9c8c, 0x001a, 0xf09c, 0x001a, 0xf49c, 0xf39c, 0xf59c, 0xf29c,
	0xf69c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf79c, 0xf89c, 0xe895, 0x001a, 0xfa9c, 0xf99c, 0x5e8f, 0x001a,
	0xac90, 0xe489, 0xfa89, 0xabed, 0xfb9c, 0x001a, 0xbd88, 0x001a,
	0x001a, 0x001a, 0xca90, 0xfc9c, 0x001a, 0xc1e6, 0x409d, 0x818c,
	0x001a, 0x419d, 0x001a, 0x001a, 0x001a, 0x001a, 0xed90, 0x001a,
	0x001a, 0x001a, 0x429d, 0x001a, 0x001a, 0x001a, 0x439d, 0x598b,
	0x449d, 0x001a, 0x459d, 0x469d, 0xd591, 0x001a, 0x001a, 0x001a,
	0xcb8c, 0x001a, 0x001a, 0xdf96, 0x001a, 0x001a, 0x001a, 0x5b96,
	0x8a8f, 0x479d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xee90,
	0xbbe7, 0xe094, 0x001a, 0xe88e, 0x001a, 0xcb8d, 0x489d, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc591, 0x001a, 0xa595, 0x001a, 0x001a,
	0xef91, 0x001a, 0x001a, 0x4b9d, 0x001a, 0x001a, 0x499d, 0x001a,
	0x4c9d, 0x001a, 0x001a, 0x4a9d, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4d9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xaf95, 0x001a,
	0x001a, 0xb588, 0x001a, 0x001a, 0x001a, 0x001a, 0x7d95, 0x001a,
	0x001a, 0xe194, 0x001a, 0x001a, 0x4e9d, 0x001a, 0x519d, 0xb38f,
	0x5a8b, 0x001a, 0x4f9d, 0x569d, 0xb48f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x509d, 0x6394, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7d97, 0x529d, 0x539d, 0x579d, 0x8a93, 0x549d, 0x528d,
	0xdc90, 0x001a, 0x001a, 0x659d, 0xb294, 0x001a, 0xf091, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xaced, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe294, 0xab9d, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf895, 0x001a, 0x001a, 0x001a, 0xef92, 0x001a, 0x001a,
	0x001a, 0x9596, 0x001a, 0x5a9d, 0x9f89, 0x8a92, 0x001a, 0x001a,
	0x001a, 0x001a, 0x639d, 0x001a, 0x001a, 0x5392, 0x5d9d, 0x649d,
	0x5f9d, 0x669d, 0x629d, 0x001a, 0x619d, 0x8f94, 0x001a, 0x5b9d,
	0xfb89, 0x599d, 0x918b, 0xf191, 0x559d, 0x001a, 0x001a, 0x589d,
	0x538d, 0xd990, 0x001a, 0xb58f, 0x609d, 0x7194, 0x001a, 0x001a,
	0x928b, 0x678a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x878a, 0x4090, 0x689d, 0x6d9d,
	0x001a, 0x699d, 0x001a, 0x9d8c, 0x001a, 0x6e9d, 0x418e, 0x898d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x458f, 0x5c9d,
	0x001a, 0x9d8e, 0x6b9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x778e,
	0x6c9d, 0xc288, 0x001a, 0x001a, 0x679d, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa792, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x938b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb28b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6a9d,
	0xa588, 0x001a, 0x001a, 0xc18d, 0x001a, 0x001a, 0x001a, 0x5590,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf092, 0x001a, 0x001a, 0xd294, 0x709d, 0x7d91,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa891, 0x001a, 0x001a, 0x4a8e, 0x719d, 0x001a, 0x739d,
	0x6f9d, 0x001a, 0x001a, 0x001a, 0x001a, 0xdf95, 0x001a, 0xbb92,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7b91, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf995,
	0xcc8e, 0x809d, 0x001a, 0x7e9d, 0x001a, 0x001a, 0x9890, 0x001a,
	0x001a, 0x001a, 0x9e8c, 0x001a, 0x001a, 0x001a, 0x789d, 0xb78f,
	0x001a, 0x001a, 0xe693, 0x5094, 0x001a, 0x001a, 0x001a, 0x001a,
	0x769d, 0x001a, 0x001a, 0x7c91, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf68e, 0x7b9d, 0x001a, 0x001a, 0xb68f, 0x001a, 0x759d, 0x7a9d,
	0x001a, 0x001a, 0x7294, 0x001a, 0x001a, 0x001a, 0x749d, 0x001a,
	0x408c, 0x001a, 0x001a, 0x7c8a, 0x001a, 0x001a, 0x001a, 0x7c9d,
	0xa997, 0xcc8d, 0x5492, 0x799d, 0x001a, 0xda90, 0x001a, 0x548d,
	0x8490, 0x8689, 0x5b91, 0x779d, 0x648b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x668c, 0x001a, 0xcd92, 0x7d9d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7e91, 0x001a, 0x001a, 0x819d, 0x001a,
	0x839d, 0x001a, 0x001a, 0xb591, 0x899d, 0x001a, 0x849d, 0x001a,
	0x001a, 0x869d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6095,
	0xf192, 0x001a, 0x879d, 0x001a, 0x001a, 0x001a, 0x4b97, 0x001a,
	0x001a, 0x001a, 0x6797, 0xb78a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xac88, 0x001a, 0x859d, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x829d, 0x001a, 0x001a, 0x001a, 0x001a, 0xf68a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8789, 0xaded, 0x889d, 0x001a,
	0x001a, 0x001a, 0x6897, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8c9d, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb991, 0x001a, 0x939d,
	0x001a, 0x001a, 0x001a, 0x8d9d, 0x001a, 0x001a, 0x8a9d, 0x919d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x729d, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8e9d, 0x001a,
	0x929d, 0x001a, 0x001a, 0x001a, 0xc094, 0x8b93, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8b9d, 0x001a, 0x8f9d, 0x001a,
	0x001a, 0x001a, 0x678c, 0x001a, 0x001a, 0x001a, 0xef8d, 0x001a,
	0x001a, 0x001a, 0xdb90, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x979d, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4593, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xaeed, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x949d,
	0x001a, 0x8096, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x959d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x969d, 0x001a,
	0xcc96, 0x001a, 0xa090, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x828c, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9d9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x548e, 0x9a9d, 0x001a, 0x999d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5194, 0x001a, 0x001a, 0xafed, 0xb393, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5093, 0x9b9d, 0x001a, 0x001a,
	0x001a, 0x9c9d, 0x001a, 0x8f95, 0x001a, 0x6494, 0x428e, 0x001a,
	0xef90, 0x001a, 0x6f96, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x688a, 0x001a, 0xa39d, 0x9e9d, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6997, 0xa59d, 0x001a, 0x001a, 0xa19d, 0x001a, 0xa29d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8091, 0xb0ed, 0x001a,
	0x001a, 0x001a, 0xa09d, 0x001a, 0x5e9d, 0x001a, 0x001a, 0x001a,
	0xa49d, 0x001a, 0x9f9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa99d, 0xaa9d, 0x4693, 0xac9d, 0x001a, 0x001a, 0x438e, 0xa79d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5b8b, 0x001a, 0x001a, 0xad9d,
	0x001a, 0xa69d, 0xb19d, 0x001a, 0xb09d, 0x001a, 0xaf9d, 0x001a,
	0x001a, 0x001a, 0xb29d, 0x001a, 0x001a, 0xb49d, 0xef8f, 0x001a,
	0xb39d, 0x001a, 0x001a, 0x001a, 0x001a, 0xb79d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb59d, 0x001a, 0x001a, 0x001a, 0xb69d, 0x909d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb99d, 0xb89d, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x989d, 0xba9d, 0xae9d, 0x001a, 0x001a, 0x788e,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbb9d, 0xbc9d, 0xbe9d, 0xbd9d,
	0xbf9d, 0xfc89, 0x001a, 0x558d, 0x001a, 0x001a, 0xfa95, 0xad90,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcc8c, 0x001a, 0x001a,
	0xc19d, 0x001a, 0x001a, 0x001a, 0x001a, 0xc49d, 0xb1ed, 0x7195,
	0x001a, 0x7e8b, 0x001a, 0x001a, 0x001a, 0xc39d, 0xc29d, 0x7394,
	0xc59d, 0xb38b, 0x001a, 0x001a, 0x001a, 0xc79d, 0xc69d, 0x001a,
	0x001a, 0x001a, 0xb88a, 0x558e, 0x001a, 0x001a, 0xd693, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x688c, 0x001a, 0x001a, 0x001a,
	0x9490, 0x001a, 0xc89d, 0x001a, 0xae90, 0x4793, 0x001a, 0x7e95,
	0xc99d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xca9d, 0xcb9d, 0x001a, 0x001a, 0x001a, 0xb695,
	0x7c9b, 0xc490, 0x001a, 0x001a, 0x6b95, 0x001a, 0xd68d, 0x001a,
	0xe394, 0xc194, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6c93,
	0x001a, 0xbf97, 0x001a, 0xcd9d, 0xce8e, 0x001a, 0x001a, 0xce9d,
	0x001a, 0xb488, 0x001a, 0x001a, 0xd28b, 0xcb90, 0x001a, 0x8095,
	0x001a, 0x001a, 0x001a, 0xcf9d, 0x618e, 0x6692, 0x001a, 0x7a8e,
	0x5690, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd09d,
	0x001a, 0xfb95, 0x001a, 0x001a, 0x9789, 0x7b8e, 0x001a, 0x001a,
	0x001a, 0xd39d, 0x001a, 0xd19d, 0xd49d, 0xb797, 0xd29d, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf990, 0xd59d, 0x001a, 0x001a, 0xb091,
	0x001a, 0x001a, 0xd69d, 0x001a, 0x001a, 0x001a, 0x001a, 0xf88a,
	0x001a, 0xd89d, 0x001a, 0xd79d, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd99d, 0xda9d, 0xf98a, 0x001a, 0x001a, 0xfa93, 0x5592, 0x8c8b,
	0x7c8e, 0x8191, 0x001a, 0x001a, 0x7b8f, 0xae88, 0x001a, 0x001a,
	0x001a, 0xdb9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa089, 0xdf9d, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb2ed, 0x001a, 0x568d, 0xde9d, 0x001a, 0x001a, 0xa98d, 0xb88f,
	0x001a, 0xb5ed, 0xdd9d, 0x001a, 0xb98f, 0x001a, 0xbe96, 0xa88d,
	0x001a, 0x001a, 0x001a, 0xd588, 0xcc90, 0xb3ed, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe49d, 0x001a, 0xb7ed, 0xaf90,
	0x6689, 0x001a, 0x001a, 0x001a, 0xb8ed, 0x748f, 0x001a, 0x8696,
	0xf08d, 0x001a, 0x001a, 0x001a, 0x001a, 0xba8f, 0xb6ed, 0xa590,
	0x001a, 0x47ed, 0x001a, 0x001a, 0xe39d, 0xe19d, 0xe29d, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb4ed, 0x8b92, 0x001a, 0x001a, 0x459e,
	0x001a, 0xe89d, 0x9e8e, 0x578d, 0xe69d, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe79d, 0x001a, 0x5790, 0x001a, 0x001a, 0x001a, 0xe59d,
	0x001a, 0x001a, 0x4e8e, 0x001a, 0x001a, 0x001a, 0x001a, 0xbaed,
	0x001a, 0xbbed, 0x001a, 0x001a, 0x001a, 0xea9d, 0xe99d, 0xee9d,
	0x001a, 0x001a, 0xef9d, 0x001a, 0xeb9d, 0xb9ed, 0x418a, 0xec9d,
	0xed9d, 0xd394, 0x001a, 0x001a, 0x001a, 0x001a, 0x8195, 0x698c,
	0xf09d, 0x001a, 0x001a, 0xbded, 0xb090, 0x001a, 0xbb8f, 0x001a,
	0x001a, 0x001a, 0x7192, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc58b, 0x001a, 0xf19d, 0xf59d, 0x001a, 0x001a, 0xc989,
	0xf29d, 0xf49d, 0x001a, 0x001a, 0x001a, 0x001a, 0xf39d, 0x001a,
	0x001a, 0x8b8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x6792, 0xc388,
	0xf69d, 0xbeed, 0x001a, 0x001a, 0x001a, 0xf79d, 0x001a, 0x001a,
	0xbfed, 0x001a, 0xa892, 0x001a, 0x001a, 0x001a, 0xef97, 0x001a,
	0x001a, 0x001a, 0x001a, 0x628e, 0x001a, 0x001a, 0xe995, 0x001a,
	0x001a, 0x001a, 0xc0ed, 0x001a, 0x5c96, 0x001a, 0x001a, 0x001a,
	0x419e, 0xf99d, 0x001a, 0x001a, 0xfc9d, 0x001a, 0xfb9d, 0xc1ed,
	0x001a, 0xf89d, 0x001a, 0x001a, 0x409e, 0x001a, 0x001a, 0xdc93,
	0x001a, 0xfa9d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x429e, 0x001a,
	0x001a, 0x8c8f, 0x439e, 0x001a, 0x6a97, 0x9894, 0x001a, 0x001a,
	0x449e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x469e, 0x001a,
	0x001a, 0x479e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x489e, 0x001a, 0xc88b, 0x6789, 0x588d, 0x499e, 0x001a, 0x4a9e,
	0x918f, 0x8291, 0xc2ed, 0x4aed, 0xd699, 0x5d91, 0x5c91, 0xd691,
	0xc58d, 0x001a, 0x001a, 0xf098, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8e8c, 0x4c97, 0x001a, 0xfc95, 0x001a, 0x9e95, 0xc3ed, 0x4b9e,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf18d, 0xbd92, 0x4c9e, 0x4e98,
	0x001a, 0x001a, 0x001a, 0x5d96, 0x001a, 0xa992, 0x4d9e, 0xfa8a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4e9e, 0x4f9e,
	0xd896, 0x001a, 0xa296, 0x9696, 0x7b96, 0x448e, 0x519e, 0x001a,
	0x001a, 0xe98e, 0x001a, 0x001a, 0x7096, 0x001a, 0x539e, 0x569e,
	0x559e, 0x001a, 0xf78a, 0x001a, 0x001a, 0x808b, 0x001a, 0x529e,
	0x001a, 0x549e, 0x001a, 0x001a, 0x001a, 0x001a, 0x579e, 0x001a,
	0x001a, 0x9990, 0x001a, 0x001a, 0x001a, 0x001a, 0x9b97, 0xc788,
	0xde8d, 0xba91, 0x001a, 0xdb8e, 0x001a, 0x001a, 0xf18f, 0x001a,
	0x001a, 0x5a9e, 0x001a, 0x001a, 0x6d93, 0x001a, 0x589e, 0xa991,
	0x599e, 0xf08f, 0xdb96, 0x5b9e, 0x5c9e, 0x8897, 0xc5ed, 0x001a,
	0x001a, 0x001a, 0x619e, 0x001a, 0x001a, 0x598d, 0x001a, 0x7494,
	0x5e9e, 0x8c93, 0xdc9d, 0xe09d, 0x001a, 0x6e8b, 0x001a, 0x6694,
	0x001a, 0x001a, 0x001a, 0x001a, 0x609e, 0x001a, 0xbc8f, 0xc294,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x669e, 0x001a, 0xf894,
	0x001a, 0x5d9e, 0x001a, 0x639e, 0x629e, 0x001a, 0x001a, 0x001a,
	0xcd90, 0x001a, 0x001a, 0x001a, 0x001a, 0x8d96, 0x001a, 0xd197,
	0x001a, 0x001a, 0x8796, 0x001a, 0xca89, 0x7d8e, 0x001a, 0x001a,
	0x6798, 0x659e, 0x9590, 0x001a, 0x001a, 0x001a, 0x649e, 0x001a,
	0x001a, 0x5f9e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcd8c,
	0x001a, 0x001a, 0x001a, 0x6b9e, 0x699e, 0x001a, 0xcb89, 0x679e,
	0x6d9e, 0x739e, 0x001a, 0xc6ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc8ed, 0xc691, 0x001a, 0x001a, 0xbf95, 0x001a, 0x759e, 0x001a,
	0x001a, 0x001a, 0x4195, 0x001a, 0x001a, 0x001a, 0x749e, 0x9094,
	0x5e96, 0xb98a, 0x001a, 0xf590, 0x5f8f, 0x001a, 0x001a, 0x001a,
	0xd192, 0x001a, 0x4d97, 0x001a, 0x001a, 0x709e, 0x6f9e, 0x001a,
	0x001a, 0x001a, 0x719e, 0x001a, 0x6e9e, 0x001a, 0x001a, 0x769e,
	0x001a, 0x6c9e, 0x001a, 0x001a, 0x6a9e, 0x001a, 0x729e, 0x689e,
	0x001a, 0x8c92, 0x001a, 0xf696, 0xc48e, 0xf28d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb88d, 0x001a, 0x001a, 0x8f96, 0x608a,
	0x001a, 0xc9ed, 0xcc92, 0xc893, 0x6889, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf090, 0x001a, 0x001a, 0xb290, 0x498c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x789e, 0x001a,
	0x001a, 0x5a8d, 0x9c8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7a9e, 0x948a, 0x819e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7d9e, 0x001a, 0xf190, 0x001a, 0x001a, 0x001a,
	0x6a8a, 0xaa8d, 0x001a, 0x001a, 0x698a, 0xcd8d, 0x001a, 0x001a,
	0x7b9e, 0x858c, 0x6a8c, 0x8d93, 0xcaed, 0x001a, 0x799e, 0x001a,
	0xc488, 0x001a, 0x001a, 0x001a, 0x001a, 0x7c9e, 0x7e9e, 0x001a,
	0xcb8b, 0x4b8c, 0xc7ed, 0xba8a, 0x6a8b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x829e, 0x001a, 0x001a, 0xf78d, 0x9196, 0x001a, 0x568e,
	0x001a, 0x001a, 0x001a, 0x839e, 0x001a, 0x001a, 0x001a, 0x4f95,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8f9e, 0x001a, 0xb189, 0x849e,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x959e, 0x859e,
	0x001a, 0xc097, 0x001a, 0x8c9e, 0x001a, 0x7e94, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x949e, 0x001a, 0x879e,
	0x001a, 0x001a, 0x001a, 0xb288, 0x899e, 0x001a, 0x001a, 0x5b8d,
	0x001a, 0x001a, 0x001a, 0x8b9e, 0x001a, 0x8a9e, 0x001a, 0x869e,
	0x919e, 0x001a, 0xbd8f, 0x001a, 0x001a, 0x001a, 0xeb9a, 0xe68c,
	0x9c97, 0x001a, 0x001a, 0x001a, 0x001a, 0x889e, 0x001a, 0xf292,
	0x428a, 0xab8d, 0x001a, 0x809e, 0x001a, 0x909e, 0x818a, 0x001a,
	0x001a, 0x8e9e, 0x929e, 0x001a, 0x8e93, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfc8a, 0x001a, 0xb09e, 0x001a,
	0x48ed, 0xc796, 0x979e, 0xfb8a, 0x001a, 0x9e9e, 0x001a, 0xcbed,
	0x001a, 0x001a, 0x5f96, 0x001a, 0x9f9e, 0xa19e, 0x001a, 0xa59e,
	0x999e, 0x001a, 0x4992, 0x001a, 0x001a, 0x001a, 0x001a, 0x8f93,
	0xa99e, 0x9c9e, 0x001a, 0xa69e, 0x001a, 0x001a, 0x001a, 0xa09e,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5890, 0xaa9e,
	0x001a, 0x001a, 0xb190, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa89e, 0xbb8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6f98, 0x969e, 0x001a, 0x001a, 0xa49e, 0xd688, 0x001a, 0x001a,
	0x989e, 0x001a, 0x001a, 0xb896, 0x9d9e, 0x4190, 0xc592, 0x939e,
	0x001a, 0x001a, 0xa39e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9a90, 0xad9e, 0x918a, 0x9f8c, 0x001a, 0x001a, 0x001a,
	0x001a, 0xaf9e, 0x9a9e, 0xae9e, 0x001a, 0xa79e, 0x9b9e, 0x001a,
	0xab9e, 0x001a, 0xac9e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbd9e, 0x001a, 0x001a, 0x001a, 0xcc93, 0x001a, 0xa29e, 0x001a,
	0x001a, 0xb99e, 0x001a, 0x001a, 0x001a, 0xbb9e, 0x001a, 0xd692,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6b97, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9695, 0xb69e, 0xc891, 0x001a, 0x001a,
	0x001a, 0xbc9e, 0x5e91, 0x001a, 0xb39e, 0xc09e, 0xbf9e, 0x001a,
	0xed93, 0xbe9e, 0xe893, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcded, 0x001a, 0xc29e, 0xb59e, 0x001a, 0xc68b, 0xb89e, 0x7c8f,
	0x001a, 0x001a, 0x001a, 0x8094, 0xba9e, 0xc98b, 0x001a, 0xb29e,
	0xb49e, 0xb19e, 0x001a, 0x001a, 0x4f98, 0x798a, 0xb79e, 0x001a,
	0x001a, 0xc19e, 0x548a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe58d, 0x001a, 0x001a, 0x001a, 0x7c89, 0x001a,
	0x001a, 0xd29e, 0x001a, 0x001a, 0x5098, 0xd59e, 0x001a, 0x001a,
	0xcfed, 0x001a, 0x001a, 0x5990, 0xd49e, 0x001a, 0x001a, 0x001a,
	0xd39e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd09e,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc49e, 0x001a,
	0x001a, 0xe19e, 0xc39e, 0x001a, 0xd69e, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xce9e, 0x001a, 0x001a, 0xc99e, 0xc69e,
	0x001a, 0xc79e, 0x001a, 0xcf9e, 0x001a, 0x001a, 0x001a, 0xa0ea,
	0x001a, 0x001a, 0xcc9e, 0x5c8d, 0xc692, 0x8491, 0xca9e, 0x001a,
	0xc59e, 0x001a, 0x001a, 0xc89e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6c97, 0x8a96, 0x001a, 0x001a, 0x001a, 0xcd9e, 0xd79e, 0x001a,
	0x001a, 0x001a, 0xd0ed, 0x001a, 0x001a, 0x001a, 0x001a, 0xdf9e,
	0xd89e, 0x001a, 0x001a, 0xe59e, 0x001a, 0xe39e, 0x001a, 0x001a,
	0x001a, 0x001a, 0xde9e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdd9e, 0x001a, 0xce92, 0x001a, 0x8591, 0x001a, 0xdb9e,
	0x001a, 0x001a, 0xd99e, 0x001a, 0x001a, 0xe09e, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe69e, 0xf394, 0xec9e, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe79e, 0xea9e, 0xe49e, 0x001a, 0x001a, 0x9492,
	0x001a, 0x5795, 0x001a, 0xda9e, 0x001a, 0x001a, 0xe29e, 0xbe8f,
	0x001a, 0xcd96, 0xf69e, 0xe99e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa08c, 0xa189, 0x7e8a, 0x001a, 0x001a, 0xd19e, 0x001a,
	0xd1ed, 0x001a, 0x001a, 0x001a, 0x001a, 0xbf8f, 0xee9e, 0x001a,
	0xf59e, 0xf78e, 0x928a, 0x001a, 0x001a, 0x4d92, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xeb9e, 0x001a, 0xd3ed, 0xf09e,
	0xf49e, 0x001a, 0x001a, 0xb48b, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6b8b, 0xf29e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x408b,
	0x001a, 0xc993, 0xf19e, 0x001a, 0x001a, 0x001a, 0xf39e, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd2ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xed9e, 0xd4ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0xef9e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd5ed, 0x808a,
	0x6892, 0x001a, 0x001a, 0x001a, 0xfa9e, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf89e, 0xe78c, 0x001a,
	0xf79e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x409f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x779e, 0x001a, 0x001a, 0x001a,
	0xf99e, 0x001a, 0xfb9e, 0xfc9e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4b9f, 0x001a, 0x479f, 0x001a, 0x8d9e, 0x001a,
	0x001a, 0x001a, 0x001a, 0x469f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x459f, 0x001a, 0x001a, 0x429f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe89e, 0x449f, 0x439f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x499f, 0x001a, 0x4598, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4c9f, 0xf98b, 0x001a, 0x001a, 0x489f, 0x4a9f,
	0x001a, 0x001a, 0xd6ed, 0x001a, 0xd7ed, 0x001a, 0x001a, 0x001a,
	0xa594, 0x001a, 0x4d9f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x519f, 0x4e9f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9397, 0x4f9f, 0x001a, 0x001a,
	0x001a, 0x001a, 0xdc9e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x529f, 0x001a, 0x001a, 0x001a, 0x539f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5489, 0x001a, 0x559f,
	0x878c, 0x9f8e, 0x001a, 0xd38b, 0x001a, 0x001a, 0x001a, 0xa289,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7e97, 0x001a, 0x001a, 0x001a, 0x001a, 0x579f,
	0x569f, 0x599f, 0x5c8b, 0x001a, 0x001a, 0xd48b, 0xbc8a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5c9f, 0x001a, 0x001a, 0x001a, 0x5b9f,
	0x001a, 0x5d9f, 0x001a, 0x001a, 0xcc89, 0x001a, 0x5692, 0x001a,
	0x5e9f, 0x001a, 0x001a, 0xbd8a, 0x609f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5f9f, 0x001a, 0x619f, 0x001a, 0x001a, 0x001a, 0x629f,
	0x001a, 0x639f, 0x7e8e, 0xb390, 0x9f8d, 0x001a, 0x9095, 0x001a,
	0x001a, 0xe095, 0x6398, 0x001a, 0x001a, 0x001a, 0x001a, 0x958e,
	0x001a, 0x001a, 0x001a, 0xce8d, 0xf097, 0x001a, 0x001a, 0x001a,
	0x649f, 0x659f, 0x001a, 0x808e, 0x001a, 0x001a, 0x001a, 0x669f,
	0x679f, 0x001a, 0x001a, 0x699f, 0x689f, 0x001a, 0x7796, 0x001a,
	0x001a, 0x7d8f, 0xea8e, 0x638e, 0x001a, 0x6a9f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6c9f, 0x4290, 0x001a,
	0x6b9f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6d9f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6e9f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6f9f, 0x709f, 0x001a, 0x001a, 0x001a, 0x719f,
	0x001a, 0x739f, 0x729f, 0x749f, 0xa389, 0x6992, 0x001a, 0x759f,
	0x001a, 0x001a, 0x458e, 0x6b8a, 0x769f, 0x001a, 0x001a, 0x6193,
	0xca9a, 0x001a, 0x001a, 0x001a, 0x001a, 0x428b, 0x779f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x789f, 0x001a, 0xea95, 0x8896, 0x001a,
	0x001a, 0x001a, 0xc593, 0x799f, 0xe494, 0x001a, 0xd8ed, 0x001a,
	0xf994, 0x001a, 0x001a, 0xd196, 0x001a, 0x001a, 0x001a, 0x7a9f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7c9f, 0x7b9f, 0x001a, 0x001a, 0x7e9f,
	0x001a, 0x001a, 0x001a, 0x7d9f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x819f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x818e,
	0x001a, 0xaf96, 0x001a, 0x829f, 0x839f, 0x001a, 0x001a, 0x438b,
	0x001a, 0x001a, 0x001a, 0x849f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x869f, 0x859f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8590, 0x001a, 0x001a, 0x5895,
	0x6989, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc394, 0xd9ed,
	0xf392, 0x608f, 0x818b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc494, 0x001a,
	0xac8e, 0x001a, 0x001a, 0x001a, 0x001a, 0x889f, 0x001a, 0xbe8a,
	0x001a, 0x001a, 0x9889, 0x001a, 0xdaed, 0xf093, 0x879f, 0x5d8d,
	0x7292, 0x001a, 0x899f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x919f, 0x001a, 0x8a9f, 0x001a, 0x001a, 0x001a, 0x001a, 0xdced,
	0xbf91, 0x001a, 0x828b, 0x929f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x888c, 0x001a, 0x001a, 0x448b, 0x909f, 0x001a,
	0x001a, 0x8e9f, 0x8b9f, 0x8097, 0x001a, 0x001a, 0xdbed, 0x001a,
	0xbe92, 0x001a, 0x001a, 0x001a, 0xd793, 0x8c9f, 0x001a, 0x001a,
	0x949f, 0x001a, 0x939f, 0x428c, 0x001a, 0x001a, 0xab89, 0x001a,
	0x001a, 0xb98d, 0x8d9f, 0x8f9f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7696, 0xf291, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9796, 0x001a, 0x001a, 0x9c9f, 0x001a,
	0x001a, 0x9d9f, 0x001a, 0xcd89, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa695, 0xfb96, 0x9f9f, 0xa18e, 0xc08f, 0x989f, 0x9e9f, 0x8889,
	0x001a, 0xb58b, 0x001a, 0x001a, 0x959f, 0x9a9f, 0x001a, 0x001a,
	0x001a, 0xf290, 0x9194, 0x001a, 0xe594, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x979f, 0x001a, 0x4096, 0x001a, 0x999f,
	0x001a, 0xa29f, 0xdded, 0xa09f, 0x001a, 0x9b9f, 0x001a, 0x001a,
	0x001a, 0x4196, 0x6794, 0x838b, 0x001a, 0x4493, 0x001a, 0x001a,
	0x8d92, 0x001a, 0xa39f, 0x001a, 0x001a, 0x001a, 0x001a, 0xa19f,
	0xd791, 0x969f, 0x001a, 0x6a89, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xdeed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6d97, 0xae9f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xad9f, 0x001a, 0x001a, 0x001a, 0x001a, 0xf490,
	0x001a, 0xaa9f, 0x001a, 0x8c97, 0x001a, 0x001a, 0xb493, 0xa49f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc392, 0x001a, 0x001a,
	0x001a, 0x6b89, 0x5e8d, 0xa79f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x468f, 0xac9f, 0x001a, 0xab9f, 0xa69f, 0x001a,
	0xa99f, 0x001a, 0x001a, 0x888a, 0x001a, 0xa89f, 0x6894, 0x001a,
	0x001a, 0xac97, 0x001a, 0x001a, 0xf28f, 0xf390, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb49f, 0xb29f, 0x001a, 0x6c95, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xaf9f, 0xb19f, 0x001a, 0x5989, 0x001a,
	0x001a, 0x5f8d, 0x5198, 0x001a, 0x5c8a, 0x001a, 0x8295, 0xe0ed,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8197, 0x001a, 0x001a, 0x438a,
	0x5a90, 0xb39f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb89f, 0x001a, 0xdfed,
	0xc18f, 0x001a, 0x001a, 0x001a, 0x4f97, 0x001a, 0xb59f, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb09f, 0x001a, 0xb69f, 0xe1ed, 0x001a,
	0x001a, 0xdc97, 0x001a, 0x9393, 0xc093, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe2ed, 0x001a, 0x001a, 0x558a,
	0x001a, 0x001a, 0x7489, 0x001a, 0x001a, 0xbc9f, 0x001a, 0x001a,
	0xbf9f, 0x001a, 0x001a, 0x001a, 0xc197, 0x001a, 0x001a, 0x001a,
	0x8497, 0x001a, 0x001a, 0x001a, 0x001a, 0xc69f, 0xc09f, 0xbd9f,
	0x001a, 0x001a, 0x001a, 0xd297, 0xc39f, 0x001a, 0x001a, 0xe3ed,
	0x001a, 0x698f, 0xc59f, 0x001a, 0x001a, 0xca9f, 0x001a, 0x001a,
	0x9193, 0xc89f, 0x001a, 0x001a, 0x001a, 0x001a, 0xc29f, 0x001a,
	0x001a, 0x5792, 0x001a, 0x001a, 0xc99f, 0x001a, 0xbe9f, 0x001a,
	0xc49f, 0x001a, 0xcb9f, 0xfa88, 0xc19f, 0x001a, 0xcc9f, 0x001a,
	0x001a, 0x5b90, 0xe5ed, 0x7e8f, 0x001a, 0xa395, 0x001a, 0xac8d,
	0xe4ed, 0xb99f, 0xc79f, 0x5993, 0xe6ed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb490, 0x001a, 0x898a,
	0xcf8d, 0xc28f, 0xbb9f, 0x618f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6b8c, 0x001a, 0xba9f, 0x001a, 0x001a,
	0x001a, 0xd09f, 0x8d8f, 0xb88c, 0x001a, 0xdf9f, 0x001a, 0xd99f,
	0x948b, 0x6e93, 0x001a, 0xd49f, 0xdd9f, 0xad88, 0x5189, 0xe9ed,
	0x001a, 0xb789, 0x001a, 0xd69f, 0xaa91, 0xcd9f, 0xcf9f, 0x608d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe09f, 0xe7ed, 0xdb9f, 0x001a, 0xeaed, 0x001a, 0xd39f, 0x001a,
	0x001a, 0x001a, 0x001a, 0xda9f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa996, 0x001a, 0x001a, 0xd89f, 0xdc9f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xce8c, 0x001a,
	0xc38f, 0x001a, 0x001a, 0x5892, 0xe8ed, 0x001a, 0x001a, 0xd29f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4e97,
	0x001a, 0x001a, 0x001a, 0xd59f, 0x001a, 0x001a, 0xce9f, 0x9293,
	0x001a, 0x001a, 0xd19f, 0x001a, 0x001a, 0x001a, 0xd79f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7098, 0xbc8e,
	0x9e96, 0x001a, 0xe19f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xac94, 0x001a, 0x001a, 0xed9f,
	0xb98c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x808f, 0x001a,
	0xe39f, 0x001a, 0x001a, 0x001a, 0xad97, 0x618d, 0x001a, 0xf09f,
	0x001a, 0x001a, 0xec88, 0x001a, 0x001a, 0xee9f, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe29f, 0x001a, 0x001a, 0x001a, 0x001a, 0xe89f,
	0x001a, 0x001a, 0xea9f, 0x001a, 0x001a, 0x001a, 0x6e97, 0xe59f,
	0x001a, 0x001a, 0x4d93, 0x001a, 0x001a, 0xe79f, 0x001a, 0xebed,
	0x001a, 0x001a, 0xef9f, 0x001a, 0xe99f, 0xc596, 0x001a, 0x001a,
	0x001a, 0xe49f, 0x001a, 0xa08e, 0xfc9f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8a8a, 0x001a, 0xe69f, 0xeb9f, 0xec9f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xea91, 0xd891, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf49f, 0x001a, 0x001a, 0xfa9f,
	0x001a, 0x001a, 0xf89f, 0x001a, 0x4893, 0x001a, 0x001a, 0x42e0,
	0xf59f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf69f, 0xde9f,
	0x001a, 0x998b, 0x5995, 0x001a, 0x001a, 0x001a, 0xbd8e, 0x001a,
	0x001a, 0x978d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5298,
	0x001a, 0xf29f, 0x001a, 0x41e0, 0x8989, 0x8691, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9994, 0x001a, 0xbf8a, 0xf897, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9f96, 0xd092, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf99f, 0xfb9f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5191, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x40e0, 0xf79f,
	0x001a, 0xf19f, 0x001a, 0x001a, 0x001a, 0xc18a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x898c, 0x001a, 0x001a, 0x001a,
	0x4ee0, 0x001a, 0x001a, 0x49e0, 0xf690, 0x001a, 0x001a, 0x838a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x818f, 0x001a, 0x52e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4be0, 0xaa92, 0x48e0,
	0xd792, 0x001a, 0x001a, 0x001a, 0x6be0, 0x001a, 0x001a, 0x001a,
	0x45e0, 0x001a, 0x44e0, 0x001a, 0x4de0, 0x001a, 0x001a, 0x001a,
	0x47e0, 0x46e0, 0x4ce0, 0x001a, 0x9f90, 0x001a, 0x43e0, 0x001a,
	0xeced, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4fe0, 0x001a,
	0x001a, 0x50e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc08a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x55e0, 0x001a, 0x54e0, 0x56e0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x59e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6293, 0x001a, 0x53e0, 0x001a, 0xeded, 0x001a, 0x001a,
	0x001a, 0x57e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x838c, 0xf791, 0x51e0, 0x5a94, 0x001a, 0x001a, 0x58e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5de0, 0x5be0, 0x001a, 0x001a,
	0x5ee0, 0x001a, 0x001a, 0x61e0, 0x001a, 0x001a, 0x001a, 0x5ae0,
	0x8a8d, 0x4794, 0x001a, 0x001a, 0xb79f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9497, 0x5ce0, 0x001a, 0x60e0, 0xf391,
	0x001a, 0x5fe0, 0x001a, 0x4ae0, 0x001a, 0xeeed, 0x89e8, 0x001a,
	0x001a, 0x001a, 0x64e0, 0x001a, 0x001a, 0x001a, 0x68e0, 0x001a,
	0x001a, 0x66e0, 0x001a, 0x001a, 0x001a, 0xefed, 0x001a, 0xf0ed,
	0x001a, 0x62e0, 0x001a, 0x63e0, 0x001a, 0x001a, 0x001a, 0x67e0,
	0x001a, 0x65e0, 0x001a, 0x001a, 0x001a, 0x6d95, 0x001a, 0x001a,
	0x6de0, 0x001a, 0x6ae0, 0x69e0, 0x001a, 0x6ce0, 0xd293, 0x6ee0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9592, 0xeb91,
	0xf1ed, 0x001a, 0x001a, 0x001a, 0xa390, 0x001a, 0x001a, 0x001a,
	0x6fe0, 0x001a, 0x71e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x70e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf39f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x72e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe593, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x73e0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xce89, 0x001a, 0x001a, 0x001a, 0x9493,
	0x448a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x848b, 0x001a, 0x001a, 0x001a, 0xdc8e, 0xd08d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf2ed, 0x001a, 0x001a,
	0x001a, 0x4698, 0x8690, 0x001a, 0x001a, 0x001a, 0x8a89, 0x001a,
	0x001a, 0x001a, 0x75e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x74e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf3ed, 0x78e0, 0x5992, 0x7be0, 0x76e0,
	0x001a, 0x001a, 0x001a, 0x7ae0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x79e0, 0x5f93, 0xd788, 0x46ed, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf397, 0x001a, 0x001a, 0x7de0, 0x001a, 0x001a, 0x001a, 0x4789,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x80e0, 0x001a, 0x001a, 0x001a, 0x7ee0, 0x001a, 0x7ce0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x77e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4296, 0x001a, 0x001a, 0x001a, 0x82e0, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf5ed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x81e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4ed,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8b89, 0x001a, 0x001a, 0x001a,
	0x001a, 0x84e0, 0xb095, 0x001a, 0x83e0, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb396, 0x001a, 0x001a, 0x001a, 0x001a, 0xc58f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5291, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc48f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7ed, 0xf8ed,
	0x001a, 0xf997, 0x001a, 0x001a, 0x8ae0, 0x001a, 0xf790, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x86e0, 0x8be0, 0x001a,
	0x001a, 0x8c89, 0x001a, 0x001a, 0xf6ed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x89e0, 0x001a, 0x8194, 0x85e0, 0x88e0, 0xc68f,
	0x001a, 0xcf94, 0x001a, 0x001a, 0x8ce0, 0x001a, 0xcf8e, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf890, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8fe0, 0x001a, 0x001a, 0x001a,
	0x87e0, 0x001a, 0x468c, 0x001a, 0x001a, 0x001a, 0x001a, 0x8de0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6f97, 0x90e0, 0x001a, 0x001a,
	0x001a, 0xa4ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6e8f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x91e0, 0x001a, 0x001a, 0x001a, 0x92e0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4d94, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x94e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x95e0, 0x001a,
	0x001a, 0xfaed, 0x001a, 0x5294, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9593, 0x97e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x99e0, 0x001a,
	0xd397, 0x001a, 0x96e0, 0x001a, 0x98e0, 0x8d89, 0x001a, 0x93e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7a9a,
	0x9ae0, 0x001a, 0x001a, 0x001a, 0x001a, 0x8791, 0x578e, 0x9ce0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9be0, 0x4390, 0xd799, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9de0, 0x001a, 0x001a,
	0x001a, 0x9fe0, 0x001a, 0x8ee0, 0x9ee0, 0x001a, 0xfbed, 0xa0e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9a94, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa1e0, 0x001a, 0x001a,
	0xa2e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa3e0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa4e0, 0x001a, 0xdc92, 0x001a, 0xa6e0, 0xa5e0, 0x001a, 0x001a,
	0xa7e0, 0x001a, 0xa8e0, 0x001a, 0x001a, 0xdd8e, 0x8395, 0x001a,
	0x001a, 0x001a, 0xea96, 0xa9e0, 0xaae0, 0x7591, 0xa28e, 0xabe0,
	0xace0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xade0, 0xd095,
	0xc594, 0x001a, 0x001a, 0xaee0, 0x7694, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xab92, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xafe0, 0xe589, 0x001a, 0x8d8b, 0x001a, 0xc496, 0x001a, 0xb496,
	0x001a, 0xb289, 0x5398, 0x001a, 0x001a, 0x001a, 0x001a, 0x7196,
	0x001a, 0xa895, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb590, 0x001a, 0xb0e0, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc193, 0x001a, 0x001a, 0x001a, 0xa18c, 0xb1e0, 0x001a,
	0xd28d, 0xb3e0, 0xb2e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xb4e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb5e0, 0x001a, 0x001a, 0x001a, 0xb6e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5d8b, 0x001a, 0xb7e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xb8e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa28c, 0x001a, 0x001a, 0xc694,
	0x001a, 0xfced, 0xbae0, 0x001a, 0x001a, 0x001a, 0xf38f, 0x001a,
	0x001a, 0xb9e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x40ee, 0x001a,
	0x001a, 0x001a, 0xb68b, 0xbbe0, 0xbde0, 0x001a, 0xbce0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbee0, 0x001a,
	0xcf8c, 0x001a, 0xbfe0, 0x001a, 0x001a, 0x001a, 0x001a, 0xe78b,
	0x001a, 0x5f91, 0x001a, 0x9d8d, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc1e0, 0xc2e0, 0xc0e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xeb8e, 0x001a, 0x001a, 0xc693, 0xb78b, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc4e0,
	0x4b92, 0xc3e0, 0x001a, 0x001a, 0x5498, 0x8294, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc7e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc9e0, 0xc6e0,
	0x001a, 0x001a, 0x001a, 0xd296, 0xc8e0, 0xcae0, 0x001a, 0xc297,
	0x001a, 0x001a, 0x001a, 0x001a, 0x41ee, 0xcee0, 0x001a, 0x001a,
	0x001a, 0xcde0, 0x9692, 0x4c94, 0x001a, 0x001a, 0xa38c, 0xcce0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xcbe0, 0x001a, 0x5097, 0x5197,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcfe0, 0x8e89,
	0x001a, 0x001a, 0x001a, 0x001a, 0x968d, 0x828e, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd0e0, 0xd1e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd3e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x628f, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd5e0, 0x001a, 0xd4e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd6e0, 0x001a, 0x6c8a, 0x001a, 0x001a, 0xd8e0, 0x001a, 0x43ee,
	0xd7e0, 0x001a, 0xdae0, 0xd9e0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xba8c, 0x001a, 0x001a, 0xa697,
	0x001a, 0xca8b, 0x001a, 0xa489, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe88b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdf8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe697, 0xdce0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdee0, 0x001a, 0x44ee, 0x001a, 0x001a,
	0xdfe0, 0x001a, 0xcf89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xdbe0, 0x45ee, 0x588e, 0x001a, 0x001a, 0xbf92, 0xdde0, 0x001a,
	0x001a, 0x001a, 0x48ee, 0x001a, 0x001a, 0x001a, 0x46ee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe2e0, 0x001a,
	0xec8e, 0x001a, 0x001a, 0x47ee, 0x001a, 0xe0e0, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5d8c, 0x001a, 0x001a, 0xc794, 0xe1e0, 0x001a,
	0x001a, 0xfce0, 0x001a, 0x001a, 0x001a, 0x4aee, 0x001a, 0x001a,
	0xe7e0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbb8c, 0x001a,
	0x001a, 0x001a, 0x001a, 0x858b, 0x001a, 0xe4e0, 0x9d97, 0x49ee,
	0x001a, 0xae97, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf491, 0x001a, 0x001a, 0xe6e0, 0x4bee, 0x001a,
	0x001a, 0x4dee, 0x4cee, 0x001a, 0x001a, 0x001a, 0x4eee, 0x001a,
	0x001a, 0x001a, 0xe8e0, 0xd497, 0xd58b, 0xfa94, 0x6994, 0x001a,
	0x001a, 0x001a, 0xe9e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xebe0,
	0x001a, 0xeee0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xeae0, 0x001a, 0x001a,
	0x001a, 0xede0, 0xe88c, 0x6c89, 0xefe0, 0x001a, 0x9090, 0xece0,
	0xda97, 0x001a, 0x4fee, 0xf2e0, 0xa2ea, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf0e0, 0xf3e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xe5e0,
	0xf1e0, 0x001a, 0x001a, 0xba8d, 0x001a, 0x001a, 0xf4e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf5e0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9e97, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x50ee, 0x001a, 0xf6e0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7e0, 0x51ee,
	0x001a, 0x001a, 0xe3e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xf8e0,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc28a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa38e, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf9e0, 0x001a, 0x001a, 0x001a, 0x001a, 0xfae0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfbe0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5a89, 0x001a, 0x001a, 0x001a,
	0x40e1, 0x001a, 0x5a95, 0x41e1, 0x001a, 0x001a, 0xa28a, 0x42e1,
	0x001a, 0x43e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x44e1, 0x001a,
	0x46e1, 0x47e1, 0x45e1, 0x001a, 0x001a, 0x001a, 0x7295, 0x49e1,
	0x48e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x52ee, 0x001a, 0x4be1, 0x4ae1, 0x4ce1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4de1, 0x4fe1, 0x4ee1, 0x001a,
	0x001a, 0x998d, 0x001a, 0x51e1, 0x001a, 0x50e1, 0x001a, 0x001a,
	0xc38a, 0x001a, 0x7290, 0x001a, 0x5b93, 0x001a, 0x52e1, 0xb690,
	0x001a, 0x001a, 0x001a, 0x598e, 0x001a, 0x9989, 0x53e1, 0x001a,
	0x7097, 0x001a, 0x001a, 0xe195, 0x54e1, 0x001a, 0x001a, 0x8ced,
	0x6393, 0x5297, 0x628d, 0x5c90, 0x001a, 0x001a, 0x001a, 0x6a92,
	0xb299, 0x001a, 0xac92, 0xe689, 0x55e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x56e1, 0x001a, 0x5be1, 0x001a,
	0x001a, 0x59e1, 0x58e1, 0xc09d, 0x458a, 0x57e1, 0x001a, 0xd888,
	0x001a, 0xa894, 0x001a, 0x001a, 0xc894, 0x001a, 0x001a, 0x001a,
	0x001a, 0xaf97, 0x5ce1, 0x5ae1, 0x7b92, 0xa490, 0x001a, 0x001a,
	0xa994, 0x001a, 0x4c95, 0x001a, 0x5ee1, 0xaa97, 0x6c8c, 0x5fe1,
	0x001a, 0x5de1, 0xd494, 0x60e1, 0x001a, 0x61e1, 0x001a, 0x53ee,
	0xd988, 0x001a, 0x001a, 0xf48f, 0x66e1, 0x001a, 0x63e1, 0xeb93,
	0x62e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x458b,
	0x001a, 0x001a, 0x69e1, 0x001a, 0x001a, 0x001a, 0x64e1, 0x65e1,
	0x001a, 0x68e1, 0x67e1, 0x4495, 0x001a, 0x001a, 0x6191, 0x6091,
	0x001a, 0x5e8b, 0x001a, 0x001a, 0x6ae1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6be1, 0x001a, 0x001a, 0x6ce1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6ee1, 0x001a, 0x6de1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7589, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x76e1, 0xe694, 0x70e1, 0x001a, 0x72e1, 0x001a, 0x001a,
	0x74e1, 0x5d90, 0x001a, 0x001a, 0x75e1, 0x73e1, 0xbe8e, 0x001a,
	0x001a, 0x001a, 0x6fe1, 0x71e1, 0x001a, 0x6195, 0x001a, 0xc78f,
	0x001a, 0x001a, 0x78e1, 0x001a, 0x001a, 0x77e1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x79e1, 0x001a, 0xa48e, 0xad8d, 0x001a, 0x001a,
	0x9793, 0x7ae1, 0x001a, 0xc992, 0x001a, 0x001a, 0x7ce1, 0x001a,
	0x001a, 0x001a, 0x9f97, 0x7be1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8991, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x82e1, 0x001a, 0x84e1, 0x85e1, 0x7392, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x83e1, 0x001a, 0x80e1, 0x001a, 0x7de1, 0x7ee1,
	0x001a, 0x81e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x88e1, 0x001a, 0x86e1, 0x001a, 0x87e1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x89e1,
	0x8be1, 0x8ce1, 0x8de1, 0x001a, 0x8ee1, 0x001a, 0x001a, 0x8ae1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x90e1, 0x001a, 0x001a, 0x001a, 0x8fe1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x91e1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc397, 0x001a, 0x001a, 0x001a, 0x94e1, 0x92e1,
	0x93e1, 0x001a, 0x001a, 0x001a, 0xe08a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xfc96, 0x001a, 0x001a, 0x001a, 0xc895, 0x001a,
	0x96e1, 0x001a, 0x001a, 0x001a, 0x95e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x97e1, 0x98e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x9ce1,
	0x99e1, 0x9ae1, 0x9be1, 0x001a, 0x9de1, 0x001a, 0x001a, 0x001a,
	0x9ee1, 0x001a, 0x9fe1, 0x001a, 0x001a, 0x001a, 0xa0e1, 0x001a,
	0xa1e1, 0x001a, 0xad94, 0x6f93, 0xa2e1, 0x9294, 0x5395, 0x001a,
	0xa3e1, 0x001a, 0x54ee, 0xa4e1, 0x4993, 0x001a, 0x468a, 0x638d,
	0xa5e1, 0x001a, 0x001a, 0xa6e1, 0x001a, 0x001a, 0xa7e1, 0x001a,
	0x488e, 0x001a, 0x001a, 0xa9e1, 0x001a, 0x001a, 0xa8e1, 0x001a,
	0x001a, 0xaae1, 0xabe1, 0x57ee, 0x55ee, 0x001a, 0x56ee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x58ee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe794, 0x001a,
	0xace1, 0x001a, 0x001a, 0x001a, 0xade1, 0x001a, 0x001a, 0x89ea,
	0xaee1, 0xafe1, 0xb0e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x4d8e,
	0x001a, 0x001a, 0xb1e1, 0x7594, 0x001a, 0x001a, 0x7e96, 0x001a,
	0x6d89, 0x001a, 0x7689, 0x001a, 0x001a, 0xb2e1, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb4e1, 0x001a, 0x001a, 0x001a, 0xb3e1, 0x9093,
	0x001a, 0x001a, 0x001a, 0xb790, 0x589f, 0x001a, 0xb5e1, 0xbf96,
	0x001a, 0xb6e1, 0x001a, 0xc48a, 0xd594, 0xb7e1, 0x001a, 0xb8e1,
	0x001a, 0x001a, 0xb9e1, 0x001a, 0x001a, 0x001a, 0xda96, 0x001a,
	0x001a, 0x001a, 0xd396, 0x001a, 0xbc92, 0x001a, 0x001a, 0x001a,
	0x8a91, 0x001a, 0x001a, 0xbbe1, 0x001a, 0x001a, 0x828f, 0x001a,
	0x001a, 0xc88f, 0x001a, 0x001a, 0xbee1, 0x001a, 0x001a, 0xbde1,
	0xbce1, 0xfb94, 0x001a, 0xc58a, 0xa78c, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc4e1, 0x001a, 0x001a, 0xc1e1, 0x5e90,
	0xb096, 0x001a, 0x001a, 0x001a, 0xc0e1, 0xc2e1, 0xc3e1, 0x001a,
	0x001a, 0xbfe1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc5e1,
	0xc6e1, 0x001a, 0xad92, 0x001a, 0xe18a, 0x001a, 0x001a, 0x001a,
	0x8592, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5aee, 0xc7e1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc8e1, 0xcbe1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8790, 0x001a, 0xc293, 0x001a, 0xcce1, 0x7296, 0x001a,
	0xc9e1, 0x001a, 0x001a, 0xcae1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcfe1, 0x001a, 0x001a, 0x001a, 0x001a, 0xcee1, 0xcde1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd1e1, 0x001a, 0x001a, 0xd0e1, 0x001a,
	0x001a, 0xd2e1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd4e1, 0x001a,
	0xd3e1, 0x001a, 0x001a, 0x001a, 0x001a, 0xcb95, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x758f, 0xc497, 0x001a, 0x001a,
	0xd5e1, 0x001a, 0x001a, 0xb593, 0x001a, 0x001a, 0xd6e1, 0x001a,
	0x001a, 0xd7e1, 0x001a, 0xdbe1, 0xd9e1, 0xdae1, 0x001a, 0xd8e1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdce1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdde1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdee1,
	0x001a, 0x001a, 0xdfe1, 0xb596, 0xe0e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xee96, 0xe1e1, 0x001a, 0x6d92, 0x001a, 0x8a94,
	0x001a, 0xe98b, 0x001a, 0x001a, 0x001a, 0x5a92, 0xe2e1, 0xb88b,
	0x001a, 0x001a, 0x001a, 0xce90, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe3e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xbb8d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe4e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe5e1, 0x001a, 0xa48c, 0xd38d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe7e1, 0x5cee, 0x001a, 0x001a, 0x001a, 0x7593, 0xd48d, 0x6d8b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4396, 0x001a, 0x6a94, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7693, 0x001a, 0x001a, 0x001a, 0x001a, 0x7b8d,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe9e1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5dee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc98f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5eee, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb097, 0x648d, 0x001a, 0x001a, 0xa58c,
	0x001a, 0x001a, 0xa194, 0x001a, 0xebe1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5fee, 0x001a, 0xede1, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe98c, 0x001a, 0x001a, 0x001a, 0x001a, 0xece1, 0xf492,
	0x001a, 0x001a, 0x001a, 0x001a, 0xefe1, 0x568a, 0xeae1, 0x001a,
	0x001a, 0xe894, 0x001a, 0x4f89, 0x001a, 0xea8d, 0x001a, 0x7198,
	0x001a, 0x001a, 0xeee1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf0e1, 0x001a, 0x001a, 0x001a, 0xc995,
	0x001a, 0xd790, 0xf2e1, 0x001a, 0x001a, 0x001a, 0x001a, 0xf3e1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf1e1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6d8a, 0x001a, 0xf9e1, 0x001a, 0xf8e1, 0x001a,
	0x001a, 0xa58e, 0x001a, 0x001a, 0x001a, 0xfae1, 0xf5e1, 0x001a,
	0x001a, 0x001a, 0xfbe1, 0xf6e1, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd694, 0xf4e1, 0x001a, 0x001a, 0xf7e1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x41e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x40e2,
	0x8196, 0x001a, 0x001a, 0x001a, 0xfce1, 0x001a, 0x001a, 0xe988,
	0x001a, 0x001a, 0x001a, 0x001a, 0x43e2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x42e2, 0x001a, 0x001a,
	0x001a, 0xca8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x44e2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6291, 0x001a,
	0x001a, 0x46e2, 0x45e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x47e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe6e1, 0x001a,
	0x001a, 0x001a, 0xe8e1, 0x49e2, 0x48e2, 0x001a, 0x001a, 0x001a,
	0x60ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa68e, 0x001a, 0xe797, 0x001a, 0xd08e, 0x001a,
	0x4ae2, 0x568c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5f8b,
	0x468b, 0x838e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5397, 0x001a, 0x001a, 0x50e2, 0x001a, 0x4fe2, 0x6391, 0x4ce2,
	0x001a, 0x001a, 0x4ee2, 0x001a, 0x001a, 0x6a8f, 0x5f90, 0x4de2,
	0x4be2, 0x001a, 0x4994, 0x001a, 0x001a, 0xcb8f, 0x001a, 0x001a,
	0x5b95, 0x001a, 0x001a, 0x001a, 0x001a, 0xd58d, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9893,
	0x001a, 0x001a, 0x51e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x52e2,
	0x68e2, 0xd68b, 0x001a, 0x001a, 0x5c98, 0x5491, 0x001a, 0x001a,
	0x001a, 0x001a, 0x53e2, 0x001a, 0x001a, 0xd089, 0xf592, 0x9f95,
	0x001a, 0x001a, 0x001a, 0x001a, 0x64ee, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x66ee, 0x001a, 0x54e2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9a8b, 0x55e2,
	0x001a, 0x001a, 0x57e2, 0x001a, 0x001a, 0x001a, 0x58e2, 0x001a,
	0x4894, 0x001a, 0x001a, 0x59e2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5ae2, 0x5be2, 0x001a, 0x001a, 0xd78b, 0xd189, 0xc393,
	0x478f, 0x848e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5ce2, 0x001a, 0x488f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc889, 0x6295, 0x001a, 0x001a, 0x5de2, 0x001a, 0x001a,
	0xe994, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6491,
	0x001a, 0x60e2, 0x001a, 0x61e2, 0x8994, 0x001a, 0x6090, 0x5ee2,
	0x001a, 0x8192, 0x001a, 0x001a, 0x5fe2, 0x001a, 0x001a, 0x001a,
	0xcc8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xda88, 0x001a, 0x001a, 0x001a, 0x001a,
	0x488b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x62e2, 0x001a, 0x001a, 0xf692, 0x001a, 0x63e2, 0xc590, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xab96, 0x001a, 0x001a, 0x4295,
	0x64e2, 0x65e2, 0x7492, 0x001a, 0xc597, 0x001a, 0x001a, 0x67e2,
	0x66e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xed8e, 0x001a,
	0x001a, 0x69e2, 0xee88, 0x001a, 0x001a, 0x001a, 0x001a, 0x6ce2,
	0x001a, 0x001a, 0x001a, 0x6ae2, 0xd289, 0x6d8c, 0x6be2, 0x658d,
	0x928d, 0x001a, 0xe495, 0x6de2, 0x001a, 0x001a, 0x7396, 0x001a,
	0x001a, 0x6fe2, 0x001a, 0x001a, 0x001a, 0xcf90, 0x6e89, 0xb889,
	0xaa88, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6ee2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x70e2, 0x71e2, 0xf58f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x72e2, 0x001a, 0x6e8a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x74e2, 0x001a, 0x001a, 0x001a, 0x8a8c, 0x001a, 0x868b, 0x001a,
	0x001a, 0x75e2, 0xf38b, 0x001a, 0x001a, 0x76e2, 0x001a, 0xfa90,
	0x001a, 0xcb93, 0x001a, 0xde90, 0xf38d, 0x001a, 0x001a, 0x001a,
	0x77e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8292, 0x8b91, 0x001a, 0x79e2, 0x7be2, 0x78e2,
	0x7ae2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x418c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7ce2, 0x458c, 0x001a, 0x001a, 0x001a, 0x878b, 0x7197,
	0x7ee2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x80e2, 0x001a,
	0x001a, 0x001a, 0x4d89, 0x001a, 0x001a, 0x001a, 0x001a, 0x83e2,
	0x001a, 0x001a, 0x001a, 0x968a, 0x82e2, 0x81e2, 0x001a, 0x85e2,
	0x7de2, 0x001a, 0x86e2, 0xa797, 0x001a, 0x87e2, 0x001a, 0x88e2,
	0x001a, 0x67ee, 0xf29a, 0x8ae2, 0x001a, 0x89e2, 0x001a, 0x001a,
	0x001a, 0x8be2, 0x8ce2, 0x001a, 0xb397, 0x8de2, 0x001a, 0xede8,
	0xcd8f, 0x8ee2, 0x8fe2, 0x768f, 0x001a, 0xb693, 0x90e2, 0x68ee,
	0x001a, 0x001a, 0x4792, 0x6aee, 0x001a, 0x91e2, 0x001a, 0x5b92,
	0x92e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa38b, 0x001a,
	0x5e99, 0x7c92, 0xb18e, 0x001a, 0x001a, 0x001a, 0x001a, 0xc68a,
	0x001a, 0x001a, 0x93e2, 0x001a, 0xa0e2, 0x001a, 0x96e2, 0x001a,
	0x888b, 0x001a, 0x95e2, 0xa2e2, 0x001a, 0x001a, 0x001a, 0x94e2,
	0x001a, 0xce8f, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x98e2, 0x99e2, 0x001a, 0x4a93, 0x001a, 0x001a, 0x9ae2, 0x001a,
	0x7d8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7990, 0x8495, 0x001a,
	0x9ce2, 0x001a, 0x001a, 0x001a, 0xe691, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x97e2, 0x001a, 0x9be2, 0x9de2, 0x001a,
	0x001a, 0xf98d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa4e2, 0x4d95, 0x001a,
	0xa494, 0x9993, 0x001a, 0xd88b, 0xa3e2, 0xa1e2, 0x001a, 0xb394,
	0x9ee2, 0x7d92, 0x9b93, 0x001a, 0x9a93, 0x001a, 0xf48d, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb6e2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6e2, 0x001a, 0xa8e2,
	0x001a, 0x001a, 0x001a, 0x001a, 0xabe2, 0x001a, 0xace2, 0x001a,
	0xa9e2, 0xaae2, 0x001a, 0x001a, 0xa7e2, 0xa5e2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x9fe2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcd95, 0xd389,
	0x001a, 0x001a, 0x001a, 0xb3e2, 0x001a, 0xb0e2, 0x001a, 0xb5e2,
	0x001a, 0x001a, 0xb4e2, 0x001a, 0x9394, 0xa596, 0x001a, 0x5a8e,
	0xaee2, 0xb7e2, 0xb2e2, 0x001a, 0xb1e2, 0xade2, 0x6bee, 0xafe2,
	0x001a, 0xc78a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5c92, 0x001a, 0x001a, 0xfb90, 0x001a, 0x001a,
	0x001a, 0xa094, 0x001a, 0x001a, 0xbce2, 0x001a, 0x001a, 0x001a,
	0xa294, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xdf90, 0xb9e2, 0x001a, 0x001a, 0xcd94, 0x001a, 0xbde2, 0xd195,
	0x001a, 0x7a92, 0x001a, 0xb8e2, 0xbae2, 0x001a, 0x001a, 0xbbe2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbee2, 0x001a, 0x001a,
	0xc28e, 0x001a, 0x001a, 0x001a, 0xc493, 0xc3e2, 0xc2e2, 0x001a,
	0x001a, 0xbfe2, 0x001a, 0x001a, 0x001a, 0x5598, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc8e2, 0x001a, 0x001a, 0xcce2, 0xc9e2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc5e2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc6e2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcbe2, 0x001a, 0x001a,
	0x001a, 0xc0e2, 0xd399, 0xc7e2, 0xc1e2, 0x001a, 0x001a, 0xcae2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd0e2,
	0x001a, 0xc88a, 0x001a, 0xcde2, 0x001a, 0x001a, 0x001a, 0xcee2,
	0x001a, 0x001a, 0xcfe2, 0xd2e2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd1e2,
	0xf494, 0x001a, 0x001a, 0x001a, 0x001a, 0xd3e2, 0xfa97, 0xeb95,
	0xd8e2, 0x001a, 0x001a, 0xd5e2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd4e2, 0xd090, 0x001a, 0xd7e2,
	0xd9e2, 0x001a, 0x001a, 0x001a, 0xd6e2, 0x001a, 0xdde2, 0x001a,
	0xdae2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdbe2,
	0xc4e2, 0x001a, 0x001a, 0x001a, 0xdce2, 0xdee2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xdfe2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc495, 0x001a, 0xe0e2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe096, 0x001a,
	0x001a, 0xcc8b, 0x488c, 0xe1e2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb295, 0x001a, 0x8890, 0x001a, 0xae96, 0x001a, 0x001a,
	0xe2e2, 0x001a, 0xb197, 0x001a, 0x001a, 0x9494, 0x001a, 0x6591,
	0x5394, 0x001a, 0x001a, 0x6c8f, 0x001a, 0x001a, 0x001a, 0xbe88,
	0x001a, 0xe7e2, 0xe5e2, 0x001a, 0xe3e2, 0x9f8a, 0x001a, 0xcf8f,
	0xe8e2, 0x001a, 0x001a, 0xe6e2, 0x001a, 0xe4e2, 0xece2, 0x001a,
	0x001a, 0xebe2, 0xeae2, 0xe9e2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xede2, 0x001a, 0x001a, 0x001a, 0xeee2, 0xb890, 0x001a,
	0xefe2, 0x001a, 0xf1e2, 0x001a, 0x001a, 0xf0e2, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd08c, 0x001a, 0x001a, 0x001a, 0x5791, 0x001a,
	0x001a, 0x001a, 0xf3e2, 0x001a, 0x001a, 0x001a, 0x9c93, 0x001a,
	0xf2e2, 0x001a, 0x001a, 0x001a, 0xf4e2, 0x001a, 0xb395, 0x8c91,
	0x668d, 0x001a, 0xf5e2, 0x001a, 0x001a, 0x001a, 0x001a, 0xc697,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7e2,
	0x001a, 0x001a, 0xf8e2, 0x001a, 0xf9e2, 0x001a, 0xfae2, 0x001a,
	0x858e, 0x001a, 0xfbe2, 0x6e8c, 0x001a, 0x001a, 0x8a8b, 0x001a,
	0x498b, 0x001a, 0x40e3, 0x001a, 0xf196, 0x678d, 0xfce2, 0x001a,
	0x001a, 0x001a, 0x43e3, 0xe496, 0x001a, 0x5b94, 0x001a, 0x001a,
	0x5295, 0x001a, 0x001a, 0x001a, 0x838f, 0x42e3, 0x001a, 0xd18e,
	0x688d, 0x868e, 0x898b, 0xb495, 0x41e3, 0x001a, 0x001a, 0x001a,
	0x6691, 0x6196, 0xf58d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x878e, 0xdb92, 0x001a, 0x46e3, 0xdd97,
	0xd78d, 0x001a, 0x47e3, 0x6190, 0x001a, 0x49e3, 0x001a, 0x001a,
	0x001a, 0xd08f, 0xae8d, 0x001a, 0x001a, 0x001a, 0x001a, 0x48e3,
	0x001a, 0x001a, 0x498f, 0xbc8c, 0x6791, 0x44e3, 0x4ae3, 0x001a,
	0x6dee, 0x001a, 0x001a, 0x45e3, 0x6f8c, 0x001a, 0x4de3, 0x51e3,
	0x8b8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ce3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x55e3, 0x6eee, 0x001a, 0x698d, 0x001a,
	0x001a, 0x8d97, 0xba88, 0x52e3, 0x001a, 0x001a, 0x8b8b, 0x001a,
	0x4fe3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x50e3, 0x001a,
	0x001a, 0x9d93, 0x4ee3, 0x4be3, 0x001a, 0x478a, 0xe290, 0x001a,
	0x001a, 0xa68c, 0x001a, 0x001a, 0x001a, 0x57e3, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x54e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x56e3,
	0x001a, 0x001a, 0x001a, 0x53e3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x708c, 0xb191, 0x58e3, 0x8e91, 0x001a, 0x001a, 0x65e3,
	0x70ee, 0x001a, 0x61e3, 0x5be3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5fe3, 0xf88e, 0xdb88, 0x5ae3, 0x62e3,
	0x66e3, 0x6a8d, 0xd496, 0x001a, 0xd492, 0x5ce3, 0x001a, 0x6fee,
	0x64e3, 0x001a, 0x59e3, 0x5d92, 0x001a, 0x5ee3, 0xbb88, 0xc896,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5de3,
	0x001a, 0x001a, 0xd98b, 0xea94, 0x001a, 0x001a, 0x001a, 0x8d91,
	0x001a, 0xce97, 0x8f8f, 0x001a, 0x001a, 0x8ee3, 0x71ee, 0x001a,
	0x67e3, 0x001a, 0xfc90, 0x001a, 0x63e3, 0x68e3, 0x6ae3, 0x001a,
	0xf792, 0x6de3, 0x001a, 0x001a, 0x69e3, 0x001a, 0x001a, 0x001a,
	0xd295, 0xc98a, 0x001a, 0x001a, 0xc996, 0x001a, 0x001a, 0xdc88,
	0x001a, 0x001a, 0x6ce3, 0x001a, 0xfb97, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6be3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8f89, 0x001a, 0x001a, 0xea93, 0x6ee3, 0x001a, 0x001a,
	0x001a, 0x75e3, 0x6fe3, 0x76e3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x72e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9b94, 0x001a, 0x001a, 0xc88e, 0x74e3,
	0x001a, 0x71e3, 0x77e3, 0x70e3, 0x001a, 0x001a, 0x638f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4496, 0x001a, 0x001a, 0x6b8f, 0x001a,
	0x001a, 0x73e3, 0x80e3, 0x001a, 0x001a, 0x7be3, 0x001a, 0x7ee3,
	0x001a, 0x7ce3, 0x81e3, 0x7ae3, 0x001a, 0x60e3, 0xd190, 0x001a,
	0x001a, 0xc994, 0x001a, 0x7de3, 0x001a, 0x001a, 0x78e3, 0x001a,
	0x001a, 0x001a, 0x4091, 0x718c, 0x001a, 0x4a8f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x72ee, 0x001a, 0x4490, 0x5591, 0x84e3, 0x001a,
	0x001a, 0x86e3, 0x87e3, 0x001a, 0x001a, 0x83e3, 0x85e3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x79e3, 0x82e3,
	0x001a, 0x8ae3, 0x89e3, 0x001a, 0x001a, 0x9a96, 0x001a, 0x001a,
	0x4a8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x88e3, 0x001a, 0x8ce3, 0x8be3, 0x8fe3, 0x001a, 0x91e3,
	0x001a, 0x001a, 0x5b8e, 0x8de3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x92e3, 0x93e3, 0x40ed, 0x001a, 0x94e3, 0x001a, 0x9ae3, 0x5a93,
	0x96e3, 0x001a, 0x95e3, 0x97e3, 0x98e3, 0x001a, 0x99e3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9be3, 0x9ce3, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xca8a, 0x001a,
	0x9de3, 0x001a, 0x9ee3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9fe3, 0x001a, 0x73ee,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa0e3, 0xa1e3, 0xa2e3, 0x001a,
	0xa3e3, 0xa4e3, 0x001a, 0x001a, 0xa6e3, 0xa5e3, 0x001a, 0x001a,
	0xa7e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa8e3,
	0xa9e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xace3,
	0xaae3, 0xabe3, 0xdf8d, 0x728c, 0x001a, 0x001a, 0x7592, 0x001a,
	0xb194, 0x001a, 0x908f, 0x001a, 0x001a, 0x6c94, 0x001a, 0xeb94,
	0xade3, 0xeb9c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xaee3, 0xb0e3, 0x001a, 0x8597, 0xafe3, 0xb2e3,
	0xb1e3, 0x001a, 0x7297, 0x001a, 0xb3e3, 0x001a, 0xfc94, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xb4e3, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb7e3, 0x001a, 0x001a, 0xb6e3, 0xb5e3, 0x001a,
	0x001a, 0x74ee, 0x001a, 0xb8e3, 0x518c, 0x001a, 0x001a, 0x001a,
	0x4191, 0x608b, 0x001a, 0x001a, 0x001a, 0x001a, 0xbce3, 0xb9e3,
	0x001a, 0x001a, 0xbae3, 0x001a, 0x001a, 0x001a, 0xbde3, 0x001a,
	0xbee3, 0xbbe3, 0x001a, 0x001a, 0x001a, 0x4889, 0x001a, 0x001a,
	0x001a, 0xa589, 0x001a, 0x001a, 0x001a, 0xc0e3, 0xc1e3, 0x001a,
	0x001a, 0x001a, 0xc2e3, 0x001a, 0x8297, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4b8f, 0x001a, 0xc4e3, 0xc3e3, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8990, 0xc5e3, 0x001a, 0x001a, 0x001a, 0x001a, 0xc6e3, 0x001a,
	0x001a, 0xc7e3, 0x001a, 0xe38a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcb8a, 0x001a, 0x001a, 0xc8e3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc9e3, 0x001a, 0x7c96, 0x8397, 0x001a, 0x001a, 0x001a,
	0x7397, 0x5698, 0x001a, 0x6c8d, 0xcce3, 0xd28e, 0xcbe3, 0x001a,
	0x001a, 0x001a, 0x001a, 0xcde3, 0xa78e, 0x001a, 0x001a, 0x001a,
	0xcf91, 0x001a, 0xcee3, 0x001a, 0x001a, 0x6b8d, 0x001a, 0xd596,
	0xcfe3, 0xd0e3, 0x001a, 0x001a, 0xd1e3, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd2e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd3e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa88e, 0x001a, 0x001a, 0xeb96, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd5e3, 0x001a, 0x5e92, 0x001a, 0xd4e3,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd7e3, 0x001a,
	0x001a, 0x001a, 0xd6e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd8e3, 0x001a, 0x001a, 0x001a, 0xb990, 0x001a,
	0xd9e3, 0x001a, 0xdae3, 0x001a, 0x001a, 0x001a, 0xb795, 0xdbe3,
	0x001a, 0x8f91, 0xdce3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xdde3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xfc97,
	0xe0e3, 0x001a, 0xdfe3, 0xdee3, 0xae92, 0x001a, 0xe1e3, 0x4590,
	0x001a, 0xe2e3, 0x001a, 0x001a, 0x001a, 0xe3e3, 0x5798, 0xe4e3,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe5e3, 0xe7e3, 0xe6e3, 0xa394,
	0x001a, 0xf793, 0x001a, 0x5d98, 0xa794, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe9e3, 0x001a, 0x001a, 0xd18f, 0x001a,
	0x4995, 0x001a, 0xeae3, 0xe8e3, 0x001a, 0xcc8a, 0x001a, 0x001a,
	0x001a, 0xd28c, 0x888e, 0x001a, 0x001a, 0xec94, 0x001a, 0x001a,
	0x001a, 0xa88c, 0x6296, 0x001a, 0xede3, 0xebe3, 0x001a, 0x6d8d,
	0x001a, 0x6e8d, 0xe788, 0x001a, 0xe68d, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7894, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdd88, 0xf2e3, 0x001a, 0x5f92, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7794, 0x001a, 0xd991, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4e3, 0x001a,
	0x001a, 0xf0e3, 0xf3e3, 0xeee3, 0x001a, 0xf1e3, 0x4596, 0x001a,
	0x001a, 0xd38c, 0x001a, 0x001a, 0xfb88, 0xefe3, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf6e3,
	0x001a, 0xf7e3, 0x001a, 0x001a, 0xb793, 0x001a, 0x001a, 0x001a,
	0xb98b, 0x001a, 0x001a, 0x001a, 0x45e4, 0x5c94, 0x001a, 0x001a,
	0x001a, 0x001a, 0x898e, 0x001a, 0x001a, 0xba8b, 0xc690, 0x6598,
	0xac96, 0xf5e3, 0xd290, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x728b, 0xf8e3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xfae3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf9e3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xfbe3,
	0x001a, 0x4592, 0x001a, 0x5d94, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xaf92, 0x001a, 0x001a, 0x001a, 0x001a, 0x42e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x41e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0xfce3, 0x001a, 0x001a, 0x7490, 0x001a,
	0x8595, 0x44e4, 0x001a, 0x43e4, 0x6f8d, 0x7298, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x54e4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x48e4, 0x49e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0xee8e, 0x001a, 0x001a, 0x47e4, 0x001a,
	0x988d, 0x46e4, 0x001a, 0x001a, 0x4ae4, 0x001a, 0x001a, 0x001a,
	0xb092, 0xa095, 0x4291, 0x001a, 0x001a, 0x001a, 0x001a, 0xda91,
	0x4ee4, 0x001a, 0x4fe4, 0x4be4, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4ce4, 0x001a, 0x4de4, 0x001a, 0x001a, 0x001a, 0x001a, 0x708d,
	0x001a, 0x001a, 0x001a, 0x55e4, 0x001a, 0x51e4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8695, 0x001a, 0x8c96, 0x4795, 0x001a, 0x001a,
	0x50e4, 0x001a, 0x001a, 0x53e4, 0x52e4, 0x001a, 0x001a, 0x001a,
	0x6396, 0x56e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x57e4, 0x001a, 0x001a, 0x5691, 0x001a, 0x58e4, 0x001a, 0x001a,
	0x5ae4, 0x001a, 0x5ee4, 0x001a, 0x001a, 0x5be4, 0x59e4, 0x5e94,
	0x5ce4, 0x001a, 0x5de4, 0x001a, 0x001a, 0x001a, 0xb089, 0x001a,
	0x64e4, 0x5fe4, 0x001a, 0x001a, 0x001a, 0x60e4, 0x001a, 0x001a,
	0x001a, 0x61e4, 0x001a, 0x9f91, 0x001a, 0x001a, 0x001a, 0x001a,
	0x63e4, 0x62e4, 0x65e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x66e4,
	0x67e4, 0x001a, 0x001a, 0x6290, 0x001a, 0xe789, 0x001a, 0x68e4,
	0xd597, 0x001a, 0xa98e, 0x001a, 0x001a, 0x4c8f, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x8a8e, 0x7692, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x69e4, 0x6ae4, 0x5089, 0x001a, 0x6be4, 0x001a,
	0x001a, 0x6ce4, 0x6de4, 0x001a, 0x001a, 0x6ee4, 0x001a, 0x6fe4,
	0xbb8b, 0xa89d, 0x70e4, 0x001a, 0xe390, 0x71e4, 0xc98e, 0x001a,
	0x72e4, 0x001a, 0xae98, 0x001a, 0x001a, 0x001a, 0x73e4, 0xdc95,
	0xda8a, 0x001a, 0x001a, 0x4391, 0x778f, 0x001a, 0x9195, 0x4d8f,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x74e4, 0x718d, 0x75e4, 0xca94, 0x001a, 0x84e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0x77e4, 0x001a, 0xc791, 0x9594, 0xbd8c,
	0x76e4, 0x4491, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x78e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf892,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7ae4, 0x79e4, 0x7ce4, 0x001a, 0x001a, 0x7be4, 0x001a, 0x7de4,
	0x001a, 0x001a, 0x80e4, 0x001a, 0x7ee4, 0x001a, 0xcd8a, 0x001a,
	0x81e4, 0x001a, 0x82e4, 0x83e4, 0x001a, 0x001a, 0xaf8d, 0xc797,
	0x001a, 0x85e4, 0x4690, 0x001a, 0x001a, 0x001a, 0x9089, 0x86e4,
	0x87e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x88e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf088, 0x001a, 0x89e4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8ae4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8795, 0x001a, 0x001a, 0x001a, 0xc58e, 0x001a, 0x8ce4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x488a, 0xb088, 0x001a,
	0x001a, 0x001a, 0x001a, 0x8be4, 0x8ee4, 0x6d94, 0x001a, 0x6390,
	0x001a, 0xd489, 0x001a, 0x4696, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7c8c, 0xda8b, 0x001a, 0x8de4, 0x001a, 0xe889, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa18a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9189, 0x92e4, 0xe897, 0xdb91, 0x001a, 0x001a, 0x6395,
	0x001a, 0x9ee4, 0x001a, 0xd589, 0x9ce4, 0x001a, 0x9ae4, 0x91e4,
	0x001a, 0x8fe4, 0x001a, 0x90e4, 0x001a, 0xe18e, 0xea8b, 0x9792,
	0x001a, 0x001a, 0x001a, 0xcf93, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7089, 0x001a, 0x94e4, 0x93e4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x99e4, 0x95e4, 0x98e4, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x76ee, 0xce96, 0x97e4, 0xd689, 0x9d8a, 0x9be4, 0x001a,
	0x001a, 0x9de4, 0x001a, 0x001a, 0x001a, 0x001a, 0x738c, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa1e4, 0xaae4,
	0xabe4, 0x001a, 0x001a, 0x001a, 0xa988, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb2e4, 0x001a, 0x001a, 0x001a, 0x001a,
	0xef88, 0x001a, 0x001a, 0xa9e4, 0x001a, 0x001a, 0x001a, 0xa8e4,
	0x001a, 0xa3e4, 0xa2e4, 0x001a, 0xa0e4, 0x9fe4, 0x8392, 0x001a,
	0xf991, 0xa5e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa4e4, 0x001a, 0x001a, 0x001a, 0x001a, 0xa7e4, 0x001a, 0x001a,
	0x001a, 0x9091, 0x748c, 0x001a, 0x001a, 0x001a, 0x001a, 0x6089,
	0xa6e4, 0x001a, 0x728d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9191, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x77ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb8e4, 0x001a, 0xb9e4, 0x001a, 0xd789,
	0x001a, 0x001a, 0x001a, 0xac89, 0xb6e4, 0x001a, 0x001a, 0x78ee,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xace4, 0x001a, 0xb4e4,
	0x001a, 0xbbe4, 0xb5e4, 0x001a, 0x001a, 0x001a, 0xb3e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0x96e4, 0x001a, 0x001a, 0xb1e4, 0x001a,
	0x001a, 0x001a, 0xade4, 0x001a, 0x001a, 0x001a, 0xce8a, 0xafe4,
	0xbae4, 0x001a, 0xb0e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbce4, 0x001a, 0xaee4, 0x9c94, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8997, 0x001a, 0x001a, 0x001a, 0xb7e4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcde4, 0x001a, 0x001a,
	0x001a, 0xc5e4, 0x001a, 0x001a, 0x001a, 0x9b90, 0x001a, 0x79ee,
	0x001a, 0x001a, 0x658b, 0x001a, 0xdb8b, 0x001a, 0xc0e4, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd989, 0x001a, 0x001a, 0xd28f, 0x001a,
	0xc3e4, 0x001a, 0x001a, 0x001a, 0xd88d, 0x001a, 0x001a, 0x7093,
	0xc8e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xec95, 0x001a, 0xbfe4, 0x001a, 0x001a, 0x001a, 0xd889,
	0xd48c, 0x4895, 0xc9e4, 0x001a, 0xbde4, 0x001a, 0x7aee, 0xc6e4,
	0x001a, 0x001a, 0x001a, 0xd0e4, 0x001a, 0xc1e4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc2e4, 0xb893, 0x001a, 0x001a, 0xc7e4,
	0x001a, 0x001a, 0x001a, 0xc4e4, 0x4796, 0xcae4, 0xde88, 0x001a,
	0x001a, 0x001a, 0x001a, 0xbee4, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcce4, 0x001a, 0xcbe4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8b94, 0xd2e4, 0x001a, 0xdde4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9e8a, 0x001a, 0x001a, 0x001a, 0xe0e4, 0x001a, 0x001a,
	0xcee4, 0x001a, 0x001a, 0x001a, 0xd3e4, 0x8e97, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdce4, 0x001a,
	0x7bee, 0x7497, 0x001a, 0x001a, 0x001a, 0x001a, 0xa897, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9892,
	0x001a, 0x001a, 0x001a, 0x8b8a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9295, 0xe2e4, 0x9f93, 0x001a, 0x001a, 0xaf88, 0x001a,
	0x001a, 0xdbe4, 0x001a, 0xd7e4, 0x9291, 0xd1e4, 0xd9e4, 0xdee4,
	0x001a, 0x4b94, 0x001a, 0x001a, 0x001a, 0xa888, 0x001a, 0xd6e4,
	0x001a, 0xdfe4, 0x9895, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xdae4, 0x001a, 0xd5e4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd38f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4e8f, 0x001a, 0x001a, 0x001a, 0xaa8e, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd696, 0x001a, 0x001a, 0x6695, 0x001a, 0x001a, 0xe5e4,
	0x001a, 0xeee4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd8e4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x978a, 0x001a, 0x7cee, 0x001a, 0x001a, 0x001a,
	0xf68f, 0xe3e4, 0x001a, 0xe8e4, 0x9391, 0x001a, 0x001a, 0xe4e4,
	0x001a, 0xebe4, 0x001a, 0x001a, 0x7e92, 0x001a, 0xece4, 0x001a,
	0x001a, 0x7597, 0xe1e4, 0x578a, 0x001a, 0xe7e4, 0x001a, 0x001a,
	0xeae4, 0xaa96, 0x001a, 0x001a, 0x001a, 0x001a, 0xede4, 0x001a,
	0x001a, 0xe6e4, 0xe9e4, 0x001a, 0x44ed, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4896, 0x001a, 0x4098, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf1e4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf8e4, 0x001a, 0x001a, 0xf0e4,
	0xc18e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcfe4, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcc95, 0x001a, 0xa096, 0xf7e4, 0xf6e4, 0x001a, 0xf2e4,
	0xf3e4, 0x001a, 0x5589, 0x001a, 0x001a, 0x001a, 0x001a, 0xf5e4,
	0x001a, 0xefe4, 0x001a, 0x001a, 0x001a, 0x001a, 0xd392, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf4e4, 0xfc88, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa091, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc195, 0x001a, 0x001a,
	0xf9e4, 0x40e5, 0x001a, 0xd794, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfce4, 0xd48f, 0xc78e, 0x42e5, 0x001a, 0x001a, 0xbc8b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7dee, 0x001a, 0x43e5, 0x001a, 0x9995,
	0xfbe4, 0x7eee, 0xd4e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xfae4, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6e98, 0xa093, 0x9395, 0x80ee, 0x001a, 0x4ae5, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x50e5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x51e5, 0x001a,
	0x44e5, 0x001a, 0x001a, 0x001a, 0x9694, 0x001a, 0x001a, 0x4ee5,
	0x46e5, 0x001a, 0x48e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x52e5, 0x47e5, 0x001a, 0x001a, 0x4be5, 0x001a, 0x001a, 0x9289,
	0x001a, 0xe393, 0x001a, 0x4ce5, 0x4fe5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x45e5, 0x001a, 0x4591, 0x001a,
	0x49e5, 0x468e, 0x6490, 0x4f8c, 0xf296, 0x001a, 0xf796, 0x928f,
	0x82ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x56e5, 0x54e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6d98, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x53e5, 0x001a, 0x001a, 0x001a, 0x9597, 0x001a, 0x55e5,
	0x57e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x58e5, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5be5, 0x59e5, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa193, 0x5ae5, 0x001a, 0x001a,
	0x001a, 0xcb94, 0x4de5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x938f,
	0x001a, 0x5ce5, 0x61e5, 0x9491, 0x001a, 0x001a, 0x60e5, 0x001a,
	0x001a, 0x001a, 0x41e5, 0x001a, 0x001a, 0x001a, 0x62e5, 0x6891,
	0x001a, 0x001a, 0x5de5, 0x5fe5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5ee5, 0x001a, 0x001a, 0x509f, 0x419f,
	0x001a, 0x001a, 0x64e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x63e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9697, 0x001a, 0xbae1,
	0x65e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x66e5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x67e5, 0xd58c, 0x001a,
	0x738b, 0x001a, 0x001a, 0x001a, 0x69e5, 0x7c99, 0x001a, 0x001a,
	0x001a, 0x001a, 0x958b, 0x001a, 0xb897, 0x001a, 0xf18b, 0x6ae5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6be5,
	0x001a, 0x001a, 0x001a, 0x8e92, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6ce5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf893, 0x001a, 0xb888, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe189, 0x71e5, 0x72e5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6de5, 0x001a, 0x5c8e, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6ee5, 0x6194, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6fe5, 0x70e5, 0x7ae5, 0x001a, 0x001a, 0x001a, 0x74e5,
	0x77e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x73e5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x75e5, 0x001a, 0x76e5, 0xd68e,
	0x001a, 0x78e5, 0x001a, 0x6092, 0x001a, 0x758c, 0x618a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7be5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5e8a, 0x001a, 0x81e5, 0x001a, 0x001a, 0x7ce5, 0x80e5,
	0x001a, 0x001a, 0x001a, 0x001a, 0xb894, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7de5, 0x001a, 0x001a, 0x7ee5, 0x6795, 0xd894, 0x82e5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfb91, 0x8ce5, 0x001a, 0x88e5, 0x001a, 0x001a, 0xe989, 0x001a,
	0x86e5, 0x001a, 0x4996, 0x87e5, 0x001a, 0x001a, 0x84e5, 0x001a,
	0x85e5, 0x8ae5, 0x8de5, 0x001a, 0x001a, 0x8be5, 0x001a, 0x001a,
	0x001a, 0x89e5, 0x83e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7792, 0x001a, 0x94e5, 0x001a, 0xa896, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x92e5, 0x001a, 0x001a,
	0x001a, 0x93e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8ee5, 0x001a, 0x001a, 0x90e5,
	0x001a, 0x001a, 0x001a, 0x91e5, 0x001a, 0x001a, 0x001a, 0x8fe5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe490, 0x001a, 0x5898, 0x98e5, 0x001a, 0x99e5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9fe5, 0x001a, 0x4990, 0x001a, 0x9be5,
	0x001a, 0x9ee5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x96e5,
	0x95e5, 0x001a, 0x001a, 0xa0e5, 0x001a, 0x001a, 0xda89, 0x001a,
	0x9ce5, 0x001a, 0xa1e5, 0x001a, 0x001a, 0x001a, 0x9de5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9ae5, 0x001a, 0xb192, 0x001a,
	0x97e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x8894,
	0x001a, 0x001a, 0xa5e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5a97, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa4e5,
	0x001a, 0x001a, 0xa3e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xace5, 0x001a, 0x001a, 0x001a, 0xa6e5,
	0x001a, 0x001a, 0x001a, 0xaee5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8697, 0xb1e5, 0x001a, 0xa8e5, 0x001a, 0x001a,
	0xa9e5, 0x001a, 0x001a, 0x001a, 0xade5, 0x001a, 0xb0e5, 0xafe5,
	0x001a, 0x001a, 0x001a, 0xa7e5, 0x001a, 0x001a, 0x001a, 0x001a,
	0xaae5, 0x001a, 0xbbe5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb4e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb2e5,
	0x001a, 0x001a, 0xb3e5, 0x001a, 0x001a, 0x001a, 0xb8e5, 0xb9e5,
	0x001a, 0x498a, 0x001a, 0x618b, 0x001a, 0x001a, 0xb7e5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa2e5, 0x001a, 0x85ee,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb6e5, 0xbae5, 0xb5e5,
	0x001a, 0xbce5, 0x001a, 0x001a, 0x001a, 0xbee5, 0xbde5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc0e5, 0xbfe5, 0x79e5, 0x001a, 0x001a, 0x001a, 0xc4e5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc1e5, 0x001a, 0x001a, 0x001a, 0x001a, 0xc2e5, 0x001a,
	0x001a, 0xc3e5, 0x001a, 0xc5e5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8c8c, 0x001a, 0xc7e5, 0x001a, 0xc6e5, 0x001a, 0x4f8f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x738d, 0xa59f, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc8e5, 0x708f, 0x001a, 0x001a, 0x001a, 0x588a,
	0x001a, 0xc9e5, 0x001a, 0x7189, 0x001a, 0xd58f, 0xcae5, 0x001a,
	0x001a, 0x748d, 0xcbe5, 0xdf88, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5c95, 0x001a, 0x001a, 0xcce5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8a90, 0x001a, 0xd3e5, 0x001a, 0x001a, 0xd0e5, 0x001a, 0x8f92,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd1e5, 0xcee5, 0xdc8b,
	0x001a, 0xcde5, 0xd4e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x558c, 0x001a, 0x001a, 0xdc91, 0x001a, 0xdae5, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd6e5, 0x001a, 0x001a, 0x001a, 0xb391, 0xd5e5,
	0x001a, 0xd8e5, 0x001a, 0x001a, 0x001a, 0x001a, 0xcfe5, 0x001a,
	0x001a, 0x001a, 0xd9e5, 0x001a, 0xdbe5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xed94, 0x001a, 0x001a, 0xd7e5, 0x001a,
	0xdce5, 0xdee5, 0x001a, 0x001a, 0xd18c, 0xd2e5, 0x001a, 0xbf88,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdde5,
	0x001a, 0xd98d, 0xf497, 0xdfe5, 0xe0e5, 0x9591, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa097,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe1e5, 0x5497, 0x001a, 0x001a,
	0xe2e5, 0xe3e5, 0x001a, 0x001a, 0xe295, 0xe4e5, 0x001a, 0xbe8d,
	0x001a, 0xa197, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe9e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xeae5, 0xd68f, 0xe8e5, 0x86ee, 0x001a, 0x001a,
	0x8797, 0xe5e5, 0x001a, 0x001a, 0xe7e5, 0xbb90, 0x9e90, 0x001a,
	0x001a, 0x001a, 0xe6e5, 0x001a, 0xebe5, 0x001a, 0x001a, 0xa195,
	0x001a, 0x001a, 0xede5, 0x001a, 0xece5, 0x001a, 0x001a, 0x001a,
	0x8c8a, 0x001a, 0x4a96, 0xeee5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x41ed, 0xfae5, 0xf0e5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf1e5, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf2e5, 0xf3e5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7e5, 0x001a,
	0xf8e5, 0x001a, 0x001a, 0xf6e5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf4e5, 0x001a, 0xefe5, 0xf5e5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf9e5, 0xb5e8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa689, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xfce5, 0xdd8b,
	0xfbe5, 0x001a, 0x001a, 0x001a, 0x41e6, 0x001a, 0x40e6, 0x001a,
	0x001a, 0x001a, 0x43e6, 0x001a, 0x001a, 0x42e6, 0x001a, 0x44e6,
	0x001a, 0x001a, 0x508f, 0x001a, 0x45e6, 0x001a, 0x001a, 0x46e6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x47e6, 0xbc90,
	0x001a, 0x7697, 0x001a, 0x48e6, 0x001a, 0x001a, 0xa295, 0x6594,
	0x49e6, 0x001a, 0x4ae6, 0xa98c, 0x001a, 0x001a, 0x001a, 0x4b8b,
	0x001a, 0x001a, 0x001a, 0x4be6, 0x001a, 0x001a, 0x8b8e, 0x6094,
	0x4ce6, 0x001a, 0x6f8a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4de6, 0x001a, 0x001a, 0x001a, 0x001a, 0x4fe6, 0x9797,
	0x001a, 0x4ee6, 0x6590, 0x001a, 0x50e6, 0x001a, 0x001a, 0x51e6,
	0x001a, 0x001a, 0x52e6, 0xcf8a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x53e6, 0x001a, 0x001a, 0x54e6, 0x001a, 0x55e6,
	0x56e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x708a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x57e6, 0x001a, 0x58e6, 0x59e6, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf089, 0x001a, 0x001a, 0x4790, 0x5ae6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5be6, 0x001a, 0x001a, 0x001a,
	0x5ce6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbe8c, 0x001a, 0xf992, 0x5de6, 0x001a, 0x001a, 0x001a, 0x001a,
	0x768c, 0x001a, 0x7590, 0x001a, 0x60e6, 0x001a, 0xa293, 0x001a,
	0x5fe6, 0x001a, 0x87ee, 0x508c, 0x001a, 0x001a, 0x5ee6, 0xf591,
	0x4c8b, 0x001a, 0x001a, 0x61e6, 0x001a, 0x62e6, 0x001a, 0xd78f,
	0x001a, 0x001a, 0x001a, 0x8d8c, 0x001a, 0x63e6, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4b96, 0x001a, 0x001a, 0xdd90, 0x001a, 0x001a,
	0x001a, 0x968b, 0x001a, 0xf396, 0x6991, 0x001a, 0x64e6, 0x88ee,
	0x001a, 0x001a, 0x6690, 0x9092, 0xd88f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x65e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x68e6, 0x001a,
	0x69e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbc8d, 0xc091, 0x67e6, 0x001a, 0xd98f, 0x5d95, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x66e6, 0x001a, 0x001a, 0x8c8e, 0x001a,
	0x7289, 0x001a, 0x6de6, 0x778c, 0x001a, 0x001a, 0x8e8e, 0x001a,
	0x001a, 0x8d8e, 0x001a, 0x6c98, 0x6ce6, 0x6be6, 0x4691, 0x001a,
	0x6c8b, 0x6298, 0x598a, 0xda8f, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x89ee, 0x001a, 0x001a, 0x6ae6, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6fe6, 0x001a, 0x70e6, 0x6ee6, 0x001a, 0xd68c,
	0x001a, 0x5f97, 0x001a, 0x001a, 0x8f8e, 0x4694, 0x001a, 0x001a,
	0x001a, 0x73e6, 0x001a, 0xbe90, 0x001a, 0x6192, 0x001a, 0x001a,
	0x5597, 0x001a, 0x76e6, 0x001a, 0x001a, 0x001a, 0xea8c, 0x001a,
	0xbd90, 0x72e6, 0x001a, 0x77e6, 0xeb8c, 0x74e6, 0x75e6, 0x8aee,
	0x71e6, 0x001a, 0x001a, 0x001a, 0xe090, 0xc793, 0x001a, 0x001a,
	0x4e92, 0x001a, 0xdb89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xee94, 0x001a, 0x001a, 0x628b, 0x001a, 0x8bee, 0xb292,
	0x001a, 0x001a, 0x7ae6, 0x001a, 0x78e6, 0x001a, 0x001a, 0x6b92,
	0x001a, 0x001a, 0x001a, 0xbf90, 0xd08a, 0x79e6, 0x001a, 0x7a90,
	0x001a, 0x001a, 0xc897, 0x001a, 0x001a, 0x001a, 0x5f98, 0x001a,
	0x001a, 0x001a, 0x7be6, 0x87e6, 0xb392, 0x001a, 0x86e6, 0x8cee,
	0x83e6, 0x8be6, 0x84e6, 0x001a, 0x80e6, 0x001a, 0xfa92, 0x7ee6,
	0x001a, 0x001a, 0x001a, 0x7ce6, 0x001a, 0x4097, 0x908e, 0x001a,
	0x001a, 0x81e6, 0x001a, 0x7de6, 0x001a, 0x001a, 0x8eee, 0x85e6,
	0x948f, 0x001a, 0xbf8c, 0x001a, 0x001a, 0x001a, 0xf891, 0x001a,
	0x6496, 0x7989, 0xe088, 0x001a, 0xa393, 0x001a, 0x001a, 0x89e6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x88e6, 0x001a, 0xe493, 0x001a,
	0x8de6, 0x001a, 0x001a, 0x001a, 0x82e6, 0x001a, 0x8ce6, 0x8ee6,
	0x001a, 0xaa8c, 0x8ae6, 0x758d, 0x001a, 0xd38e, 0x001a, 0x001a,
	0x8fe6, 0x7797, 0x001a, 0x001a, 0x001a, 0x001a, 0x92e6, 0x001a,
	0x95e6, 0x001a, 0x001a, 0x93e6, 0x5495, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x90e6, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xde8b, 0x001a, 0x001a, 0x001a, 0x001a, 0x94e6, 0x001a,
	0x001a, 0x96e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9ae6, 0x001a, 0x001a, 0x97e6, 0x001a, 0x99e6, 0x98e6,
	0x001a, 0x001a, 0x001a, 0x8fee, 0x001a, 0x001a, 0x9be6, 0x001a,
	0xaf8e, 0x001a, 0x9de6, 0x9ce6, 0x8895, 0x001a, 0x001a, 0x9fe6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x788c, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9ee6, 0xa0e6, 0x001a, 0x001a, 0xa1e6,
	0x638b, 0xbfe3, 0xf78f, 0x001a, 0xa2e6, 0x001a, 0x001a, 0xec8c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa3e6, 0x001a, 0x90ee,
	0xa4e6, 0x001a, 0x001a, 0x5d8e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcc9d, 0x001a, 0xa5e6, 0x001a, 0xa6e6, 0x001a,
	0x518f, 0x001a, 0xa7e6, 0xa8e6, 0x001a, 0x001a, 0xa9e6, 0x001a,
	0x001a, 0xaae6, 0xabe6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4a92,
	0x001a, 0x001a, 0xace6, 0x001a, 0x001a, 0x001a, 0x001a, 0xaee6,
	0x001a, 0xade6, 0x001a, 0x001a, 0x001a, 0x001a, 0xa493, 0x001a,
	0xafe6, 0x001a, 0x4c96, 0x001a, 0xb0e6, 0x001a, 0xb1e6, 0x001a,
	0xb2e6, 0x001a, 0x001a, 0x001a, 0x001a, 0xb3e6, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd893, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdb8f, 0xb4e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8b8d, 0xac98, 0xb5e6, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb6e6, 0x5e95, 0xb7e6, 0x001a, 0xbfe6, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb8e6, 0x001a, 0x001a, 0xbae6, 0x001a, 0x001a,
	0x001a, 0xb9e6, 0xbbe6, 0x001a, 0x6596, 0xbce6, 0xbde6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbee6, 0x001a, 0x001a, 0x001a,
	0xc0e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x4c8a, 0xe592, 0x001a,
	0x8995, 0xe08d, 0x768d, 0x001a, 0x001a, 0x001a, 0x001a, 0x6e95,
	0xdd89, 0xcc94, 0xc3e6, 0xd18a, 0xd390, 0xc2e6, 0xc7e6, 0x9992,
	0xe196, 0x001a, 0xc5e6, 0xc6e6, 0x4d8b, 0x001a, 0xc8e6, 0x8394,
	0xdd91, 0x001a, 0x001a, 0xef94, 0x5c93, 0xc4e6, 0x001a, 0x6696,
	0xea89, 0xcae6, 0x4798, 0xc092, 0x6498, 0x001a, 0x001a, 0x918e,
	0xc9e6, 0x001a, 0xaf91, 0x001a, 0x001a, 0xdae6, 0x4791, 0x001a,
	0x001a, 0xf693, 0x001a, 0x6f95, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcde6, 0x5e8e, 0x928e, 0x001a, 0xdc8f, 0x001a,
	0x8594, 0x001a, 0xab8c, 0xcce6, 0xcbe6, 0x001a, 0x8a95, 0x001a,
	0x001a, 0x001a, 0xbf8e, 0x001a, 0x001a, 0x7193, 0x001a, 0x001a,
	0x91ee, 0x001a, 0x001a, 0x001a, 0x92ee, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcfe6, 0xd0e6, 0x778d, 0xcee6, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd1e6, 0xd2e6, 0x001a, 0xd4e6,
	0xa191, 0x001a, 0xd3e6, 0xe48a, 0x001a, 0xd6e6, 0x001a, 0xd5e6,
	0xd7e6, 0x001a, 0x93ee, 0xd9e6, 0xdbe6, 0x001a, 0xdce6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd490, 0x001a, 0xcd8e, 0xdde6,
	0x001a, 0x001a, 0x001a, 0x718a, 0x001a, 0xdee6, 0x001a, 0x001a,
	0x9691, 0xdfe6, 0x001a, 0xe0e6, 0x8b95, 0x001a, 0x94ee, 0x4e8b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe1e6, 0x001a, 0x001a, 0x001a, 0xb492, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7a89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe2e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xef8e, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9690, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xab91, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe5e6, 0x001a, 0x001a, 0x001a, 0xe4e6, 0x001a,
	0x001a, 0x001a, 0xe3e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xebe6, 0xe9e6, 0x001a, 0x001a, 0xe6e6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe8e6, 0x001a,
	0x001a, 0x001a, 0xe7e6, 0xeae6, 0x001a, 0x978b, 0x001a, 0xeee6,
	0x001a, 0xd590, 0x001a, 0xefe6, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd78c, 0x001a, 0xece6, 0xede6, 0x001a, 0x001a, 0x001a, 0x4898,
	0x001a, 0x001a, 0x001a, 0xb592, 0x001a, 0x4891, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf0e6, 0x001a, 0x001a, 0xf3e6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf1e6, 0xf2e6, 0x7897, 0x001a, 0x001a, 0x001a, 0x001a, 0xa593,
	0xf6e6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4e6, 0xf5e6, 0xf7e6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x48e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfae6, 0x001a, 0x001a, 0x001a, 0xfbe6, 0xf9e6, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf8e6, 0x001a, 0xfb92, 0x001a, 0x001a, 0x40e7,
	0x44e7, 0x41e7, 0xfce6, 0x001a, 0x42e7, 0x001a, 0x001a, 0x001a,
	0x43e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ae7, 0x001a, 0x001a,
	0x001a, 0x45e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd690,
	0x47e7, 0x001a, 0x001a, 0x49e7, 0x46e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4ce7, 0x001a, 0x528f, 0x001a, 0x4be7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4de7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4ee7, 0x001a, 0x001a, 0x51e7, 0x50e7, 0x001a, 0x4fe7,
	0x001a, 0x001a, 0x53e7, 0x52e7, 0x001a, 0xf496, 0x001a, 0x001a,
	0x001a, 0x55e7, 0x001a, 0x54e7, 0x56e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x57e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x59e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x58e7, 0x6790, 0x5ae7, 0x001a, 0x001a, 0xeb8b,
	0x5be7, 0x5de7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5ee7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5fe7, 0x5ce7, 0x001a,
	0x60e7, 0x001a, 0xd48e, 0x61e7, 0x4f8b, 0x528c, 0x001a, 0x96ee,
	0x001a, 0x001a, 0xac8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x62e7, 0x001a, 0x001a, 0x001a, 0xee93,
	0x001a, 0x001a, 0x5d93, 0x63e7, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x66e7, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb28e, 0x001a, 0x001a, 0x65e7, 0x64e7, 0x798c, 0x67e7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x728a, 0x001a, 0x69e7, 0x001a, 0x001a,
	0x001a, 0xda8d, 0x68e7, 0x001a, 0x71e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6be7, 0x6de7, 0xe395, 0x6ae7, 0x001a, 0x001a,
	0x001a, 0x6ce7, 0x001a, 0x70e7, 0x6ee7, 0x508b, 0x001a, 0x6fe7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x72e7, 0x001a,
	0x001a, 0x7994, 0xd697, 0x001a, 0x001a, 0x001a, 0x001a, 0x538f,
	0x001a, 0x001a, 0x001a, 0x73e7, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4197, 0x75e7, 0x001a, 0x74e7, 0x001a, 0x001a, 0x78e7, 0x6097,
	0x001a, 0x001a, 0x77e7, 0x001a, 0x8d8a, 0x76e7, 0x7be7, 0x001a,
	0x001a, 0x7ae7, 0x001a, 0x001a, 0x79e7, 0x5193, 0x7ce7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7de7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7ee7, 0x001a, 0x001a, 0x8c8d,
	0x001a, 0x448c, 0x80e7, 0x81e7, 0x82e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6890, 0x83e7, 0x001a, 0xab8e, 0x84e7,
	0x001a, 0x001a, 0x001a, 0x85e7, 0x001a, 0x001a, 0x001a, 0x9f99,
	0x9e99, 0x001a, 0x001a, 0x001a, 0x001a, 0x86e7, 0x90e3, 0x87e7,
	0x4392, 0x4a90, 0x5f94, 0x001a, 0x001a, 0x001a, 0x001a, 0x88e7,
	0x001a, 0x001a, 0xd395, 0xd292, 0x9e8d, 0x001a, 0x001a, 0x4892,
	0x001a, 0x001a, 0x4989, 0x001a, 0x9896, 0x7690, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7d8c, 0x001a,
	0x001a, 0xdf8b, 0x001a, 0x001a, 0xd495, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x89e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8be7, 0x001a, 0x001a, 0x8ae7, 0xde89, 0x001a,
	0x001a, 0xf493, 0x8ce7, 0x9794, 0x001a, 0x5293, 0x001a, 0x8de7,
	0x718f, 0x001a, 0x001a, 0x001a, 0x8fe7, 0x001a, 0x001a, 0xc096,
	0x9ee7, 0x91e7, 0x92e7, 0x001a, 0x001a, 0xc792, 0x001a, 0x001a,
	0xde91, 0x9791, 0x001a, 0xa693, 0x001a, 0x90e7, 0x748b, 0x001a,
	0x001a, 0x001a, 0x001a, 0x99e7, 0x001a, 0x96e7, 0xa3e7, 0xa793,
	0x8092, 0x93e7, 0x001a, 0xfc92, 0x7293, 0x94e7, 0x98e7, 0x8090,
	0x001a, 0x8794, 0xca92, 0x001a, 0x001a, 0xc090, 0x97e7, 0xac91,
	0xa291, 0x95e7, 0xa788, 0x4198, 0x001a, 0x001a, 0x001a, 0x9ae7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdf91, 0x001a,
	0x001a, 0x548f, 0x6990, 0x001a, 0x001a, 0x9ce7, 0x9be7, 0x001a,
	0xed88, 0x9de7, 0x001a, 0x001a, 0x4e95, 0x001a, 0xa5e7, 0x001a,
	0x001a, 0xd993, 0x8b90, 0x001a, 0x001a, 0x7892, 0x001a, 0xf68b,
	0x001a, 0xa4e7, 0x5697, 0x5e89, 0x001a, 0xd595, 0xdf89, 0x9fe7,
	0xa0e7, 0xa1e7, 0xa2e7, 0xb993, 0x4292, 0xe188, 0xa6e7, 0x001a,
	0xa7e7, 0xa1ea, 0x001a, 0x001a, 0xbb91, 0x001a, 0xa8e7, 0x001a,
	0x9389, 0x6b91, 0x001a, 0xad8c, 0x001a, 0x7997, 0x001a, 0x99ee,
	0xa9e7, 0x4b93, 0x001a, 0x001a, 0x001a, 0x9891, 0xd58e, 0xaae7,
	0x001a, 0x001a, 0xade7, 0x001a, 0x001a, 0x858f, 0xabe7, 0x4a91,
	0x4991, 0x001a, 0xe288, 0x001a, 0xc997, 0xafe7, 0x001a, 0xf094,
	0xb1e7, 0xb0e7, 0xaee7, 0x84e2, 0xd28a, 0x001a, 0x001a, 0x8ee7,
	0x001a, 0xb3e7, 0xb2e7, 0x001a, 0x001a, 0x001a, 0x001a, 0xb4e7,
	0x001a, 0x5797, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdf93, 0x001a, 0x001a, 0x4d96, 0x001a,
	0xb5e7, 0x001a, 0xd78e, 0x001a, 0x001a, 0x001a, 0x001a, 0xb6e7,
	0x001a, 0xb7e7, 0x001a, 0x001a, 0x001a, 0xb8e7, 0x001a, 0x001a,
	0x4093, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe888, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x788d, 0x001a, 0x001a, 0x001a, 0x5998, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xbce7, 0x001a, 0x001a, 0x9aee, 0x001a,
	0x001a, 0x538c, 0xb9e7, 0x001a, 0xbae7, 0x001a, 0x001a, 0x001a,
	0x9495, 0x001a, 0x001a, 0x001a, 0x001a, 0x738a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5897, 0x001a, 0xbd8b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7393, 0x001a, 0x001a,
	0x001a, 0x001a, 0xbde7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xbee7, 0x001a, 0x001a, 0x9cee, 0x001a, 0x001a,
	0x001a, 0xbfe7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9dee,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4193, 0x001a, 0x001a,
	0xc1e7, 0x001a, 0xc0e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd193, 0xc2e7, 0x558f, 0xde8e, 0x7a94, 0x9192, 0x001a,
	0x001a, 0x001a, 0xf08e, 0x001a, 0x8c90, 0x001a, 0xc3e7, 0x001a,
	0xc4e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7c90, 0xc5e7, 0x001a, 0xc6e7, 0x001a, 0x001a,
	0x001a, 0xc7e7, 0x8f97, 0x001a, 0x568f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc9e7, 0xc8e7, 0x001a, 0x798d, 0x001a, 0x938d,
	0x5f8e, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcce7, 0x001a, 0x001a, 0x001a, 0x001a, 0x868f,
	0x001a, 0xcbe7, 0x001a, 0xcae7, 0x001a, 0xe791, 0x001a, 0x001a,
	0xed8c, 0x001a, 0xc190, 0x001a, 0x001a, 0x001a, 0x001a, 0xae94,
	0x001a, 0x001a, 0x001a, 0x001a, 0x588f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcde7, 0x001a, 0xdd8f, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd0e7, 0xcee7, 0x001a, 0x001a, 0x001a, 0xcfe7,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd2e7, 0xd1e7, 0x001a, 0x001a,
	0xf88f, 0x001a, 0xd3e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd4e7, 0xd5e7, 0x001a, 0x001a, 0x001a, 0x001a, 0xce94, 0xd18d,
	0xdf8e, 0xd6e7, 0x001a, 0xd7e7, 0xa297, 0x648f, 0xec96, 0xca97,
	0xd8e7, 0xe08b, 0x001a, 0x001a, 0x001a, 0x001a, 0xd9e7, 0x9fee,
	0x4293, 0x001a, 0x9eee, 0xdce7, 0x988a, 0x6a90, 0xa0ee, 0xdae7,
	0x001a, 0xdbe7, 0x001a, 0xde92, 0xa3ee, 0xa4ee, 0x7496, 0xfa8b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa1ee, 0xa2ee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdee7, 0xdfe7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xdde7, 0x001a, 0x001a, 0xe1e7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa5ee, 0x001a,
	0x001a, 0x001a, 0xa7ee, 0x001a, 0x001a, 0xdd93, 0x628a, 0x001a,
	0xa6ee, 0xe5e7, 0x001a, 0x001a, 0xe2e7, 0xe4e7, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe0e7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6ee8, 0x001a, 0x001a, 0xe3e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe997, 0x001a, 0x001a, 0xd88c,
	0x001a, 0xaeee, 0xa8ee, 0x001a, 0xaaee, 0x001a, 0x001a, 0xede7,
	0xa9ee, 0x001a, 0x001a, 0x001a, 0x5393, 0xe8e7, 0x001a, 0x001a,
	0xebe7, 0xe9e7, 0x001a, 0xeee7, 0x001a, 0x001a, 0xabee, 0x001a,
	0xefe7, 0xadee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe7e7,
	0x001a, 0xacee, 0xf4e7, 0x9489, 0x001a, 0x001a, 0xe6e7, 0x001a,
	0x001a, 0x001a, 0xab94, 0x001a, 0xeae7, 0x001a, 0xde8f, 0xafee,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7a8d, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb1ee,
	0xb2ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6796, 0x001a,
	0xe28b, 0x001a, 0x001a, 0x658f, 0x001a, 0xba93, 0x001a, 0x001a,
	0x43ed, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4c91, 0x001a, 0xf2e7, 0x001a, 0xece7, 0xf1e7, 0x001a,
	0xc196, 0x001a, 0xb692, 0xf3e7, 0xf0e7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb0ee,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4b91, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7e7,
	0x001a, 0xf6e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf5e7,
	0xb6ee, 0x001a, 0x4e96, 0xbaee, 0x001a, 0xb8ee, 0x001a, 0xb4ee,
	0x001a, 0xb5ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb9ee, 0x001a, 0x001a, 0x001a, 0x9b8f, 0x001a, 0x001a, 0xb3ee,
	0x001a, 0xf8e7, 0xdd95, 0x001a, 0x001a, 0x7389, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6595, 0x9292, 0x001a, 0x001a, 0x001a, 0x001a,
	0x988b, 0x49ed, 0xfae7, 0xbdee, 0x7c8d, 0x001a, 0x001a, 0xc0ee,
	0x001a, 0x001a, 0xc2ee, 0x001a, 0x001a, 0x001a, 0x4b8e, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf9e7,
	0x8d90, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x8e90, 0x40e8, 0x42e8, 0x001a, 0x001a, 0xc1ee, 0xbfee, 0x001a,
	0xf98f, 0xbcee, 0x41e8, 0x43e8, 0x001a, 0xbbee, 0xd18b, 0x001a,
	0x6495, 0x001a, 0x001a, 0xe08e, 0x4298, 0x001a, 0xfce7, 0xf68d,
	0x001a, 0x001a, 0x5e98, 0x001a, 0x001a, 0x45e8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x44e8, 0x46e8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfbe7, 0x001a, 0x001a, 0x001a,
	0x42ed, 0x001a, 0x001a, 0xe793, 0x001a, 0x7493, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd592, 0x001a, 0x4be8, 0xc4ee,
	0x001a, 0x001a, 0x001a, 0x6292, 0x47e8, 0x001a, 0x001a, 0x001a,
	0x48e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4c8c, 0x001a, 0x4ae8, 0x001a,
	0xc3ee, 0x001a, 0x001a, 0x001a, 0x001a, 0xae8c, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x49e8, 0x001a, 0xdf8f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x998a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4fe8, 0x001a, 0xbd8d, 0x9991,
	0x001a, 0x001a, 0xc892, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc5ee, 0x001a, 0x001a, 0x5a8a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4de8, 0x4ee8, 0xc192, 0x001a,
	0x4ce8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x50e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x56e8, 0x001a, 0x001a, 0xc6ee, 0x001a,
	0x59e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x58e8, 0x4c93, 0x001a, 0x001a, 0x001a, 0x001a, 0x51e8, 0x52e8,
	0x55e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x57e8, 0xc7ee, 0x001a,
	0x001a, 0xbe8b, 0x001a, 0x001a, 0x5ae8, 0x54e8, 0x001a, 0x001a,
	0x53e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc8ee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5ee8, 0x001a, 0x001a, 0x001a, 0x5fe8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x60e8, 0x001a, 0x001a, 0x5de8, 0x5ce8, 0x001a, 0x001a, 0x001a,
	0xe08f, 0xa893, 0x5be8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x64e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x62e8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc9ee, 0x001a, 0x001a, 0x001a, 0x63e8, 0x61e8, 0x001a,
	0xf691, 0x001a, 0x65e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x66e8, 0x001a, 0x001a, 0x68e8, 0xcaee, 0x001a, 0x001a,
	0xcbee, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd38a, 0x67e8, 0xf896, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x73e8, 0x69e8, 0x001a, 0x001a, 0x6ce8, 0x001a,
	0x6ae8, 0x001a, 0x6be8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6de8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6fe8, 0x001a, 0x001a, 0x001a, 0x001a, 0x70e8, 0x001a, 0x71e8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x74e8, 0x72e8, 0x75e8, 0x77e8,
	0x001a, 0x76e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb792,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe596, 0x001a, 0x78e8, 0x4d91, 0x001a, 0x001a, 0x001a, 0x79e8,
	0x001a, 0xc295, 0x7ae8, 0x4a8a, 0x001a, 0x001a, 0x001a, 0x5b89,
	0x001a, 0xd58a, 0xccee, 0xd48a, 0x7be8, 0x001a, 0x7ce8, 0x001a,
	0x7de8, 0x7ee8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x80e8, 0x001a, 0xd68a, 0x748a, 0x7d8d, 0xb494, 0x001a, 0x82e8,
	0x81e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x83e8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x7b89, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x86e8, 0x001a, 0x85e8, 0x84e8, 0x001a, 0x87e8, 0x001a,
	0x001a, 0x001a, 0x001a, 0x8ae8, 0x001a, 0x001a, 0x001a, 0xc588,
	0x001a, 0x001a, 0x88e8, 0x001a, 0x8ce8, 0x8be8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8ee8, 0x8de8, 0x8fe8, 0x001a,
	0xac93, 0x001a, 0x001a, 0x001a, 0x90e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x91e8, 0x93e8, 0x001a, 0x001a, 0x92e8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8c95, 0x001a, 0x001a, 0x001a,
	0x001a, 0x94e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x95e8, 0x001a, 0xe38d, 0x001a, 0x001a, 0x001a, 0x96e8, 0x97e8,
	0x001a, 0x001a, 0x6896, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6a91, 0x001a, 0x001a, 0x001a, 0xa288,
	0xc991, 0x001a, 0x98e8, 0x001a, 0x8d95, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9be8, 0x99e8, 0x7e8d, 0x001a, 0x9ae8,
	0xc08c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc395, 0x9de8, 0x9fe8, 0x9ee8, 0xa0e8,
	0x001a, 0x001a, 0x4089, 0x7790, 0x9c8f, 0xd78a, 0xa1e8, 0x001a,
	0x001a, 0x001a, 0x8694, 0x001a, 0xa3e8, 0x001a, 0x001a, 0x001a,
	0x4189, 0x001a, 0xa2e8, 0xc292, 0x001a, 0xcb97, 0xa993, 0x9ce8,
	0xa497, 0x001a, 0xaf8c, 0x001a, 0x001a, 0x7a97, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf78b, 0xb297, 0x001a,
	0x478c, 0x001a, 0xe091, 0x40e4, 0x001a, 0xa4e8, 0x4b8a, 0x8f90,
	0x001a, 0x001a, 0x001a, 0x001a, 0x758a, 0xa6e8, 0x001a, 0xa7e8,
	0xa5e8, 0x848c, 0x001a, 0xdb8d, 0xe18f, 0xcfee, 0x001a, 0x001a,
	0x4289, 0x001a, 0x001a, 0xd797, 0x001a, 0x001a, 0x001a, 0xa9e8,
	0xace7, 0x001a, 0xa8e8, 0x001a, 0x001a, 0x001a, 0x001a, 0xd0ee,
	0xace8, 0xaae8, 0xabe8, 0x001a, 0xade8, 0x001a, 0xaee8, 0xea97,
	0xafe8, 0xb0e8, 0x001a, 0xc790, 0xb994, 0x001a, 0x001a, 0x001a,
	0x9d90, 0xe58a, 0x001a, 0x001a, 0x5997, 0xeb89, 0x578f, 0xd98c,
	0x001a, 0xb3e8, 0x001a, 0xb2e8, 0x938e, 0xb4e8, 0xb1e8, 0x001a,
	0x001a, 0x478e, 0x001a, 0x001a, 0x001a, 0xb8e8, 0xabe5, 0x001a,
	0x001a, 0xd499, 0x001a, 0x9790, 0xb6e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa397, 0xef93, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4a89, 0x001a, 0xe190, 0xb48e, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb595, 0x001a, 0x5f89, 0x001a, 0x001a, 0x001a, 0xeb97, 0x8b97,
	0x001a, 0xb9e8, 0x001a, 0x6493, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf98e, 0x001a, 0x001a, 0x001a, 0xbae8, 0x001a, 0xbbe8, 0x6b90,
	0xbce8, 0x001a, 0xec97, 0x001a, 0x001a, 0xb7e8, 0xbee8, 0xc0e8,
	0x001a, 0xbfe8, 0x001a, 0xbde8, 0x001a, 0x001a, 0xc1e8, 0x001a,
	0x001a, 0xc2e8, 0x001a, 0x001a, 0x9a91, 0x001a, 0xe089, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc3e8, 0x001a, 0x001a, 0xb696,
	0x001a, 0x001a, 0xc4e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc5e8, 0x001a, 0x4998, 0xd1ee, 0x001a, 0x001a, 0x001a, 0x001a,
	0x509e, 0xc6e8, 0x001a, 0xd2ee, 0x001a, 0xc7e8, 0xc8e8, 0x001a,
	0x001a, 0x001a, 0xcce8, 0xd3ee, 0xc9e8, 0x001a, 0xcae8, 0x001a,
	0xcbe8, 0xcde8, 0x001a, 0x001a, 0x001a, 0xd4ee, 0x001a, 0xd5ee,
	0x001a, 0xd6ee, 0xc290, 0x001a, 0x001a, 0xd7ee, 0xf596, 0x001a,
	0x001a, 0xc390, 0x001a, 0x001a, 0xcee8, 0x001a, 0xf194, 0x001a,
	0xcfe8, 0x72ea, 0xca96, 0x001a, 0xd0e8, 0x001a, 0xd1e8, 0x001a,
	0xd2e8, 0x768a, 0x001a, 0xd4e8, 0x001a, 0x7890, 0x001a, 0x001a,
	0x001a, 0xd5e8, 0x001a, 0x001a, 0x438c, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd6e8, 0xdae8, 0x001a, 0xd8e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd9e8, 0x001a, 0x001a, 0x938a, 0xd7e8, 0xdbe8, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdce8, 0x001a, 0xc688, 0x001a, 0xdde8,
	0xdee8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe28f, 0x001a, 0x001a, 0x001a, 0xdfe8, 0x001a, 0x001a, 0x001a,
	0x668b, 0x001a, 0x001a, 0xe2e8, 0x001a, 0x001a, 0xe1e8, 0x001a,
	0xe0e8, 0x001a, 0x001a, 0x91e6, 0x001a, 0xda95, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe3e8, 0xe4e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe5e8, 0x001a, 0x001a, 0xe6e8, 0x001a,
	0xe7e8, 0x001a, 0x001a, 0xe8e8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd88a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe9e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xeae8, 0x4294, 0x001a,
	0x001a, 0x001a, 0xece8, 0xb989, 0x001a, 0xefe8, 0xeee8, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4389, 0x001a, 0x001a, 0x001a, 0xbf8b,
	0x001a, 0xc595, 0xb892, 0xa08d, 0x001a, 0x808d, 0x878f, 0x001a,
	0x7b90, 0x001a, 0x001a, 0x001a, 0xf1e8, 0x001a, 0x001a, 0xf0e8,
	0x6197, 0xe68a, 0xd094, 0xda93, 0x001a, 0x001a, 0x001a, 0x9c90,
	0xcc97, 0x001a, 0x7a8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf4e8, 0x001a, 0x001a, 0xf3e8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6a96, 0xaa93, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6f89, 0x001a, 0x001a, 0xf5e8,
	0xf2e8, 0x001a, 0x001a, 0x7095, 0x8a97, 0xf6e8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf7e8, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf9e8, 0xe891, 0x7a8a, 0x7b8a, 0xf8e8,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe78a, 0xb08c, 0x001a, 0xd8ee,
	0xe88a, 0x001a, 0x001a, 0x5e93, 0x001a, 0x001a, 0xde97, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd9ee, 0x001a, 0xda8c,
	0x001a, 0x001a, 0x001a, 0xfae8, 0x001a, 0x001a, 0x001a, 0xfbe8,
	0xfce8, 0x40e9, 0x001a, 0x42e9, 0x41e9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9795, 0x001a, 0x43e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x44e9,
	0x001a, 0x45e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x46e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x48e9, 0x47e9, 0x001a, 0x49e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf294, 0xcae3, 0x001a, 0x001a, 0x4890,
	0x001a, 0x001a, 0x518b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4ae9, 0x001a, 0x4be9, 0x001a, 0xaa99, 0x5a9f, 0xd194,
	0x001a, 0x001a, 0xf988, 0x001a, 0xb988, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x948e, 0x4f96, 0xfc8f, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4ce9, 0x001a, 0xdd96, 0x001a, 0x001a,
	0x001a, 0x4de9, 0x7b97, 0x001a, 0x6189, 0x001a, 0x001a, 0x001a,
	0x608e, 0x001a, 0x4ee9, 0xec89, 0x4fe9, 0x001a, 0x001a, 0x001a,
	0x50e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x52e9, 0x53e9, 0x001a,
	0x55e9, 0x51e9, 0x001a, 0x001a, 0x54e9, 0x001a, 0x001a, 0xdcee,
	0xd98a, 0x001a, 0x001a, 0x001a, 0x56e9, 0x001a, 0x57e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x58e9, 0x59e9, 0x001a,
	0x001a, 0x001a, 0x5ae9, 0x001a, 0x001a, 0x5ce9, 0x001a, 0x001a,
	0x001a, 0x5be9, 0x001a, 0x5ee9, 0x61e9, 0x001a, 0x001a, 0x001a,
	0x5de9, 0x5fe9, 0x60e9, 0x001a, 0x001a, 0x62e9, 0x001a, 0xc08b,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf18e, 0x63e9,
	0x64e9, 0x818d, 0x001a, 0x001a, 0x001a, 0x001a, 0xdeee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x65e9, 0x001a, 0x001a,
	0x5d8a, 0x001a, 0x001a, 0x001a, 0x6e94, 0x66e9, 0x67e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x7992, 0xe993, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x68e9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x9d94, 0x001a, 0x001a, 0xca91, 0x7789, 0xec8b, 0x001a,
	0xed8b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9392, 0x6de9, 0xee8b, 0x001a, 0x001a, 0xed89, 0x001a, 0x001a,
	0x6ce9, 0x001a, 0x001a, 0x6ae9, 0x001a, 0x6be9, 0x001a, 0x69e9,
	0x001a, 0x001a, 0x77e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6ee9, 0x6fe9, 0x001a,
	0x001a, 0x70e9, 0x71e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x73e9, 0x001a, 0x001a, 0x72e9, 0x001a, 0x001a, 0x001a, 0x788f,
	0x001a, 0x74e9, 0x001a, 0x001a, 0x001a, 0x76e9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x528b, 0x75e9,
	0x001a, 0x001a, 0x9b91, 0xb18c, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x78e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcb91, 0x001a, 0x001a, 0x79e9, 0x001a, 0x001a, 0x001a, 0x001a,
	0xab93, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7ae9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x80e9, 0x001a,
	0x7de9, 0x001a, 0x7ce9, 0x7ee9, 0x001a, 0x7be9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x82e9, 0xdfee, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x81e9, 0x001a, 0x84e9,
	0x001a, 0x001a, 0xc18b, 0x83e9, 0x001a, 0x001a, 0x001a, 0x85e9,
	0x001a, 0x001a, 0x86e9, 0x001a, 0x88e9, 0x87e9, 0x001a, 0x001a,
	0x001a, 0x89e9, 0x8be9, 0x8ae9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9c8d, 0x001a, 0x001a, 0x001a, 0x001a, 0x8ce9, 0x001a, 0x001a,
	0x8de9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5b8a, 0x001a, 0x001a, 0x001a, 0x8ee9, 0x001a, 0x001a, 0x001a,
	0x8fe9, 0x001a, 0x001a, 0x001a, 0x9190, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x90e9,
	0x001a, 0x91e9, 0x001a, 0x92e9, 0x93e9, 0x001a, 0x001a, 0x001a,
	0x828d, 0xe0ee, 0x001a, 0x001a, 0xe1ee, 0x001a, 0x94e9, 0x95e9,
	0x001a, 0x001a, 0x96e9, 0x97e9, 0x001a, 0x001a, 0x98e9, 0x001a,
	0x001a, 0x001a, 0xaf94, 0x9ae9, 0x001a, 0x4595, 0x9be9, 0x99e9,
	0x001a, 0x9de9, 0x001a, 0x001a, 0x9ce9, 0x001a, 0x001a, 0x9ee9,
	0x001a, 0x001a, 0x001a, 0x9fe9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa0e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa1e9, 0x001a, 0xa2e9, 0x001a, 0x001a, 0x001a, 0x001a, 0xa3e9,
	0x001a, 0x001a, 0xa4e9, 0xa5e9, 0x001a, 0xa6e9, 0x001a, 0xa7e9,
	0xa8e9, 0xa9e9, 0xaae9, 0x001a, 0x001a, 0x001a, 0xabe9, 0xace9,
	0x001a, 0x549f, 0xade9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf6e2, 0x538b, 0x001a, 0x001a, 0x001a,
	0x001a, 0x408a, 0xb08d, 0xafe9, 0xaee9, 0xa396, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb1e9, 0xb2e9, 0xb0e9,
	0x001a, 0xb3e9, 0x001a, 0x001a, 0x8296, 0x001a, 0x001a, 0x001a,
	0xb4e9, 0x001a, 0x9b8b, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4498,
	0x001a, 0x001a, 0xe3ee, 0x001a, 0xb5e9, 0xe2ee, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb7e9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbc88, 0xe4ee,
	0x001a, 0xb8e9, 0xa995, 0xb6e9, 0x001a, 0x001a, 0xb9e9, 0xbae9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbbe9,
	0xbce9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbde9, 0x001a, 0x8e96, 0x4c8e, 0x001a, 0xf88d, 0x4e91, 0x001a,
	0x001a, 0xe5ee, 0x001a, 0x001a, 0xbee9, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc1e9, 0x001a, 0xe6ee, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbfe9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc2e9, 0x001a,
	0x001a, 0xef8c, 0xc0e9, 0x001a, 0x001a, 0x001a, 0x001a, 0xc3e9,
	0x001a, 0xc4e9, 0xc5e9, 0x001a, 0xc9e9, 0x001a, 0x498e, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe291, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcae9, 0xc7e9, 0xc6e9, 0xc8e9, 0x001a, 0x001a, 0x001a,
	0x7e8c, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcee9, 0xcde9, 0xcce9, 0x001a, 0x001a, 0xb188, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe7ee, 0x001a, 0x001a, 0x001a, 0xd8e9, 0x001a, 0xd4e9, 0x001a,
	0xd5e9, 0xd1e9, 0xd7e9, 0x001a, 0xd3e9, 0x828a, 0x001a, 0x001a,
	0x6b98, 0x001a, 0xd6e9, 0xd2e9, 0xd0e9, 0xcfe9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdae9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdde9, 0x001a, 0x001a, 0xdce9, 0xdbe9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6895, 0xd9e9, 0xf188,
	0xdee9, 0x001a, 0xe0e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x8f8a, 0xcbe9, 0x5689, 0x001a, 0x001a, 0xe2e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe1e9, 0xdfe9,
	0x4c92, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x9096, 0x001a, 0x001a, 0x001a, 0x001a, 0xd897,
	0x001a, 0x001a, 0xe3e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe4e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe5e9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe6e9, 0x001a,
	0xe7e9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb992, 0x001a, 0xe8e9,
	0x001a, 0xb594, 0x001a, 0xede9, 0xe9e9, 0x001a, 0x001a, 0x001a,
	0xeae9, 0x001a, 0x001a, 0x5096, 0xc296, 0x001a, 0xce93, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xeee9, 0x001a, 0x001a, 0xefe9, 0xbc93,
	0xece9, 0xebe9, 0x001a, 0x001a, 0x001a, 0x001a, 0xa889, 0x001a,
	0x001a, 0x001a, 0xf7e9, 0x001a, 0x001a, 0xf6e9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x9589, 0x001a, 0x001a, 0x001a, 0xf4e9,
	0x001a, 0x001a, 0x001a, 0xf3e9, 0x001a, 0x001a, 0xf1e9, 0x001a,
	0x9b8a, 0x001a, 0xf0e9, 0xb08e, 0xa789, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x838d, 0x001a, 0x001a, 0xfae9, 0xf9e9,
	0x001a, 0xf8e9, 0x001a, 0x001a, 0xf5e9, 0x001a, 0xfbe9, 0x001a,
	0xfce9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x44ea, 0x43ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x45ea, 0x001a, 0x001a, 0x4c89, 0x40ea, 0x41ea, 0x001a,
	0x948d, 0xb796, 0x001a, 0x001a, 0x42ea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe9ee, 0x5196, 0x001a, 0x001a, 0x4aea,
	0xe8ee, 0x001a, 0x46ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4bea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x48ea,
	0x001a, 0x47ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7b8c,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4cea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4dea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x4eea, 0x001a, 0x49ea, 0x001a, 0x001a, 0x001a, 0xf2e9,
	0x001a, 0x001a, 0x4fea, 0x001a, 0xdf92, 0x001a, 0x001a, 0x001a,
	0x53ea, 0x001a, 0x54ea, 0x52ea, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x51ea, 0x57ea, 0x001a, 0x50ea, 0x001a, 0x55ea, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x56ea,
	0x001a, 0x001a, 0x001a, 0x59ea, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x58ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5bea, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5cea, 0x001a, 0x5dea,
	0x001a, 0x001a, 0x6898, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5aea, 0xe991, 0xeb8d, 0x001a, 0x001a, 0x5eea, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xebee, 0x5fea, 0x60ea, 0x001a, 0x001a, 0x61ea, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x62ea, 0x001a, 0x001a,
	0xb28c, 0x63ea, 0x001a, 0x001a, 0x001a, 0x64ea, 0x001a, 0xad8e,
	0x001a, 0x65ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x66ea, 0x001a, 0x001a, 0x67ea, 0x68ea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x6bea, 0x69ea, 0x5b98, 0x001a, 0x6aea, 0x001a, 0xed97,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6cea, 0x001a, 0xd997,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6dea, 0x9e94, 0x001a,
	0x001a, 0x6eea, 0x70ea, 0x001a, 0x001a, 0x71ea, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6fea, 0x8d8d, 0xcb96, 0x8396, 0xf59b, 0x001a, 0x809f, 0x9b96,
	0x001a, 0x001a, 0x001a, 0x001a, 0xa989, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x73ea, 0x6f8b, 0x74ea, 0x75ea,
	0x76ea, 0xecee, 0x958d, 0x001a, 0x77ea, 0x001a, 0x001a, 0x001a,
	0xd2e0, 0xd996, 0x001a, 0xe191, 0x78ea, 0x7aea, 0x79ea, 0x001a,
	0x7bea, 0x001a, 0x001a, 0x001a, 0x001a, 0x7cea, 0x001a, 0x001a,
	0x7dea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7eea,
	0x001a, 0x001a, 0x001a, 0x001a, 0x80ea, 0x001a, 0x81ea, 0x82ea,
	0x001a, 0x83ea, 0x001a, 0x84ea, 0x85ea, 0x86ea, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x87ea,
	0x88ea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4393, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdb8c, 0x001a, 0x8aea, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6c91, 0x8bea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x8cea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4095, 0x001a, 0x001a, 0x8dea, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x8eea, 0x56e2, 0x001a, 0x001a, 0xd8e6, 0xebe8,
	0x001a, 0x001a, 0x8fea, 0x001a, 0x90ea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x92ea,
	0x93ea, 0x94ea, 0xee97, 0x91ea, 0x001a, 0x001a, 0x95ea, 0x96ea,
	0x001a, 0x001a, 0x98ea, 0x001a, 0x97ea, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x9aea, 0x001a, 0x001a, 0x001a, 0x9bea, 0x99ea,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb497, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x9cea, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x9dea, 0x73e2, 0x001a, 0x001a,
	0x9eea, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0xf900[ 320 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc4ed, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xcdee, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x73ed, 0x7eed,
	0x80ed, 0x95ed, 0xbced, 0xcced, 0xceed, 0xf9ed, 0x42ee, 0x59ee,
	0x61ee, 0x62ee, 0x63ee, 0x65ee, 0x69ee, 0x6cee, 0x75ee, 0x81ee,
	0x83ee, 0x84ee, 0x8dee, 0x95ee, 0x97ee, 0x98ee, 0x9bee, 0xb7ee,
	0xbeee, 0xceee, 0xdaee, 0xdbee, 0xddee, 0xeaee, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_932_unicode_to_byte_stream_base_0xff00[ 256 ] = {
	0x001a, 0x4981, 0xfcee, 0x9481, 0x9081, 0x9381, 0x9581, 0xfbee,
	0x6981, 0x6a81, 0x9681, 0x7b81, 0x4381, 0x7c81, 0x4481, 0x5e81,
	0x4f82, 0x5082, 0x5182, 0x5282, 0x5382, 0x5482, 0x5582, 0x5682,
	0x5782, 0x5882, 0x4681, 0x4781, 0x8381, 0x8181, 0x8481, 0x4881,
	0x9781, 0x6082, 0x6182, 0x6282, 0x6382, 0x6482, 0x6582, 0x6682,
	0x6782, 0x6882, 0x6982, 0x6a82, 0x6b82, 0x6c82, 0x6d82, 0x6e82,
	0x6f82, 0x7082, 0x7182, 0x7282, 0x7382, 0x7482, 0x7582, 0x7682,
	0x7782, 0x7882, 0x7982, 0x6d81, 0x5f81, 0x6e81, 0x4f81, 0x5181,
	0x4d81, 0x8182, 0x8282, 0x8382, 0x8482, 0x8582, 0x8682, 0x8782,
	0x8882, 0x8982, 0x8a82, 0x8b82, 0x8c82, 0x8d82, 0x8e82, 0x8f82,
	0x9082, 0x9182, 0x9282, 0x9382, 0x9482, 0x9582, 0x9682, 0x9782,
	0x9882, 0x9982, 0x9a82, 0x6f81, 0x6281, 0x7081, 0x6081, 0x001a,
	0x001a, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
	0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
	0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
	0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
	0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
	0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x9181, 0x9281, 0xca81, 0x5081, 0xfaee, 0x8f81, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

/* Determines the size of a byte stream character from an Unicode character
 * Adds the size to the byte stream character size value
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_932_unicode_character_size_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     size_t *byte_stream_character_size,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_932_unicode_character_size_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream_character_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream character size.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x0080 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0100 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x0380 )
	      && ( unicode_character < 0x0480 ) )
	{
		unicode_character -= 0x0380;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0380[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2340 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2440 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2440;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2440[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3100 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3200 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3200;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3200[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fc0 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xf900 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xf900;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0xf900[ unicode_character ];
	}
	else if( ( unicode_character >= 0xff00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xff00;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0xff00[ unicode_character ];
	}
	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_character_size += 2;
	}
	else
	{
		*byte_stream_character_size += 1;
	}
	return( 1 );
}

/* Copies a Unicode character from a byte stream
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_932_copy_from_byte_stream(
     libuna_unicode_character_t *unicode_character,
     const uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function         = "libuna_codepage_windows_932_copy_from_byte_stream";
	uint8_t additional_character  = 0;
	uint8_t byte_stream_character = 0;

	if( unicode_character == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid Unicode character.",
		 function );

		return( -1 );
	}
	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	byte_stream_character = byte_stream[ *byte_stream_index ];

	if( byte_stream_character < 0x80 )
	{
		*unicode_character = byte_stream_character;
	}
	else if( ( byte_stream_character >= 0xa1 )
	      && ( byte_stream_character < 0xe0 ) )
	{
		*unicode_character = 0xff61 + byte_stream_character - 0xa1;
	}
	else if( ( *byte_stream_index + 1 ) <= byte_stream_size )
	{
		*byte_stream_index += 1;

		additional_character = byte_stream[ *byte_stream_index ];

		if( byte_stream_character == 0x81 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xad ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8140[ additional_character ];
			}
			else if( ( additional_character >= 0xb8 )
			      && ( additional_character < 0xc0 ) )
			{
				additional_character -= 0xb8;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81b8[ additional_character ];
			}
			else if( ( additional_character >= 0xc8 )
			      && ( additional_character < 0xcf ) )
			{
				additional_character -= 0xc8;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81c8[ additional_character ];
			}
			else if( ( additional_character >= 0xda )
			      && ( additional_character < 0xea ) )
			{
				additional_character -= 0xda;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81da[ additional_character ];
			}
			else if( ( additional_character >= 0xf0 )
			      && ( additional_character < 0xf8 ) )
			{
				additional_character -= 0xf0;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x81f0[ additional_character ];
			}
			else if( additional_character == 0xfc )
			{
				*unicode_character = 0x25ef;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x82 )
		{
			if( ( additional_character >= 0x4f )
			 && ( additional_character < 0x59 ) )
			{
				*unicode_character = 0xff10 - 0x4f + additional_character;
			}
			else if( ( additional_character >= 0x60 )
			      && ( additional_character < 0x7a ) )
			{
				*unicode_character = 0xff21 - 0x60 + additional_character;
			}
			else if( ( additional_character >= 0x81 )
			      && ( additional_character < 0x9b ) )
			{
				*unicode_character = 0xff41 - 0x81 + additional_character;
			}
			else if( ( additional_character >= 0x9f )
			      && ( additional_character < 0xf2 ) )
			{
				*unicode_character = 0x3041 - 0x9f + additional_character;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x83 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				*unicode_character = 0x30a1 - 0x40 + additional_character;
			}
			else if( ( additional_character >= 0x80 )
			      && ( additional_character < 0x97 ) )
			{
				*unicode_character = 0x30e0 - 0x80 + additional_character;
			}
			else if( ( additional_character >= 0x9f )
			      && ( additional_character < 0xb0 ) )
			{
				*unicode_character = 0x0391 - 0x9f + additional_character;
			}
			else if( ( additional_character >= 0xb0 )
			      && ( additional_character < 0xb7 ) )
			{
				*unicode_character = 0x03a3 - 0xb0 + additional_character;
			}
			else if( ( additional_character >= 0xbf )
			      && ( additional_character < 0xd0 ) )
			{
				*unicode_character = 0x03b1 - 0xbf + additional_character;
			}
			else if( ( additional_character >= 0xd0 )
			      && ( additional_character < 0xd7 ) )
			{
				*unicode_character = 0x03c3 - 0xd0 + additional_character;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x84 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x61 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8440[ additional_character ];
			}
			else if( ( additional_character >= 0x70 )
			      && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x70;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8470[ additional_character ];
			}
			else if( ( additional_character >= 0x80 )
			      && ( additional_character < 0x92 ) )
			{
				*unicode_character = 0x043e - 0x80 + additional_character;
			}
			else if( ( additional_character >= 0x9f )
			      && ( additional_character < 0xbf ) )
			{
				additional_character -= 0x9f;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x849f[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x87 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x54 ) )
			{
				*unicode_character = 0x2460 - 0x40 + additional_character;
			}
			else if( ( additional_character >= 0x54 )
			      && ( additional_character < 0x5e ) )
			{
				*unicode_character = 0x2160 - 0x54 + additional_character;
			}
			else if( ( additional_character >= 0x5f )
			      && ( additional_character < 0x76 ) )
			{
				additional_character -= 0x5f;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x875f[ additional_character ];
			}
			else if( additional_character == 0x7e )
			{
				*unicode_character = 0x337b;
			}
			else if( ( additional_character >= 0x80 )
			      && ( additional_character < 0x9d ) )
			{
				additional_character -= 0x80;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8780[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x88 )
		{
			if( ( additional_character >= 0x9f )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x9f;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x889f[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0x89 )
		      && ( byte_stream_character <= 0x97 ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0x89:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8940[ additional_character ];
						break;

					case 0x8a:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8a40[ additional_character ];
						break;

					case 0x8b:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8b40[ additional_character ];
						break;

					case 0x8c:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8c40[ additional_character ];
						break;

					case 0x8d:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8d40[ additional_character ];
						break;

					case 0x8e:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8e40[ additional_character ];
						break;

					case 0x8f:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x8f40[ additional_character ];
						break;

					case 0x90:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9040[ additional_character ];
						break;

					case 0x91:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9140[ additional_character ];
						break;

					case 0x92:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9240[ additional_character ];
						break;

					case 0x93:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9340[ additional_character ];
						break;

					case 0x94:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9440[ additional_character ];
						break;

					case 0x95:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9540[ additional_character ];
						break;

					case 0x96:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9640[ additional_character ];
						break;

					case 0x97:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9740[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0x98 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x73 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9840[ additional_character ];
			}
			else if( ( additional_character >= 0x9f )
			      && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x9f;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x989f[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0x99 )
		      && ( byte_stream_character <= 0x9f ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0x99:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9940[ additional_character ];
						break;

					case 0x9a:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9a40[ additional_character ];
						break;

					case 0x9b:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9b40[ additional_character ];
						break;

					case 0x9c:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9c40[ additional_character ];
						break;

					case 0x9d:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9d40[ additional_character ];
						break;

					case 0x9e:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9e40[ additional_character ];
						break;

					case 0x9f:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0x9f40[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xe0 )
		      && ( byte_stream_character <= 0xe9 ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xe0:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe040[ additional_character ];
						break;

					case 0xe1:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe140[ additional_character ];
						break;

					case 0xe2:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe240[ additional_character ];
						break;

					case 0xe3:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe340[ additional_character ];
						break;

					case 0xe4:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe440[ additional_character ];
						break;

					case 0xe5:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe540[ additional_character ];
						break;

					case 0xe6:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe640[ additional_character ];
						break;

					case 0xe7:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe740[ additional_character ];
						break;

					case 0xe8:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe840[ additional_character ];
						break;

					case 0xe9:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xe940[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xea )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xa5 ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xea40[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xed )
		      && ( byte_stream_character <= 0xee ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xed:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xed40[ additional_character ];
						break;

					case 0xee:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xee40[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xfa )
		      && ( byte_stream_character <= 0xfb ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0xfd ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xfa:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfa40[ additional_character ];
						break;

					case 0xfb:
						*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfb40[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xfc )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x4c ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_932_byte_stream_to_unicode_base_0xfc40[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else
		{
			*unicode_character = 0xfffd;
		}
	}
	else
	{
		*unicode_character = 0xfffd;
	}
	*byte_stream_index += 1;

	return( 1 );
}

/* Copies a Unicode character to a byte stream
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_932_copy_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_932_copy_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x0080 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0100 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x0380 )
	      && ( unicode_character < 0x0480 ) )
	{
		unicode_character -= 0x0380;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x0380[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2340 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2440 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2440;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x2440[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3100 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3200 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3200;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x3200[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fc0 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xf900 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xf900;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0xf900[ unicode_character ];
	}
	else if( ( unicode_character >= 0xff00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xff00;
		byte_stream_value  = libuna_codepage_windows_932_unicode_to_byte_stream_base_0xff00[ unicode_character ];
	}
	byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );

	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_index += 1;

		byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );
	}
	*byte_stream_index += 1;

	return( 1 );
}

