/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.optimistic;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.config.CacheLoaderConfig;
import org.jboss.cache.factories.XmlConfigurationParser;
import org.jboss.cache.loader.CacheLoader;
import org.jboss.cache.transaction.DummyTransactionManager;
import org.jboss.cache.xml.XmlHelper;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNull;
import org.testng.annotations.Test;
import org.w3c.dom.Element;

/**
 * Tests optimistic locking with pasivation
 *
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 */
@Test(groups = "functional")
public class OptimisticWithPassivationTest extends AbstractOptimisticTestCase
{
   protected CacheLoaderConfig getCacheLoaderConfig() throws Exception
   {
      String xml = "            <config>\n" +
            "                \n" +
            "                <passivation>true</passivation>\n" +
            "                <preload></preload>\n" +
            "\n" +
            "                <cacheloader>\n" +
            "                    <class>org.jboss.cache.loader.DummyInMemoryCacheLoader</class>\n" +
            "                    <async>false</async>\n" +
            "                    <fetchPersistentState>false</fetchPersistentState>\n" +
            "                    <ignoreModifications>false</ignoreModifications>\n" +
            "                </cacheloader>\n" +
            "                \n" +
            "            </config>";
      Element element = XmlHelper.stringToElement(xml);
      return XmlConfigurationParser.parseCacheLoaderConfig(element);
   }

   private CacheSPI createLocalCache() throws Exception
   {
      CacheSPI cache = createCacheUnstarted(false);
      cache.getConfiguration().setCacheLoaderConfig(getCacheLoaderConfig());

      cache.create();
      cache.start();
      return cache;
   }

   public void testPassivationLocal() throws Exception
   {
      CacheSPI cache = createLocalCache();
      CacheLoader loader = cache.getCacheLoaderManager().getCacheLoader();

      // clean up
      cache.removeNode(fqn);
      loader.remove(fqn);

      assertNull(loader.get(fqn));

      DummyTransactionManager mgr = DummyTransactionManager.getInstance();

      // put something in the cache
      mgr.begin();
      cache.put(fqn, key, value);
      mgr.commit();

      // should be nothing in the loader
      assertEquals(value, cache.get(fqn, key));
      assertNull(loader.get(fqn));

      // evict from cache
      mgr.begin();
      cache.evict(fqn);
      mgr.commit();

      mgr.begin();
      // should now be passivated in the loader
      // don't do a cache.get() first as this will activate this node from the loader again!
      // assertNull( cache.get( fqn ) );
      assertEquals(value, loader.get(fqn).get(key));

      // now do a cache.get()...
      assertEquals(value, cache.get(fqn, key));

      // and the object should now be removed from the loader
      assertNull(loader.get(fqn));
      mgr.commit();

      // clean up
      mgr.begin();
      cache.removeNode(fqn);
      loader.remove(fqn);
      mgr.commit();

   }
}
