/*
 * Copyright (c) 2019 Mellanox Technologies, Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include "common/def.h"
#include "common/log.h"
#include "common/sys.h"
#include "common/base.h"

#include "udp_base.h"


class udp_bind : public udp_base {};

/**
 * @test udp_bind.ti_1
 * @brief
 *    bind(SOCK_DGRAM) socket to local ip
 * @details
 */
TEST_F(udp_bind, ti_1) {
	int rc = EOK;
	int fd;

	fd = udp_base::sock_create();
	ASSERT_LE(0, fd);

	errno = EOK;
	rc = bind(fd, (struct sockaddr *)&client_addr, sizeof(client_addr));
	EXPECT_EQ(EOK, errno);
	EXPECT_EQ(0, rc);

	close(fd);
}

/**
 * @test udp_bind.ti_2
 * @brief
 *    bind(SOCK_DGRAM) socket to remote ip
 * @details
 */
TEST_F(udp_bind, ti_2) {
	int rc = EOK;
	int fd;

	fd = udp_base::sock_create();
	ASSERT_LE(0, fd);

	errno = EOK;
	rc = bind(fd, (struct sockaddr *)&remote_addr, sizeof(remote_addr));
	EXPECT_EQ(EADDRNOTAVAIL, errno);
	EXPECT_GT(0, rc);

	close(fd);
}

/**
 * @test udp_bind.ti_3
 * @brief
 *    bind(SOCK_DGRAM) socket twice
 * @details
 */
TEST_F(udp_bind, ti_3) {
	int rc = EOK;
	int fd;
	struct sockaddr_in addr;

	fd = udp_base::sock_create();
	ASSERT_LE(0, fd);

	errno = EOK;
	rc = bind(fd, (struct sockaddr *)&client_addr, sizeof(client_addr));
	EXPECT_EQ(EOK, errno);
	EXPECT_EQ(0, rc);

	memcpy(&addr, &client_addr, sizeof(addr));
	addr.sin_port = htons(bogus_port);

	errno = EOK;
	rc = bind(fd, (struct sockaddr *)&addr, sizeof(addr));
	EXPECT_EQ(EINVAL, errno);
	EXPECT_GT(0, rc);

	close(fd);
}

/**
 * @test udp_bind.ti_4
 * @brief
 *    bind(SOCK_DGRAM) two sockets on the same ip
 * @details
 */
TEST_F(udp_bind, ti_4) {
	int rc = EOK;
	int fd;
	int fd2;

	fd = udp_base::sock_create();
	ASSERT_LE(0, fd);

	errno = EOK;
	rc = bind(fd, (struct sockaddr *)&client_addr, sizeof(client_addr));
	EXPECT_EQ(EOK, errno);
	EXPECT_EQ(0, rc);

	fd2 = udp_base::sock_create();
	ASSERT_LE(0, fd);

	errno = EOK;
	rc = bind(fd2, (struct sockaddr *)&client_addr, sizeof(client_addr));
	EXPECT_EQ(EADDRINUSE, errno);
	EXPECT_GT(0, rc);

	close(fd);
	close(fd2);
}
