/* igmp.c
  
   PDU builder for IGMP messages

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-igmp.h"

static void
build(const GNode *pdu, void *dest)
{
    struct igmp *igmp;
    ip_addr_t group; 

    igmp = dest;

    igmp->igmp_type = num_next(_pb_pdata(pdu, "type") );
    igmp->igmp_code = num_next(_pb_pdata(pdu, "resptime") );

    group = ip_next(_pb_pdata(pdu, "group") );
    SIVAL(igmp, offsetof(struct igmp, igmp_group), group);
}

static void
postbuild(const GNode *pdu, void *dest, void *prev_pdu_hdr _U_)
{
    struct igmp *igmp;
    uint16_t cksum;
    void *field;
    size_t opts_len, payload_len;

    igmp = dest;

    if ( (field = _pb_pdata(pdu, "cksum") ) ) {
	cksum = htons(num_next(field) );
    } else {
	opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
	payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

	/* Must be 0 before calculating cksum! */
	SSVAL(igmp, offsetof(struct igmp, igmp_cksum), 0);

	cksum = _pb_cksum(dest,
			  sizeof(struct igmp) + opts_len + payload_len);
    }

    SSVAL(igmp, offsetof(struct igmp, igmp_cksum), cksum);
}

#if 0
static void
pdu_igmphdr_dumper(pdu_t *p, const char *prefix)
{
    struct igmphdr_options *hdr_data;
    char addr[INET_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Type: %s\n", prefix, num_info(hdr_data->type) );
    printf("%s    Response time: %s\n", prefix, num_info(hdr_data->resptime) );
    printf("%s    Group address: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->group, addr, INET_ADDRSTRLEN) );
    printf("%s    IGMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

static pdu_t pdu_igmp = {
    .name = "igmp",
    .description = "IGMP message",
    .documented_in = "RFC 2236",
    .len = sizeof(struct igmp),
    .fields = (field_t []) {
	{.name = "type", .type = PDU_FTYPE_NUMTYPE},
	{.name = "resptime", .type = PDU_FTYPE_NUMTYPE},
	{.name = "group", .type = PDU_FTYPE_IP},
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL, .type = 0}
    },
    .build = &build,
    .postbuild = &postbuild
};

void
_pb_register_igmp(void)
{
    _pb_register_protocol(&pdu_igmp);
}
