///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_OBJECT_MODIFIER_BASE_H
#define __ATOMS_OBJECT_MODIFIER_BASE_H

#include <core/Core.h>
#include <core/scene/objects/Modifier.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <core/reference/CloneHelper.h>
#include <core/reference/RefTargetListener.h>

#include <atomviz/AtomViz.h>
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/DataChannel.h>

namespace AtomViz {

/******************************************************************************
* Abstract base class for modifiers that operate on an AtomsObject.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomsObjectModifierBase : public Modifier
{
protected:
	/// Constructor.
	AtomsObjectModifierBase(bool isLoading = false) : Modifier(isLoading), inputAtoms(NULL), modApp(NULL) {}

public:

	/// This modifies the input object.
	virtual EvaluationStatus modifyObject(TimeTicks time, ModifierApplication* modApp, PipelineFlowState& state);

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);
	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// Modifies the atoms object. This function must be implemented by sub-classes
	/// do the modifier specific work. The time interval passed
	/// to the function should be reduced to the interval where the returned object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval) = 0;

	/// Returns the input atoms. The returned object may not be modified.
	/// If you want to modify the AtomsObject then use the result() method instead.
	/// It will return a copy of the input object that may be modified.
	AtomsObject* input() const { OVITO_ASSERT(inputAtoms != NULL); return inputAtoms; }

	/// Creates a shallow copy of the input object on demand and returns it.
	AtomsObject* output();

	/// Completely replaces the output object with a new AtomsObject.
	void setOutput(AtomsObject* newOutput) { outputAtoms = newOutput; }

	/// Returns the standard channel with the given identifier from the output object.
	/// The requested data channel will be created or deep copied as needed.
	DataChannel* outputStandardChannel(DataChannel::DataChannelIdentifier which);

	/// Returns the standard channel with the given identifier from the input object.
	/// The returned channel may be NULL if it does not exist. Its contents
	/// may not be modified.
	DataChannel* inputStandardChannel(DataChannel::DataChannelIdentifier which) const;

	/// Returns the given standard channel from the input object.
	/// The returned channel may not be modified. If they input object does
	/// not contain the standard channel then an exception is thrown.
	DataChannel* expectStandardChannel(DataChannel::DataChannelIdentifier which) const;

	/// Returns the channel with the given name from the input object.
	/// The returned channel may not be modified. If they input object does
	/// not contain a channel with the given name then an exception is thrown.
	/// If there is a channel with the given name but with another data type then
	/// an exception is thrown too.
	DataChannel* expectCustomChannel(const QString& channelName, int channelDataType, size_t componentCount = 1) const;

	/// Returns the current ModifierApplication object for this modifier.
	ModifierApplication* modifierApplication() const { OVITO_ASSERT(modApp != NULL); return modApp; }

	/// Returns a clone helper object that should be used to create shallow and deep copies
	/// of the atoms object and its data.
	CloneHelper* cloneHelper() {
		if(!_cloneHelper) _cloneHelper.reset(new CloneHelper());
		return _cloneHelper.get();
	}

protected:

	/// The clone helper object that is used to create shallow and deep copies
	/// of the atoms object and its channels.
	scoped_ptr<CloneHelper> _cloneHelper;

	/// The current ModifierApplication object.
	ModifierApplication* modApp;

	/// The input atoms of the modifier.
	AtomsObject* inputAtoms;

	/// The output atoms of the modifier.
	AtomsObject::SmartPtr outputAtoms;

private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AtomsObjectModifierBase)
};

/******************************************************************************
* Base class for properties editors that are made for objects derived
* from the AtomsObjectModifierBase class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomsObjectModifierEditorBase : public PropertiesEditor
{
public:

	/// Constructor.
	AtomsObjectModifierEditorBase() : PropertiesEditor(),
		modifierStatusInfoIcon(":/atomviz/icons/modifier_status_info.png"),
		modifierStatusWarningIcon(":/atomviz/icons/modifier_status_warning.png"),
		modifierStatusErrorIcon(":/atomviz/icons/modifier_status_error.png")
	{
		connect(this, SIGNAL(contentsReplaced(RefTarget*)), this, SLOT(onContentsReplaced(RefTarget*)));
		connect(&_modAppListener, SIGNAL(notificationMessage(RefTargetMessage*)), this, SLOT(onModAppNotificationMessage(RefTargetMessage*)));
	}

	/// Returns a widget that displays a message sent by the modifier that
	/// states the outcome of the modifier evaluation. Derived classes of this
	/// editor base class can add the widget to their user interface.
	QWidget* statusLabel();

private Q_SLOTS:

	/// This handler is called when a new edit object has been loaded into the editor.
	void onContentsReplaced(RefTarget* newEditObject);

	/// This handler is called when the current ModifierApplication sends a notification message.
	void onModAppNotificationMessage(RefTargetMessage* msg);

private:

	/// Updates the text of the result label.
	void updateStatusLabel(ModifierApplication* modApp);

	RefTargetListener _modAppListener;
	QPointer<QWidget> _statusLabel;
	QPointer<QLabel> _statusTextLabel;
	QPointer<QLabel> _statusIconLabel;

	QPixmap modifierStatusInfoIcon;
	QPixmap modifierStatusWarningIcon;
	QPixmap modifierStatusErrorIcon;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AtomsObjectModifierEditorBase)
};

};	// End of namespace AtomViz

#endif // __ATOMS_OBJECT_MODIFIER_BASE_H
