/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "filtrationrule.h"

#include <QRegExp>
#include <QHostAddress>

#include "message.h"

FiltrationRule::FiltrationRule() :
 m_isRegExp(false),
 m_activated(true)
{
}


FiltrationRule::~FiltrationRule()
{
}

bool FiltrationRule::checkMessage(Message* msg)
{
  if(
     (m_packetType.contains(msg->type    ()) || m_packetType.isEmpty()) &&
     (m_srcIp     .contains(msg->srcIp   ()) || m_srcIp     .isEmpty()) &&
     (m_srcUid    .contains(msg->srcUid  ()) || m_srcUid    .isEmpty()) &&
     (m_userName  .contains(msg->userName()) || m_userName  .isEmpty()) &&
     (m_compName  .contains(msg->compName()) || m_compName  .isEmpty()))
  {
    if(m_messageFilter.isEmpty())
      if(m_packetType.isEmpty() &&
         m_srcIp     .isEmpty() &&
         m_srcUid    .isEmpty() &&
         m_userName  .isEmpty() &&
         m_compName  .isEmpty())
         return false;
      else
         return true;

    QStringList words = m_messageFilter.split("\n");
    QRegExp rx;

    if(isRegExp())
    {
      foreach(QString s, words)
      {
        rx.setPattern(s);
        if(msg->msg().contains(rx))
          return true;
      }
    }
    else
    {
      foreach(QString s, words)
        if(msg->msg().contains(QRegExp("\\b" + QRegExp().escape(s) + "\\b")))
          return true;
    }
  }

  return false;
}

void FiltrationRule::setUserNames(const QString & str)
{
  if(str.isEmpty())
  {
    m_userName.clear();
    return;
  }

  QStringList list = str.split("\n");

  m_userName.clear();

  foreach(QString s, list)
    if(!m_userName.contains(s))
      m_userName.append(s);
}

void FiltrationRule::setIPs(const QString & str)
{
  if(str.isEmpty())
  {
    m_srcIp.clear();
    return;
  }

  QStringList list = str.split("\n");
  QHostAddress addr;
  quint64 ip;

  m_srcIp.clear();

  foreach(QString s, list)
  {
    addr.setAddress(s);
    ip = addr.toIPv4Address();
    if(ip != 0 && !m_srcIp.contains(ip))
      m_srcIp.append(ip);
  }
}

void FiltrationRule::setCompNames(const QString & str)
{
  if(str.isEmpty())
  {
    m_compName.clear();
    return;
  }

  QStringList list = str.split("\n");

  m_compName.clear();

  foreach(QString s, list)
    if(!m_compName.contains(s))
      m_compName.append(s);
}

QString FiltrationRule::userNames() const
{
  QString res;

  foreach(QString s, m_userName)
    res += s + '\n';

  res.remove(res.size() - 1, 1);

  return res;
}

QString FiltrationRule::compNames() const
{
  QString res;

  foreach(QString s, m_compName)
    res += s + '\n';

  res.remove(res.size() - 1, 1);

  return res;
}

QString FiltrationRule::IPs() const
{
  QString res;

  foreach(quint64 ip, m_srcIp)
    res += QHostAddress(ip).toString() + '\n';

  res.remove(res.size() - 1, 1);

  return res;
}

void FiltrationRule::save(QSettings* settings) const
{
  settings->setValue("Name"         , m_name);
  settings->setValue("IsRegExp"     , m_isRegExp);
  settings->setValue("Activated"    , m_activated);
  settings->setValue("IPs"          , IPs());
  settings->setValue("UserNames"    , userNames());
  settings->setValue("CompNames"    , compNames());
  settings->setValue("MessageFilter", messageFilter());
}

void FiltrationRule::load(QSettings* settings)
{
  setName         (settings->value("Name").toString());
  setIsRegExp     (settings->value("IsRegExp").toBool());
  setActivated    (settings->value("Activated").toBool());
  setIPs          (settings->value("IPs").toString());
  setUserNames    (settings->value("UserNames").toString());
  setCompNames    (settings->value("CompNames").toString());
  setMessageFilter(settings->value("MessageFilter").toString());
}
