// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief e+e- > pi0 pi0 gamma
  class KLOE_2007_I725483 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(KLOE_2007_I725483);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      // get the CMS point
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(10*eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"0"s], 1, ih, 1);
        book(_h[en+"1"s], 1, ih, 2);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      Particles pi0, gamma;
      map<long,Particles> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        if (p.pid() == PID::PI0)          pi0 += p;
        else if (p.pid() == PID::GAMMA) gamma += p;
        ++ntotal;
      }
      // three particles (pi0 pi0 gamma)
      if (ntotal!=3 || pi0.size()!=2 || gamma.size()!=1) vetoEvent;
      _h[_sqs+"0"]->fill((pi0[0].mom()+  pi0[1].mom()).mass()/MeV);
      _h[_sqs+"1"]->fill((pi0[0].mom()+gamma[0].mom()).mass()/MeV);
      _h[_sqs+"1"]->fill((pi0[1].mom()+gamma[0].mom()).mass()/MeV);
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(KLOE_2007_I725483);

}
