// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief R measurement
  class MARKI_1975_I100733 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(MARKI_1975_I100733);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      declare(FinalState(), "FS");
      // Book histograms
      book(_h["hadrons0"], 1, 1, 1);
      book(_h["hadrons1"], "/TMP/sigma_hadrons", refData<YODA::BinnedEstimate<string>>(2,1,1));
      book(_h["muons"], "/TMP/sigma_muons", refData<YODA::BinnedEstimate<string>>(2,1,1));
      for (size_t ix=0; ix<2; ++ix) {
        for (const string& en : _h["hadrons"+toString(ix)].binning().edges<0>()) {
          const size_t idx = en.find("-");
          if (idx!=std::string::npos) {
            const double emin = std::stod(en.substr(0,idx));
            const double emax = std::stod(en.substr(idx+1,string::npos));
            if (inRange(sqrtS()/GeV, emin, emax)) {
              _sqs[ix] = en; break;
            }
          }
          else {
            const double eval = stod(en)*GeV;
            if (isCompatibleWithSqrtS(eval)) {
              _sqs[ix] = en; break;
            }
          }
        }
      }
      raiseBeamErrorIf(_sqs[0].empty() && _sqs[1].empty());
      if (_sqs[1] == "3.0"s) {
      	book(_h[_sqs[1]+"charged"], 3, 1, 1);
      }
      else if (_sqs[1] == "4.8"s) {
      	book(_h[_sqs[1]+"charged"], 3, 1, 2);
      }
      else if (_sqs[1] == "7.4"s) {
      	book(_h[_sqs[1]+"charged"], 3, 1, 3);
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
      	nCount[p.pid()] += 1;
      	++ntotal;
      }
      if (nCount[-13]==1 && nCount[13]==1 && ntotal==2+nCount[22]) {
      	_h["muons"]->fill(_sqs[1]); // mu+mu- + photons
      }
      // everything else
      else {
        _h["hadrons0"]->fill(_sqs[0]);
        _h["hadrons1"]->fill(_sqs[1]);
        if (_sqs[1] == "3.0"s || _sqs[1] == "4.8"s || _sqs[1] == "7.4"s) {
          if (_edges.empty()) _edges = _h[_sqs[1]+"charged"]->xEdges();
          for (const Particle& p : fs.particles()) {
            if (PID::isCharged(p.pid())) {
              double x = 2.*p.p3().mod()/sqrtS();
              const size_t idx = _axis.index(x);
              if (idx && idx <=_edges.size()) _h[_sqs[1]+"charged"]->fill(_edges[idx-1]);
            }
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      const double sf = crossSection()/ sumOfWeights() /nanobarn;
      for (auto& item : _h) {
        if (item.first.find("charged") != string::npos) {
          scale(item.second, crossSection()/sumOfWeights()/microbarn*sqr(sqrtS()));
          for (auto& b : item.second->bins()) {
            b.scaleW(1./_axis.width(b.index()));
          }
          continue;
        }
        scale(item.second, sf);
      }
      BinnedEstimatePtr<string> mult;
      book(mult, 2, 1, 1);
      divide(_h["hadrons1"], _h["muons"], mult);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    vector<string> _edges;
    string _sqs[2];
    YODA::Axis<double> _axis{0.,0.10,0.12,0.14,0.16,0.18,0.20,0.22,0.24,0.26,0.28,0.30,
                             0.32,0.34,0.36,0.38,0.40,0.44,0.48,0.52,0.56,0.64,0.72,0.80,0.88};
    /// @}


  };


  RIVET_DECLARE_PLUGIN(MARKI_1975_I100733);


}
