SYD(2)

# NAME

*/dev/syd* virtual system call interface

# SYNOPSIS

```
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Magic commands return `/dev/null' metadata in stat.
struct stat stat;

/* Execute sandbox commands */
int stat("/dev/syd/[command]", &stat); // $ syd -m command
int stat("/dev/syd/[config]?", &stat);
int stat("/dev/syd/[config]:[value]", &stat); // $ syd -m config:value
int stat("/dev/syd/[list]+[value]", &stat); // $ syd -m list+value
int stat("/dev/syd/[list]-[value]", &stat); // $ syd -m list-value
int stat("/dev/syd/[list]^[value]", &stat); // $ syd -m list^value
int stat("/dev/syd/[command]![value]", &stat);

/* Read sandbox state as JSON */
int open("/dev/syd", O_RDONLY);

/* Read syd.el which is the Emacs Lisp implementation of the API */
int open("/dev/syd.el", O_RDONLY);

/* Read syd.sh which exports esyd shell function */
int open("/dev/syd.sh", O_RDONLY);
```

# DESCRIPTION

The */dev/syd* virtual system call interface is a unique mechanism
designed for runtime configuration of the Syd sandbox environment. It
enables sandboxed processes to interact with the Syd process to
dynamically adjust sandbox settings or query its state. This interaction
is facilitated through the use of virtual system calls, specifically via
the _stat_(2) system call, applied to specially constructed paths under
*/dev/syd*.

This interface allows for a range of operations, including enabling or
disabling sandbox features, appending or removing elements from lists,
querying the sandbox state, and executing special Syd commands.
Operations are specified through paths constructed with */dev/syd* as
the prefix, followed by a sandbox command and an operation character
that denotes the desired action:

- *:* for setting a value (boolean, integer, string),
- *?* for querying a value,
- *+* for appending to a string vector,
- *-* for removing an element from a string vector,
- *^* for removing all matching elements from a string vector, and
- *!* for executing a special Syd command.

The type *string-map* is similar to *string-vec*, except the operator
*^* does not accept an argument and removes all elements from the string
map. The *-* operator of a string map is functionally equivalent to the
*^* operator of a string vector in that both remove all matching elements
from the respective set.

This interface supports a flexible and powerful method for managing sandbox
policies dynamically, allowing for real-time adjustments to the security and
operational behavior of sandboxed processes. *libsyd* is a comprehensive C
library designed for interfacing with the Syd stat interface. It offers
functionalities for managing sandbox states, and facilitating runtime
configuration and interaction with the Syd sandboxing environment. *gosyd* is
a Go module that uses _cgo_ to use *libsyd*. *plsyd* is a Perl module that uses
_FFI::Platypus_ to use *libsyd*. *pysyd* is a Python module that uses _ctypes_
to use *libsyd*. *rbsyd* is a Ruby module that uses _ffi_ gem to use *libsyd*.
*syd.el* is an _Emacs Lisp_ implementation of the Syd stat interface.

# COMMANDS

The */dev/syd* interface supports the following commands for runtime
configuration of the sandbox. Each command can be invoked through the _stat_(2)
system call on special paths under */dev/syd*. Note, Syd provides similar
interfaces for the _stat_(2) interface, *-m* command-line option and the
configuration file. Some sandbox commands only take affect when they're
submitted on startup, such as *unshare/user*, and *sandbox/lock*. Such commands
are noted as *static* in the descriptions below.

## stat

This command causes Syd to output sandbox state on standard error.

## reset

This command causes Syd to reset sandboxing to the default state.
Allowlists, denylists and filters are going to be cleared. The state of
the sandbox lock is not affected by reset. This ensures an unintended
reset cannot open window for a sandbox bypass. In addition, the state of
Crypt sandboxing is not affected by reset too. This ensures concurrent
or near-concurrent encryption operations continue uninterrupted.

## panic

This command causes Syd to exit immediately with code 127.

Due to security reasons, this command is only available via the virtual
_stat_(2) call, it may not be used with the *-m* command line switch or
in a configuration file.

Due to safety reasons, panic may not be called when Crypt sandboxing is
on. In this case the virtual _stat_(2) returns -1 and sets _errno_(3) to
*EBUSY*. This ensures concurrent or near-concurrent encryption
operations continue uninterrupted.

## ghost

This command initiates Ghost mode. Ghost mode is irreversible so you can
call this command only once during Syd runtime. Refer to the *Ghost
mode* section of the _syd_(7) manual page for more information. This
command implies *reset*, ie. the sandbox state is reset before Ghost
mode initiation to ensure there're no run-away exec processes after the
invocation of the *ghost* command. Ghost mode is only available via the
virtual _stat_(2) call, it can not be used with the *-m* command line
switch or in a configuration file.

## config/expand

[[ type
:[ *integer (u64)*
|  default
:  *0*
|  static
:  *yes*

Given zero as timeout in seconds, which is the default, enables
environment variable and tilde expansion using the *shellexpand* crate.
This runs much faster as it does not require confinement, however it
does not support command substitution and recursive environment variable
expansion like _wordexp_(3) does. Unset environment variables are not
expanded to empty strings. On environment variable lookup errors and
UTF-8 decoding errors Syd stops parsing and exits with error. This is
done for safety as an unintended empty-string expansion can potentially
cause the resulting sandboxing rule to allowlist unintended paths
without the user easily noticing it. The user is recommended to set
default values for environment variables using the familiar
*${HOME:-/var/empty}* notation. Empty-string expansion on unset
environment variables can still be done using the notation *${HOME:-}*
but this is not recommended and should be used with care.

Given a positive integer as timeout in seconds, enables environment
variable expansion and command substitutiton for configuration using
_wordexp_(3). The fork process which calls */bin/sh* for expansion is
executed in a confined environment and it is terminated if its runtime
exceeds the given timeout. Confinement is done using _landlock_(7),
_namespaces_(7) and _seccomp_(2).

This is a startup-only setting. For safety reasons, no expansion is
performed for runtime configuration.

## ipc

[[ type
:[ *string*
|  static
:  *yes*

Configure sandbox during runtime using the given UNIX socket address
with kernel-validated peer authentication. Authentication leverages
*SCM_CREDENTIALS* and *SO_PASSCRED* mechanisms to verify that connecting
processes share identical UID and GID with the IPC worker process.
Authentication UID and GID may be overriden by *ipc/uid* and *ipc/gid*
options at startup. This kernel-enforced authentication prevents
privilege escalation and unauthorized access by validating credentials
on every message, ensuring only the specified user and group or the
system administrator can execute IPC commands.

If the argument starts with the character *@*, the address is taken to
be an _abstract_ UNIX socket. Use the keywords *none* or *off* to unset
a previously set IPC address. The IPC implementation is inspired by
HAProxy's stats socket implementation. All responses except the *stats*
command are in compact JSON. User is recommended to use the *version*
command to check the API version prior to use. As a safety measure, the
IPC service is provided as long as the sandbox is unlocked. When the
sandbox is locked, the _syd_ipc_ thread exits. This thread makes no
attempt to _unlink_(2) the UNIX domain socket path at startup or exit.
The user should perform the cleanup or use abstract sockets which is
recommended. To access the socket, an external utility such as
_socat_(1) is required. Socat is a swiss-army knife to connect anything
to anything.  We use it to connect terminals to the socket, or a couple
of stdin/stdout pipes to it for scripts. The two main syntaxes we'll use
are the following:

```
# socat ~/.syd/sandbox.sock stdio
# socat ~/.syd/sandbox.sock readline
```

The first one is used with scripts. It is possible to send the output of
a script to Syd, and pass Syd's output to another script. That's useful
for retrieving sandbox configuration as JSON for example. The second one
is only useful for issuing commands by hand. It has the benefit that the
terminal is handled by the readline library which supports line editing
and history, which is very convenient when issuing repeated commands
(eg: watch a counter).

The socket supports three operation modes:
- non-interactive, silent
- interactive, silent
- interactive with prompt

The non-interactive mode is the default when _socat_(1) connects to the
socket. In this mode, a single line may be sent. It is processed as a
whole, responses are sent back, and the connection closes after the end
of the response. This is the mode that scripts and monitoring tools use.
A single command may be sent at a time only. The interactive mode allows
new commands to be sent after the ones from the previous lines finish.
It exists in two variants, one silent, which works like the
non-interactive mode except that the socket waits for a new command
instead of closing, and one where a prompt is displayed (';') at the
beginning of the line. The interactive mode is preferred for advanced
tools while the prompt mode is preferred for humans.

The mode can be changed using the *prompt* command. By default, it
toggles the interactive+prompt modes. Entering *prompt* in interactive
mode will switch to prompt mode. The command optionally takes a specific
mode among the following:

- *n*: non-interactive mode (single command and quits)
- *i*: interactive mode (multiple commands, no prompt)
- *p*: prompt mode (multiple commands with a prompt)

Since the default mode is non-interactive, *prompt* must be used as the
first command in order to switch it, otherwise the previous command will
cause the connection to be closed. Switching to non-interactive mode
will result in the connection to be closed after all the commands of the
same line complete.

For this reason, when debugging by hand, it's quite common to start with
the *prompt* command:

```
# socat ~/.syd/sandbox.sock readline
prompt
; stats
...
; 
```

Interactive tools might prefer starting with *prompt i* to switch to
interactive mode without the prompt.

The following commands are supported in addition to the _syd_(2) API:

- stat: Prints sandbox state in compact JSON.
- stats: Prints sandbox state in human-readable format.
- version: Prints IPC api version in compact JSON.

The commands *quit* and *exit* may be used to close a socket connection.
The command *ping* is supported for aliveness checks.

## ipc/uid

[[ type
:[ *uid*
|  default
:  *Uid::current*
|  static
:  *yes*

User ID override for IPC authentication. Specifies the UID that
connecting processes must possess to authenticate with the IPC worker.
Accepts either numeric user IDs or user names. When specified as a user
name, the system resolves it to the corresponding UID using
_getpwnam_(3). Defaults to the current process UID obtained via
_getuid_(2). When set, the IPC worker validates that all connecting
clients have this exact UID via *SCM_CREDENTIALS* authentication. This
setting allows privilege delegation scenarios where the IPC worker runs
as one user but accepts connections from processes running as a
different specific UID. Set the option to *none* or *off* to
disable UID authentication for IPC.

## ipc/gid

[[ type
:[ *gid*
|  default
:  *Gid::current*
|  static
:  *yes*

Group ID override for IPC authentication. Specifies the GID that
connecting processes must possess to authenticate with the IPC worker.
Accepts either numeric group IDs or group names. When specified as a
group name, the system resolves it to the corresponding GID using
_getgrnam_(3). Defaults to the current process GID obtained via
_getgid_(2). When set, the IPC worker validates that all connecting
clients have this exact GID via *SCM_CREDENTIALS* authentication. This
setting enables group-based access control where multiple users
belonging to the same group can access the IPC interface. Set the
option to *none* or *off* to disable GID authentication for IPC.

## lock

[[ type
:[ *string*

Set the state of the sandbox lock. Possible values are *on*, *off*,
*exec*, *ipc*, and *read* or shortly just *1*, *0*, *x*, *i*, and *r*.
The values are case-sensitive. The values *ro*, *readonly* and
*read-only* are also permitted for *read* mode which was added as of
version 3.39.0. Specifying just *lock* without value or shortly *l* is
permitted as a short-hand for *lock:on*.

If the sandbox lock is *on* no sandbox commands are allowed. If sandbox
lock is *read*, only reads are allowed but NOT edits. A read locked
sandbox makes available only the read-only _open_(2) hooks of the
_syd_(2) virtual system call API to the sandbox process. _stat_(2) hooks
for edits are NOT permitted in a read locked sandbox.

If *exec* is specified, the sandbox lock is set to *on* for all
processes except the initial process, aka Syd exec child. If the sandbox
lock is *ipc*, sandbox commands may only be specified using the IPC
socket. The IPC socket is a UNIX socket which may or may not be
accessible from within the sandbox depending on sandbox ACL rules.

Transition from lock modes *off*, *exec*, and *ipc* into one of *read*
and *on* is one-way and idempotent: It results in the sandbox policy
getting sealed in memory using the _mseal_(2) system call either
immediately or simultaneously with sandbox process startup.  Transitions
between lock modes *read* and *on* are not permitted.

The sandbox lock used to default to *exec* but as a hardening measure
and to ensure security by default, as of version 3.17.0, this has been
changed such that the default is *unset* and if no lock clause has been
specified by the time Syd executes the initial sandbox process, then the
sandbox lock is automatically set to *on*. This means if no *lock*
clause is specified in any of the profiles, configuration files or *-m*
CLI arguments, the lock will be *on* by default. As of version 3.35.2,
this default is set to *ipc* if the *ipc* command was specified but lock
was not set explicitly. Setting lock to *on* at any point during
configuration parsing prevents further commands from being emitted by
the sandbox. This feature may be used to lock site-wide defaults for a
Syd login shell by adding a *lock:on* clause at the end of the site-wide
configuration file which prevents Syd from subsequently parsing the user
configuration file, practically enforcing the site-wide defaults.

Setting lock to *off*, *exec*, or *ipc* at startup makes Syd skip
preventing _execve_(2) and _execveat_(2) system calls as part of the
*Execution Control (EEC)* feature. This is done to allow *cmd/exec*
command to execute commands outside the sandbox. This filter to prevent
_exec_(3) is only applied when the sandbox is locked.

## log/level

[[ type
:[ *string*
|  default
:  *warn*

Set the log level. Available log levels are *emerg*, *alert*, *crit*,
*error*, *warn*, *notice*, *info*, and *debug*. Defaults to *warn*
unless *SYD_LOG* environment variable is set at startup. An integer in
the closed range of *[0,7]* can also be used as an argument to set the
log level, where *0* corresponds to *emerg* and *7* corresponds to
*debug*. All access violations except the *stat* and *walk* categories
are logged with the *warn* level. Stat and Walk categories are logged
with the *notice* level. Startup messages are logged with the *info*
level.

## log/lock/same_exec_off

[[ type
:[ *boolean*
|  default
:  *off*

Disables logging of denied accesses originating from the thread creating
the _landlock_(7) domain, as well as its children, as long as they
continue running the same executable code (i.e., without an intervening
_execve_(2) call). This is intended for programs that execute unknown
code without invoking _execve_(2), such as script interpreters. Programs
that only sandbox themselves should not set this flag, so users can be
notified of unauthorized access attempts via system logs.

This option requires _landlock_(7) ABI 7 support which is new in
Linux-6.15. Setting this option is a NO-OP otherwise. Setting this
option is also a NO-OP when *sandbox/lock* is off. Multiple options may
be set or unset at once by passing them as a comma-delimited list.
Environment variables in the value are expanded.

## log/lock/new_exec_on

[[ type
:[ *boolean*
|  default
:  *off*

Enables logging of denied accesses after an _execve_(2) call, providing
visibility into unauthorized access attempts by newly executed programs
within the created _landlock_(7) domain. This flag is recommended only
when all potential executables in the domain are expected to comply with
the access restrictions, as excessive audit log entries could make it
more difficult to identify critical events.

This option requires _landlock_(7) ABI 7 support which is new in
Linux-6.15. Setting this option is a NO-OP otherwise. Setting this
option is also a NO-OP when *sandbox/lock* is off. Multiple options may
be set or unset at once by passing them as a comma-delimited list.
Environment variables in the value are expanded. A sandboxer should not
log denied access requests to avoid spamming logs, therefore this option
is off by default. Use this option to test audit logging.

## log/lock/subdomains_off

[[ type
:[ *boolean*
|  default
:  *off*

Disables logging of denied accesses originating from nested
_landlock_(7) domains created by the caller or its descendants. This
flag should be set according to runtime configuration, not hardcoded, to
avoid suppressing important security events. It is useful for container
runtimes or sandboxing tools that may launch programs which themselves
create _landlock_(7) domains and could otherwise generate excessive
logs. Unlike *log/lock/same_exec_off*, this flag only affects future
nested domains, not the one being created.

This option requires _landlock_(7) ABI 7 support which is new in
Linux-6.15. Setting this option is a NO-OP otherwise. Setting this
option is also a NO-OP when *sandbox/lock* is off. Multiple options may
be set or unset at once by passing them as a comma-delimited list.
Environment variables in the value are expanded.

## log/verbose

[[ type
:[ *boolean*
|  default
:  *off*

Set verbose logging. In verbose mode, Syd acquires various details about
the current _seccomp_(2) request and adds this information to the access
violation JSON payload under the *req* key. This is primarily intended
for malware analysis. When verbose logging is off, which is the default,
Syd only logs the process ID of the current _seccomp_(2) request
in addition to the access violation information.

As of version 3.38.0, process name changes using the *PR_SET_NAME*
_prctl_(2) are logged when verbose mode is on.

## pty/row

[[ type
:[ *ushort*
|  default
:  *<inherit>*
|  static
:  *yes*

Set row size for PTY sandboxing. Default is to inherit the window-size.
Use the keyword *none* to unset a previously set value. You may shortly
refer to this option as *pty/x*.

## pty/col

[[ type
:[ *ushort*
|  default
:  *<inherit>*
|  static
:  *yes*

Set column size for PTY sandboxing. Default is to inherit the
window-size. Use the keyword *none* to unset a previously set value.
You may shortly refer to this option as *pty/y*.

## sandbox/walk

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Walk sandboxing *on* or *off*.

## sandbox/stat

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Stat sandboxing *on* or *off*.

## sandbox/read

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Read sandboxing *on* or *off*.

## sandbox/write

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Write sandboxing *on* or *off*.

## sandbox/exec

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Exec sandboxing *on* or *off*.

## sandbox/ioctl

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Ioctl sandboxing *on* or *off*.

As of version 3.36.0, _ioctl_(2) requests to block devices are always
denied, and _ioctl_(2) requests to magic links are denied unless
*trace/allow_unsafe_magiclinks:true* is set.

## sandbox/create

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Create sandboxing *on* or *off*.

## sandbox/delete

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Delete sandboxing *on* or *off*.

## sandbox/rename

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Rename sandboxing *on* or *off*.

## sandbox/symlink

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Symlink sandboxing *on* or *off*.

## sandbox/truncate

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Truncate sandboxing *on* or *off*.

## sandbox/chdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chdir sandboxing *on* or *off*.

## sandbox/readdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Readdir sandboxing *on* or *off*.

## sandbox/mkdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkdir sandboxing *on* or *off*.

## sandbox/rmdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Rmdir sandboxing *on* or *off*.

## sandbox/chown

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chown sandboxing *on* or *off*.

## sandbox/chgrp

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chgrp sandboxing *on* or *off*.

## sandbox/chmod

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chmod sandboxing *on* or *off*.

## sandbox/chattr

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chattr sandboxing *on* or *off*.

## sandbox/chroot

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chroot sandboxing *on* or *off*.

## sandbox/utime

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Utime sandboxing *on* or *off*.

## sandbox/mkdev

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkdev sandboxing *on* or *off*.

## sandbox/mkfifo

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkfifo sandboxing *on* or *off*.

## sandbox/mktemp

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mktemp sandboxing *on* or *off*.

## sandbox/net

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Network sandboxing *on* or *off*.

## sandbox/lock

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Turn Landlock sandboxing *on* or *off*.

## sandbox/force

[[ type
:[ *boolean*
|  query
:  *yes*

Turn Force sandboxing *on* or *off*.

## sandbox/tpe

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Trusted Path Execution (TPE) sandboxing *on* or *off*.

## sandbox/crypt

[[ type
:[ *boolean*
|  default
:  *off*
|  query
:  *yes*

Turn Crypt sandboxing *on* or *off*.

To set this option *on*, a key must have already been specified with
*crypt/key* or _syd_(1) will exit with the _errno_(3) *ENOKEY*.

Note, setting this sandboxing type to *on* implies
*trace/allow_safe_kcapi:true* to allow cryptographic operations using the
Kernel Cryptography API (KCAPI).

Note, setting this sandboxing type to *on* implies
*trace/exit_wait_all:true* so as not to leave any ongoing encryption
processes behind on sandbox process exit.

## sandbox/proxy

[[ type
:[ *boolean*
|  default
:  *off*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Turn Proxy sandboxing *on* or *off*.

Defaults to proxying through TOR. See the options "proxy/addr",
"proxy/port", "proxy/ext/host", and "proxy/ext/port" to configure a
different proxy.

Implies *unshare/net:true*.

Requires _syd-tor_(1) helper utility to be under PATH. _syd-tor_(1) is
executed once at startup, it runs as a single process and this process
runs at most as long as the owner Syd process. See the _syd-tor_(1)
manual page for more information.

## sandbox/pty

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Turn PTY sandboxing *on* or *off*.

Requires _syd-pty_(1) helper utility to be under PATH. _syd-pty_(1) is
executed once at startup, it runs as a single process and this process
runs at most as long as the owner Syd process. See the _syd-pty_(1)
manual page for more information. Note, this option has no effect unless
both standard input and standard output are attached to a TTY at
startup.

## sandbox/mem

[[ type
:[ *boolean*
|  default
:  *off*
|  query
:  *yes*
|  static
:  *yes*

Turn Memory sandboxing *on* or *off*.

For performance reasons, this only works at startup. If not given at startup,
Syd will just allow _brk_(2), _mmap_(2), _mmap2_(2), and _mremap_(2) system
calls at seccomp-bpf level. Turning this sandboxing off during runtime is still
possible, in this case the respective system calls handlers will do nothing and
just continue the calls.

## sandbox/pid

[[ type
:[ *boolean*
|  default
:  *off*
|  query
:  *yes*

Turn PID sandboxing *on* or *off*.

## default/walk

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Walk sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/stat

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Stat sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/read

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Read sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/write

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Write sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/exec

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Exec sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/ioctl

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Ioctl sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/create

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Create sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/delete

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Delete sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/rename

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Rename sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/symlink

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Symlink sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/truncate

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Truncate sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/readdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Readdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/rmdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Rmdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chown

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chown sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chgrp

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chgrp sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chmod

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chmod sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chattr

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chattr sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chroot

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chattr sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/utime

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Utime sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkdev

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkdev sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkfifo

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkfifo sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mktemp

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mktemp sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/net

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Network sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/block

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for IP blocklist violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny".

## default/force

[[ type
:[ *string*
|  default
:  *deny*

For force sandboxing, define the default action to take when the path of
a binary is not in the Integrity Force map.

The value must be either one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", "exit", where the default is "deny".

## default/segvguard

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for SegvGuard access violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny".

## default/tpe

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for TPE sandboxing access violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny".

## default/mem

[[ type
:[ *string*
|  default
:  *kill*

Specify the action for Memory sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "kill".

## default/pid

[[ type
:[ *string*
|  default
:  *kill*

Specify the action for PID sandboxing access violations.

The value must be either one of "warn", "filter", "stop", "abort",
"kill", "exit", where the default is "kill".

## default/lock

[[ type
:[  *string*
|  default
:  *kill*
|  static
:  *yes*

Specify the compatibility level for Lock sandboxing.

The value must be either one of *kill*, *deny*, *warn*. *kill*
stands for the Landlock compatibility level *hard-requirement*,
whereas *deny* stands for *soft-requirement* and *warn* stands
for *best-effort*.

As of version 3.35.0, the default level has been promoted from *warn* to *kill*
to adhere to the principle of secure defaults. Again, as of this version
*ENOENT*, i.e. *No such file or directory* errors are fatal unless compatibility
level is set to *best-effort* at startup using *default/lock:warn*.

For more information on Landlock compatibility levels, see:
https://landlock.io/rust-landlock/landlock/trait.Compatible.html

## unshare/mount

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create Mount namespace on startup, implies *unshare/pid:true*.

## unshare/uts

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create UTS namespace on startup.

## unshare/ipc

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create IPC namespace on startup.

## unshare/user

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create User namespace on startup.

## unshare/pid

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create Pid namespace on startup, implies *unshare/mount:true*.

Syd mounts private _procfs_(5) in this mode.

As of version 3.37.2 _procfs_(5) is mounted with *hidepid=4* option which is Linux>=5.8.

As of version 3.39.0 _procfs_(5) is mounted with *subset=pid* option which is
Linux>=5.8, unless *trace/allow_unsafe_proc_files:true* is set at startup.

## unshare/net

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create Net namespace on startup.

## unshare/cgroup

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create CGroup namespace on startup.

## unshare/time

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*
|  oci
:  *no*

Create Time namespace on startup. Syd resets the boot-time clock
such that _uptime_(1) will report container uptime rather than
host uptime. Use _time_ command to override default and set
alternative time.

## root

[[ type
:[ *string*
|  static
:  *yes*
|  oci
:  *no*

Change the root mount to the given new root directory at startup using
_pivot_root_(2). Destination path arguments of "bind" commands are
interpreted relative to this directory. The directories "$root/dev", and
"$root/proc" _must_ exist to mount private filesystems. In addition,
target paths of the "bind" commands _must_ also be manually created by
the user.

This option does nothing without "unshare/mount:1".

As of version 3.23.14, symbolic links are not followed in any part of
the root directory and path traversal using ".." is not permitted. In
addition, root directory must be an absolute path, relative paths are
not permitted.

As of version 3.35.0, the special keyword *tmpfs* is supported to make
Syd create a temporary, private new root directory with the path
"/tmp/syd.XXXXXX" where the last 6 characters are replaced by random
characters. See _mkdtemp_(3) for more information. Syd uses this
directory to mount a _tmpfs_(5) filesystem over the new root filesystem.
In this mode, Syd is going to attempt to create target paths inside
the private temporary filesystem.

As of version 3.35.2, the special keywords *none* and *off* may be used
to unset a previously set _root_ directory.

## root/map

[[ type
:[ *boolean*
|  static
:  *yes*
|  oci
:  *no*

Map current user to root in the sandbox on startup.

This option does nothing without "unshare/user:1".

## root/fake

[[ type
:[ *boolean*
|  static
:  *yes*

In *fakeroot* mode, the system will return a user/group id of *0*,
mimicking the *root* user. This allows users to execute commands with
apparent root privileges, without actual superuser rights. It's useful
for tasks like package building where root-like environment is needed,
but not actual root permissions.

## time

[[ type
:[ *time_t*
|  static
:  *yes*
|  oci
:  *no*

Set clock monotonic and boottime offset (seconds) in Time Namespace.

## uts/host

[[ type
:[ *string*
|  default
:  *localhost*

Set UTS host name in the sandbox. Name is limited to 64 characters.
Name may be empty. Name may not have nul bytes. Default is *localhost*.

Useful when combined with *unshare/uts:true*. As of version 3.40.0, the
value of this option is returned at _uname_(2) boundary in *nodename*
field of the *utsname* structure regardless of the *unshare/uts* option.

## uts/domain

[[ type
:[ *string*
|  default
:  *(none)*

Set NIS/YP domain name in the sandbox. Name is limited to 64 characters.
Name may be empty. Name may not have nul bytes. Default is *(none)*.

Useful when combined with *unshare/uts:true*. As of version 3.40.0, the
value of this option is returned at _uname_(2) boundary in *domainname*
field of the *utsname* structure regardless of the *unshare/uts* option.

## uts/version

[[ type
:[ *string*
|  default
:  *<empty>*

Set version level of the operating system as returned in *version* field
of the *utsname* structure at _uname_(2) boundary. Name is limited to 64
characters. Name may be empty. Name may not have nul bytes. Default is
the empty string.

## ioctl/allow

[[ type
:[ *integer or string*
|  default
:  *[...]*

Add to or remove a request from the _ioctl_(2) request allowlist.
Accepts an unsigned 64-bit integer as argument. Prefix with *0x* for
hexadecimal and *0o* for octal input. Use *ioctl/allow+<request>* to add
to, and *ioctl/allow-<request>* to remove from the allowlist. As of
version 3.38.0, ioctl(2) requests may also be specified by
case-insensitive name and multiple requests may be added or removed by
separating them as a comma-delimited list. Specifying ioctl(2) requests
by name is strongly recommended because request numbers may vary by
architecture which is handled transparently when the request is
specified as a name. As of version 3.38.6, the _ioctl_(2) name may be
prepended with an optional exclamation mark, i.e. *!*, to denote Syd
should not return EINVAL ("Invalid argument") _errno_(3) in case the
name is not defined for any of the current supported architectures. This
allows for writing rules generic across multiple incompatible
architectures.

By default the list contains the _ioctl_(2) requests FIOCLEX, FIONCLEX,
FIONBIO, FIONREAD, FIOASYNC, FIOQSIZE, FIFREEZE, FITHAW, FS_IOC_FIEMAP,
FIGETBSZ, FICLONE, FICLONERANGE, FIDEDUPERANGE, FS_IOC_GETFSUUID,
FS_IOC_GETFSSYSFSPATH, and RNDGETENTCNT.

For rules added at startup deny rules have precedence over allow rules
because the denylist is checked at kernel-space, whereas the allowlist
is checked at user-space. For rules added after startup, the last
matching rule wins.

## ioctl/deny

[[ type
:[ *integer or string*
|  default
:  *[...]*
|  static
:  *add is dynamic, remove is partly static*

Add to or remove a request from the _ioctl_(2) request denylist. Accepts
an unsigned 64-bit integer as argument. Prefix with *0x* for hexadecimal
and *0o* for octal input. Use *ioctl/deny+<request>* to add to, and
*ioctl/deny-<request>* to remove from the allowlist. As of version
3.38.0, ioctl(2) requests may also be specified by case-insensitive name
and multiple requests may be added or removed by separating them as a
comma-delimited list. Specifying ioctl(2) requests by name is strongly
recommended because request numbers may vary by architecture which is
handled transparently when the request is specified as a name. As of
version 3.38.6, the _ioctl_(2) name may be prepended with an optional
exclamation mark, i.e. *!*, to denote Syd should not return EINVAL
("Invalid argument") _errno_(3) in case the name is not defined for any
of the current supported architectures. This allows for writing rules
generic across multiple incompatible architectures.

By default the list of denylisted _ioctl_(2) requests are FIBMAP,
FS_IOC_FSGETXATTR, FS_IOC_FSSETXATTR, FS_IOC_SETFLAGS, KDSETKEYCODE,
KDSIGACCEPT, RNDADDTOENTCNT, RNDGETPOOL, RNDADDENTROPY, RNDZAPENTCNT,
RNDCLEARPOOL, SECCOMP_IOCTL_NOTIF_RECV, SECCOMP_IOCTL_NOTIF_SEND,
SECCOMP_IOCTL_NOTIF_ID_VALID, SECCOMP_IOCTL_NOTIF_ADDFD,
SECCOMP_IOCTL_NOTIF_SET_FLAGS, TIOCCONS, TIOCLINUX, TIOCSETD, and
TIOCSTI.

For security reasons, the _ioctl_(2) denylist is applied at the parent
seccomp-bpf filter at startup. This means the Syd process is included in
this restriction as well. This also means, removing elements from this
list after startup has no effect. However, if Ioctl sandboxing was
enabled at startup, adding new elements to the _ioctl_(2) denylist will
further restrict the _ioctl_(2) request space.

For rules added at startup, deny rules have precedence over allow rules
because the denylist is checked at kernel-space, whereas the allowlist
is checked at user-space. For rules added after startup, the last
matching rule wins.

Further reading about denylisted _ioctl_(2) requests:

- https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2023-1523
- https://a13xp0p0v.github.io/2017/03/24/CVE-2017-2636.html
- http://phrack.org/issues/52/6.html#article
- https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=83efeeeb3d04b22aaed1df99bc70a48fe9d22c4d
- https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=8d1b43f6a6df7bcea20982ad376a000d90906b42
- https://seclists.org/oss-sec/2024/q1/13
- https://seclists.org/oss-sec/2024/q1/14
- https://forums.grsecurity.net/viewtopic.php?f=7&t=2522
- http://lkml.indiana.edu/hypermail/linux/kernel/9907.0/0132.html
- http://linux.derkeiler.com/Mailing-Lists/Kernel/2007-11/msg07723.html

## mem/max

[[ type
:[ *positive integer (u64)*
|  default
:  *256MB*

This setting specifies the limit on per-process memory usage. Setting this value
to *0* disables testing for this type of memory usage. Note, the value is parsed
using the *parse-size* crate. Refer to their documentation for information on
formatting.

## mem/vm_max

[[ type
:[ *positive integer (u64)*
|  default
:  *0*

This setting specifies the limit on per-process virtual memory usage. Setting
this value to *0* disables testing for this type of memory usage. Note, the
value is parsed using the *parse-size* crate. Refer to their documentation for
information on formatting.

## pid/max

[[ type
:[ *positive integer (u64)*
|  default
:  *1*

This setting specifies the limit on the number of running tasks for pid
sandboxing. Setting this value to *0* is functionally equivalent to setting
*sandbox/pid* to *off*.

As of version 3.40.0, when *unshare/pid:true* is set, PID sandboxing
counts and enforces the limit in the current PID namespace; on Linux
6.14 and newer, to account for the kernel's 300 reserved PIDs the
namespaced kernel.pid_max is set to *max(pid/max, 301)*, while on older
kernels *kernel.pid_max* _sysctl_(8) is left unchanged.

## bind

[[ type
:[ *string-vec*
|  static
:  *yes*
|  oci
:  *no*

This command causes Syd to bind mount a directory on startup. The
format is *source-dir:target-dir:mount-options,...* where _the source and
target directories may be equal_. Mount options are a comma-separated list
of a combination of the following options:

- *ro* to mount the filesystem read-only.
- *nodev* to not interpret character or block special devices on the filesystem.
- *noexec* to not permit direct execution of any binaries on the mounted \
filesystem.
- *nosuid* to not honour set-user-ID and set-group-ID bits or file capabilities \
when executing programs from this filesystem. In addition, SELinux domain \
transitions require permission *nosuid_transition*, which in turn needs also \
policy capability *nnp_nosuid_transition*.
- *nosymfollow* to not follow symbolic links when resolving paths. \
Symbolic links can still be created, and _readlink_(1), \
_readlink_(2), _realpath_(1), and _realpath_(3) all still work \
properly.
- *noatime* to not update inode access times on this filesystem (e.g. for \
faster access on the news spool to speed up news servers). This works for all \
inode types (directories too), so it implies *nodiratime*.
- *nodiratime* to not update directory inode access times on this filesystem. \
(This option is implied when *noatime* is set.)
- *relatime* to update inode access times relative to modify or change time.

Mount options may be omitted. If the source directory is not an absolute
path, it is interpreted as the filesystem type rather than the source
directory. This may be used to mount special filesystems such as
cgroupfs, overlayfs or _tmpfs_(5) into the mount namespace. In this
case, any mount options supported by this filesystem type may be
submitted in options argument not just the ones listed above. You may
find some examples below:

- bind+/:/:ro
- bind+tmpfs:/tmp:noexec,size=16M
- bind+cgroup2:/sys/fs/cgroup:nodev,noexec,nosuid
- bind+overlay:/tmp/target:lowerdir=/tmp/lower,upperdir=/tmp/upper,workdir=/tmp/work,nosuid

This option does nothing without *unshare/mount:true*.

This command may be used to create immutable containers. For example,
the command *bind+/:/:ro* is functionally equivalent to
*deny/write+/\*\*\** except the restriction happens at kernel VFS layer
rather than at user level using _seccomp_(2) notify. Alternatively this
can also be achieved at the kernel level using _landlock_(7).

As of version 3.23.14, symbolic links are not followed in any part of
the source or target directory paths and path traversal using *..* is
not permitted. In addition, target directory must be an absolute path,
relative paths are not permitted.

As of version 3.23.14, mounting the special _proc_(5) filesystem under a
custom path is not permitted. Syd handles this mount itself specially
after all bind mounts are processed.

## crypt

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to encrypt for *Crypt sandboxing*.

## crypt/key

[[ type
:[ *i32*
|  static
:  *yes*

Specify _keyrings_(7) IDs of the 256-bit AES-CTR encryption key and
HMAC-SHA256 authentication key for *Crypt sandboxing*. The ID must be a
32-bit integer. To set encryption and authentication keys separately,
use the options *crypt/key/enc*, and *crypt/key/mac*. This option is a
shorthand to set both at the same time to the same key serial ID.

Setting an encryption key with this option implies *sandbox/crypt:on*.

Session keyring must be attached to the user keyring or this will fail
at startup with the EKEYREVOKED ("Key has been revoked") _errno_(3).
Use the _syd-key_(1) utility to safely generate a key and save to
_keyrings_(7) interface.

## crypt/key/enc

[[ type
:[ *i32*
|  static
:  *yes*

Specify _keyrings_(7) ID of the 256-bit AES-CTR encryption key for
*Crypt sandboxing*. The ID must be a 32-bit integer.

Setting an encryption key with this option implies *sandbox/crypt:on*.

Session keyring must be attached to the user keyring or this will fail
at startup with the EKEYREVOKED ("Key has been revoked") _errno_(3).
Use the _syd-key_(1) utility to safely generate a key and save to
_keyrings_(7) interface.

## crypt/key/mac

[[ type
:[ *i32*
|  static
:  *yes*

Specify _keyrings_(7) ID of the 256-bit HMAC-SHA256 authentication key
for *Crypt sandboxing*. The ID must be a 32-bit integer.

Setting an encryption key with this option implies *sandbox/crypt:on*.

Session keyring must be attached to the user keyring or this will fail
at startup with the EKEYREVOKED ("Key has been revoked") _errno_(3).
Use the _syd-key_(1) utility to safely generate a key and save to
_keyrings_(7) interface.

## crypt/tmp

[[ type
:[ *string*
|  default
:  mem
|  static
:  *yes*

Specify temporary backing directory for transparent file decryption. The
argument must be an absolute path or the special value *mem*. The user
must ensure this directory is secure as decrypted contents will be
written to temporary files under this directory. Specify the special
value *mem* to use anonymous files which live in RAM with a volatile
backing storage created with _memfd_create_(2). This is the default. The
user is encouraged to specify this option for efficient handling of
large files for *Crypt sandboxing*.

Setting this option implies *sandbox/crypt:on*.

## force

[[ type
:[ *string-vec*

Add or remove an integrity force rule for Force Sandboxing. The format
is _force+/path:hashhex:action_ for addition and _force-/path_ for
removal. Use _force^_ to clear the Integrity Force map. Available
actions are "warn", "filter", "deny", "panic", "stop", "abort", "kill"
and "exit" where the default is "deny". hashhex is either a 8-character
CRC32 checksum, 16-character CRC64 checksum, 32-character MD5 checksum,
a 40-character SHA1 checksum, a 64-character SHA3-256 checksum, a
96-character SHA3-384 checksum or a 128-character SHA3-512 checksum.

- _syd-sha_(1) is a helper tool to calculate checksums of files.
- _syd-path_(1) is a helper tool to write integrity force rules for binaries under PATH.

## proxy/addr

[[ type
:[ *IP address*
|  default
:  *127.0.0.1*
|  static
:  *yes*
|  oci
:  *no*

Set internal address for Proxy sandboxing. This must be an IPv4 or an
IPv6 address. Defaults to 127.0.0.1.

## proxy/port

[[ type
:[ *integer*
|  default
:  *9050*
|  static
:  *yes*
|  oci
:  *no*

Set internal port for Proxy sandboxing. Defaults to 9050.

## proxy/ext/host

[[ type
:[ *Hostname or IP*
|  default
:  *127.0.0.1*
|  static
:  *yes*
|  oci
:  *no*

Set external address for Proxy sandboxing. This must either be an IPv4
address or an IPv6 address or a hostname. If the argument does not parse
as an IP address, Syd resolves the name using the system DNS resolver
and selects a response IP randomly.

Defaults to "127.0.0.1", which may be be overriden with the environment variable
*SYD_PROXY_HOST* at startup.

## proxy/ext/port

[[ type
:[ *integer*
|  default
:  *9050*
|  static
:  *yes*
|  oci
:  *no*

Set external port for Proxy sandboxing.

Defaults to 9050, which may be overriden with the environment variable
*SYD_PROXY_PORT* at startup.

## proxy/ext/unix

[[ type
:[ *string*
|  static
:  *yes*
|  oci
:  *no*

Set external UNIX domain socket for Proxy sandboxing.

The argument may also be set using the environment variable
*SYD_PROXY_UNIX* at startup.

This option has precedence over the option "proxy/ext/host", ie. when
both are given Syd will connect to the UNIX domain socket.

## segvguard/expiry

[[ type
:[ *integer (u64)*
|  default
:  120

Specify SegvGuard expiry timeout in seconds. Set to 0 to disable SegvGuard.

## segvguard/suspension

[[ type
:[ *integer (u64)*
|  default
:  600

Specify SegvGuard suspension timeout in seconds.

## segvguard/maxcrashes

[[ type
:[ *integer (u8)*
|  default
:  5

Specify SegvGuard max crashes.

## tpe/gid

[[ type
:[ *integer (gid_t)*

Specify untrusted GID for Trusted Path Execution (TPE). By default, TPE
is applied to users of all groups including root and this setting can be
used to limit it to a certain group. To unset a previously set GID and
return to the default state set "none" as the value.

## tpe/negate

[[ type
:[ *boolean*

Negate GID logic for Trusted Path Execution (TPE). This turns "tpe/gid"
from untrusted into trusted such that users belonging to this group will
be exempt from TPE.

## tpe/root_owned

[[ type
:[ *boolean*

Ensure file and parent directory are root-owned for Trusted Path
Execution (TPE).

Note, this option will misbehave with "unshare/user:1" if the real root
user is not mapped inside the container.

## tpe/user_owned

[[ type
:[ *boolean*

Ensure file and parent directory are user-owned or root-owned for
Trusted Path Execution (TPE).

Note, this option may misbehave with "unshare/user:1" if the real root
user is not mapped inside the container.

## tpe/root_mount

[[ type
:[ *boolean*

Ensure file and parent directory are on root filesystem for Trusted Path
Execution (TPE).

This option may be used to pin all executions to a single safe mountpoint.

## allow/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Walk sandboxing*.

## allow/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Stat sandboxing*.

## allow/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Read sandboxing*.

## allow/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Write sandboxing*.

## allow/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Exec sandboxing*.

## allow/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Ioctl sandboxing*.

## allow/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Create sandboxing*.

## allow/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Delete sandboxing*.

## allow/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Rename sandboxing*.

## allow/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Symlink sandboxing*.

## allow/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Truncate sandboxing*.

## allow/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chdir sandboxing*.

## allow/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Readdir sandboxing*.

## allow/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Mkdir sandboxing*.

## allow/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Rmdir sandboxing*.

## allow/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chown sandboxing*.

## allow/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chgrp sandboxing*.

## allow/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chmod sandboxing*.

## allow/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chattr sandboxing*.

## allow/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Chroot sandboxing*.

## allow/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Utime sandboxing*.

## allow/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Mkdev sandboxing*.

## allow/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Mkfifo sandboxing*.

## allow/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to allow for *Mktemp sandboxing*.

## allow/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to allow for *Bind network sandboxing*.

## allow/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to allow for *Accept network sandboxing*.

## allow/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to allow for *Connect network sandboxing*.

## allow/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to allow for *SendFd network sandboxing*.

## allow/net/link

[[ type
:[ *string-vec*
|  static
:  *yes*

Specifies a list of netlink families to allow for *Link network sandboxing*.

Accepts a comma-delimited list of the following items: *route*,
*usersock*, *firewall*, *sock_diag*, *nflog*, *xfrm*, *selinux*,
*iscsi*, *audit*, *fib_lookup*, *connector*, *netfilter*, *ip6_fw*,
*dnrtmsg*, *kobject_uevent*, *generic*, *scsitransport*, *ecryptfs*,
*rdma*, *crypto*, and *smc*. Use *all* to specifiy all families.

## allow/lock/read

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null", "/proc")_

Specifies a set of beneath paths to grant file read access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_READ_FILE" and only applies to the content of the
directory not the directory itself. As of version 3.21.0, this set
includes the paths "/dev/null" and "/proc" by default as Syd is included
in the Landlock sandbox and Syd requires read access to these paths to
function correctly.

## allow/lock/write

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null")_

Specifies a set of beneath paths to grant file write access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_WRITE_FILE" and only applies to the content of the
directory not the directory itself. As of version 3.21.0, this set
includes the path "/dev/null" by default as Syd is included in the
Landlock sandbox and Syd requires write access to this file to function
correctly.

## allow/lock/exec

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file execute access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_EXECUTE" and only applies to the content of the
directory not the directory itself.

## allow/lock/ioctl

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant _ioctl_(2) access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_IOCTL_DEV" and only applies to the content of the
directory not the directory itself. Landlock _ioctl_(2) support requires
ABI 5 or later. Fifth Landlock ABI was introduced with Linux 6.10. On
older kernels, this command is a no-op and is not going to confine
_ioctl_(2) operations.

## allow/lock/create

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file creation, rename and link
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_REG" and only applies to the
content of the directory not the directory itself.

## allow/lock/delete

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file unlink, rename and link
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_REMOVE_FILE" and only applies to the
content of the directory not the directory itself.

## allow/lock/rename

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant access to link or rename a
file from or to a different directory (i.e. reparent a file hierarchy)
for *Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_REFER" and only applies to the content of the
directory not the directory itself. Landlock rename support requires ABI
2 or later. Second Landlock ABI was introduced with Linux 5.19. On older
kernels, this type of access is always denied with Landlock.

## allow/lock/symlink

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant symbolic link creation, rename
and link access for *Lock sandboxing*. This category corresponds to the
Landlock access right "LANDLOCK_ACCESS_FS_MAKE_SYM" and only applies
to the content of the directory not the directory itself.

## allow/lock/truncate

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null")_

Specifies a set of beneath paths to grant file truncation access for
*Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_TRUNCATE" and only applies to the content of
the directory not the directory itself. Landlock file truncation support
requires ABI 3 or later. Third Landlock ABI was introduced with Linux
6.2. On older kernels, this command is a no-op and is not going to
confine file truncation operations. As of version 3.21.0, this set
includes the path "/dev/null" by default as Syd is included in the
Landlock sandbox and Syd requires truncation access to this file to
function correctly.

## allow/lock/readdir

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/proc")_

Specifies a set of beneath paths to grant directory list access for
*Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_READ_DIR" and applies to the directory
and the directories beneath it. As of version 3.21.0, this set includes
the directory "/proc" by default as Syd is included in the Landlock
sandbox and Syd requires readdir access to this directory to function
correctly.

## allow/lock/mkdir

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant directory creation and rename
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_DIR" and only applies to the
content of the directory not the directory itself.

## allow/lock/rmdir

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant directory deletion and rename
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_REMOVE_DIR" and only applies to the
content of the directory not the directory itself.

## allow/lock/mkdev

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant character device creation
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_CHAR" and only applies to the
content of the directory not the directory itself.

## allow/lock/mkfifo

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant named pipe (FIFO) creation
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_FIFO" and only applies to the
content of the directory not the directory itself.

## allow/lock/bind

[[ type
:[ *(u16-set, string-set)*
|  static
:  *yes*

Specifies a list of allowed _bind_(2) ports and UNIX domain socket paths
for *Lock sandboxing*. This category corresponds to the Landlock access
rights "LANDLOCK_ACCESS_NET_BIND_TCP" and "LANDLOCK_ACCESS_FS_MAKE_SOCK"
and only applies to the content of the directory not the directory
itself. Argument is either a single port or a closed range in format
*port1-port2*, or an absolute UNIX domain socket path. Landlock network
support requires ABI 4 or later. Fourth Landlock ABI was introduced with
Linux 6.7. On older kernels, this command is a no-op when specified with
port arguments and does not do any network confinement.

## allow/lock/connect

[[ type
:[ *u16-set*
|  static
:  *yes*

Specifies a list of allowed _connect_(2) ports for *Lock sandboxing*.
This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_NET_BIND_CONNECT". Argument is either a single port or
a closed range in format *port1-port2*. Landlock network support
requires ABI 4 or later. Fourth Landlock ABI was introduced with Linux
6.7. On older kernels, this command is a no-op and does not do any
network confinement.

## warn/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Walk sandboxing*.

## warn/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Stat sandboxing*.

## warn/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Read sandboxing*.

## warn/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Write sandboxing*.

## warn/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Exec sandboxing*.

## warn/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Ioctl sandboxing*.

## warn/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Create sandboxing*.

## warn/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Delete sandboxing*.

## warn/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Rename sandboxing*.

## warn/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Symlink sandboxing*.

## warn/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Truncate sandboxing*.

## warn/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chdir sandboxing*.

## warn/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Readdir sandboxing*.

## warn/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Mkdir sandboxing*.

## warn/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Rmdir sandboxing*.

## warn/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chown sandboxing*.

## warn/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chgrp sandboxing*.

## warn/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chmod sandboxing*.

## warn/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chattr sandboxing*.

## warn/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Chroot sandboxing*.

## warn/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Utime sandboxing*.

## warn/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Mkdev sandboxing*.

## warn/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Mkfifo sandboxing*.

## warn/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to warn for *Mktemp sandboxing*.

## warn/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to warn for *Bind network sandboxing*.

## warn/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to warn for *Accept network sandboxing*.

## warn/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to warn for *Connect network sandboxing*.

## warn/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to warn for *SendFd network sandboxing*.

## deny/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Walk sandboxing*.

## deny/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Stat sandboxing*.

## deny/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Read sandboxing*.

## deny/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Write sandboxing*.

## deny/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Exec sandboxing*.

## deny/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Ioctl sandboxing*.

## deny/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Create sandboxing*.

## deny/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Delete sandboxing*.

## deny/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Rename sandboxing*.

## deny/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Symlink sandboxing*.

## deny/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Truncate sandboxing*.

## deny/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chdir sandboxing*.

## deny/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Readdir sandboxing*.

## deny/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Mkdir sandboxing*.

## deny/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Rmdir sandboxing*.

## deny/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chown sandboxing*.

## deny/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chgrp sandboxing*.

## deny/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chmod sandboxing*.

## deny/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chattr sandboxing*.

## deny/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Chroot sandboxing*.

## deny/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Utime sandboxing*.

## deny/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Mkdev sandboxing*.

## deny/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Mkfifo sandboxing*.

## deny/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to deny for *Mktemp sandboxing*.

## deny/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to deny for *Bind network sandboxing*.

## deny/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to deny for *Accept network sandboxing*.

## deny/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to deny for *Connect network sandboxing*.

## deny/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to deny for *SendFd network sandboxing*.

## panic/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Walk sandboxing*.

## panic/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Stat sandboxing*.

## panic/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Read sandboxing*.

## panic/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Write sandboxing*.

## panic/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Exec sandboxing*.

## panic/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Ioctl sandboxing*.

## panic/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Create sandboxing*.

## panic/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Delete sandboxing*.

## panic/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Rename sandboxing*.

## panic/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Symlink sandboxing*.

## panic/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Truncate sandboxing*.

## panic/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chdir sandboxing*.

## panic/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Readdir sandboxing*.

## panic/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Mkdir sandboxing*.

## panic/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Rmdir sandboxing*.

## panic/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chown sandboxing*.

## panic/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chgrp sandboxing*.

## panic/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chmod sandboxing*.

## panic/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chattr sandboxing*.

## panic/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Chroot sandboxing*.

## panic/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Utime sandboxing*.

## panic/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Mkdev sandboxing*.

## panic/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Mkfifo sandboxing*.

## panic/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to panic for *Mktemp sandboxing*.

## panic/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to panic for *Bind network sandboxing*.

## panic/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to panic for *Accept network sandboxing*.

## panic/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to panic for *Connect network sandboxing*.

## panic/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to panic for *SendFd network sandboxing*.

## stop/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Walk sandboxing*.

## stop/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Stat sandboxing*.

## stop/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Read sandboxing*.

## stop/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Write sandboxing*.

## stop/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Exec sandboxing*.

## stop/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Ioctl sandboxing*.

## stop/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Create sandboxing*.

## stop/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Delete sandboxing*.

## stop/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Rename sandboxing*.

## stop/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Symlink sandboxing*.

## stop/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Truncate sandboxing*.

## stop/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chdir sandboxing*.

## stop/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Readdir sandboxing*.

## stop/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Mkdir sandboxing*.

## stop/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Rmdir sandboxing*.

## stop/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chown sandboxing*.

## stop/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chgrp sandboxing*.

## stop/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chmod sandboxing*.

## stop/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chattr sandboxing*.

## stop/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Chroot sandboxing*.

## stop/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Utime sandboxing*.

## stop/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Mkdev sandboxing*.

## stop/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Mkfifo sandboxing*.

## stop/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to stop for *Mktemp sandboxing*.

## stop/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to stop for *Bind network sandboxing*.

## stop/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to stop for *Accept network sandboxing*.

## stop/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to stop for *Connect network sandboxing*.

## stop/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to stop for *SendFd network sandboxing*.

## abort/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Walk sandboxing*.

## abort/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Stat sandboxing*.

## abort/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Read sandboxing*.

## abort/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Write sandboxing*.

## abort/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Exec sandboxing*.

## abort/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Ioctl sandboxing*.

## abort/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Create sandboxing*.

## abort/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Delete sandboxing*.

## abort/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Rename sandboxing*.

## abort/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Symlink sandboxing*.

## abort/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Truncate sandboxing*.

## abort/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chdir sandboxing*.

## abort/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Readdir sandboxing*.

## abort/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Mkdir sandboxing*.

## abort/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Rmdir sandboxing*.

## abort/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chown sandboxing*.

## abort/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chgrp sandboxing*.

## abort/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chmod sandboxing*.

## abort/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chattr sandboxing*.

## abort/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Chroot sandboxing*.

## abort/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Utime sandboxing*.

## abort/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Mkdev sandboxing*.

## abort/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Mkfifo sandboxing*.

## abort/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to abort for *Mktemp sandboxing*.

## abort/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to abort for *Bind network sandboxing*.

## abort/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to abort for *Accept network sandboxing*.

## abort/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to abort for *Connect network sandboxing*.

## abort/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to abort for *SendFd network sandboxing*.

## kill/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Walk sandboxing*.

## kill/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Stat sandboxing*.

## kill/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Read sandboxing*.

## kill/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Write sandboxing*.

## kill/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Exec sandboxing*.

## kill/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Ioctl sandboxing*.

## kill/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Create sandboxing*.

## kill/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Delete sandboxing*.

## kill/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Rename sandboxing*.

## kill/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Symlink sandboxing*.

## kill/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Truncate sandboxing*.

## kill/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chdir sandboxing*.

## kill/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Readdir sandboxing*.

## kill/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Mkdir sandboxing*.

## kill/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Rmdir sandboxing*.

## kill/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chown sandboxing*.

## kill/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chgrp sandboxing*.

## kill/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chmod sandboxing*.

## kill/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chattr sandboxing*.

## kill/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Chroot sandboxing*.

## kill/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Utime sandboxing*.

## kill/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Mkdev sandboxing*.

## kill/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Mkfifo sandboxing*.

## kill/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to kill for *Mktemp sandboxing*.

## kill/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to kill for *Bind network sandboxing*.

## kill/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to kill for *Accept network sandboxing*.

## kill/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to kill for *Connect network sandboxing*.

## kill/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to kill for *SendFd network sandboxing*.

## exit/walk

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Walk sandboxing*.

## exit/stat

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Stat sandboxing*.

## exit/read

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Read sandboxing*.

## exit/write

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Write sandboxing*.

## exit/exec

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Exec sandboxing*.

## exit/ioctl

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Ioctl sandboxing*.

## exit/create

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Create sandboxing*.

## exit/delete

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Delete sandboxing*.

## exit/rename

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Rename sandboxing*.

## exit/symlink

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Symlink sandboxing*.

## exit/truncate

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Truncate sandboxing*.

## exit/chdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chdir sandboxing*.

## exit/readdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Readdir sandboxing*.

## exit/mkdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Mkdir sandboxing*.

## exit/rmdir

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Rmdir sandboxing*.

## exit/chown

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chown sandboxing*.

## exit/chgrp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chgrp sandboxing*.

## exit/chmod

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chmod sandboxing*.

## exit/chattr

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chattr sandboxing*.

## exit/chroot

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Chroot sandboxing*.

## exit/utime

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Utime sandboxing*.

## exit/mkdev

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Mkdev sandboxing*.

## exit/mkfifo

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Mkfifo sandboxing*.

## exit/mktemp

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to exit for *Mktemp sandboxing*.

## exit/net/bind

[[ type
:[ *string-vec*

Specifies a list of network address patterns to exit for *Bind network sandboxing*.

## exit/net/accept

[[ type
:[ *string-vec*

Specifies a list of network address patterns to exit for *Accept network sandboxing*.

## exit/net/connect

[[ type
:[ *string-vec*

Specifies a list of network address patterns to exit for *Connect network sandboxing*.

## exit/net/sendfd

[[ type
:[ *string-vec*

Specifies a list of network address patterns to exit for *SendFd network sandboxing*.

## append

[[ type
:[ *string-vec*

Specifies a list of _glob_(3p) patterns to files that should be made
append-only for *Write sandboxing*.

If a path is append-only, Syd adds *O_APPEND* and removes *O_TRUNC* from
flags on any sandbox granted attempt to _open_(2) this path. Unsetting
the *O_APPEND* flag using _fcntl_(2) *F_SETFL* command is prevented.
Similarly, any attempt to _rename_(2), _truncate_(2) and _unlink_(2) the
file is prevented. This is typically useful for history and log files.

## mask

[[ type
:[ *string-map*

Specifies a list of _glob_(3p) patterns to mask for *Read & Write sandboxing*.

If a path is masked, Syd returns a file descriptor to */dev/null* on any
sandbox granted attempt to _open_(2) this path. Masking can effectively
be used to hide the contents of a file in a more relaxed and compatible
way than denying read/write access to it. _stat_(2) calls on a masked
file returns the original file metadata and a masked file may be
executed. After a successful mask operation, the mask path is _not_
checked for sandbox access.

As of version 3.35.1, the default mask path */dev/null* may be changed
by specifying a colon-separated extra path to the mask-add command, e.g.
*mask+/dev/[fn]ull:/dev/zero* when both of the paths */dev/full* and
*/dev/null* will be masked with the path */dev/zero*. The mask path must
be a fully canonicalized path without symbolic links.

As of version 3.36.0, the default mask path may be overriden for
directories by specifying an additional colon-separated extra path to
the mask-add command, e.g. *mask+/proc/acpi/\*\*\*:/dev/null:/var/empty*
when the path */proc/acpi/wakeup* which is a regular file will return
*/dev/null* at _open_(2) boundary but the directory */proc/acpi* and any
subdirectory within will return */var/empty* at _open_(2) boundary. The
mask path must be a fully canonicalized path without symbolic links.

This feature provides a non-privileged alternative to the _bind_ command
because it does not require the creation of a mount namespace. Moreover,
_mask_ commands may be specified dynamically after startup using the
_syd_(2) API allowing for fine-tuned and/or incremental confinement.

## block

[[ type
:[ *ip-range*

Specifies a range of IP networks to be blocked when specified as the
target address of *connect* group system calls which are _connect_(2),
_sendto_(2), _sendmsg_(2), _sendmmsg_(2) and when received as the source
address in return from _accept_(2) and _accept4_(2) system calls for
IPv4 and IPv6 family sockets. Use *block+<net>* and *block-<net>* to add
and remove ip networks from the range. Alternatively the range can also
be populated by including *ipset* and *netset* files from within Syd
configuration. Use *block^* to clear the list and *block!* to simplify
the ip range by aggregating networks together. *block!* is useful to
call after importing big IP blocklists, it helps reduce memory
consumption and improve matching performance. Below is a configuration
snippet that imports Feodo and DShield blocklists:

```
# Enable IP blocklists
# Source: https://github.com/firehol/blocklist-ipsets.git
include /usr/src/blocklist-ipsets/feodo.ipset
include /usr/src/blocklist-ipsets/feodo_badips.ipset
include /usr/src/blocklist-ipsets/dshield.netset
include /usr/src/blocklist-ipsets/dshield_1d.netset
include /usr/src/blocklist-ipsets/dshield_30d.netset
include /usr/src/blocklist-ipsets/dshield_7d.netset
include /usr/src/blocklist-ipsets/dshield_top_1000.ipset
block!
```

## cmd/exec

[[ type
:[ *command*

Makes Syd execute an external command without sandboxing. The process is
executed in a new process group with its standard input attached to
*/dev/null*. Standard output and standard error file descriptors are
inherited. Syd also ensures no non-standard file descriptors leak into
the new process utilizing the _close_range_(2) system call. Current
working directory is changed to the root directory, i.e. */*. The
_umask_(2) is set to 077. The program name and arguments must be
separated with the *US* (unit separator, hex: 0x1f, octal: 037)
character. To ease usage, the _syd-exec_(1) helper utility is provided
to construct a sandbox command of this type:

```
; syd -puser -mlock:exec -- sh -c 'test -c $(syd-exec echo hello world)'
hello world
;
```

## load

[[ type
:[ *integer (fd) or string (profile-name)*

Read configuration from the given file descriptor, the file must be open for
reading. Syd uses _pidfd_getfd_(2) to acquire the file descriptor and reads
sandbox configuration from it. This command is useful to load a set of sandbox
commands into Syd in a single step and is typically used with *reset*, e.g:

```
int fd = open("/tmp", O_RDWR | O_TMPFILE | O_CLOEXEC, 0);
if (fd == -1) errx(1, "Failed to open temporary file");

const char *syd = "sandbox:stat/on\\nallow/stat+/***\\ndeny/stat+/\\nlock:on\\n";
errx(write(fd, syd, strlen(syd)) == -1, "Failed to write config");
errx(lseek(fd, 0, SEEK_SET) == -1, "Failed to seek in file");

char load[64];
sprintf(load, "/dev/syd/load/%d", fd);
errx(stat("/dev/syd/reset", NULL) == -1, "Failed to reset syd");
errx(stat(load, NULL) == -1, "Failed to load syd profile");

errx(execvp("/bin/sh", (char *[]){"/bin/sh", "-l", NULL}) == -1, "execvp failed");
```

Due to security reasons, this command is only available via the virtual
_stat_(2) call, it may not be used with the *-m* command line switch or
in a configuration file.

As of version 3.30.0, this command may be used to load builtin profiles,
when Syd falls back to parsing the "load" argument as a profile name if
parsing the argument as a file descriptor fails.

## trace/allow_safe_setuid

[[ type
:[ *boolean*
|  static
:  *yes*

Enable *SafeSetID* and retain the Linux capability *CAP_SETUID*. This
option is implied at startup if any UID transits were defined with the
*setuid* command. This feature allows Syd to change UID simultaneously
with the sandbox process. Because NPTL uses reserved signals to ensure
all threads share the same UID/GID, setting this option disables the
SROP mitigator. See the *Enhanced Execution Control (EEC)* section of
the _syd_(7) manual page for more information.

## trace/allow_safe_setgid

[[ type
:[ *boolean*
|  static
:  *yes*

Enable *SafeSetID* and retain the Linux capability *CAP_SETGID*. This
option is implied at startup if any GID transits were defined with the
*setuid* command. This feature allows Syd to change GID simultaneously
with the sandbox process. Because NPTL uses reserved signals to ensure
all threads share the same UID/GID, setting this option disables the
SROP mitigator. See the *Enhanced Execution Control (EEC)* section of
the _syd_(7) manual page for more information.

## setuid

[[ type
:[ *[(uid, uid)]*
|  static
:  *yes*

Add, remove a UID transition or reset UID transitions. Only a single
transition from a source UID can be defined. Target UID can not be lower
than the build default *11*, which is typically the *operator* user.
Defining a UID transit with this option implies
*trace/allow_safe_setuid:true*.

Usage:

```
setuid+0:65534      # Define a UID transition from root to nobody.
setuid+root:nobody  # Same as above but using user names.
setuid-0:65534      # Remove a previously defined UID transition.
setuid^0            # Remove all UID transitions matching source UID.
setuid^             # Remove all UID transitions.
```

## setgid

[[ type
:[ *[(gid, gid)]*
|  static
:  *yes*

Add, remove a GID transition or reset GID transitions. Only a single
transition from a source GID can be defined. Target GID can not be lower
than the build default *14*, which is typically the *uucp* user.
Defining a GID transit with this option implies
*trace/allow_safe_setgid:true*.

Usage:

```
setgid+0:65534      # Define a GID transition from root to nogroup.
setgid+root:nogroup # Same as above but using group names.
setgid-0:65534      # Remove a previously defined GID transition.
setgid^0            # Remove all GID transitions matching source GID.
setgid^             # Remove all GID transitions.
```

## trace/allow_unsafe_cbpf

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether Syd should allow additional _seccomp_(2)
cbpf filters to be installed by sandbox processes. By default, this is
denied to mitigate confused deputy problems and _errno_(3) is set
to *EINVAL*, i.e. *Invalid argument*, for compatibility reasons. On the one
hand, stacked _seccomp_(2) cbpf filters allow for incremental
confinement and therefore added hardening, on the other hand they may be
abused to install system call filters with more precedent actions than
user-notify thereby bypassing Syd's own _seccomp_(2) cbpf filters. To
quote the _seccomp_unotify_(2): "... a user-space notifier can be
bypassed if the existing filters allow the use of _seccomp_(2) or
_prctl_(2) to install a filter that returns an action value with a
higher precedence than *SECCOMP_RET_USER_NOTIF* (see _seccomp_(2))."
Setting the option *trace/allow_unsafe_prctl:true* overrides this option
and allows the *PR_SET_SECCOMP* _prctl_(2) operation inside the sandbox.
This may be changed in the future for clearer separation of mitigations.

## trace/allow_unsafe_ebpf

[[ type
:[ *boolean*
|  static
:  *yes*

Allows direct eBPF use inside the Syd sandbox using the _bpf_(2) system
call, whose unprivileged use is permitted since Linux-4.4. On the one
hand, eBPF programs can be used for additional hardening, on the other
hand eBPF is a frequent source of vulnerabilities due to churn,
complexity, improper validation and complexity of validation. eBPF may
also be abused to implement efficient and portable rootkits.

Note, as of version 3.32.8, Syd includes the _uretprobe_(2) system call
into this mitigation. _uretprobe_(2) system call is implemented in
Linux-6.11 or newer and is used by the kernel to execute pending return
uprobes.

Note, as of version 3.37.0, Syd drops the capability *CAP_BPF* and
denies the privileged _bpf_(2) commands *BPF_MAP_CREATE* and
*BPF_PROG_LOAD* with the _errno_(3) *EPERM*, i.e. *Operation not
permitted*, *regardless* of the value of this option. This is in
consistence with the Linux kernel checks for the
*kernel.unprivileged_bpf_disabled* _sysctl_(8). Consult the _bpf_(2)
and _capabilities_(7) manual pages for more information about the
*CAP_BPF* Linux capability which is implemented in Linux-5.8 or newer.

## trace/allow_unsafe_dumpable

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether Syd should skip from setting its process
dumpable attribute to false. This allows core dumps for the Syd process,
and allows debugging/profiling/tracing the Syd process. You should not
set this option unless you're developing Syd.

## trace/allow_unsafe_exec_ldso

[[ type
:[ *boolean*

A boolean specifying whether _ld.so_(8) exec indirection should be
allowed. This is not allowed by default to harden noexec boundaries.

## trace/allow_unsafe_exec_libc

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether turning on secure-execution mode for libc
should be skipped. Refer to the *Enforcing AT_SECURE and UID/GID
Verification* section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_exec_memory

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Memory-Deny-Write-Execute (MDWE) protections should be
bypassed. See *Memory-Deny-Write-Execute Protections* section of the _syd_(7)
manual page for more information.

## trace/allow_unsafe_exec_nopie

[[ type
:[ *boolean*

A boolean specifying whether execution of non-PIE binaries should be
allowed. This is generally not recommended but may be necessary on some
systems. Refer to the *Enforcing Position-Independent Executables (PIE)*
section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_exec_null

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether exec calls with NULL argument and
environment pointers should be allowed. Refer to the *Enhanced execve
and execveat Syscall Validation* section of the _syd_(7) manual page for
more information.

## trace/allow_unsafe_exec_stack

[[ type
:[ *boolean*

A boolean specifying whether execution of binaries with executable stack
should be allowed. This is generally not recommended by may be necessary
on some systems. Refer to the *Enforcing Non-Executable Stack* section
of the _syd_(7) manual page for more information.

## trace/allow_unsafe_exec_script

[[ type
:[ *boolean*
|  static
:  *yes*

Opt out of file vetting for interpreted exec. When off (default) on
Linux 6.14 and newer, Syd sets SECBIT_EXEC_RESTRICT_FILE and
SECBIT_EXEC_RESTRICT_FILE_LOCKED at startup so interpreters/dynamic
linkers must only execute a file if _execveat_(2) with AT_EXECVE_CHECK
flag on its file descriptor would succeed (FD-based check avoids
TOCTOU). When on, Syd does not set these bits (legacy behavior). No-op
on kernels < 6.14. Bits are unprivileged-settable; locks make the
policy sticky across exec. Refer to the *Securebits and Kernel-Assisted
Executability* section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_exec_interactive

[[ type
:[ *boolean*
|  static
:  *yes*

Opt out of interactive snippet denial. When off (default) on Linux 6.14
and newer, Syd sets SECBIT_EXEC_DENY_INTERACTIVE and its lock so
interpreters refuse interactive code (-e, -c, REPL, etc.) unless content
arrives via an FD and passes AT_EXECVE_CHECK of _execveat_(2). When on,
Syd does not set these bits (legacy behavior). No-op on kernels < 6.14.
Bits are unprivileged-settable; locks persist the policy across exec.
Refer to the *Securebits and Kernel-Assisted Executability* section of
the _syd_(7) manual page for more information.

## trace/allow_unsafe_exec_speculative

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether speculation controls should not be set to
enable Speculative Execution mitigations using the _prctl_(2) interface
at startup.  When this option is enabled, the _prctl_(2) operations
PR_GET_SPECULATION_CTRL, and PR_SET_SPECULATION_CTRL are allowed within
the sandbox. Refer to the *Speculative Execution Mitigation* section of
the _syd_(7) manual page for more information.

## trace/allow_unsafe_ptrace

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether _ptrace_(2) should be used to secure the
exec handler. Setting this option to true effectively removes the
_ptrace_(2) dependency from the sandbox. This is necessary to trace syd
together with its children, e.g. with *strace -f*. *Warning*, this
option makes _syd_(1) keep the *CAP_SYS_PTRACE* capability and disables
Force Sandboxing, SegvGuard and the exec-TOCTOU mitigator. It allows the
sandbox process to trivially break out of the sandbox by e.g. attaching
to the _syd_(1) main thread with _ptrace_(2) and getting a handle to the
_seccomp_(2) notify file descriptor. Therefore, *this option should
only be used in trusted environments.*

## trace/allow_unsafe_perf

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether perf calls should be allowed within the sandbox.

As of version 3.40.0, the _prctl_(2) operations
*PR_TASK_PERF_EVENTS_ENABLE*, and *PR_TASK_PERF_EVENTS_DISABLE* are also
allowed if this option is set at startup.

## trace/allow_unsafe_create

[[ type
:[ *boolean*

A boolean specifying whether to allow unsafe file creation. Refer to the
*Trusted File Creation* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_filename

[[ type
:[ *boolean*

A boolean specifying whether the restrictions on file names should be
lifted. By default, file names with control characters, forbidden
characters or invalid UTF-8 are denied with *EINVAL* as necessary.
Read *Enhanced Path Integrity Measures* of the _syd_(7) manual page for
more information.

## trace/allow_unsafe_hardlinks

[[ type
:[ *boolean*

A boolean specifying whether to allow unsafe hardlink targets. Refer to
the *Trusted Hardlinks* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_machine_id

[[ type
:[ *boolean*

Specify whether the sandbox substitutes _machine-id_(5) with a
synthetic, per-Syd random identifier or exposes the host value. The
substitution is done at _open_(2) boundary after the access checks
grants access to this file. The files */etc/hostid* and
*/var/adm/hostid* which are part of the _gethostid_(3) interface of
POSIX.1-2008 are also substituted as part of this mitigation. When
*false* (default), at startup Syd computes a SHA3-512 digest using
AT_RANDOM bytes and formats the result as a 128-character lowercase
hexadecimal sandbox ID; the first 32 characters of this string (*which
must not be all zeroes*) are presented in place of _machine-id_(5),
/etc/hostid, and /var/adm/hostid to limit information leakage.
Users may override the sandbox ID by setting *SYD_ID* environment
variable to a 128-character lowercase hexadecimal string that satisfies
the same non-all-zero 32-character prefix constraint. When *true*, no
substitution is performed and the real system _machine-id_(5),
/etc/hostid, and /var/adm/hostid files are made visible to the
sandbox process (i.e., the mitigation is disabled). Refer to the
following links for more information:

- https://man7.org/linux/man-pages/man5/machine-id.5.html
- https://pubs.opengroup.org/onlinepubs/9699919799/functions/gethostid.html

## trace/allow_unsafe_proc_files

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether internal _procfs_(5) should NOT be mounted with the
option *subset=pid*. This option is a no-op unless *unshare/pid:true* is
also set.

## trace/allow_unsafe_proc_pid_status

[[ type
:[ *boolean*

A boolean specifying whether masking security-sensitive fields in
_proc_pid_status_(5) files should be disabled. Refer to the *Hardening
proc_pid_status(5)* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_magiclinks

[[ type
:[ *boolean*

A boolean specifying whether /proc magic links should be followed even when
per-process directory id differs from the caller process id. Magic links are
symbolic link-like objects that are most notably found in _proc_(5); examples
include */proc/pid/exe* and */proc/pid/fd/\**. See _symlink_(7) for more
details. Unknowingly opening magic links can be risky for some applications.
Examples of such risks include the following:

- If the process opening a pathname is a controlling process that
  currently has no controlling terminal (see _credentials_(7)), then
  opening a magic link inside */proc/pid/fd* that happens to refer to a
  terminal would cause the process to acquire a controlling terminal.
- In a containerized environment, a magic link inside */proc* may refer to
  an object outside the container, and thus may provide a means to escape
  from the container.

Because of such risks, Syd denies access to magic links which do not
belong to the current process by default.

As of version 3.36.0, _ioctl_(2) requests to magic links are denied
unless this option is set.

## trace/allow_unsafe_symlinks

[[ type
:[ *boolean*

A boolean specifying whether to allow following symlinks in _untrusted_
directories. _Untrusted_ directories are either group-writable,
world-writable, or have the sticky-bit set. Refer to the *Trusted
Symbolic Links* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_namespace

[[ type
:[ *string-vec*
|  static
:  *yes*

A list of namespaces to allow creation under the sandbox. Must be a
comma-separated list of *mount*, *uts*, *ipc*, *user*, *pid*, *net*,
*cgroup* and *time*. The special value *all* is supported as a
placeholder to specify all namespaces. An invocation of this command
overrides all previous invocations, ie only the list of subnamespaces in
the last invocation of this command will be allowed. By default,
subnamespace creation is not allowed. As of version 3.35.2, the system
calls _sethostname_(2) and _setdomainname_(2) are only allowed in the
sandbox if *uts* subnamespace is allowed. This is similar to the mount
family system calls which are only allowed if *mount* subnamespace is
allowed.

## trace/allow_unsafe_nice

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether process and I/O priority changes are
allowed for the sandbox. See the *Process Priority and Resource
Management* section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_nocookie

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether enforcement of syscall argument cookies
should be disabled. See the *Syscall Argument Cookies* section of the
_syd_(7) manual page for more information.


## trace/allow_unsafe_nomseal

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether read-only sealing critical regions of the Syd
sandbox policy using _mseal_(2) when sandbox is locked should be disabled.
See the *Memory Sealing of Sandbox Policy Regions on Lock* section of the
_syd_(7) manual page for more information.

## trace/allow_unsafe_sigreturn

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifiying whether signal counting to mitigate
Sigreturn Oriented Programming, aka SROP, should be disabled. Refer to
the *Mitigation against Sigreturn Oriented Programming (SROP)* section
of the _syd_(7) manual page for more information.

## trace/allow_unsafe_chown

[[ type
:[ *boolean*
|  static
:  *yes*

Makes Syd keep the capability *CAP_CHOWN* and sandbox process will
inherit the capability from Syd.

## trace/allow_unsafe_chroot

[[ type
:[ *boolean*
|  static
:  *yes*

Disable Chroot sandboxing and turn _chroot_(2) system call into a no-op.
Refer to the explanation of *chroot* sandbox category in the SANDBOXING
section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_pivot_root

[[ type
:[ *boolean*
|  static
:  *yes*

Turn _pivot_root_(2) system call into a no-op rather than
unconditionally denying it with the _errno_(3) *EPERM*. Refer to the
explanation of *chroot* sandbox category in the SANDBOXING section of
the _syd_(7) manual page for more information.

## trace/allow_unsafe_oob

[[ type
:[ *boolean*
|  static
:  *yes*

Allow the *MSG_OOB* flag for _send_(2), _sendto_(2), _sendmsg_(2), and
_sendmmsg_(2) system calls to send out-of-band data. Refer to the
*Denying MSG_OOB Flag in send System Calls* subsection of the _syd_(7)
manual page for more information.

## trace/allow_unsafe_open_kfd

[[ type
:[ *boolean*

A boolean specifying whether _open_(2) calls to AMD KFD character
devices should be continued in the sandbox process rather than opening
them in the Syd emulator thread and sending the file descriptor.
The */dev/kfd* character device requires per-application access to the GPU
device, therefore opening the device in the Syd emulator thread and then
continuing the subsequent _ioctl_(2) system calls in the sandbox process
is going to return *EBADF*, i.e. *Bad file number*. Until Syd has a way
to fully emulate the _ioctl_(2) request space and is able to call the
_ioctl_(2) system call directly from Syd emulator threads, this option
may be used to access such character devices. *Setting this option opens
a TOCTOU attack vector, whereby the sandbox process can open an
arbitrary file instead of the character device in question!* Syd applies
the following mitigations to limit the scope of the attack vector:

- Syd *continues* the system call if and only if *O_RDWR* is set in the flags
  argument.
- Syd does not *continue* the system call if at least one of the flags
  *O_CREAT*, *O_TRUNC* or *O_TMPFILE* is set in the flags argument.
- Syd returns *ENOSYS*, i.e. *Function not implemented*, for the
  _openat2_(2) system call rather than *continuing* it in the sandbox
  process to prevent the *struct open_how* pointer indirection to bypass
  the restrictions applied to the flags argument. Refer to the
  _openat2_(2) manual page for more information.
- This option may be changed at runtime, and it is highly recommended to
  unset this option using the _syd_(2) virtual system call API right after
  the character device is opened.

## trace/allow_unsafe_open_path

[[ type
:[ *boolean*

A boolean specifying whether the mitigation to turn *O_PATH* file
descriptors into *O_RDONLY* file descriptors for safe emulation should
be disabled. With this option, syd continues the _open_(2) system calls
with the *O_PATH* in the sandbox process which opens a TOCTOU vector.

## trace/allow_unsafe_mkbdev

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe block device access should be allowed. When set,
Syd does not drop the capability *CAP_MKNOD* on startup for itself, but
it is still dropped for the sandbox process. This allows:

- block device creation with _mknod_(2).
- _ioctl_(2) calls on block devices.
- open block devices with _open_(2).
- list block devices with _getdents64_(2).

## trace/allow_unsafe_mkcdev

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe character device creation should be allowed. When
set, Syd does not drop the capability *CAP_MKNOD* on startup for itself,
but it is still dropped for the sandbox process. This allows creation of
character devices with _mknod_(2).

## trace/allow_unsafe_stat_bdev

[[ type
:[ *boolean*

Specify whether _stat_(2) family calls on block devices should return
last access and modification times as-is. Refer to the *Device
Sidechannel Mitigations* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_stat_cdev

[[ type
:[ *boolean*

Specify whether _stat_(2) family calls on character devices should
return last access and modification times as-is. Refer to the *Device
Sidechannel Mitigations* section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_notify_bdev

[[ type
:[ *boolean*

Specify whether unsafe event generation for _fanotify_mark_(2) and
_inotify_add_watch_(2) system calls should be allowed for block devices.
Refer to the *Device Sidechannel Mitigations* section of the _syd_(7)
manual page for more information.

## trace/allow_unsafe_notify_cdev

[[ type
:[ *boolean*

Specify whether unsafe event generation for _fanotify_mark_(2) and
_inotify_add_watch_(2) system calls should be allowed for character devices.
Refer to the *Device Sidechannel Mitigations* section of the _syd_(7)
manual page for more information.

## trace/allow_unsafe_cpu

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether CPU emulation system calls should be allowed. By
default, as of version 3.22.1, Syd denies the _modify_ldt_(2),
_subpage_prot_(2), _switch_endian_(2), _vm86_(2), and _vm86old_(2)
system calls, which are associated with CPU emulation functionalities.
Enabling this option (trace/allow_unsafe_cpu:1) permits these calls,
thus relaxing the restriction. This option should be used with caution,
as allowing these system calls can introduce potential vulnerabilities
by enabling processes to modify CPU state or memory protections. Use
this setting only in trusted environments where the execution of these
system calls is necessary.

## trace/allow_unsafe_deprecated

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether deprecated system calls such as _remap_file_pages_(2),
_stime_(2), and _uselib_(2) should be allowed. Refer to the output of
the command *syd-ls deprecated* for the full list of deprecated system
calls for your installation.

## trace/allow_unsafe_keyring

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _add_key_(2), _keyctl_(2), and _request_key_(2)
system calls should be allowed. Enabling this setting permits key
management within the sandbox, which can introduce security risks by
allowing keyring manipulations. Use only in trusted environments.

## trace/allow_unsafe_pipe

[[ type
:[ *boolean*
|  static
:  *yes*

Allow creating notification pipes using the "O_NOTIFICATION_PIPE" flag
to the _pipe2_(2) system call. See the "Denying O_NOTIFICATION_PIPE Flag
in pipe2" subsection of the _syd_(7) manual page for more information.

## trace/allow_unsafe_pkey

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the _pkey_alloc_(2), _pkey_free_(2), and
_pkey_mprotect_(2) system calls should be allowed. By default, these
calls are denied to enhance security. Setting this option to true
enables these system calls, allowing the use of memory protection keys.
This option should be used with caution and only in trusted environments
where the use of these system calls is necessary.

## trace/allow_unsafe_madvise

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether _madvise_(2) system call should NOT be hardened. By
default, only a subset of advices are permitted. see *syd-ls madvise*.
Refer to the *madvise(2) Hardening* section of the _syd_(7) manual page
for more information.

## trace/allow_unsafe_mbind

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the _mbind_(2) system call should be allowed. By
default, this call is denied to enhance security, as changing NUMA
memory policy and triggering page migration over large address ranges
can be abused to create prolonged kernel work and resource pressure,
which can serve as a denial-of-service vector. It may also make memory
placement more predictable, weakening certain mitigation techniques.
Enable only if required for compatibility with applications that need
explicit NUMA policy control.

## trace/allow_unsafe_msgsnd

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the _msgsnd_(2) system call should be allowed. By
default, this call is denied to enhance security as the ability of this
system call to allocate large, contiguous blocks of memory in the kernel
heap is often used to orchestrate kernel heap spraying attacks. See the
"Mitigation Against Heap Spraying" section of the _syd_(7) manual page
for more information.

## trace/allow_unsafe_page_cache

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the system calls _cachestat_(2) and _mincore_(2)
should be allowed. By default, these calls are denied to enhance
security as it has been documented that they can be misused to perform
page-cache attacks. See the "Mitigation against Page Cache Attacks"
section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_time

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether system calls which adjust the system time are
allowed. Note, this also causes Syd to keep the *CAP_SYS_TIME* capability.
Use *syd-ls time* to see the list of system calls allowed by this setting.

## trace/allow_unsafe_uring

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether system calls of the _io_uring_(7) interface are
allowed. Normally, these are denied because they may be used to bypass path
sandboxing. Use *syd-ls uring* to see the list of system calls allowed by
this setting.

## trace/allow_unsafe_xattr

[[ type
:[ *boolean*

A boolean specifying whether the extended attributes restrictions on
"user.syd.\*" and "security.\*" should be lifted. If this option is not
set only sandbox processes with access to the sandbox lock can view
or change these extended attributes.

## trace/allow_unsafe_caps

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether Syd should skip dropping Linux capabilities
at startup. This setting can be used to construct _privileged
containers_ and should be used with extreme care.

Note, _syd-oci_(1) sets this option to honour the list of capabilities
specified by the container engine. You may unset it using the container
configuration file. See CONFIGURATION section in _syd-oci_(1) manual
page for more information.

## trace/allow_unsafe_env

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe environment variables should be allowed into the
environment of the sandbox process. See *syd-ls env* for the list of unsafe
environment variables.

## trace/allow_safe_kcapi

[[ type
:[ *boolean*

Specify whether access to the Linux kernel cryptography API (aka:
"KCAPI") should be allowed when network sandboxing is on. This option
has no effect when network sandboxing is off.

As most things in life, cryptography has good and evil uses: KCAPI is
convenient as it may be used to implement cryptography without depending
on user-space libraries such as OpenSSL but it may also enable malicious
code to efficiently turn itself into ransomware. Adhering to the goal to
be secure by default Syd disallows this access by default.

Note, Syd does not hook into _setsockopt_(2) and the "ALG_SET_KEY"
operation to set the encryption key is directly handled by the host
kernel therefore the encryption key is not copied into Syd's address
space.

Note again, Syd hooks into _bind_(2), _sendto_(2), _sendmsg_(2), and
_sendmmsg_(2) but *not* _read_(2), _write_(2), _recv_(2), or
_splice_(2). To reduce syscall overhead, user is recommended to use the
unhooked system calls when they can to interact with KCAPI.

## trace/allow_safe_syslog

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unprivileged sandbox processes can access Syd's
_syslog_(2) emulation using _dmesg_(8). Unprivileged processes include
the set of _all_ sandbox processes with the sandbox lock "off", and _all
but the initial_ sandbox process with the sandbox lock set to "exec".
Note, this option has nothing to do with access to the host syslog
which is never allowed.

## trace/allow_safe_bind

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the socket address arguments of successful _bind_(2)
calls should be allowed for _connect_(2), _sendto_(2), _sendmsg_(2), and
_sendmmsg_(2) system calls.

*Note*, these addresses are allowed globally and not per-process for
usability reasons. Thus, for example, a process which forks to call
_bind_(2) will have its address allowed for their parent as well.

## trace/allow_unsafe_bind

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability *CAP_NET_BIND_SERVICE*, which
allows a process to _bind_(2) to ports lower than 1024, should be
retained. When this option is set, Syd keeps the capability on startup
for itself, but it is still dropped for the sandbox process. 

## trace/allow_unsafe_socket

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe socket families should be allowed. When set, Syd
does not drop the capability *CAP_NET_RAW* on startup for itself, but it
is still dropped for the sandbox process. This allows:

- use of RAW and PACKET sockets.
- bind to any address for transparent proxying.
- make use of the _ping_(1) command.

## trace/allow_unsupp_socket

[[ type
:[ *boolean*

Specify whether unsupported socket families such as netlink sockets
should be allowed access when network sandboxing is on. By default Syd
allows sandboxed access to unix, ipv4 and ipv6 sockets. This option has
no effect when network sandboxing is off.

As of version 3.16.6 Syd allows access to alg sockets with the
*trace/allow_safe_kcapi* option rather than with this option. Alg
sockets are used to interact with the Linux kernel cryptography API.

Note, on architectures with a multiplexed _socketcall_(2) system call,
enabling this option is insecure because it is vulnerable to TOCTOU.
You may use _syd-sys_(1) utility to check if this system call is
available on your architecture using e.g: *syd-sys socketcall && echo
vulnerable*.

## trace/allow_unsafe_personality

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether _personality_(2) restrictions should be lifted. See *syd-ls
personality* for the list of allowlisted _personality_(2) personas. See the
*Personality Syscall Restrictions* of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_prctl

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether _prctl_(2) restrictions should be lifted. See *syd-ls prctl* for
the list of allowed prctl requests.

## trace/allow_unsafe_prlimit

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether _prlimit_(2) restrictions should be lifted.

## trace/allow_unsafe_mqueue

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe permissions in mode argument of _mq_open_(2)
system call should be permitted. See the *Shared Memory Permissions
Hardening* section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_rseq

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe Restartable Sequences with the _rseq_(2) system
call should be permitted. See the *Denying Restartable Sequences*
section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_shm

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe permissions in mode arguments of _shmget_(2),
_msgget_(2), and _semget_(2) system calls and the *IPC_SET* operation of
_shmctl_(2), _msgctl_(2), and _semctl_(2) system calls should be
permitted. See the *Shared Memory Permissions Hardening* section of the
_syd_(7) manual page for more information.

## trace/allow_unsafe_sysinfo

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _sysinfo_(2) randomizer should be disabled at startup.
If this option is set at startup the _sysinfo_(2) system call becomes
allowed and provides identical info to the files */proc/loadavg* and
*/proc/meminfo* which are disabled by default by common profiles such
as the *linux* and *user* profiles. Notably this mitigation is unset
for the *paludis* profile because leaking this side-channel is
irrelevant for package builds.

## trace/allow_unsafe_syslog

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability *CAP_SYSLOG* should be retained. This
allows the process to perform privileged _syslog_(2) operations. This is
useful when sandboxing a service such as syslogd.

## trace/allow_unsafe_sync

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _sync_(2) and _syncfs_(2) system calls should be
allowed inside the sandbox. By default these system calls are turned
into no-ops to prevent potential local DoS, however it may be useful to
disable this restriction in scenarios where sync is actually expected to
work such as when sandboxing databases.

## trace/allow_unsafe_memfd

[[ type
:[ *boolean*

A boolean specifying whether secret memory file descriptors and
executable memory file descriptors should be enabled. By default Syd
strips the *MFD_EXEC* and adds the *MFD_NOEXEC_SEAL* flag to
_memfd_create_(2) flags argument. This ensures the memory file
descriptor can never be made executable. The *MFD_NOEXEC_SEAL* flag
requires Linux-6.3 or newer therefore on older kernels this option must
be enabled to make memory file descriptors work. However, the user
should be aware that allowing encrypted memory file descriptors does
allow an attacker to bypass Exec, Force and TPE sandboxing and execute
denylisted code.

## trace/allow_unsafe_vmsplice

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _vmsplice_(2) system call should be allowed inside
the sandbox. By default this system call is not permitted, refer to the
*Restricting vmsplice System Call* section of the _syd_(7) manual page
for more information.

## trace/deny_dotdot

[[ type
:[ *boolean*

Specify whether *..* components should be denied during path resolution for
_chdir_(2) and _open_(2) family system calls. This is useful in mitigating path
traversal attacks. See *Path Resolution Restriction For Chdir and Open Calls* of
the _syd_(7) manual page for more information.

## trace/deny_exec_elf32

[[ type
:[ *boolean*

Deny the execution of 32-bit ELF binaries.

## trace/deny_exec_elf_dynamic

[[ type
:[ *boolean*

Deny the execution of dynamically linked ELF binaries.

## trace/deny_exec_elf_static

[[ type
:[ *boolean*

Deny the execution of statically linked ELF binaries.

## trace/deny_exec_script

[[ type
:[ *boolean*

Deny the execution of scripts (files with #!<interpreter> on first line).

The _execve_(2) TOCTOU mitigations do not cover this option which means
*the functionality is vulnerable to TOCTOU*. This allows an attacker to
execute a script whose path is denylisted. This TOCTOU is limited to
scripts and requires the interpreter binary to be allowlisted for exec.
Hence this vulnerability does not allow an attacker to execute
denylisted binaries. This is why the user is recommended to deny the
respective interpreter binaries for execution instead for a safe and
secure approach.

On Linux-6.14 and newer, kernel-assisted executability provides a safe
way to deny execution of scripts in cooperation with enlightened
interpreters. Refer to the *Securebits and Kernel-Assisted
Executability* section of the _syd_(7) manual page for more information.

## trace/deny_tsc

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether reading the timestamp counter should be denied. Without an
accurate timer, many timing attacks are going to be harder to perform.

- This works on *x86 only*.
- This breaks time related calls in the *vDSO*, which can be trivially worked \
around by writing a *LD_PRELOAD* library to call the respective system calls \
directly. See *libsydtime*, https://lib.rs/libsydtime, for a reference \
implementation.
- This has a negative performance impact on programs that rely on \
_gettimeofday_(2) being a *vDSO* call.

## trace/exit_wait_all

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether Syd should wait for all processes to exit before exiting. By
default, Syd exits with the eldest process and any leftover processes in the
background are automatically killed.

## trace/force_cloexec

[[ type
:[ *boolean*

Specify whether the "O_CLOEXEC" flag should be enforced for all
_creat_(2), _open_(2), _openat_(2), _openat2_(2), _memfd_create_(2),
_socket_(2), _accept_(2), and _accept4_(2) system calls made by the
sandbox process. When this feature is enabled, Syd ensures that every
file descriptor opened by the sandbox process is automatically set with
the "O_CLOEXEC" flag, which prevents these file descriptors from being
inherited by newly executed programs. This measure enhances security by
closing file descriptors during _exec_(3) calls, thereby mitigating the
risk of file descriptor leakage which could lead to unauthorized access
to sensitive files or resources. The feature can be toggled at runtime
using Syd's virtual _stat_(2) API, providing flexible control over the
confinement level of sandboxed processes.

## trace/force_rand_fd

[[ type
:[ *boolean*

Specify whether file descriptors returned by all _creat_(2), _open_(2),
_openat_(2), _openat2_(2), _memfd_create_(2), _socket_(2), _accept_(2),
and _accept4_(2) system calls made by the sandbox process should be
randomized.  When this feature is enabled, Syd specifies a random
available slot (rather than the lowest-numbered one) to the
*SECCOMP_IOCTL_NOTIF_ADDFD* operation which is used to install a file
descriptor to the sandbox process. Randomizing file descriptor numbers
makes it significantly harder for an attacker to predict or deliberately
reuse critical descriptors, thereby raising the bar against
file-descriptor reuse and collision attacks. Note that enabling this may
break programs which rely on the POSIX guarantee that _open_(2) returns
the lowest available descriptor. This behavior can be toggled at runtime
via Syd's virtual _stat_(2) API, allowing operators to enable or disable
descriptor randomization without restarting or recompiling the sandboxed
process.  We're also cooperating with the HardenedBSD project to
implement a similar feature in the BSD kernel. Refer to the following
link for more information:
https://git.hardenedbsd.org/hardenedbsd/HardenedBSD/-/issues/117

This feature uses the _kcmp_(2) system call and requires a Linux kernel
configured with the *CONFIG_KCMP* option. On a kernel without this
option, all system calls that are part of this feature will return
*ENOSYS* (*Function not implemented*).

As of version 3.38.0, this option is enabled for the *user* profile.

## trace/force_ro_open

[[ type
:[ *boolean*

Specify whether creating and writing _open_(2) family system calls
should be denied regardless of the path argument. This option is
restricted to _creat_(2), _open_(2), _openat_(2), and _openat2_(2)
system calls and provided for convenience. To stop all write-like access
completely, including e.g. _mkdir_(2), _truncate_(2) etc., use the
*readonly* profile instead which uses the rule "deny/wrset/\*\*\*" to
prevent all write-like access. See "PROFILES" section of the _syd_(5)
manual page for more information.

## trace/force_no_xdev

[[ type
:[ *boolean*

Specify whether path resolution for the _open_(2) family is forced to
use the *RESOLVE_NO_XDEV* resolve flag. This flag is forced during path
canonicalization, therefore this mitigation applies to all hooked path
system calls, not just the _open_(2) family. When enabled, traversal of
mount points, including bind mounts, is disallowed during lookup; the
path must reside on the same mount as the directory referenced by dirfd
(or the current working directory when dirfd == AT_FDCWD). This affects
only the pathname resolution step and does not modify other flags or
access checks.

## trace/force_umask

[[ type
:[ *octal*
|  static
:  *yes*

Specify an umask mode to force. To unset a previously configured force
umask use -1 as the value. As of version 3.15.6, _chmod_(2) family
system calls also honour force umask for added hardening. As of version
3.22.1, this setting does not apply to directory creation for _mkdir_(2)
and _mkdirat_(2) system calls. As of version 3.26.2, this setting does
not apply to UNIX domain socket creation for _bind_(2) system calls, and
non-regular file creation for _mknod_(2) and _mknodat_(2) system calls.

## trace/memory_access

[[ type
:[ *integer*
|  default
:  *2*
|  static
:  *yes*

Set mode on cross memory attach and _proc_pid_mem_(5) usage. Cross
memory attach is done using the system calls _process_vm_readv_(2) and
_process_vm_writev_(2) which requires a Linux kernel configured with the
*CONFIG_CROSS_MEMORY_ATTACH* option enabled. Supported modes are:

- *0*: Use cross memory attach if available, use _proc_pid_mem_(5) otherwise.
- *1*: Use _/proc/pid/mem_(5) unconditionally.
- *2*: Use cross memory attach unconditionally.

From a security point of view, these two modes of access have an
important distinction where cross memory attach honours page protections
of the target process, however using _/proc/pid/mem_(5) does not. This
makes direct _proc_pid_mem_(5) access dangerous in that a Syd deputy
process may be confused into corrupting or even controlling memory
regions the sandbox process otherwise does not have direct access to.
This is the main reason why mode *2* has been added as of version 3.32.6
as a secure default alternative to the previous default mode *0* whose
fallback behaviour can be unpredictable and is against the idea of
secure defaults. Therefore as of version 3.32.6, the user is asked
to change the memory access mode explicitly if their Linux kernel
is not configured with the *CONFIG_CROSS_MEMORY_ATTACH* option.
You may also use the environment variables *SYD_NO_CROSS_MEMORY_ATTACH*
and *SYD_PROC_PID_MEM_FALLBACK*, see the "ENVIRONMENT" section of the
_syd_(1) manual page for more information. For further information
about the security impact of _proc_pid_mem_(5) writes refer to the
following links:

- https://lore.kernel.org/lkml/202403011451.C236A38@keescook/T/
- https://lwn.net/Articles/476947/
- https://issues.chromium.org/issues/40089045

```
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:0 true
% time     seconds  usecs/call     calls    errors syscall
------ ----------- ----------- --------- --------- -----------------
100.00    0.000031          10         3           process_vm_readv
------ ----------- ----------- --------- --------- -----------------
100.00    0.000031          10         3           total
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:1 true
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:2 true
% time     seconds  usecs/call     calls    errors syscall
------ ----------- ----------- --------- --------- -----------------
100.00    0.000008           2         3           process_vm_readv
------ ----------- ----------- --------- --------- -----------------
100.00    0.000008           2         3           total
```

## trace/sync_seccomp

[[ type
:[ *boolean*
|  default
:  *true*
|  static
:  *yes*

Use synchronous mode for seccomp-notify so each Syd syscall handler
thread wakes up on the same CPU as the respective sandbox thread that
executed the system call. This option makes no functional difference and
typically helps with performance. Use _perf_(1) to benchmark seccomp
synchronous mode on your system:

```
; perf bench sched seccomp-notify
# Running 'sched/seccomp-notify' benchmark:
# Executed 1000000 system calls
Total time: 6.736 [sec]
6.736395 usecs/op
148447 ops/sec
; perf bench sched seccomp-notify --sync-mode
# Running 'sched/seccomp-notify' benchmark:
# Executed 1000000 system calls
Total time: 4.188 [sec]
4.188846 usecs/op
238729 ops/sec
```

# PATTERN MATCHING

Syd uses shell-style pattern matching for allowlists and filters. The
matching code is based on _rsync_(1). See the *PATTERN MATCHING RULES*
section of the _rsync_(1) manual for more information. Notably, Syd
applies the *triple star* extension to patterns, i.e. */dev/\*\*\**
matches both */dev* and any file recursively under */dev*. Note also,
Syd gets patterns from multiple sources: a configuration file, a
profile, the *-m* command line switch, or a _stat_(1) call with
*/dev/syd* prefix. There is no precedence between different sources. All
patterns in a list are compiled together in an array and pattern
matching during access control happens in a single step where *the last
matching pattern decides the outcome.*

# ADDRESS MATCHING

Syd has a simple address scheme to match network addresses. The addresses can
either be a *glob* pattern to match *UNIX* and *abstract UNIX* socket addresses,
or *IP CIDR* followed by a port range to match *IPv4* and *IPv6* addresses. Port
range can either be a single port or a closed range in format *port1-port2*. The
address and the port range must be split by the character *!*.	The precedence
logic is same as *Pattern Matching* where *the last matching pattern decides the
outcome.*

In addition there are some *aliases*, you may use instead of specifying an address:

- *any*: Expanded to *any4* + *any6*.
- *any4*: Expanded to *0.0.0.0/0* which matches the whole Ipv4 address space.
- *any6*: Expanded to *::/0* which matches the whole Ipv6 address space.
- *loopback*: Expanded to *loopback4* + *loopback6*.
- *loopback4*: Expanded to *127.0.0.0/8*
- *loopback6*: Expanded to *::1/128*
- *linklocal*: Expanded to *linklocal4* + *linklocal6*.
- *linklocal4*: Expanded to *169.254.0.0/16*
- *linklocal6*: Expanded to *fe80::/10*
- *local*: Expanded to *local4* + *local6*.
- *local4*: Expanded to four addresses as defined in *RFC1918*:
    - *127.0.0.0/8*
    - *10.0.0.0/8*
    - *172.16.0.0/12*
    - *192.168.0.0/16*
- *local6*: Expanded to four addresses:
    - *::1/128*
    - *fe80::/7*
    - *fc00::/7*
    - *fec0::/7*

# SECURITY

The interface is only available if the sandbox lock is not set for the calling
process. Similarly, command-line option parsing and configuration file parsing
stops once a *lock:on* clause is executed.

# RETURN VALUE

For _stat_(2) calls, on success, zero is returned. On error, -1 is returned, and
errno is set to indicate the error.

For _open_(2) calls, on success the new file descriptor (a nonnegative integer)
is returned. On error, -1 is returned and errno is set to indicate the error.

# ERRORS

|[ *EBUSY*
:< Attempted to edit a setting at runtime that must be configured at startup

|[ *EEXIST*
:< Attempted to add an UID/GID transition for SafeSetID but a transition with the same source UID/GID exists.

|[ *EINVAL*
:< Sandbox command is syntactically incorrect.

|[ *ENOENT*
:< Result of the given sandbox query is false (e.g. *test -c /dev/syd/sandbox/stat?*).

|[ *ENOENT*
:< Sandbox lock is on, no commands are allowed.

|[ *ENOKEY*
:< Crypt sandboxing is on but no encryption key was supplied.

|[ *EKEYREVOKED*
:< Session keyring is not linked to the user keyring for Crypt sandboxing.

|[ *ENODATA*
:< SafeSetID is on but no UID/GID transits were defined for the current user/group.

|[ *EOPNOTSUPP*
:< Sandbox command is not supported.

# SEE ALSO

_syd_(1), _syd_(5), _syd_(7), _syd-ls_(1) _open_(2), _stat_(2),
_perf_(1), _pledge_(2), _ptrace_(2), _seccomp_(2), _strace_(1),
_glob_(3p), _io_uring_(7), _gdb_(1), _valgrind_(1), _wordexp_(3)

- *syd* homepage: https://sydbox.exherbo.org
- *libsyd* homepage: https://libsyd.exherbo.org
- *gosyd* homepage: https://gosyd.exherbo.org
- *plsyd* homepage:  https://plsyd.exherbo.org
- *pysyd* homepage: https://pysyd.exherbo.org
- *rbsyd* homepage: https://rbsyd.exherbo.org
- *syd.el* homepage: https://sydel.exherbo.org
- *libsydtime* homepage: https://lib.rs/libsydtime
- *LandLock* homepage: https://landlock.io/
- *vDSO* wiki: https://en.wikipedia.org/wiki/VDSO
- *parse-size* documentation: https://docs.rs/parse-size/1.0.0/parse_size/
- *ipnetwork* documentation: https://docs.rs/ipnetwork

# AUTHORS

Maintained by Ali Polatel. Up-to-date sources can be found at
https://gitlab.exherbo.org/sydbox/sydbox.git and bugs/patches can be
submitted to https://gitlab.exherbo.org/groups/sydbox/-/issues. Discuss
in #sydbox on Libera Chat or in #sydbox:mailstation.de on Matrix.
