#!/usr/bin/env python3
#
# Public Domain 2014-present MongoDB, Inc.
# Public Domain 2008-2014 WiredTiger, Inc.
#
# This is free and unencumbered software released into the public domain.
#
# Anyone is free to copy, modify, publish, use, compile, sell, or
# distribute this software, either in source code form or as a compiled
# binary, for any purpose, commercial or non-commercial, and by any
# means.
#
# In jurisdictions that recognize copyright laws, the author or authors
# of this software dedicate any and all copyright interest in the
# software to the public domain. We make this dedication for the benefit
# of the public at large and to the detriment of our heirs and
# successors. We intend this dedication to be an overt act of
# relinquishment in perpetuity of all present and future rights to this
# software under copyright law.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


# ------------------------------------------------------------------------------
# Given a database directory generated by Workgen, find and compare the mirrored
# tables. Returns zero on success if all mirror pairs contain the same data.
# Returns a non-zero value on failure if any mirror pair contains a mismatch.
#
# Note that the wiredtiger/tools directory must be on your PYTHONPATH.
# ------------------------------------------------------------------------------
import os, pathlib, re, sys
from contextlib import redirect_stdout
from wt_tools_common import wiredtiger_open
from wt_cmp_uri import wiredtiger_compare_uri

# Print usage and exit with a failure status.
def usage_exit():

    print('Usage: python3 validate_mirror_tables database_dir')
    print('  database_dir is a POSIX pathname to a WiredTiger home directory')
    sys.exit(1)

# Given a database directory, return all workgen tables.
def get_wiredtiger_db_files(dir):

    files = [f[:-3] for f in os.listdir(dir) \
             if f.endswith('.wt') and not f.startswith('WiredTiger')]
    return files

# Given a list of database tables, return a list of mirrored uri pairs.
def get_mirrors(db_dir, db_files):

    mirrors = []
    db_files_remaining = set(db_files)

    for filename in db_files:
        if filename not in db_files_remaining:
            continue
        db_files_remaining.remove(filename)
        mirror_filename = get_mirror_file(db_dir, f'table:{filename}')
        if mirror_filename is not None:
            db_files_remaining.remove(mirror_filename)
            mirrors.append([f'{db_dir}/table:{filename}',
                            f'{db_dir}/table:{mirror_filename}'])

    return mirrors

# Get the mirror for the specified file by examining the file's metadata. Mirror names
# are stored in the 'app_metadata' by workgen when mirroring is enabled. If the file has
# a mirror, the name of the mirror is returned. Otherwise, the function returns None.
def get_mirror_file(db_dir, filename):

    connection = wiredtiger_open(db_dir, 'readonly')
    session = connection.open_session()
    c = session.open_cursor('metadata:', None, None)
    metadata = c[filename]
    c.close()

    result = re.findall('app_metadata="([^"]*)"', metadata)
    mirror = None

    if result:
        app_metadata = dict((a.strip(), b.strip())  
            for a, b in (element.split('=')  
                for element in result[0].split(',')))  

        if app_metadata.get('workgen_dynamic_table') == 'true' and \
           app_metadata.get('workgen_table_mirror') != None :
            mirror = app_metadata['workgen_table_mirror']
            mirror = mirror.split(':')[1]

    session.close()    
    connection.close()
    return mirror

# ------------------------------------------------------------------------------

def main(sysargs):

    if len(sysargs) != 1:
        usage_exit()

    db_dir = sysargs[0]
    db_files = get_wiredtiger_db_files(db_dir)
    mirrors = get_mirrors(db_dir, db_files)
    failure_count = 0

    for item in mirrors:
        stdout = None
        try:
            with open(os.devnull, "w") as f, redirect_stdout(f):       
                stdout = wiredtiger_compare_uri(item)
        except SystemExit as e:
            if e.code != 0:
                print(f"Mirror mismatch {item}: {stdout}")
                failure_count += 1

    if failure_count == 0:
        print(f"Successfully validated {len(mirrors)} table mirrors in " \
              f"database directory '{db_dir}'.")
    else:
        print(f"Mirrored table validation failed for {failure_count} of " \
              f"{len(mirrors)} table mirrors in database directory '{db_dir}'.")

    sys.exit(1 if failure_count > 0 else 0)

# ------------------------------------------------------------------------------

if __name__ == "__main__":

    main(sys.argv[1:])
