// Copyright 2019 The TCMalloc Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "absl/types/span.h"
#include "tcmalloc/common.h"
#include "tcmalloc/internal/config.h"
#include "tcmalloc/size_class_info.h"

GOOGLE_MALLOC_SECTION_BEGIN
namespace tcmalloc {
namespace tcmalloc_internal {

// Columns in the following tables:
// - bytes: size of the size class
// - pages: number of pages per span
// - batch: preferred number of objects for transfers between caches
// - cap: maximum cpu cache capacity
// - class: size class number
// - objs: number of objects per span
// - waste/fixed: fixed per-size-class overhead due to end-of-span fragmentation
//   and other factors. For instance, if we have a 96 byte size class, and use
//   a single 8KiB page, then we will hold 85 objects per span, and have 32
//   bytes left over. There is also a fixed component of 48 bytes of TCMalloc
//   metadata per span. Together, the fixed overhead would be wasted/allocated
//   = (32 + 48) / (8192 - 32) ~= 0.98%.
// - waste/sampling: overhead due to heap sampling
//   (rounding to page size, proxy object, metadata).
// - inc: increment from the previous size class. This caps the dynamic
//   overhead component based on mismatches between the number of bytes
//   requested and the number of bytes provided by the size class. Together
//   they sum to the total overhead; for instance if you asked for a 50-byte
//   allocation that rounds up to a 64-byte size class, the dynamic overhead
//   would be 28%, and if waste were 22% it would mean (on average) 25 bytes
//   of overhead for allocations of that size.

// clang-format off
#if defined(__cpp_aligned_new) && __STDCPP_DEFAULT_NEW_ALIGNMENT__ <= 8
#if TCMALLOC_PAGE_SHIFT == 13
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2281},  //  0  1024  0.58%    0.42%   0.00%
  {    16,    1,   32, 2281},  //  1   512  0.58%    0.42% 100.00%
  {    24,    1,   32, 2281},  //  2   341  0.68%    0.42%  50.00%
  {    32,    1,   32, 2281},  //  3   256  0.58%    0.42%  33.33%
  {    40,    1,   32, 1398},  //  4   204  0.97%    0.42%  25.00%
  {    48,    1,   32, 2281},  //  5   170  0.97%    0.42%  20.00%
  {    56,    1,   32,  779},  //  6   146  0.78%    0.42%  16.67%
  {    64,    1,   32, 1554},  //  7   128  0.58%    0.42%  14.29%
  {    72,    1,   32,  565},  //  8   113  1.26%    0.42%  12.50%
  {    80,    1,   32,  841},  //  9   102  0.97%    0.42%  11.11%
  {    88,    1,   32,  476},  // 10    93  0.68%    0.42%  10.00%
  {    96,    1,   32,  539},  // 11    85  0.97%    0.42%   9.09%
  {   104,    1,   32,  276},  // 12    78  1.55%    0.42%   8.33%
  {   112,    1,   32,  392},  // 13    73  0.78%    0.42%   7.69%
  {   120,    1,   32,  282},  // 14    68  0.97%    0.42%   7.14%
  {   128,    1,   32,  448},  // 15    64  0.58%    0.42%   6.67%
  {   136,    1,   32,  226},  // 16    60  0.97%    0.42%   6.25%
  {   144,    1,   32,  334},  // 17    56  2.14%    0.42%   5.88%
  {   160,    1,   32,  361},  // 18    51  0.97%    0.42%  11.11%
  {   176,    1,   32,  235},  // 19    46  1.75%    0.42%  10.00%
  {   192,    1,   32,  277},  // 20    42  2.14%    0.42%   9.09%
  {   208,    1,   32,  216},  // 21    39  1.55%    0.42%   8.33%
  {   224,    1,   32,  200},  // 22    36  2.14%    0.42%   7.69%
  {   240,    1,   32,  194},  // 23    34  0.97%    0.42%   7.14%
  {   256,    1,   32,  322},  // 24    32  0.58%    0.42%   6.67%
  {   264,    1,   32,  154},  // 25    31  0.68%    0.42%   3.12%
  {   280,    1,   32,  204},  // 26    29  1.46%    0.42%   6.06%
  {   312,    1,   32,  224},  // 27    26  1.55%    0.42%  11.43%
  {   336,    1,   32,  228},  // 28    24  2.14%    0.42%   7.69%
  {   352,    1,   32,  165},  // 29    23  1.75%    0.42%   4.76%
  {   384,    1,   32,  186},  // 30    21  2.14%    0.42%   9.09%
  {   408,    1,   32,  174},  // 31    20  0.97%    0.42%   6.25%
  {   424,    1,   32,  156},  // 32    19  2.23%    0.42%   3.92%
  {   448,    1,   32,  182},  // 33    18  2.14%    0.42%   5.66%
  {   480,    1,   32,  167},  // 34    17  0.97%    0.42%   7.14%
  {   512,    1,   32,  246},  // 35    16  0.58%    0.42%   6.67%
  {   576,    1,   32,  188},  // 36    14  2.14%    0.42%  12.50%
  {   640,    1,   32,  175},  // 37    12  6.80%    0.43%  11.11%
  {   704,    1,   32,  165},  // 38    11  6.02%    0.43%  10.00%
  {   768,    1,   32,  164},  // 39    10  6.80%    0.43%   9.09%
  {   896,    1,   32,  170},  // 40     9  2.14%    0.43%  16.67%
  {  1024,    1,   32,  235},  // 41     8  0.58%    0.42%  14.29%
  {  1152,    2,   32,  167},  // 42    14  1.85%    0.43%  12.50%
  {  1280,    2,   32,  158},  // 43    12  6.52%    0.43%  11.11%
  {  1408,    2,   32,  155},  // 44    11  5.74%    0.43%  10.00%
  {  1536,    2,   32,  157},  // 45    10  6.52%    0.43%   9.09%
  {  1792,    2,   32,  156},  // 46     9  1.85%    0.43%  16.67%
  {  2048,    2,   32,  170},  // 47     8  0.29%    0.42%  14.29%
  {  2304,    2,   28,  154},  // 48     7  1.85%    0.43%  12.50%
  {  2688,    2,   24,  151},  // 49     6  1.85%    0.43%  16.67%
  {  2816,    3,   23,  146},  // 50     8  8.51%    0.44%   4.76%
  {  3200,    2,   20,  148},  // 51     5  2.63%    0.43%  13.64%
  {  3456,    3,   18,  145},  // 52     7  1.75%    0.43%   8.00%
  {  3584,    4,   18,  145},  // 53     9  1.71%    0.43%   3.70%
  {  4096,    1,   16,  225},  // 54     2  0.58%    0.43%  14.29%
  {  4736,    3,   13,  147},  // 55     5  3.83%    0.43%  15.62%
  {  5376,    2,   12,  145},  // 56     3  1.85%    0.43%  13.51%
  {  6144,    3,   10,  147},  // 57     4  0.19%    0.42%  14.29%
  {  7168,    7,    9,  146},  // 58     8  0.08%    0.42%  16.67%
  {  8192,    1,    8,  172},  // 59     1  0.58%    0.03%  14.29%
  {  9472,    5,    6,  146},  // 60     4  7.61%    0.85%  15.62%
  { 10240,    4,    6,  144},  // 61     3  6.39%    0.85%   8.11%
  { 12288,    3,    5,  146},  // 62     2  0.19%    0.82%  20.00%
  { 13568,    5,    4,  144},  // 63     3  0.74%    0.82%  10.42%
  { 14336,    7,    4,  143},  // 64     4  0.08%    0.82%   5.66%
  { 16384,    2,    4,  148},  // 65     1  0.29%    0.03%  14.29%
  { 20480,    5,    3,  145},  // 66     2  0.12%    1.21%  25.00%
  { 24576,    3,    2,  145},  // 67     1  0.19%    0.03%  20.00%
  { 28672,    7,    2,  144},  // 68     2  0.08%    1.60%  16.67%
  { 32768,    4,    2,  149},  // 69     1  0.15%    0.03%  14.29%
  { 40960,    5,    2,  144},  // 70     1  0.12%    0.03%  25.00%
  { 49152,    6,    2,  143},  // 71     1  0.10%    0.03%  20.00%
  { 57344,    7,    2,  143},  // 72     1  0.08%    0.03%  16.67%
  { 65536,    8,    2,  145},  // 73     1  0.07%    0.03%  14.29%
  { 73728,    9,    2,  144},  // 74     1  0.07%    0.03%  12.50%
  { 81920,   10,    2,  143},  // 75     1  0.06%    0.03%  11.11%
  { 98304,   12,    2,  144},  // 76     1  0.05%    0.03%  20.00%
  {114688,   14,    2,  143},  // 77     1  0.04%    0.03%  16.67%
  {131072,   16,    2,  143},  // 78     1  0.04%    0.03%  14.29%
  {139264,   17,    2,  144},  // 79     1  0.03%    0.03%   6.25%
  {155648,   19,    2,  143},  // 80     1  0.03%    0.03%  11.76%
  {172032,   21,    2,  135},  // 81     1  0.03%    0.03%  10.53%
  {204800,   25,    2,  143},  // 82     1  0.02%    0.03%  19.05%
  {229376,   28,    2,  143},  // 83     1  0.02%    0.03%  12.00%
  {262144,   32,    2,  143},  // 84     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 15
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2485},  //  0  4096  0.15%    1.60%   0.00%
  {    16,    1,   32, 2485},  //  1  2048  0.15%    1.60% 100.00%
  {    24,    1,   32, 2485},  //  2  1365  0.17%    1.60%  50.00%
  {    32,    1,   32, 2485},  //  3  1024  0.15%    1.60%  33.33%
  {    40,    1,   32, 1587},  //  4   819  0.17%    1.60%  25.00%
  {    48,    1,   32, 2485},  //  5   682  0.24%    1.60%  20.00%
  {    56,    1,   32,  781},  //  6   585  0.17%    1.60%  16.67%
  {    64,    1,   32, 1520},  //  7   512  0.15%    1.60%  14.29%
  {    72,    1,   32,  436},  //  8   455  0.17%    1.60%  12.50%
  {    80,    1,   32,  580},  //  9   409  0.29%    1.60%  11.11%
  {    88,    1,   32,  442},  // 10   372  0.24%    1.60%  10.00%
  {    96,    1,   32,  522},  // 11   341  0.24%    1.60%   9.09%
  {   104,    1,   32,  306},  // 12   315  0.17%    1.60%   8.33%
  {   112,    1,   32,  336},  // 13   292  0.34%    1.60%   7.69%
  {   120,    1,   32,  295},  // 14   273  0.17%    1.60%   7.14%
  {   128,    1,   32,  487},  // 15   256  0.15%    1.60%   6.67%
  {   136,    1,   32,  233},  // 16   240  0.54%    1.60%   6.25%
  {   144,    1,   32,  275},  // 17   227  0.39%    1.60%   5.88%
  {   160,    1,   32,  312},  // 18   204  0.54%    1.60%  11.11%
  {   176,    1,   32,  241},  // 19   186  0.24%    1.60%  10.00%
  {   192,    1,   32,  277},  // 20   170  0.54%    1.60%   9.09%
  {   208,    1,   32,  201},  // 21   157  0.49%    1.60%   8.33%
  {   240,    1,   32,  297},  // 22   136  0.54%    1.60%  15.38%
  {   256,    1,   32,  329},  // 23   128  0.15%    1.60%   6.67%
  {   280,    1,   32,  222},  // 24   117  0.17%    1.60%   9.38%
  {   304,    1,   32,  199},  // 25   107  0.88%    1.60%   8.57%
  {   320,    1,   32,  177},  // 26   102  0.54%    1.60%   5.26%
  {   352,    1,   32,  226},  // 27    93  0.24%    1.60%  10.00%
  {   400,    1,   32,  201},  // 28    81  1.27%    1.60%  13.64%
  {   448,    1,   32,  191},  // 29    73  0.34%    1.60%  12.00%
  {   512,    1,   32,  246},  // 30    64  0.15%    1.60%  14.29%
  {   576,    1,   32,  186},  // 31    56  1.71%    1.60%  12.50%
  {   640,    1,   32,  198},  // 32    51  0.54%    1.60%  11.11%
  {   704,    1,   32,  182},  // 33    46  1.32%    1.60%  10.00%
  {   768,    1,   32,  176},  // 34    42  1.71%    1.60%   9.09%
  {   896,    1,   32,  191},  // 35    36  1.71%    1.60%  16.67%
  {  1024,    1,   32,  217},  // 36    32  0.15%    1.60%  14.29%
  {  1152,    1,   32,  176},  // 37    28  1.71%    1.60%  12.50%
  {  1280,    1,   32,  170},  // 38    25  2.49%    1.60%  11.11%
  {  1408,    1,   32,  170},  // 39    23  1.32%    1.60%  10.00%
  {  1536,    1,   32,  167},  // 40    21  1.71%    1.60%   9.09%
  {  1792,    1,   32,  166},  // 41    18  1.71%    1.60%  16.67%
  {  1920,    1,   32,  159},  // 42    17  0.54%    1.60%   7.14%
  {  2048,    1,   32,  170},  // 43    16  0.15%    1.60%   6.67%
  {  2176,    1,   30,  167},  // 44    15  0.54%    1.60%   6.25%
  {  2304,    1,   28,  159},  // 45    14  1.71%    1.60%   5.88%
  {  2688,    1,   24,  165},  // 46    12  1.71%    1.60%  16.67%
  {  3200,    1,   20,  163},  // 47    10  2.49%    1.60%  19.05%
  {  3584,    1,   18,  159},  // 48     9  1.71%    1.60%  12.00%
  {  4096,    1,   16,  206},  // 49     8  0.15%    1.60%  14.29%
  {  4608,    1,   14,  160},  // 50     7  1.71%    1.60%  12.50%
  {  5376,    1,   12,  159},  // 51     6  1.71%    1.60%  16.67%
  {  6528,    1,   10,  163},  // 52     5  0.54%    1.60%  21.43%
  {  8192,    1,    8,  167},  // 53     4  0.15%    1.60%  25.49%
  {  9344,    2,    7,  159},  // 54     7  0.27%    1.60%  14.06%
  { 10880,    1,    6,  157},  // 55     3  0.54%    1.60%  16.44%
  { 13056,    2,    5,  158},  // 56     5  0.46%    1.60%  20.00%
  { 13952,    3,    4,  155},  // 57     7  0.70%    1.60%   6.86%
  { 16384,    1,    4,  164},  // 58     2  0.15%    1.60%  17.43%
  { 19072,    3,    3,  158},  // 59     5  3.04%    1.62%  16.41%
  { 21760,    2,    3,  157},  // 60     3  0.46%    1.60%  14.09%
  { 24576,    3,    2,  155},  // 61     4  0.05%    1.60%  12.94%
  { 28672,    7,    2,  157},  // 62     8  0.02%    1.60%  16.67%
  { 32768,    1,    2,  160},  // 63     1  0.15%    0.03%  14.29%
  { 38144,    5,    2,  155},  // 64     4  6.90%    3.28%  16.41%
  { 40960,    4,    2,  155},  // 65     3  6.28%    3.28%   7.38%
  { 49152,    3,    2,  155},  // 66     2  0.05%    3.16%  20.00%
  { 57344,    7,    2,  155},  // 67     4  0.02%    3.16%  16.67%
  { 65536,    2,    2,  158},  // 68     1  0.07%    0.03%  14.29%
  { 81920,    5,    2,  157},  // 69     2  0.03%    4.72%  25.00%
  { 98304,    3,    2,  155},  // 70     1  0.05%    0.03%  20.00%
  {114688,    7,    2,  155},  // 71     2  0.02%    6.28%  16.67%
  {131072,    4,    2,  163},  // 72     1  0.04%    0.03%  14.29%
  {163840,    5,    2,  155},  // 73     1  0.03%    0.03%  25.00%
  {196608,    6,    2,  155},  // 74     1  0.02%    0.03%  20.00%
  {229376,    7,    2,  155},  // 75     1  0.02%    0.03%  16.67%
  {262144,    8,    2,  157},  // 76     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 18
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2331},  //  0 32768  0.02%   12.53%   0.00%
  {    16,    1,   32, 2331},  //  1 16384  0.02%   12.53% 100.00%
  {    24,    1,   32, 2331},  //  2 10922  0.02%   12.53%  50.00%
  {    32,    1,   32, 2331},  //  3  8192  0.02%   12.53%  33.33%
  {    40,    1,   32, 1334},  //  4  6553  0.03%   12.53%  25.00%
  {    48,    1,   32, 2331},  //  5  5461  0.02%   12.53%  20.00%
  {    56,    1,   32,  648},  //  6  4681  0.02%   12.53%  16.67%
  {    64,    1,   32, 1459},  //  7  4096  0.02%   12.53%  14.29%
  {    72,    1,   32,  691},  //  8  3640  0.04%   12.53%  12.50%
  {    80,    1,   32,  624},  //  9  3276  0.04%   12.53%  11.11%
  {    88,    1,   32,  302},  // 10  2978  0.05%   12.53%  10.00%
  {    96,    1,   32,  382},  // 11  2730  0.04%   12.53%   9.09%
  {   104,    1,   32,  264},  // 12  2520  0.04%   12.53%   8.33%
  {   112,    1,   32,  340},  // 13  2340  0.04%   12.53%   7.69%
  {   128,    1,   32,  473},  // 14  2048  0.02%   12.53%  14.29%
  {   144,    1,   32,  411},  // 15  1820  0.04%   12.53%  12.50%
  {   160,    1,   32,  365},  // 16  1638  0.04%   12.53%  11.11%
  {   176,    1,   32,  215},  // 17  1489  0.05%   12.53%  10.00%
  {   192,    1,   32,  236},  // 18  1365  0.04%   12.53%   9.09%
  {   208,    1,   32,  200},  // 19  1260  0.04%   12.53%   8.33%
  {   232,    1,   32,  299},  // 20  1129  0.10%   12.53%  11.54%
  {   256,    1,   32,  333},  // 21  1024  0.02%   12.53%  10.34%
  {   280,    1,   32,  221},  // 22   936  0.04%   12.53%   9.38%
  {   312,    1,   32,  273},  // 23   840  0.04%   12.53%  11.43%
  {   336,    1,   32,  246},  // 24   780  0.04%   12.53%   7.69%
  {   376,    1,   32,  176},  // 25   697  0.05%   12.53%  11.90%
  {   416,    1,   32,  199},  // 26   630  0.04%   12.53%  10.64%
  {   472,    1,   32,  193},  // 27   555  0.09%   12.53%  13.46%
  {   512,    1,   32,  213},  // 28   512  0.02%   12.53%   8.47%
  {   576,    1,   32,  190},  // 29   455  0.04%   12.53%  12.50%
  {   704,    1,   32,  207},  // 30   372  0.12%   12.53%  22.22%
  {   768,    1,   32,  165},  // 31   341  0.12%   12.53%   9.09%
  {   896,    1,   32,  174},  // 32   292  0.21%   12.53%  16.67%
  {  1024,    1,   32,  213},  // 33   256  0.02%   12.53%  14.29%
  {  1152,    1,   32,  168},  // 34   227  0.26%   12.53%  12.50%
  {  1280,    1,   32,  162},  // 35   204  0.41%   12.53%  11.11%
  {  1408,    1,   32,  155},  // 36   186  0.12%   12.53%  10.00%
  {  1664,    1,   32,  188},  // 37   157  0.36%   12.53%  18.18%
  {  1920,    1,   32,  174},  // 38   136  0.41%   12.53%  15.38%
  {  2048,    1,   32,  165},  // 39   128  0.02%   12.53%   6.67%
  {  2176,    1,   30,  204},  // 40   120  0.41%   12.53%   6.25%
  {  2304,    1,   28,  156},  // 41   113  0.70%   12.53%   5.88%
  {  2560,    1,   25,  154},  // 42   102  0.41%   12.53%  11.11%
  {  2816,    1,   23,  149},  // 43    93  0.12%   12.53%  10.00%
  {  3072,    1,   21,  149},  // 44    85  0.41%   12.53%   9.09%
  {  3328,    1,   19,  154},  // 45    78  0.99%   12.54%   8.33%
  {  3584,    1,   18,  148},  // 46    73  0.21%   12.53%   7.69%
  {  3840,    1,   17,  148},  // 47    68  0.41%   12.53%   7.14%
  {  4096,    1,   16,  192},  // 48    64  0.02%   12.53%   6.67%
  {  4224,    1,   15,  149},  // 49    62  0.12%   12.53%   3.12%
  {  4736,    1,   13,  151},  // 50    55  0.65%   12.54%  12.12%
  {  5248,    1,   12,  154},  // 51    49  1.92%   12.54%  10.81%
  {  5760,    1,   11,  148},  // 52    45  1.14%   12.54%   9.76%
  {  6528,    1,   10,  150},  // 53    40  0.41%   12.53%  13.33%
  {  7168,    1,    9,  147},  // 54    36  1.58%   12.54%   9.80%
  {  8192,    1,    8,  160},  // 55    32  0.02%   12.53%  14.29%
  {  9344,    1,    7,  149},  // 56    28  0.21%   12.53%  14.06%
  { 10880,    1,    6,  148},  // 57    24  0.41%   12.54%  16.44%
  { 11904,    1,    5,  148},  // 58    22  0.12%   12.53%   9.41%
  { 13056,    1,    5,  148},  // 59    20  0.41%   12.54%   9.68%
  { 13696,    1,    4,  146},  // 60    19  0.75%   12.54%   4.90%
  { 14464,    1,    4,  147},  // 61    18  0.70%   12.54%   5.61%
  { 15360,    1,    4,  146},  // 62    17  0.41%   12.54%   6.19%
  { 16384,    1,    4,  151},  // 63    16  0.02%   12.53%   6.67%
  { 17408,    1,    3,  147},  // 64    15  0.41%   12.54%   6.25%
  { 18688,    1,    3,  148},  // 65    14  0.21%   12.54%   7.35%
  { 20096,    1,    3,  146},  // 66    13  0.36%   12.54%   7.53%
  { 21760,    1,    3,  146},  // 67    12  0.41%   12.54%   8.28%
  { 23808,    1,    2,  147},  // 68    11  0.12%   12.53%   9.41%
  { 26112,    1,    2,  147},  // 69    10  0.41%   12.54%   9.68%
  { 29056,    1,    2,  146},  // 70     9  0.26%   12.54%  11.27%
  { 32768,    1,    2,  162},  // 71     8  0.02%   12.53%  12.78%
  { 37376,    1,    2,  147},  // 72     7  0.21%   12.54%  14.06%
  { 43648,    1,    2,  146},  // 73     6  0.12%   12.54%  16.78%
  { 45568,    2,    2,  146},  // 74    11  4.40%   12.63%   4.40%
  { 52352,    1,    2,  146},  // 75     5  0.16%   12.54%  14.89%
  { 56064,    2,    2,  146},  // 76     9  3.77%   12.63%   7.09%
  { 65536,    1,    2,  147},  // 77     4  0.02%   12.53%  16.89%
  { 74880,    2,    2,  146},  // 78     7  0.03%   12.53%  14.26%
  { 87296,    1,    2,  146},  // 79     3  0.12%   12.54%  16.58%
  {104832,    2,    2,  146},  // 80     5  0.03%   12.54%  20.09%
  {112256,    3,    2,  146},  // 81     7  0.09%   12.54%   7.08%
  {131072,    1,    2,  146},  // 82     2  0.02%   12.53%  16.76%
  {149760,    3,    2,  146},  // 83     5  4.79%   12.88%  14.26%
  {174720,    2,    2,  146},  // 84     3  0.03%   12.54%  16.67%
  {196608,    3,    2,  137},  // 85     4  0.01%   12.53%  12.53%
  {209664,    4,    2,  146},  // 86     5  0.03%   12.54%   6.64%
  {262144,    1,    2,  147},  // 87     1  0.02%    0.03%  25.03%
};
#elif TCMALLOC_PAGE_SHIFT == 12
static_assert(kMaxSize == 8192, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = false,
  .span_size = 48,
  .sampling_rate = 524288,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2072},  //  0   512  1.16%    0.92%   0.00%
  {    16,    1,   32, 2810},  //  1   256  1.16%    0.92% 100.00%
  {    24,    1,   32, 2810},  //  2   170  1.54%    0.92%  50.00%
  {    32,    1,   32, 2810},  //  3   128  1.16%    0.92%  33.33%
  {    40,    1,   32, 2557},  //  4   102  1.54%    0.92%  25.00%
  {    48,    1,   32, 2810},  //  5    85  1.54%    0.92%  20.00%
  {    56,    1,   32, 1180},  //  6    73  1.35%    0.92%  16.67%
  {    64,    1,   32, 2810},  //  7    64  1.16%    0.92%  14.29%
  {    72,    1,   32,  475},  //  8    56  2.70%    0.92%  12.50%
  {    80,    1,   32, 1139},  //  9    51  1.54%    0.92%  11.11%
  {    96,    1,   32,  958},  // 10    42  2.70%    0.92%  20.00%
  {   104,    1,   32,  375},  // 11    39  2.12%    0.92%   8.33%
  {   112,    1,   32,  580},  // 12    36  2.70%    0.92%   7.69%
  {   128,    1,   32,  741},  // 13    32  1.16%    0.92%  14.29%
  {   144,    1,   32,  617},  // 14    28  2.70%    0.92%  12.50%
  {   160,    1,   32,  568},  // 15    25  3.47%    0.92%  11.11%
  {   176,    1,   32,  332},  // 16    23  2.32%    0.92%  10.00%
  {   192,    1,   32,  350},  // 17    21  2.70%    0.92%   9.09%
  {   208,    1,   32,  266},  // 18    19  4.63%    0.92%   8.33%
  {   224,    1,   32,  327},  // 19    18  2.70%    0.92%   7.69%
  {   240,    1,   32,  236},  // 20    17  1.54%    0.92%   7.14%
  {   256,    1,   32,  346},  // 21    16  1.16%    0.92%   6.67%
  {   272,    1,   32,  213},  // 22    15  1.54%    0.92%   6.25%
  {   288,    1,   32,  229},  // 23    14  2.70%    0.92%   5.88%
  {   336,    1,   32,  339},  // 24    12  2.70%    0.92%  16.67%
  {   408,    1,   32,  210},  // 25    10  1.54%    0.92%  21.43%
  {   448,    1,   32,  200},  // 26     9  2.70%    0.92%   9.80%
  {   512,    1,   32,  401},  // 27     8  1.16%    0.92%  14.29%
  {   576,    2,   32,  244},  // 28    14  2.14%    0.92%  12.50%
  {   640,    2,   32,  185},  // 29    12  6.80%    0.92%  11.11%
  {   768,    2,   32,  243},  // 30    10  6.80%    0.93%  20.00%
  {   896,    2,   32,  224},  // 31     9  2.14%    0.92%  16.67%
  {  1024,    2,   32,  488},  // 32     8  0.58%    0.92%  14.29%
  {  1152,    3,   32,  193},  // 33    10  6.61%    0.93%  12.50%
  {  1280,    3,   32,  176},  // 34     9  6.61%    0.93%  11.11%
  {  1536,    3,   32,  191},  // 35     8  0.39%    0.92%  20.00%
  {  2048,    4,   32,  318},  // 36     8  0.29%    0.92%  33.33%
  {  2304,    4,   28,  187},  // 37     7  1.85%    0.92%  12.50%
  {  2688,    4,   24,  182},  // 38     6  1.85%    0.93%  16.67%
  {  3200,    4,   20,  177},  // 39     5  2.63%    0.93%  19.05%
  {  4096,    4,   16,  355},  // 40     4  0.29%    0.92%  28.00%
  {  4736,    5,   13,  195},  // 41     4  7.72%    1.77%  15.62%
  {  6144,    3,   10,  177},  // 42     2  0.39%    1.70%  29.73%
  {  7168,    7,    9,  178},  // 43     4  0.17%    1.70%  16.67%
  {  8192,    4,    8,  204},  // 44     2  0.29%    1.70%  14.29%
};
#else
#error "Unsupported TCMALLOC_PAGE_SHIFT value!"
#endif
#else
#if TCMALLOC_PAGE_SHIFT == 13
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2530},  //  0  1024  0.58%    0.42%   0.00%
  {    16,    1,   32, 2530},  //  1   512  0.58%    0.42% 100.00%
  {    32,    1,   32, 2534},  //  2   256  0.58%    0.42% 100.00%
  {    48,    1,   32, 2530},  //  3   170  0.97%    0.42%  50.00%
  {    64,    1,   32, 2530},  //  4   128  0.58%    0.42%  33.33%
  {    80,    1,   32, 1459},  //  5   102  0.97%    0.42%  25.00%
  {    96,    1,   32, 1007},  //  6    85  0.97%    0.42%  20.00%
  {   112,    1,   32,  605},  //  7    73  0.78%    0.42%  16.67%
  {   128,    1,   32,  678},  //  8    64  0.58%    0.42%  14.29%
  {   144,    1,   32,  481},  //  9    56  2.14%    0.42%  12.50%
  {   160,    1,   32,  415},  // 10    51  0.97%    0.42%  11.11%
  {   176,    1,   32,  267},  // 11    46  1.75%    0.42%  10.00%
  {   192,    1,   32,  316},  // 12    42  2.14%    0.42%   9.09%
  {   208,    1,   32,  245},  // 13    39  1.55%    0.42%   8.33%
  {   224,    1,   32,  226},  // 14    36  2.14%    0.42%   7.69%
  {   240,    1,   32,  219},  // 15    34  0.97%    0.42%   7.14%
  {   256,    1,   32,  368},  // 16    32  0.58%    0.42%   6.67%
  {   272,    1,   32,  206},  // 17    30  0.97%    0.42%   6.25%
  {   288,    1,   32,  226},  // 18    28  2.14%    0.42%   5.88%
  {   304,    1,   32,  191},  // 19    26  4.08%    0.42%   5.56%
  {   320,    1,   32,  217},  // 20    25  2.91%    0.42%   5.26%
  {   336,    1,   32,  230},  // 21    24  2.14%    0.42%   5.00%
  {   352,    1,   32,  184},  // 22    23  1.75%    0.42%   4.76%
  {   368,    1,   32,  173},  // 23    22  1.75%    0.42%   4.55%
  {   384,    1,   32,  194},  // 24    21  2.14%    0.42%   4.35%
  {   400,    1,   32,  183},  // 25    20  2.91%    0.42%   4.17%
  {   416,    1,   32,  180},  // 26    19  4.08%    0.43%   4.00%
  {   448,    1,   32,  210},  // 27    18  2.14%    0.42%   7.69%
  {   480,    1,   32,  188},  // 28    17  0.97%    0.42%   7.14%
  {   512,    1,   32,  280},  // 29    16  0.58%    0.42%   6.67%
  {   576,    1,   32,  212},  // 30    14  2.14%    0.42%  12.50%
  {   640,    1,   32,  196},  // 31    12  6.80%    0.43%  11.11%
  {   704,    1,   32,  184},  // 32    11  6.02%    0.43%  10.00%
  {   768,    1,   32,  183},  // 33    10  6.80%    0.43%   9.09%
  {   896,    1,   32,  191},  // 34     9  2.14%    0.43%  16.67%
  {  1024,    1,   32,  266},  // 35     8  0.58%    0.42%  14.29%
  {  1152,    2,   32,  187},  // 36    14  1.85%    0.43%  12.50%
  {  1280,    2,   32,  177},  // 37    12  6.52%    0.43%  11.11%
  {  1408,    2,   32,  173},  // 38    11  5.74%    0.43%  10.00%
  {  1536,    2,   32,  175},  // 39    10  6.52%    0.43%   9.09%
  {  1792,    2,   32,  174},  // 40     9  1.85%    0.43%  16.67%
  {  2048,    2,   32,  190},  // 41     8  0.29%    0.42%  14.29%
  {  2304,    2,   28,  172},  // 42     7  1.85%    0.43%  12.50%
  {  2688,    2,   24,  168},  // 43     6  1.85%    0.43%  16.67%
  {  2816,    3,   23,  162},  // 44     8  8.51%    0.44%   4.76%
  {  3200,    2,   20,  164},  // 45     5  2.63%    0.43%  13.64%
  {  3456,    3,   18,  161},  // 46     7  1.75%    0.43%   8.00%
  {  3584,    4,   18,  161},  // 47     9  1.71%    0.43%   3.70%
  {  4096,    1,   16,  254},  // 48     2  0.58%    0.43%  14.29%
  {  4736,    3,   13,  164},  // 49     5  3.83%    0.43%  15.62%
  {  5376,    2,   12,  161},  // 50     3  1.85%    0.43%  13.51%
  {  6144,    3,   10,  164},  // 51     4  0.19%    0.42%  14.29%
  {  6528,    4,   10,  159},  // 52     5  0.54%    0.43%   6.25%
  {  7168,    7,    9,  161},  // 53     8  0.08%    0.42%   9.80%
  {  8192,    1,    8,  193},  // 54     1  0.58%    0.03%  14.29%
  {  9472,    5,    6,  162},  // 55     4  7.61%    0.85%  15.62%
  { 10240,    4,    6,  159},  // 56     3  6.39%    0.85%   8.11%
  { 12288,    3,    5,  162},  // 57     2  0.19%    0.82%  20.00%
  { 13568,    5,    4,  159},  // 58     3  0.74%    0.82%  10.42%
  { 14336,    7,    4,  158},  // 59     4  0.08%    0.82%   5.66%
  { 16384,    2,    4,  164},  // 60     1  0.29%    0.03%  14.29%
  { 20480,    5,    3,  161},  // 61     2  0.12%    1.21%  25.00%
  { 24576,    3,    2,  161},  // 62     1  0.19%    0.03%  20.00%
  { 28672,    7,    2,  159},  // 63     2  0.08%    1.60%  16.67%
  { 32768,    4,    2,  166},  // 64     1  0.15%    0.03%  14.29%
  { 40960,    5,    2,  159},  // 65     1  0.12%    0.03%  25.00%
  { 49152,    6,    2,  158},  // 66     1  0.10%    0.03%  20.00%
  { 57344,    7,    2,  159},  // 67     1  0.08%    0.03%  16.67%
  { 65536,    8,    2,  161},  // 68     1  0.07%    0.03%  14.29%
  { 73728,    9,    2,  159},  // 69     1  0.07%    0.03%  12.50%
  { 81920,   10,    2,  159},  // 70     1  0.06%    0.03%  11.11%
  { 90112,   11,    2,  158},  // 71     1  0.05%    0.03%  10.00%
  { 98304,   12,    2,  159},  // 72     1  0.05%    0.03%   9.09%
  {106496,   13,    2,  158},  // 73     1  0.05%    0.03%   8.33%
  {114688,   14,    2,  158},  // 74     1  0.04%    0.03%   7.69%
  {131072,   16,    2,  158},  // 75     1  0.04%    0.03%  14.29%
  {139264,   17,    2,  159},  // 76     1  0.03%    0.03%   6.25%
  {147456,   18,    2,  158},  // 77     1  0.03%    0.03%   5.88%
  {155648,   19,    2,  158},  // 78     1  0.03%    0.03%   5.56%
  {172032,   21,    2,  158},  // 79     1  0.03%    0.03%  10.53%
  {188416,   23,    2,  158},  // 80     1  0.03%    0.03%   9.52%
  {204800,   25,    2,  158},  // 81     1  0.02%    0.03%   8.70%
  {221184,   27,    2,  158},  // 82     1  0.02%    0.03%   8.00%
  {237568,   29,    2,  158},  // 83     1  0.02%    0.03%   7.41%
  {262144,   32,    2,  158},  // 84     1  0.02%    0.03%  10.34%
};
#elif TCMALLOC_PAGE_SHIFT == 15
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2724},  //  0  4096  0.15%    1.60%   0.00%
  {    16,    1,   32, 2724},  //  1  2048  0.15%    1.60% 100.00%
  {    32,    1,   32, 2724},  //  2  1024  0.15%    1.60% 100.00%
  {    48,    1,   32, 2724},  //  3   682  0.24%    1.60%  50.00%
  {    64,    1,   32, 2724},  //  4   512  0.15%    1.60%  33.33%
  {    80,    1,   32, 1075},  //  5   409  0.29%    1.60%  25.00%
  {    96,    1,   32, 1011},  //  6   341  0.24%    1.60%  20.00%
  {   112,    1,   32,  599},  //  7   292  0.34%    1.60%  16.67%
  {   128,    1,   32,  781},  //  8   256  0.15%    1.60%  14.29%
  {   144,    1,   32,  427},  //  9   227  0.39%    1.60%  12.50%
  {   160,    1,   32,  374},  // 10   204  0.54%    1.60%  11.11%
  {   176,    1,   32,  282},  // 11   186  0.24%    1.60%  10.00%
  {   192,    1,   32,  329},  // 12   170  0.54%    1.60%   9.09%
  {   208,    1,   32,  230},  // 13   157  0.49%    1.60%   8.33%
  {   224,    1,   32,  263},  // 14   146  0.34%    1.60%   7.69%
  {   240,    1,   32,  262},  // 15   136  0.54%    1.60%   7.14%
  {   256,    1,   32,  395},  // 16   128  0.15%    1.60%   6.67%
  {   288,    1,   32,  274},  // 17   113  0.83%    1.60%  12.50%
  {   304,    1,   32,  210},  // 18   107  0.88%    1.60%   5.56%
  {   320,    1,   32,  200},  // 19   102  0.54%    1.60%   5.26%
  {   352,    1,   32,  262},  // 20    93  0.24%    1.60%  10.00%
  {   384,    1,   32,  214},  // 21    85  0.54%    1.60%   9.09%
  {   400,    1,   32,  185},  // 22    81  1.27%    1.60%   4.17%
  {   448,    1,   32,  215},  // 23    73  0.34%    1.60%  12.00%
  {   480,    1,   32,  218},  // 24    68  0.54%    1.60%   7.14%
  {   512,    1,   32,  241},  // 25    64  0.15%    1.60%   6.67%
  {   576,    1,   32,  210},  // 26    56  1.71%    1.60%  12.50%
  {   640,    1,   32,  225},  // 27    51  0.54%    1.60%  11.11%
  {   704,    1,   32,  205},  // 28    46  1.32%    1.60%  10.00%
  {   768,    1,   32,  197},  // 29    42  1.71%    1.60%   9.09%
  {   832,    1,   32,  198},  // 30    39  1.12%    1.60%   8.33%
  {   896,    1,   32,  188},  // 31    36  1.71%    1.60%   7.69%
  {  1024,    1,   32,  251},  // 32    32  0.15%    1.60%  14.29%
  {  1152,    1,   32,  197},  // 33    28  1.71%    1.60%  12.50%
  {  1280,    1,   32,  190},  // 34    25  2.49%    1.60%  11.11%
  {  1408,    1,   32,  190},  // 35    23  1.32%    1.60%  10.00%
  {  1536,    1,   32,  186},  // 36    21  1.71%    1.60%   9.09%
  {  1792,    1,   32,  185},  // 37    18  1.71%    1.60%  16.67%
  {  1920,    1,   32,  174},  // 38    17  0.54%    1.60%   7.14%
  {  2048,    1,   32,  189},  // 39    16  0.15%    1.60%   6.67%
  {  2176,    1,   30,  186},  // 40    15  0.54%    1.60%   6.25%
  {  2304,    1,   28,  176},  // 41    14  1.71%    1.60%   5.88%
  {  2432,    1,   26,  174},  // 42    13  3.66%    1.60%   5.56%
  {  2688,    1,   24,  178},  // 43    12  1.71%    1.60%  10.53%
  {  2944,    1,   22,  174},  // 44    11  1.32%    1.60%   9.52%
  {  3200,    1,   20,  176},  // 45    10  2.49%    1.60%   8.70%
  {  3584,    1,   18,  174},  // 46     9  1.71%    1.60%  12.00%
  {  4096,    1,   16,  237},  // 47     8  0.15%    1.60%  14.29%
  {  4608,    1,   14,  177},  // 48     7  1.71%    1.60%  12.50%
  {  5376,    1,   12,  174},  // 49     6  1.71%    1.60%  16.67%
  {  6528,    1,   10,  180},  // 50     5  0.54%    1.60%  21.43%
  {  7168,    2,    9,  172},  // 51     9  1.63%    1.60%   9.80%
  {  8192,    1,    8,  186},  // 52     4  0.15%    1.60%  14.29%
  {  9344,    2,    7,  174},  // 53     7  0.27%    1.60%  14.06%
  { 10880,    1,    6,  172},  // 54     3  0.54%    1.60%  16.44%
  { 13056,    2,    5,  173},  // 55     5  0.46%    1.60%  20.00%
  { 13952,    3,    4,  170},  // 56     7  0.70%    1.60%   6.86%
  { 16384,    1,    4,  181},  // 57     2  0.15%    1.60%  17.43%
  { 19072,    3,    3,  174},  // 58     5  3.04%    1.62%  16.41%
  { 21760,    2,    3,  172},  // 59     3  0.46%    1.60%  14.09%
  { 24576,    3,    2,  172},  // 60     4  0.05%    1.60%  12.94%
  { 26112,    4,    2,  170},  // 61     5  0.43%    1.60%   6.25%
  { 28672,    7,    2,  172},  // 62     8  0.02%    1.60%   9.80%
  { 32768,    1,    2,  177},  // 63     1  0.15%    0.03%  14.29%
  { 38144,    5,    2,  172},  // 64     4  6.90%    3.28%  16.41%
  { 40960,    4,    2,  170},  // 65     3  6.28%    3.28%   7.38%
  { 49152,    3,    2,  170},  // 66     2  0.05%    3.16%  20.00%
  { 57344,    7,    2,  172},  // 67     4  0.02%    3.16%  16.67%
  { 65536,    2,    2,  173},  // 68     1  0.07%    0.03%  14.29%
  { 81920,    5,    2,  172},  // 69     2  0.03%    4.72%  25.00%
  { 98304,    3,    2,  170},  // 70     1  0.05%    0.03%  20.00%
  {114688,    7,    2,  170},  // 71     2  0.02%    6.28%  16.67%
  {131072,    4,    2,  180},  // 72     1  0.04%    0.03%  14.29%
  {163840,    5,    2,  170},  // 73     1  0.03%    0.03%  25.00%
  {196608,    6,    2,  170},  // 74     1  0.02%    0.03%  20.00%
  {229376,    7,    2,  166},  // 75     1  0.02%    0.03%  16.67%
  {262144,    8,    2,  172},  // 76     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 18
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2502},  //  0 32768  0.02%   12.53%   0.00%
  {    16,    1,   32, 2502},  //  1 16384  0.02%   12.53% 100.00%
  {    32,    1,   32, 2506},  //  2  8192  0.02%   12.53% 100.00%
  {    48,    1,   32, 2502},  //  3  5461  0.02%   12.53%  50.00%
  {    64,    1,   32, 2502},  //  4  4096  0.02%   12.53%  33.33%
  {    80,    1,   32, 1473},  //  5  3276  0.04%   12.53%  25.00%
  {    96,    1,   32,  666},  //  6  2730  0.04%   12.53%  20.00%
  {   112,    1,   32,  563},  //  7  2340  0.04%   12.53%  16.67%
  {   128,    1,   32,  585},  //  8  2048  0.02%   12.53%  14.29%
  {   144,    1,   32,  503},  //  9  1820  0.04%   12.53%  12.50%
  {   160,    1,   32,  443},  // 10  1638  0.04%   12.53%  11.11%
  {   176,    1,   32,  248},  // 11  1489  0.05%   12.53%  10.00%
  {   192,    1,   32,  275},  // 12  1365  0.04%   12.53%   9.09%
  {   208,    1,   32,  227},  // 13  1260  0.04%   12.53%   8.33%
  {   224,    1,   32,  345},  // 14  1170  0.04%   12.53%   7.69%
  {   256,    1,   32,  415},  // 15  1024  0.02%   12.53%  14.29%
  {   288,    1,   32,  334},  // 16   910  0.04%   12.53%  12.50%
  {   320,    1,   32,  271},  // 17   819  0.04%   12.53%  11.11%
  {   336,    1,   32,  259},  // 18   780  0.04%   12.53%   5.00%
  {   368,    1,   32,  187},  // 19   712  0.07%   12.53%   9.52%
  {   400,    1,   32,  221},  // 20   655  0.07%   12.53%   8.70%
  {   448,    1,   32,  222},  // 21   585  0.04%   12.53%  12.00%
  {   480,    1,   32,  181},  // 22   546  0.04%   12.53%   7.14%
  {   512,    1,   32,  232},  // 23   512  0.02%   12.53%   6.67%
  {   576,    1,   32,  215},  // 24   455  0.04%   12.53%  12.50%
  {   640,    1,   32,  208},  // 25   409  0.16%   12.53%  11.11%
  {   704,    1,   32,  184},  // 26   372  0.12%   12.53%  10.00%
  {   768,    1,   32,  181},  // 27   341  0.12%   12.53%   9.09%
  {   896,    1,   32,  193},  // 28   292  0.21%   12.53%  16.67%
  {  1024,    1,   32,  244},  // 29   256  0.02%   12.53%  14.29%
  {  1152,    1,   32,  186},  // 30   227  0.26%   12.53%  12.50%
  {  1280,    1,   32,  177},  // 31   204  0.41%   12.53%  11.11%
  {  1408,    1,   32,  167},  // 32   186  0.12%   12.53%  10.00%
  {  1536,    1,   32,  169},  // 33   170  0.41%   12.53%   9.09%
  {  1664,    1,   32,  199},  // 34   157  0.36%   12.53%   8.33%
  {  1920,    1,   32,  193},  // 35   136  0.41%   12.53%  15.38%
  {  2048,    1,   32,  182},  // 36   128  0.02%   12.53%   6.67%
  {  2176,    1,   30,  233},  // 37   120  0.41%   12.53%   6.25%
  {  2304,    1,   28,  170},  // 38   113  0.70%   12.53%   5.88%
  {  2560,    1,   25,  167},  // 39   102  0.41%   12.53%  11.11%
  {  2816,    1,   23,  160},  // 40    93  0.12%   12.53%  10.00%
  {  3072,    1,   21,  160},  // 41    85  0.41%   12.53%   9.09%
  {  3328,    1,   19,  167},  // 42    78  0.99%   12.54%   8.33%
  {  3584,    1,   18,  159},  // 43    73  0.21%   12.53%   7.69%
  {  3840,    1,   17,  159},  // 44    68  0.41%   12.53%   7.14%
  {  4096,    1,   16,  217},  // 45    64  0.02%   12.53%   6.67%
  {  4224,    1,   15,  161},  // 46    62  0.12%   12.53%   3.12%
  {  4736,    1,   13,  164},  // 47    55  0.65%   12.54%  12.12%
  {  5120,    1,   12,  162},  // 48    51  0.41%   12.53%   8.11%
  {  5632,    1,   11,  164},  // 49    46  1.19%   12.54%  10.00%
  {  6144,    1,   10,  159},  // 50    42  1.58%   12.54%   9.09%
  {  6528,    1,   10,  160},  // 51    40  0.41%   12.53%   6.25%
  {  7168,    1,    9,  158},  // 52    36  1.58%   12.54%   9.80%
  {  8192,    1,    8,  176},  // 53    32  0.02%   12.53%  14.29%
  {  8704,    1,    7,  158},  // 54    30  0.41%   12.54%   6.25%
  {  9344,    1,    7,  159},  // 55    28  0.21%   12.53%   7.35%
  { 10368,    1,    6,  158},  // 56    25  1.14%   12.54%  10.96%
  { 11392,    1,    5,  160},  // 57    23  0.07%   12.53%   9.88%
  { 12416,    1,    5,  159},  // 58    21  0.56%   12.54%   8.99%
  { 13056,    1,    5,  158},  // 59    20  0.41%   12.54%   5.15%
  { 13696,    1,    4,  156},  // 60    19  0.75%   12.54%   4.90%
  { 14464,    1,    4,  158},  // 61    18  0.70%   12.54%   5.61%
  { 15360,    1,    4,  158},  // 62    17  0.41%   12.54%   6.19%
  { 16384,    1,    4,  164},  // 63    16  0.02%   12.53%   6.67%
  { 17408,    1,    3,  158},  // 64    15  0.41%   12.54%   6.25%
  { 18688,    1,    3,  159},  // 65    14  0.21%   12.54%   7.35%
  { 20096,    1,    3,  156},  // 66    13  0.36%   12.54%   7.53%
  { 21760,    1,    3,  158},  // 67    12  0.41%   12.54%   8.28%
  { 23808,    1,    2,  159},  // 68    11  0.12%   12.53%   9.41%
  { 26112,    1,    2,  158},  // 69    10  0.41%   12.54%   9.68%
  { 29056,    1,    2,  156},  // 70     9  0.26%   12.54%  11.27%
  { 32768,    1,    2,  176},  // 71     8  0.02%   12.53%  12.78%
  { 37376,    1,    2,  158},  // 72     7  0.21%   12.54%  14.06%
  { 43648,    1,    2,  156},  // 73     6  0.12%   12.54%  16.78%
  { 45568,    2,    2,  156},  // 74    11  4.40%   12.63%   4.40%
  { 52352,    1,    2,  156},  // 75     5  0.16%   12.54%  14.89%
  { 56064,    2,    2,  156},  // 76     9  3.77%   12.63%   7.09%
  { 65536,    1,    2,  158},  // 77     4  0.02%   12.53%  16.89%
  { 74880,    2,    2,  156},  // 78     7  0.03%   12.53%  14.26%
  { 87296,    1,    2,  156},  // 79     3  0.12%   12.54%  16.58%
  {104832,    2,    2,  156},  // 80     5  0.03%   12.54%  20.09%
  {112256,    3,    2,  156},  // 81     7  0.09%   12.54%   7.08%
  {131072,    1,    2,  156},  // 82     2  0.02%   12.53%  16.76%
  {149760,    3,    2,  156},  // 83     5  4.79%   12.88%  14.26%
  {174720,    2,    2,  156},  // 84     3  0.03%   12.54%  16.67%
  {196608,    3,    2,  156},  // 85     4  0.01%   12.53%  12.53%
  {209664,    4,    2,  156},  // 86     5  0.03%   12.54%   6.64%
  {262144,    1,    2,  158},  // 87     1  0.02%    0.03%  25.03%
};
#elif TCMALLOC_PAGE_SHIFT == 12
static_assert(kMaxSize == 8192, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = false,
  .span_size = 48,
  .sampling_rate = 524288,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 3071},  //  0   512  1.16%    0.92%   0.00%
  {    16,    1,   32, 3071},  //  1   256  1.16%    0.92% 100.00%
  {    32,    1,   32, 3078},  //  2   128  1.16%    0.92% 100.00%
  {    48,    1,   32, 3071},  //  3    85  1.54%    0.92%  50.00%
  {    64,    1,   32, 3071},  //  4    64  1.16%    0.92%  33.33%
  {    80,    1,   32, 2098},  //  5    51  1.54%    0.92%  25.00%
  {    96,    1,   32, 1380},  //  6    42  2.70%    0.92%  20.00%
  {   112,    1,   32, 1108},  //  7    36  2.70%    0.92%  16.67%
  {   128,    1,   32, 1051},  //  8    32  1.16%    0.92%  14.29%
  {   144,    1,   32,  865},  //  9    28  2.70%    0.92%  12.50%
  {   160,    1,   32,  790},  // 10    25  3.47%    0.92%  11.11%
  {   176,    1,   32,  432},  // 11    23  2.32%    0.92%  10.00%
  {   192,    1,   32,  459},  // 12    21  2.70%    0.92%   9.09%
  {   208,    1,   32,  331},  // 13    19  4.63%    0.92%   8.33%
  {   224,    1,   32,  423},  // 14    18  2.70%    0.92%   7.69%
  {   240,    1,   32,  286},  // 15    17  1.54%    0.92%   7.14%
  {   256,    1,   32,  451},  // 16    16  1.16%    0.92%   6.67%
  {   272,    1,   32,  252},  // 17    15  1.54%    0.92%   6.25%
  {   288,    1,   32,  276},  // 18    14  2.70%    0.92%   5.88%
  {   304,    1,   32,  247},  // 19    13  4.63%    0.92%   5.56%
  {   336,    1,   32,  384},  // 20    12  2.70%    0.92%  10.53%
  {   368,    1,   32,  223},  // 21    11  2.32%    0.92%   9.52%
  {   400,    1,   32,  210},  // 22    10  3.47%    0.92%   8.70%
  {   448,    1,   32,  234},  // 23     9  2.70%    0.92%  12.00%
  {   512,    1,   32,  535},  // 24     8  1.16%    0.92%  14.29%
  {   576,    2,   32,  298},  // 25    14  2.14%    0.92%  12.50%
  {   640,    2,   32,  208},  // 26    12  6.80%    0.92%  11.11%
  {   768,    2,   32,  295},  // 27    10  6.80%    0.93%  20.00%
  {   896,    2,   32,  267},  // 28     9  2.14%    0.92%  16.67%
  {  1024,    2,   32,  669},  // 29     8  0.58%    0.92%  14.29%
  {  1152,    3,   32,  222},  // 30    10  6.61%    0.93%  12.50%
  {  1280,    3,   32,  193},  // 31     9  6.61%    0.93%  11.11%
  {  1536,    3,   32,  217},  // 32     8  0.39%    0.92%  20.00%
  {  1792,    4,   32,  199},  // 33     9  1.85%    0.92%  16.67%
  {  2048,    4,   32,  402},  // 34     8  0.29%    0.92%  14.29%
  {  2304,    4,   28,  210},  // 35     7  1.85%    0.92%  12.50%
  {  2688,    4,   24,  204},  // 36     6  1.85%    0.93%  16.67%
  {  3200,    4,   20,  196},  // 37     5  2.63%    0.93%  19.05%
  {  3584,    7,   18,  193},  // 38     8  0.17%    0.92%  12.00%
  {  4096,    4,   16,  463},  // 39     4  0.29%    0.92%  14.29%
  {  4736,    5,   13,  223},  // 40     4  7.72%    1.77%  15.62%
  {  5376,    4,   12,  192},  // 41     3  1.85%    1.72%  13.51%
  {  6144,    3,   10,  195},  // 42     2  0.39%    1.70%  14.29%
  {  7168,    7,    9,  198},  // 43     4  0.17%    1.70%  16.67%
  {  8192,    4,    8,  237},  // 44     2  0.29%    1.70%  14.29%
};
#else
#error "Unsupported TCMALLOC_PAGE_SHIFT value!"
#endif
#endif
// clang-format on

static_assert(sizeof(List) / sizeof(List[0]) <= kNumBaseClasses);
extern constexpr SizeClasses kLegacySizeClasses{List, Assumptions};

}  // namespace tcmalloc_internal
}  // namespace tcmalloc
GOOGLE_MALLOC_SECTION_END
