// SPDX-License-Identifier: GPL-3.0-or-later

package yugabytedb

import (
	"context"
	"net/http"
	"net/http/httptest"
	"net/url"
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/netdata/netdata/go/plugins/plugin/go.d/agent/module"
	"github.com/netdata/netdata/go/plugins/plugin/go.d/pkg/web"
)

var (
	dataConfigJSON, _ = os.ReadFile("testdata/config.json")
	dataConfigYAML, _ = os.ReadFile("testdata/config.yaml")

	dataVer223Master, _  = os.ReadFile("testdata/v2.23.1/master-metrics.txt")
	dataVer223Tserver, _ = os.ReadFile("testdata/v2.23.1/tserver-metrics.txt")
	dataVer223YCQL, _    = os.ReadFile("testdata/v2.23.1/ycql-metrics.txt")
	dataVer223YSQL, _    = os.ReadFile("testdata/v2.23.1/ysql-metrics.txt")
)

func Test_testDataIsValid(t *testing.T) {
	for name, data := range map[string][]byte{
		"dataConfigJSON":    dataConfigJSON,
		"dataConfigYAML":    dataConfigYAML,
		"dataVer223Master":  dataVer223Master,
		"dataVer223Tserver": dataVer223Tserver,
		"dataVer223YCQL":    dataVer223YCQL,
		"dataVer223YSQL":    dataVer223YSQL,
	} {
		require.NotNil(t, data, name)
	}
}

func TestCollector_ConfigurationSerialize(t *testing.T) {
	module.TestConfigurationSerialize(t, &Collector{}, dataConfigJSON, dataConfigYAML)
}

func TestCollector_Init(t *testing.T) {
	tests := map[string]struct {
		wantFail bool
		config   Config
	}{
		"success with default": {
			wantFail: false,
			config:   New().Config,
		},
		"fail when URL not set": {
			wantFail: true,
			config: Config{
				HTTPConfig: web.HTTPConfig{
					RequestConfig: web.RequestConfig{URL: ""},
				},
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			collr := New()
			collr.Config = test.config

			if test.wantFail {
				assert.Error(t, collr.Init(context.Background()))
			} else {
				assert.NoError(t, collr.Init(context.Background()))
			}
		})
	}
}

func TestCollector_Check(t *testing.T) {
	tests := map[string]struct {
		wantFail bool
		prepare  func(t *testing.T) (nu *Collector, cleanup func())
	}{
		"success on master response": {
			wantFail: false,
			prepare:  caseMasterOk,
		},
		"success on tserver response": {
			wantFail: false,
			prepare:  caseTserverOk,
		},
		"success on ycql response": {
			wantFail: false,
			prepare:  caseYcqlOk,
		},
		"success on ysql response": {
			wantFail: false,
			prepare:  caseYsqlOk,
		},
		"fail on invalid data response": {
			wantFail: true,
			prepare:  caseInvalidDataResponse,
		},
		"fail on connection refused": {
			wantFail: true,
			prepare:  caseConnectionRefused,
		},
		"fail on 404 response": {
			wantFail: true,
			prepare:  case404,
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			collr, cleanup := test.prepare(t)
			defer cleanup()

			if test.wantFail {
				assert.Error(t, collr.Check(context.Background()))
			} else {
				assert.NoError(t, collr.Check(context.Background()))
			}
		})
	}
}

func TestCollector_Charts(t *testing.T) {
	assert.NotNil(t, New().Charts())
}

func TestCollector_Collect(t *testing.T) {
	tests := map[string]struct {
		prepare         func(t *testing.T) (nu *Collector, cleanup func())
		wantNumOfCharts int
		wantMetrics     map[string]int64
	}{
		"success on master response": {
			prepare:         caseMasterOk,
			wantNumOfCharts: 423,
			wantMetrics: map[string]int64{
				"handler_latency_yb_consensus_ConsensusService_ChangeConfig_count":                           0,
				"handler_latency_yb_consensus_ConsensusService_ChangeConfig_sum":                             0,
				"handler_latency_yb_consensus_ConsensusService_GetConsensusState_count":                      0,
				"handler_latency_yb_consensus_ConsensusService_GetConsensusState_sum":                        0,
				"handler_latency_yb_consensus_ConsensusService_GetLastOpId_count":                            0,
				"handler_latency_yb_consensus_ConsensusService_GetLastOpId_sum":                              0,
				"handler_latency_yb_consensus_ConsensusService_GetNodeInstance_count":                        1,
				"handler_latency_yb_consensus_ConsensusService_GetNodeInstance_sum":                          570,
				"handler_latency_yb_consensus_ConsensusService_LeaderElectionLost_count":                     0,
				"handler_latency_yb_consensus_ConsensusService_LeaderElectionLost_sum":                       0,
				"handler_latency_yb_consensus_ConsensusService_LeaderStepDown_count":                         0,
				"handler_latency_yb_consensus_ConsensusService_LeaderStepDown_sum":                           0,
				"handler_latency_yb_consensus_ConsensusService_MultiRaftUpdateConsensus_count":               0,
				"handler_latency_yb_consensus_ConsensusService_MultiRaftUpdateConsensus_sum":                 0,
				"handler_latency_yb_consensus_ConsensusService_RequestConsensusVote_count":                   0,
				"handler_latency_yb_consensus_ConsensusService_RequestConsensusVote_sum":                     0,
				"handler_latency_yb_consensus_ConsensusService_RunLeaderElection_count":                      0,
				"handler_latency_yb_consensus_ConsensusService_RunLeaderElection_sum":                        0,
				"handler_latency_yb_consensus_ConsensusService_StartRemoteBootstrap_count":                   0,
				"handler_latency_yb_consensus_ConsensusService_StartRemoteBootstrap_sum":                     0,
				"handler_latency_yb_consensus_ConsensusService_UnsafeChangeConfig_count":                     0,
				"handler_latency_yb_consensus_ConsensusService_UnsafeChangeConfig_sum":                       0,
				"handler_latency_yb_consensus_ConsensusService_UpdateConsensus_count":                        0,
				"handler_latency_yb_consensus_ConsensusService_UpdateConsensus_sum":                          0,
				"handler_latency_yb_master_MasterClient_GetIndexBackfillProgress_count":                      0,
				"handler_latency_yb_master_MasterClient_GetIndexBackfillProgress_sum":                        0,
				"handler_latency_yb_master_MasterClient_GetStatefulServiceLocation_count":                    0,
				"handler_latency_yb_master_MasterClient_GetStatefulServiceLocation_sum":                      0,
				"handler_latency_yb_master_MasterClient_GetTableLocations_count":                             30392,
				"handler_latency_yb_master_MasterClient_GetTableLocations_sum":                               1556670,
				"handler_latency_yb_master_MasterClient_GetTabletLocations_count":                            0,
				"handler_latency_yb_master_MasterClient_GetTabletLocations_sum":                              0,
				"handler_latency_yb_master_MasterClient_GetTransactionStatusTablets_count":                   0,
				"handler_latency_yb_master_MasterClient_GetTransactionStatusTablets_sum":                     0,
				"handler_latency_yb_master_MasterClient_GetYsqlCatalogConfig_count":                          0,
				"handler_latency_yb_master_MasterClient_GetYsqlCatalogConfig_sum":                            0,
				"handler_latency_yb_master_MasterClient_RedisConfigGet_count":                                0,
				"handler_latency_yb_master_MasterClient_RedisConfigGet_sum":                                  0,
				"handler_latency_yb_master_MasterClient_RedisConfigSet_count":                                0,
				"handler_latency_yb_master_MasterClient_RedisConfigSet_sum":                                  0,
				"handler_latency_yb_master_MasterClient_ReservePgsqlOids_count":                              0,
				"handler_latency_yb_master_MasterClient_ReservePgsqlOids_sum":                                0,
				"handler_latency_yb_master_MasterDdl_AlterNamespace_count":                                   0,
				"handler_latency_yb_master_MasterDdl_AlterNamespace_sum":                                     0,
				"handler_latency_yb_master_MasterDdl_AlterTable_count":                                       0,
				"handler_latency_yb_master_MasterDdl_AlterTable_sum":                                         0,
				"handler_latency_yb_master_MasterDdl_BackfillIndex_count":                                    0,
				"handler_latency_yb_master_MasterDdl_BackfillIndex_sum":                                      0,
				"handler_latency_yb_master_MasterDdl_CreateNamespace_count":                                  0,
				"handler_latency_yb_master_MasterDdl_CreateNamespace_sum":                                    0,
				"handler_latency_yb_master_MasterDdl_CreateTable_count":                                      0,
				"handler_latency_yb_master_MasterDdl_CreateTable_sum":                                        0,
				"handler_latency_yb_master_MasterDdl_CreateTablegroup_count":                                 0,
				"handler_latency_yb_master_MasterDdl_CreateTablegroup_sum":                                   0,
				"handler_latency_yb_master_MasterDdl_CreateUDType_count":                                     0,
				"handler_latency_yb_master_MasterDdl_CreateUDType_sum":                                       0,
				"handler_latency_yb_master_MasterDdl_DeleteNamespace_count":                                  0,
				"handler_latency_yb_master_MasterDdl_DeleteNamespace_sum":                                    0,
				"handler_latency_yb_master_MasterDdl_DeleteTable_count":                                      0,
				"handler_latency_yb_master_MasterDdl_DeleteTable_sum":                                        0,
				"handler_latency_yb_master_MasterDdl_DeleteTablegroup_count":                                 0,
				"handler_latency_yb_master_MasterDdl_DeleteTablegroup_sum":                                   0,
				"handler_latency_yb_master_MasterDdl_DeleteUDType_count":                                     0,
				"handler_latency_yb_master_MasterDdl_DeleteUDType_sum":                                       0,
				"handler_latency_yb_master_MasterDdl_GetBackfillJobs_count":                                  0,
				"handler_latency_yb_master_MasterDdl_GetBackfillJobs_sum":                                    0,
				"handler_latency_yb_master_MasterDdl_GetBackfillStatus_count":                                0,
				"handler_latency_yb_master_MasterDdl_GetBackfillStatus_sum":                                  0,
				"handler_latency_yb_master_MasterDdl_GetColocatedTabletSchema_count":                         0,
				"handler_latency_yb_master_MasterDdl_GetColocatedTabletSchema_sum":                           0,
				"handler_latency_yb_master_MasterDdl_GetNamespaceInfo_count":                                 1,
				"handler_latency_yb_master_MasterDdl_GetNamespaceInfo_sum":                                   99,
				"handler_latency_yb_master_MasterDdl_GetTableDiskSize_count":                                 0,
				"handler_latency_yb_master_MasterDdl_GetTableDiskSize_sum":                                   0,
				"handler_latency_yb_master_MasterDdl_GetTableSchema_count":                                   30386,
				"handler_latency_yb_master_MasterDdl_GetTableSchema_sum":                                     2398868,
				"handler_latency_yb_master_MasterDdl_GetTablegroupSchema_count":                              0,
				"handler_latency_yb_master_MasterDdl_GetTablegroupSchema_sum":                                0,
				"handler_latency_yb_master_MasterDdl_GetUDTypeInfo_count":                                    0,
				"handler_latency_yb_master_MasterDdl_GetUDTypeInfo_sum":                                      0,
				"handler_latency_yb_master_MasterDdl_IsAlterTableDone_count":                                 0,
				"handler_latency_yb_master_MasterDdl_IsAlterTableDone_sum":                                   0,
				"handler_latency_yb_master_MasterDdl_IsCreateNamespaceDone_count":                            0,
				"handler_latency_yb_master_MasterDdl_IsCreateNamespaceDone_sum":                              0,
				"handler_latency_yb_master_MasterDdl_IsCreateTableDone_count":                                0,
				"handler_latency_yb_master_MasterDdl_IsCreateTableDone_sum":                                  0,
				"handler_latency_yb_master_MasterDdl_IsDeleteNamespaceDone_count":                            0,
				"handler_latency_yb_master_MasterDdl_IsDeleteNamespaceDone_sum":                              0,
				"handler_latency_yb_master_MasterDdl_IsDeleteTableDone_count":                                0,
				"handler_latency_yb_master_MasterDdl_IsDeleteTableDone_sum":                                  0,
				"handler_latency_yb_master_MasterDdl_IsTruncateTableDone_count":                              0,
				"handler_latency_yb_master_MasterDdl_IsTruncateTableDone_sum":                                0,
				"handler_latency_yb_master_MasterDdl_IsYsqlDdlVerificationDone_count":                        0,
				"handler_latency_yb_master_MasterDdl_IsYsqlDdlVerificationDone_sum":                          0,
				"handler_latency_yb_master_MasterDdl_LaunchBackfillIndexForTable_count":                      0,
				"handler_latency_yb_master_MasterDdl_LaunchBackfillIndexForTable_sum":                        0,
				"handler_latency_yb_master_MasterDdl_ListNamespaces_count":                                   0,
				"handler_latency_yb_master_MasterDdl_ListNamespaces_sum":                                     0,
				"handler_latency_yb_master_MasterDdl_ListTablegroups_count":                                  0,
				"handler_latency_yb_master_MasterDdl_ListTablegroups_sum":                                    0,
				"handler_latency_yb_master_MasterDdl_ListTables_count":                                       1,
				"handler_latency_yb_master_MasterDdl_ListTables_sum":                                         895,
				"handler_latency_yb_master_MasterDdl_ListUDTypes_count":                                      0,
				"handler_latency_yb_master_MasterDdl_ListUDTypes_sum":                                        0,
				"handler_latency_yb_master_MasterDdl_ReportYsqlDdlTxnStatus_count":                           0,
				"handler_latency_yb_master_MasterDdl_ReportYsqlDdlTxnStatus_sum":                             0,
				"handler_latency_yb_master_MasterDdl_TruncateTable_count":                                    0,
				"handler_latency_yb_master_MasterDdl_TruncateTable_sum":                                      0,
				"handler_latency_yb_tserver_PgClientService_ActiveSessionHistory_count":                      0,
				"handler_latency_yb_tserver_PgClientService_ActiveSessionHistory_sum":                        0,
				"handler_latency_yb_tserver_PgClientService_AlterDatabase_count":                             0,
				"handler_latency_yb_tserver_PgClientService_AlterDatabase_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_AlterTable_count":                                0,
				"handler_latency_yb_tserver_PgClientService_AlterTable_sum":                                  0,
				"handler_latency_yb_tserver_PgClientService_BackfillIndex_count":                             0,
				"handler_latency_yb_tserver_PgClientService_BackfillIndex_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_CancelTransaction_count":                         0,
				"handler_latency_yb_tserver_PgClientService_CancelTransaction_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_CheckIfPitrActive_count":                         0,
				"handler_latency_yb_tserver_PgClientService_CheckIfPitrActive_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_CreateDatabase_count":                            0,
				"handler_latency_yb_tserver_PgClientService_CreateDatabase_sum":                              0,
				"handler_latency_yb_tserver_PgClientService_CreateReplicationSlot_count":                     0,
				"handler_latency_yb_tserver_PgClientService_CreateReplicationSlot_sum":                       0,
				"handler_latency_yb_tserver_PgClientService_CreateSequencesDataTable_count":                  0,
				"handler_latency_yb_tserver_PgClientService_CreateSequencesDataTable_sum":                    0,
				"handler_latency_yb_tserver_PgClientService_CreateTable_count":                               0,
				"handler_latency_yb_tserver_PgClientService_CreateTable_sum":                                 0,
				"handler_latency_yb_tserver_PgClientService_CreateTablegroup_count":                          0,
				"handler_latency_yb_tserver_PgClientService_CreateTablegroup_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_DeleteDBSequences_count":                         0,
				"handler_latency_yb_tserver_PgClientService_DeleteDBSequences_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_DeleteSequenceTuple_count":                       0,
				"handler_latency_yb_tserver_PgClientService_DeleteSequenceTuple_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_DropDatabase_count":                              0,
				"handler_latency_yb_tserver_PgClientService_DropDatabase_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_DropReplicationSlot_count":                       0,
				"handler_latency_yb_tserver_PgClientService_DropReplicationSlot_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_DropTable_count":                                 0,
				"handler_latency_yb_tserver_PgClientService_DropTable_sum":                                   0,
				"handler_latency_yb_tserver_PgClientService_DropTablegroup_count":                            0,
				"handler_latency_yb_tserver_PgClientService_DropTablegroup_sum":                              0,
				"handler_latency_yb_tserver_PgClientService_FetchData_count":                                 0,
				"handler_latency_yb_tserver_PgClientService_FetchData_sum":                                   0,
				"handler_latency_yb_tserver_PgClientService_FetchSequenceTuple_count":                        0,
				"handler_latency_yb_tserver_PgClientService_FetchSequenceTuple_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_FinishTransaction_count":                         0,
				"handler_latency_yb_tserver_PgClientService_FinishTransaction_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_GetActiveTransactionList_count":                  0,
				"handler_latency_yb_tserver_PgClientService_GetActiveTransactionList_sum":                    0,
				"handler_latency_yb_tserver_PgClientService_GetCatalogMasterVersion_count":                   0,
				"handler_latency_yb_tserver_PgClientService_GetCatalogMasterVersion_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_GetDatabaseInfo_count":                           0,
				"handler_latency_yb_tserver_PgClientService_GetDatabaseInfo_sum":                             0,
				"handler_latency_yb_tserver_PgClientService_GetIndexBackfillProgress_count":                  0,
				"handler_latency_yb_tserver_PgClientService_GetIndexBackfillProgress_sum":                    0,
				"handler_latency_yb_tserver_PgClientService_GetLockStatus_count":                             0,
				"handler_latency_yb_tserver_PgClientService_GetLockStatus_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_GetNewObjectId_count":                            0,
				"handler_latency_yb_tserver_PgClientService_GetNewObjectId_sum":                              0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlotStatus_count":                  0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlotStatus_sum":                    0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlot_count":                        0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlot_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_GetTableDiskSize_count":                          0,
				"handler_latency_yb_tserver_PgClientService_GetTableDiskSize_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_GetTableKeyRanges_count":                         0,
				"handler_latency_yb_tserver_PgClientService_GetTableKeyRanges_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_GetTablePartitionList_count":                     0,
				"handler_latency_yb_tserver_PgClientService_GetTablePartitionList_sum":                       0,
				"handler_latency_yb_tserver_PgClientService_GetTserverCatalogVersionInfo_count":              0,
				"handler_latency_yb_tserver_PgClientService_GetTserverCatalogVersionInfo_sum":                0,
				"handler_latency_yb_tserver_PgClientService_Heartbeat_count":                                 0,
				"handler_latency_yb_tserver_PgClientService_Heartbeat_sum":                                   0,
				"handler_latency_yb_tserver_PgClientService_InsertSequenceTuple_count":                       0,
				"handler_latency_yb_tserver_PgClientService_InsertSequenceTuple_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_IsInitDbDone_count":                              0,
				"handler_latency_yb_tserver_PgClientService_IsInitDbDone_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_IsObjectPartOfXRepl_count":                       0,
				"handler_latency_yb_tserver_PgClientService_IsObjectPartOfXRepl_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_ListLiveTabletServers_count":                     0,
				"handler_latency_yb_tserver_PgClientService_ListLiveTabletServers_sum":                       0,
				"handler_latency_yb_tserver_PgClientService_ListReplicationSlots_count":                      0,
				"handler_latency_yb_tserver_PgClientService_ListReplicationSlots_sum":                        0,
				"handler_latency_yb_tserver_PgClientService_OpenTable_count":                                 0,
				"handler_latency_yb_tserver_PgClientService_OpenTable_sum":                                   0,
				"handler_latency_yb_tserver_PgClientService_Perform_count":                                   0,
				"handler_latency_yb_tserver_PgClientService_Perform_sum":                                     0,
				"handler_latency_yb_tserver_PgClientService_ReadSequenceTuple_count":                         0,
				"handler_latency_yb_tserver_PgClientService_ReadSequenceTuple_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_ReserveOids_count":                               0,
				"handler_latency_yb_tserver_PgClientService_ReserveOids_sum":                                 0,
				"handler_latency_yb_tserver_PgClientService_RollbackToSubTransaction_count":                  0,
				"handler_latency_yb_tserver_PgClientService_RollbackToSubTransaction_sum":                    0,
				"handler_latency_yb_tserver_PgClientService_SetActiveSubTransaction_count":                   0,
				"handler_latency_yb_tserver_PgClientService_SetActiveSubTransaction_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_TabletServerCount_count":                         0,
				"handler_latency_yb_tserver_PgClientService_TabletServerCount_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_TabletsMetadata_count":                           0,
				"handler_latency_yb_tserver_PgClientService_TabletsMetadata_sum":                             0,
				"handler_latency_yb_tserver_PgClientService_TruncateTable_count":                             0,
				"handler_latency_yb_tserver_PgClientService_TruncateTable_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_UpdateSequenceTuple_count":                       0,
				"handler_latency_yb_tserver_PgClientService_UpdateSequenceTuple_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_ValidatePlacement_count":                         0,
				"handler_latency_yb_tserver_PgClientService_ValidatePlacement_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_WaitForBackendsCatalogVersion_count":             0,
				"handler_latency_yb_tserver_PgClientService_WaitForBackendsCatalogVersion_sum":               0,
				"handler_latency_yb_tserver_PgClientService_YCQLStatementStats_count":                        0,
				"handler_latency_yb_tserver_PgClientService_YCQLStatementStats_sum":                          0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession_count":        0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession_sum":          0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession_count": 0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession_sum":   0,
				"handler_latency_yb_tserver_RemoteBootstrapService_ChangePeerRole_count":                     0,
				"handler_latency_yb_tserver_RemoteBootstrapService_ChangePeerRole_sum":                       0,
				"handler_latency_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive_count":  0,
				"handler_latency_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive_sum":    0,
				"handler_latency_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession_count":          0,
				"handler_latency_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession_sum":            0,
				"handler_latency_yb_tserver_RemoteBootstrapService_FetchData_count":                          0,
				"handler_latency_yb_tserver_RemoteBootstrapService_FetchData_sum":                            0,
				"handler_latency_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive_count":                 0,
				"handler_latency_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive_sum":                   0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RegisterLogAnchor_count":                  0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RegisterLogAnchor_sum":                    0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession_count":       0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession_sum":         0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor_count":                0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor_sum":                  0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UpdateLogAnchor_count":                    0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UpdateLogAnchor_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_AbortTransaction_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_AbortTransaction_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_AcquireObjectLocks_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_AcquireObjectLocks_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_AdminExecutePgsql_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_AdminExecutePgsql_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_CancelTransaction_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_CancelTransaction_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_CheckTserverTabletHealth_count":              0,
				"handler_latency_yb_tserver_TabletServerService_CheckTserverTabletHealth_sum":                0,
				"handler_latency_yb_tserver_TabletServerService_Checksum_count":                              0,
				"handler_latency_yb_tserver_TabletServerService_Checksum_sum":                                0,
				"handler_latency_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer_count":            0,
				"handler_latency_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer_sum":              0,
				"handler_latency_yb_tserver_TabletServerService_ClearMetacache_count":                        0,
				"handler_latency_yb_tserver_TabletServerService_ClearMetacache_sum":                          0,
				"handler_latency_yb_tserver_TabletServerService_ClearUniverseUuid_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_ClearUniverseUuid_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_GetCompatibleSchemaVersion_count":            0,
				"handler_latency_yb_tserver_TabletServerService_GetCompatibleSchemaVersion_sum":              0,
				"handler_latency_yb_tserver_TabletServerService_GetLockStatus_count":                         0,
				"handler_latency_yb_tserver_TabletServerService_GetLockStatus_sum":                           0,
				"handler_latency_yb_tserver_TabletServerService_GetLogLocation_count":                        0,
				"handler_latency_yb_tserver_TabletServerService_GetLogLocation_sum":                          0,
				"handler_latency_yb_tserver_TabletServerService_GetMasterAddresses_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_GetMasterAddresses_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_GetOldSingleShardWaiters_count":              0,
				"handler_latency_yb_tserver_TabletServerService_GetOldSingleShardWaiters_sum":                0,
				"handler_latency_yb_tserver_TabletServerService_GetOldTransactions_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_GetOldTransactions_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_GetSharedData_count":                         0,
				"handler_latency_yb_tserver_TabletServerService_GetSharedData_sum":                           0,
				"handler_latency_yb_tserver_TabletServerService_GetSplitKey_count":                           0,
				"handler_latency_yb_tserver_TabletServerService_GetSplitKey_sum":                             0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletKeyRanges_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletKeyRanges_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletStatus_count":                       0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletStatus_sum":                         0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant_count":     0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant_sum":       0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatus_count":                  0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatus_sum":                    0,
				"handler_latency_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo_count":          0,
				"handler_latency_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo_sum":            0,
				"handler_latency_yb_tserver_TabletServerService_ImportData_count":                            0,
				"handler_latency_yb_tserver_TabletServerService_ImportData_sum":                              0,
				"handler_latency_yb_tserver_TabletServerService_IsTabletServerReady_count":                   0,
				"handler_latency_yb_tserver_TabletServerService_IsTabletServerReady_sum":                     0,
				"handler_latency_yb_tserver_TabletServerService_ListMasterServers_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_ListMasterServers_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_ListTabletsForTabletServer_count":            0,
				"handler_latency_yb_tserver_TabletServerService_ListTabletsForTabletServer_sum":              0,
				"handler_latency_yb_tserver_TabletServerService_ListTablets_count":                           0,
				"handler_latency_yb_tserver_TabletServerService_ListTablets_sum":                             0,
				"handler_latency_yb_tserver_TabletServerService_NoOp_count":                                  0,
				"handler_latency_yb_tserver_TabletServerService_NoOp_sum":                                    0,
				"handler_latency_yb_tserver_TabletServerService_ProbeTransactionDeadlock_count":              0,
				"handler_latency_yb_tserver_TabletServerService_ProbeTransactionDeadlock_sum":                0,
				"handler_latency_yb_tserver_TabletServerService_Publish_count":                               0,
				"handler_latency_yb_tserver_TabletServerService_Publish_sum":                                 0,
				"handler_latency_yb_tserver_TabletServerService_Read_count":                                  10,
				"handler_latency_yb_tserver_TabletServerService_Read_sum":                                    36076,
				"handler_latency_yb_tserver_TabletServerService_ReleaseObjectLocks_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_ReleaseObjectLocks_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer_count":           0,
				"handler_latency_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer_sum":             0,
				"handler_latency_yb_tserver_TabletServerService_Truncate_count":                              0,
				"handler_latency_yb_tserver_TabletServerService_Truncate_sum":                                0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionStatusLocation_count":       0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionStatusLocation_sum":         0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus_count":     0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus_sum":       0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransaction_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransaction_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_VerifyTableRowRange_count":                   0,
				"handler_latency_yb_tserver_TabletServerService_VerifyTableRowRange_sum":                     0,
				"handler_latency_yb_tserver_TabletServerService_Write_count":                                 0,
				"handler_latency_yb_tserver_TabletServerService_Write_sum":                                   0,
				"hybrid_clock_skew": 0,
				"service_request_bytes_yb_consensus_ConsensusService_ChangeConfig":                            0,
				"service_request_bytes_yb_consensus_ConsensusService_GetConsensusState":                       0,
				"service_request_bytes_yb_consensus_ConsensusService_GetLastOpId":                             0,
				"service_request_bytes_yb_consensus_ConsensusService_GetNodeInstance":                         57,
				"service_request_bytes_yb_consensus_ConsensusService_LeaderElectionLost":                      0,
				"service_request_bytes_yb_consensus_ConsensusService_LeaderStepDown":                          0,
				"service_request_bytes_yb_consensus_ConsensusService_MultiRaftUpdateConsensus":                0,
				"service_request_bytes_yb_consensus_ConsensusService_RequestConsensusVote":                    0,
				"service_request_bytes_yb_consensus_ConsensusService_RunLeaderElection":                       0,
				"service_request_bytes_yb_consensus_ConsensusService_StartRemoteBootstrap":                    0,
				"service_request_bytes_yb_consensus_ConsensusService_UnsafeChangeConfig":                      0,
				"service_request_bytes_yb_consensus_ConsensusService_UpdateConsensus":                         0,
				"service_request_bytes_yb_tserver_PgClientService_ActiveSessionHistory":                       0,
				"service_request_bytes_yb_tserver_PgClientService_AlterDatabase":                              0,
				"service_request_bytes_yb_tserver_PgClientService_AlterTable":                                 0,
				"service_request_bytes_yb_tserver_PgClientService_BackfillIndex":                              0,
				"service_request_bytes_yb_tserver_PgClientService_CancelTransaction":                          0,
				"service_request_bytes_yb_tserver_PgClientService_CheckIfPitrActive":                          0,
				"service_request_bytes_yb_tserver_PgClientService_CreateDatabase":                             0,
				"service_request_bytes_yb_tserver_PgClientService_CreateReplicationSlot":                      0,
				"service_request_bytes_yb_tserver_PgClientService_CreateSequencesDataTable":                   0,
				"service_request_bytes_yb_tserver_PgClientService_CreateTable":                                0,
				"service_request_bytes_yb_tserver_PgClientService_CreateTablegroup":                           0,
				"service_request_bytes_yb_tserver_PgClientService_DeleteDBSequences":                          0,
				"service_request_bytes_yb_tserver_PgClientService_DeleteSequenceTuple":                        0,
				"service_request_bytes_yb_tserver_PgClientService_DropDatabase":                               0,
				"service_request_bytes_yb_tserver_PgClientService_DropReplicationSlot":                        0,
				"service_request_bytes_yb_tserver_PgClientService_DropTable":                                  0,
				"service_request_bytes_yb_tserver_PgClientService_DropTablegroup":                             0,
				"service_request_bytes_yb_tserver_PgClientService_FetchData":                                  0,
				"service_request_bytes_yb_tserver_PgClientService_FetchSequenceTuple":                         0,
				"service_request_bytes_yb_tserver_PgClientService_FinishTransaction":                          0,
				"service_request_bytes_yb_tserver_PgClientService_GetActiveTransactionList":                   0,
				"service_request_bytes_yb_tserver_PgClientService_GetCatalogMasterVersion":                    0,
				"service_request_bytes_yb_tserver_PgClientService_GetDatabaseInfo":                            0,
				"service_request_bytes_yb_tserver_PgClientService_GetIndexBackfillProgress":                   0,
				"service_request_bytes_yb_tserver_PgClientService_GetLockStatus":                              0,
				"service_request_bytes_yb_tserver_PgClientService_GetNewObjectId":                             0,
				"service_request_bytes_yb_tserver_PgClientService_GetReplicationSlot":                         0,
				"service_request_bytes_yb_tserver_PgClientService_GetReplicationSlotStatus":                   0,
				"service_request_bytes_yb_tserver_PgClientService_GetTableDiskSize":                           0,
				"service_request_bytes_yb_tserver_PgClientService_GetTableKeyRanges":                          0,
				"service_request_bytes_yb_tserver_PgClientService_GetTablePartitionList":                      0,
				"service_request_bytes_yb_tserver_PgClientService_GetTserverCatalogVersionInfo":               0,
				"service_request_bytes_yb_tserver_PgClientService_Heartbeat":                                  0,
				"service_request_bytes_yb_tserver_PgClientService_InsertSequenceTuple":                        0,
				"service_request_bytes_yb_tserver_PgClientService_IsInitDbDone":                               0,
				"service_request_bytes_yb_tserver_PgClientService_IsObjectPartOfXRepl":                        0,
				"service_request_bytes_yb_tserver_PgClientService_ListLiveTabletServers":                      0,
				"service_request_bytes_yb_tserver_PgClientService_ListReplicationSlots":                       0,
				"service_request_bytes_yb_tserver_PgClientService_OpenTable":                                  0,
				"service_request_bytes_yb_tserver_PgClientService_Perform":                                    0,
				"service_request_bytes_yb_tserver_PgClientService_ReadSequenceTuple":                          0,
				"service_request_bytes_yb_tserver_PgClientService_ReserveOids":                                0,
				"service_request_bytes_yb_tserver_PgClientService_RollbackToSubTransaction":                   0,
				"service_request_bytes_yb_tserver_PgClientService_SetActiveSubTransaction":                    0,
				"service_request_bytes_yb_tserver_PgClientService_TabletServerCount":                          0,
				"service_request_bytes_yb_tserver_PgClientService_TabletsMetadata":                            0,
				"service_request_bytes_yb_tserver_PgClientService_TruncateTable":                              0,
				"service_request_bytes_yb_tserver_PgClientService_UpdateSequenceTuple":                        0,
				"service_request_bytes_yb_tserver_PgClientService_ValidatePlacement":                          0,
				"service_request_bytes_yb_tserver_PgClientService_WaitForBackendsCatalogVersion":              0,
				"service_request_bytes_yb_tserver_PgClientService_YCQLStatementStats":                         0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession":         0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession":  0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_ChangePeerRole":                      0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive":   0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession":           0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_FetchData":                           0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive":                  0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_RegisterLogAnchor":                   0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession":        0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor":                 0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_UpdateLogAnchor":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_AbortTransaction":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_AcquireObjectLocks":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_AdminExecutePgsql":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_CancelTransaction":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_CheckTserverTabletHealth":               0,
				"service_request_bytes_yb_tserver_TabletServerService_Checksum":                               0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer":             0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearMetacache":                         0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearUniverseUuid":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_GetCompatibleSchemaVersion":             0,
				"service_request_bytes_yb_tserver_TabletServerService_GetLockStatus":                          0,
				"service_request_bytes_yb_tserver_TabletServerService_GetLogLocation":                         0,
				"service_request_bytes_yb_tserver_TabletServerService_GetMasterAddresses":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_GetOldSingleShardWaiters":               0,
				"service_request_bytes_yb_tserver_TabletServerService_GetOldTransactions":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_GetSharedData":                          0,
				"service_request_bytes_yb_tserver_TabletServerService_GetSplitKey":                            0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTabletKeyRanges":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTabletStatus":                        0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTransactionStatus":                   0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant":      0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo":           0,
				"service_request_bytes_yb_tserver_TabletServerService_ImportData":                             0,
				"service_request_bytes_yb_tserver_TabletServerService_IsTabletServerReady":                    0,
				"service_request_bytes_yb_tserver_TabletServerService_ListMasterServers":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_ListTablets":                            0,
				"service_request_bytes_yb_tserver_TabletServerService_ListTabletsForTabletServer":             0,
				"service_request_bytes_yb_tserver_TabletServerService_NoOp":                                   0,
				"service_request_bytes_yb_tserver_TabletServerService_ProbeTransactionDeadlock":               0,
				"service_request_bytes_yb_tserver_TabletServerService_Publish":                                0,
				"service_request_bytes_yb_tserver_TabletServerService_Read":                                   8118,
				"service_request_bytes_yb_tserver_TabletServerService_ReleaseObjectLocks":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer":            0,
				"service_request_bytes_yb_tserver_TabletServerService_Truncate":                               0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransaction":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransactionStatusLocation":        0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus":      0,
				"service_request_bytes_yb_tserver_TabletServerService_VerifyTableRowRange":                    0,
				"service_request_bytes_yb_tserver_TabletServerService_Write":                                  0,
				"service_response_bytes_yb_consensus_ConsensusService_ChangeConfig":                           0,
				"service_response_bytes_yb_consensus_ConsensusService_GetConsensusState":                      0,
				"service_response_bytes_yb_consensus_ConsensusService_GetLastOpId":                            0,
				"service_response_bytes_yb_consensus_ConsensusService_GetNodeInstance":                        130,
				"service_response_bytes_yb_consensus_ConsensusService_LeaderElectionLost":                     0,
				"service_response_bytes_yb_consensus_ConsensusService_LeaderStepDown":                         0,
				"service_response_bytes_yb_consensus_ConsensusService_MultiRaftUpdateConsensus":               0,
				"service_response_bytes_yb_consensus_ConsensusService_RequestConsensusVote":                   0,
				"service_response_bytes_yb_consensus_ConsensusService_RunLeaderElection":                      0,
				"service_response_bytes_yb_consensus_ConsensusService_StartRemoteBootstrap":                   0,
				"service_response_bytes_yb_consensus_ConsensusService_UnsafeChangeConfig":                     0,
				"service_response_bytes_yb_consensus_ConsensusService_UpdateConsensus":                        0,
				"service_response_bytes_yb_tserver_PgClientService_ActiveSessionHistory":                      0,
				"service_response_bytes_yb_tserver_PgClientService_AlterDatabase":                             0,
				"service_response_bytes_yb_tserver_PgClientService_AlterTable":                                0,
				"service_response_bytes_yb_tserver_PgClientService_BackfillIndex":                             0,
				"service_response_bytes_yb_tserver_PgClientService_CancelTransaction":                         0,
				"service_response_bytes_yb_tserver_PgClientService_CheckIfPitrActive":                         0,
				"service_response_bytes_yb_tserver_PgClientService_CreateDatabase":                            0,
				"service_response_bytes_yb_tserver_PgClientService_CreateReplicationSlot":                     0,
				"service_response_bytes_yb_tserver_PgClientService_CreateSequencesDataTable":                  0,
				"service_response_bytes_yb_tserver_PgClientService_CreateTable":                               0,
				"service_response_bytes_yb_tserver_PgClientService_CreateTablegroup":                          0,
				"service_response_bytes_yb_tserver_PgClientService_DeleteDBSequences":                         0,
				"service_response_bytes_yb_tserver_PgClientService_DeleteSequenceTuple":                       0,
				"service_response_bytes_yb_tserver_PgClientService_DropDatabase":                              0,
				"service_response_bytes_yb_tserver_PgClientService_DropReplicationSlot":                       0,
				"service_response_bytes_yb_tserver_PgClientService_DropTable":                                 0,
				"service_response_bytes_yb_tserver_PgClientService_DropTablegroup":                            0,
				"service_response_bytes_yb_tserver_PgClientService_FetchData":                                 0,
				"service_response_bytes_yb_tserver_PgClientService_FetchSequenceTuple":                        0,
				"service_response_bytes_yb_tserver_PgClientService_FinishTransaction":                         0,
				"service_response_bytes_yb_tserver_PgClientService_GetActiveTransactionList":                  0,
				"service_response_bytes_yb_tserver_PgClientService_GetCatalogMasterVersion":                   0,
				"service_response_bytes_yb_tserver_PgClientService_GetDatabaseInfo":                           0,
				"service_response_bytes_yb_tserver_PgClientService_GetIndexBackfillProgress":                  0,
				"service_response_bytes_yb_tserver_PgClientService_GetLockStatus":                             0,
				"service_response_bytes_yb_tserver_PgClientService_GetNewObjectId":                            0,
				"service_response_bytes_yb_tserver_PgClientService_GetReplicationSlot":                        0,
				"service_response_bytes_yb_tserver_PgClientService_GetReplicationSlotStatus":                  0,
				"service_response_bytes_yb_tserver_PgClientService_GetTableDiskSize":                          0,
				"service_response_bytes_yb_tserver_PgClientService_GetTableKeyRanges":                         0,
				"service_response_bytes_yb_tserver_PgClientService_GetTablePartitionList":                     0,
				"service_response_bytes_yb_tserver_PgClientService_GetTserverCatalogVersionInfo":              0,
				"service_response_bytes_yb_tserver_PgClientService_Heartbeat":                                 0,
				"service_response_bytes_yb_tserver_PgClientService_InsertSequenceTuple":                       0,
				"service_response_bytes_yb_tserver_PgClientService_IsInitDbDone":                              0,
				"service_response_bytes_yb_tserver_PgClientService_IsObjectPartOfXRepl":                       0,
				"service_response_bytes_yb_tserver_PgClientService_ListLiveTabletServers":                     0,
				"service_response_bytes_yb_tserver_PgClientService_ListReplicationSlots":                      0,
				"service_response_bytes_yb_tserver_PgClientService_OpenTable":                                 0,
				"service_response_bytes_yb_tserver_PgClientService_Perform":                                   0,
				"service_response_bytes_yb_tserver_PgClientService_ReadSequenceTuple":                         0,
				"service_response_bytes_yb_tserver_PgClientService_ReserveOids":                               0,
				"service_response_bytes_yb_tserver_PgClientService_RollbackToSubTransaction":                  0,
				"service_response_bytes_yb_tserver_PgClientService_SetActiveSubTransaction":                   0,
				"service_response_bytes_yb_tserver_PgClientService_TabletServerCount":                         0,
				"service_response_bytes_yb_tserver_PgClientService_TabletsMetadata":                           0,
				"service_response_bytes_yb_tserver_PgClientService_TruncateTable":                             0,
				"service_response_bytes_yb_tserver_PgClientService_UpdateSequenceTuple":                       0,
				"service_response_bytes_yb_tserver_PgClientService_ValidatePlacement":                         0,
				"service_response_bytes_yb_tserver_PgClientService_WaitForBackendsCatalogVersion":             0,
				"service_response_bytes_yb_tserver_PgClientService_YCQLStatementStats":                        0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession":        0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession": 0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_ChangePeerRole":                     0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive":  0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession":          0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_FetchData":                          0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive":                 0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_RegisterLogAnchor":                  0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession":       0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor":                0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_UpdateLogAnchor":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_AbortTransaction":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_AcquireObjectLocks":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_AdminExecutePgsql":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_CancelTransaction":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_CheckTserverTabletHealth":              0,
				"service_response_bytes_yb_tserver_TabletServerService_Checksum":                              0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer":            0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearMetacache":                        0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearUniverseUuid":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_GetCompatibleSchemaVersion":            0,
				"service_response_bytes_yb_tserver_TabletServerService_GetLockStatus":                         0,
				"service_response_bytes_yb_tserver_TabletServerService_GetLogLocation":                        0,
				"service_response_bytes_yb_tserver_TabletServerService_GetMasterAddresses":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_GetOldSingleShardWaiters":              0,
				"service_response_bytes_yb_tserver_TabletServerService_GetOldTransactions":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_GetSharedData":                         0,
				"service_response_bytes_yb_tserver_TabletServerService_GetSplitKey":                           0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTabletKeyRanges":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTabletStatus":                       0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTransactionStatus":                  0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant":     0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo":          0,
				"service_response_bytes_yb_tserver_TabletServerService_ImportData":                            0,
				"service_response_bytes_yb_tserver_TabletServerService_IsTabletServerReady":                   0,
				"service_response_bytes_yb_tserver_TabletServerService_ListMasterServers":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_ListTablets":                           0,
				"service_response_bytes_yb_tserver_TabletServerService_ListTabletsForTabletServer":            0,
				"service_response_bytes_yb_tserver_TabletServerService_NoOp":                                  0,
				"service_response_bytes_yb_tserver_TabletServerService_ProbeTransactionDeadlock":              0,
				"service_response_bytes_yb_tserver_TabletServerService_Publish":                               0,
				"service_response_bytes_yb_tserver_TabletServerService_Read":                                  3010360,
				"service_response_bytes_yb_tserver_TabletServerService_ReleaseObjectLocks":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer":           0,
				"service_response_bytes_yb_tserver_TabletServerService_Truncate":                              0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransaction":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransactionStatusLocation":       0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus":     0,
				"service_response_bytes_yb_tserver_TabletServerService_VerifyTableRowRange":                   0,
				"service_response_bytes_yb_tserver_TabletServerService_Write":                                 0,
			},
		},
		"success on tserver response": {
			prepare:         caseTserverOk,
			wantNumOfCharts: 399,
			wantMetrics: map[string]int64{
				"handler_latency_yb_consensus_ConsensusService_ChangeConfig_count":                            0,
				"handler_latency_yb_consensus_ConsensusService_ChangeConfig_sum":                              0,
				"handler_latency_yb_consensus_ConsensusService_GetConsensusState_count":                       0,
				"handler_latency_yb_consensus_ConsensusService_GetConsensusState_sum":                         0,
				"handler_latency_yb_consensus_ConsensusService_GetLastOpId_count":                             0,
				"handler_latency_yb_consensus_ConsensusService_GetLastOpId_sum":                               0,
				"handler_latency_yb_consensus_ConsensusService_GetNodeInstance_count":                         0,
				"handler_latency_yb_consensus_ConsensusService_GetNodeInstance_sum":                           0,
				"handler_latency_yb_consensus_ConsensusService_LeaderElectionLost_count":                      0,
				"handler_latency_yb_consensus_ConsensusService_LeaderElectionLost_sum":                        0,
				"handler_latency_yb_consensus_ConsensusService_LeaderStepDown_count":                          0,
				"handler_latency_yb_consensus_ConsensusService_LeaderStepDown_sum":                            0,
				"handler_latency_yb_consensus_ConsensusService_MultiRaftUpdateConsensus_count":                0,
				"handler_latency_yb_consensus_ConsensusService_MultiRaftUpdateConsensus_sum":                  0,
				"handler_latency_yb_consensus_ConsensusService_RequestConsensusVote_count":                    0,
				"handler_latency_yb_consensus_ConsensusService_RequestConsensusVote_sum":                      0,
				"handler_latency_yb_consensus_ConsensusService_RunLeaderElection_count":                       9,
				"handler_latency_yb_consensus_ConsensusService_RunLeaderElection_sum":                         7219,
				"handler_latency_yb_consensus_ConsensusService_StartRemoteBootstrap_count":                    0,
				"handler_latency_yb_consensus_ConsensusService_StartRemoteBootstrap_sum":                      0,
				"handler_latency_yb_consensus_ConsensusService_UnsafeChangeConfig_count":                      0,
				"handler_latency_yb_consensus_ConsensusService_UnsafeChangeConfig_sum":                        0,
				"handler_latency_yb_consensus_ConsensusService_UpdateConsensus_count":                         0,
				"handler_latency_yb_consensus_ConsensusService_UpdateConsensus_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_ActiveSessionHistory_count":                       0,
				"handler_latency_yb_tserver_PgClientService_ActiveSessionHistory_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_AlterDatabase_count":                              0,
				"handler_latency_yb_tserver_PgClientService_AlterDatabase_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_AlterTable_count":                                 0,
				"handler_latency_yb_tserver_PgClientService_AlterTable_sum":                                   0,
				"handler_latency_yb_tserver_PgClientService_BackfillIndex_count":                              0,
				"handler_latency_yb_tserver_PgClientService_BackfillIndex_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_CancelTransaction_count":                          0,
				"handler_latency_yb_tserver_PgClientService_CancelTransaction_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_CheckIfPitrActive_count":                          0,
				"handler_latency_yb_tserver_PgClientService_CheckIfPitrActive_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_CreateDatabase_count":                             0,
				"handler_latency_yb_tserver_PgClientService_CreateDatabase_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_CreateReplicationSlot_count":                      0,
				"handler_latency_yb_tserver_PgClientService_CreateReplicationSlot_sum":                        0,
				"handler_latency_yb_tserver_PgClientService_CreateSequencesDataTable_count":                   0,
				"handler_latency_yb_tserver_PgClientService_CreateSequencesDataTable_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_CreateTable_count":                                0,
				"handler_latency_yb_tserver_PgClientService_CreateTable_sum":                                  0,
				"handler_latency_yb_tserver_PgClientService_CreateTablegroup_count":                           0,
				"handler_latency_yb_tserver_PgClientService_CreateTablegroup_sum":                             0,
				"handler_latency_yb_tserver_PgClientService_DeleteDBSequences_count":                          0,
				"handler_latency_yb_tserver_PgClientService_DeleteDBSequences_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_DeleteSequenceTuple_count":                        0,
				"handler_latency_yb_tserver_PgClientService_DeleteSequenceTuple_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_DropDatabase_count":                               0,
				"handler_latency_yb_tserver_PgClientService_DropDatabase_sum":                                 0,
				"handler_latency_yb_tserver_PgClientService_DropReplicationSlot_count":                        0,
				"handler_latency_yb_tserver_PgClientService_DropReplicationSlot_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_DropTable_count":                                  0,
				"handler_latency_yb_tserver_PgClientService_DropTable_sum":                                    0,
				"handler_latency_yb_tserver_PgClientService_DropTablegroup_count":                             0,
				"handler_latency_yb_tserver_PgClientService_DropTablegroup_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_FetchData_count":                                  0,
				"handler_latency_yb_tserver_PgClientService_FetchData_sum":                                    0,
				"handler_latency_yb_tserver_PgClientService_FetchSequenceTuple_count":                         0,
				"handler_latency_yb_tserver_PgClientService_FetchSequenceTuple_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_FinishTransaction_count":                          0,
				"handler_latency_yb_tserver_PgClientService_FinishTransaction_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_GetActiveTransactionList_count":                   0,
				"handler_latency_yb_tserver_PgClientService_GetActiveTransactionList_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_GetCatalogMasterVersion_count":                    0,
				"handler_latency_yb_tserver_PgClientService_GetCatalogMasterVersion_sum":                      0,
				"handler_latency_yb_tserver_PgClientService_GetDatabaseInfo_count":                            1,
				"handler_latency_yb_tserver_PgClientService_GetDatabaseInfo_sum":                              611,
				"handler_latency_yb_tserver_PgClientService_GetIndexBackfillProgress_count":                   0,
				"handler_latency_yb_tserver_PgClientService_GetIndexBackfillProgress_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_GetLockStatus_count":                              0,
				"handler_latency_yb_tserver_PgClientService_GetLockStatus_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_GetNewObjectId_count":                             0,
				"handler_latency_yb_tserver_PgClientService_GetNewObjectId_sum":                               0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlotStatus_count":                   0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlotStatus_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlot_count":                         0,
				"handler_latency_yb_tserver_PgClientService_GetReplicationSlot_sum":                           0,
				"handler_latency_yb_tserver_PgClientService_GetTableDiskSize_count":                           0,
				"handler_latency_yb_tserver_PgClientService_GetTableDiskSize_sum":                             0,
				"handler_latency_yb_tserver_PgClientService_GetTableKeyRanges_count":                          0,
				"handler_latency_yb_tserver_PgClientService_GetTableKeyRanges_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_GetTablePartitionList_count":                      0,
				"handler_latency_yb_tserver_PgClientService_GetTablePartitionList_sum":                        0,
				"handler_latency_yb_tserver_PgClientService_GetTserverCatalogVersionInfo_count":               1,
				"handler_latency_yb_tserver_PgClientService_GetTserverCatalogVersionInfo_sum":                 105,
				"handler_latency_yb_tserver_PgClientService_Heartbeat_count":                                  186,
				"handler_latency_yb_tserver_PgClientService_Heartbeat_sum":                                    6621,
				"handler_latency_yb_tserver_PgClientService_InsertSequenceTuple_count":                        0,
				"handler_latency_yb_tserver_PgClientService_InsertSequenceTuple_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_IsInitDbDone_count":                               0,
				"handler_latency_yb_tserver_PgClientService_IsInitDbDone_sum":                                 0,
				"handler_latency_yb_tserver_PgClientService_IsObjectPartOfXRepl_count":                        0,
				"handler_latency_yb_tserver_PgClientService_IsObjectPartOfXRepl_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_ListLiveTabletServers_count":                      0,
				"handler_latency_yb_tserver_PgClientService_ListLiveTabletServers_sum":                        0,
				"handler_latency_yb_tserver_PgClientService_ListReplicationSlots_count":                       0,
				"handler_latency_yb_tserver_PgClientService_ListReplicationSlots_sum":                         0,
				"handler_latency_yb_tserver_PgClientService_OpenTable_count":                                  37,
				"handler_latency_yb_tserver_PgClientService_OpenTable_sum":                                    30320,
				"handler_latency_yb_tserver_PgClientService_Perform_count":                                    1,
				"handler_latency_yb_tserver_PgClientService_Perform_sum":                                      32553,
				"handler_latency_yb_tserver_PgClientService_ReadSequenceTuple_count":                          0,
				"handler_latency_yb_tserver_PgClientService_ReadSequenceTuple_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_ReserveOids_count":                                0,
				"handler_latency_yb_tserver_PgClientService_ReserveOids_sum":                                  0,
				"handler_latency_yb_tserver_PgClientService_RollbackToSubTransaction_count":                   0,
				"handler_latency_yb_tserver_PgClientService_RollbackToSubTransaction_sum":                     0,
				"handler_latency_yb_tserver_PgClientService_SetActiveSubTransaction_count":                    0,
				"handler_latency_yb_tserver_PgClientService_SetActiveSubTransaction_sum":                      0,
				"handler_latency_yb_tserver_PgClientService_TabletServerCount_count":                          0,
				"handler_latency_yb_tserver_PgClientService_TabletServerCount_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_TabletsMetadata_count":                            0,
				"handler_latency_yb_tserver_PgClientService_TabletsMetadata_sum":                              0,
				"handler_latency_yb_tserver_PgClientService_TruncateTable_count":                              0,
				"handler_latency_yb_tserver_PgClientService_TruncateTable_sum":                                0,
				"handler_latency_yb_tserver_PgClientService_UpdateSequenceTuple_count":                        0,
				"handler_latency_yb_tserver_PgClientService_UpdateSequenceTuple_sum":                          0,
				"handler_latency_yb_tserver_PgClientService_ValidatePlacement_count":                          0,
				"handler_latency_yb_tserver_PgClientService_ValidatePlacement_sum":                            0,
				"handler_latency_yb_tserver_PgClientService_WaitForBackendsCatalogVersion_count":              0,
				"handler_latency_yb_tserver_PgClientService_WaitForBackendsCatalogVersion_sum":                0,
				"handler_latency_yb_tserver_PgClientService_YCQLStatementStats_count":                         0,
				"handler_latency_yb_tserver_PgClientService_YCQLStatementStats_sum":                           0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession_count":         0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession_sum":           0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession_count":  0,
				"handler_latency_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession_sum":    0,
				"handler_latency_yb_tserver_RemoteBootstrapService_ChangePeerRole_count":                      0,
				"handler_latency_yb_tserver_RemoteBootstrapService_ChangePeerRole_sum":                        0,
				"handler_latency_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive_count":   0,
				"handler_latency_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive_sum":     0,
				"handler_latency_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession_count":           0,
				"handler_latency_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession_sum":             0,
				"handler_latency_yb_tserver_RemoteBootstrapService_FetchData_count":                           0,
				"handler_latency_yb_tserver_RemoteBootstrapService_FetchData_sum":                             0,
				"handler_latency_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive_count":                  0,
				"handler_latency_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive_sum":                    0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RegisterLogAnchor_count":                   0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RegisterLogAnchor_sum":                     0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession_count":        0,
				"handler_latency_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession_sum":          0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor_count":                 0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor_sum":                   0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UpdateLogAnchor_count":                     0,
				"handler_latency_yb_tserver_RemoteBootstrapService_UpdateLogAnchor_sum":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_AddTableToTablet_count":                  0,
				"handler_latency_yb_tserver_TabletServerAdminService_AddTableToTablet_sum":                    0,
				"handler_latency_yb_tserver_TabletServerAdminService_AlterSchema_count":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_AlterSchema_sum":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_BackfillDone_count":                      0,
				"handler_latency_yb_tserver_TabletServerAdminService_BackfillDone_sum":                        0,
				"handler_latency_yb_tserver_TabletServerAdminService_BackfillIndex_count":                     0,
				"handler_latency_yb_tserver_TabletServerAdminService_BackfillIndex_sum":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_ClonePgSchema_count":                     0,
				"handler_latency_yb_tserver_TabletServerAdminService_ClonePgSchema_sum":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_CloneTablet_count":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_CloneTablet_sum":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_CountIntents_count":                      0,
				"handler_latency_yb_tserver_TabletServerAdminService_CountIntents_sum":                        0,
				"handler_latency_yb_tserver_TabletServerAdminService_CreateTablet_count":                      9,
				"handler_latency_yb_tserver_TabletServerAdminService_CreateTablet_sum":                        606318,
				"handler_latency_yb_tserver_TabletServerAdminService_DeleteTablet_count":                      0,
				"handler_latency_yb_tserver_TabletServerAdminService_DeleteTablet_sum":                        0,
				"handler_latency_yb_tserver_TabletServerAdminService_EnableDbConns_count":                     0,
				"handler_latency_yb_tserver_TabletServerAdminService_EnableDbConns_sum":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_FlushTablets_count":                      0,
				"handler_latency_yb_tserver_TabletServerAdminService_FlushTablets_sum":                        0,
				"handler_latency_yb_tserver_TabletServerAdminService_GetSafeTime_count":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_GetSafeTime_sum":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_PrepareDeleteTransactionTablet_count":    0,
				"handler_latency_yb_tserver_TabletServerAdminService_PrepareDeleteTransactionTablet_sum":      0,
				"handler_latency_yb_tserver_TabletServerAdminService_RemoveTableFromTablet_count":             0,
				"handler_latency_yb_tserver_TabletServerAdminService_RemoveTableFromTablet_sum":               0,
				"handler_latency_yb_tserver_TabletServerAdminService_SplitTablet_count":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_SplitTablet_sum":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_TestRetry_count":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_TestRetry_sum":                           0,
				"handler_latency_yb_tserver_TabletServerAdminService_UpdateTransactionTablesVersion_count":    0,
				"handler_latency_yb_tserver_TabletServerAdminService_UpdateTransactionTablesVersion_sum":      0,
				"handler_latency_yb_tserver_TabletServerAdminService_UpgradeYsql_count":                       0,
				"handler_latency_yb_tserver_TabletServerAdminService_UpgradeYsql_sum":                         0,
				"handler_latency_yb_tserver_TabletServerAdminService_WaitForYsqlBackendsCatalogVersion_count": 0,
				"handler_latency_yb_tserver_TabletServerAdminService_WaitForYsqlBackendsCatalogVersion_sum":   0,
				"handler_latency_yb_tserver_TabletServerBackupService_TabletSnapshotOp_count":                 0,
				"handler_latency_yb_tserver_TabletServerBackupService_TabletSnapshotOp_sum":                   0,
				"handler_latency_yb_tserver_TabletServerService_AbortTransaction_count":                       0,
				"handler_latency_yb_tserver_TabletServerService_AbortTransaction_sum":                         0,
				"handler_latency_yb_tserver_TabletServerService_AcquireObjectLocks_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_AcquireObjectLocks_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_AdminExecutePgsql_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_AdminExecutePgsql_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_CancelTransaction_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_CancelTransaction_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_CheckTserverTabletHealth_count":               0,
				"handler_latency_yb_tserver_TabletServerService_CheckTserverTabletHealth_sum":                 0,
				"handler_latency_yb_tserver_TabletServerService_Checksum_count":                               0,
				"handler_latency_yb_tserver_TabletServerService_Checksum_sum":                                 0,
				"handler_latency_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer_count":             0,
				"handler_latency_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer_sum":               0,
				"handler_latency_yb_tserver_TabletServerService_ClearMetacache_count":                         0,
				"handler_latency_yb_tserver_TabletServerService_ClearMetacache_sum":                           0,
				"handler_latency_yb_tserver_TabletServerService_ClearUniverseUuid_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_ClearUniverseUuid_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_GetCompatibleSchemaVersion_count":             0,
				"handler_latency_yb_tserver_TabletServerService_GetCompatibleSchemaVersion_sum":               0,
				"handler_latency_yb_tserver_TabletServerService_GetLockStatus_count":                          0,
				"handler_latency_yb_tserver_TabletServerService_GetLockStatus_sum":                            0,
				"handler_latency_yb_tserver_TabletServerService_GetLogLocation_count":                         0,
				"handler_latency_yb_tserver_TabletServerService_GetLogLocation_sum":                           0,
				"handler_latency_yb_tserver_TabletServerService_GetMasterAddresses_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_GetMasterAddresses_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_GetOldSingleShardWaiters_count":               0,
				"handler_latency_yb_tserver_TabletServerService_GetOldSingleShardWaiters_sum":                 0,
				"handler_latency_yb_tserver_TabletServerService_GetOldTransactions_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_GetOldTransactions_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_GetSharedData_count":                          0,
				"handler_latency_yb_tserver_TabletServerService_GetSharedData_sum":                            0,
				"handler_latency_yb_tserver_TabletServerService_GetSplitKey_count":                            0,
				"handler_latency_yb_tserver_TabletServerService_GetSplitKey_sum":                              0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletKeyRanges_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletKeyRanges_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletStatus_count":                        0,
				"handler_latency_yb_tserver_TabletServerService_GetTabletStatus_sum":                          0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant_count":      0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant_sum":        0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatus_count":                   0,
				"handler_latency_yb_tserver_TabletServerService_GetTransactionStatus_sum":                     0,
				"handler_latency_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo_count":           0,
				"handler_latency_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo_sum":             0,
				"handler_latency_yb_tserver_TabletServerService_ImportData_count":                             0,
				"handler_latency_yb_tserver_TabletServerService_ImportData_sum":                               0,
				"handler_latency_yb_tserver_TabletServerService_IsTabletServerReady_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_IsTabletServerReady_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_ListMasterServers_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_ListMasterServers_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_ListTabletsForTabletServer_count":             0,
				"handler_latency_yb_tserver_TabletServerService_ListTabletsForTabletServer_sum":               0,
				"handler_latency_yb_tserver_TabletServerService_ListTablets_count":                            0,
				"handler_latency_yb_tserver_TabletServerService_ListTablets_sum":                              0,
				"handler_latency_yb_tserver_TabletServerService_NoOp_count":                                   0,
				"handler_latency_yb_tserver_TabletServerService_NoOp_sum":                                     0,
				"handler_latency_yb_tserver_TabletServerService_ProbeTransactionDeadlock_count":               0,
				"handler_latency_yb_tserver_TabletServerService_ProbeTransactionDeadlock_sum":                 0,
				"handler_latency_yb_tserver_TabletServerService_Publish_count":                                0,
				"handler_latency_yb_tserver_TabletServerService_Publish_sum":                                  0,
				"handler_latency_yb_tserver_TabletServerService_Read_count":                                   159,
				"handler_latency_yb_tserver_TabletServerService_Read_sum":                                     213379,
				"handler_latency_yb_tserver_TabletServerService_ReleaseObjectLocks_count":                     0,
				"handler_latency_yb_tserver_TabletServerService_ReleaseObjectLocks_sum":                       0,
				"handler_latency_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer_count":            0,
				"handler_latency_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer_sum":              0,
				"handler_latency_yb_tserver_TabletServerService_Truncate_count":                               0,
				"handler_latency_yb_tserver_TabletServerService_Truncate_sum":                                 0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionStatusLocation_count":        0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionStatusLocation_sum":          0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus_count":      0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus_sum":        0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransaction_count":                      0,
				"handler_latency_yb_tserver_TabletServerService_UpdateTransaction_sum":                        0,
				"handler_latency_yb_tserver_TabletServerService_VerifyTableRowRange_count":                    0,
				"handler_latency_yb_tserver_TabletServerService_VerifyTableRowRange_sum":                      0,
				"handler_latency_yb_tserver_TabletServerService_Write_count":                                  30245,
				"handler_latency_yb_tserver_TabletServerService_Write_sum":                                    514906289,
				"hybrid_clock_skew": 0,
				"service_request_bytes_yb_consensus_ConsensusService_ChangeConfig":                             0,
				"service_request_bytes_yb_consensus_ConsensusService_GetConsensusState":                        0,
				"service_request_bytes_yb_consensus_ConsensusService_GetLastOpId":                              0,
				"service_request_bytes_yb_consensus_ConsensusService_GetNodeInstance":                          0,
				"service_request_bytes_yb_consensus_ConsensusService_LeaderElectionLost":                       0,
				"service_request_bytes_yb_consensus_ConsensusService_LeaderStepDown":                           0,
				"service_request_bytes_yb_consensus_ConsensusService_MultiRaftUpdateConsensus":                 0,
				"service_request_bytes_yb_consensus_ConsensusService_RequestConsensusVote":                     0,
				"service_request_bytes_yb_consensus_ConsensusService_RunLeaderElection":                        1170,
				"service_request_bytes_yb_consensus_ConsensusService_StartRemoteBootstrap":                     0,
				"service_request_bytes_yb_consensus_ConsensusService_UnsafeChangeConfig":                       0,
				"service_request_bytes_yb_consensus_ConsensusService_UpdateConsensus":                          0,
				"service_request_bytes_yb_tserver_PgClientService_ActiveSessionHistory":                        0,
				"service_request_bytes_yb_tserver_PgClientService_AlterDatabase":                               0,
				"service_request_bytes_yb_tserver_PgClientService_AlterTable":                                  0,
				"service_request_bytes_yb_tserver_PgClientService_BackfillIndex":                               0,
				"service_request_bytes_yb_tserver_PgClientService_CancelTransaction":                           0,
				"service_request_bytes_yb_tserver_PgClientService_CheckIfPitrActive":                           0,
				"service_request_bytes_yb_tserver_PgClientService_CreateDatabase":                              0,
				"service_request_bytes_yb_tserver_PgClientService_CreateReplicationSlot":                       0,
				"service_request_bytes_yb_tserver_PgClientService_CreateSequencesDataTable":                    0,
				"service_request_bytes_yb_tserver_PgClientService_CreateTable":                                 0,
				"service_request_bytes_yb_tserver_PgClientService_CreateTablegroup":                            0,
				"service_request_bytes_yb_tserver_PgClientService_DeleteDBSequences":                           0,
				"service_request_bytes_yb_tserver_PgClientService_DeleteSequenceTuple":                         0,
				"service_request_bytes_yb_tserver_PgClientService_DropDatabase":                                0,
				"service_request_bytes_yb_tserver_PgClientService_DropReplicationSlot":                         0,
				"service_request_bytes_yb_tserver_PgClientService_DropTable":                                   0,
				"service_request_bytes_yb_tserver_PgClientService_DropTablegroup":                              0,
				"service_request_bytes_yb_tserver_PgClientService_FetchData":                                   0,
				"service_request_bytes_yb_tserver_PgClientService_FetchSequenceTuple":                          0,
				"service_request_bytes_yb_tserver_PgClientService_FinishTransaction":                           0,
				"service_request_bytes_yb_tserver_PgClientService_GetActiveTransactionList":                    0,
				"service_request_bytes_yb_tserver_PgClientService_GetCatalogMasterVersion":                     0,
				"service_request_bytes_yb_tserver_PgClientService_GetDatabaseInfo":                             58,
				"service_request_bytes_yb_tserver_PgClientService_GetIndexBackfillProgress":                    0,
				"service_request_bytes_yb_tserver_PgClientService_GetLockStatus":                               0,
				"service_request_bytes_yb_tserver_PgClientService_GetNewObjectId":                              0,
				"service_request_bytes_yb_tserver_PgClientService_GetReplicationSlot":                          0,
				"service_request_bytes_yb_tserver_PgClientService_GetReplicationSlotStatus":                    0,
				"service_request_bytes_yb_tserver_PgClientService_GetTableDiskSize":                            0,
				"service_request_bytes_yb_tserver_PgClientService_GetTableKeyRanges":                           0,
				"service_request_bytes_yb_tserver_PgClientService_GetTablePartitionList":                       0,
				"service_request_bytes_yb_tserver_PgClientService_GetTserverCatalogVersionInfo":                71,
				"service_request_bytes_yb_tserver_PgClientService_Heartbeat":                                   9397,
				"service_request_bytes_yb_tserver_PgClientService_InsertSequenceTuple":                         0,
				"service_request_bytes_yb_tserver_PgClientService_IsInitDbDone":                                0,
				"service_request_bytes_yb_tserver_PgClientService_IsObjectPartOfXRepl":                         0,
				"service_request_bytes_yb_tserver_PgClientService_ListLiveTabletServers":                       0,
				"service_request_bytes_yb_tserver_PgClientService_ListReplicationSlots":                        0,
				"service_request_bytes_yb_tserver_PgClientService_OpenTable":                                   3071,
				"service_request_bytes_yb_tserver_PgClientService_Perform":                                     6652,
				"service_request_bytes_yb_tserver_PgClientService_ReadSequenceTuple":                           0,
				"service_request_bytes_yb_tserver_PgClientService_ReserveOids":                                 0,
				"service_request_bytes_yb_tserver_PgClientService_RollbackToSubTransaction":                    0,
				"service_request_bytes_yb_tserver_PgClientService_SetActiveSubTransaction":                     0,
				"service_request_bytes_yb_tserver_PgClientService_TabletServerCount":                           0,
				"service_request_bytes_yb_tserver_PgClientService_TabletsMetadata":                             0,
				"service_request_bytes_yb_tserver_PgClientService_TruncateTable":                               0,
				"service_request_bytes_yb_tserver_PgClientService_UpdateSequenceTuple":                         0,
				"service_request_bytes_yb_tserver_PgClientService_ValidatePlacement":                           0,
				"service_request_bytes_yb_tserver_PgClientService_WaitForBackendsCatalogVersion":               0,
				"service_request_bytes_yb_tserver_PgClientService_YCQLStatementStats":                          0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession":          0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession":   0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_ChangePeerRole":                       0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive":    0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession":            0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_FetchData":                            0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive":                   0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_RegisterLogAnchor":                    0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession":         0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor":                  0,
				"service_request_bytes_yb_tserver_RemoteBootstrapService_UpdateLogAnchor":                      0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_AddTableToTablet":                   0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_AlterSchema":                        0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_BackfillDone":                       0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_BackfillIndex":                      0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_ClonePgSchema":                      0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_CloneTablet":                        0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_CountIntents":                       0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_CreateTablet":                       3891,
				"service_request_bytes_yb_tserver_TabletServerAdminService_DeleteTablet":                       0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_EnableDbConns":                      0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_FlushTablets":                       0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_GetSafeTime":                        0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_PrepareDeleteTransactionTablet":     0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_RemoveTableFromTablet":              0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_SplitTablet":                        0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_TestRetry":                          0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_UpdateTransactionTablesVersion":     0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_UpgradeYsql":                        0,
				"service_request_bytes_yb_tserver_TabletServerAdminService_WaitForYsqlBackendsCatalogVersion":  0,
				"service_request_bytes_yb_tserver_TabletServerBackupService_TabletSnapshotOp":                  0,
				"service_request_bytes_yb_tserver_TabletServerService_AbortTransaction":                        0,
				"service_request_bytes_yb_tserver_TabletServerService_AcquireObjectLocks":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_AdminExecutePgsql":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_CancelTransaction":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_CheckTserverTabletHealth":                0,
				"service_request_bytes_yb_tserver_TabletServerService_Checksum":                                0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer":              0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearMetacache":                          0,
				"service_request_bytes_yb_tserver_TabletServerService_ClearUniverseUuid":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_GetCompatibleSchemaVersion":              0,
				"service_request_bytes_yb_tserver_TabletServerService_GetLockStatus":                           0,
				"service_request_bytes_yb_tserver_TabletServerService_GetLogLocation":                          0,
				"service_request_bytes_yb_tserver_TabletServerService_GetMasterAddresses":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_GetOldSingleShardWaiters":                0,
				"service_request_bytes_yb_tserver_TabletServerService_GetOldTransactions":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_GetSharedData":                           0,
				"service_request_bytes_yb_tserver_TabletServerService_GetSplitKey":                             0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTabletKeyRanges":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTabletStatus":                         0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTransactionStatus":                    0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant":       0,
				"service_request_bytes_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo":            0,
				"service_request_bytes_yb_tserver_TabletServerService_ImportData":                              0,
				"service_request_bytes_yb_tserver_TabletServerService_IsTabletServerReady":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_ListMasterServers":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_ListTablets":                             0,
				"service_request_bytes_yb_tserver_TabletServerService_ListTabletsForTabletServer":              0,
				"service_request_bytes_yb_tserver_TabletServerService_NoOp":                                    0,
				"service_request_bytes_yb_tserver_TabletServerService_ProbeTransactionDeadlock":                0,
				"service_request_bytes_yb_tserver_TabletServerService_Publish":                                 0,
				"service_request_bytes_yb_tserver_TabletServerService_Read":                                    0,
				"service_request_bytes_yb_tserver_TabletServerService_ReleaseObjectLocks":                      0,
				"service_request_bytes_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer":             0,
				"service_request_bytes_yb_tserver_TabletServerService_Truncate":                                0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransaction":                       0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransactionStatusLocation":         0,
				"service_request_bytes_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus":       0,
				"service_request_bytes_yb_tserver_TabletServerService_VerifyTableRowRange":                     0,
				"service_request_bytes_yb_tserver_TabletServerService_Write":                                   77026294,
				"service_response_bytes_yb_consensus_ConsensusService_ChangeConfig":                            0,
				"service_response_bytes_yb_consensus_ConsensusService_GetConsensusState":                       0,
				"service_response_bytes_yb_consensus_ConsensusService_GetLastOpId":                             0,
				"service_response_bytes_yb_consensus_ConsensusService_GetNodeInstance":                         0,
				"service_response_bytes_yb_consensus_ConsensusService_LeaderElectionLost":                      0,
				"service_response_bytes_yb_consensus_ConsensusService_LeaderStepDown":                          0,
				"service_response_bytes_yb_consensus_ConsensusService_MultiRaftUpdateConsensus":                0,
				"service_response_bytes_yb_consensus_ConsensusService_RequestConsensusVote":                    0,
				"service_response_bytes_yb_consensus_ConsensusService_RunLeaderElection":                       90,
				"service_response_bytes_yb_consensus_ConsensusService_StartRemoteBootstrap":                    0,
				"service_response_bytes_yb_consensus_ConsensusService_UnsafeChangeConfig":                      0,
				"service_response_bytes_yb_consensus_ConsensusService_UpdateConsensus":                         0,
				"service_response_bytes_yb_tserver_PgClientService_ActiveSessionHistory":                       0,
				"service_response_bytes_yb_tserver_PgClientService_AlterDatabase":                              0,
				"service_response_bytes_yb_tserver_PgClientService_AlterTable":                                 0,
				"service_response_bytes_yb_tserver_PgClientService_BackfillIndex":                              0,
				"service_response_bytes_yb_tserver_PgClientService_CancelTransaction":                          0,
				"service_response_bytes_yb_tserver_PgClientService_CheckIfPitrActive":                          0,
				"service_response_bytes_yb_tserver_PgClientService_CreateDatabase":                             0,
				"service_response_bytes_yb_tserver_PgClientService_CreateReplicationSlot":                      0,
				"service_response_bytes_yb_tserver_PgClientService_CreateSequencesDataTable":                   0,
				"service_response_bytes_yb_tserver_PgClientService_CreateTable":                                0,
				"service_response_bytes_yb_tserver_PgClientService_CreateTablegroup":                           0,
				"service_response_bytes_yb_tserver_PgClientService_DeleteDBSequences":                          0,
				"service_response_bytes_yb_tserver_PgClientService_DeleteSequenceTuple":                        0,
				"service_response_bytes_yb_tserver_PgClientService_DropDatabase":                               0,
				"service_response_bytes_yb_tserver_PgClientService_DropReplicationSlot":                        0,
				"service_response_bytes_yb_tserver_PgClientService_DropTable":                                  0,
				"service_response_bytes_yb_tserver_PgClientService_DropTablegroup":                             0,
				"service_response_bytes_yb_tserver_PgClientService_FetchData":                                  0,
				"service_response_bytes_yb_tserver_PgClientService_FetchSequenceTuple":                         0,
				"service_response_bytes_yb_tserver_PgClientService_FinishTransaction":                          0,
				"service_response_bytes_yb_tserver_PgClientService_GetActiveTransactionList":                   0,
				"service_response_bytes_yb_tserver_PgClientService_GetCatalogMasterVersion":                    0,
				"service_response_bytes_yb_tserver_PgClientService_GetDatabaseInfo":                            62,
				"service_response_bytes_yb_tserver_PgClientService_GetIndexBackfillProgress":                   0,
				"service_response_bytes_yb_tserver_PgClientService_GetLockStatus":                              0,
				"service_response_bytes_yb_tserver_PgClientService_GetNewObjectId":                             0,
				"service_response_bytes_yb_tserver_PgClientService_GetReplicationSlot":                         0,
				"service_response_bytes_yb_tserver_PgClientService_GetReplicationSlotStatus":                   0,
				"service_response_bytes_yb_tserver_PgClientService_GetTableDiskSize":                           0,
				"service_response_bytes_yb_tserver_PgClientService_GetTableKeyRanges":                          0,
				"service_response_bytes_yb_tserver_PgClientService_GetTablePartitionList":                      0,
				"service_response_bytes_yb_tserver_PgClientService_GetTserverCatalogVersionInfo":               17,
				"service_response_bytes_yb_tserver_PgClientService_Heartbeat":                                  1995,
				"service_response_bytes_yb_tserver_PgClientService_InsertSequenceTuple":                        0,
				"service_response_bytes_yb_tserver_PgClientService_IsInitDbDone":                               0,
				"service_response_bytes_yb_tserver_PgClientService_IsObjectPartOfXRepl":                        0,
				"service_response_bytes_yb_tserver_PgClientService_ListLiveTabletServers":                      0,
				"service_response_bytes_yb_tserver_PgClientService_ListReplicationSlots":                       0,
				"service_response_bytes_yb_tserver_PgClientService_OpenTable":                                  18604,
				"service_response_bytes_yb_tserver_PgClientService_Perform":                                    2839192,
				"service_response_bytes_yb_tserver_PgClientService_ReadSequenceTuple":                          0,
				"service_response_bytes_yb_tserver_PgClientService_ReserveOids":                                0,
				"service_response_bytes_yb_tserver_PgClientService_RollbackToSubTransaction":                   0,
				"service_response_bytes_yb_tserver_PgClientService_SetActiveSubTransaction":                    0,
				"service_response_bytes_yb_tserver_PgClientService_TabletServerCount":                          0,
				"service_response_bytes_yb_tserver_PgClientService_TabletsMetadata":                            0,
				"service_response_bytes_yb_tserver_PgClientService_TruncateTable":                              0,
				"service_response_bytes_yb_tserver_PgClientService_UpdateSequenceTuple":                        0,
				"service_response_bytes_yb_tserver_PgClientService_ValidatePlacement":                          0,
				"service_response_bytes_yb_tserver_PgClientService_WaitForBackendsCatalogVersion":              0,
				"service_response_bytes_yb_tserver_PgClientService_YCQLStatementStats":                         0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteBootstrapSession":         0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_BeginRemoteSnapshotTransferSession":  0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_ChangePeerRole":                      0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_CheckRemoteBootstrapSessionActive":   0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_EndRemoteBootstrapSession":           0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_FetchData":                           0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_KeepLogAnchorAlive":                  0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_RegisterLogAnchor":                   0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_RemoveRemoteBootstrapSession":        0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_UnregisterLogAnchor":                 0,
				"service_response_bytes_yb_tserver_RemoteBootstrapService_UpdateLogAnchor":                     0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_AddTableToTablet":                  0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_AlterSchema":                       0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_BackfillDone":                      0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_BackfillIndex":                     0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_ClonePgSchema":                     0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_CloneTablet":                       0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_CountIntents":                      0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_CreateTablet":                      90,
				"service_response_bytes_yb_tserver_TabletServerAdminService_DeleteTablet":                      0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_EnableDbConns":                     0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_FlushTablets":                      0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_GetSafeTime":                       0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_PrepareDeleteTransactionTablet":    0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_RemoveTableFromTablet":             0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_SplitTablet":                       0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_TestRetry":                         0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_UpdateTransactionTablesVersion":    0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_UpgradeYsql":                       0,
				"service_response_bytes_yb_tserver_TabletServerAdminService_WaitForYsqlBackendsCatalogVersion": 0,
				"service_response_bytes_yb_tserver_TabletServerBackupService_TabletSnapshotOp":                 0,
				"service_response_bytes_yb_tserver_TabletServerService_AbortTransaction":                       0,
				"service_response_bytes_yb_tserver_TabletServerService_AcquireObjectLocks":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_AdminExecutePgsql":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_CancelTransaction":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_CheckTserverTabletHealth":               0,
				"service_response_bytes_yb_tserver_TabletServerService_Checksum":                               0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearAllMetaCachesOnServer":             0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearMetacache":                         0,
				"service_response_bytes_yb_tserver_TabletServerService_ClearUniverseUuid":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_GetCompatibleSchemaVersion":             0,
				"service_response_bytes_yb_tserver_TabletServerService_GetLockStatus":                          0,
				"service_response_bytes_yb_tserver_TabletServerService_GetLogLocation":                         0,
				"service_response_bytes_yb_tserver_TabletServerService_GetMasterAddresses":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_GetOldSingleShardWaiters":               0,
				"service_response_bytes_yb_tserver_TabletServerService_GetOldTransactions":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_GetSharedData":                          0,
				"service_response_bytes_yb_tserver_TabletServerService_GetSplitKey":                            0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTabletKeyRanges":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTabletStatus":                        0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTransactionStatus":                   0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTransactionStatusAtParticipant":      0,
				"service_response_bytes_yb_tserver_TabletServerService_GetTserverCatalogVersionInfo":           0,
				"service_response_bytes_yb_tserver_TabletServerService_ImportData":                             0,
				"service_response_bytes_yb_tserver_TabletServerService_IsTabletServerReady":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_ListMasterServers":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_ListTablets":                            0,
				"service_response_bytes_yb_tserver_TabletServerService_ListTabletsForTabletServer":             0,
				"service_response_bytes_yb_tserver_TabletServerService_NoOp":                                   0,
				"service_response_bytes_yb_tserver_TabletServerService_ProbeTransactionDeadlock":               0,
				"service_response_bytes_yb_tserver_TabletServerService_Publish":                                0,
				"service_response_bytes_yb_tserver_TabletServerService_Read":                                   0,
				"service_response_bytes_yb_tserver_TabletServerService_ReleaseObjectLocks":                     0,
				"service_response_bytes_yb_tserver_TabletServerService_StartRemoteSnapshotTransfer":            0,
				"service_response_bytes_yb_tserver_TabletServerService_Truncate":                               0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransaction":                      0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransactionStatusLocation":        0,
				"service_response_bytes_yb_tserver_TabletServerService_UpdateTransactionWaitingForStatus":      0,
				"service_response_bytes_yb_tserver_TabletServerService_VerifyTableRowRange":                    0,
				"service_response_bytes_yb_tserver_TabletServerService_Write":                                  2267216,
			},
		},
		"success on ycql response": {
			prepare:         caseYcqlOk,
			wantNumOfCharts: 12,
			wantMetrics: map[string]int64{
				"handler_latency_yb_cqlserver_SQLProcessor_AnalyzeRequest_count":      89,
				"handler_latency_yb_cqlserver_SQLProcessor_AnalyzeRequest_sum":        6348,
				"handler_latency_yb_cqlserver_SQLProcessor_DeleteStmt_count":          0,
				"handler_latency_yb_cqlserver_SQLProcessor_DeleteStmt_sum":            0,
				"handler_latency_yb_cqlserver_SQLProcessor_ExecuteRequest_count":      161,
				"handler_latency_yb_cqlserver_SQLProcessor_ExecuteRequest_sum":        241399,
				"handler_latency_yb_cqlserver_SQLProcessor_InsertStmt_count":          0,
				"handler_latency_yb_cqlserver_SQLProcessor_InsertStmt_sum":            0,
				"handler_latency_yb_cqlserver_SQLProcessor_NumFlushesToExecute_count": 161,
				"handler_latency_yb_cqlserver_SQLProcessor_NumFlushesToExecute_sum":   161,
				"handler_latency_yb_cqlserver_SQLProcessor_NumRetriesToExecute_count": 161,
				"handler_latency_yb_cqlserver_SQLProcessor_NumRetriesToExecute_sum":   0,
				"handler_latency_yb_cqlserver_SQLProcessor_NumRoundsToAnalyze_count":  89,
				"handler_latency_yb_cqlserver_SQLProcessor_NumRoundsToAnalyze_sum":    89,
				"handler_latency_yb_cqlserver_SQLProcessor_OtherStmts_count":          0,
				"handler_latency_yb_cqlserver_SQLProcessor_OtherStmts_sum":            0,
				"handler_latency_yb_cqlserver_SQLProcessor_ParseRequest_count":        89,
				"handler_latency_yb_cqlserver_SQLProcessor_ParseRequest_sum":          8246,
				"handler_latency_yb_cqlserver_SQLProcessor_ResponseSize_count":        161,
				"handler_latency_yb_cqlserver_SQLProcessor_ResponseSize_sum":          1570190,
				"handler_latency_yb_cqlserver_SQLProcessor_SelectStmt_count":          161,
				"handler_latency_yb_cqlserver_SQLProcessor_SelectStmt_sum":            241399,
				"handler_latency_yb_cqlserver_SQLProcessor_Transaction_count":         0,
				"handler_latency_yb_cqlserver_SQLProcessor_Transaction_sum":           0,
				"handler_latency_yb_cqlserver_SQLProcessor_UpdateStmt_count":          0,
				"handler_latency_yb_cqlserver_SQLProcessor_UpdateStmt_sum":            0,
				"handler_latency_yb_cqlserver_SQLProcessor_UseStmt_count":             0,
				"handler_latency_yb_cqlserver_SQLProcessor_UseStmt_sum":               0,
				"hybrid_clock_skew": 0,
			},
		},
		"success on ysql response": {
			prepare:         caseYsqlOk,
			wantNumOfCharts: 20,
			wantMetrics: map[string]int64{
				"handler_latency_yb_ysqlserver_SQLProcessor_AggregatePushdowns_count":        0,
				"handler_latency_yb_ysqlserver_SQLProcessor_AggregatePushdowns_sum":          0,
				"handler_latency_yb_ysqlserver_SQLProcessor_BeginStmt_count":                 0,
				"handler_latency_yb_ysqlserver_SQLProcessor_BeginStmt_sum":                   0,
				"handler_latency_yb_ysqlserver_SQLProcessor_CommitStmt_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_CommitStmt_sum":                  0,
				"handler_latency_yb_ysqlserver_SQLProcessor_DeleteStmt_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_DeleteStmt_sum":                  0,
				"handler_latency_yb_ysqlserver_SQLProcessor_InsertStmt_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_InsertStmt_sum":                  0,
				"handler_latency_yb_ysqlserver_SQLProcessor_OtherStmts_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_OtherStmts_sum":                  0,
				"handler_latency_yb_ysqlserver_SQLProcessor_RollbackStmt_count":              0,
				"handler_latency_yb_ysqlserver_SQLProcessor_RollbackStmt_sum":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_SelectStmt_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_SelectStmt_sum":                  0,
				"handler_latency_yb_ysqlserver_SQLProcessor_SingleShardTransactions_count":   0,
				"handler_latency_yb_ysqlserver_SQLProcessor_SingleShardTransactions_sum":     0,
				"handler_latency_yb_ysqlserver_SQLProcessor_Single_Shard_Transactions_count": 0,
				"handler_latency_yb_ysqlserver_SQLProcessor_Single_Shard_Transactions_sum":   0,
				"handler_latency_yb_ysqlserver_SQLProcessor_Transactions_count":              0,
				"handler_latency_yb_ysqlserver_SQLProcessor_Transactions_sum":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_UpdateStmt_count":                0,
				"handler_latency_yb_ysqlserver_SQLProcessor_UpdateStmt_sum":                  0,
				"yb_ysqlserver_active_connection_total":                                      0,
				"yb_ysqlserver_connection_available":                                         299,
				"yb_ysqlserver_connection_over_limit_total":                                  0,
				"yb_ysqlserver_connection_total":                                             1,
				"yb_ysqlserver_max_connection_total":                                         300,
				"yb_ysqlserver_new_connection_total":                                         1,
			},
		},
		"fail on invalid data response": {
			prepare:     caseInvalidDataResponse,
			wantMetrics: nil,
		},
		"fail on connection refused": {
			prepare:     caseConnectionRefused,
			wantMetrics: nil,
		},
		"fail on 404 response": {
			prepare:     case404,
			wantMetrics: nil,
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			collr, cleanup := test.prepare(t)
			defer cleanup()

			_ = collr.Check(context.Background())

			mx := collr.Collect(context.Background())

			require.Equal(t, test.wantMetrics, mx)

			if len(test.wantMetrics) > 0 {
				assert.Equal(t, test.wantNumOfCharts, len(*collr.Charts()), "want charts")

				module.TestMetricsHasAllChartsDims(t, collr.Charts(), mx)
			}
		})
	}
}

func caseMasterOk(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			switch r.URL.Path {
			case "/prometheus-metrics":
				_, _ = w.Write(dataVer223Master)
			default:
				w.WriteHeader(http.StatusNotFound)
			}
		}))
	collr := New()
	collr.URL, _ = url.JoinPath(srv.URL, "/prometheus-metrics")
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}

func caseTserverOk(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			switch r.URL.Path {
			case "/prometheus-metrics":
				_, _ = w.Write(dataVer223Tserver)
			default:
				w.WriteHeader(http.StatusNotFound)
			}
		}))
	collr := New()
	collr.URL, _ = url.JoinPath(srv.URL, "/prometheus-metrics")
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}

func caseYcqlOk(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			switch r.URL.Path {
			case "/prometheus-metrics":
				_, _ = w.Write(dataVer223YCQL)
			default:
				w.WriteHeader(http.StatusNotFound)
			}
		}))
	collr := New()
	collr.URL, _ = url.JoinPath(srv.URL, "/prometheus-metrics")
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}

func caseYsqlOk(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			switch r.URL.Path {
			case "/prometheus-metrics":
				_, _ = w.Write(dataVer223YSQL)
			default:
				w.WriteHeader(http.StatusNotFound)
			}
		}))
	collr := New()
	collr.URL, _ = url.JoinPath(srv.URL, "/prometheus-metrics")
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}

func caseInvalidDataResponse(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			_, _ = w.Write([]byte("hello and\n goodbye"))
		}))
	collr := New()
	collr.URL = srv.URL
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}

func caseConnectionRefused(t *testing.T) (*Collector, func()) {
	t.Helper()
	collr := New()
	collr.URL = "http://127.0.0.1:65001"
	require.NoError(t, collr.Init(context.Background()))

	return collr, func() {}
}

func case404(t *testing.T) (*Collector, func()) {
	t.Helper()
	srv := httptest.NewServer(http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(http.StatusNotFound)
		}))
	collr := New()
	collr.URL = srv.URL
	require.NoError(t, collr.Init(context.Background()))

	return collr, srv.Close
}
