﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/athena/model/WorkGroupConfiguration.h>
#include <aws/athena/model/WorkGroupState.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Athena {
namespace Model {

/**
 * <p>A workgroup, which contains a name, description, creation time, state, and
 * other configuration, listed under <a>WorkGroup$Configuration</a>. Each workgroup
 * enables you to isolate queries for you or your group of users from other queries
 * in the same account, to configure the query results location and the encryption
 * configuration (known as workgroup settings), to enable sending query metrics to
 * Amazon CloudWatch, and to establish per-query data usage control limits for all
 * queries in a workgroup. The workgroup settings override is specified in
 * <code>EnforceWorkGroupConfiguration</code> (true/false) in the
 * <code>WorkGroupConfiguration</code>. See
 * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/WorkGroup">AWS
 * API Reference</a></p>
 */
class WorkGroup {
 public:
  AWS_ATHENA_API WorkGroup() = default;
  AWS_ATHENA_API WorkGroup(Aws::Utils::Json::JsonView jsonValue);
  AWS_ATHENA_API WorkGroup& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ATHENA_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The workgroup name.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  WorkGroup& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the workgroup: ENABLED or DISABLED.</p>
   */
  inline WorkGroupState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(WorkGroupState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline WorkGroup& WithState(WorkGroupState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the workgroup, which includes the location in Amazon S3
   * where query and calculation results are stored, the encryption configuration, if
   * any, used for query and calculation results; whether the Amazon CloudWatch
   * Metrics are enabled for the workgroup; whether workgroup settings override
   * client-side settings; and the data usage limits for the amount of data scanned
   * per query or per workgroup. The workgroup settings override is specified in
   * <code>EnforceWorkGroupConfiguration</code> (true/false) in the
   * <code>WorkGroupConfiguration</code>. See
   * <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.</p>
   */
  inline const WorkGroupConfiguration& GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = WorkGroupConfiguration>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = WorkGroupConfiguration>
  WorkGroup& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The workgroup description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  WorkGroup& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time the workgroup was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  WorkGroup& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the IAM Identity Center enabled application associated with the
   * workgroup.</p>
   */
  inline const Aws::String& GetIdentityCenterApplicationArn() const { return m_identityCenterApplicationArn; }
  inline bool IdentityCenterApplicationArnHasBeenSet() const { return m_identityCenterApplicationArnHasBeenSet; }
  template <typename IdentityCenterApplicationArnT = Aws::String>
  void SetIdentityCenterApplicationArn(IdentityCenterApplicationArnT&& value) {
    m_identityCenterApplicationArnHasBeenSet = true;
    m_identityCenterApplicationArn = std::forward<IdentityCenterApplicationArnT>(value);
  }
  template <typename IdentityCenterApplicationArnT = Aws::String>
  WorkGroup& WithIdentityCenterApplicationArn(IdentityCenterApplicationArnT&& value) {
    SetIdentityCenterApplicationArn(std::forward<IdentityCenterApplicationArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  WorkGroupState m_state{WorkGroupState::NOT_SET};
  bool m_stateHasBeenSet = false;

  WorkGroupConfiguration m_configuration;
  bool m_configurationHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Utils::DateTime m_creationTime{};
  bool m_creationTimeHasBeenSet = false;

  Aws::String m_identityCenterApplicationArn;
  bool m_identityCenterApplicationArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
