/* html.c
 * Parse html tags.
 * This file is part of the edbrowse project, released under GPL.
 */

#include "eb.h"

#ifdef _MSC_VER			// sleep(secs) macro
#define SLEEP(a) Sleep(a * 1000)
extern int gettimeofday(struct timeval *tp, void *tzp);	// from tidys.lib
#else // !_MSC_VER
#define SLEEP sleep
#endif // _MSC_VER y/n

// OSX has no pthread_tryjoin_np, so we can't do our
// asynchronous timers under OSX, which is really no great loss.
#if defined(__APPLE__) || defined(__ANDROID__) || defined(__FreeBSD__)
#define pthread_tryjoin_np pthread_join
#endif

uchar browseLocal;
bool showHover, doColors;

static Tag *js_reset, *js_submit;
static const int asyncTimer = 700;

// start a document.write on current frame, i.e. cf
void dwStart(void)
{
	if (cf->dw)
		return;
	cf->dw = initString(&cf->dw_l);
	stringAndString(&cf->dw, &cf->dw_l, "<!DOCTYPE public><body>");
}

bool handlerPresent(const Tag *t, const char *name)
{
	const char *name2 = tack_fn(name);
	return (typeof_property_t(t, name) == EJ_PROP_FUNCTION ||
	(name2 && typeof_property_t(t, name2) == EJ_PROP_FUNCTION));
}

// called with click, dblclick, reset, or submit
bool tagHandler(int seqno, const char *name)
{
	Tag *t = tagList[seqno];
/* check the htnl tag attributes first */
	if (t->onclick && stringEqual(name, "onclick"))
		return true;
	if (t->onsubmit && stringEqual(name, "onsubmit"))
		return true;
	if (t->onreset && stringEqual(name, "onreset"))
		return true;
	if (t->onchange && stringEqual(name, "onchange"))
		return true;

	if (!t->jslink)
		return false;
	if (!isJSAlive)
		return false;
	if (!handlerPresent(t, name))
		return false;

	if (stringEqual(name, "onclick"))
		t->onclick = true;
	if (stringEqual(name, "onsubmit"))
		t->onsubmit = true;
	if (stringEqual(name, "onreset"))
		t->onreset = true;
	if (stringEqual(name, "onchange"))
		t->onchange = true;
	return true;
}

static void formReset(const Tag *form);

/*********************************************************************
This function was originally written to incorporate any strings generated by
document.write(), and it still does that,
but now it does much more.
It handles any side effects that occur from running js.
innerHTML tags generated, form input values set, timers,
form.reset(), form.submit(), document.location = url, etc.
Every js activity should start with jSyncup() and end with jSideEffects().
WARNING: this routine mucks with cf, so you need to set it afterwards,
the button being pushed or the onclick code or whatever frame is appropriate.
*********************************************************************/

void jSideEffects(void)
{
	if (!cw->browseMode || !isJSAlive)
		return;
	debugPrint(4, "jSideEffects starts");
	runScriptsPending(false);
	cw->mustrender = true;
	rebuildSelectors();
	debugPrint(4, "jSideEffects ends");
}

static Tag *locateOptionByName(const Tag *sel,
					  const char *name, int *pmc,
					  bool exact)
{
	Tag *t, *em = 0, *pm = 0;
	int pmcount = 0;	/* partial match count */
	const char *s;

	for (t = cw->optlist; t; t = t->same) {
		if (t->controller != sel)
			continue;
		if (!(s = t->textval))
			continue;
		if (stringEqualCI(s, name)) {
			em = t;
			continue;
		}
		if (exact)
			continue;
		if (strcasestr(s, name)) {
			pm = t;
			++pmcount;
		}
	}
	if (em)
		return em;
	if (pmcount == 1)
		return pm;
	*pmc = (pmcount > 0);
	return 0;
}

static Tag *locateOptionByNum(const Tag *sel, int n)
{
	Tag *t;
	int cnt = 0;

	for (t = cw->optlist; t; t = t->same) {
		if (t->controller != sel)
			continue;
		if (!t->textval)
			continue;
		++cnt;
		if (cnt == n)
			return t;
	}
	return 0;
}

static bool
locateOptions(const Tag *sel, const char *input,
	      char **disp_p, char **val_p, bool setcheck)
{
	Tag *t;
	char *disp, *val;
	int disp_l, val_l;
	int len = strlen(input);
	int n, pmc;
	const char *s, *e;	/* start and end of an option */
	char *iopt;		/* individual option */

	iopt = (char *)allocMem(len + 1);
	disp = initString(&disp_l);
	val = initString(&val_l);

	if (setcheck) {
/* Uncheck all existing options, then check the ones selected. */
		if (sel->jslink && allowJS)
			set_property_number_t(sel, "selectedIndex", -1);
		for (t = cw->optlist; t; t = t->same) {
			if (t->controller == sel && t->textval) {
				t->checked = false;
				if (t->jslink && allowJS)
					set_property_bool_t(t, "selected",   false);
			}
		}
	}

	s = input;
	while (*s) {
		e = 0;
		if (sel->multiple)
			e = strchr(s, selsep);
		if (!e)
			e = s + strlen(s);
		len = e - s;
		strncpy(iopt, s, len);
		iopt[len] = 0;
		s = e;
		if (*s == selsep)
			++s;

		t = locateOptionByName(sel, iopt, &pmc, true);
		if (!t) {
			n = stringIsNum(iopt);
			if (n >= 0)
				t = locateOptionByNum(sel, n);
		}
		if (!t)
			t = locateOptionByName(sel, iopt, &pmc, false);
		if (!t) {
			if (n >= 0)
				setError(MSG_XOutOfRange, n);
			else
				setError(pmc + MSG_OptMatchNone, iopt);
/* This should never happen when we're doing a set check */
			if (setcheck) {
				runningError(MSG_OptionSync, iopt);
				continue;
			}
			goto fail;
		}

		if (val_p) {
			if (*val)
				stringAndChar(&val, &val_l, '\1');
			stringAndString(&val, &val_l, t->value);
		}

		if (disp_p) {
			if (*disp)
				stringAndChar(&disp, &disp_l, selsep);
// special exception for <input> with datalist - revert back to value.
			stringAndString(&disp, &disp_l,
			(sel->action == TAGACT_DATAL ? t->value : t->textval));
		}

		if (setcheck) {
			t->checked = true;
			if (t->jslink && allowJS) {
				set_property_bool_t(t, "selected", true);
				if (sel->jslink && allowJS)
					set_property_number_t(sel, "selectedIndex", t->lic);
			}
		}
	}			/* loop over multiple options */

	if (val_p)
		*val_p = val;
	if (disp_p)
		*disp_p = disp;
	free(iopt);
	return true;

fail:
	free(iopt);
	nzFree(val);
	nzFree(disp);
	if (val_p)
		*val_p = 0;
	if (disp_p)
		*disp_p = 0;
	return false;
}

/*********************************************************************
Sync up the javascript variables with the input fields.
This is required before running any javascript, e.g. an onclick function.
After all, the input fields may have changed.
You may have changed the last name from Flintstone to Rubble.
This has to propagate down to the javascript strings in the DOM.
This is quick and stupid; I just update everything.
Most of the time I'm setting the strings to what they were before;
that's the way it goes.
*********************************************************************/

void jSyncup(bool fromtimer, const Tag *active)
{
	Tag *t;
	int itype, j, cx;
	char *value, *cxbuf;

	if (!cw->browseMode)
		return;		/* not necessary */
	if (cw->sank)
		return;		/* already done */
	cw->sank = true;
	if (!isJSAlive)
		return;
	debugPrint(4, "jSyncup starts");
	if (!fromtimer)
		cw->nextrender = 0;

	if(active)
		set_property_object_doc(cf, "activeElement", active);

	for (t = cw->inputlist; t; t = t->same) {
		itype = t->itype;
		if (itype <= INP_HIDDEN)
			continue;

		if (itype >= INP_RADIO) {
			int checked = fieldIsChecked(t->seqno);
			if (checked < 0)
				continue;
			t->checked = checked;
			set_property_bool_t(t, "checked", checked);
			continue;
		}

		value = getFieldFromBuffer(t->seqno);
/* If that line has been deleted from the user's buffer,
 * indicated by value = 0,
 * then don't do anything. */
		if (!value)
			continue;

		if (itype == INP_SELECT) {
/* set option.selected in js based on the option(s) in value */
			locateOptions(t, (value ? value : t->value), 0, 0,
				      true);
			if (value) {
				nzFree(t->value);
				t->value = value;
			}
			continue;
		}

		if (itype == INP_TA) {
			if (!value) {
				set_property_string_t(t, "value", 0);
				continue;
			}
/* Now value is just <buffer 3>, which is meaningless. */
			nzFree(value);
			cx = t->lic;
			if (!cx)
				continue;
// unfoldBuffer could fail if we have quit that session.
			if (!unfoldBuffer(cx, false, &cxbuf, &j))
				continue;
			set_property_string_t(t, "value", cxbuf);
			nzFree(cxbuf);
			continue;
		}

		if (value) {
			set_property_string_t(t, "value", value);
			nzFree(t->value);
			t->value = value;
		}
	}			/* loop over tags */

	debugPrint(4, "jSyncup ends");
}

void jClearSync(void)
{
	if (cw->browseMode) {
		cw->sank = false;
		return;
	}
/* when we are able to jSyncup windows other than the foreground window,
 * which we can't do yet, then the rest of this will make sense. */
#if 0
	for (cx = 1; cx <= maxSession; ++cx) {
		w = sessionList[cx].lw;
		while (w) {
			w->sank = false;
			w = w->prev;
		}
	}
#endif
}

/*********************************************************************
This function is called for a new web page, by http refresh,
or by document.location = new_url, or by new Window().
If delay is 0 or less then the action should happen now.
The refresh parameter means replace the current page.
This is false only if js creates a new window, which should stack up on top of the old.
*********************************************************************/

char *newlocation;
int newloc_d;			/* possible delay */
bool newloc_r;			/* replace the buffer */
Frame *newloc_f;	/* frame calling for new web page */
bool js_redirects;
static void gotoLocation(char *url, int delay, bool rf)
{
	if (!allowRedirection) {
		debugPrint(1, "javascript redirection disabled: %s", url);
		nzFree(url);
		return;
	}
	if (newlocation && delay >= newloc_d) {
		nzFree(url);
		return;
	}
	nzFree(newlocation);
	newlocation = url;
	newloc_d = delay;
	newloc_r = rf;
	newloc_f = cf;
	if (!delay)
		js_redirects = true;
}

/* helper function for meta tag */
void htmlMetaHelper(const Tag *t)
{
	char *name;
	const char *content, *heq, *charset;
	char **ptr;
	char *copy = 0;

/* if we're generating a cookie, we better get the frame right,
 * because that's the url that the cookie belongs to.
 * I think the frame is correct anyways, because we are parsing html,
 * but just to be safe ... */
	cf = t->f0;

// if multiple charsets, the first one wins
	charset = attribVal(t, "charset");
	if(charset && *charset && !cf->charset)
		cf->charset = charset;

	name = t->name;
	content = attribVal(t, "content");
	copy = cloneString(content);
	heq = attribVal(t, "http-equiv");

	if (heq && content) {
		bool rc;
		int delay;

/* It's not clear if we should process the http refresh command
 * immediately, the moment we spot it, or if we finish parsing
 * all the html first.
 * Does it matter?  It might.
 * A subsequent meta tag could use http-equiv to set a cooky,
 * and we won't see that cooky if we jump to the new page right now.
 * And there's no telling what subsequent javascript might do.
 * So I'm going to postpone the refresh until everything is parsed.
 * Bear in mind, we really don't want to refresh if we're working
 * on a local file. */

		if (stringEqualCI(heq, "Set-Cookie")) {
			rc = receiveCookie(cf->fileName, content);
			debugPrint(3, rc ? "jar" : "rejected");
		}

		if (allowRedirection && !browseLocal
		    && stringEqualCI(heq, "Refresh")) {
			if (parseRefresh(copy, &delay)) {
				char *newcontent;
				unpercentURL(copy);
				newcontent = resolveURL(cf->hbase, copy);
				gotoLocation(newcontent, delay, true);
			}
		}
	}

	if (name) {
		ptr = 0;
		if (stringEqualCI(name, "description"))
			ptr = &cw->htmldesc;
		if (stringEqualCI(name, "keywords"))
			ptr = &cw->htmlkey;
		if (ptr && !*ptr && content) {
			stripWhite(copy);
			*ptr = copy;
			copy = 0;
		}
	}

	nzFree(copy);
}

/* pre is the predecoration from edbrowse-js, if appropriate */
static void runGeneratedHtml(Tag *t, const char *h)
{
	int l = cw->numTags;

	debugPrint(3, "parse html from docwrite");
	if (t)
		debugPrint(4, "parse under %s %d", t->info->name, t->seqno);
	else
		debugPrint(4, "parse under top");
	debugGenerated(h);

	html2nodes(h, false);
	htmlGenerated = true;
	htmlNodesIntoTree(l, t);
	prerender(false);
	decorate(0);
	debugPrint(3, "end parse html from docwrite");
}

/* helper function to prepare an html script.
 * steps: 1 parsed as html, 2 decorated with a coresponding javascript object
 * 3 downloading in background,
 * 4 data fetched and in the js world and possibly deminimized,
 * 5 script has run, 6 script could not run. */

void prepareScript(Tag *t)
{
	const char *js_file = "generated";
	char *js_text = 0;
	const char *filepart;
	char *b;
	int blen;
	Frame *f = t->f0;

	if (intFlag)
		goto fail;

/* As per the starting line number, we cant distinguish between
 * <script> foo </script>  and
 * <script>
 * foo
 * </script>
 * so make a guess towards the first form, knowing we could be off by 1.
 * Just leave it at t->js_ln */
	if (f->fileName && !t->scriptgen)
		js_file = f->fileName;

	if (t->jslink) {
// js might have set, or changed, the source url.
		char *new_url = get_property_url_t(t, false);
		if (new_url && *new_url) {
			if (t->href && !stringEqual(t->href, new_url))
				debugPrint(3, "js replaces script %s with %s",
					   t->href, new_url);
			nzFree(t->href);
			t->href = new_url;
		}
		t->async = get_property_bool_t(t, "async");
// A side effect of tidy + edbrowse is that the text of the script is a
// childNode of script, but I don't think it should be.
		if (t->firstchild && t->firstchild->action == TAGACT_TEXT)
			run_function_onearg_t(t, "removeChild", t->firstchild);
	}

	if (t->href) {		/* fetch the javascript page */
		const char *altsource = 0, *realsource = 0;
		bool from_data;
		if (!javaOK(t->href))
			goto fail;
		from_data = isDataURI(t->href);
		if (!from_data) {
			altsource = fetchReplace(t->href);
			realsource = (altsource ? altsource : t->href);
		}
		debugPrint(3, "js source %s",
			   !from_data ? realsource : "data URI");
		if (from_data) {
			char *mediatype;
			int data_l = 0;
			if (parseDataURI(t->href, &mediatype,
					 &js_text, &data_l)) {
				prepareForBrowse(js_text, data_l);
				nzFree(mediatype);
			} else {
				debugPrint(3,
					   "Unable to parse data URI containing JavaScript");
				goto fail;
			}
		} else if ((browseLocal || altsource) && !isURL(realsource)) {
			char *h = cloneString(realsource);
			unpercentString(h);
			if (!fileIntoMemory(h, &b, &blen)) {
				if (debugLevel >= 1)
					i_printf(MSG_GetLocalJS);
				nzFree(h);
				goto fail;
			}
			js_text = force_utf8(b, blen);
			if (!js_text)
				js_text = b;
			else
				nzFree(b);
			nzFree(h);
		} else {
			struct i_get g;

// this has to happen before threads spin off
			if (!curlActive) {
				eb_curl_global_init();
				cookiesFromJar();
				setupEdbrowseCache();
			}

			if (down_jsbg && !demin && !uvw
			    && !pthread_create(&t->loadthread, NULL,
					       httpConnectBack2, (void *)t)) {
				t->js_ln = 1;
				js_file = realsource;
				filepart = getFileURL(js_file, true);
				t->js_file = cloneString(filepart);
// stop here and wait for the child process to download
				t->step = 3;
				return;
			}
			memset(&g, 0, sizeof(g));
			g.thisfile = f->fileName;
			g.uriEncoded = true;
			g.url = realsource;
			if (!httpConnect(&g)) {
				if (debugLevel >= 3)
					i_printf(MSG_GetJS2);
				goto fail;
			}
			if (g.code == 200) {
				js_text = force_utf8(g.buffer, g.length);
				if (!js_text)
					js_text = g.buffer;
				else
					nzFree(g.buffer);
			} else {
				nzFree(g.buffer);
				if (debugLevel >= 3)
					i_printf(MSG_GetJS, g.url, g.code);
				goto fail;
			}
		}
		t->js_ln = 1;
		js_file = (!from_data ? realsource : "data_URI");
	} else {
		js_text = t->textval;
		t->textval = 0;
	}

	if (!js_text) {
// we can only run if javascript has supplied the code forr this scrip,
// because we have none to offer.
// Such code cannot be deminimized.
		goto success;
	}
	set_property_string_t(t, "text", js_text);
	nzFree(js_text);

	filepart = getFileURL(js_file, true);
	t->js_file = cloneString(filepart);

// deminimize the code if we're debugging.
	if (demin)
		run_function_onearg_win(f, "eb$demin", t);
	if (uvw)
		run_function_onearg_win(f, "eb$watch", t);

success:
	t->step = 4;
	return;

fail:
	t->step = 6;
}

static bool is_subframe(Frame *f1, Frame *f2)
{
	Tag *t;
	int n;
	if (f1 == f2)
		return true;
	while (true) {
		for (n = 0; n < cw->numTags; ++n) {
			t = tagList[n];
			if (t->f1 == f1)
				goto found;
		}
		return false;
found:
		f1 = t->f0;
		if (f1 == f2)
			return true;
	}
}

/*********************************************************************
Run pending scripts, and perform other actions that have been queued up by javascript.
This includes document.write, linkages, perhaps even form.submit.
Things stop however if we detect document.location = new_url,
i.e. a page replacement, as indicated by the newlocation variable being set.
The old page doesn't matter any more.
I run the scripts linked to the current frame.
That way the scripts in a subframe will run, then return, then the scripts
in the parent frame pick up where they left off.
The algorithm for which sripts to run when is far from obvious,
and nowhere documented.
I had to write several contrived web pages and run them through chrome
and firefox and document the results.

1. Scripts that come from the internet (src=url) are different from
inline scripts, i.e. those that are part of the home page, or generated
dynamically with s.text set.
An internet script is loaded, that is, fetched from the internet and then run,
and after that, it's onload handler is run.
An inline script does not run its onload handler, even if it has one.

2. <style> is inline, but <link href=url rel=stylesheet> is internet.
As above, onload code is run after an internet css page is fetched.
This is rather asynchronous, relative to the other scripts.
Just run the onload code when you can.

3. All the scripts on the home page run in sequence.
Each internet script must fetch and run before the next script runs.
Of course all the internet scripts can download in parallel, to save time,
and I do that if down_jsbg is true,
but still, we have to execute them in order.

4. A script could have async set, and in theory I could
skip that one and do the next one if it is available (postpone), or even do the
async script in another thread, but I can't, because duktape is not threadsafe,
as is clearly documented.
So I allow for postponement, that is, two passes,
the first pass runs scripts in order and skips async scripts,
the second pass runs the async scripts.
Pass 2 runs the async scripts in order, and it doesn't have to, but it's
the easiest way to go, and how often do we have several async scripts,
some ready several seconds before others? Not very often.

5. These scripts can generate other scripts, which run in the next wave.
However, if the generated script is inline, not from the internet,
it runs right now.
The first script pauses, runs the second script, then resumes.
I demonstrated this in my contrived web page,
but not sure it ever happens in the real world.
If your second script sets s.text = "some code here",
then why not embed that code in the first script and be done with it?
So I haven't gone to the bother of implementing this.
All generated scripts, inline and internet, run later.
But if we wanted to implement this, you can probably follow the pattern
set by URL and several other classes.
Create a member text$2, that's where the code lives.
A getter returns text$2 when you ask for text.
A setter runs the code through eval(),
then stores it in text$2 for future reference.
The inline script is executed now.
All the C code deals with text$2 so there are no unintended side effects.
So it's not too hard I suppose, but I haven't seen the need yet.

6. DOMContentLoaded is a strange event.
It fires when the first wave of scripts is complete, but before the second wave or any onload handlers etc.
The startbrowse argument tells runScriptsPending() we are calling it
because a new page is being browsed.
It should dispatch DOMContentLoaded after the first scripts have run.

7. Generated scripts do not run unless and until they are connected to the tree.
If they are connected later, that is when they run.
This is true of both inline and internet.
The first is problematic, because my implementation, in 5,
that just calls eval(s.text) on a setter,
would run all the time, whether the new script was connected or not.
So my 5 implementation would fix one problem and create another.
Not sure what to do about that.
The function isRooted(t) determines whether t is rooted;
this is needed in several places, besides running scripts.

8. Scripts are run in creation order, not in tree order.
Or maybe the order is arbitrary and not prescribed,
in which case creation order is fine.
That is easiest, and it's what I do.
*********************************************************************/

bool isRooted(const Tag *t)
{
	const Tag *up;
	for(up = t; up; up = up->parent)
		if(up->action == TAGACT_HEAD || up->action == TAGACT_BODY)
			return true;
	return false;
}

void runScriptsPending(bool startbrowse)
{
	Tag *t;
	char *js_file;
	const char *a;
	int ln;
	bool change, async;
	Frame *f, *save_cf = cf;

	if (newlocation && newloc_r)
		return;

// Not sure where document.write objects belong.
// For now I'm putting them under body.
// Each write corresponds to the frame containing document.write.
	for (f = &(cw->f0); f; f = f->next) {
		if (!f->dw)
			continue;
		cf = f;
// completely different behavior before and after browse
// After browse, it clobbers the page.
		if(cf->browseMode) {
			run_function_onestring_t(cf->bodytag, "eb$dbih",
			strstr(cf->dw, "<body>")+6);
		} else {
			stringAndString(&cf->dw, &cf->dw_l, "</body>");
			runGeneratedHtml(cf->bodytag, cf->dw);
		}
		nzFree(cf->dw);
		cf->dw = 0;
		cf->dw_l = 0;
	}

top:
	change = false;

	for (t = cw->scriptlist; t; t = t->same) {
		if (t->dead || !t->jslink || t->step >= 3)
			continue;

// don't execute a script until it is linked into the tree.
		if(!isRooted(t))
			continue;

		cf = t->f0;
		prepareScript(t);
// step will now be 3, load in background, 4, loaded, or 6, failure.
	}

	async = false;
passes:

	for (t = cw->scriptlist; t; t = t->same) {
		if (t->dead || !t->jslink || t->step >= 5 || t->step <= 2 || t->async != async)
			continue;
		cf = t->f0;
		if (!is_subframe(cf, save_cf))
			continue;
		if (intFlag) {
			t->step = 6;
			continue;
		}

		if (async && down_jsbg && cw->browseMode) {
			if (!t->intimer) {
				scriptSetsTimeout(t);
				t->intimer = true;
			}
			continue;
		}

		if (t->step == 3) {
// waiting for background process to load
			pthread_join(t->loadthread, NULL);
			if (!t->loadsuccess || t->hcode != 200) {
				if (debugLevel >= 3)
					i_printf(MSG_GetJS, t->href, t->hcode);
				t->step = 6;
				continue;
			}
			set_property_string_t(t,
					    (t->inxhr ? "$entire" : "text"),
					    t->value);
			nzFree(t->value);
			t->value = 0;
			t->step = 4;	// loaded
		}

		t->step = 5;	// now running the script

		if (t->inxhr) {
// xhr looks like an asynchronous script before browse
			char *gc_name;
			run_function_bool_t(t, "parseResponse");
/*********************************************************************
Ok this is subtle. I put it on a script tag, and t.jv.onload exists!
That is the function that is run by xhr.
So runOnload() comes along and runs it again, unless we do something.
I will disconnect here, and also check for inxhr in runOnload().
*********************************************************************/
			disconnectTagObject(t);
			t->dead = true;
// allow garbage collection to recapture the object if it wants to
			gc_name = get_property_string_t(t, "backlink");
			if (gc_name)
				delete_property_win(cf, gc_name);
			goto afterscript;
		}

// If no language is specified, javascript is default.
		a = get_property_string_t(t, "language");
		if (a && *a && (!memEqualCI(a, "javascript", 10) || isalphaByte(a[10]))) {
			debugPrint(3, "script tag %d language %s not executed", t->seqno, a);
			cnzFree(a);
			goto afterscript;
		}
		cnzFree(a);
// Also reject a script if a type is specified and it is not JS.
// For instance, some JSON pairs in script tags on amazon.com
		a = get_property_string_t(t, "type");
// allow for type 5e5857709a179301c738ca91-text/javascript, which really happens.
// Also application/javascript.
		if (a && *a && !stringEqualCI(a, "javascript") &&
		((ln = strlen(a)) < 11 || !stringEqualCI(a + ln - 11, "/javascript"))) {
			debugPrint(3, "script tag %d type %s not executed", t->seqno, a);
			cnzFree(a);
			goto afterscript;
		}
		cnzFree(a);

		js_file = t->js_file;
		if (!js_file)
			js_file = "generated";
		if (cf != save_cf)
			debugPrint(4, "running script at a lower frame %s",
				   js_file);
		ln = t->js_ln;
		if (!ln)
			ln = 1;
		debugPrint(3, "exec %s at %d", js_file, ln);
		jsRunData(t, js_file, ln);
		debugPrint(3, "exec complete");

afterscript:
		if (newlocation && newloc_r) {
			cf = save_cf;
			return;
		}

/* look for document.write from this script */
		if (cf->dw) {
// completely different behavior before and after browse
// After browse, it clobbers the page.
			if(cf->browseMode) {
				run_function_onestring_t(cf->bodytag,
				"eb$dbih",
				strstr(cf->dw, "<body>")+6);
			} else {
// Any newly generated scripts have to run next. Move them up in the linked list.
				Tag *t1, *t2, *u;
				for(u = t; u; u = u->same)
					if(!u->slash) t1 = u;
// t1 is now last real script in the list.
				stringAndString(&cf->dw, &cf->dw_l, "</body>");
				runGeneratedHtml(t, cf->dw);
				run_function_onearg_win(cf, "eb$uplift", t);
				for(u = t1; u; u = u->same)
					if(!u->slash) t2 = u;
				if(t1 != t && t2 != t1) {
					Tag *t3 = t->same;
					t->same = t1->same;
					t2->same = t3;
					t1->same = 0;
					for(u = t->same; u != t3; u = u->same)
						if(u->jslink)
	prepareScript(u);
				}
			}
			nzFree(cf->dw);
			cf->dw = 0;
			cf->dw_l = 0;
		}

		change = true;
	}

// after each pass, see if there is a link onload to run.
	for (t = cw->linklist; t; t = t->same) {
		if(t->lic == 1 && t->jslink && !t->dead &&
		save_cf == t->f0) {
			cf = save_cf;
			run_event_t(t, "link", "onload");
			t->lic = 0;
			change = true;
		}
	}

// If a frame has an onload function, that function might need to run.
// We don't get to wait around for on-demand expansion; we have to expand.
// Fortunately this doesn't happen often.
	for (t = cw->framelist; t; t = t->same) {
		if(!t->f1 && // not expanded yet
		!t->expf && // we haven't tried to expand it yet
		isRooted(t) && // it's in our tree
		typeof_property_t(t, "onload") == EJ_PROP_FUNCTION)
			forceFrameExpand(t);
	}

	if (!async) {
		if(startbrowse)
// I think it's ok to use cf here, but let's be safe.
			run_event_doc(save_cf, "document", "onDOMContentLoaded");
		startbrowse = false;
		async = true;
		goto passes;
	}

	if (change)
		goto top;

	if ((t = js_reset)) {
		js_reset = 0;
		formReset(t);
	}

	if ((t = js_submit)) {
		char *post;
		bool rc;
		js_submit = 0;
		rc = infPush(t->seqno, &post);
		if (rc)
			gotoLocation(post, 0, false);
		else
			showError();
	}

	cf = save_cf;
}

void preFormatCheck(int tagno, bool * pretag, bool * slash)
{
	const Tag *t;
	*pretag = *slash = false;
	if (tagno >= 0 && tagno < cw->numTags) {
		t = tagList[tagno];
		*pretag = (t->action == TAGACT_PRE);
		*slash = t->slash;
	}
}

/* is there a doorway from html to js? */
static bool jsDoorway(void)
{
	const Tag *t;
	int j;
	for (j = 0; j < cw->numTags; ++j) {
		t = tagList[j];
		if (t->doorway)
			return true;
	}
	debugPrint(3, "no js doorway");
	return false;
}

char *htmlParse(char *buf, int remote)
{
	char *a, *newbuf;

	if (tagList)
		i_printfExit(MSG_HtmlNotreentrant);
	if (remote >= 0)
		browseLocal = !remote;
	initTagArray();
	cf->baseset = false;
	cf->hbase = cloneString(cf->fileName);

	debugPrint(3, "parse html from browse");
/* call the tidy parser to build the html nodes */
	html2nodes(buf, true);
	nzFree(buf);
	htmlGenerated = false;
	htmlNodesIntoTree(0, NULL);
	prerender(false);

/* if the html doesn't use javascript, then there's
 * no point in generating it.
 * This is typical of generated html, from pdf for instance,
 * or the html that is in email. */
	if (cf->jslink && !jsDoorway())
		freeJSContext(cf);

	if (isJSAlive) {
		decorate(0);
		set_basehref(cf->hbase);
		run_function_bool_win(cf, "eb$qs$start");
		runScriptsPending(true);
		runOnload();
		runScriptsPending(false);
		run_function_bool_win(cf, "readyStateComplete");
		run_event_win(cf, "window", "onfocus");
		run_event_doc(cf, "document", "onfocus");

		runScriptsPending(false);
		rebuildSelectors();
	}
	debugPrint(3, "end parse html from browse");

	a = render(0);
	debugPrint(6, "|%s|\n", a);
	newbuf = htmlReformat(a);
	nzFree(a);

	return newbuf;
}

/* See if there are simple tags like <p> or </font> */
bool htmlTest(void)
{
	int j, ln;
	int cnt = 0;
	int fsize = 0;		/* file size */
	char look[12];
	bool firstline = true;

	for (ln = 1; ln <= cw->dol; ++ln) {
		char *p = (char *)fetchLine(ln, -1);
		char c;
		int state = 0;

		while (isspaceByte(*p) && *p != '\n')
			++p;
		if (*p == '\n')
			continue;	/* skip blank line */
		if (firstline && *p == '<') {
/* check for <!doctype and other things */
			if (memEqualCI(p + 1, "!doctype", 8))
				return true;
			if (memEqualCI(p + 1, "?xml", 4))
				return true;
			if (memEqualCI(p + 1, "!--", 3))
				return true;
/* If it starts with <tag, for any tag we recognize,
 * we'll call it good. */
			for (j = 1; j < 10; ++j) {
				if (!isalnumByte(p[j]))
					break;
				look[j - 1] = p[j];
			}
			look[j - 1] = 0;
			if (j > 1 && (p[j] == '>' || isspaceByte(p[j]))) {
/* something we recognize? */
				const struct tagInfo *ti;
				for (ti = availableTags; ti->name[0]; ++ti)
					if (stringEqualCI(ti->name, look))
						return true;
			}	/* leading tag */
		}		/* leading < */
		firstline = false;

/* count tags through the buffer */
		for (j = 0; (c = p[j]) != '\n'; ++j) {
			if (state == 0) {
				if (c == '<')
					state = 1;
				continue;
			}
			if (state == 1) {
				if (c == '/')
					state = 2;
				else if (isalphaByte(c))
					state = 3;
				else
					state = 0;
				continue;
			}
			if (state == 2) {
				if (isalphaByte(c))
					state = 3;
				else
					state = 0;
				continue;
			}
			if (isalphaByte(c))
				continue;
			if (c == '>')
				++cnt;
			state = 0;
		}
		fsize += j;
	}			/* loop over lines */

/* we need at least one of these tags every 300 characters.
 * And we need at least 4 such tags.
 * Remember, you can always override by putting <html> at the top. */
	return (cnt >= 4 && cnt * 300 >= fsize);
}

// Connect an input field to its datalist.
// I use the field ninp for this, rather nonobvious, sorry.
static void connectDatalist(Tag *t)
{
	const char *lista = 0; // list from attributes
	char *listj = 0; // list from javascript
	const Tag *u;
	if(t->action != TAGACT_INPUT || t->itype != INP_TEXT ||
	t->ninp)
		return;
	lista = attribVal(t, "list");
	if(t->jslink && allowJS)
		listj = get_property_string_t(t, "list");
	if(listj && *listj)
		lista = listj;
	if(!lista)
		return;
	if((u = gebi_c(t, lista, false)))
		t->ninp = u->seqno;
	nzFree(listj);
}

// Show an input field
void infShow(int tagno, const char *search)
{
	Tag *t = tagList[tagno], *v;
	const char *s;
	int cnt;
	bool show;

	connectDatalist(t);

	s = inp_types[t->itype];
	if(t->ninp && t->itype == INP_TEXT)
		s = "suggested select";
	printf("%s", s);
	if (t->multiple)
		i_printf(MSG_Many);
	if (t->itype == INP_TEXT && t->lic)
		printf("[%d]", t->lic);
	if (t->itype_minor != INP_NO_MINOR)
		printf(" (%s)", inp_others[t->itype_minor]);
	if (t->itype == INP_TA) {
		const char *rows = attribVal(t, "rows");
		const char *cols = attribVal(t, "cols");
		const char *wrap = attribVal(t, "wrap");
		if (rows && cols) {
			printf("[%s×%s", rows, cols);
			if (wrap && stringEqualCI(wrap, "virtual"))
				i_printf(MSG_Recommended);
			i_printf(MSG_Close);
		}
	}			/* text area */
	if (t->rdonly)
		printf(" readonly");
	if (t->name)
		printf(" %s", t->name);
	nl();

	if(t->ninp && t->itype == INP_TEXT)
		t = tagList[t->ninp];
	else if (t->itype != INP_SELECT)
		return;

/* display the options in a pick list */
/* If a search string is given, display the options containing that string. */
	cnt = 0;
	show = false;
	for (v = cw->optlist; v; v = v->same) {
		if (v->controller != t)
			continue;
		if (!v->textval)
			continue;
		++cnt;
		if (*search && !strcasestr(v->textval, search))
			continue;
		if(v->custom_h)
			printf("    %s\n", v->custom_h);
		show = true;
		printf("%3d %s\n", cnt, v->textval);
	}
	if (!show) {
		if (!*search)
			i_puts(MSG_NoOptions);
		else
			i_printf(MSG_NoOptionsMatch, search);
	}
}

static bool inputDisabled(const Tag *t)
{
	if (allowJS && t->jslink)
		return get_property_bool_t(t, "disabled");
	return t->disabled;
}

/*********************************************************************
Update an input field in the current edbrowse buffer.
This can be done for one of two reasons.
First, the user has interactively entered a value in the form, such as
	i=foobar
In this case fromForm will be set to true.
I need to find the tag in the current buffer.
He just modified it, so it ought to be there.
If it isn't there, print an error and do nothing.
The second case: the value has been changed by form reset,
either the user has pushed the reset button or javascript has called form.reset.
Here fromForm is false.
I'm not sure why js would reset a form before the page was even rendered;
that's the only way the line should not be found,
or perhaps if that section of the web page was deleted.
notify = true causes the line to be printed after the change is made.
Notify true and fromForm false is impossible.
You don't need to be notified as each variable is changed during a reset.
The new line replaces the old, and the old is freed.
This works because undo is disabled in browse mode.
*********************************************************************/

static void
updateFieldInBuffer(int tagno, const char *newtext, bool notify, bool fromForm)
{
	int ln, n, plen;
	char *p, *s, *t, *new;

	if (locateTagInBuffer(tagno, &ln, &p, &s, &t)) {
		n = (plen = pstLength((pst) p)) + strlen(newtext) - (t - s);
		new = allocMem(n);
		memcpy(new, p, s - p);
		strcpy(new + (s - p), newtext);
		memcpy(new + strlen(new), t, plen - (t - p));
		free(cw->map[ln].text);
		cw->map[ln].text = (pst) new;
		if (notify && debugLevel> 0)
			displayLine(ln);
		return;
	}

	if (fromForm)
		i_printf(MSG_NoTagFound, tagno, newtext);
}

/* Update an input field. */
bool infReplace(int tagno, const char *newtext, bool notify)
{
	Tag *t = tagList[tagno];
	const Tag *v;
	const Tag *form = t->controller;
	char *display = 0;
	int itype = t->itype;
	int itype_minor = t->itype_minor;
	int newlen = strlen(newtext);

/* sanity checks on the input */
	if (itype <= INP_SUBMIT) {
		int b = MSG_IsButton;
		if (itype == INP_SUBMIT || itype == INP_IMAGE)
			b = MSG_SubmitButton;
		if (itype == INP_RESET)
			b = MSG_ResetButton;
		setError(b);
		return false;
	}

	if (itype == INP_TA) {
		setError((t->lic ? MSG_Textarea : MSG_Textarea0), t->lic);
		return false;
	}

	if(allowJS && t->jslink)
		t->rdonly = get_property_bool_t(t, "readonly");
	if (t->rdonly) {
		setError(MSG_Readonly);
		return false;
	}
	if (inputDisabled(t)) {
		setError(MSG_Disabled);
		return false;
	}

	if (strchr(newtext, '\n')) {
		setError(MSG_InputNewline);
		return false;
	}

	if (itype >= INP_RADIO) {
		if ((newtext[0] != '+' && newtext[0] != '-') || newtext[1]) {
			setError(MSG_InputRadio);
			return false;
		}
		if (itype == INP_RADIO && newtext[0] == '-') {
			setError(MSG_ClearRadio);
			return false;
		}
	}

	if (itype == INP_SELECT) {
		if (!locateOptions(t, newtext, 0, 0, false))
			return false;
		locateOptions(t, newtext, &display, 0, false);
		updateFieldInBuffer(tagno, display, notify, true);
	}

	if (itype == INP_FILE) {
		int u_l;
		char *u = initString(&u_l);
		if(!newtext[0]) { // empty
			updateFieldInBuffer(tagno, newtext, notify, true);
			return true;
		}
		if(!t->multiple) {
			if (!envFile(newtext, &newtext))
				return false;
			if (newtext[0] && (access(newtext, 4) || fileTypeByName(newtext, false) != 'f')) {
				setError(MSG_FileAccess, newtext);
				return false;
			}
			u = cloneString(newtext);
		} else {
			const char *v = newtext, *w, *z2;
			char *z;
			while(*v) {
				if(!(w = strchr(v, selsep)))
					w = v + strlen(v);
				z = pullString(v, w-v);
				v = *w ? w+1 : w; // point to next file
				if(!*z) { // empty
					nzFree(z);
					continue;
				}
				if (!envFile(z, &z2)) {
					nzFree(z), nzFree(u);
					return false;
				}
				if (z2[0] && (access(z2, 4) || fileTypeByName(z2, false) != 'f')) {
					setError(MSG_FileAccess, z2);
					nzFree(z), nzFree(u);
					return false;
				}
				if(*u)
					stringAndChar(&u, &u_l, selsep);
				stringAndString(&u, &u_l, z2);
				nzFree(z);
			}
		}
		updateFieldInBuffer(tagno, u, notify, true);
		nzFree(u);
		return true;
	}

	if(itype == INP_TEXT) {
		connectDatalist(t);
		if (t->ninp) { // the suggested select
// this is a strange puppy.
// ` to override
			if(newtext[0] == '`') {
				++newtext, --newlen;
			} else {
				const Tag *options = tagList[t->ninp];
				if (!locateOptions(options, newtext, 0, 0, false))
					return false;
				locateOptions(options, newtext, &display, 0, false);
				newtext = display;
				newlen = strlen(newtext);
			}
		}

		if (t->lic && newlen > t->lic) {
			setError(MSG_InputLong, t->lic);
			goto fail;
		}

		if (itype_minor == INP_NUMBER && (*newtext && stringIsNum(newtext) < 0)) {
			setError(MSG_NumberExpected);
			goto fail;
		}

		if (itype_minor == INP_EMAIL && (*newtext && !isEmailAddress(newtext))) {
			setError(MSG_EmailInput);
			goto fail;
		}

		if (itype_minor == INP_URL && (*newtext && !isURL(newtext))) {
			setError(MSG_UrlInput);
			goto fail;
		}
	}

	if (itype == INP_RADIO && form && t->name && *newtext == '+') {
/* clear the other radio button */
		for (v = cw->inputlist; v; v = v->same) {
			if (v->controller != form)
				continue;
			if (v->itype != INP_RADIO)
				continue;
			if (!v->name)
				continue;
			if (!stringEqual(v->name, t->name))
				continue;
			if (fieldIsChecked(v->seqno) == true)
				updateFieldInBuffer(v->seqno, "-", false, true);
		}
	}

	if (itype != INP_SELECT) {
		updateFieldInBuffer(tagno, newtext, notify, true);
	}

	if (itype >= INP_TEXT) {
		jSyncup(false, t);
		cf = t->f0;
		if (itype >= INP_RADIO) {
// The change has already been made;
// if onclick returns false, should that have prevented the change??
			bubble_event_t(t, "onclick");
			if (js_redirects)
				return true;
		}
		if (itype != INP_SELECT)
			bubble_event_t(t, "oninput");
		if (js_redirects)
			goto success;
		bubble_event_t(t, "onchange");
		if (js_redirects)
			goto success;
		jSideEffects();
	}

success:
	nzFree(display);
	return true;

fail:
	nzFree(display);
	return false;
}

/*********************************************************************
Reset or submit a form.
This function could be called by javascript, as well as a human.
It must therefore update the js variables and the text simultaneously.
Most of this work is done by resetVar().
To reset a variable, copy its original value, in the html tag,
back to the text buffer, and over to javascript.
*********************************************************************/

static void resetVar(Tag *t)
{
	int itype = t->itype;
	const char *w = t->rvalue;
	bool bval;

/* This is a kludge - option looks like INP_SELECT */
	if (t->action == TAGACT_OPTION)
		itype = INP_SELECT;

	if (itype <= INP_SUBMIT)
		return;

	if (itype >= INP_SELECT && itype != INP_TA) {
		bval = t->rchecked;
		t->checked = bval;
		w = bval ? "+" : "-";
	}

	if (itype == INP_TA) {
		int cx = t->lic;
		if (cx)
			sideBuffer(cx, w, -1, 0);
	} else if (itype != INP_HIDDEN && itype != INP_SELECT)
		updateFieldInBuffer(t->seqno, w, false, false);

	if ((itype >= INP_TEXT && itype <= INP_FILE) || itype == INP_TA) {
		nzFree(t->value);
		t->value = cloneString(t->rvalue);
	}

	if (!t->jslink || !allowJS)
		return;

	if (itype >= INP_RADIO) {
		set_property_bool_t(t, "checked", bval);
	} else if (itype == INP_SELECT) {
/* remember this means option */
		set_property_bool_t(t, "selected", bval);
		if (bval && !t->controller->multiple && t->controller->jslink)
			set_property_number_t(t->controller,
					    "selectedIndex", t->lic);
	} else
		set_property_string_t(t, "value", w);
}

static void formReset(const Tag *form)
{
	Tag *t;
	int i, itype;
	char *display;

	rebuildSelectors();

	for (i = 0; i < cw->numTags; ++i) {
		t = tagList[i];
		if (t->action == TAGACT_OPTION) {
			resetVar(t);
			continue;
		}

		if (t->action != TAGACT_INPUT)
			continue;
		if (t->controller != form)
			continue;
		itype = t->itype;
		if (itype != INP_SELECT) {
			resetVar(t);
			continue;
		}
		if (t->jslink && allowJS)
			set_property_number_t(t, "selectedIndex", -1);
	}			/* loop over tags */

/* loop again to look for select, now that options are set */
	for (t = cw->inputlist; t; t = t->same) {
		if (t->controller != form)
			continue;
		itype = t->itype;
		if (itype != INP_SELECT)
			continue;
		display = displayOptions(t);
		updateFieldInBuffer(t->seqno, display, false, false);
		nzFree(t->value);
		t->value = display;
/* this should now be the same as t->rvalue, but I guess I'm
 * not going to check for that, or take advantage of it. */
	}			/* loop over tags */

	i_puts(MSG_FormReset);
}

/* Fetch a field value (from a form) to post. */
/* The result is allocated */
static char *fetchTextVar(const Tag *t)
{
	char *v;

// js must not muck with the value of a file field
	if (t->itype != INP_FILE) {
		if (t->jslink && allowJS)
			return get_property_string_t(t, "value");
	}

	if (t->itype > INP_HIDDEN) {
		v = getFieldFromBuffer(t->seqno);
		if (v)
			return v;
	}

/* Revert to the default value */
	return cloneString(t->value);
}

static bool fetchBoolVar(const Tag *t)
{
	int checked;

	if (t->jslink && allowJS)
		return get_property_bool_t(t,
					 (t->action == TAGACT_OPTION ? "selected" : "checked"));

	checked = fieldIsChecked(t->seqno);
	if (checked < 0)
		checked = t->rchecked;
	return checked;
}

/* Some information on posting forms can be found here.
 * http://www.w3.org/TR/REC-html40/interact/forms.html */

static char *pfs;		/* post form string */
static int pfs_l;
static const char *boundary;

static void postDelimiter(char fsep)
{
	char c = pfs[strlen(pfs) - 1];
	if (c == '?' || c == '\1')
		return;
	if (fsep == '-') {
		stringAndString(&pfs, &pfs_l, "--");
		stringAndString(&pfs, &pfs_l, boundary);
		stringAndChar(&pfs, &pfs_l, '\r');
		fsep = '\n';
	}
	stringAndChar(&pfs, &pfs_l, fsep);
}

static bool
postNameVal(const char *name, const char *val, char fsep, uchar isfile)
{
	char *enc;
	const char *ct, *ce;	/* content type, content encoding */
	const char *cut;

	if (!val)
		val = emptyString;

	if(fsep)
		postDelimiter(fsep);

	switch (fsep) {
	case '&':
		enc = encodePostData(name, NULL);
		stringAndString(&pfs, &pfs_l, enc);
		nzFree(enc);
		stringAndChar(&pfs, &pfs_l, '=');
		break;

	case 0:
		stringAndString(&pfs, &pfs_l, name);
		stringAndChar(&pfs, &pfs_l, '=');
		break;

	case '\n':
		stringAndString(&pfs, &pfs_l, name);
		stringAndString(&pfs, &pfs_l, "=\r\n");
		break;

	case '-':
		stringAndString(&pfs, &pfs_l,
				"Content-Disposition: form-data; name=\"");
		stringAndString(&pfs, &pfs_l, name);
		stringAndChar(&pfs, &pfs_l, '"');
/* I'm leaving nl off, in case we need ; filename */
		break;
	}			/* switch */

	if (!*val && fsep == '&')
		return true;

	switch (fsep) {
	case '&':
		enc = encodePostData(val, NULL);
		stringAndString(&pfs, &pfs_l, enc);
		nzFree(enc);
		break;

	case '\n': case 0:
		stringAndString(&pfs, &pfs_l, val);
		stringAndString(&pfs, &pfs_l, eol);
		break;

	case '-':
		if (isfile) {
			if (isfile & 2) {
				stringAndString(&pfs, &pfs_l, "; filename=\"");
// only show the filename
				cut = strrchr(val, '/');
				stringAndString(&pfs, &pfs_l, (cut ? cut + 1 : val));
				stringAndChar(&pfs, &pfs_l, '"');
			}
			if (!encodeAttachment(val, 0, true, &ct, &ce, &enc))
				return false;
			if(!(isfile&2))
				enc = makeDosNewlines(enc);
/* remember to free enc in this case */
			val = enc;
		} else {
			const char *s;
			ct = "text/plain";
/* Anything nonascii makes it 8bit */
			ce = "7bit";
			for (s = val; *s; ++s)
				if (*s & 0x80) {
					ce = "8bit";
					break;
				}
		}
		if(!stringEqual(ct, "text/plain")) {
			stringAndString(&pfs, &pfs_l, "\r\nContent-Type: ");
			stringAndString(&pfs, &pfs_l, ct);
		}
		stringAndString(&pfs, &pfs_l, "\r\nContent-Transfer-Encoding: ");
		stringAndString(&pfs, &pfs_l, ce);
		stringAndString(&pfs, &pfs_l, "\r\n\r\n");
		stringAndString(&pfs, &pfs_l, val);
		stringAndString(&pfs, &pfs_l, eol);
		if (isfile)
			nzFree(enc);
		break;
	}			/* switch */

	return true;
}

static bool formSubmit(const Tag *form, const Tag *submit, bool dopost)
{
	const Tag *t;
	int j, itype;
	char *name, *dynamicvalue = NULL;
/* dynamicvalue needs to be freed with nzFree. */
	const char *value;
	char fsep = '&';	/* field separator */
	bool rc;
	bool bval;
	const char *eo1; // enctype override from attribute
	char *eo2; // enctype override from js

/* js could rebuild an option list then submit the form. */
	rebuildSelectors();

	if (form->bymail)
		fsep = '\n';

// if method is not post, these other encodings are not honored
	if(!dopost)
		goto skip_encode;

	if (form->mime)
		fsep = '-';
	if(form->plain)
		fsep = 0;

// <input enctype=blah> can override
	if(submit) {
		eo1 = attribVal(submit, "formenctype"), eo2 = 0;
		if(submit->jslink && allowJS)
			eo2 = get_property_string_t(submit, "formenctype");
		if(eo2 && *eo2)
			eo1 = eo2;
		if(eo1 && *eo1) {
			fsep = '&';
			if (stringEqualCI(eo1, "multipart/form-data"))
				fsep = '-';
			else if (stringEqualCI(eo1, "text/plain"))
				fsep = 0;
			else if (!stringEqualCI(eo1, 
				  "application/x-www-form-urlencoded"))
				debugPrint(3,
					   "unrecognized enctype, plese use multipart/form-data or application/x-www-form-urlencoded or text/plain");
		}
		nzFree(eo2);
	}

	if (fsep == '-') {
		boundary = makeBoundary();
		stringAndString(&pfs, &pfs_l, "`mfd~");
		stringAndString(&pfs, &pfs_l, boundary);
		stringAndString(&pfs, &pfs_l, eol);
	}

skip_encode:

	for (t = cw->inputlist; t; t = t->same) {
		if (t->controller != form)
			continue;
		itype = t->itype;
		if (itype <= INP_SUBMIT && t != submit)
			continue;
		if (inputDisabled(t))
			continue;
		name = t->name;
		if (!name || !*name)
			continue;

		if (t == submit) {	/* the submit button you pushed */
			int namelen;
			char *nx;
			value = t->value;
			if (!value || !*value)
				value = "Submit";
			if (t->itype != INP_IMAGE)
				goto success;
			namelen = strlen(name);
			nx = (char *)allocMem(namelen + 3);
			strcpy(nx, name);
			strcpy(nx + namelen, ".x");
			postNameVal(nx, "0", fsep, false);
			nx[namelen + 1] = 'y';
			postNameVal(nx, "0", fsep, false);
			nzFree(nx);
			goto success;
		}

		if (itype >= INP_RADIO) {
			value = t->value;
			bval = fetchBoolVar(t);
			if (!bval)
				continue;
			if (!name[0])
				if (value && !*value)
					value = 0;
			if (itype == INP_CHECKBOX && value == 0)
				value = "on";
			goto success;
		}

// special case for charset
		if(itype == INP_HIDDEN && stringEqualCI(name, "_charset_")) {
			value = (cf->charset ? cf->charset : "UTF-8");
			eo1 = attribVal(form, "accept-charset"), eo2 = 0;
			if(form->jslink && allowJS)
				eo2 = get_property_string_t(form, "accept-charset");
			if(eo2 && *eo2)
				eo1 = eo2;
			if(eo1 && *eo1)
				value = eo1;
			postNameVal(name, value, fsep, false);
			nzFree(eo2);
			continue;
		}

		if (itype < INP_FILE) {
/* Even a hidden variable can be adjusted by js.
 * fetchTextVar allows for this possibility.
 * I didn't allow for it in the above, the value of a radio button;
 * hope that's not a problem. */
			dynamicvalue = fetchTextVar(t);
			postNameVal(name, dynamicvalue, fsep, false);
			if(t->required && !dynamicvalue && !*dynamicvalue)
				goto required;
			nzFree(dynamicvalue);
			dynamicvalue = NULL;
			continue;
		}

		if (itype == INP_TA) {
			int cx = t->lic;
			char *cxbuf;
			int cxlen;
			if (cx) {
				if (fsep == '-') {
					char cxstring[12];
/* do this as an attachment */
					sprintf(cxstring, "%d", cx);
					if (!postNameVal
					    (name, cxstring, fsep, 1))
						goto fail;
					continue;
				}	/* attach */
				if (!unfoldBuffer(cx, true, &cxbuf, &cxlen))
					goto fail;
				for (j = 0; j < cxlen; ++j)
					if (cxbuf[j] == 0) {
						setError(MSG_SessionNull, cx);
						nzFree(cxbuf);
						goto fail;
					}
				if (j && cxbuf[j - 1] == '\n')
					--j;
				if (j && cxbuf[j - 1] == '\r')
					--j;
				cxbuf[j] = 0;
				rc = postNameVal(name, cxbuf, fsep, false);
				nzFree(cxbuf);
				if (!rc)
					goto fail;
				if(t->required && !j)
					goto required;
				continue;
			}

			postNameVal(name, 0, fsep, false);
			if(t->required)
				goto required;
			continue;
		}

		if (itype == INP_SELECT) {
			char *display = getFieldFromBuffer(t->seqno);
			char *s, *e;
			if (!display) {	/* off the air */
				Tag *v;
/* revert back to reset state */
				for (v = cw->optlist; v; v = v->same)
					if (v->controller == t)
						v->checked = v->rchecked;
				display = displayOptions(t);
			}
			rc = locateOptions(t, display, 0, &dynamicvalue, false);
			nzFree(display);
			if (!rc)
				goto fail;	/* this should never happen */
/* option could have an empty value, usually the null choice,
 * before you have made a selection. */
			if (!*dynamicvalue) {
				if(t->required)
					goto required;
				if (!t->multiple)
					postNameVal(name, dynamicvalue, fsep, false);
				continue;
			}
/* Step through the options */
			for (s = dynamicvalue; *s; s = e) {
				char more;
				e = 0;
				if (t->multiple)
					e = strchr(s, '\1');
				if (!e)
					e = s + strlen(s);
				more = *e, *e = 0;
				postNameVal(name, s, fsep, false);
				if (more)
					++e;
			}
			nzFree(dynamicvalue);
			dynamicvalue = NULL;
			continue;
		}

		if (itype == INP_FILE) {	/* the only one left */
			uchar isfile = 3;
			dynamicvalue = fetchTextVar(t);
			if (!dynamicvalue || !*dynamicvalue) {
				if(t->required)
					goto required;
				postNameVal(name, emptyString, fsep, 0);
				continue;
			}
			if (!dopost  || fsep != '-') {
				if(fsep == '\n') {
					setError(MSG_FilePost);
					nzFree(dynamicvalue);
					goto fail;
				}
// we'll try to truck along
				isfile = 0;
				if(debugLevel >= 3)
					i_puts(MSG_FilePost);
			}

			if(!t->multiple) {
				rc = postNameVal(name, dynamicvalue, fsep, isfile);
				nzFree(dynamicvalue);
				dynamicvalue = NULL;
				if (!rc)
					goto fail;
			} else {
				const char *v = dynamicvalue, *w;
				char *z;

				while(*v) {
					if(!(w = strchr(v, selsep)))
						w = v + strlen(v);
					z = pullString(v, w-v);
					v = *w ? w+1 : w; // point to next file
					if(!*z) { // empty
						nzFree(z);
						continue;
					}
					rc = postNameVal(name, z, fsep, isfile);
					nzFree(z);
					if (!rc)
						goto fail;
				}

				nzFree(dynamicvalue);
			}
			continue;
		}

		i_printfExit(MSG_UnexSubmitForm);

success:
		postNameVal(name, value, fsep, false);
	}			/* loop over tags */

	if (fsep == '-') {	// the last boundary
		stringAndString(&pfs, &pfs_l, "--");
		stringAndString(&pfs, &pfs_l, boundary);
		stringAndString(&pfs, &pfs_l, "--\r\n");
	}

	if(debugLevel >= 1)
		i_puts(MSG_FormSubmit);
	return true;

fail:
	return false;

required:
	setError(MSG_ReqField, name);
	return false;
}

/*********************************************************************
Push the reset or submit button.
This routine must be reentrant.
You push submit, which calls this routine, which runs the onsubmit code,
which checks the fields and calls form.submit(),
which calls this routine.  Happens all the time.
*********************************************************************/

/* jSyncup has been called before we enter this function */
bool infPush(int tagno, char **post_string)
{
	Tag *t = tagList[tagno];
	Frame *f = t->f0;
	Tag *form;
	int itype;
	int actlen;
	const char *action = 0;
	char *action2 = 0; // allocated action
	char *section;
	const char *prot;
	bool rc, dopost;

	*post_string = 0;

/* If the tag is actually a form, then infPush() was invoked
 * by form.submit().
 * Revert t back to 0, since there may be multiple submit buttons
 * on the form, and we don't know which one was pushed. */
	if (t->action == TAGACT_FORM) {
		form = t;
		t = 0;
		itype = INP_SUBMIT;
	} else {
		form = t->controller;
		itype = t->itype;
	}

	if (itype > INP_SUBMIT) {
		setError(MSG_NoButton);
		return false;
	}

	if (t) {
		if (inputDisabled(t)) {
			setError(MSG_Disabled);
			return false;
		}
		if (tagHandler(t->seqno, "onclick") && !allowJS)
			runningError(itype ==
				     INP_BUTTON ? MSG_NJNoAction :
				     MSG_NJNoOnclick);
		bubble_event_t(t, "onclick");
		if (js_redirects)
			return true;
// At this point onclick has run, be it button or submit or reset
	}

	if (itype == INP_BUTTON) {
/* I use to error here, but click could be captured by a node higher up in the tree
   and do what it is suppose to do, so we might not want an error here.
		if (allowJS && t->jslink && !t->onclick) {
			setError(MSG_ButtonNoJS);
			return false;
		}
*/
		return true;
	}
// Now submit or reset
	if (itype == INP_RESET) {
		if (!form) {
			setError(MSG_NotInForm);
			return false;
		}
// Before we reset, run the onreset code.
// I read somewhere that onreset and onsubmit only run if you
// pushed the button - rather like onclick.
// Thus t, the reset button, must be nonzero.
		if (t && tagHandler(form->seqno, "onreset")) {
			if (!allowJS)
				runningError(MSG_NJNoReset);
			else {
				rc = true;
				if (form->jslink)
					rc = run_event_t(form, "form", "onreset");
				if (!rc)
					return true;
				if (js_redirects)
					return true;
			}
		}		/* onreset */
		formReset(form);
		return true;
	}
// now it's submit
	if (!form && !(t && t->onclick)) {
		setError(MSG_NotInForm);
		return false;
	}
// <button> could turn into submit, which we don't want to do if it is not in a form.
	if (!form)
		return true;
	// Before we submit, run the onsubmit code
	if (t && tagHandler(form->seqno, "onsubmit")) {
		if (!allowJS)
			runningError(MSG_NJNoSubmit);
		else {
			rc = true;
			if (form->jslink)
				rc = bubble_event_t(form, "onsubmit");
			if (!rc)
				return true;
			if (js_redirects)
				return true;
		}
	}

	dopost = form->post;
	action = form->href;
/* But we defer to the js variable */
	if (form->jslink && allowJS) {
		char *jh = get_property_url_t(form, true);
		if (jh && (!action || !stringEqual(jh, action))) {
			nzFree(form->href);
			action = form->href = jh;
			jh = NULL;
		}
		nzFree(jh);
	}

	if(t) { // submit button pressed
		const char *va; // value from attribute
		char *vj; // value from javascript
// spec says formAction and formMethod are camelcase, when coming from js.
// Are they lowercase when coming from html? or case insensitive?
		va = attribVal(t, "formmethod");
		vj = 0;
		if(t->jslink && allowJS)
			vj = get_property_string_t(t, "formMethod");
		if(vj && *vj)
			va = vj;
		if(va && *va) {
			dopost = false;
			if (stringEqualCI(va, "post"))
				dopost = true;
			else if (!stringEqualCI(va, "get"))
				debugPrint(3, "unrecognized method, please use get or post");
		}
		nzFree(vj);
		va = attribVal(t, "formaction");
		if(t->jslink && allowJS)
			action2 = vj = get_property_string_t(t, "formAction");
		if(vj && *vj)
			va = vj;
		if(va && *va)
			action = va;
	}

// if no action, or action is "#", the default is the current location.
// And yet, with onclick on the submit button, no action means no action,
// and I believe the same is true for onsubmit.
// Just assume javascript has done the submit.
	if (!action || !*action || stringEqual(action, "#")) {
		if (t && (t->onclick | form->onsubmit))
			goto success;
		action = f->hbase;
	}

	prot = getProtURL(action);
	if (!prot) {
		if (t && t->onclick)
			goto success;
		setError(MSG_FormBadURL);
fail:
		nzFree(action2);
		return false;
	}

	debugPrint(2, "* %s", action);

	if (stringEqualCI(prot, "javascript")) {
		if (!allowJS) {
			setError(MSG_NJNoForm);
			goto fail;
		}
		jsRunScript_t(form, action, 0, 0);
		goto success;
	}

	form->bymail = false;
	if (stringEqualCI(prot, "mailto")) {
		if (!validAccount(localAccount))
			goto fail;
		form->bymail = true;
	} else if (stringEqualCI(prot, "http")) {
		if (form->secure) {
			setError(MSG_BecameInsecure);
			goto fail;
		}
	} else if (!stringEqualCI(prot, "https") &&
		   !stringEqualCI(prot, "gopher")) {
		setError(MSG_SubmitProtBad, prot);
		goto fail;
	}

	pfs = initString(&pfs_l);
	stringAndString(&pfs, &pfs_l, action);
	section = findHash(pfs);
	if (section) {
		i_printf(MSG_SectionIgnored, section);
		*section = 0;
		pfs_l = section - pfs;
	}
	section = strpbrk(pfs, "?\1");
	if (section && (*section == '\1' || !(form->bymail | dopost))) {
		debugPrint(3,
			   "the url already specifies some data, which will be overwritten by the data in this form");
		*section = 0;
		pfs_l = section - pfs;
	}

	stringAndChar(&pfs, &pfs_l, (dopost ? '\1' : '?'));
	actlen = strlen(pfs);

	if (!formSubmit(form, t, dopost)) {
		nzFree(pfs);
		goto fail;
	}

	debugPrint(3, "%s %s", dopost ? "post" : "get", pfs + actlen);

/* Handle the mail method here and now. */
	if (form->bymail) {
		char *addr, *subj, *q;
		const char *tolist[MAXCC + 2], *atlist[MAXCC + 2];
		const char *name = form->name;
		int newlen = strlen(pfs) - actlen;	/* the new string could be longer than post */
		char key;
		decodeMailURL(action, &addr, &subj, 0);
		tolist[0] = addr;
		tolist[1] = 0;
		atlist[0] = 0;
		newlen += 9;	/* subject: \n */
		if (subj)
			newlen += strlen(subj);
		else
			newlen += 11 + (name ? strlen(name) : 1);
		++newlen;	/* null */
		++newlen;	/* encodeAttachment might append another nl */
		q = (char *)allocMem(newlen);
		if (subj)
			sprintf(q, "subject:%s\n", subj);
		else
			sprintf(q, "subject:html form(%s)\n",
				name ? name : "?");
		strcpy(q + strlen(q), pfs + actlen);
		nzFree(pfs);
		printf("sending mail to %s", addr);
		if(subj)
			printf(", subject %s", subj);
		printf(", is this ok? ");
		fflush(stdout);
		key = getLetter("ynYN");
		puts("");
		if(key == 'y' || key == 'Y') {
			rc = sendMail(localAccount, tolist, q, -1, atlist, 0, 0, false);
			if (rc)
				i_puts(MSG_MailSent);
		} else
			rc = true;
		nzFree(addr);
		nzFree(subj);
		nzFree(q);
		nzFree(action2);
		*post_string = 0;
		return rc;
	}
// gopher submit is one input field with no name;
// the leading = doesn't belong.
	if (pfs[actlen] == '=' && stringEqualCI(prot, "gopher"))
		strmove(pfs + actlen, pfs + actlen + 1);

	*post_string = pfs;
success:
	nzFree(action2);
	return true;
}

void domSubmitsForm(Tag *t, bool reset)
{
	if (reset)
		js_reset = t;
	else
		js_submit = t;
}

void domSetsTagValue(Tag *t, const char *newtext)
{
	if (t->itype == INP_HIDDEN || t->itype == INP_RADIO
	    || t->itype == INP_FILE)
		return;
	if (t->itype == INP_TA) {
		int side = t->lic;
		if (side <= 0 || side >= MAXSESSION || side == context)
			return;
		if (sessionList[side].lw == NULL)
			return;
		if (cw->browseMode)
			i_printf(MSG_BufferUpdated, side);
		sideBuffer(side, newtext, -1, 0);
		return;
	}
	nzFree(t->value);
	t->value = cloneString(newtext);
}

bool charInOptions(char c)
{
	const Window *w;
	const Tag *t;
	int i;
	for(i = 0; i < MAXSESSION; ++i) {
		for(w = sessionList[i].lw; w; w = w->prev) {
			if(!w->browseMode)
				continue;
			for (t = w->optlist; t; t = t->same)
				if(t->textval && strchr(t->textval, c) &&
				t->controller && t->controller->multiple)
					return true;
		}
	}
	return false;
}

void charFixOptions(char c)
{
	Window *w, *save_w = cw;
	Tag *t;
	int i, j;
	char *u;
	for(i = 0; i < MAXSESSION; ++i) {
		for(w = sessionList[i].lw; w; w = w->prev) {
			if(!w->browseMode)
				continue;
			for(j = 0; j < w->numTags; ++j) {
				t = w->tags[j];
				if(t->action != TAGACT_INPUT || !t->value ||
				t->itype != INP_SELECT || !t->multiple)
					continue;
				for(u = t->value; *u; ++u)
					if(*u == c)
						*u = selsep;
			}
			cw = w, rerender(cw == save_w ? 0 : -1);
		}
	}
	cw = save_w;
}

bool charInFiles(char c)
{
	const Window *w;
	const Tag *t;
	int i, j;
	char *u;
	for(i = 0; i < MAXSESSION; ++i) {
		for(w = sessionList[i].lw; w; w = w->prev) {
			if(!w->browseMode)
				continue;
			for(j = 0; j < w->numTags; ++j) {
				t = w->tags[j];
				if(t->action != TAGACT_INPUT ||
				t->itype != INP_FILE || !t->multiple ||
				!(u = fetchTextVar(t)))
					continue;
				if(strchr(u, c)) {
					nzFree(u);
					return true;
				}
				nzFree(u);
			}
		}
	}
	return false;
}

void charFixFiles(char c)
{
	Window *w, *save_w = cw;
	Tag *t;
	int i, j;
	char *u, *v;
	for(i = 0; i < MAXSESSION; ++i) {
		for(w = sessionList[i].lw; w; w = w->prev) {
			if(!w->browseMode)
				continue;
			cw = w;
			for(j = 0; j < w->numTags; ++j) {
				t = w->tags[j];
				if(t->action != TAGACT_INPUT ||
				t->itype != INP_FILE || !t->multiple ||
				!(u = fetchTextVar(t)))
					continue;
				for(v = u; *v; ++v)
					if(*v == c)
						*v = selsep;
				updateFieldInBuffer(t->seqno, u, false, false);
				nzFree(t->value);
				t->value = u;
			}
		}
	}
	cw = save_w;
}

// getElementById in C.
// Don't scan through all tags; we have to stay within our frame.
static bool anchorlook;
static const char *idsearch;
static const Tag *gebi_r(const Tag *t); // recursive
const Tag *gebi_c(const Tag *t, const char *id, bool lookname)
{
	int action;
	if(!id || !*id)
		return 0;
// this should be easy, but bugs keep popping up.
	debugPrint(4, "search %s", id);
	while (true) {
		action = t->action;
		debugPrint(4, "up %d,%s", t->seqno, t->info->name);
// don't go past document and up to a higher frame
		if(action == TAGACT_HTML || action == TAGACT_FRAME)
			return 0;
		if(action == TAGACT_BODY)
			break;
		if(!(t = t->parent))
			return 0; // should never happen
	}
// t is <body> at the top of the current frame
	idsearch = id;
	anchorlook = lookname;
	return gebi_r(t);
}

static const Tag *gebi_r(const Tag *t)
{
	const Tag *c; // children
	const Tag *u;
	debugPrint(4, "look %d,%s,%s", t->seqno, t->info->name, t->id);
	if(t->id && stringEqual(t->id, idsearch))
		return t;
	if (anchorlook && t->action == TAGACT_A &&
	t->name && stringEqual(t->name, idsearch))
		return t;
// do not descend into a new frame
	if(t->action == TAGACT_FRAME)
		return 0;
// look through children
	for (c = t->firstchild; c; c = c->sibling)
		if((u = gebi_r(c)))
			return u;
	return 0;
}

/* Javascript errors, we need to see these no matter what. */
void runningError(int msg, ...)
{
	va_list p;
	if (ismc)
		return;
	if (debugLevel <= 2)
		return;
	va_start(p, msg);
	vprintf(i_getString(msg), p);
	va_end(p);
	nl();
}

/*********************************************************************
Diff the old screen with the new rendered screen.
This is a simple front back diff algorithm.
Compare the two strings from the start, how many lines are the same.
Compare the two strings from the back, how many lines are the same.
That zeros in on the line that has changed.
Most of the time one line has changed,
or a couple of adjacent lines, or a couple of nearby lines.
So this should do it.
sameFront counts the lines from the top that are the same.
We're here because the buffers are different, so sameFront will not equal $.
Lines after sameFront are different.
Lines past sameBack1 and same back2 are the same to the bottom in the two buffers.
To be a bit more sophisticated, front1z and front2z
become nonzero if just one line was added, updated, or deleted at sameFront.
they march on beyond this point, as long as lines are the same.
In the same way, back1z and back2z march backwards
past a one line anomaly.
*********************************************************************/

static int sameFront, sameBack1, sameBack2;
static int front1z, front2z, back1z, back2z;
static const char *newChunkStart, *newChunkEnd;

// need a reverse strchr to help us out.
static const char *rstrchr(const char *s, const char *mark)
{
	for (--s; s > mark; --s)
		if (s[-1] == '\n')
			return s;
	return (s == mark ? s : NULL);
}

static void frontBackDiff(const char *b1, const char *b2)
{
	const char *f1, *f2, *s1, *s2, *e1, *e2;
	const char *g1, *g2, *h1, *h2;

	sameFront = front1z = front2z = 0;
	back1z = back2z = 0;

	s1 = b1, s2 = b2;
	f1 = b1, f2 = b2;
	while (*s1 == *s2 && *s1) {
		if (*s1 == '\n') {
			f1 = s1 + 1, f2 = s2 + 1;
			++sameFront;
		}
		++s1, ++s2;
	}

	g1 = strchr(f1, '\n');
	g2 = strchr(f2, '\n');
	if (g1 && g2) {
		++g1, ++g2;
		h1 = strchr(g1, '\n');
		h2 = strchr(g2, '\n');
		if (h1 && h2) {
			++h1, ++h2;
			if (g1 - f1 == h2 - g2 && !memcmp(f1, g2, g1 - f1)) {
				e1 = f1, e2 = g2;
				s1 = g1, s2 = h2;
				front1z = sameFront + 1;
				front2z = sameFront + 2;
			} else if (h1 - g1 == g2 - f2
				   && !memcmp(g1, f2, h1 - g1)) {
				e1 = g1, e2 = f2;
				s1 = h1, s2 = g2;
				front1z = sameFront + 2;
				front2z = sameFront + 1;
			} else if (h1 - g1 == h2 - g2
				   && !memcmp(g1, g2, h1 - g1)) {
				e1 = g1, e2 = g2;
				s1 = h1, s2 = h2;
				front1z = sameFront + 2;
				front2z = sameFront + 2;
			}
		}
	}

	if (front1z || front2z) {
		sameBack1 = front1z - 1, sameBack2 = front2z - 1;
		while (*s1 == *s2 && *s1) {
			if (*s1 == '\n')
				++front1z, ++front2z;
			++s1, ++s2;
		}
		if (!*s1 && !*s2) {
			front1z = front2z = 0;
			goto done;
		}
	}

	s1 = b1 + strlen(b1);
	s2 = b2 + strlen(b2);
	while (s1 > f1 && s2 > f2 && s1[-1] == s2[-1])
		--s1, --s2;
	if (s1 == f1 && s2[-1] == '\n')
		goto mark_e;
	if (s2 == f2 && s1[-1] == '\n')
		goto mark_e;
/* advance both pointers to newline or null */
	while (*s1 && *s1 != '\n')
		++s1, ++s2;
/* these buffers should always end in newline, so the next if should always be true */
	if (*s1 == '\n')
		++s1, ++s2;
mark_e:
	e1 = s1, e2 = s2;

	sameBack1 = sameFront;
	for (s1 = f1; s1 < e1; ++s1)
		if (*s1 == '\n')
			++sameBack1;
	if (s1 > f1 && s1[-1] != '\n')	// should never happen
		++sameBack1;

	sameBack2 = sameFront;
	for (s2 = f2; s2 < e2; ++s2)
		if (*s2 == '\n')
			++sameBack2;
	if (s2 > f2 && s2[-1] != '\n')	// should never happen
		++sameBack2;

	if (front1z || front2z) {
// front2z can run past sameBack2 if lines are deleted.
// This because front2z is computed before sameBack2.
		while (front1z > sameBack1 || front2z > sameBack2)
			--front1z, --front2z;
		if (front1z <= sameFront || front2z <= sameFront)
			front1z = front2z = 0;
		goto done;
	}

	h1 = rstrchr(e1, f1);
	h2 = rstrchr(e2, f2);
	if (h1 && h2) {
		g1 = rstrchr(h1, f1);
		g2 = rstrchr(h2, f2);
		if (g1 && g2) {
			if (e1 - h1 == h2 - g2 && !memcmp(h1, g2, e1 - h1)) {
				s1 = h1, s2 = g2;
				back1z = sameBack1, back2z = sameBack2 - 1;
			} else if (h1 - g1 == e2 - h2
				   && !memcmp(g1, h2, h1 - g1)) {
				s1 = g1, s2 = h2;
				back1z = sameBack1 - 1, back2z = sameBack2;
			} else if (h1 - g1 == h2 - g2
				   && !memcmp(g1, g2, h1 - g1)) {
				s1 = g1, s2 = g2;
				back1z = sameBack1 - 1, back2z = sameBack2 - 1;
			}
		}
	}

	if (back1z || back2z) {
		--s1, --s2;
		while (*s1 == *s2 && s1 >= f1 && s2 >= f2) {
			if (s1[-1] == '\n' && s2[-1] == '\n')
				--back1z, --back2z;
			--s1, --s2;
		}
	}

done:
	newChunkStart = f2;
	newChunkEnd = e2;
}

// Believe it or not, I have exercised all the pathways in this routine.
// It's rather mind numbing.
static bool reportZ(void)
{
// low and high operations are ad, update, delete
	char oplow, ophigh;
// lines affected in the second group
	int act1, act2;
	int d_start, d_end;

	if (!(front1z || front2z || back1z || back2z))
		return false;

	if (front1z || front2z) {
		if (front2z > front1z)
			oplow = 1;
		if (front2z == front1z)
			oplow = 2;
		if (front2z < front1z)
			oplow = 3;
		act1 = sameBack1 - front1z;
		act2 = sameBack2 - front2z;
		ophigh = 2;
		if (!act1)
			ophigh = 1;
		if (!act2)
			ophigh = 3;
// delete delete is the easy case, but very rare
		if (oplow == 3 && ophigh == 3) {
			if (act1 == 1)
				i_printf(MSG_LineDeleteZ1, sameFront + 1,
					 sameBack1);
			else
				i_printf(MSG_LineDeleteZ2, sameFront + 1,
					 front1z + 1, sameBack1);
			goto done;
		}
// double add is more common, and also unambiguous.
// If this algorithm says we added 100 lines, then we added 100 lines.
		if (oplow == 1 && ophigh == 1) {
			if (act2 == 1)
				i_printf(MSG_LineAddZ1, sameFront + 1,
					 sameBack2);
			else
				i_printf(MSG_LineAddZ2, sameFront + 1,
					 front2z + 1, sameBack2);
			goto done;
		}
		if (oplow == 3) {
// delete mixed with something else, and I just don't care about the delete.
			if (ophigh == 1)
				i_printf(MSG_LineAdd2, front2z + 1, sameBack2);
			else if (act2 <= 10)
				i_printf(MSG_LineUpdate3, front2z + 1,
					 sameBack2);
			else
				i_printf(MSG_LineUpdateRange, front2z + 1,
					 sameBack2);
			goto done;
		}
		if (ophigh == 3) {
// if the deleted block is big then report it, otherwise ignore it.
			if (act1 >= 10)
				i_printf(MSG_LineDelete2, act1, front1z);
			else if (oplow == 1)
				i_printf(MSG_LineAdd1, sameFront + 1);
			else
				i_printf(MSG_LineUpdate1, sameFront + 1);
			goto done;
		}
// a mix of add and update, call it an update.
// If the second group is big then switch to range message.
		if (act2 > 10 && ophigh == 2)
			i_printf(MSG_LineUpdateRange,
				 (front2z - sameFront <
				  10 ? sameFront + 1 : front2z + 1), sameBack2);
		else if (act2 == 1)
			i_printf(MSG_LineUpdateZ1, sameFront + 1, sameBack2);
		else
			i_printf(MSG_LineUpdateZ2, sameFront + 1, front2z + 1,
				 sameBack2);
		goto done;
	}
// At this point the single line change comes second,
// we have to look at back1z and back2z.
	d_start = sameBack2 - sameBack1;
	d_end = back2z - back1z;
	ophigh = 2;
	if (d_end > d_start)
		ophigh = 3;
	if (d_end < d_start)
		ophigh = 1;
	act1 = back1z - sameFront - 1;
	act2 = back2z - sameFront - 1;
	oplow = 2;
	if (!act1)
		oplow = 1;
	if (!act2)
		oplow = 3;
// delete delete is the easy case, but very rare
	if (oplow == 3 && ophigh == 3) {
// act1 should never be 1, because then one line was deleted earlier,
// and we would be in the front1z case.
		i_printf(MSG_LineDeleteZ3, sameFront + 1, back1z - 1,
			 sameBack1);
		goto done;
	}
// double add is more common, and also unambiguous.
// If this algorithm says we added 100 lines, then we added 100 lines.
	if (oplow == 1 && ophigh == 1) {
		i_printf(MSG_LineAddZ3, sameFront + 1, back2z - 1, sameBack2);
		goto done;
	}
	if (ophigh == 3) {
// delete mixed with something else, and I just don't care about the delete.
		if (oplow == 1)
			i_printf(MSG_LineAdd2, sameFront + 1, back2z - 1);
		else if (act2 <= 10)
			i_printf(MSG_LineUpdate3, sameFront + 1, back2z - 1);
		else
			i_printf(MSG_LineUpdateRange, sameFront + 1,
				 back2z - 1);
		goto done;
	}
	if (oplow == 3) {
// if the deleted block is big then report it, otherwise ignore it.
		if (act1 >= 10)
			i_printf(MSG_LineDelete2, act1, sameFront);
		else if (ophigh == 1)
			i_printf(MSG_LineAdd1, sameBack2);
		else
			i_printf(MSG_LineUpdate1, sameBack2);
		goto done;
	}
// a mix of add and update, call it an update.
// If the first group is big then switch to range message.
	if (act2 > 10 && oplow == 2)
		i_printf(MSG_LineUpdateRange,
			 sameFront + 1,
			 (sameBack2 - back2z < 10 ? sameBack2 : back2z - 1));
	else
		i_printf(MSG_LineUpdateZ3, sameFront + 1, back2z - 1,
			 sameBack2);

done:
	return true;
}

static time_t now_sec;
static int now_ms;
static void currentTime(void)
{
	struct timeval tv;
	gettimeofday(&tv, NULL);
	now_sec = tv.tv_sec;
	now_ms = tv.tv_usec / 1000;
}

static void silent(int msg, ...)
{
}

// Is there an active tag below?
static bool activeBelow(Tag *t)
{
	bool rc;
	int action = t->action;
	if (action == TAGACT_INPUT || action == TAGACT_SELECT ||
	    action == TAGACT_A || action == TAGACT_AREA ||
	((action == TAGACT_SPAN || action == TAGACT_DIV) && t->onclick))
		return true;
	t = t->firstchild;
	while (t) {
		rc = activeBelow(t);
		if (rc)
			return rc;
		t = t->sibling;
	}
	return false;
}

static int hovcount, invcount, injcount;

/* Rerender the buffer and notify of any lines that have changed */
int rr_interval = 20;
void rerender(int rr_command)
{
	char *a, *snap, *newbuf;
	int j;
	int markdot, wasdot, addtop;
	bool z;
	void (*say_fn) (int, ...);

	debugPrint(4, "rerender");
	cw->mustrender = false;
	time(&cw->nextrender);
	cw->nextrender += rr_interval;
	hovcount = invcount = injcount = 0;

// not sure if we have to do this here
	rebuildSelectors();

	if (rr_command > 0) {
// You might have changed some input fields on the screen, then typed rr
		jSyncup(true, 0);
	}
// screen snap, to compare with the new screen.
	if (!unfoldBufferW(cw, false, &snap, &j)) {
		snap = 0;
		puts("no screen snap available");
		return;
	}

/* and the new screen */
	a = render(0);
	newbuf = htmlReformat(a);
	nzFree(a);

	if (rr_command > 0 && debugLevel >= 3) {
		char buf[120];
		buf[0] = 0;
		if (hovcount)
			sprintf(buf, "%d nodes under hover", hovcount);
		if (invcount) {
			if (buf[0])
				strcat(buf, ", ");
			sprintf(buf + strlen(buf),
				"%d nodes invisible", invcount);
		}
		if (injcount) {
			if (buf[0])
				strcat(buf, ", ");
			sprintf(buf + strlen(buf), "%d nodes injected by css",
				injcount);
		}
		if (buf[0])
			debugPrint(3, "%s", buf);
	}

/* the high runner case, most of the time nothing changes,
 * and we can check that efficiently with strcmp */
	if (stringEqual(newbuf, snap)) {
		if (rr_command > 0)
			i_puts(MSG_NoChange);
		nzFree(newbuf);
		nzFree(snap);
		return;
	}

/* mark dot, so it stays in place */
	cw->labels[MARKDOT] = wasdot = cw->dot;
	frontBackDiff(snap, newbuf);
	addtop = 0;
	if (sameBack1 > sameFront)
		delText(sameFront + 1, sameBack1);
	if (sameBack2 > sameFront) {
		addTextToBuffer((pst) newChunkStart,
				newChunkEnd - newChunkStart, sameFront, false);
		addtop = sameFront + 1;
	}
	markdot = cw->labels[MARKDOT];
	if (markdot)
		cw->dot = markdot;
	else if (sameBack1 == sameBack2)
		cw->dot = wasdot;
	else if (addtop)
		cw->dot = addtop;
	cw->undoable = false;

/*********************************************************************
It's almost easier to do it than to report it.
First, run diff again with the hidden numbers gone, so we only report
the visible differences. It's annoying to hear that line 27 has been updated,
and it looks just like it did before.
This happens when a periodic timer updates a section through innerHTML.
If the text is the same every time that's fine, but it's new tags each time,
and new internal numbers each time, and that use to trip this algorithm.
*********************************************************************/

	if(rr_command < 0)
		goto done;

	removeHiddenNumbers((pst) snap, 0);
	removeHiddenNumbers((pst) newbuf, 0);
	if (stringEqual(snap, newbuf)) {
		if (rr_command > 0)
			i_puts(MSG_NoChange);
		goto done;
	}
	frontBackDiff(snap, newbuf);
	debugPrint(4, "front %d back %d,%d front z %d,%d back z %d,%d",
		   sameFront, sameBack1, sameBack2,
		   front1z, front2z, back1z, back2z);
	z = reportZ();

// Update from javascript means the lines move, and our undo is unreliable.
// Here is a complicated if, cause often the current line is unaffected.
	if(undo1line <= sameFront || // before any changes
	(sameBack1 == sameBack2 && (
	undo1line > sameBack1 || (
	front1z == front2z && back1z == back2z && (
	(back1z > 0 && undo1line >= back1z && undo1line < sameBack1) ||
	(front1z > 0 && undo1line <= front1z && undo1line > sameFront + 1))))))
		;
	else
		undoSpecialClear();

// Even if the change has been reported above,
// I march on here because it puts dot back where it belongs.
	say_fn = (z ? silent : i_printf);
	if (sameBack2 == sameFront) {	/* delete */
		if (sameBack1 == sameFront + 1)
			(*say_fn) (MSG_LineDelete1, sameFront);
		else
			(*say_fn) (MSG_LineDelete2, sameBack1 - sameFront,
				   sameFront);
	} else if (sameBack1 == sameFront) {
		if (sameBack2 == sameFront + 1)
			(*say_fn) (MSG_LineAdd1, sameFront + 1);
		else {
			(*say_fn) (MSG_LineAdd2, sameFront + 1, sameBack2);
/* put dot back to the start of the new block */
			if (!markdot)
				cw->dot = sameFront + 1;
		}
	} else {
		if (sameBack1 == sameFront + 1 && sameBack2 == sameFront + 1)
			(*say_fn) (MSG_LineUpdate1, sameFront + 1);
		else if (sameBack2 == sameFront + 1)
			(*say_fn) (MSG_LineUpdate2, sameBack1 - sameFront,
				   sameFront + 1);
		else {
			if (sameBack2 - sameFront <= 10 ||
			    sameBack1 - sameFront <= 10)
				(*say_fn) (MSG_LineUpdate3, sameFront + 1,
					   sameBack2);
			else
				(*say_fn) (MSG_LineUpdateRange, sameFront + 1,
					   sameBack2);
/* put dot back to the start of the new block */
			if (!markdot && sameBack1 != sameBack2)
				cw->dot = sameFront + 1;
		}
	}

done:
	nzFree(newbuf);
	nzFree(snap);
}

/* mark the tags on the deleted lines as deleted */
void delTags(int startRange, int endRange)
{
	pst p;
	int j, tagno, action;
	Tag *t;

/* no javascript, no cause to ever rerender */
	if (!cf->cx)
		return;

	for (j = startRange; j <= endRange; ++j) {
		p = fetchLine(j, -1);
		for (; *p != '\n'; ++p) {
			if (*p != InternalCodeChar)
				continue;
			tagno = strtol((char *)p + 1, (char **)&p, 10);
/* could be 0, but should never be negative */
			if (tagno <= 0)
				continue;
			t = tagList[tagno];
/* Only mark certain tags as deleted.
 * If you mark <div> deleted, it could wipe out half the page. */
			action = t->action;
			if (action == TAGACT_TEXT ||
			    action == TAGACT_HR ||
			    action == TAGACT_LI || action == TAGACT_IMAGE)
				t->deleted = true;
		}
	}
}

/* turn an onunload function into a clickable hyperlink */
static void unloadHyperlink(const char *js_function, const char *where)
{
	dwStart();
	stringAndString(&cf->dw, &cf->dw_l, "<P>Onclose <A href='javascript:");
	stringAndString(&cf->dw, &cf->dw_l, js_function);
	stringAndString(&cf->dw, &cf->dw_l, "()'>");
	stringAndString(&cf->dw, &cf->dw_l, where);
	stringAndString(&cf->dw, &cf->dw_l, "</A><br>");
}

/* Run the various onload functions */
/* Turn the onunload functions into hyperlinks */
/* This runs after the page is parsed and before the various javascripts run, is that right? */
void runOnload(void)
{
	int i, action;
	int fn;			/* form number */
	Tag *t;

	if (!isJSAlive)
		return;
	if (intFlag)
		return;

/* window and document onload */
	run_event_win(cf, "window", "onload");
	if (intFlag)
		return;
	run_event_doc(cf, "document", "onload");
	if (intFlag)
		return;

	fn = -1;
	for (i = 0; i < cw->numTags; ++i) {
		if (intFlag)
			return;
		t = tagList[i];
		if (t->slash)
			continue;
		if (t->f0 != cf)
			continue;
		action = t->action;
		if (action == TAGACT_FORM)
			++fn;
		if (!t->jslink)
			continue;
		if (action == TAGACT_BODY)
			run_event_t(t, "body", "onload");
		if (action == TAGACT_BODY && t->onunload)
			unloadHyperlink("document.body.onunload", "Body");
		if (action == TAGACT_FORM)
			run_event_t(t, "form", "onload");
/* tidy5 says there is no form.onunload */
		if (action == TAGACT_FORM && t->onunload) {
			char formfunction[48];
			sprintf(formfunction, "document.forms[%d].onunload", fn);
			unloadHyperlink(formfunction, "Form");
		}
		if (action == TAGACT_H)
			run_event_t(t, "h1", "onload");
	}
}

// In one place, tack on the $$fn to turn onfoo into onfoo$$fn
const char *tack_fn(const char *e)
{
	static char buf[64];
	int l = strlen(e);
	if((unsigned)l + 4 >= sizeof(buf)) {
		debugPrint(3, "%s$$fn too long", e);
		return 0;
	}
// ontimer is our simulated event handler.
	if(stringEqual(e, "ontimer"))
		return 0;
	sprintf(buf, "%s$$fn", e);
	return buf;
}

/*********************************************************************
Manage js timers here.
It's a simple list of timers, assuming there aren't too many.
Store the seconds and milliseconds when the timer should fire,
and an interval flag to repeat.
The usual pathway is setTimeout(), whence backlink is the name
of the timer object under window.
Timer object.backlink also holds the name, so we don't forget it.
jt->t will be 0. There is no tag with this timer.
jt->tsn is a timer sequence number, globally, to help us keep track.
Another path is an asynchronous script.
If we have browsed the page, and down_jsbg is true,
downloading js in background,
then runScriptsPending doesn't run the script, it callse scriptSetsTimeout(),
and thereby puts the script on a timer.
The timer runs as an interval, according to asyncTimer ms.
The script is out of the hands of runScriptsPending,
and eventually executed by runTimer.
The object on the tag is the script object.
There is yet another path, asynchronous xhr.
Like the above, the page must be browsed, and down_jsbg true.
A tag is created, of type Object, not Script.
The tag is connected to the XHR object, not a Script object.
This is given a backlink name from window, with o.backlink having the same name.
This is the same procedure as the timer objects.
The links protect these objects from garbage collection,
but we have to remember to unlink them.
*********************************************************************/

struct jsTimer {
	struct jsTimer *next, *prev;
	Frame *f;	/* edbrowse frame holding this timer */
	Tag *t;	// for an asynchronous script
	time_t sec;
	int ms;
	bool isInterval;
	bool running;
	bool deleted;
	bool pending;
	int jump_sec;		/* for interval */
	int jump_ms;
	int tsn;
	char *backlink;
};

/* list of pending timers */
struct listHead timerList = {
	&timerList, &timerList
};

/*********************************************************************
the spec says you can't run a timer less than 10 ms but here we currently use
3000 ms. This really should be a configurable limit.
If less than 200ms the load average jumps way up.  e.g.nasa.gov
We only rerender the screen every 20 seconds or so anyways.
But, the acid test uses a timer to schedule each of its 100 tests,
and is crazy slow if we throttle them.
So ... the first few timers can run as fast  as they like,and we're ok
with that, then timers slow down as we proceed.
*********************************************************************/
static const int timerSpread = 3000;
static const int timerStep = 7;
int timer_sn;			// timer sequence number

void domSetsTimeout(int n, const char *jsrc, const char *backlink, bool isInterval)
{
	struct jsTimer *jt;
	int seqno;
	int n2;

	if (jsrc[0] == 0)
		return;		/* nothing to run */

	if (stringEqual(jsrc, "-")) {
// Delete a timer. Comes from clearTimeout(obj).
		seqno = n;
		foreach(jt, timerList) {
			if (jt->tsn != seqno)
				continue;
			debugPrint(3, "timer %d delete from context %d", seqno,
			jt->f ? jt->f->gsn: -1);
// a running timer will often delete itself.
			if (jt->running) {
				jt->deleted = true;
			} else {
				if (backlink)
					delete_property_win(jt->f, backlink);
				delFromList(jt);
				nzFree(jt->backlink);
				nzFree(jt);
			}
			return;
		}
// not found, just return.
		return;
	}

// now adding a timer
	jt = allocZeroMem(sizeof(struct jsTimer));
	if(stringEqual(jsrc, "@@pending"))
		jt->pending = true;
	else {
		if(n < cf->jtmin)
			n = cf->jtmin;
		if(!n)
			n = 10;
		if (n < timerSpread)
			cf->jtmin = n += timerStep;
	}
	if ((jt->isInterval = isInterval))
		jt->jump_sec = n / 1000, jt->jump_ms = n % 1000;
	n2 = n;
// promise jobs not throttled by timerspeed
	if(timerspeed > 1 && !jt->pending &&
		0x40000000 / timerspeed >= n)
		n2 *= timerspeed;
	jt->sec = n2 / 1000;
	jt->ms = n2 % 1000;
	currentTime();
	jt->sec += now_sec;
	jt->ms += now_ms;
	if (jt->ms >= 1000)
		jt->ms -= 1000, ++jt->sec;
	jt->backlink = cloneString(backlink);
	jt->f = cf;
	addToListBack(&timerList, jt);
	seqno = timer_sn;
	debugPrint(3, "timer %d add to context %d under %s",
	seqno, (cf ? cf->gsn : -1), backlink);
	jt->tsn = seqno;
}

void scriptSetsTimeout(Tag *t)
{
	struct jsTimer *jt = allocZeroMem(sizeof(struct jsTimer));
// asychronous scripts or xhr are not throttled by timerspeed
	jt->sec = 0;
	jt->ms = asyncTimer;
	jt->isInterval = true;
	jt->jump_sec = 0, jt->jump_ms = asyncTimer;
	currentTime();
	jt->sec += now_sec;
	jt->ms += now_ms;
	if (jt->ms >= 1000)
		jt->ms -= 1000, ++jt->sec;
	jt->t = t;
	jt->f = cf;
	addToListBack(&timerList, jt);
	debugPrint(3, "timer %s%d=%s context %d",
		   (t->action == TAGACT_SCRIPT ? "script" : "xhr"),
		   ++timer_sn, t->href, cf->gsn);
	jt->tsn = timer_sn;
}

static struct jsTimer *soonest(void)
{
	struct jsTimer *t, *best_t = 0;
	const Window *w;
	if (listIsEmpty(&timerList))
		return 0;
	foreach(t, timerList) {
		if(!t->pending) {
// regular timer, not the pending jobs timer
			if(!gotimers)
				continue;
// Browsing a new web page in the current session pushes the old one, like ^z
// in Linux. The prior page suspends, and the timers suspend.
// ^ is like fg, bringing it back to life.
			w = t->f->owner;
			if(sessionList[w->sno].lw != w)
				continue;
		}
		if (!best_t || t->sec < best_t->sec ||
		    (t->sec == best_t->sec && t->ms < best_t->ms))
			best_t = t;
	}
	return best_t;
}

bool timerWait(int *delay_sec, int *delay_ms)
{
	struct jsTimer *jt;
	time_t now;
	int remaining;

// if js is not active then we don't need any timers; even if they are there.
	if(!allowJS)
		return false;

	if (cw->mustrender) {
		time(&now);
		remaining = 0;
		if (now < cw->nextrender)
			remaining = cw->nextrender - now;
	}

	if (!(jt = soonest())) {
		if (!cw->mustrender)
			return false;
		*delay_sec = remaining;
		*delay_ms = 0;
		return true;
	}

	currentTime();
	if (now_sec > jt->sec || (now_sec == jt->sec && now_ms >= jt->ms))
		*delay_sec = *delay_ms = 0;
	else {
		*delay_sec = jt->sec - now_sec;
		*delay_ms = (jt->ms - now_ms);
		if (*delay_ms < 0)
			*delay_ms += 1000, --*delay_sec;
	}

	if (cw->mustrender && remaining <= *delay_sec) {
		*delay_sec = remaining;
		*delay_ms = 0;
	}

	return true;
}

void delTimers(const Frame *f)
{
	int delcount = 0;
	struct jsTimer *jt, *jnext;
	for (jt = timerList.next; jt != (void *)&timerList; jt = jnext) {
		jnext = jt->next;
		if (jt->f == f) {
			++delcount;
			delFromList(jt);
			nzFree(jt->backlink);
			nzFree(jt);
		}
	}
	if(delcount)
		debugPrint(3, "%d timers deleted from context %d", delcount, f->gsn);

	delPendings(f);
}

void runTimer(void)
{
	struct jsTimer *jt;
	Window *save_cw = cw;
	Frame *save_cf = cf;
	Tag *t;

	currentTime();

	if (!(jt = soonest()) ||
	    (jt->sec > now_sec || (jt->sec == now_sec && jt->ms > now_ms)))
		goto done;

	if(jt->pending) { // pending jobs
		my_ExecutePendingJobs();
		my_ExecutePendingMessages();
		my_ExecutePendingMessagePorts();
// promise jobs not throttled by timerspeed
		int n = jt->jump_sec * 1000 + jt->jump_ms;
		jt->sec = now_sec + n / 1000;
		jt->ms = now_ms + n % 1000;
		if (jt->ms >= 1000)
			jt->ms -= 1000, ++jt->sec;
		goto done;
	}

	if (!gotimers)
		goto skip_execution;

	cf = jt->f;
	cw = cf->owner;

/*********************************************************************
Only syncing the foreground window is right almost all the time,
but not every time.
The forground could be just text, buffer for a textarea in another window.
You should sync that other window before running javascript, so it has
the latest text, the text you are editing right now.
I can't do that because jSyncup calls fetchLine() to pull text lines
out of the buffer, which has to be the foreground window.
We need to fix this someday, though it is a very rare corner case.
*********************************************************************/
	if (foregroundWindow)
		jSyncup(true, 0);
	jt->running = true;
	if ((t = jt->t)) {
// asynchronous script or xhr
		if (t->step == 3) {	// background load
			int rc =
			    pthread_tryjoin_np(t->loadthread, NULL);
			if (rc != 0 && rc != EBUSY) {
// should never happen
				debugPrint(3,
					   "script background thread test returns %d",
					   rc);
				pthread_join(t->loadthread, NULL);
				rc = 0;
			}
			if (!rc) {	// it's done
				if (!t->loadsuccess ||
				(t->action == TAGACT_SCRIPT &&  t->hcode != 200)) {
					if (debugLevel >= 3)
						i_printf(MSG_GetJS,
							 t->href, t->hcode);
					t->step = 6;
				} else {
					if (t->action == TAGACT_SCRIPT) {
						set_property_string_t(t, "text", t->value);
						nzFree(t->value);
						t->value = 0;
					}
					t->step = 4;	// loaded
				}
			}
		}
		if (t->step == 4 && t->action == TAGACT_SCRIPT) {
			char *js_file = t->js_file;
			int ln = t->js_ln;
			t->step = 5;	// running
			if (!js_file)
				js_file = "generated";
			if (!ln)
				ln = 1;
			if (ln > 1)
				++ln;
			if (cf != save_cf)
				debugPrint(4,
					   "running script at a lower frame %s",
					   js_file);
			debugPrint(3, "async exec timer %d %s at %d",
				   jt->tsn, js_file, ln);
			jsRunData(t, js_file, ln);
			debugPrint(3, "async exec complete");
		}
		if (t->step == 4 && t->action != TAGACT_SCRIPT) {
			t->step = 5;
			set_property_string_t(t, "$entire", t->value);
			nzFree(t->value);
			t->value = 0;
			debugPrint(3, "run xhr %d", jt->tsn);
			run_function_bool_t(t, "parseResponse");
		}
		if (t->step >= 5)
			jt->deleted = true;
	} else {
// regular timer
		debugPrint(4, "exec timer %d context %d", jt->tsn, jt->f->gsn);
		run_ontimer(jt->f, jt->backlink);
		debugPrint(4, "exec complete");
	}
	jt->running = false;
skip_execution:

	if (!jt->isInterval || jt->deleted) {
		debugPrint(3, "timer %d complete in context %d under %s",
		jt->tsn, (jt->f ? jt->f->gsn : -1), jt->backlink);
// at debug 3 or higher, keep these around, in case you have to
// track down an error.
		if(debugLevel < 3 && jt->backlink)
			delete_property_win(jt->f, jt->backlink);
		t = jt->t;
		delFromList(jt);
		nzFree(jt->backlink);
		nzFree(jt);
		if(t) {
// this will free the xhr object and allow for garbage collection.
			disconnectTagObject(t);
			t->dead = true;
			t->action = TAGACT_NOP;
		}
	} else {
		int n = jt->jump_sec * 1000 + jt->jump_ms;
		if(n < cf->jtmin)
			n = cf->jtmin;
		if(!n)
			n = 10;
		if (n < timerSpread)
			cf->jtmin = n += timerStep;
		if(timerspeed > 1 && !jt->pending && !jt->t &&
		0x40000000 / timerspeed >= n)
			n *= timerspeed;
		jt->sec = now_sec + n / 1000;
		jt->ms = now_ms + n % 1000;
		if (jt->ms >= 1000)
			jt->ms -= 1000, ++jt->sec;
	}

	if (gotimers)
		jSideEffects();

done:
	cw = save_cw, cf = save_cf;
}

void showTimers(void)
{
	const struct jsTimer *t;
	int n;
	bool printed = false;

	currentTime();
	foreach(t, timerList) {
		if(t->pending)
			continue;
		if(t->f->owner != cw)
			continue;
		printed = true;
		if(t->isInterval)
			printf("interval ");
		else if(t->t)
			printf("%s ",
			   (t->t->action == TAGACT_SCRIPT ? "script" : "xhr"));
		else
			printf("timer ");
		printf("%d cx%d %s ", t->tsn, t->f->gsn, t->backlink);
		n = (t->sec - now_sec) * 1000;
		n += t->ms - now_ms;
		if(n >= 1000 || n < -1000)
			printf("in %ds", n / 1000);
		else
			printf("in %dms", n);
		if(t->isInterval) {
			n = t->jump_sec * 1000 + t->jump_ms;
			if(n >= 1000)
				printf(" freq %ds", n / 1000);
			else
				printf(" freq %dms", n);
		}
		puts("");
	}

	if(!printed)
		i_puts(MSG_Empty);
}

void domOpensWindow(const char *href, const char *name)
{
	char *copy, *r;
	const char *a;
	bool replace = false;

	if (*href == 'r')
		replace = true;
	++href;
	if (!*href) {
		debugPrint(3, "javascript is opening a blank window");
		return;
	}

	copy = cloneString(href);
	unpercentURL(copy);
	r = resolveURL(cf->hbase, copy);
	nzFree(copy);
	if ((replace || cw->browseMode) && foregroundWindow) {
		gotoLocation(r, 0, replace);
		return;
	}

/* Turn the new window into a hyperlink. */
/* just shovel this onto dw, as though it came from document.write() */
	dwStart();
	stringAndString(&cf->dw, &cf->dw_l, "<P>");
	stringAndString(&cf->dw, &cf->dw_l,
			i_getString(replace ? MSG_Redirect : MSG_NewWindow));
	stringAndString(&cf->dw, &cf->dw_l, ": <A href=");
	stringAndString(&cf->dw, &cf->dw_l, r);
	stringAndChar(&cf->dw, &cf->dw_l, '>');
	a = altText(r);
	nzFree(r);
/* I'll assume this is more helpful than the name of the window */
	if (a)
		name = a;
	r = htmlEscape(name);
	stringAndString(&cf->dw, &cf->dw_l, r);
	nzFree(r);
	stringAndString(&cf->dw, &cf->dw_l, "</A><br>\n");
}

/* the new string, the result of the render operation */
static char *ns;
static int ns_l;
static bool invisible, tdfirst;
static Tag *inv2, *inv3;	// invisible via css
static int listnest;		/* count nested lists */
/* None of these tags nest, so it is reasonable to talk about
 * the current open tag. */
static Tag *currentForm, *currentA;

static char *backArrow(char *s)
{
	if (!s)
		s = ns + ns_l;
	while (s > ns) {
		if ((uchar) (*--s) == 0xe2 && (uchar) s[1] == 0x89 &&
		    ((uchar) s[2] == 0xaa || (uchar) s[2] == 0xab))
			return s;
	}
	return 0;
}

static char *backColon(char *s)
{
	while (s > ns)
		if (*--s == ':')
			break;
	return s;
}

static void swapArrow(void)
{
	char *s = ns + ns_l - 6;
	if (s > ns &&
	    !strncmp(s, "≫\0020", 5) && (s[5] == '>' || s[5] == '}')) {
		strmove(s, s + 3);
		strcpy(s + 3, "≫");
	}
}

// Is this table a matrix of data, or just for layout purposes?
// 0 means we can't tell, 1 is data, 2 is presentation
// t is the cell.
static int tableType(const Tag *t)
{
	const char *role;
	if(stringEqual(t->info->name, "th"))
		return 1;
	t = t->parent;
// no containing row; don't know
	if(!t || t->action != TAGACT_TR)
		return 0;
// row header = data
	if(stringEqual(t->firstchild->info->name, "th"))
		return 1;
	while((t = t->parent)) {
		if(t->action == TAGACT_TABLE)
			break;
	}
	if(!t) // no table
		return 0;
	role = attribVal(t, "role");
	if(role && stringEqual(role, "presentation"))
		return 2;
// descend and look for caption or thead
	for(t = t->firstchild; t; t = t->sibling)
		if(t->action == TAGACT_THEAD ||
		stringEqual(t->info->name, "caption"))
			return 1;
	return 0;
}

static char *td_text;
static int td_text_l;
static void td_textUnder(const Tag *u)
{
	if(u->action == TAGACT_TEXT && u->textval)
		stringAndString(&td_text,&td_text_l, u->textval);
	for(u = u->firstchild; u; u = u->sibling)
		td_textUnder(u);
}

// return a column heading by number.
static void findHeading(const Tag *t, int colno)
{
	int j = 0;
	const Tag *u;
	td_text = initString(&td_text_l);
	if(!t->parent ||
	((t = t->parent)->action != TAGACT_TABLE &&
	t->action != TAGACT_THEAD &&
	t->action != TAGACT_TBODY))
		return;
	if(t->action != TAGACT_TABLE) {
// it is tbody or thead
		t = t->parent;
		if(!t || t->action != TAGACT_TABLE)
			return;
	}
	t = t->firstchild;
	for(u = t; u; u = u->sibling)
		if(u->action == TAGACT_THEAD) {
			t = u;
			break;
		}
	if(t && t->action == TAGACT_THEAD)
		t = t->firstchild;
	if(!t || t->action != TAGACT_TR || !t->firstchild)
		return;
	t = t->firstchild;
	if(t->action != TAGACT_TD ||
	!stringEqual(t->info->name, "th"))
		return;
	while(t) {
		if(t->action == TAGACT_TD) {
			if(++j == colno) {
// this is the header we want, descend to the text field
				td_textUnder(t);
				return;
			}
		}
		t = t->sibling;
	}
	return;
}

// return allocated string, as may come from js
static char *arialabel(const Tag *t)
{
	const char *a;
	if(allowJS && t->jslink) {
		char *u = get_property_string_t(t, "aria-label");
		if(u && *u)
			return u;
		nzFree(u);  // empty string?
	}
	a = attribVal(t, "aria-label");
	return (a && *a) ? cloneString(a) : 0;
}

static void tagInStream(int tagno)
{
	char buf[32];
	sprintf(buf, "%c%d*", InternalCodeChar, tagno);
	stringAndString(&ns, &ns_l, buf);
}

/* see if a number or star is pending, waiting to be printed */
static void liCheck(Tag *t)
{
	Tag *ltag;	/* the start list tag */
	if (listnest && (ltag = findOpenList(t)) && ltag->post) {
		char olbuf[32];
		if (ltag->ninp)
			tagInStream(ltag->ninp);
		if (ltag->action == TAGACT_OL) {
			int j = ++ltag->lic, k;
			Tag *tli = tagList[ltag->ninp];
// this checks for <li value=7>, but does not check for javascript
// dynamically setting value after the html is parsed.
// Add that in someday.
			if(tli->value && (k = stringIsNum(tli->value)) >= 0)
				ltag->lic = j = k;
			sprintf(olbuf, "%d. ", j);
		} else {
			strcpy(olbuf, "* ");
		}
		if (!invisible)
			stringAndString(&ns, &ns_l, olbuf);
		ltag->post = false;
	}
}

static Tag *deltag;

static void renderNode(Tag *t, bool opentag)
{
	int tagno = t->seqno;
	Frame *f = t->f0;
	char hnum[40];		/* hidden number */
#define ns_hnum() stringAndString(&ns, &ns_l, hnum)
#define ns_ic() stringAndChar(&ns, &ns_l, InternalCodeChar)
	int j, l;
	int itype;		/* input type */
	const struct tagInfo *ti = t->info;
	int action = t->action;
	char c;
	bool endcolor;
	bool retainTag;
	const char *a;		/* usually an attribute */
	char *u;
	Tag *ltag;	/* list tag */

	debugPrint(6, "rend %c%s", (opentag ? ' ' : '/'), t->info->name);

	if (deltag) {
		if (t == deltag && !opentag)
			deltag = 0;
li_hide:
/* we can skate past the li tag, but still need to increment the count */
		if (action == TAGACT_LI && opentag &&
		    (ltag = findOpenList(t)) && ltag->action == TAGACT_OL)
			++ltag->lic;
		return;
	}
	if (t->deleted) {
		deltag = t;
		goto li_hide;
	}

	if (inv2) {
		if (inv2 == t)
			inv2 = NULL;
		return;
	}

	if (inv3 == t) {
		inv3 = NULL;
// I tried to remove an empty invisible section,
// but it's never really empty due to tag markers.
		stringAndString(&ns, &ns_l, "\r]]\r");
		return;
	}

	endcolor = false;
	if (doColors && !opentag && t->iscolor) {
		char *u0, *u1, *u3;
// don't put a color around whitespace
		u1 = backArrow(0);
// there should always be a previous color marker
		if (!u1)
			goto nocolorend;
		if ((uchar) u1[2] == 0xab)	// close
			goto yescolorend;
		for (u3 = u1 + 3; *u3; ++u3) {
			if (*u3 == InternalCodeChar) {
				for (++u3; isdigit(*u3); ++u3) ;
				if (*u3 == '*' && !*++u3)
					break;
			}
			if (!isspace(*u3))
				goto yescolorend;
		}
		u0 = backColon(u1);
		if (*u0 != ':')
			goto yescolorend;
		for (u3 = u0 + 1; u3 < u1; ++u3)
			if (*u3 != ' ' && !isalpha(*u3))
				goto yescolorend;
		u1 += 3;
		strmove(u0, u1);
		ns_l -= (u1 - u0);
		goto nocolorend;
yescolorend:
		stringAndString(&ns, &ns_l, "≫");
		endcolor = true;
	}
nocolorend:

	if (!opentag && ti->bits & TAG_NOSLASH)
		return;

	if (opentag) {
// what is the visibility now?
		uchar v_now = 2;
		if(allowJS && t->jslink) {
			t->disval =
			    run_function_onearg_win(f, "eb$visible", t);
// If things appear upon hover, they do this sometimes if your mouse
// is anywhere in that section, so maybe we should see them.
// Also if color is transparent then it surely changes to a color
// if the mouse is somewhere or on some circumstance,
// so just bring this to light as well.
			if(t->disval != DIS_INVISIBLE)
				t->disval = DIS_COLOR;
		} else {
// allow html to hide sections, even if js is not running.
			t->disval = 0;
			if(((a = attribVal(t, "hidden")) && !stringEqual(a, "false")) ||
			((a = attribVal(t, "aria-hidden")) && !stringEqual(a, "false")))
				t->disval = DIS_INVISIBLE;
		}
		if (t->disval == DIS_INVISIBLE)
			v_now = DIS_INVISIBLE;
		if (t->disval == DIS_HOVER)
			v_now = DIS_COLOR;
		if (t->action == TAGACT_TEXT && v_now == DIS_HOVER) {
			Tag *y = t;
			while (y && y->f0 == f) {
				uchar dv = y->disval;
				if (dv == DIS_TRANSPARENT)
					v_now = DIS_INVISIBLE;
				if (dv == DIS_HOVERCOLOR)
					v_now = DIS_COLOR;
				if (dv >= DIS_COLOR)
					break;
				y = y->parent;
			}
		}
// gather some stats
		if (v_now == DIS_INVISIBLE)
			++invcount;
		if (v_now == DIS_COLOR)
			++hovcount;
		if (v_now == DIS_INVISIBLE) {
			if (!showHover) {
				inv2 = t;
				return;
			}
			if (!inv3) {
				inv3 = t;
// merge adjacent invisible sections together
				if (ns_l >= 4
				    && stringEqual(ns + ns_l - 4, "\r]]\r"))
					ns_l -= 4;
				else
					stringAndString(&ns, &ns_l, "\r[[\r");
			}
		}
		if (!showHover && v_now == DIS_COLOR && !activeBelow(t)) {
			inv2 = t;
			return;
		}
		if (action == TAGACT_TEXT && t->jslink &&
		    get_property_bool_t(t, "inj$css")) {
			++injcount;
			if (!showHover) {
				inv2 = t;
				return;
			}
		}
	}

	retainTag = true;
	if (invisible)
		retainTag = false;
	if (ti->bits & TAG_INVISIBLE) {
		retainTag = false;
		invisible = opentag;
/* special case for noscript with no js */
		if (action == TAGACT_NOSCRIPT && !f->cx)
			invisible = false;
	}

	if (doColors && opentag) {
		char *u0, *u1, *u2, *u3;
		char *color, *recolor = 0;
		t->iscolor = false;
		color = get_style_string_t(t, "color");
		if (!color || !color[0])
			goto nocolor;
		caseShift(color, 'l');
		recolor = closeColor(color);
		if (!recolor) {
			nzFree(color);
			goto nocolor;
		}
		if (recolor != color)
			nzFree(color);
		if (stringEqual(recolor, "inherit")) {	// not a color
			nzFree(recolor);
			goto nocolor;
		}
// is this the same as the previous?
		u2 = backArrow(0);
		if (!u2)
			goto yescolor;
		if ((uchar) u2[2] == 0xaa) {	// open
			u1 = u2;
			u2 = 0;	// no closing
		} else {
			u1 = backArrow(u2);
			if (!u1 || (uchar) u1[2] != 0xaa)
				goto yescolor;
		}
// back up to :
		u0 = backColon(u1);
		if (*u0++ != ':' ||
		    (unsigned)(u1 - u0) != strlen(recolor) || memcmp(u0, recolor, u1 - u0))
			goto yescolor;
		if (!u2) {
// it's the same color, orange inside orange
			nzFree(recolor);
			goto nocolor;
		}
// merge sections if there are no words in between
		for (u3 = u2; *u3; ++u3) {
			if (*u3 == InternalCodeChar)
				for (++u3; isdigit(*u3); ++u3) ;
			if (isalnum(*u3))
				goto yescolor;
		}
		strmove(u2, u2 + 3);
		ns_l -= 3;
		nzFree(recolor);
		t->iscolor = true;
		goto nocolor;
yescolor:
		stringAndChar(&ns, &ns_l, ':');
		stringAndString(&ns, &ns_l, recolor);
		stringAndString(&ns, &ns_l, "≪");
		nzFree(recolor);
		t->iscolor = true;
	}
nocolor:

	switch (action) {
	case TAGACT_TEXT:
		if (t->jslink) {
// defer to the javascript text.
// either we query js every time, on every piece of text, as we do now,
// or we include a setter so that TextNode.data assignment has a side effect.
			char *u = get_property_string_t(t, "data");
			if (u) {
				nzFree(t->textval);
				t->textval = u;
			}
		}
		if (!t->textval)
			break;
		liCheck(t);
		if (!invisible) {
			tagInStream(tagno);
			stringAndString(&ns, &ns_l, t->textval);
		}
		break;

	case TAGACT_A:
		liCheck(t);
		currentA = (opentag ? t : 0);
		if (!retainTag)
			break;
// Javascript might have set or changed this url.
		if (opentag && t->jslink) {
			char *new_url = get_property_url_t(t, false);
			if (new_url && *new_url) {
				nzFree(t->href);
				t->href = new_url;
			}
		}
		if (opentag && !t->href) {
// onclick turns this into a hyperlink.
			if (tagHandler(tagno, "onclick"))
				t->href = cloneString("#");
		}
		if (t->href) {
			if (opentag) {
				sprintf(hnum, "%c%d{", InternalCodeChar, tagno);
				if((a = arialabel(t))) {
// for <a>,  aria-label replaces anything that was below; this takes precedence
					ns_hnum();
					stringAndString(&ns, &ns_l, a);
					cnzFree(a);
					sprintf(hnum, "%c0}", InternalCodeChar);
					deltag = t;
// <a title=x>   x appears on hover
				} else if (t->jslink     && (a =
					get_property_string_t(t, "title"))) {
					++hovcount;
					if (showHover) {
						stringAndString(&ns, &ns_l, a);
						stringAndChar(&ns, &ns_l, ' ');
					}
					cnzFree(a);
				}
			} else // open or closed
				sprintf(hnum, "%c0}", InternalCodeChar);
		} else { // href or no href
			if (opentag)
				sprintf(hnum, "%c%d*", InternalCodeChar, tagno);
			else
				hnum[0] = 0;
		} // href or no href
		ns_hnum();
		if (endcolor)
			swapArrow();
		break;

// check for span onclick and make it look like a link.
// Same for div, maybe for others too.
	case TAGACT_SPAN: case TAGACT_DIV:
		a = 0, u = opentag ? arialabel(t) : 0;
// If nothing in the span then the title becomes important.
		if (!t->firstchild && opentag && !u) {
			a = attribVal(t, "title");
			if (allowJS && t->jslink)
				u = get_property_string_t(t, "title");
		}
// If an onclick function, then turn this into a hyperlink, thus clickable.
// At least one site adds the onclick function via javascript, not html.
// But only at the start, so maybe we only need to check on the first render.
// But maybe some other site adds onclick later. Do we have to check every time?
// This rerender function is getting more and more js intensive!
		if (!t->onclick && t->jslink && handlerPresent(t, "onclick"))
			t->onclick = true;
		if (!(t->onclick & allowJS)) {
// regular span
			if((u || a) && action == TAGACT_DIV)
				stringAndChar(&ns, &ns_l, '\n');
			if (u)
				stringAndString(&ns, &ns_l, u), nzFree(u);
			else if (a)
				stringAndString(&ns, &ns_l, a);
			if((u || a) && t->firstchild)
				stringAndChar(&ns, &ns_l, ' ');
			goto nop;
		}
// this span has click, so turn into {text}
		if (opentag) {
			if((u || a) && action == TAGACT_DIV)
				stringAndChar(&ns, &ns_l, '\n');
			sprintf(hnum, "%c%d{", InternalCodeChar, tagno);
			ns_hnum();
			if (u)
				stringAndString(&ns, &ns_l, u), nzFree(u);
			else if (a)
				stringAndString(&ns, &ns_l, a);
			if((u || a) && t->firstchild)
				stringAndChar(&ns, &ns_l, ' ');
		} else {
			sprintf(hnum, "%c0}", InternalCodeChar);
			ns_hnum();
			if (endcolor)
				swapArrow();
		}
		break;

	case TAGACT_BQ:
		if (invisible)
			break;
		stringAndString(&ns, &ns_l,
		(opentag ? "\f``" : "''\f"));
		break;

	case TAGACT_SVG:
		if (!invisible && opentag) {
// I use to print "graphics" here, but that conveys virtually no information.
// Maybe at some point I'll find something useful to insert
// to say yes there's some visual thing here.
// Meantime, I better at least put in a space, because some graphic
// might separate two words.
			stringAndChar(&ns, &ns_l, ' ');
		}
		break;

	case TAGACT_OL:
	case TAGACT_UL:
		t->lic = t->slic;
		t->post = false;
		if (opentag)
			++listnest;
		else
			--listnest;

	case TAGACT_DL:
	case TAGACT_DT:
	case TAGACT_DD:
	case TAGACT_OBJECT:
	case TAGACT_BR:
	case TAGACT_P:
	case TAGACT_H:
	case TAGACT_NOP:
nop:
		if (invisible)
			break;
		j = ti->para;
		if (opentag)
			j &= 3;
		else
			j >>= 2;
// <blockquote><p> don't put in the extra whitespace after ``
		if(ns_l >= 3 && stringEqual(ns + ns_l - 3, "\f``"))
			j = 0;
		if (j) {
			c = '\f';
			if (j == 1) {
				c = '\r';
				if (action == TAGACT_BR)
					c = '\n';
			}
			stringAndChar(&ns, &ns_l, c);
			if (doColors && t->iscolor &&
			    ns_l > 4 && !memcmp(ns + ns_l - 4, "≪", 3)) {
// move the newline before the color
				char *u0 = ns + ns_l - 4;
				u0 = backColon(u0);
				if (*u0 == ':') {
					int j = strlen(u0);
					memmove(u0 + 1, u0, j);
					*u0 = c;
				}
			}
			if (opentag && action == TAGACT_H) {
				strcpy(hnum, ti->name);
				strcat(hnum, " ");
				ns_hnum();
			}
		}
/* tags with id= have to be part of the screen, so you can jump to them */
		if (t->id && opentag && action != TAGACT_LI)
			tagInStream(tagno);
		break;

	case TAGACT_PRE:
		if (!retainTag)
			break;
/* one of those rare moments when I really need </tag> in the text stream */
		j = (opentag ? tagno : t->balance->seqno);
/* I need to manage the paragraph breaks here, rather than t->info->para,
 * which would rule if I simply redirected to nop.
 * But the order is wrong if I do that. */
		if (opentag)
			stringAndChar(&ns, &ns_l, '\f');
		sprintf(hnum, "%c%d*", InternalCodeChar, j);
		ns_hnum();
		if (!opentag)
			stringAndChar(&ns, &ns_l, '\f');
		break;

	case TAGACT_FORM:
		currentForm = (opentag ? t : 0);
		goto nop;

	case TAGACT_INPUT:
		if (!retainTag)
			break;
		if (!opentag) {
// button tag opens and closes, like anchor.
// Check and make sure it's not </select>
			if (!stringEqual(t->info->name, "button"))
				break;
// <button></button> with no text yields "push".
			while (ns_l && isspace(ns[ns_l - 1]))
				--ns_l;
			if (ns_l >= 3 && ns[ns_l - 1] == '<'
			    && isdigit(ns[ns_l - 2]))
				stringAndString(&ns, &ns_l,
						i_getString(MSG_Push));
			ns_ic();
			stringAndString(&ns, &ns_l, "0>");
			if (endcolor)
				swapArrow();
			break;
		}
// value has to be something.
		if (!t->value)
			t->value = emptyString;
		itype = t->itype;
		if (itype == INP_HIDDEN)
			break;
		liCheck(t);
		if (itype == INP_TA) {
			j = t->lic;
			if (j)
				sprintf(hnum, "%c%d<buffer %d%c0>",
					InternalCodeChar, t->seqno, j,
					InternalCodeChar);
			else if (t->value[0])
				sprintf(hnum, "%c%d<buffer text%c0>",
					InternalCodeChar, t->seqno,
					InternalCodeChar);
			else
				sprintf(hnum, "%c%d<buffer ?%c0>",
					InternalCodeChar, t->seqno,
					InternalCodeChar);
			ns_hnum();
			break;
		}
		sprintf(hnum, "%c%d<", InternalCodeChar, tagno);
		ns_hnum();
// button stops here, until </button>
		if (stringEqual(t->info->name, "button"))
			break;
		if (itype < INP_RADIO) {
			if (t->value[0])
				stringAndString(&ns, &ns_l, t->value);
			else if (itype == INP_SUBMIT || itype == INP_IMAGE) {
				u = (char *)i_getString(MSG_Submit);
				if ((a = attribVal(t, "alt"))) {
					u = altText(a);
					a = NULL;
	/* see if js has changed the alt tag */
					if (allowJS && t->jslink) {
						char *aa =
						    get_property_string_t(t, "alt");
						if (aa)
							u = altText(aa);
						nzFree(aa);
					}
				}
				stringAndString(&ns, &ns_l, u);
			} else if (itype == INP_RESET)
				stringAndString(&ns, &ns_l,
						i_getString(MSG_Reset));
			else if (itype == INP_BUTTON)
				stringAndString(&ns, &ns_l,
						i_getString(MSG_Push));
		} else {
// in case js checked or unchecked
			if (allowJS && t->jslink)
				t->checked =
				    get_property_bool_t(t, "checked");
			stringAndChar(&ns, &ns_l, (t->checked ? '+' : '-'));
		}
		if (currentForm && (itype == INP_SUBMIT || itype == INP_IMAGE)) {
			if (currentForm->secure)
				stringAndString(&ns, &ns_l,
						i_getString(MSG_Secure));
			if (currentForm->bymail)
				stringAndString(&ns, &ns_l,
						i_getString(MSG_Bymail));
		}
		ns_ic();
		stringAndString(&ns, &ns_l, "0>");
		break;

	case TAGACT_LI:
		if ((ltag = findOpenList(t))) {
			ltag->post = true;
/* borrow ninp to store the tag number of <li> */
			ltag->ninp = t->seqno;
		}
		goto nop;

	case TAGACT_HR:
		liCheck(t);
		if (retainTag) {
			tagInStream(tagno);
			stringAndString(&ns, &ns_l, "\r----------\r");
		}
		break;

	case TAGACT_TR:
		if (opentag)
			tdfirst = true;
		if(t->ur && opentag && (ltag = t->parent)
		&& (ltag->action == TAGACT_TABLE || ltag->action == TAGACT_TBODY)) {
// print the row number
			Tag *v = ltag->firstchild;
			j = 1;
			while(v && v != t) {
				if(v->action == TAGACT_TR)
					++j;
				v = v->sibling;
			}
			if(v) { // should always happen
				char rowbuf[20];
				sprintf(rowbuf, "row %d\n", j);
				stringAndString(&ns, &ns_l, rowbuf);
			}
		}
	case TAGACT_TABLE:
		goto nop;

	case TAGACT_TD:
		if (!retainTag)
			break;
		if(!(ltag = t->parent)
		|| ltag->action != TAGACT_TR || !ltag->ur ||
		t->info->name[1] == 'h') {
// Traditional table format, pipe separated,
// on one line if it fits, or wraps in unpredictable ways if it doesn't.
			if (tdfirst)
				tdfirst = false;
			else {
				liCheck(t);
				j = ns_l;
				while (j && ns[j - 1] == ' ')
					--j;
				ns[j] = 0;
				ns_l = j;
				j = tableType(t);
				stringAndChar(&ns, &ns_l, "\3\4 "[j]);
			}
		} else {
// unfolded row, find the column number.
			Tag *v = ltag->firstchild;
			if (tdfirst)
				tdfirst = false;
			else
				stringAndChar(&ns, &ns_l, '\n');
			j = 1;
			while(v && v != t) {
				if(v->action == TAGACT_TD)
					++j;
				v = v->sibling;
			}
			if(v) { // should always happen
				findHeading(ltag, j);
				if(td_text_l) {
					stringAndString(&ns, &ns_l, td_text);
				nzFree(td_text);
				} else
					stringAndNum(&ns, &ns_l, j);
				stringAndString(&ns, &ns_l, ": ");
			}
		}
		tagInStream(tagno);
		break;

/* This is strictly for rendering math pages written with my particular css.
* <span class=sup> becomes TAGACT_SUP, which means superscript.
* sub is subscript and ovb is overbar.
* Sorry to put my little quirks into this program, but hey,
* it's my program. */
	case TAGACT_SUP:
	case TAGACT_SUB:
	case TAGACT_OVB:
		if (!retainTag)
			break;
		if (action == TAGACT_SUB)
			j = 1;
		if (action == TAGACT_SUP)
			j = 2;
		if (action == TAGACT_OVB)
			j = 3;
		if (opentag) {
			static const char *openstring[] = { 0,
				"[", "^(", "`"
			};
			t->lic = ns_l;
			liCheck(t);
			stringAndString(&ns, &ns_l, openstring[j]);
			break;
		}
		if (j == 3) {
			stringAndChar(&ns, &ns_l, '\'');
			break;
		}
/* backup, and see if we can get rid of the parentheses or brackets */
		l = t->lic + j;
		u = ns + l;
/* skip past <span> tag indicator */
		if (*u == InternalCodeChar) {
			++u;
			while (isdigit(*u))
				++u;
			++u;
		}
		if (j == 2 && isalphaByte(u[0]) && !u[1])
			goto unparen;
		if (j == 2 && (stringEqual(u, "th") || stringEqual(u, "rd")
			       || stringEqual(u, "nd") || stringEqual(u, "st"))) {
			strmove(ns + l - 2, ns + l);
			ns_l -= 2;
			break;
		}
		while (isdigitByte(*u))
			++u;
		if (!*u)
			goto unparen;
		stringAndChar(&ns, &ns_l, (j == 2 ? ')' : ']'));
		break;
unparen:
/* ok, we can trash the original ( or [ */
		l = t->lic + j;
		strmove(ns + l - 1, ns + l);
		--ns_l;
		if (j == 2)
			stringAndChar(&ns, &ns_l, ' ');
		break;

	case TAGACT_AREA:
	case TAGACT_FRAME:
		if (!retainTag)
			break;

		if (t->f1 && !t->contracted) {	/* expanded frame */
			sprintf(hnum, "\r%c%d*%s\r", InternalCodeChar, tagno,
				(opentag ? "`--" : "--`"));
			ns_hnum();
			break;
		}

/* back to unexpanded frame or area */
		if (!opentag)
			break;
		liCheck(t);
		stringAndString(&ns, &ns_l,
				(action == TAGACT_FRAME ? "\rFrame " : "\r"));
// js often creates frames dynamically, so check for src
		if(allowJS && t->jslink) {
			nzFree(t->href);
			t->href = get_property_string_t(t,
			(t->action == TAGACT_AREA ? "href" : "src"));
		}
		a = 0;
		if (action == TAGACT_AREA)
			a = attribVal(t, "alt");
		u = (char *)a;
		if (!u) {
			u = t->name;
			if (!u)
				u = altText(t->href);
		}
		if (!u)
			u = (action == TAGACT_FRAME ? "???" : "area");
		if (t->href) {
			sprintf(hnum, "%c%d{", InternalCodeChar, tagno);
			ns_hnum();
		}
		if (t->href || action == TAGACT_FRAME)
			stringAndString(&ns, &ns_l, u);
		if (t->href) {
			ns_ic();
			stringAndString(&ns, &ns_l, "0}");
		}
		stringAndChar(&ns, &ns_l, '\r');
		if (t->f1 && t->contracted)	/* contracted frame */
			deltag = t;
		break;

	case TAGACT_MUSIC:
		liCheck(t);
		if (!retainTag)
			break;
		if (!t->href)
			break;
		sprintf(hnum, "\r%c%d{", InternalCodeChar, tagno);
		ns_hnum();
		stringAndString(&ns, &ns_l,
				(ti->name[0] ==
				 'b' ? "Background Music" : "Audio passage"));
		sprintf(hnum, "%c0}\r", InternalCodeChar);
		ns_hnum();
		break;

	case TAGACT_IMAGE:
		liCheck(t);
		tagInStream(tagno);
		if (!currentA) {
			if ((a = attribVal(t, "alt"))) {
				u = altText(a);
				a = NULL;
/* see if js has changed the alt tag */
				if (allowJS && t->jslink) {
					char *aa =
					    get_property_string_t(t, "alt");
					if (aa)
						u = altText(aa);
					nzFree(aa);
				}
				if (u && *u && !invisible) {
					stringAndChar(&ns, &ns_l, '[');
					stringAndString(&ns, &ns_l, u);
					stringAndChar(&ns, &ns_l, ']');
				}
			}
			break;
		}
/* image is part of a hyperlink */
		if (!retainTag || !currentA->href || currentA->textin)
			break;
		u = 0;
		a = attribVal(t, "alt");
		if (a)
			u = altText(a);
		if (!u)
			u = altText(t->name);
		if (!u)
			u = altText(currentA->href);
		if (!u)
			u = altText(t->href);
		if (!u)
			u = "image";
		stringAndString(&ns, &ns_l, u);
		break;

// This is for <unrecognized id=foo> and somewhere else <a href=#foo>
// We have to have this unknown tag in the stream or we can't jump to it.
	default:
		if(opentag && t->id)
			tagInStream(tagno);
		break;
	}			/* switch */
}

/* returns an allocated string */
char *render(int start)
{
	Frame *f;
	for (f = &cw->f0; f; f = f->next)
		if (f->cx)
			set_property_bool_win(f, "rr$start", true);
	ns = initString(&ns_l);
	invisible = false;
	inv2 = inv3 = NULL;
	listnest = 0;
	currentForm = currentA = NULL;
	traverse_callback = renderNode;
	traverseAll(start);
	return ns;
}

// Create buffers for text areas, so the user can type in comments or whatever
// and send them to the website in a fill-out form.
bool itext(int d)
{
	int ln = cw->dot;	// line number
	pst p;			// the raw line to scan
	int n;
	Tag *t;
	char newtext[20];
	bool change = false, inp = false;

	p = fetchLine(ln, -1);
	while (*p != '\n') {
		if (*p != InternalCodeChar) {
			++p;
			continue;
		}
		n = strtol((char *)p + 1, (char **)&p, 10);
		if (*p != '<')
			continue;
		inp = true;
		t = tagList[n];
		if (t->itype != INP_TA || t->lic)
			continue;
		t->lic = sideBuffer(d, t->value, -1, 0);
		change = true;
		sprintf(newtext, "buffer %d", t->lic);
// updateFieldInBuffer is crazy inefficient in that it searches through the
// whole buffer, and we know it's on the current line, but really, how often
// do you invoke this command?
		updateFieldInBuffer(n, newtext, false, false);
// And now all the pointers are invalid so break out.
// If there's another textarea on the same line you have to issue the command
// again, but really, how often does that happen?
		break;
	}

	if (change) {
		displayLine(ln);
		return true;
	}
		setError(inp ? MSG_NoChange : MSG_NoInputFields);
	return false;
}

struct htmlTag *line2tr(int ln)
{
	char *p;
struct htmlTag *t;
	int tagno;

	if(!ln) {
		setError(MSG_EmptyBuffer);
		return 0;
	}
	p = (char *)fetchLine(ln, -1);
	while(*p != '\n') {
		if (*p != InternalCodeChar) {
			++p;
			continue;
		}
		tagno = strtol((char *)p + 1, (char **)&p, 10);
/* could be 0, but should never be negative */
		if (tagno <= 0)
			continue;
		t = tagList[tagno];
		if(t->action != TAGACT_TD)
			continue;
		t = t->parent;
		if(t && t->action == TAGACT_TR)
			return t;
	}
	setError(MSG_NoTable);
	return 0;
}

static struct htmlTag *line2table(int ln)
{
	struct htmlTag *t;
	if(!ln) {
		setError(MSG_EmptyBuffer);
		return 0;
	}
	t = line2tr(ln);
	if(!t || !t->parent ||
	((t = t->parent)->action != TAGACT_TABLE &&
	t->action != TAGACT_THEAD &&
	t->action != TAGACT_TBODY)) {
		setError(MSG_NoTable);
		return 0;
	}
	if(t->action != TAGACT_TABLE) {
// it is tbody or thead
		t = t->parent;
		if(!t || t->action != TAGACT_TABLE) {
			setError(MSG_NoTable);
			return 0;
		}
	}
	return t;
}

// This routine is rather unforgiving.
// Has to look like <table><thead><tr><th>text</th>...
// Any intervening tags will throw it off.
// Clearly this routine has to be expanded to cover more html layouts.
bool showHeaders(int ln)
{
	const Tag *t = line2table(ln);
	int colno;
	if(!t)
		return false;
	for(t = t->firstchild; t; t = t->sibling) {
		if(t->action == TAGACT_THEAD)
			break;
	}
	if(!t)
		goto fail;
	t = t->firstchild;
	if(!t || t->action != TAGACT_TR || !t->firstchild)
		goto fail;
	t = t->firstchild;
	if(t->action != TAGACT_TD ||
	!stringEqual(t->info->name, "th"))
		goto fail;
	colno = 1;
	while(t) {
		if(t->action == TAGACT_TD) {
			printf("%d ", colno++);
			td_text = initString(&td_text_l);
			td_textUnder(t);
			if(!td_text_l) {
				printf("?");
			} else {
				printf("%s", td_text);
				nzFree(td_text);
			}
			printf("\n");
			}
		t = t->sibling;
	}
	return true;

fail:
	setError(MSG_NoColumnHeaders);
	return false;
}

