/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <math.h>
#include <limits.h>
#include <values.h>
#include <stdio.h>
#include "exrinput.h"

/**
 * Luminance function for sRGB.
 */
static double luminance( double r, double g, double b )
{
    return ( 0.2126 * r ) + ( 0.7152 * g ) + ( 0.0722 * b );
}

int main( int argc, char **argv )
{
    exrinput_t *exrin;
    int width, height;
    double minr, maxr;
    double ming, maxg;
    double minb, maxb;
    double mina, maxa;
    double minl, maxl;
    double lav, rav, gav, bav;
    double llav, lrav, lgav, lbav;
    int y;

    if( argc < 2 ) {
        fprintf( stderr, "Usage: exrstats image.exr\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrstats: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );

    fprintf( stdout, "\n%s [%dx%d]:\n\n", argv[ 1 ], width, height );

    minl = minr = ming = minb = mina =  DBL_MAX;
    maxl = maxr = maxg = maxb = maxa = -DBL_MAX;
    lav = rav = gav = bav = 0.0;
    llav = lrav = lgav = lbav = 0.0;

    for( y = 0; y < height; y++ ) {
        double *scanline = exrinput_get_scanline( exrin, y );
        int x;

        fprintf( stderr, "exrstats: Scanning %d\r", y );
        for( x = 0; x < width; x++ ) {
            double r = scanline[ (x * 4) + 0 ];
            double g = scanline[ (x * 4) + 1 ];
            double b = scanline[ (x * 4) + 2 ];
            double a = scanline[ (x * 4) + 3 ];
            double Y = luminance( r, g, b );

            if( Y > 0.0 ) {
                llav += log( Y );
                if( Y < minl ) minl = Y; if( Y > maxl ) maxl = Y;
            }
            if( r > 0.0 ) {
                lrav += log( r );
                if( r < minr ) minr = r; if( r > maxr ) maxr = r;
            }
            if( g > 0.0 ) {
                lgav += log( g );
                if( g < ming ) ming = g; if( g > maxg ) maxg = g;
            }
            if( b > 0.0 ) {
                lbav += log( b );
                if( b < minb ) minb = b; if( b > maxb ) maxb = b;
            }

            rav += r;
            gav += g;
            bav += b;
            lav += b;

            if( a < mina ) mina = a; if( a > maxa ) maxa = a;
        }
    }
    fprintf( stderr, "\nexrstats: Done.\n\n" );

    llav /= (double) (width * height);
    lrav /= (double) (width * height);
    lgav /= (double) (width * height);
    lbav /= (double) (width * height);
    lav /= (double) (width * height);
    rav /= (double) (width * height);
    gav /= (double) (width * height);
    bav /= (double) (width * height);

    fprintf( stdout, "  Alpha     min is %7.3f, max %7.3f\n\n", mina, maxa );

    fprintf( stdout, "  Luminance min is %7.3f, avg %7.3f, max %7.3f\n",
             minl, lav, maxl );
    fprintf( stdout, "  Red       min is %7.3f, avg %7.3f, max %7.3f\n",
             minr, rav, maxr );
    fprintf( stdout, "  Green     min is %7.3f, avg %7.3f, max %7.3f\n",
             ming, gav, maxg );
    fprintf( stdout, "  Blue      min is %7.3f, avg %7.3f, max %7.3f\n\n",
             minb, bav, maxb );

    fprintf( stdout, "  Luminance log min is %7.3f, avg log %7.3f, log max %7.3f\n",
             log( minl ), llav, log( maxl ) );
    fprintf( stdout, "  Red       log min is %7.3f, avg log %7.3f, log max %7.3f\n",
             log( minr ), lrav, log( maxr ) );
    fprintf( stdout, "  Green     log min is %7.3f, avg log %7.3f, log max %7.3f\n",
             log( ming ), lgav, log( maxg ) );
    fprintf( stdout, "  Blue      log min is %7.3f, avg log %7.3f, log max %7.3f\n",
             log( minb ), lbav, log( maxb ) );

    exrinput_delete( exrin );
    return 0;
}

