/* $Id: StandardTraversal.hpp 5006 2010-10-22 11:11:46Z potyra $ 
 *
 * Perform a standard traversal on the intermediate code.
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __STANDARD_TRAVERSAL_HPP_INCLUDED
#define __STANDARD_TRAVERSAL_HPP_INCLUDED

#include "intermediate/visitor/Visitor.hpp"

namespace intermediate {

//! Perform a standard traversal on the intermediate code.
/** This visitor will perform a standard traversal on the intermediate
 *  code. Visitors, that are only interested in a subset of the nodes
 *  can inherit from this visitor.
 */
class StandardTraversal : public Visitor {
public:
	//! c'tor
	StandardTraversal();

protected:
	//! visit a CodeContainer node
	/** @param node node that gets visited.
	 */
	virtual void visit(CodeContainer &node);

	//! visit a ImmediateOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(ImmediateOperand &node);

	//! visit a BeginTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(BeginTransfer &node);

	//! visit an EndTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(EndTransfer &node);

	//! visit a SetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(SetParam &node);

	//! visit a GetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetParam &node);

	//! visit a Connect node
	/** @param node node that gets visited.
	 */
	virtual void visit(Connect &node);

	//! visit a Mov node
	/** @param node node that gets visited.
	 */
	virtual void visit(Mov &node);

	//! visit a Je node
	/** @param node node that gets visited.
	 */
	virtual void visit(Je &node);

	//! visit a Jbe node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jbe &node);

	//! visit a Jne node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jne &node);

	//! visit a Jb node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jb &node);

	//! visit a Jmp node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jmp &node);

	//! visit a Label node
	/** @param node node that gets visited.
	 */
	virtual void visit(Label &node);

	//! visit an Abort node
	/** @param node node that gets visited.
	 */
	virtual void visit(Abort &node);

	//! visit an Add node
	/** @param node node that gets visited.
	 */
	virtual void visit(Add &node);

	//! visit a Sub node
	/** @param node node that gets visited.
	 */
	virtual void visit(Sub &node);

	//! visit a Call node
	/** @param node node that gets visited.
	 */
	virtual void visit(Call &node);

	//! visit a Return node
	/** @param node node that gets visited.
	 */
	virtual void visit(Return &node);

	//! visit a Proc node
	/** @param node node that gets visited.
	 */
	virtual void visit(Proc &node);

	//! visit an Update node
	/** @param node node that gets visited.
	 */
	virtual void visit(Update &node);

	//! visit a GetSig node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetSig &node);

	//! visit a GetSimTime node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetSimTime &node);

	//! visit an IMul node
	/** @param node node that gets visited.
	 */
	virtual void visit(IMul &node);

	//! visit a Div node
	/** @param node node that gets visited.
	 */
	virtual void visit(Div &node);

	//! visit an ROffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(ROffset &node);

	//! visit an AOffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(AOffset &node);

	//! visit a Suspend node
	/** @param node node that gets visited.
	 */
	virtual void visit(Suspend &node);

	//! visit a WakeOn node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeOn &node);

	//! visit a WakeAt node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeAt &node);

	//! visit a Log node
	/** @param node node that gets visited.
	 */
	virtual void visit(Log &node);

	//! visit an IndirectOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(IndirectOperand &node);

	//! visit a Reference node
	/** @param node node that gets visited.
	 */
	virtual void visit(Reference &node);

	//! visit a Register node
	/** @param node node that gets visited.
	 */
	virtual void visit(Register &node);

	//! visit a Data node
	/** @param node node that gets visited.
	 */
	virtual void visit(Data &node);

	//! visit a TypeElement node
	/** @param node node that gets visited.
	 */
	virtual void visit(TypeElement &node);

	//! visit a Type node
	/** @param node node that gets visited.
	 */
	virtual void visit(Type &node);

private:
	//! traverse a conditional jump
	/** @param node conditional jump node.
	 */
	template <typename T>
	void traverseCondJmp(T &node);

	//! traverse a binary operator
	/** @param node binary operator.
	 */
	template <typename T>
	void traverseBinOp(T &node);
}; 

}; /* namespace intermediate */

#endif /* __STANDARD_TRAVERSAL_HPP_INCLUDED */
