{- |
   Module      : Text.Highlighting.Kate.Syntax
   Copyright   : Copyright (C) 2008-2011 John MacFarlane
   License     : GNU GPL, version 2 or above

   Maintainer  : John MacFarlane <jgm@berkeley.edu>
   Stability   : alpha
   Portability : portable

Token lexer for various languages.
-}

module Text.Highlighting.Kate.Syntax ( highlightAs,
                                       languages,
                                       languageByFullName,
                                       languagesByExtension,
                                       languagesByFilename ) where
import Data.Char (toLower)
import Text.Highlighting.Kate.Types
import Text.Highlighting.Kate.Common (matchGlobs)
import qualified Text.Highlighting.Kate.Syntax.Abc as Abc
import qualified Text.Highlighting.Kate.Syntax.Actionscript as Actionscript
import qualified Text.Highlighting.Kate.Syntax.Ada as Ada
import qualified Text.Highlighting.Kate.Syntax.Agda as Agda
import qualified Text.Highlighting.Kate.Syntax.Alert as Alert
import qualified Text.Highlighting.Kate.Syntax.Alert_indent as Alert_indent
import qualified Text.Highlighting.Kate.Syntax.Apache as Apache
import qualified Text.Highlighting.Kate.Syntax.Asn1 as Asn1
import qualified Text.Highlighting.Kate.Syntax.Asp as Asp
import qualified Text.Highlighting.Kate.Syntax.Awk as Awk
import qualified Text.Highlighting.Kate.Syntax.Bash as Bash
import qualified Text.Highlighting.Kate.Syntax.Bibtex as Bibtex
import qualified Text.Highlighting.Kate.Syntax.Boo as Boo
import qualified Text.Highlighting.Kate.Syntax.C as C
import qualified Text.Highlighting.Kate.Syntax.Changelog as Changelog
import qualified Text.Highlighting.Kate.Syntax.Clojure as Clojure
import qualified Text.Highlighting.Kate.Syntax.Cmake as Cmake
import qualified Text.Highlighting.Kate.Syntax.Coffee as Coffee
import qualified Text.Highlighting.Kate.Syntax.Coldfusion as Coldfusion
import qualified Text.Highlighting.Kate.Syntax.Commonlisp as Commonlisp
import qualified Text.Highlighting.Kate.Syntax.Cpp as Cpp
import qualified Text.Highlighting.Kate.Syntax.Cs as Cs
import qualified Text.Highlighting.Kate.Syntax.Css as Css
import qualified Text.Highlighting.Kate.Syntax.Curry as Curry
import qualified Text.Highlighting.Kate.Syntax.D as D
import qualified Text.Highlighting.Kate.Syntax.Diff as Diff
import qualified Text.Highlighting.Kate.Syntax.Djangotemplate as Djangotemplate
import qualified Text.Highlighting.Kate.Syntax.Dockerfile as Dockerfile
import qualified Text.Highlighting.Kate.Syntax.Dot as Dot
import qualified Text.Highlighting.Kate.Syntax.Doxygen as Doxygen
import qualified Text.Highlighting.Kate.Syntax.Doxygenlua as Doxygenlua
import qualified Text.Highlighting.Kate.Syntax.Dtd as Dtd
import qualified Text.Highlighting.Kate.Syntax.Eiffel as Eiffel
import qualified Text.Highlighting.Kate.Syntax.Email as Email
import qualified Text.Highlighting.Kate.Syntax.Erlang as Erlang
import qualified Text.Highlighting.Kate.Syntax.Fasm as Fasm
import qualified Text.Highlighting.Kate.Syntax.Fortran as Fortran
import qualified Text.Highlighting.Kate.Syntax.Fsharp as Fsharp
import qualified Text.Highlighting.Kate.Syntax.Gcc as Gcc
import qualified Text.Highlighting.Kate.Syntax.Glsl as Glsl
import qualified Text.Highlighting.Kate.Syntax.Gnuassembler as Gnuassembler
import qualified Text.Highlighting.Kate.Syntax.Go as Go
import qualified Text.Highlighting.Kate.Syntax.Haskell as Haskell
import qualified Text.Highlighting.Kate.Syntax.Haxe as Haxe
import qualified Text.Highlighting.Kate.Syntax.Html as Html
import qualified Text.Highlighting.Kate.Syntax.Idris as Idris
import qualified Text.Highlighting.Kate.Syntax.Ini as Ini
import qualified Text.Highlighting.Kate.Syntax.Isocpp as Isocpp
import qualified Text.Highlighting.Kate.Syntax.Java as Java
import qualified Text.Highlighting.Kate.Syntax.Javadoc as Javadoc
import qualified Text.Highlighting.Kate.Syntax.Javascript as Javascript
import qualified Text.Highlighting.Kate.Syntax.Json as Json
import qualified Text.Highlighting.Kate.Syntax.Jsp as Jsp
import qualified Text.Highlighting.Kate.Syntax.Julia as Julia
import qualified Text.Highlighting.Kate.Syntax.Kotlin as Kotlin
import qualified Text.Highlighting.Kate.Syntax.Latex as Latex
import qualified Text.Highlighting.Kate.Syntax.Lex as Lex
import qualified Text.Highlighting.Kate.Syntax.Lilypond as Lilypond
import qualified Text.Highlighting.Kate.Syntax.LiterateCurry as LiterateCurry
import qualified Text.Highlighting.Kate.Syntax.LiterateHaskell as LiterateHaskell
import qualified Text.Highlighting.Kate.Syntax.Llvm as Llvm
import qualified Text.Highlighting.Kate.Syntax.Lua as Lua
import qualified Text.Highlighting.Kate.Syntax.M4 as M4
import qualified Text.Highlighting.Kate.Syntax.Makefile as Makefile
import qualified Text.Highlighting.Kate.Syntax.Mandoc as Mandoc
import qualified Text.Highlighting.Kate.Syntax.Markdown as Markdown
import qualified Text.Highlighting.Kate.Syntax.Mathematica as Mathematica
import qualified Text.Highlighting.Kate.Syntax.Matlab as Matlab
import qualified Text.Highlighting.Kate.Syntax.Maxima as Maxima
import qualified Text.Highlighting.Kate.Syntax.Mediawiki as Mediawiki
import qualified Text.Highlighting.Kate.Syntax.Metafont as Metafont
import qualified Text.Highlighting.Kate.Syntax.Mips as Mips
import qualified Text.Highlighting.Kate.Syntax.Modelines as Modelines
import qualified Text.Highlighting.Kate.Syntax.Modula2 as Modula2
import qualified Text.Highlighting.Kate.Syntax.Modula3 as Modula3
import qualified Text.Highlighting.Kate.Syntax.Monobasic as Monobasic
import qualified Text.Highlighting.Kate.Syntax.Nasm as Nasm
import qualified Text.Highlighting.Kate.Syntax.Noweb as Noweb
import qualified Text.Highlighting.Kate.Syntax.Objectivec as Objectivec
import qualified Text.Highlighting.Kate.Syntax.Objectivecpp as Objectivecpp
import qualified Text.Highlighting.Kate.Syntax.Ocaml as Ocaml
import qualified Text.Highlighting.Kate.Syntax.Octave as Octave
import qualified Text.Highlighting.Kate.Syntax.Opencl as Opencl
import qualified Text.Highlighting.Kate.Syntax.Pascal as Pascal
import qualified Text.Highlighting.Kate.Syntax.Perl as Perl
import qualified Text.Highlighting.Kate.Syntax.Php as Php
import qualified Text.Highlighting.Kate.Syntax.Pike as Pike
import qualified Text.Highlighting.Kate.Syntax.Postscript as Postscript
import qualified Text.Highlighting.Kate.Syntax.Prolog as Prolog
import qualified Text.Highlighting.Kate.Syntax.Pure as Pure
import qualified Text.Highlighting.Kate.Syntax.Python as Python
import qualified Text.Highlighting.Kate.Syntax.R as R
import qualified Text.Highlighting.Kate.Syntax.Relaxng as Relaxng
import qualified Text.Highlighting.Kate.Syntax.Relaxngcompact as Relaxngcompact
import qualified Text.Highlighting.Kate.Syntax.Rest as Rest
import qualified Text.Highlighting.Kate.Syntax.Rhtml as Rhtml
import qualified Text.Highlighting.Kate.Syntax.Roff as Roff
import qualified Text.Highlighting.Kate.Syntax.Ruby as Ruby
import qualified Text.Highlighting.Kate.Syntax.Rust as Rust
import qualified Text.Highlighting.Kate.Syntax.Scala as Scala
import qualified Text.Highlighting.Kate.Syntax.Scheme as Scheme
import qualified Text.Highlighting.Kate.Syntax.Sci as Sci
import qualified Text.Highlighting.Kate.Syntax.Sed as Sed
import qualified Text.Highlighting.Kate.Syntax.Sgml as Sgml
import qualified Text.Highlighting.Kate.Syntax.Sql as Sql
import qualified Text.Highlighting.Kate.Syntax.SqlMysql as SqlMysql
import qualified Text.Highlighting.Kate.Syntax.SqlPostgresql as SqlPostgresql
import qualified Text.Highlighting.Kate.Syntax.Tcl as Tcl
import qualified Text.Highlighting.Kate.Syntax.Tcsh as Tcsh
import qualified Text.Highlighting.Kate.Syntax.Texinfo as Texinfo
import qualified Text.Highlighting.Kate.Syntax.Verilog as Verilog
import qualified Text.Highlighting.Kate.Syntax.Vhdl as Vhdl
import qualified Text.Highlighting.Kate.Syntax.Xml as Xml
import qualified Text.Highlighting.Kate.Syntax.Xorg as Xorg
import qualified Text.Highlighting.Kate.Syntax.Xslt as Xslt
import qualified Text.Highlighting.Kate.Syntax.Xul as Xul
import qualified Text.Highlighting.Kate.Syntax.Yacc as Yacc
import qualified Text.Highlighting.Kate.Syntax.Yaml as Yaml
import qualified Text.Highlighting.Kate.Syntax.Zsh as Zsh

-- | List of supported languages.
languages :: [String]
languages = ["Abc","Actionscript","Ada","Agda","Alert","Alert_indent","Apache","Asn1","Asp","Awk","Bash","Bibtex","Boo","C","Changelog","Clojure","Cmake","Coffee","Coldfusion","Commonlisp","Cpp","Cs","Css","Curry","D","Diff","Djangotemplate","Dockerfile","Dot","Doxygen","Doxygenlua","Dtd","Eiffel","Email","Erlang","Fasm","Fortran","Fsharp","Gcc","Glsl","Gnuassembler","Go","Haskell","Haxe","Html","Idris","Ini","Isocpp","Java","Javadoc","Javascript","Json","Jsp","Julia","Kotlin","Latex","Lex","Lilypond","LiterateCurry","LiterateHaskell","Llvm","Lua","M4","Makefile","Mandoc","Markdown","Mathematica","Matlab","Maxima","Mediawiki","Metafont","Mips","Modelines","Modula2","Modula3","Monobasic","Nasm","Noweb","Objectivec","Objectivecpp","Ocaml","Octave","Opencl","Pascal","Perl","Php","Pike","Postscript","Prolog","Pure","Python","R","Relaxng","Relaxngcompact","Rest","Rhtml","Roff","Ruby","Rust","Scala","Scheme","Sci","Sed","Sgml","Sql","SqlMysql","SqlPostgresql","Tcl","Tcsh","Texinfo","Verilog","Vhdl","Xml","Xorg","Xslt","Xul","Yacc","Yaml","Zsh"]

-- | List of language extensions.
languageExtensions :: [(String, String)]
languageExtensions = [("Abc", Abc.syntaxExtensions), ("Actionscript", Actionscript.syntaxExtensions), ("Ada", Ada.syntaxExtensions), ("Agda", Agda.syntaxExtensions), ("Alert", Alert.syntaxExtensions), ("Alert_indent", Alert_indent.syntaxExtensions), ("Apache", Apache.syntaxExtensions), ("Asn1", Asn1.syntaxExtensions), ("Asp", Asp.syntaxExtensions), ("Awk", Awk.syntaxExtensions), ("Bash", Bash.syntaxExtensions), ("Bibtex", Bibtex.syntaxExtensions), ("Boo", Boo.syntaxExtensions), ("C", C.syntaxExtensions), ("Changelog", Changelog.syntaxExtensions), ("Clojure", Clojure.syntaxExtensions), ("Cmake", Cmake.syntaxExtensions), ("Coffee", Coffee.syntaxExtensions), ("Coldfusion", Coldfusion.syntaxExtensions), ("Commonlisp", Commonlisp.syntaxExtensions), ("Cpp", Cpp.syntaxExtensions), ("Cs", Cs.syntaxExtensions), ("Css", Css.syntaxExtensions), ("Curry", Curry.syntaxExtensions), ("D", D.syntaxExtensions), ("Diff", Diff.syntaxExtensions), ("Djangotemplate", Djangotemplate.syntaxExtensions), ("Dockerfile", Dockerfile.syntaxExtensions), ("Dot", Dot.syntaxExtensions), ("Doxygen", Doxygen.syntaxExtensions), ("Doxygenlua", Doxygenlua.syntaxExtensions), ("Dtd", Dtd.syntaxExtensions), ("Eiffel", Eiffel.syntaxExtensions), ("Email", Email.syntaxExtensions), ("Erlang", Erlang.syntaxExtensions), ("Fasm", Fasm.syntaxExtensions), ("Fortran", Fortran.syntaxExtensions), ("Fsharp", Fsharp.syntaxExtensions), ("Gcc", Gcc.syntaxExtensions), ("Glsl", Glsl.syntaxExtensions), ("Gnuassembler", Gnuassembler.syntaxExtensions), ("Go", Go.syntaxExtensions), ("Haskell", Haskell.syntaxExtensions), ("Haxe", Haxe.syntaxExtensions), ("Html", Html.syntaxExtensions), ("Idris", Idris.syntaxExtensions), ("Ini", Ini.syntaxExtensions), ("Isocpp", Isocpp.syntaxExtensions), ("Java", Java.syntaxExtensions), ("Javadoc", Javadoc.syntaxExtensions), ("Javascript", Javascript.syntaxExtensions), ("Json", Json.syntaxExtensions), ("Jsp", Jsp.syntaxExtensions), ("Julia", Julia.syntaxExtensions), ("Kotlin", Kotlin.syntaxExtensions), ("Latex", Latex.syntaxExtensions), ("Lex", Lex.syntaxExtensions), ("Lilypond", Lilypond.syntaxExtensions), ("LiterateCurry", LiterateCurry.syntaxExtensions), ("LiterateHaskell", LiterateHaskell.syntaxExtensions), ("Llvm", Llvm.syntaxExtensions), ("Lua", Lua.syntaxExtensions), ("M4", M4.syntaxExtensions), ("Makefile", Makefile.syntaxExtensions), ("Mandoc", Mandoc.syntaxExtensions), ("Markdown", Markdown.syntaxExtensions), ("Mathematica", Mathematica.syntaxExtensions), ("Matlab", Matlab.syntaxExtensions), ("Maxima", Maxima.syntaxExtensions), ("Mediawiki", Mediawiki.syntaxExtensions), ("Metafont", Metafont.syntaxExtensions), ("Mips", Mips.syntaxExtensions), ("Modelines", Modelines.syntaxExtensions), ("Modula2", Modula2.syntaxExtensions), ("Modula3", Modula3.syntaxExtensions), ("Monobasic", Monobasic.syntaxExtensions), ("Nasm", Nasm.syntaxExtensions), ("Noweb", Noweb.syntaxExtensions), ("Objectivec", Objectivec.syntaxExtensions), ("Objectivecpp", Objectivecpp.syntaxExtensions), ("Ocaml", Ocaml.syntaxExtensions), ("Octave", Octave.syntaxExtensions), ("Opencl", Opencl.syntaxExtensions), ("Pascal", Pascal.syntaxExtensions), ("Perl", Perl.syntaxExtensions), ("Php", Php.syntaxExtensions), ("Pike", Pike.syntaxExtensions), ("Postscript", Postscript.syntaxExtensions), ("Prolog", Prolog.syntaxExtensions), ("Pure", Pure.syntaxExtensions), ("Python", Python.syntaxExtensions), ("R", R.syntaxExtensions), ("Relaxng", Relaxng.syntaxExtensions), ("Relaxngcompact", Relaxngcompact.syntaxExtensions), ("Rest", Rest.syntaxExtensions), ("Rhtml", Rhtml.syntaxExtensions), ("Roff", Roff.syntaxExtensions), ("Ruby", Ruby.syntaxExtensions), ("Rust", Rust.syntaxExtensions), ("Scala", Scala.syntaxExtensions), ("Scheme", Scheme.syntaxExtensions), ("Sci", Sci.syntaxExtensions), ("Sed", Sed.syntaxExtensions), ("Sgml", Sgml.syntaxExtensions), ("Sql", Sql.syntaxExtensions), ("SqlMysql", SqlMysql.syntaxExtensions), ("SqlPostgresql", SqlPostgresql.syntaxExtensions), ("Tcl", Tcl.syntaxExtensions), ("Tcsh", Tcsh.syntaxExtensions), ("Texinfo", Texinfo.syntaxExtensions), ("Verilog", Verilog.syntaxExtensions), ("Vhdl", Vhdl.syntaxExtensions), ("Xml", Xml.syntaxExtensions), ("Xorg", Xorg.syntaxExtensions), ("Xslt", Xslt.syntaxExtensions), ("Xul", Xul.syntaxExtensions), ("Yacc", Yacc.syntaxExtensions), ("Yaml", Yaml.syntaxExtensions), ("Zsh", Zsh.syntaxExtensions)]

-- | List of full names of languages.
languageFullNames :: [(String, String)]
languageFullNames = [("Abc", Abc.syntaxName), ("Actionscript", Actionscript.syntaxName), ("Ada", Ada.syntaxName), ("Agda", Agda.syntaxName), ("Alert", Alert.syntaxName), ("Alert_indent", Alert_indent.syntaxName), ("Apache", Apache.syntaxName), ("Asn1", Asn1.syntaxName), ("Asp", Asp.syntaxName), ("Awk", Awk.syntaxName), ("Bash", Bash.syntaxName), ("Bibtex", Bibtex.syntaxName), ("Boo", Boo.syntaxName), ("C", C.syntaxName), ("Changelog", Changelog.syntaxName), ("Clojure", Clojure.syntaxName), ("Cmake", Cmake.syntaxName), ("Coffee", Coffee.syntaxName), ("Coldfusion", Coldfusion.syntaxName), ("Commonlisp", Commonlisp.syntaxName), ("Cpp", Cpp.syntaxName), ("Cs", Cs.syntaxName), ("Css", Css.syntaxName), ("Curry", Curry.syntaxName), ("D", D.syntaxName), ("Diff", Diff.syntaxName), ("Djangotemplate", Djangotemplate.syntaxName), ("Dockerfile", Dockerfile.syntaxName), ("Dot", Dot.syntaxName), ("Doxygen", Doxygen.syntaxName), ("Doxygenlua", Doxygenlua.syntaxName), ("Dtd", Dtd.syntaxName), ("Eiffel", Eiffel.syntaxName), ("Email", Email.syntaxName), ("Erlang", Erlang.syntaxName), ("Fasm", Fasm.syntaxName), ("Fortran", Fortran.syntaxName), ("Fsharp", Fsharp.syntaxName), ("Gcc", Gcc.syntaxName), ("Glsl", Glsl.syntaxName), ("Gnuassembler", Gnuassembler.syntaxName), ("Go", Go.syntaxName), ("Haskell", Haskell.syntaxName), ("Haxe", Haxe.syntaxName), ("Html", Html.syntaxName), ("Idris", Idris.syntaxName), ("Ini", Ini.syntaxName), ("Isocpp", Isocpp.syntaxName), ("Java", Java.syntaxName), ("Javadoc", Javadoc.syntaxName), ("Javascript", Javascript.syntaxName), ("Json", Json.syntaxName), ("Jsp", Jsp.syntaxName), ("Julia", Julia.syntaxName), ("Kotlin", Kotlin.syntaxName), ("Latex", Latex.syntaxName), ("Lex", Lex.syntaxName), ("Lilypond", Lilypond.syntaxName), ("LiterateCurry", LiterateCurry.syntaxName), ("LiterateHaskell", LiterateHaskell.syntaxName), ("Llvm", Llvm.syntaxName), ("Lua", Lua.syntaxName), ("M4", M4.syntaxName), ("Makefile", Makefile.syntaxName), ("Mandoc", Mandoc.syntaxName), ("Markdown", Markdown.syntaxName), ("Mathematica", Mathematica.syntaxName), ("Matlab", Matlab.syntaxName), ("Maxima", Maxima.syntaxName), ("Mediawiki", Mediawiki.syntaxName), ("Metafont", Metafont.syntaxName), ("Mips", Mips.syntaxName), ("Modelines", Modelines.syntaxName), ("Modula2", Modula2.syntaxName), ("Modula3", Modula3.syntaxName), ("Monobasic", Monobasic.syntaxName), ("Nasm", Nasm.syntaxName), ("Noweb", Noweb.syntaxName), ("Objectivec", Objectivec.syntaxName), ("Objectivecpp", Objectivecpp.syntaxName), ("Ocaml", Ocaml.syntaxName), ("Octave", Octave.syntaxName), ("Opencl", Opencl.syntaxName), ("Pascal", Pascal.syntaxName), ("Perl", Perl.syntaxName), ("Php", Php.syntaxName), ("Pike", Pike.syntaxName), ("Postscript", Postscript.syntaxName), ("Prolog", Prolog.syntaxName), ("Pure", Pure.syntaxName), ("Python", Python.syntaxName), ("R", R.syntaxName), ("Relaxng", Relaxng.syntaxName), ("Relaxngcompact", Relaxngcompact.syntaxName), ("Rest", Rest.syntaxName), ("Rhtml", Rhtml.syntaxName), ("Roff", Roff.syntaxName), ("Ruby", Ruby.syntaxName), ("Rust", Rust.syntaxName), ("Scala", Scala.syntaxName), ("Scheme", Scheme.syntaxName), ("Sci", Sci.syntaxName), ("Sed", Sed.syntaxName), ("Sgml", Sgml.syntaxName), ("Sql", Sql.syntaxName), ("SqlMysql", SqlMysql.syntaxName), ("SqlPostgresql", SqlPostgresql.syntaxName), ("Tcl", Tcl.syntaxName), ("Tcsh", Tcsh.syntaxName), ("Texinfo", Texinfo.syntaxName), ("Verilog", Verilog.syntaxName), ("Vhdl", Vhdl.syntaxName), ("Xml", Xml.syntaxName), ("Xorg", Xorg.syntaxName), ("Xslt", Xslt.syntaxName), ("Xul", Xul.syntaxName), ("Yacc", Yacc.syntaxName), ("Yaml", Yaml.syntaxName), ("Zsh", Zsh.syntaxName)]

languageShortNames :: [(String, String)]
languageShortNames =
  [(map toLower y, map toLower x) | (x, y) <- languageFullNames]

-- | Lookup canonical language name by full syntaxName (e.g. "C#" for "Cs").
languageByFullName :: String -> Maybe String
languageByFullName s = lookup s languageShortNames

-- | Returns a list of languages appropriate for the given file extension.
languagesByExtension :: String -> [String]
languagesByExtension ('.':ext) = languagesByFilename ("*." ++ ext)
languagesByExtension ext       = languagesByFilename ("*." ++ ext)

-- | Returns a list of languages appropriate for the given filename.
languagesByFilename :: FilePath -> [String]
languagesByFilename fn = [lang | (lang, globs) <- languageExtensions, matchGlobs fn globs]

-- | Highlight source code. The source language may be specified
-- by its canonical name (case-insensitive) or by a canonical
-- extension (if unique).
-- The parsers read the input lazily and parse line by line;
-- results are returned immediately.
-- Supported languages: @abc@, @actionscript@, @ada@, @agda@, @alert@, @alert_indent@, @apache@, @asn1@, @asp@, @awk@, @bash@, @bibtex@, @boo@, @c@, @changelog@, @clojure@, @cmake@, @coffee@, @coldfusion@, @commonlisp@, @cpp@, @cs@, @css@, @curry@, @d@, @diff@, @djangotemplate@, @dockerfile@, @dot@, @doxygen@, @doxygenlua@, @dtd@, @eiffel@, @email@, @erlang@, @fasm@, @fortran@, @fsharp@, @gcc@, @glsl@, @gnuassembler@, @go@, @haskell@, @haxe@, @html@, @idris@, @ini@, @isocpp@, @java@, @javadoc@, @javascript@, @json@, @jsp@, @julia@, @kotlin@, @latex@, @lex@, @lilypond@, @literatecurry@, @literatehaskell@, @llvm@, @lua@, @m4@, @makefile@, @mandoc@, @markdown@, @mathematica@, @matlab@, @maxima@, @mediawiki@, @metafont@, @mips@, @modelines@, @modula2@, @modula3@, @monobasic@, @nasm@, @noweb@, @objectivec@, @objectivecpp@, @ocaml@, @octave@, @opencl@, @pascal@, @perl@, @php@, @pike@, @postscript@, @prolog@, @pure@, @python@, @r@, @relaxng@, @relaxngcompact@, @rest@, @rhtml@, @roff@, @ruby@, @rust@, @scala@, @scheme@, @sci@, @sed@, @sgml@, @sql@, @sqlmysql@, @sqlpostgresql@, @tcl@, @tcsh@, @texinfo@, @verilog@, @vhdl@, @xml@, @xorg@, @xslt@, @xul@, @yacc@, @yaml@, @zsh@.
highlightAs :: String         -- ^ Language syntax (e.g. "haskell") or extension (e.g. "hs").
            -> String         -- ^ Source code to highlight
            -> [SourceLine]   -- ^ List of highlighted source lines
highlightAs lang =
  let lang'  = map toLower lang
      lang'' = if lang' `elem` map (map toLower) languages
                  then lang'
                  else case languageByFullName lang' of
                            Just l -> l
                            Nothing ->
                              case languagesByExtension lang' of
                                    -- go by extension if unambiguous
                                    [l]  -> map toLower l
                                    _    -> lang'
  in  case lang'' of
        "abc" -> Abc.highlight
        "actionscript" -> Actionscript.highlight
        "ada" -> Ada.highlight
        "agda" -> Agda.highlight
        "alert" -> Alert.highlight
        "alert_indent" -> Alert_indent.highlight
        "apache" -> Apache.highlight
        "asn1" -> Asn1.highlight
        "asp" -> Asp.highlight
        "awk" -> Awk.highlight
        "bash" -> Bash.highlight
        "bibtex" -> Bibtex.highlight
        "boo" -> Boo.highlight
        "c" -> C.highlight
        "changelog" -> Changelog.highlight
        "clojure" -> Clojure.highlight
        "cmake" -> Cmake.highlight
        "coffee" -> Coffee.highlight
        "coldfusion" -> Coldfusion.highlight
        "commonlisp" -> Commonlisp.highlight
        "cpp" -> Cpp.highlight
        "cs" -> Cs.highlight
        "css" -> Css.highlight
        "curry" -> Curry.highlight
        "d" -> D.highlight
        "diff" -> Diff.highlight
        "djangotemplate" -> Djangotemplate.highlight
        "dockerfile" -> Dockerfile.highlight
        "dot" -> Dot.highlight
        "doxygen" -> Doxygen.highlight
        "doxygenlua" -> Doxygenlua.highlight
        "dtd" -> Dtd.highlight
        "eiffel" -> Eiffel.highlight
        "email" -> Email.highlight
        "erlang" -> Erlang.highlight
        "fasm" -> Fasm.highlight
        "fortran" -> Fortran.highlight
        "fsharp" -> Fsharp.highlight
        "gcc" -> Gcc.highlight
        "glsl" -> Glsl.highlight
        "gnuassembler" -> Gnuassembler.highlight
        "go" -> Go.highlight
        "haskell" -> Haskell.highlight
        "haxe" -> Haxe.highlight
        "html" -> Html.highlight
        "idris" -> Idris.highlight
        "ini" -> Ini.highlight
        "isocpp" -> Isocpp.highlight
        "java" -> Java.highlight
        "javadoc" -> Javadoc.highlight
        "javascript" -> Javascript.highlight
        "json" -> Json.highlight
        "jsp" -> Jsp.highlight
        "julia" -> Julia.highlight
        "kotlin" -> Kotlin.highlight
        "latex" -> Latex.highlight
        "lex" -> Lex.highlight
        "lilypond" -> Lilypond.highlight
        "literatecurry" -> LiterateCurry.highlight
        "literatehaskell" -> LiterateHaskell.highlight
        "llvm" -> Llvm.highlight
        "lua" -> Lua.highlight
        "m4" -> M4.highlight
        "makefile" -> Makefile.highlight
        "mandoc" -> Mandoc.highlight
        "markdown" -> Markdown.highlight
        "mathematica" -> Mathematica.highlight
        "matlab" -> Matlab.highlight
        "maxima" -> Maxima.highlight
        "mediawiki" -> Mediawiki.highlight
        "metafont" -> Metafont.highlight
        "mips" -> Mips.highlight
        "modelines" -> Modelines.highlight
        "modula2" -> Modula2.highlight
        "modula3" -> Modula3.highlight
        "monobasic" -> Monobasic.highlight
        "nasm" -> Nasm.highlight
        "noweb" -> Noweb.highlight
        "objectivec" -> Objectivec.highlight
        "objectivecpp" -> Objectivecpp.highlight
        "ocaml" -> Ocaml.highlight
        "octave" -> Octave.highlight
        "opencl" -> Opencl.highlight
        "pascal" -> Pascal.highlight
        "perl" -> Perl.highlight
        "php" -> Php.highlight
        "pike" -> Pike.highlight
        "postscript" -> Postscript.highlight
        "prolog" -> Prolog.highlight
        "pure" -> Pure.highlight
        "python" -> Python.highlight
        "r" -> R.highlight
        "relaxng" -> Relaxng.highlight
        "relaxngcompact" -> Relaxngcompact.highlight
        "rest" -> Rest.highlight
        "rhtml" -> Rhtml.highlight
        "roff" -> Roff.highlight
        "ruby" -> Ruby.highlight
        "rust" -> Rust.highlight
        "scala" -> Scala.highlight
        "scheme" -> Scheme.highlight
        "sci" -> Sci.highlight
        "sed" -> Sed.highlight
        "sgml" -> Sgml.highlight
        "sql" -> Sql.highlight
        "sqlmysql" -> SqlMysql.highlight
        "sqlpostgresql" -> SqlPostgresql.highlight
        "tcl" -> Tcl.highlight
        "tcsh" -> Tcsh.highlight
        "texinfo" -> Texinfo.highlight
        "verilog" -> Verilog.highlight
        "vhdl" -> Vhdl.highlight
        "xml" -> Xml.highlight
        "xorg" -> Xorg.highlight
        "xslt" -> Xslt.highlight
        "xul" -> Xul.highlight
        "yacc" -> Yacc.highlight
        "yaml" -> Yaml.highlight
        "zsh" -> Zsh.highlight
        -- special cases:
        "csharp" -> highlightAs "cs"
        _ -> map (\l -> [(NormalTok, l)]) . lines
