// ========================================================================
// Copyright 2004-2005 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ========================================================================

package org.mortbay.jetty;

import java.io.UnsupportedEncodingException;

import junit.framework.TestCase;

import org.mortbay.io.Buffer;
import org.mortbay.io.ByteArrayBuffer;
import org.mortbay.io.SimpleBuffers;
import org.mortbay.io.bio.StringEndPoint;
import org.mortbay.util.StringUtil;

/**
 * @author gregw
 *
 * To change this generated comment edit the template variable "typecomment":
 * Window>Preferences>Java>Templates.
 * To enable and disable the creation of type comments go to
 * Window>Preferences>Java>Code Generation.
 */
public class HttpParserTest extends TestCase
{
    /**
     * Constructor for HttpParserTest.
     * @param arg0
     */
    public HttpParserTest(String arg0)
    {
        super(arg0);
    }

    public static void main(String[] args)
    {
        junit.textui.TestRunner.run(HttpParserTest.class);
    }

    /**
     * @see TestCase#setUp()
     */
    protected void setUp() throws Exception
    {
        super.setUp();
    }

    /**
     * @see TestCase#tearDown()
     */
    protected void tearDown() throws Exception
    {
        super.tearDown();
    }

    public void testLineParse0()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("POST /foo HTTP/1.0\015\012" + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("POST", f0);
        assertEquals("/foo", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(-1, h);
    }

    public void testLineParse1()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("GET /999\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        f2= null;
        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("GET", f0);
        assertEquals("/999", f1);
        assertEquals(null, f2);
        assertEquals(-1, h);
    }

    public void testLineParse2()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("POST /222  \015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        f2= null;
        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("POST", f0);
        assertEquals("/222", f1);
        assertEquals(null, f2);
        assertEquals(-1, h);
    }

    public void testLineParse3()
        throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("POST /fo\u0690 HTTP/1.0\015\012" + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("POST", f0);
        assertEquals("/fo\u0690", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(-1, h);
    }

    public void testLineParse4()
        throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("POST /foo?param=\u0690 HTTP/1.0\015\012" + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("POST", f0);
        assertEquals("/foo?param=\u0690", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(-1, h);
    }

    public void testConnect()
        throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput("CONNECT 192.168.1.2:80 HTTP/1.1\015\012" + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertTrue(handler.request);
        assertEquals("CONNECT", f0);
        assertEquals("192.168.1.2:80", f1);
        assertEquals("HTTP/1.1", f2);
        assertEquals(-1, h);
    }
    
    public void testHeaderParse()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
            "GET / HTTP/1.0\015\012"
                + "Header1: value1\015\012"
                + "Header2  :   value 2a  \015\012"
                + "                    value 2b  \015\012"
                + "Header3: \015\012"
                + "Header4 \015\012"
                + "  value4\015\012"
                + "Server5: notServer\015\012"
                + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("GET", f0);
        assertEquals("/", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals("Header1", hdr[0]);
        assertEquals("value1", val[0]);
        assertEquals("Header2", hdr[1]);
        assertEquals("value 2a value 2b", val[1]);
        assertEquals("Header3", hdr[2]);
        assertEquals("", val[2]);
        assertEquals("Header4", hdr[3]);
        assertEquals("value4", val[3]);
        assertEquals("Server5", hdr[4]);
        assertEquals("notServer", val[4]);
        assertEquals(4, h);
    }

    public void testChunkParse()
    	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
            "GET /chunk HTTP/1.0\015\012"
                + "Header1: value1\015\012"
				+ "Transfer-Encoding: chunked\015\012"
                + "\015\012"
                + "a;\015\012"
                + "0123456789\015\012"
                + "1a\015\012"
                + "ABCDEFGHIJKLMNOPQRSTUVWXYZ\015\012"
                + "0\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        ByteArrayBuffer content=new ByteArrayBuffer(8192);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer,content});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), content.capacity());
        parser.parse();
        assertEquals("GET", f0);
        assertEquals("/chunk", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(1, h);
        assertEquals("Header1", hdr[0]);
        assertEquals("value1", val[0]);
        assertEquals("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _content);
    }

    public void testMultiParse()
		throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
            "GET /mp HTTP/1.0\015\012"
                + "Header1: value1\015\012"
				+ "Transfer-Encoding: chunked\015\012"
                + "\015\012"
                + "a;\015\012"
                + "0123456789\015\012"
                + "1a\015\012"
                + "ABCDEFGHIJKLMNOPQRSTUVWXYZ\015\012"
                + "0\015\012"
                + "POST /foo HTTP/1.0\015\012"
                + "Header2: value2\015\012"
				+ "Content-Length: 0\015\012"
                + "\015\012"
                + "PUT /doodle HTTP/1.0\015\012"
                + "Header3: value3\015\012"
				+ "Content-Length: 10\015\012"
                + "\015\012"
                + "0123456789\015\012");

        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        ByteArrayBuffer content=new ByteArrayBuffer(8192);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer,content});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), content.capacity());
        parser.parse();
        assertEquals("GET", f0);
        assertEquals("/mp", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(1, h);
        assertEquals("Header1", hdr[0]);
        assertEquals("value1", val[0]);
        assertEquals("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _content);

        parser.parse();
        assertEquals("POST", f0);
        assertEquals("/foo", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(1, h);
        assertEquals("Header2", hdr[0]);
        assertEquals("value2", val[0]);
        assertEquals(null, _content);

        parser.parse();
        assertEquals("PUT", f0);
        assertEquals("/doodle", f1);
        assertEquals("HTTP/1.0", f2);
        assertEquals(1, h);
        assertEquals("Header3", hdr[0]);
        assertEquals("value3", val[0]);
        assertEquals("0123456789", _content);
    }

    public void testStreamParse() throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        String http="GET / HTTP/1.0\015\012"
                + "Header1: value1\015\012"
				+ "Transfer-Encoding: chunked\015\012"
                + "\015\012"
                + "a;\015\012"
                + "0123456789\015\012"
                + "1a\015\012"
                + "ABCDEFGHIJKLMNOPQRSTUVWXYZ\015\012"
                + "0\015\012"
                + "POST /foo HTTP/1.0\015\012"
                + "Header2: value2\015\012"
                + "Content-Length: 0\015\012"
                + "\015\012"
                + "PUT /doodle HTTP/1.0\015\012"
                + "Header3: value3\015\012"
				+ "Content-Length: 10\015\012"
                + "\015\012"
                + "0123456789\015\012";

        
        int[] tests=
            {
                1024,
                http.length() + 3,
                http.length() + 2,
                http.length() + 1,
                http.length() + 0,
                http.length() - 1,
                http.length() - 2,
                http.length() / 2,
                http.length() / 3,
                64,
                32
            };
        
        for (int t= 0; t < tests.length; t++)
        {
            String tst="t"+tests[t];
            try
            {
                ByteArrayBuffer buffer= new ByteArrayBuffer(tests[t]);
                ByteArrayBuffer content=new ByteArrayBuffer(8192);
                SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer,content});

                Handler handler = new Handler();
                HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), content.capacity());
                
                
                io.setInput(http);
                
                parser.parse();
                assertEquals(tst,"GET", f0);
                assertEquals(tst,"/", f1);
                assertEquals(tst,"HTTP/1.0", f2);
                assertEquals(tst,1, h);
                assertEquals(tst,"Header1", hdr[0]);
                assertEquals(tst,"value1", val[0]);
                assertEquals(tst,"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _content);
                
                parser.parse();
                assertEquals(tst,"POST", f0);
                assertEquals(tst,"/foo", f1);
                assertEquals(tst,"HTTP/1.0", f2);
                assertEquals(tst,1, h);
                assertEquals(tst,"Header2", hdr[0]);
                assertEquals(tst,"value2", val[0]);
                assertEquals(tst,null, _content);
                
                parser.parse();
                assertEquals(tst,"PUT", f0);
                assertEquals(tst,"/doodle", f1);
                assertEquals(tst,"HTTP/1.0", f2);
                assertEquals(tst,1, h);
                assertEquals(tst,"Header3", hdr[0]);
                assertEquals(tst,"value3", val[0]);
                assertEquals(tst,"0123456789", _content);
            }
            catch(Exception e)
            {
                if (t+1 < tests.length)
                    throw e;
                assertTrue(e.toString().indexOf("FULL")>=0);
            }
        }
    }

    public void testResponseParse0()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
	    "HTTP/1.1 200 Correct\015\012" 
	    + "Content-Length: 10\015\012"
	    + "Content-Type: text/plain\015\012"
	    + "\015\012"
	    + "0123456789\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("HTTP/1.1", f0);
        assertEquals("200", f1);
        assertEquals("Correct", f2);
	assertEquals(_content.length(), 10);
	assertTrue(headerCompleted);
	assertTrue(messageCompleted);
    }

    public void testResponseParse1()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
	    "HTTP/1.1 304 Not-Modified\015\012" 
	    + "Connection: close\015\012"
	    + "\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("HTTP/1.1", f0);
        assertEquals("304", f1);
        assertEquals("Not-Modified", f2);
	assertTrue(headerCompleted);
	assertTrue(messageCompleted);
    }

    public void testResponseParse2()
	throws Exception
    {
        StringEndPoint io=new StringEndPoint();
        io.setInput(
	    "HTTP/1.1 204 No-Content\015\012" 
	    + "Connection: close\015\012"
	    + "\015\012"
	    + "HTTP/1.1 200 Correct\015\012" 
	    + "Content-Length: 10\015\012"
	    + "Content-Type: text/plain\015\012"
	    + "\015\012"
	    + "0123456789\015\012");
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler();
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        parser.parse();
        assertEquals("HTTP/1.1", f0);
        assertEquals("204", f1);
        assertEquals("No-Content", f2);
	assertTrue(headerCompleted);
	assertTrue(messageCompleted);

        parser.parse();
        assertEquals("HTTP/1.1", f0);
        assertEquals("200", f1);
        assertEquals("Correct", f2);
	assertEquals(_content.length(), 10);
	assertTrue(headerCompleted);
	assertTrue(messageCompleted);
    }

    String _content;
    String f0;
    String f1;
    String f2;
    String[] hdr;
    String[] val;
    int h;
    
    boolean headerCompleted;
    boolean messageCompleted;

    class Handler extends HttpParser.EventHandler
    {   
        boolean request;
        HttpFields fields;
        
        public void content(Buffer ref)
        {
            if (_content==null)
                _content="";
            _content= _content + ref;
        }


        public void startRequest(Buffer tok0, Buffer tok1, Buffer tok2)
        {
            try
            {
                request=true;
                h= -1;
                hdr= new String[9];
                val= new String[9];
                f0= tok0.toString();
                f1=new String(tok1.array(),tok1.getIndex(),tok1.length(),StringUtil.__UTF8);
                if (tok2!=null)
                    f2= tok2.toString();
                else
                    f2=null;

                fields=new HttpFields();
            }
            catch (UnsupportedEncodingException e)
            {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }

	    messageCompleted = false;
	    headerCompleted = false;
        }

        public void parsedHeader(Buffer name, Buffer value)
        {
            fields.add(name, value);
            hdr[++h]= name.toString();
            val[h]= value.toString();
        }

        public void headerComplete()
        {
            _content= null;
            String s0=fields.toString();
            String s1=fields.toString();
            if (!s0.equals(s1))
            {
                //System.err.println(s0);
                //System.err.println(s1);
                throw new IllegalStateException();
            }

	    headerCompleted = true;
        }

        public void messageComplete(long contentLength)
        {
	    messageCompleted = true;
        }


        public void startResponse(Buffer version, int status, Buffer reason)
        {
            request=false;
            f0 = version.toString();
	    f1 = Integer.toString(status);
	    f2 = reason.toString();

            fields=new HttpFields();
            hdr= new String[9];
            val= new String[9];

	    messageCompleted = false;
	    headerCompleted = false;
        }
    }
    
    
    byte[][] data={ 
            
            {
                0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x74, 0x72, 
                0x30, 0x36, 0x39, 0x20, 0x48, 0x54, 0x54, 0x50, 
                0x2f, 0x31, 0x2e, 0x31, 0x0d, 0x0a, 0x41, 0x75, 
                0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x61, 0x74, 
                0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x44, 0x69, 0x67, 
                0x65, 0x73, 0x74, 0x20, 0x75, 0x73, 0x65, 0x72, 
                0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x22, 0x30, 0x30, 
                0x32, 0x31, 0x39, 0x34, 0x2d, 0x4e, 0x50, 0x41, 
                0x32, 0x30, 0x31, 0x45, 0x2d, 0x30, 0x30, 0x32, 
                0x31, 0x39, 0x34, 0x30, 0x30, 0x34, 0x33, 0x33, 
                0x41, 0x22, 0x2c, 0x20, 0x72, 0x65, 0x61, 0x6c, 
                0x6d, 0x3d, 0x22, 0x74, 0x74, 0x70, 0x73, 0x22, 
                0x2c, 0x20, 0x71, 0x6f, 0x70, 0x3d, 0x22, 0x61, 
                0x75, 0x74, 0x68, 0x22, 0x2c, 0x20, 0x61, 0x6c, 
                0x6f, 0x67, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3d, 
                0x22, 0x4d, 0x44, 0x35, 0x22, 0x2c, 0x20, 0x75, 
                0x72, 0x69, 0x3d, 0x22, 0x2f, 0x74, 0x72, 0x30, 
                0x36, 0x39, 0x22, 0x2c, 0x20, 0x6e, 0x6f, 0x6e, 
                0x63, 0x65, 0x3d, 0x22, 0x79, 0x43, 0x38, 0x54, 
                0x63, 0x53, 0x63, 0x42, 0x41, 0x41, 0x44, 0x50, 
                0x6e, 0x4f, 0x44, 0x38, 0x46, 0x41, 0x6e, 0x33, 
                0x43, 0x34, 0x59, 0x4d, 0x51, 0x69, 0x77, 0x6c, 
                0x37, 0x4f, 0x31, 0x31, 0x22, 0x2c, 0x20, 0x63, 
                0x6e, 0x6f, 0x6e, 0x63, 0x65, 0x3d, 0x22, 0x79, 
                0x43, 0x38, 0x54, 0x63, 0x53, 0x63, 0x42, 0x41, 
                0x41, 0x44, 0x50, 0x6e, 0x4f, 0x44, 0x38, 0x46, 
                0x41, 0x6e, 0x33, 0x43, 0x34, 0x59, 0x4d, 0x51, 
                0x69, 0x77, 0x6c, 0x37, 0x4f, 0x31, 0x31, 0x22, 
                0x2c, 0x20, 0x6e, 0x63, 0x3d, 0x30, 0x30, 0x30, 
                0x30, 0x30, 0x30, 0x30, 0x31, 0x2c, 0x20, 0x72, 
                0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73, 0x65, 0x3d, 
                0x22, 0x30, 0x36, 0x61, 0x35, 0x38, 0x35, 0x61, 
                0x31, 0x36, 0x35, 0x63, 0x35, 0x33, 0x34, 0x31, 
                0x64, 0x66, 0x63, 0x34, 0x66, 0x34, 0x36, 0x62, 
                0x32, 0x63, 0x38, 0x62, 0x37, 0x63, 0x33, 0x64, 
                0x62, 0x22, 0x0d, 0x0a, 0x48, 0x6f, 0x73, 0x74, 
                0x3a, 0x20, 0x31, 0x39, 0x32, 0x2e, 0x31, 0x36, 
                0x38, 0x2e, 0x31, 0x2e, 0x39, 0x36, 0x0d, 0x0a, 
                0x53, 0x4f, 0x41, 0x50, 0x41, 0x63, 0x74, 0x69, 
                0x6f, 0x6e, 0x3a, 0x20, 0x49, 0x6e, 0x66, 0x6f, 
                0x72, 0x6d, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x6e, 
                0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 
                0x4b, 0x65, 0x65, 0x70, 0x2d, 0x41, 0x6c, 0x69, 
                0x76, 0x65, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74, 
                0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65, 
                0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 0x2f, 0x78, 
                0x6d, 0x6c, 0x3b, 0x20, 0x63, 0x68, 0x61, 0x72, 
                0x73, 0x65, 0x74, 0x3d, 0x22, 0x75, 0x74, 0x66, 
                0x2d, 0x38, 0x22, 0x0d, 0x0a, 0x0d, 0x0a, 0x54, 
                0x72, 0x61, 0x6e, 0x73, 0x66, 0x65, 0x72, 0x2d, 
                0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 
                0x3a, 0x20, 0x43, 0x68, 0x75, 0x6e, 0x6b, 0x65, 
                0x64, 0x0d, 0x0a },
                /*
            
            {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x74, 0x72, 
        0x30, 0x36, 0x39, 0x20, 0x48, 0x54, 0x54, 0x50, 
        0x2f, 0x31, 0x2e, 0x31, 0x0d, 0x0a, 0x48, 0x6f, 
        0x73, 0x74, 0x3a, 0x20, 0x31, 0x39, 0x32, 0x2e, 
        0x31, 0x36, 0x38, 0x2e, 0x31, 0x2e, 0x39, 0x36, 
        0x0d, 0x0a, 0x53, 0x4f, 0x41, 0x50, 0x41, 0x63, 
        0x74, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x49, 0x6e, 
        0x66, 0x6f, 0x72, 0x6d, 0x0d, 0x0a, 0x43, 0x6f, 
        0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 
        0x3a, 0x20, 0x4b, 0x65, 0x65, 0x70, 0x2d, 0x41, 
        0x6c, 0x69, 0x76, 0x65, 0x0d, 0x0a, 0x43, 0x6f, 
        0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 
        0x70, 0x65, 0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 
        0x2f, 0x78, 0x6d, 0x6c, 0x3b, 0x20, 0x63, 0x68, 
        0x61, 0x72, 0x73, 0x65, 0x74, 0x3d, 0x22, 0x75, 
        0x74, 0x66, 0x2d, 0x38, 0x22, 0x0d, 0x0a, 0x54, 
        0x72, 0x61, 0x6e, 0x73, 0x66, 0x65, 0x72, 0x2d, 
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 
        0x3a, 0x20, 0x43, 0x68, 0x75, 0x6e, 0x6b, 0x65, 
        0x64, 0x0d, 0x0a, 0x0d, 0x0a },
        */
            {
                0x31, 0x30, 0x62, 0x0d, 0x0a, 0x0a, 0x3c, 0x73, 
                0x6f, 0x61, 0x70, 0x3a, 0x45, 0x6e, 0x76, 0x65, 
                0x6c, 0x6f, 0x70, 0x65, 0x20, 0x78, 0x6d, 0x6c, 
                0x6e, 0x73, 0x3a, 0x73, 0x6f, 0x61, 0x70, 0x3d, 
                0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 
                0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x73, 0x2e, 
                0x78, 0x6d, 0x6c, 0x73, 0x6f, 0x61, 0x70, 0x2e, 
                0x6f, 0x72, 0x67, 0x2f, 0x73, 0x6f, 0x61, 0x70, 
                0x2f, 0x65, 0x6e, 0x76, 0x65, 0x6c, 0x6f, 0x70, 
                0x65, 0x2f, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 
                0x73, 0x3a, 0x73, 0x6f, 0x61, 0x70, 0x65, 0x6e, 
                0x63, 0x3d, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 
                0x2f, 0x2f, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x61, 
                0x73, 0x2e, 0x78, 0x6d, 0x6c, 0x73, 0x6f, 0x61, 
                0x70, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x73, 0x6f, 
                0x61, 0x70, 0x2f, 0x65, 0x6e, 0x63, 0x6f, 0x64, 
                0x69, 0x6e, 0x67, 0x2f, 0x22, 0x20, 0x78, 0x6d, 
                0x6c, 0x6e, 0x73, 0x3a, 0x63, 0x77, 0x6d, 0x70, 
                0x3d, 0x22, 0x75, 0x72, 0x6e, 0x3a, 0x64, 0x73, 
                0x6c, 0x66, 0x6f, 0x72, 0x75, 0x6d, 0x2d, 0x6f, 
                0x72, 0x67, 0x3a, 0x63, 0x77, 0x6d, 0x70, 0x2d, 
                0x31, 0x2d, 0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 
                0x6e, 0x73, 0x3a, 0x78, 0x73, 0x69, 0x3d, 0x22, 
                0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 
                0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e, 0x6f, 0x72, 
                0x67, 0x2f, 0x32, 0x30, 0x30, 0x31, 0x2f, 0x58, 
                0x4d, 0x4c, 0x53, 0x63, 0x68, 0x65, 0x6d, 0x61, 
                0x2d, 0x69, 0x6e, 0x73, 0x74, 0x61, 0x6e, 0x63, 
                0x65, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 
                0x3a, 0x78, 0x73, 0x64, 0x3d, 0x22, 0x68, 0x74, 
                0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 
                0x2e, 0x77, 0x33, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 
                0x32, 0x30, 0x30, 0x31, 0x2f, 0x58, 0x4d, 0x4c, 
                0x53, 0x63, 0x68, 0x65, 0x6d, 0x61, 0x22, 0x3e, 
                0x0d, 0x0a 
            },{
                0x39, 0x61, 0x0d, 0x0a, 0x0a, 0x09, 0x3c, 0x73, 
                0x6f, 0x61, 0x70, 0x3a, 0x48, 0x65, 0x61, 0x64, 
                0x65, 0x72, 0x3e, 0x0a, 0x09, 0x09, 0x3c, 0x63, 
                0x77, 0x6d, 0x70, 0x3a, 0x49, 0x44, 0x20, 0x73, 
                0x6f, 0x61, 0x70, 0x3a, 0x6d, 0x75, 0x73, 0x74, 
                0x55, 0x6e, 0x64, 0x65, 0x72, 0x73, 0x74, 0x61, 
                0x6e, 0x64, 0x3d, 0x22, 0x31, 0x22, 0x3e, 0x75, 
                0x62, 0x69, 0x63, 0x6f, 0x6d, 0x5f, 0x74, 0x72, 
                0x30, 0x36, 0x39, 0x5f, 0x69, 0x64, 0x5f, 0x30, 
                0x3c, 0x2f, 0x63, 0x77, 0x6d, 0x70, 0x3a, 0x49, 
                0x44, 0x3e, 0x0a, 0x09, 0x09, 0x3c, 0x63, 0x77, 
                0x6d, 0x70, 0x3a, 0x4e, 0x6f, 0x4d, 0x6f, 0x72, 
                0x65, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 
                0x73, 0x3e, 0x31, 0x3c, 0x2f, 0x63, 0x77, 0x6d, 
                0x70, 0x3a, 0x4e, 0x6f, 0x4d, 0x6f, 0x72, 0x65, 
                0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 
                0x3e, 0x0a, 0x09, 0x3c, 0x2f, 0x73, 0x6f, 0x61, 
                0x70, 0x3a, 0x48, 0x65, 0x61, 0x64, 0x65, 0x72, 
                0x3e, 0x0a, 0x09, 0x3c, 0x73, 0x6f, 0x61, 0x70, 
                0x3a, 0x42, 0x6f, 0x64, 0x79, 0x3e, 0x0d, 0x0a 
            },{
                0x33, 0x37, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x3c, 
                0x63, 0x77, 0x6d, 0x70, 0x3a, 0x49, 0x6e, 0x66, 
                0x6f, 0x72, 0x6d, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 
                0x73, 0x3a, 0x63, 0x77, 0x6d, 0x70, 0x3d, 0x22, 
                0x75, 0x72, 0x6e, 0x3a, 0x64, 0x73, 0x6c, 0x66, 
                0x6f, 0x72, 0x75, 0x6d, 0x2d, 0x6f, 0x72, 0x67, 
                0x3a, 0x63, 0x77, 0x6d, 0x70, 0x2d, 0x31, 0x2d, 
                0x30, 0x22, 0x3e, 0x0d, 0x0a 
            },{
                0x62, 0x65, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x09, 
                0x3c, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x49, 
                0x64, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 
                0x4d, 0x61, 0x6e, 0x75, 0x66, 0x61, 0x63, 0x74, 
                0x75, 0x72, 0x65, 0x72, 0x3e, 0x50, 0x69, 0x6e, 
                0x67, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x75, 0x6e, 
                0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3c, 
                0x2f, 0x4d, 0x61, 0x6e, 0x75, 0x66, 0x61, 0x63, 
                0x74, 0x75, 0x72, 0x65, 0x72, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x3c, 0x4f, 0x55, 0x49, 0x3e, 
                0x30, 0x30, 0x32, 0x31, 0x39, 0x34, 0x3c, 0x2f, 
                0x4f, 0x55, 0x49, 0x3e, 0x0a, 0x09, 0x09, 0x09, 
                0x09, 0x3c, 0x50, 0x72, 0x6f, 0x64, 0x75, 0x63, 
                0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x3e, 0x4e, 
                0x50, 0x41, 0x32, 0x30, 0x31, 0x45, 0x3c, 0x2f, 
                0x50, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x43, 
                0x6c, 0x61, 0x73, 0x73, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x53, 0x65, 0x72, 0x69, 0x61, 
                0x6c, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3e, 
                0x30, 0x30, 0x32, 0x31, 0x39, 0x34, 0x30, 0x30, 
                0x34, 0x33, 0x33, 0x41, 0x3c, 0x2f, 0x53, 0x65, 
                0x72, 0x69, 0x61, 0x6c, 0x4e, 0x75, 0x6d, 0x62, 
                0x65, 0x72, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x3c, 
                0x2f, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x49, 
                0x64, 0x3e, 0x0d, 0x0a 
            },{
                0x31, 0x30, 0x66, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 
                0x09, 0x3c, 0x45, 0x76, 0x65, 0x6e, 0x74, 0x20, 
                0x73, 0x6f, 0x61, 0x70, 0x65, 0x6e, 0x63, 0x3a, 
                0x61, 0x72, 0x72, 0x61, 0x79, 0x54, 0x79, 0x70, 
                0x65, 0x3d, 0x22, 0x63, 0x77, 0x6d, 0x70, 0x3a, 
                0x45, 0x76, 0x65, 0x6e, 0x74, 0x53, 0x74, 0x72, 
                0x75, 0x63, 0x74, 0x5b, 0x32, 0x5d, 0x22, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x45, 0x76, 
                0x65, 0x6e, 0x74, 0x53, 0x74, 0x72, 0x75, 0x63, 
                0x74, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 
                0x3c, 0x45, 0x76, 0x65, 0x6e, 0x74, 0x43, 0x6f, 
                0x64, 0x65, 0x3e, 0x31, 0x20, 0x42, 0x4f, 0x4f, 
                0x54, 0x3c, 0x2f, 0x45, 0x76, 0x65, 0x6e, 0x74, 
                0x43, 0x6f, 0x64, 0x65, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x09, 0x3c, 0x43, 0x6f, 0x6d, 0x6d, 
                0x61, 0x6e, 0x64, 0x4b, 0x65, 0x79, 0x3e, 0x3c, 
                0x2f, 0x43, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 
                0x4b, 0x65, 0x79, 0x3e, 0x0a, 0x09, 0x09, 0x09, 
                0x09, 0x3c, 0x2f, 0x45, 0x76, 0x65, 0x6e, 0x74, 
                0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 0x0a, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x45, 0x76, 0x65, 
                0x6e, 0x74, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 
                0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 
                0x45, 0x76, 0x65, 0x6e, 0x74, 0x43, 0x6f, 0x64, 
                0x65, 0x3e, 0x4d, 0x20, 0x52, 0x65, 0x62, 0x6f, 
                0x6f, 0x74, 0x3c, 0x2f, 0x45, 0x76, 0x65, 0x6e, 
                0x74, 0x43, 0x6f, 0x64, 0x65, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x43, 0x6f, 0x6d, 
                0x6d, 0x61, 0x6e, 0x64, 0x4b, 0x65, 0x79, 0x3e, 
                0x3c, 0x2f, 0x43, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 
                0x64, 0x4b, 0x65, 0x79, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x2f, 0x45, 0x76, 0x65, 0x6e, 
                0x74, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x3c, 0x2f, 0x45, 0x76, 
                0x65, 0x6e, 0x74, 0x3e, 0x0d, 0x0a 
            },{
                0x37, 0x32, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x09, 
                0x3c, 0x4d, 0x61, 0x78, 0x45, 0x6e, 0x76, 0x65, 
                0x6c, 0x6f, 0x70, 0x65, 0x73, 0x3e, 0x31, 0x3c, 
                0x2f, 0x4d, 0x61, 0x78, 0x45, 0x6e, 0x76, 0x65, 
                0x6c, 0x6f, 0x70, 0x65, 0x73, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x3c, 0x43, 0x75, 0x72, 0x72, 0x65, 
                0x6e, 0x74, 0x54, 0x69, 0x6d, 0x65, 0x3e, 0x30, 
                0x30, 0x30, 0x31, 0x2d, 0x30, 0x31, 0x2d, 0x30, 
                0x31, 0x54, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x3a, 
                0x30, 0x33, 0x3c, 0x2f, 0x43, 0x75, 0x72, 0x72, 
                0x65, 0x6e, 0x74, 0x54, 0x69, 0x6d, 0x65, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x3c, 0x52, 0x65, 0x74, 
                0x72, 0x79, 0x43, 0x6f, 0x75, 0x6e, 0x74, 0x3e, 
                0x30, 0x3c, 0x2f, 0x52, 0x65, 0x74, 0x72, 0x79, 
                0x43, 0x6f, 0x75, 0x6e, 0x74, 0x3e, 0x0d, 0x0a 
            },{
                0x34, 0x34, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x09, 
                0x3c, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 
                0x65, 0x72, 0x4c, 0x69, 0x73, 0x74, 0x20, 0x73, 
                0x6f, 0x61, 0x70, 0x65, 0x6e, 0x63, 0x3a, 0x61, 
                0x72, 0x72, 0x61, 0x79, 0x54, 0x79, 0x70, 0x65, 
                0x3d, 0x22, 0x63, 0x77, 0x6d, 0x70, 0x3a, 0x50, 
                0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 
                0x56, 0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 
                0x75, 0x63, 0x74, 0x5b, 0x37, 0x5d, 0x22, 0x3e, 
                0x0d, 0x0a 
            },{
                0x35, 0x30, 0x64, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x50, 0x61, 0x72, 0x61, 0x6d, 
                0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 0x75, 
                0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x4e, 
                0x61, 0x6d, 0x65, 0x3e, 0x49, 0x6e, 0x74, 0x65, 
                0x72, 0x6e, 0x65, 0x74, 0x47, 0x61, 0x74, 0x65, 
                0x77, 0x61, 0x79, 0x44, 0x65, 0x76, 0x69, 0x63, 
                0x65, 0x2e, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 
                0x49, 0x6e, 0x66, 0x6f, 0x2e, 0x53, 0x70, 0x65, 
                0x63, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 
                0x3c, 0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x0a, 
                0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 0x61, 
                0x6c, 0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 0x3a, 
                0x74, 0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 0x73, 
                0x64, 0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 
                0x22, 0x3e, 0x31, 0x2e, 0x30, 0x3c, 0x2f, 0x56, 
                0x61, 0x6c, 0x75, 0x65, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x2f, 0x50, 0x61, 0x72, 0x61, 
                0x6d, 0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 
                0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x50, 
                0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 
                0x56, 0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 
                0x75, 0x63, 0x74, 0x3e, 0x0a, 0x09, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 
                0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 
                0x47, 0x61, 0x74, 0x65, 0x77, 0x61, 0x79, 0x44, 
                0x65, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x44, 0x65, 
                0x76, 0x69, 0x63, 0x65, 0x49, 0x6e, 0x66, 0x6f, 
                0x2e, 0x48, 0x61, 0x72, 0x64, 0x77, 0x61, 0x72, 
                0x65, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 
                0x3c, 0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x0a, 
                0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 0x61, 
                0x6c, 0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 0x3a, 
                0x74, 0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 0x73, 
                0x64, 0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 
                0x22, 0x3e, 0x41, 0x31, 0x3c, 0x2f, 0x56, 0x61, 
                0x6c, 0x75, 0x65, 0x3e, 0x0a, 0x09, 0x09, 0x09, 
                0x09, 0x3c, 0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 
                0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 0x75, 
                0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x50, 0x61, 
                0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x56, 
                0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 0x75, 
                0x63, 0x74, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 
                0x09, 0x3c, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x49, 
                0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x47, 
                0x61, 0x74, 0x65, 0x77, 0x61, 0x79, 0x44, 0x65, 
                0x76, 0x69, 0x63, 0x65, 0x2e, 0x44, 0x65, 0x76, 
                0x69, 0x63, 0x65, 0x49, 0x6e, 0x66, 0x6f, 0x2e, 
                0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 
                0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x3c, 
                0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 0x3a, 0x74, 
                0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 0x73, 0x64, 
                0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x22, 
                0x3e, 0x6e, 0x70, 0x61, 0x32, 0x30, 0x31, 0x65, 
                0x2d, 0x36, 0x2e, 0x30, 0x2e, 0x32, 0x2d, 0x74, 
                0x65, 0x6c, 0x61, 0x76, 0x6f, 0x78, 0x2d, 0x72, 
                0x33, 0x30, 0x35, 0x36, 0x35, 0x3c, 0x2f, 0x56, 
                0x61, 0x6c, 0x75, 0x65, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x2f, 0x50, 0x61, 0x72, 0x61, 
                0x6d, 0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 
                0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x50, 
                0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 
                0x56, 0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 
                0x75, 0x63, 0x74, 0x3e, 0x0a, 0x09, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 
                0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 
                0x47, 0x61, 0x74, 0x65, 0x77, 0x61, 0x79, 0x44, 
                0x65, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x44, 0x65, 
                0x76, 0x69, 0x63, 0x65, 0x49, 0x6e, 0x66, 0x6f, 
                0x2e, 0x50, 0x72, 0x6f, 0x76, 0x69, 0x73, 0x69, 
                0x6f, 0x6e, 0x69, 0x6e, 0x67, 0x43, 0x6f, 0x64, 
                0x65, 0x3c, 0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 
                0x61, 0x6c, 0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 
                0x3a, 0x74, 0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 
                0x73, 0x64, 0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 
                0x67, 0x22, 0x3e, 0x3c, 0x2f, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 
                0x3c, 0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 
                0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 0x75, 0x65, 
                0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 0x0a, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x50, 0x61, 0x72, 
                0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 
                0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 
                0x74, 0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 
                0x3c, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x49, 0x6e, 
                0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x47, 0x61, 
                0x74, 0x65, 0x77, 0x61, 0x79, 0x44, 0x65, 0x76, 
                0x69, 0x63, 0x65, 0x2e, 0x4d, 0x61, 0x6e, 0x61, 
                0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x53, 0x65, 
                0x72, 0x76, 0x65, 0x72, 0x2e, 0x43, 0x6f, 0x6e, 
                0x6e, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x52, 
                0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x55, 0x52, 
                0x4c, 0x3c, 0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 
                0x61, 0x6c, 0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 
                0x3a, 0x74, 0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 
                0x73, 0x64, 0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 
                0x67, 0x22, 0x3e, 0x68, 0x74, 0x74, 0x70, 0x3a, 
                0x2f, 0x2f, 0x31, 0x39, 0x32, 0x2e, 0x31, 0x36, 
                0x38, 0x2e, 0x35, 0x38, 0x2e, 0x31, 0x32, 0x3a, 
                0x39, 0x36, 0x39, 0x39, 0x2f, 0x63, 0x6f, 0x6e, 
                0x6e, 0x5f, 0x72, 0x65, 0x71, 0x5f, 0x75, 0x72, 
                0x6c, 0x3c, 0x2f, 0x56, 0x61, 0x6c, 0x75, 0x65, 
                0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x2f, 
                0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 
                0x72, 0x56, 0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 
                0x72, 0x75, 0x63, 0x74, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x50, 0x61, 0x72, 0x61, 0x6d, 
                0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 0x75, 
                0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x4e, 
                0x61, 0x6d, 0x65, 0x3e, 0x49, 0x6e, 0x74, 0x65, 
                0x72, 0x6e, 0x65, 0x74, 0x47, 0x61, 0x74, 0x65, 
                0x77, 0x61, 0x79, 0x44, 0x65, 0x76, 0x69, 0x63, 
                0x65, 0x2e, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x65, 
                0x6d, 0x65, 0x6e, 0x74, 0x53, 0x65, 0x72, 0x76, 
                0x65, 0x72, 0x2e, 0x50, 0x61, 0x72, 0x61, 0x6d, 
                0x65, 0x74, 0x65, 0x72, 0x4b, 0x65, 0x79, 0x3c, 
                0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 0x3a, 0x74 
            },{
                0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 0x73, 0x64, 
                0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x22, 
                0x3e, 0x3c, 0x2f, 0x56, 0x61, 0x6c, 0x75, 0x65, 
                0x3e, 0x0a, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x2f, 
                0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 
                0x72, 0x56, 0x61, 0x6c, 0x75, 0x65, 0x53, 0x74, 
                0x72, 0x75, 0x63, 0x74, 0x3e, 0x0a, 0x09, 0x09, 
                0x09, 0x09, 0x3c, 0x50, 0x61, 0x72, 0x61, 0x6d, 
                0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 0x6c, 0x75, 
                0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 0x74, 0x3e, 
                0x0a, 0x09, 0x09, 0x09, 0x09, 0x09, 0x3c, 0x4e, 
                0x61, 0x6d, 0x65, 0x3e, 0x49, 0x6e, 0x74, 0x65, 
                0x72, 0x6e, 0x65, 0x74, 0x47, 0x61, 0x74, 0x65, 
                0x77, 0x61, 0x79, 0x44, 0x65, 0x76, 0x69, 0x63, 
                0x65, 0x2e, 0x57, 0x41, 0x4e, 0x44, 0x65, 0x76, 
                0x69, 0x63, 0x65, 0x2e, 0x31, 0x2e, 0x57, 0x41, 
                0x4e, 0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 
                0x69, 0x6f, 0x6e, 0x44, 0x65, 0x76, 0x69, 0x63, 
                0x65, 0x2e, 0x31, 0x2e, 0x57, 0x41, 0x4e, 0x49, 
                0x50, 0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 
                0x69, 0x6f, 0x6e, 0x2e, 0x31, 0x2e, 0x45, 0x78, 
                0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c, 0x49, 0x50, 
                0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3c, 
                0x2f, 0x4e, 0x61, 0x6d, 0x65, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x09, 0x3c, 0x56, 0x61, 0x6c, 
                0x75, 0x65, 0x20, 0x78, 0x73, 0x69, 0x3a, 0x74, 
                0x79, 0x70, 0x65, 0x3d, 0x22, 0x78, 0x73, 0x64, 
                0x3a, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x22, 
                0x3e, 0x31, 0x39, 0x32, 0x2e, 0x31, 0x36, 0x38, 
                0x2e, 0x35, 0x38, 0x2e, 0x31, 0x32, 0x3c, 0x2f, 
                0x56, 0x61, 0x6c, 0x75, 0x65, 0x3e, 0x0a, 0x09, 
                0x09, 0x09, 0x09, 0x3c, 0x2f, 0x50, 0x61, 0x72, 
                0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x56, 0x61, 
                0x6c, 0x75, 0x65, 0x53, 0x74, 0x72, 0x75, 0x63, 
                0x74, 0x3e, 0x0d, 0x0a 
            },{
                0x31, 0x34, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x09, 
                0x3c, 0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 
                0x74, 0x65, 0x72, 0x4c, 0x69, 0x73, 0x74, 0x3e, 
                0x0d, 0x0a 
            },{
                0x31, 0x31, 0x0d, 0x0a, 0x0a, 0x09, 0x09, 0x3c, 
                0x2f, 0x63, 0x77, 0x6d, 0x70, 0x3a, 0x49, 0x6e, 
                0x66, 0x6f, 0x72, 0x6d, 0x3e, 0x0d, 0x0a 
            },{
                0x31, 0x66, 0x0d, 0x0a, 0x0a, 0x09, 0x3c, 0x2f, 
                0x73, 0x6f, 0x61, 0x70, 0x3a, 0x42, 0x6f, 0x64, 
                0x79, 0x3e, 0x0a, 0x3c, 0x2f, 0x73, 0x6f, 0x61, 
                0x70, 0x3a, 0x45, 0x6e, 0x76, 0x65, 0x6c, 0x6f, 
                0x70, 0x65, 0x3e, 0x0d, 0x0a 
            },{
                0x30, 0x0d, 0x0a, 0x0d, 0x0a 
            }
    };

    public void testRealData()
        throws Exception
    {
        String request = "";
        
        for (int i=0;i<data.length;i++)
        {
            String chunk = new String(data[i]);
            //System.err.println(chunk);
            request+=chunk;
        }
        

        StringEndPoint io=new StringEndPoint();
        io.setInput(request);
        ByteArrayBuffer buffer= new ByteArrayBuffer(4096);
        SimpleBuffers buffers=new SimpleBuffers(new Buffer[]{buffer});

        Handler handler = new Handler()
        {

            /* (non-Javadoc)
             * @see org.mortbay.jetty.HttpParserTest.Handler#content(org.mortbay.io.Buffer)
             */
            public void content(Buffer ref)
            {
                super.content(ref);
                System.err.println(ref);
            }

            /* (non-Javadoc)
             * @see org.mortbay.jetty.HttpParserTest.Handler#headerComplete()
             */
            public void headerComplete()
            {
                System.err.println("---");
                super.headerComplete();
            }

            /* (non-Javadoc)
             * @see org.mortbay.jetty.HttpParserTest.Handler#messageComplete(long)
             */
            public void messageComplete(long contentLength)
            {
                System.err.println("===");
                super.messageComplete(contentLength);
            }

            /* (non-Javadoc)
             * @see org.mortbay.jetty.HttpParserTest.Handler#parsedHeader(org.mortbay.io.Buffer, org.mortbay.io.Buffer)
             */
            public void parsedHeader(Buffer name, Buffer value)
            {
                System.err.println(name+": "+value);
                super.parsedHeader(name,value);
            }

            /* (non-Javadoc)
             * @see org.mortbay.jetty.HttpParserTest.Handler#startRequest(org.mortbay.io.Buffer, org.mortbay.io.Buffer, org.mortbay.io.Buffer)
             */
            public void startRequest(Buffer tok0, Buffer tok1, Buffer tok2)
            {
                System.err.println(tok0+" "+tok1+" "+tok2);
                super.startRequest(tok0,tok1,tok2);
            }
            
            
        };
        HttpParser parser= new HttpParser(buffers,io, handler, buffer.capacity(), 0);
        
        parser.parse();
    }
}
