# Using the CMake tools to create the file arma_config.hpp, which contains
# information on the Armadillo configuration when mlpack was compiled.  This
# assumes ${ARMADILLO_INCLUDE_DIR} is set.  In addition, we must be careful to
# avoid overwriting arma_config.hpp with the exact same information, because
# this may trigger a new complete rebuild, which is undesired.
if(EXISTS "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp")
  file(READ "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp"
      OLD_FILE_CONTENTS)
else(EXISTS "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp")
  set(OLD_FILE_CONTENTS "")
endif(EXISTS "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp")

# If we are using Armadillo 5+, ARMA_64BIT_WORD is implicitly enabled.
set(ARMA_HAS_64BIT_WORD 0) # This may be unnecessary.
if(NOT (${ARMADILLO_VERSION_MAJOR} LESS 5))
  # ARMA_64BIT_WORD is only set if we are on a 64-bit system.
  if (CMAKE_SIZEOF_VOID_P EQUAL 8)
    set(ARMA_HAS_64BIT_WORD 1)
  else (CMAKE_SIZEOF_VOID_P EQUAL 8)
    set(ARMA_HAS_64BIT_WORD 0)
  endif (CMAKE_SIZEOF_VOID_P EQUAL 8)
else(NOT (${ARMADILLO_VERSION_MAJOR} LESS 5))
  # Otherwise, we'll need to open the config.hpp we are using and inspect the
  # setting of ARMA_64BIT_WORD.
  if(EXISTS "${ARMADILLO_INCLUDE_DIR}/armadillo_bits/config.hpp")
    file(READ "${ARMADILLO_INCLUDE_DIR}/armadillo_bits/config.hpp"
        ARMA_CONFIG)

    # Extract ARMA_64BIT_WORD.
    string(REGEX MATCH
        "[\r\n][ ]*#define ARMA_64BIT_WORD"
        ARMA_HAS_64BIT_WORD_PRE
        "${ARMA_CONFIG}")

    string(LENGTH "${ARMA_HAS_64BIT_WORD_PRE}" ARMA_HAS_64BIT_WORD)
  else(EXISTS "${ARMADILLO_INCLUDE_DIR}/armadillo_bits/config.hpp")
    # Assumes ARMA_64BIT_WORD is not set.
    message(WARNING "Armadillo configuration file
        (${ARMADILLO_INCLUDE_DIR}/armadillo_bits/config.hpp) does not exist!")
  endif(EXISTS "${ARMADILLO_INCLUDE_DIR}/armadillo_bits/config.hpp")
endif(NOT (${ARMADILLO_VERSION_MAJOR} LESS 5))

# Now use the value we gathered to generate the new file contents.
if(ARMA_HAS_64BIT_WORD EQUAL 0)
  set(ARMA_64BIT_WORD_DEFINE "#define MLPACK_ARMA_NO64BIT_WORD")
else(ARMA_HAS_64BIT_WORD EQUAL 0)
  set(ARMA_64BIT_WORD_DEFINE "#define MLPACK_ARMA_64BIT_WORD")
endif(ARMA_HAS_64BIT_WORD EQUAL 0)

set(NEW_FILE_CONTENTS
"/**
 * @file arma_config.hpp
 *
 * This is an autogenerated file which contains the configuration of Armadillo
 * at the time mlpack was built.  If you modify anything in here by hand, your
 * warranty is void, your house may catch fire, and we're not going to call the
 * police when your program segfaults so hard that robbers come to your house
 * and take everything you own.  If you do decide, against better judgment, to
 * modify anything at all in this file, and you are reporting a bug, be
 * absolutely certain to mention that you've done something stupid in this file
 * first.
 *
 * In short: don't touch this file.
 */
#ifndef __MLPACK_CORE_UTIL_ARMA_CONFIG_HPP
#define __MLPACK_CORE_UTIL_ARMA_CONFIG_HPP

${ARMA_64BIT_WORD_DEFINE}

#endif
")

# Did the contents of the file change at all?  If not, don't write it.
if(NOT "${OLD_FILE_CONTENTS}" STREQUAL "${NEW_FILE_CONTENTS}")
  # We have a reason to write the new file.
  message(STATUS "Regenerating arma_config.hpp.")
  file(REMOVE "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp")
  file(WRITE "${CMAKE_CURRENT_SOURCE_DIR}/src/mlpack/core/util/arma_config.hpp"
      "${NEW_FILE_CONTENTS}")
endif(NOT "${OLD_FILE_CONTENTS}" STREQUAL "${NEW_FILE_CONTENTS}")



