/*	test_Reference_Resolver

PIRL CVS ID: test_Reference_Resolver.java,v 1.19 2012/04/16 06:04:12 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

import java.util.Vector;
import java.util.Iterator;
import java.text.ParseException;

import PIRL.Conductor.Reference_Resolver;
import PIRL.Conductor.Unresolved_Reference;
import PIRL.Database.Database;
import PIRL.Database.Database_Exception;
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;
import PIRL.PVL.Parameter;
import PIRL.PVL.Value;
import PIRL.PVL.PVL_Exception;


public class test_Reference_Resolver
{
private static final String
	CONFIG_FILENAME	= "Conductor.conf",
	TEST_CATALOG	= "test",
	TEST_TABLE		= TEST_CATALOG + '.' + "test_table";

private static int
	Tests_Total = 0,
	Tests_Passed = 0;


public static void main
	(
	String[]	args
	)
{
System.out.println ("*** test_Reference_Resolver: " + Reference_Resolver.ID);

String
	server = null,
	config_filename = null;
boolean
	Verbose = false;

for (int count = 0;
	 count < args.length;
	 count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].charAt (1))
			{
			case 'C':
			case 'c':
				if (config_filename != null)
					{
					System.out.println
						("Only one config filename may be specified.");
					System.exit (1);
					}
				if (++count == args.length ||
					args[count].charAt (0) == '-')
					{
					System.out.println
						("Missing config filename.");
					System.exit (1);
					}
				config_filename = args[count];
				break;
			case 'V':
			case 'v':
				Verbose = true;
				break;
			default:
				System.out.println
					("Unknown option: " + args[count]);
			case 'H':
			case 'h':
				System.out.println
					("Usage: java test_Reference_Resolver [-Help] [-Verbose] [-Config filename] [server]");
				System.exit (1);
			}
		}
	else
		{
		if (server != null)
			{
			System.out.println
				("Only one server may be specified.");
			System.exit (1);
			}
		server = args[count];
		}
	}



//	Construct a new Configuration.
if (config_filename == null)
	config_filename = CONFIG_FILENAME;
Configuration
	configuration = null;
try {configuration = new Configuration (config_filename);}
catch (Configuration_Exception exception)
	{
	System.out.println
		("Unable to construct the Configuration.\n"
		+ exception.getMessage ());
	System.exit (1);
	}
//	Construct a new Database and connect to it.
Database
	database = null;
try {database = new Database (configuration);}
catch (Database_Exception exception)
	{
	System.out.println
		("Unable to construct the Database.\n"
		+ exception.getMessage ());
	System.exit (1);
	}
try {database.Connect (server);}
catch (Database_Exception exception)
	{
	System.out.println
		("Unable to Connect to the Database.\n"
		+ exception.getMessage ());
	System.exit (1);
	}
catch (Configuration_Exception exception)
	{
	System.out.println
		("Unable to Connect to the Database.\n"
		+ exception.getMessage ());
	System.exit (1);
	}
//	Construct a new Reference_Resolver.
Reference_Resolver
	resolver = new Reference_Resolver (database);

int
	Status = 1;

try
	{
	String
		description,
		pattern,
		expected,
		obtained;

	//	Unresolved reference.
	pattern = "${Test_Parameter}";
	try
		{
		expected = "Unresolved_Reference";
		obtained = resolver.Resolve (pattern);
		}
	catch (Unresolved_Reference exception)
		{
		expected = obtained = exception.getMessage ();
		}
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	expected = "Default Value";
	resolver.Default_Value (expected);
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Default_Value (\"" + expected + "\")",
			expected, obtained, Verbose);

	//	Malformed pattern.
	pattern = "${Test_Parameter";
	try
		{
		expected = "ParseException";
		obtained = resolver.Resolve (pattern);
		}
	catch (ParseException exception)
		{
		expected = obtained = exception.getMessage ();
		}
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "$Test_}Parameter";
	try
		{
		expected = "ParseException";
		obtained = resolver.Resolve (pattern);
		}
	catch (ParseException exception)
		{
		expected = obtained = exception.getMessage ();
		}
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "{}";
	try
		{
		expected = "ParseException";
		obtained = resolver.Resolve (pattern);
		}
	catch (ParseException exception)
		{
		expected = obtained = exception.getMessage ();
		}
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "{..:}";
	try
		{
		expected = "ParseException";
		obtained = resolver.Resolve (pattern);
		}
	catch (ParseException exception)
		{
		expected = obtained = exception.getMessage ();
		}
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Valid parameter.
	if (Verbose)
		System.out.println
			("--- configuration.Set (\"Test_Parameter\", \"" + expected + "\");");
	expected = "Test";
	configuration.Set ("Test_Parameter", expected);
	pattern = "${Test_Parameter}";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Multiple parameter references.
	pattern = "${Test_Parameter}-${Test_Parameter}";
	expected = "Test-Test";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Nested parameter references.
	pattern = "${${Test_Parameter}_Parameter}";
	expected = "Test";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Parameter reference resolves to parameter reference.
	if (Verbose)
		System.out.println
			("--- configuration.Set "
			+"(\"Test_Parameter\", \"${Rescan_Parameter}\");");
	configuration.Set ("Test_Parameter", "${Rescan_Parameter}");
	if (Verbose)
		System.out.println
			("--- configuration.Set "
			+"(\"Rescan_Parameter\", \"" + expected + "\");");
	configuration.Set ("Rescan_Parameter", expected);
	pattern = "${Test_Parameter}";
	expected = "Test";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Mulitiple parameter selection.
	if (Verbose)
		System.out.println
			("--- configuration.Add "
			+"(new Parameter (\"Test_Parameter\").Value (\"Single Value\"));");
	configuration.Add (new Parameter ("Test_Parameter").Value ("Single Value"));
	if (Verbose)
		System.out.println
			("--- configuration.Add\n"
			+"      (new Parameter (\"Test_Parameter\").Value\n"
			+"        (new Value (\"0\").Add (\"1\").Add (\"2\")\n"
			+"          .Add (new Value (\"10\").Add (\"20\").Add (\"30\"))\n"
			+"        .Add(\"3\")));");
	configuration.Add (new Parameter ("Test_Parameter").Value
		(new Value ("0").Add ("1").Add ("2")
				.Add (new Value ("10").Add ("20").Add ("30"))
			.Add ("3")));
	if (Verbose)
		System.out.println
			("---- configuration.Description ();\n"
			+ configuration.Description ());

	pattern = "${Test_Parameter@1}";
	expected = "Single Value";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2}";
	expected = "0";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[1]}";
	expected = "1";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[3][2]}";
	expected = "30";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[3][-1]}";
	expected = "10,20,30";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@1[-1]}";
	expected = "Single Value";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[-1]}";
	expected = "0,1,2,10,20,30,3";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[2][-1]}";
	expected = "2";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[${Test_Parameter@2[4]}][1]}";
	expected = "20";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	pattern = "${Test_Parameter@2[ (${Test_Parameter@2[3][1]} / 10) + 1 ][1]}";
	expected = "20";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Escaped characters.
	pattern = "\\$\\{Test_Parameter\\}";
	expected = pattern;
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);


	//	"Resolve_to_End_Reference" method for a parameter reference.
	pattern = "${Test_Parameter}";
	expected = "Test_Parameter";
	obtained = resolver.Resolve_to_End_Reference (pattern);
	Check ("resolver.Resolve_to_End_Reference (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	"Resolve_to_End_Reference" method for a field reference.
	pattern = "{MOC.MGSC.FILE_SPECIFICATION_NAME:PRODUCT_ID = 'SPO-1-210/03'}";
	expected = "MOC.MGSC.FILE_SPECIFICATION_NAME:PRODUCT_ID = 'SPO-1-210/03'";
	obtained = resolver.Resolve_to_End_Reference (pattern);
	Check ("resolver.Resolve_to_End_Reference (\"" + pattern + "\")",
		expected, obtained, Verbose);


	//	Setup the test table.

	if (Verbose)
		System.out.println
			("--- database.Delete (\"" + TEST_TABLE + "\", (Vector)null);");
	try {database.Delete (TEST_TABLE, (Vector)null);}
	catch (Database_Exception exception) {/* No such table expected. */}

	Vector
		fields = new Vector (),
		values = new Vector ();
	fields.add ("ID");			values.add ("INT PRIMARY KEY AUTO_INCREMENT NOT NULL");
	fields.add ("VOLUME");		values.add ("INT");
	fields.add ("NAME");		values.add ("VARCHAR(20)");
	fields.add ("DESCRIPTION");	values.add ("VARCHAR(80)");
		if (Verbose)
		System.out.println
			("--- database.Create (\"" + TEST_TABLE + "\",\n\""
			+ fields + "\",\n\""
			+ values + "\");");
	database.Create (TEST_TABLE, fields, values);

	values.clear ();
	values.add ("NULL");	//	Auto_increment the ID.
	values.add ("");
	values.add ("");
	values.add ("");
	for (int count = 0;
	    	 count < 5;
			 count++)
		{
		values.set (1, new Integer (count));
		values.set (2, "Volume_" + count + "/Name_" + count);
		values.set (3, "Description for entry " + count);
		if (Verbose)
			System.out.println
				("--- database.Insert (\"" + TEST_TABLE + "\",\n\""
				+ fields + "\",\n\""
				+ values + "\");");
		database.Insert (TEST_TABLE, fields, values);
		}
	if (Verbose)
		Print_Table (database.Select (TEST_TABLE));

	//	Valid field reference.
	pattern = "{" + TEST_TABLE + ".VOLUME:ID = 2}";
	expected = "1";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);

	//	Nested feild references.
	pattern =
		"{" + TEST_TABLE + ".DESCRIPTION:NAME = "
			+ "'Volume_{" + TEST_TABLE + ".Volume:ID = 3}"
			+   "/Name_{" + TEST_TABLE + ".Volume:ID = 3}'}";
	expected = "Description for entry 2";
	obtained = resolver.Resolve (pattern);
	Check ("resolver.Resolve (\"" + pattern + "\")",
		expected, obtained, Verbose);


	Status = 0;
	}
catch (Configuration_Exception exception)
	{
	System.out.println 
		("!!!!: Configuration_Exception -\n"
		+ exception.getMessage ());
	}
catch (Database_Exception exception)
	{
	System.out.println 
		("!!!!: Database_Exception -\n"
		+ exception.getMessage ());
	}
catch (ParseException exception)
	{
	System.out.println
		("!!!!: ParseException -\n"
		+ exception.getMessage ());
	}
catch (PVL_Exception exception)
	{
	System.out.println
		("!!!!: PVL_Exception -\n"
		+ exception.getMessage ());
	}
catch (Unresolved_Reference exception)
	{
	System.out.println
		("!!!!: Unresolved_Reference -\n"
		+ exception.getMessage ());
	}

System.out.println ('\n'
	+ "Total tests: " + Tests_Total + '\n'
	+ "Passed tests: " + Tests_Passed);
System.exit (Status);
}


public static void Print_Table
	(
	Vector	table
	)
{
Iterator
	table_list = table.iterator ();
while (table_list.hasNext ())
	System.out.println (table_list.next ());
System.out.println ();
}


public static boolean Check
	(
	String	description,
	String	expected,
	String	obtained,
	boolean	verbose
	)
{
boolean
	matches = obtained.equals (expected);
System.out.print ((matches ? "PASS" : "FAIL") + ": " + description);
if (verbose || ! matches)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else
	System.out.println (" - " + obtained);
Tests_Total++;
if (matches)
	Tests_Passed++;
return matches;
}

}
