/*	Memory_Chart_Panel

CVS ID: Memory_Chart_Panel.java,v 1.4 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	javax.swing.JPanel;
import	javax.swing.BorderFactory;
import	javax.swing.JPopupMenu;
import	javax.swing.JMenuItem;
import	javax.swing.JCheckBoxMenuItem;
import	javax.swing.JOptionPane;
import	javax.swing.JFileChooser;
import	javax.swing.Box;
import	javax.swing.InputMap;
import	javax.swing.ActionMap;
import	javax.swing.Action;
import	javax.swing.AbstractAction;
import	javax.swing.KeyStroke;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Insets;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.awt.event.MouseAdapter;
import	java.awt.event.MouseEvent;
import	java.awt.event.KeyListener;
import	java.awt.event.KeyEvent;
import	java.io.File;
import	java.io.FileNotFoundException;
import	java.io.PrintStream;
import	java.io.FileOutputStream;

/**	A <i>Memory_Chart_Panel</i> provides panel for a Memory_Chart.
<p>
	The panel may optionally be provided with a popup menu that includes
	Memory_Chart controls that are also bound to accelerator keys.
	In addition to basic Memory_Chart controls, a Save History action is
	provided that will save the current memory sampling history to a
	file in comma separated value (CSV) format.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.4
	@see	Memory_Chart
*/
public class Memory_Chart_Panel
	extends JPanel
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Viewers.Memory_Chart_Panel (1.4 2012/04/16 06:22:58)";

/**	The default {@link Memory_Chart} sampling rate.
*/
public static final int
	DEFAULT_SAMPLING_RATE	= 1;

private Memory_Chart
	Chart;

private JPopupMenu
	Popup_Menu		= null;
private JMenuItem
	Start_Stop_MenuItem;
private JCheckBoxMenuItem
	Annotation_MenuItem,
	Seconds_Scale_MenuItem;

/**	Menu mnemonic and keyboard accellerator keys.
<p>
	When the {@link #Menu() menu} is visible, each key combined with the
	ALT key will select the corresponding menu item. When the component
	has focus, each key combined with the CTL (control) key will cause
	the corresponding action to occur.
*/
public static final int
	STOP_START_KEY		= KeyEvent.VK_S,
	SAMPLING_RATE_KEY	= KeyEvent.VK_R,
	ANNOTATION_KEY		= KeyEvent.VK_A,
	SECONDS_SCALE_KEY	= KeyEvent.VK_E,
	SAVE_HISTORY_KEY	= KeyEvent.VK_H;

private JFileChooser
	File_Chooser	= new JFileChooser (System.getProperty ("user.dir"));

/*==============================================================================
	Constructors
*/
/**	Construct a Memory_Chart_Panel using a sampling rate and optional
	popup menu.
<p>
	A border with the title "Memory Use" is placed around a {@link
	Memory_Chart} that uses the specified sampling rate. A popup {@link
	#Menu() menu} is provided if requested.
<p>
	@param	rate	The memory sampling rate for the Memory_Chart.
	@param	provide_menu	If true, a popup menu is provided.
*/
public Memory_Chart_Panel
	(
	int		rate,
	boolean	provide_menu
	)
{
setLayout (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Memory chart.
Chart = new Memory_Chart (rate);
setBorder (BorderFactory.createTitledBorder ("Memory Use"));
location.fill = GridBagConstraints.BOTH;
location.weightx =
location.weighty = 1.0;
add (Chart, location);

if (provide_menu)
	{
	//	Popup menu.
	Popup_Menu = Menu ();
	addMouseListener (new Popup_Listener ());
	}
}

/**	Construct a Memory_Chart_Panel using a sampling rate.
<p>
	A popup {@link #Menu() menu} is provided.
<p>
	@param	rate	The memory sampling rate for the Memory_Chart.
*/
public Memory_Chart_Panel
	(
	int		rate
	)
{this (rate, true);}

/**	Construct a Memory_Chart_Panel with an optional popup menu.
<p>
	The Memory_Chart will use the {@link #DEFAULT_SAMPLING_RATE}.
<p>
	@param	provide_menu	If true, a popup menu is provided.
*/
public Memory_Chart_Panel
	(
	boolean	provide_menu
	)
{this (DEFAULT_SAMPLING_RATE, provide_menu);}

/**	Construct a Memory_Chart_Panel with a popup menu and using the
	{@link #DEFAULT_SAMPLING_RATE}.
*/
public Memory_Chart_Panel ()
{this (DEFAULT_SAMPLING_RATE, true);}

/*==============================================================================
	Accessors
*/
/**	Test if the Memory_Chart {@link Memory_Chart#Annotation(boolean)
	annotation} is enabled.
<p>
	@return	true if the annotation is enabled; false otherwise.
*/
public boolean Annotation ()
{return Chart.Annotation ();}

/**	Test if the Memory_Chart {@link Memory_Chart#Seconds_Scale(boolean)
	seconds scale} is enabled.
<p>
	@return	true if the seconds scale is enabled; false otherwise.
*/
public boolean Seconds_Scale ()
{return Chart.Seconds_Scale ();}

/**	Get the memory sampling rate.
<p>
	@return	The time, in seconds, between memory samples.
	@see	Memory_Chart#Rate(int)
*/
public int Rate ()
{return Chart.Rate ();}

/**	Get the previous memory sampling rate.
<p>
	@return	The previous memory history sampling rate. This will be
		zero only if there was no non-zero previous sampling rate.
	@see	Memory_Chart#Previous_Rate()
*/
public int Previous_Rate ()
{return Chart.Previous_Rate ();}

/**	Get the amount of memory allocated by the Java runtime environment
	but not currently in use.
<p>
	@return	The amount, in bytes, of allocated but free memory.
	@see	Memory_History#Available_Memory()
*/
public static long Free_Memory ()
{return Memory_History.Free_Memory ();}

/**	Get the amount of memory currently in use by the Java runtime
	environment.
<p>
	@return	The amount, in bytes, of allocated memory.
	@see	Memory_History#Available_Memory()
*/
public static long Allocated_Memory ()
{return Memory_History.Allocated_Memory ();}

/**	Get the total amount of memory available to the Java runtime
	environment.
<p>
	@return	The total amount of memory, in bytes, available to the Java
		runtime environment.
	@see	Memory_History#Available_Memory()
*/
public static long Available_Memory ()
{return Memory_History.Available_Memory ();}

/*==============================================================================
	Menu
*/
/**	Get a popup menu with Memory_Chart controls.
<p>
	Controls to {@link Memory_Chart#Stop() stop} and {@link
	Memory_Chart#Start() start} memory sampling, set the
	{@link Memory_Chart#Rate(int) sampling rate}, enable and
	disable the {@link Memory_Chart#Annotation(boolean) annotations}
	and {@link Memory_Chart#Seconds_Scale(boolean) seconds scale},
	and save the current {@link Memory_History#History() sampling
	history} to a file are provided.
<p>
	The menu actions are also bound to keyboard accelerators.
<p>
	The Save History action will use a file chooser to select a file in
	which the current memory sampling history will be written in comma
	separated values (CSV) format. The first line provides the names of
	the values on the following lines (one line per sample): "Sample" is
	the sample index, starting with zero; "Allocated" followed by a
	slash characater ('/') and the total amount of memory available is
	the amount of {@link Memory_Chart#Allocated_Memory() allocated
	memory}; and "Free" is the amount of {@link
	Memory_Chart#Free_Memory() free memory} in the allocated pool.
<p>
	@return	A JPopupMenu. 
*/
public JPopupMenu Menu ()
{
if (Popup_Menu == null)
	{
	Popup_Menu = new JPopupMenu ("Chart Controls");

	JMenuItem
		menu_item;
	Action
		action;
	InputMap
		input_map = getInputMap ();
	ActionMap
		action_map = getActionMap ();
	KeyStroke
		keystroke;

	Start_Stop_MenuItem = new JMenuItem
		((Chart.Rate () == 0) ? "Start" : "Stop");
	Start_Stop_MenuItem.setMnemonic
		(STOP_START_KEY);
	keystroke = KeyStroke.getKeyStroke
		(STOP_START_KEY, ActionEvent.CTRL_MASK);
	Start_Stop_MenuItem.setAccelerator (keystroke);
	action = new AbstractAction ()
		{public void actionPerformed (ActionEvent event)
		{Run (Chart.Rate () == 0);}};
	Start_Stop_MenuItem.addActionListener (action);
	input_map.put (keystroke, "Run");
	action_map.put ("Run", action);
	Popup_Menu.add (Start_Stop_MenuItem);

	menu_item = new JMenuItem ("Sampling rate ...");
	menu_item.setMnemonic
		(SAMPLING_RATE_KEY);
	keystroke = KeyStroke.getKeyStroke
		(SAMPLING_RATE_KEY, ActionEvent.CTRL_MASK);
	menu_item.setAccelerator (keystroke);
	action = new AbstractAction ()
		{public void actionPerformed (ActionEvent event)
			{Select_Rate ();}};
	menu_item.addActionListener (action);
	input_map.put (keystroke, "Rate");
	action_map.put ("Rate", action);
	Popup_Menu.add (menu_item);

	Annotation_MenuItem = new JCheckBoxMenuItem ("Annotation",
		Chart.Annotation ());
	Annotation_MenuItem.setMnemonic
		(ANNOTATION_KEY);
	keystroke = KeyStroke.getKeyStroke
		(ANNOTATION_KEY, ActionEvent.CTRL_MASK);
	Annotation_MenuItem.setAccelerator (keystroke);
	action = new AbstractAction ()
		{public void actionPerformed (ActionEvent event)
		{Annotation (! Chart.Annotation ());}};
	Annotation_MenuItem.addActionListener (action);
	input_map.put (keystroke, "Annotation");
	action_map.put ("Annotation", action);
	Popup_Menu.add (Annotation_MenuItem);

	Seconds_Scale_MenuItem = new JCheckBoxMenuItem ("Seconds Scale",
		Chart.Seconds_Scale ());
	Seconds_Scale_MenuItem.setMnemonic
		(SECONDS_SCALE_KEY);
	keystroke = KeyStroke.getKeyStroke
		(SECONDS_SCALE_KEY, ActionEvent.CTRL_MASK);
	Seconds_Scale_MenuItem.setAccelerator (keystroke);
	action = new AbstractAction ()
		{public void actionPerformed (ActionEvent event)
		{Seconds_Scale (! Chart.Seconds_Scale ());}};
	Seconds_Scale_MenuItem.addActionListener (action);
	input_map.put (keystroke, "Seconds");
	action_map.put ("Seconds", action);
	Popup_Menu.add (Seconds_Scale_MenuItem);

	menu_item = new JMenuItem ("Save History ...");
	menu_item.setMnemonic
		(SAVE_HISTORY_KEY);
	keystroke = KeyStroke.getKeyStroke
		(SAVE_HISTORY_KEY, ActionEvent.CTRL_MASK);
	menu_item.setAccelerator (keystroke);
	action = new AbstractAction ()
		{public void actionPerformed (ActionEvent event)
			{Save_History ();}};
	menu_item.addActionListener (action);
	input_map.put (keystroke, "History");
	action_map.put ("History", action);
	Popup_Menu.add (menu_item);
	}
return Popup_Menu;
}

/*------------------------------------------------------------------------------
	Actions
*/
/**	Enable or disable the operation of the Memory_Chart.
<p>
	@param	run	If true, memory sampling will be {@link Memory_Chart#Start()
		started} if it is not already running. If false, sampling will be
		{@link Memory_Chart#Stop() stopped}.
	@return	This Memory_Chart_Panel.
*/
public Memory_Chart_Panel Run
	(
	boolean	run
	)
{
if (run)
	{
	if (Chart.Rate () == 0)
		{
		Chart.Start ();
		Start_Stop_MenuItem.setText ("Stop");
		}
	}
else if (Chart.Rate () != 0)
	{
	Chart.Stop ();
	Start_Stop_MenuItem.setText ("Start");
	}
return this;
}

/**	Set the memory sampling rate.
<p>
	@param	rate	The memory {@link Memory_Chart#Rate(int) sampling
		rate}.
	@return	This Memory_Chart_Panel.
*/
public Memory_Chart_Panel Rate
	(
	int		rate
	)
{
Chart.Rate (rate);
return this;
}

/**	Use an input dialog to select the memory sampling rate.
*/
private void Select_Rate ()
{
String
	rate = JOptionPane.showInputDialog (this,
		"Memory chart sampling rate (seconds):",
		String.valueOf (Chart.Rate ()));
if (rate != null)
	{
	try {Chart.Rate (Integer.parseInt (rate));}
	catch (NumberFormatException exception)
		{
		Dialog_Box.Error
			(rate + " is not a valid sampling rate number.", this);
		}
	}
}

/**	Enable or disable the Memory_Chart annotations.
<p>
	@param	enabled	If true, chart {@link
		Memory_Chart#Annotation(boolean) annotations} will be shown. If
		false they will not be shown.
	@return	This Memory_Chart_Panel.
*/
public Memory_Chart_Panel Annotation
	(
	boolean	enabled
	)
{
Chart.Annotation (enabled);
Annotation_MenuItem.setSelected (enabled);
return this;
}

/**	Enable or disable the Memory_Chart seconds scale.
<p>
	@param	enabled	If true, chart {@link
		Memory_Chart#Seconds_Scale(boolean) seconds scale} will be
		shown. If false it will not be shown.
	@return	This Memory_Chart_Panel.
*/
public Memory_Chart_Panel Seconds_Scale
	(
	boolean	enabled
	)
{
Chart.Seconds_Scale (enabled);
Seconds_Scale_MenuItem.setSelected (enabled);
return this;
}

/**	Use a file chooser to specify a file in which the memory sampling
	history will be saved.
*/
private void Save_History ()
{
if (File_Chooser.showSaveDialog (this)
		!= JFileChooser.APPROVE_OPTION)
	return;
File
	file = File_Chooser.getSelectedFile ();
try {CWD (file);}
catch (FileNotFoundException exception)
	{
	Dialog_Box.Notice
		("The pathname to " + file.getPath () + '\n'
		+"does not exist.\n\n"
		+ exception.getMessage (),
		this);
	return;
	}
if (file.exists ())
	{
	if (file.isDirectory ())
		{
		Dialog_Box.Notice
			("Can't replace directory " + file.getAbsolutePath (), this);
		return;
		}
	if (! Dialog_Box.Confirm
		("Replace " + file.getAbsolutePath () + '?', this))
		return;
	}

try
	{
	PrintStream
		output = new PrintStream (new FileOutputStream (file), true);
	long[][]
		history = Chart.getModel ().History ();
	output.println
		("Sample,Allocated/" + Chart.Available_Memory () + ",Free");
	for (int
			index = 0;
			index < history.length;
			index++)
		output.println (String.valueOf (index) + ','
			+history[index][0] + ',' + history[index][1]);
	output.close ();
	if (output.checkError ())
		Dialog_Box.Error
			("There was a problem writing the file "
			+ file.getAbsolutePath () + "\n\n",
			this);
	}
catch (FileNotFoundException exception)
	{
	Dialog_Box.Error
		("Can't open file " + file.getAbsolutePath () + "\n\n"
		+ exception.getMessage (),
		this);
	}
catch (SecurityException exception)
	{
	Dialog_Box.Error
		("Denied write access to file " + file.getAbsolutePath () + "\n\n"
		+ exception.getMessage (),
		this);
	}
}

/**	Sets the current working directory for finding parameter files.
<P>
	@param	file	A File to which to set the CWD. If null, the
		"user.dir" System property will be used. If the file does not
		refer to an existing directory, the file's parent will be used.
	@return	This Parameter_View.
	@throws	FileNotFoundException	If neither the file nor the file's
		parent refers to an existing directory.
*/
public Memory_Chart_Panel CWD
	(
	File	file
	)
	throws FileNotFoundException
{
if (file == null)
	file = new File (System.getProperty ("user.dir"));
if (! file.isAbsolute ())
	file = file.getAbsoluteFile ();
if (! file.isDirectory ())
	//	Can only set directories.
	file = file.getParentFile ();
if (file.exists ())
	File_Chooser.setCurrentDirectory (file);
else
	throw new FileNotFoundException
		(ID + '\n'
		+"Can't set working directory to " + file.getAbsolutePath () + '\n'
		+"No such directory.");
return this;
}

/*..............................................................................
	Popup Menu Listener
*/
/**	The <i>Popup_Listener</i> is a MouseAdapter used to listen for
	mouse events.
*/
class Popup_Listener
	extends MouseAdapter
{
public void mousePressed (MouseEvent event)
{maybeShowPopup (event);}

public void mouseReleased (MouseEvent event)
{maybeShowPopup (event);}

private void maybeShowPopup
	(
	MouseEvent	event
	)
{
if (event.isPopupTrigger ())
	Popup_Menu.show (event.getComponent (), event.getX (), event.getY ());
}
}


}
