/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Ricardo Mendoza: ricardo.mendoza@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "powerd-internal.h"
#include "powerd-sensors.h"
#include "log.h"
#include <atomic>

#include <ubuntu/application/sensors/light.h>
#include <ubuntu/application/sensors/proximity.h>

namespace
{

UASensorsProximity* prox_sensor;
UASensorsLight* light_sensor;
GMainLoop* main_loop = NULL;
std::atomic<bool> allow_synthetic{false};
enum class Proximity { unknown, near, far };
std::atomic<Proximity> current_proximity{Proximity::unknown};

void update_and_emit_proximity(Proximity proximity)
{
    current_proximity = proximity;
    if (proximity == Proximity::near)
        powerd_proximity_event(TRUE);
    else if (proximity == Proximity::far)
        powerd_proximity_event(FALSE);
}

gboolean emit_synthetic_proximity_far(void *context)
{
    if (allow_synthetic)
        update_and_emit_proximity(Proximity::far);

    return FALSE;
}

void on_new_proximity_event(UASProximityEvent *event, void *context)
{
    allow_synthetic = false;
    switch (uas_proximity_event_get_distance(event))
    {
        case U_PROXIMITY_NEAR:
        {
            update_and_emit_proximity(Proximity::near);
            break;
        }
        case U_PROXIMITY_FAR:
        {
            update_and_emit_proximity(Proximity::far);
            break;
        }
    }
}

}

void powerd_sensors_proximity_enable(void)
{
    //FIXME: Some proximity sensors do not
    //send an initial event when enabled and nothing is close
    //To work around this, we schedule a synthetic proximity
    //far event in case no event has been emitted 500ms
    //after enabling the proximity sensor
    allow_synthetic = true;
    current_proximity = Proximity::unknown;
    g_timeout_add(500, emit_synthetic_proximity_far, NULL);

    ua_sensors_proximity_enable(prox_sensor);
}

void powerd_sensors_proximity_disable(void)
{
    allow_synthetic = false;
    ua_sensors_proximity_disable(prox_sensor);
    current_proximity = Proximity::unknown;
}

void powerd_sensors_proximity_emit(void)
{
    update_and_emit_proximity(current_proximity);
}

void on_new_als_event(UASLightEvent *event, void *context)
{
    float lux;
    uas_light_event_get_light(event, &lux);
    powerd_new_als_event(lux);
}

void powerd_sensors_als_enable(void)
{
    if (light_sensor)
        ua_sensors_light_enable(light_sensor);
}

void powerd_sensors_als_disable(void)
{
    if (light_sensor)
        ua_sensors_light_disable(light_sensor);
}

void powerd_sensors_init(GMainLoop *ml)
{
    main_loop = ml;
    prox_sensor = ua_sensors_proximity_new();
    if (prox_sensor != NULL) {
        ua_sensors_proximity_set_reading_cb(prox_sensor,
                                            on_new_proximity_event,
                                            NULL);
    } else {
        powerd_warn("Failed to allocate proximity sensor");
    }

    light_sensor = ua_sensors_light_new();
    if (light_sensor) {
        ua_sensors_light_set_reading_cb(light_sensor, on_new_als_event,
                                        NULL);
    } else {
        powerd_warn("Failed to allocate ambient light sensor");
    }
}
